/*
 * This file is part of din.
 *
 * din is copyright (c) 2006 - 2012 S Jagannathan <jag@dinisnoise.org>
 * For more information, please visit http://dinisnoise.org
 *
 * din is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * din is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with din.  If not, see <http://www.gnu.org/licenses/>.
 *
*/
#ifndef __SOLVER
#define __SOLVER

#include "crvpt.h"
#include "multi_curve.h"
#include <vector>


struct solver;

struct xhandler {
  virtual void operator() (solver& s, float& x, float& dx) = 0;
  virtual ~xhandler () {}
};

struct atmin: xhandler {
  void operator() (solver& s, float& x, float& dx);
};

struct atmax: xhandler {
  void operator() (solver& s, float& x, float& dx);
};

struct tomin: xhandler {
  void operator() (solver& s, float& x, float& dx);
};

struct loopmin : atmin {
  void operator() (solver& s, float& x, float& dx);
};

struct loopmax : tomin {
  void operator() (solver& s, float& x, float& dx);
};

struct pongmax: atmax {
  void operator() (solver& s, float& x, float& dx);
};

struct pongmin: atmin {
  void operator() (solver& s, float& x, float& dx);
};

struct curve_editor;

struct gotog : xhandler {
  float g;
  curve_editor* ed;
  gotog (float gg, curve_editor* e); 
  void operator() (solver& s, float& x, float& dx);
  void set (float gg);
};

extern atmin _atmin;
extern atmax _atmax;
extern tomin _tomin;
extern loopmin _loopmin;
extern loopmax _loopmax;
extern pongmin _pongmin;
extern pongmax _pongmax;
extern gotog _gotog;

struct solver {

  //
  // given x return y
  //
  // assumes x such that
  // x0 <= x1 <= x2 <= ...... <= xlast
  //

  // solved curve
  multi_curve* mcrv;

  int ncurvs, icurv, last_curv;
  int iseg;

  // line segment on curve expected to contain x
  float startx, starty;
  float endx, endy;
  float m, inf;
  float ycomp;

  // first and last points
  float firstx, firsty;
  float lastx, lasty;

  float result; // last result

  inline int inseg (float x, int c, int i) {
    std::vector<curve>& curv = mcrv->curv;
    std::vector<crvpt>& vpts = curv[c].vpts;
    float leftx = vpts[i].x, rightx = vpts[i+1].x;

    if (x >= leftx && x <= rightx) {
        setseg (c, i);
        return 1;
    }
    return 0;
  }

  void setseg (int c, int i) {

    std::vector<curve>& curv = mcrv->curv;
    std::vector<crvpt>& vpts = curv[c].vpts;

    crvpt& start = vpts[i];
    crvpt& end = vpts[i+1];

    startx = start.x;
    starty = start.y;
    endx = end.x;
    endy = end.y;
    m = start.m;
    inf = start.inf;

    if (inf == 1) ycomp = endy; else ycomp = starty - m * startx;

    icurv = c;
    iseg = i;

  }

  inline int lastseg (int c) {
    std::vector<curve>& curv = mcrv->curv;
    std::vector<crvpt>& vpts = curv[c].vpts;
    return vpts.size() - 2;
    return 0;
  }

  inline int getsegs (int c) {
    return (mcrv->curv[c].vpts.size() - 1);
  }

  inline int searchleft (float x) {
    int lseg = iseg, lcurv = icurv;
    while (1) {
      if (--lseg < 0) {
        --lcurv;
        if (lcurv < 0) break;
        lseg = lastseg (lcurv);
      }
      if (inseg (x, lcurv, lseg)) return 1;
    }
    return 0;

  }

  inline int searchright (float x) {
    int rseg = iseg, rcurv = icurv;
    int rsegs = getsegs (rcurv);
    while (1) {
      if (++rseg < rsegs); else {
        ++rcurv;
        if (rcurv >= ncurvs) break;
        rsegs = getsegs (rcurv);
        rseg = 0;
      }
      if (inseg (x, rcurv, rseg)) return 1;
    }
    return 0;

  }

  inline int findseg (float x) {

    int fcurv = 0, fseg = 0, nsegs = getsegs (fcurv);
    while (1) {
      if (inseg (x, fcurv, fseg)) return 1; else {
        ++fseg;
        if (fseg >= nsegs) {
          if (++fcurv >= ncurvs) {
            setseg (ncurvs - 1, nsegs - 1);
            break;
          }
          fseg = 0;
          nsegs = getsegs (fcurv);
        }
      }
    }
    return 0;

  }

  solver ();
  solver (multi_curve* crv);

  void operator() (multi_curve* crv);

  // given x, solve y
  float operator() (float x);
  float operator() (float &x, float& dx, xhandler& xmin = _atmin, xhandler& xmax = _tomin);

  void operator() (float& x, float& dx, int n, float* soln, xhandler& xmin = _atmin, xhandler& xmax = _tomin); // fast solver when x is incremented by dx n times, solution stored in soln
  void operator() (float& x, float& dx, int n, float* mod, float* soln, xhandler& xmin = _atmin, xhandler& xmax = _tomin); // same as last solver but x is also modulated by mod.
  void operator() (float* ax, int n, xhandler& xmin = _atmin, xhandler& xmax = _atmax); // given an array of x, store solution in same array --> used by compressor

  void init ();
  
  void update (); // when multi curve has changed


};




#endif
