// Reconcile DEP-5 debian/copyright to licensecheck
//
// Copyright : 2025 P Blackman
// License   : BSD-2-clause
//
// Routine to encapsulate array of file data

unit filedata;
{$mode delphi}

interface uses Classes;

procedure GlobSearch (Str : String; out S1 : AnsiString);
Function  SetDep5License   (F, PD : Integer; lic : string) : Boolean;
procedure SetActualLicense (F : Integer; lic : string);
procedure InitFileData (SourceList : tStringList);
procedure UnMangleFileNames;

function FindThisFile (const FileStr : AnsiString) : Integer;
function GetMatch (out FileNum : Integer; Filename : string) : Boolean;
function CheckFiles : Boolean;


implementation uses SysUtils, StrUtils, rstrings, support,
        exclude, gfdl, gpl, gpla, spdx, spdx2, dotzero,
        apache, ntprsa, eclipse, psf, andor, bsd0, options;

type
    tFileLic =
    record
        PathDepth : Integer;
        FName : AnsiString;
        Dep5,
        Actual : String;  // License short names
    end;
    tFileLicArray = array of tFileLic;

var
    MyFiles : tFileLicArray;


// use * to match anything in a file name string
// Effects a recursive directory match
procedure GlobSearch (Str : String; out S1 : AnsiString);
var P : Integer;
    BStr, EStr : String;
    MyFile : tFileLic;
begin
    S1 := '';
    P    := PosEx ('*', Str);
    Assert (P <> 0, 'Lost the asterisk!');

    BStr := Copy (Str, 1, P-1);
    EStr := Copy (Str, P+1, length(Str) -P);

    for MyFile in MyFiles do
        if  StartsStr (BStr, MyFile.FName)
        and EndsStr   (EStr, MyFile.FName) then
            S1 := S1 + MyFile.FName + LineEnding;
end;

function SetDep5License (F, PD : Integer; lic : string) : Boolean;
begin
    result := (PD = -1)                         // Not loaded via a globbing pattern
              or (MyFiles[F].Dep5 = '')         // First time for this file
              or (PD >= MyFiles[F].PathDepth);  // Depth OK if not lower

    MyFiles[F].PathDepth := PD;
    MyFiles[F].Dep5      := lic;
end;

procedure SetActualLicense (F : Integer; lic : string);
begin
    MyFiles[F].Actual := lic;
end;

procedure InitFileData (SourceList : tStringList);
var C, Posn : Integer;
    Line    : AnsiString;
begin
    SetLength (MyFiles, SourceList.Count);
    for C := 0 to SourceList.Count -1 do
    begin
        Posn             := 3; // Strip leading ./
        Line             := SourceList.Strings[C];
        MyFiles[C].Fname := ExtractSubstr (Line, Posn, []);
        MyFiles[C].Dep5  := '';
        MyFiles[C].Actual:= '';
        MyFiles[C].PathDepth := 0;
    end;
end;

procedure UnMangleFileNames;
var F : Integer;
begin
    for F := 0 to High (MyFiles) do
        Unmanglename (MyFiles[F].Fname);
end;

// locate a file from d/copyright, find its index in the source file array
function FindThisFile (const FileStr : AnsiString) : Integer;
var F : Integer;
    Found : Boolean;
begin
    Found := false;
    F := 0;

    while not found and (F < Length (MyFiles)) do
    begin
        if MyFiles [F].FName = FileStr then
            found := true
        else
            inc(F);
    end;

    if not found then
    begin
        writeln ('** ' + rsSfp + ' ', FileStr); // Superfluous file pattern
        F := -1;
    end;

    result := F;
end;

// Search for given file name, and return its index
function GetMatch (out FileNum : Integer; Filename : string) : Boolean;
begin
    FileNum  := 0;
    result := FileName = MyFiles[FileNum].Fname;
    While not result and (FileNum < High (MyFiles)) do
    begin
        FileNum  := FileNum +1;
        result := FileName = MyFiles[FileNum].Fname;
    end;
end;

// Traverse the file data array,
// looking for mismatch in license strings.
// Output the main body of the report.
function CheckFiles : Boolean;
var F : tFileLic;
    Header,
    GotOne,
    MisMatch,
    FalsePositive : Boolean;
    last_Dep5,
    Last_Actual : String;

begin
    Header      := False;
    GotOne      := False;
    MisMatch    := False;
    Last_Dep5   := '';
    Last_Actual := '';

    for F in MyFiles do
      with F do
        if (Actual = '') then
            // Nothing to do
        else
        begin
            MisMatch := not SameText(Dep5, Actual);
            FalsePositive := false;

            if MisMatch and not IgnoreFile (Fname) then
                FalsePositive :=
                       // Workarounds for various problems with licensecheck
                       CheckGPL     (Fname, Dep5, Actual)
                    or CheckGPLa    (Fname, Dep5, Actual)
                    or CheckSPDX    (Fname, Dep5, Actual)
                    or CheckSPDX2   (Fname, Dep5, Actual)
                    or CheckApache  (Fname, Dep5, Actual)
                    or CheckGFDL    (Fname, Dep5, Actual)
                    or CheckEclipse (Fname, Dep5, Actual)
                    or CheckPSF2    (Fname, Dep5, Actual)
                    or CheckBSD0    (Fname, Dep5, Actual)
                    or CheckDotZero (Dep5, Actual)
                    or CheckNTPRSA  (Dep5, Actual)
                    or CheckANDOR   (Dep5, Actual)
                    or ContainsStr (Actual, 'Autoconf-data');

            if not IgnoreFile (Fname) and (Option_Long
            or MisMatch and not FalsePositive) then
            begin
               if not Header and not Option_Format then
                begin
                    Writeln ('d/copyright      | licensecheck');
                    Writeln;
                    Header := True;
                end;

                if Option_Short and (Dep5 = last_Dep5) and (Actual = Last_Actual) then
                    // skip this file
                else
                if Option_Format then
                begin
                    Writeln (Dep5);
                    Writeln (Actual);
                    Writeln (FName);
                    Writeln;
                end
                else
                    Writeln (PadRight(Dep5,17), '| ', PadRight(Actual,17), ' ',FName);

                Last_Dep5   := Dep5;
                Last_Actual := Actual;
                GotOne      := GotOne or (MisMatch and not FalsePositive);
            end;
        end;

    result := GotOne;
end;

end.
