# NAME

goto::file - Stop parsing the current file and move on to a different one.

# DESCRIPTION

It is rare, but there are times where you want to swap out the currently
compiling file for a different one. This module does that. From the point you
`use` the module perl will be parsing the new file instead of the original.

# WHY?!

This was created specifically for [Test2::Harness](https://metacpan.org/pod/Test2::Harness) which can preload modules
and fork to run each test. The problem was that using `do` to execute the test
files post-fork was resuling in extra frames in the stack trace... in other
words there are a lot of tests that assume the test file is the bottom of the
stack. This happens all the time, specially if stack traces need to be
verified.

This module allows Test2::Harness to swap out the main script for the new file
without adding a stack frame.

# SYNOPSIS

Plain and simple:

    #!/usr/bin/perl

    use goto::file 'some_file.pl';

    die "This will never be seen!";

    __DATA__

    This data will not be seen by <DATA>

More useful:

    #!/usr/bin/perl

    BEGIN {
        my $file = should_switch_files();

        if ($file) {
            print "about to switch to file '$file'\n";
            require goto::file;
            goto::file->import($file);
        }
    }

    print "Did not go to a file\n";

Another thing you can do:

    use goto::file [
        'print "Hi!\n";',
        "exit 0",
    ];

    die "Will not get here";

# NOTES

- \_\_DATA\_\_ and <DATA>

    This module does its very best to make sure the data you get from <DATA> comes
    from the NEW file, and not the old. At the moment there are no known failure
    cases, but there could be some.

# IMPLEMENTATION DETAILS

This is a source filter. The source filter simply disgards the lines from the
original file and instead feeds perl lines from the new file. There is also a
small source injection at the start that sets up `<DATA>` and makes sure
line numbers and file name are all correct.

# SOURCE

The source code repository for goto-file can be found at
`http://github.com/exodist/goto-file/`.

# MAINTAINERS

- Chad Granum <exodist@cpan.org>

# AUTHORS

- Chad Granum <exodist@cpan.org>

# COPYRIGHT

Copyright 2017 Chad Granum <exodist7@gmail.com>.

This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

See `http://dev.perl.org/licenses/`
