Benchmarks
---

*Dislcaimer: remember benchmarks are generally fraught with issues and their
legitimacy depends on a multitude of factors.*

I've used basic, but seemingly nice, [`pytest-benchmark`](https://pytest-benchmark.readthedocs.io/en/stable/usage.html)
to produce the following benchmarks on a fairly modern machine.

If you'd like to run these yourself, you'll need whatever according system 
dependency, if any, required by the specific algorithm as well as the Python
lib installed. Can install all via `pip install -r benchmark-requirements.txt`. 

The resulting output from benchmarks indicate what file and lib was used, (`cramjam` vs `gzip`)
`test_gzip[urls.10K-cramjam]` indicates `cramjam` and `benchmarks/data/urls.10K` file was used during the gzip benchmark.


Special performance notes:
---
In general, if the length of the output from de/compression is known, for any variant,  
you can provide `output_len=<<some int>>` to get a 1-4x performance increase. This
is allows us to avoid double allocation. 

Furthermore, you can provide `output_len` equal to the "at most" length when 
de/compressing `bytearray` objects; as these can be resized after the de/compression 
is finished and actual size is known for basically the same performance. This is not
true for `bytes` objects, as we cannot resize after de/compression and the result
will be trialing null bytes.

For snappy used in conjunction with `bytearray`s, this is only midly helpful as
we automatically estimate the buffer size and can resize the `bytearray` after.


--- 

#### Gzip

`make bench-gzip`

(spoiler alert, although cramjam is generally faster than the builtin gzip module, you should pretty much always use [igzip](https://github.com/pycompression/python-isal) when possible.)

```bash 
----------------------------------------------------------------------------------------------------------------- benchmark: 42 tests ------------------------------------------------------------------------------------------------------------------
Name (time in us)                                           Min                       Max                      Mean                StdDev                    Median                   IQR            Outliers          OPS            Rounds  Iterations
--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
test_gzip[Mark.Twain-Tom.Sawyer.txt-cramjam]           192.0910 (3.41)           248.1120 (2.56)           211.3754 (3.45)         6.1655 (2.03)           210.4330 (3.48)         7.5975 (3.27)       924;75   4,730.9196 (0.29)       3255           1
test_gzip[Mark.Twain-Tom.Sawyer.txt-gzip]              231.0250 (4.10)           306.7920 (3.17)           252.1350 (4.11)         7.4904 (2.47)           251.5680 (4.16)         8.2273 (3.54)       520;89   3,966.1291 (0.24)       2153           1
test_gzip[Mark.Twain-Tom.Sawyer.txt-isal]               56.3420 (1.0)             96.9110 (1.0)             61.3174 (1.0)          3.0346 (1.0)             60.5350 (1.0)          2.3235 (1.0)       695;447  16,308.5874 (1.0)        5641           1
test_gzip[alice29.txt-cramjam]                       2,313.8000 (41.07)        2,937.7980 (30.31)        2,433.7892 (39.69)       59.4811 (19.60)        2,432.6420 (40.19)       64.1962 (27.63)       82;11     410.8819 (0.03)        391           1
test_gzip[alice29.txt-gzip]                          2,706.1680 (48.03)        3,048.9910 (31.46)        2,838.6805 (46.29)       55.2535 (18.21)        2,839.8855 (46.91)       86.1010 (37.06)       105;3     352.2763 (0.02)        318           1
test_gzip[alice29.txt-isal]                            638.2960 (11.33)          806.4610 (8.32)           679.3579 (11.08)       18.2648 (6.02)           672.1430 (11.10)       24.8000 (10.67)      162;21   1,471.9782 (0.09)       1232           1
test_gzip[asyoulik.txt-cramjam]                      2,026.6650 (35.97)        2,242.3150 (23.14)        2,136.0455 (34.84)       41.7069 (13.74)        2,143.3210 (35.41)       62.2349 (26.78)       159;0     468.1548 (0.03)        442           1
test_gzip[asyoulik.txt-gzip]                         2,365.9300 (41.99)        2,610.0570 (26.93)        2,472.7299 (40.33)       50.8724 (16.76)        2,469.8730 (40.80)       77.2808 (33.26)       152;0     404.4113 (0.02)        389           1
test_gzip[asyoulik.txt-isal]                           522.5980 (9.28)           629.7140 (6.50)           561.3238 (9.15)        14.8072 (4.88)           563.7030 (9.31)        23.9787 (10.32)       565;8   1,781.5029 (0.11)       1539           1
test_gzip[fifty-four-mb-random-cramjam]          1,201,188.9220 (>1000.0)  1,201,985.9070 (>1000.0)  1,201,683.9626 (>1000.0)    324.7614 (107.02)   1,201,663.1259 (>1000.0)    464.0428 (199.72)        1;0       0.8322 (0.00)          5           1
test_gzip[fifty-four-mb-random-gzip]               932,466.0240 (>1000.0)    943,055.2960 (>1000.0)    935,937.0100 (>1000.0)  4,266.8755 (>1000.0)    935,577.6800 (>1000.0)  4,915.4140 (>1000.0)       1;0       1.0684 (0.00)          5           1
test_gzip[fifty-four-mb-random-isal]               234,743.2040 (>1000.0)    244,949.3090 (>1000.0)    238,258.1728 (>1000.0)  4,771.0711 (>1000.0)    235,085.6990 (>1000.0)  7,732.6717 (>1000.0)       1;0       4.1971 (0.00)          5           1
test_gzip[fifty-four-mb-repeating-cramjam]         127,197.2460 (>1000.0)    131,845.0370 (>1000.0)    129,026.2873 (>1000.0)  1,373.3095 (452.56)     128,740.0955 (>1000.0)  1,293.4150 (556.66)        2;1       7.7504 (0.00)          8           1
test_gzip[fifty-four-mb-repeating-gzip]            138,558.1080 (>1000.0)    139,676.8730 (>1000.0)    139,084.2045 (>1000.0)    363.8480 (119.90)     138,942.7640 (>1000.0)    494.4985 (212.82)        2;0       7.1899 (0.00)          8           1
test_gzip[fifty-four-mb-repeating-isal]            125,758.5940 (>1000.0)    131,267.6750 (>1000.0)    126,878.8431 (>1000.0)  1,794.9466 (591.50)     126,403.0415 (>1000.0)    459.1045 (197.59)        1;1       7.8815 (0.00)          8           1
test_gzip[fireworks.jpeg-cramjam]                    2,341.1080 (41.55)        2,564.7090 (26.46)        2,424.7822 (39.54)       27.0851 (8.93)         2,421.5450 (40.00)       33.1180 (14.25)      112;10     412.4082 (0.03)        402           1
test_gzip[fireworks.jpeg-gzip]                       1,939.3610 (34.42)        2,444.8310 (25.23)        2,054.7399 (33.51)       40.8401 (13.46)        2,049.9410 (33.86)       39.9370 (17.19)       98;17     486.6796 (0.03)        486           1
test_gzip[fireworks.jpeg-isal]                         547.6960 (9.72)           694.9480 (7.17)           580.3026 (9.46)        16.9312 (5.58)           578.8780 (9.56)        13.2060 (5.68)      342;116   1,723.2388 (0.11)       1322           1
test_gzip[geo.protodata-cramjam]                       554.1570 (9.84)           689.9020 (7.12)           611.5268 (9.97)        14.4540 (4.76)           610.2805 (10.08)       18.6529 (8.03)       425;31   1,635.2512 (0.10)       1458           1
test_gzip[geo.protodata-gzip]                          618.9990 (10.99)          858.5870 (8.86)           668.6560 (10.90)       19.7861 (6.52)           665.6425 (11.00)       20.9209 (9.00)       293;40   1,495.5374 (0.09)       1378           1
test_gzip[geo.protodata-isal]                          324.7680 (5.76)           437.8440 (4.52)           352.5797 (5.75)        13.1510 (4.33)           351.2230 (5.80)        15.8620 (6.83)       682;54   2,836.2380 (0.17)       2153           1
test_gzip[html-cramjam]                                583.1560 (10.35)          694.8190 (7.17)           633.9863 (10.34)       12.3700 (4.08)           633.2520 (10.46)       16.5625 (7.13)       429;16   1,577.3210 (0.10)       1415           1
test_gzip[html-gzip]                                   653.0910 (11.59)          809.4880 (8.35)           702.4723 (11.46)       17.4990 (5.77)           699.6340 (11.56)       21.0187 (9.05)       299;36   1,423.5437 (0.09)       1259           1
test_gzip[html-isal]                                   303.0450 (5.38)           385.6550 (3.98)           317.2875 (5.17)         9.4691 (3.12)           314.4690 (5.19)        14.0475 (6.05)       512;25   3,151.7153 (0.19)       2468           1
test_gzip[html_x_4-cramjam]                          2,754.2840 (48.89)        3,373.9300 (34.81)        2,939.6069 (47.94)       63.6499 (20.98)        2,938.6090 (48.54)       78.8350 (33.93)        91;5     340.1815 (0.02)        353           1
test_gzip[html_x_4-gzip]                             2,754.2509 (48.88)        3,352.4790 (34.59)        2,892.2294 (47.17)       62.8312 (20.71)        2,886.7505 (47.69)       61.3300 (26.40)        56;7     345.7540 (0.02)        312           1
test_gzip[html_x_4-isal]                             1,317.6360 (23.39)        2,061.8500 (21.28)        1,410.1803 (23.00)       75.0201 (24.72)        1,400.9050 (23.14)       52.8415 (22.74)       19;11     709.1292 (0.04)        636           1
test_gzip[kppkn.gtb-cramjam]                         2,275.8590 (40.39)        2,514.6150 (25.95)        2,398.5709 (39.12)       37.1257 (12.23)        2,404.1010 (39.71)       46.5598 (20.04)       119;8     416.9149 (0.03)        405           1
test_gzip[kppkn.gtb-gzip]                            3,551.6540 (63.04)        3,999.0250 (41.26)        3,730.4021 (60.84)       67.8474 (22.36)        3,732.6630 (61.66)       87.0247 (37.45)        61;8     268.0676 (0.02)        259           1
test_gzip[kppkn.gtb-isal]                              641.7160 (11.39)          762.3440 (7.87)           664.1646 (10.83)       18.2096 (6.00)           662.0445 (10.94)       27.2695 (11.74)      270;22   1,505.6510 (0.09)       1220           1
test_gzip[lcet10.txt-cramjam]                        6,604.3800 (117.22)       7,548.9010 (77.90)        6,831.8665 (111.42)     112.5211 (37.08)        6,828.3140 (112.80)     118.4703 (50.99)        26;3     146.3729 (0.01)        143           1
test_gzip[lcet10.txt-gzip]                           7,557.3400 (134.13)       8,364.3020 (86.31)        7,843.2906 (127.91)     143.6573 (47.34)        7,807.8630 (128.98)     162.8670 (70.09)        31;7     127.4975 (0.01)        126           1
test_gzip[lcet10.txt-isal]                           2,029.7630 (36.03)        2,855.2201 (29.46)        2,104.7033 (34.32)       75.4166 (24.85)        2,093.5150 (34.58)       58.7273 (25.28)        11;7     475.1263 (0.03)        437           1
test_gzip[paper-100k.pdf-cramjam]                    1,679.8870 (29.82)        1,860.9260 (19.20)        1,748.9655 (28.52)       23.1390 (7.63)         1,747.3105 (28.86)       26.9440 (11.60)      151;14     571.7666 (0.04)        552           1
test_gzip[paper-100k.pdf-gzip]                       1,690.6669 (30.01)        2,068.6880 (21.35)        1,783.8001 (29.09)       29.9959 (9.88)         1,782.6200 (29.45)       32.8138 (14.12)      132;16     560.6009 (0.03)        567           1
test_gzip[paper-100k.pdf-isal]                         416.8270 (7.40)           578.1950 (5.97)           448.2110 (7.31)        16.6530 (5.49)           441.5740 (7.29)        16.5207 (7.11)       313;94   2,231.0920 (0.14)       1761           1
test_gzip[plrabn12.txt-cramjam]                      8,900.4810 (157.97)       9,788.2350 (101.00)       9,164.0241 (149.45)     114.8119 (37.83)        9,169.1470 (151.47)     123.6400 (53.21)        25;2     109.1224 (0.01)        103           1
test_gzip[plrabn12.txt-gzip]                        10,147.7650 (180.11)      11,477.3950 (118.43)      10,481.0197 (170.93)     170.7084 (56.25)       10,461.9600 (172.82)     214.5829 (92.35)        23;1      95.4106 (0.01)         90           1
test_gzip[plrabn12.txt-isal]                         2,288.2160 (40.61)        3,318.0070 (34.24)        2,387.4327 (38.94)       89.3594 (29.45)        2,388.3040 (39.45)       65.8750 (28.35)        14;5     418.8600 (0.03)        394           1
test_gzip[urls.10K-cramjam]                         10,137.8160 (179.93)      11,621.6400 (119.92)      10,464.5132 (170.66)     175.7993 (57.93)       10,423.2070 (172.18)     154.5588 (66.52)        13;4      95.5611 (0.01)         91           1
test_gzip[urls.10K-gzip]                             9,798.4010 (173.91)      10,841.0180 (111.87)      10,145.5872 (165.46)     202.4173 (66.70)       10,114.5900 (167.09)     247.6572 (106.59)       30;3      98.5650 (0.01)        103           1
test_gzip[urls.10K-isal]                             3,105.6050 (55.12)        4,504.9600 (46.49)        3,276.1175 (53.43)      119.1319 (39.26)        3,268.3510 (53.99)       58.6328 (25.23)       28;28     305.2394 (0.02)        289           1
--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
```

#### Snappy

There are two types here, `framed` and `raw`; the recommended one being snappy's `framed` format.

`make bench-snappy-framed`

```bash
------------------------------------------------------------------------------------------------------------------------ benchmark: 28 tests ------------------------------------------------------------------------------------------------------------------------
Name (time in us)                                                    Min                       Max                      Mean                  StdDev                    Median                     IQR            Outliers          OPS            Rounds  Iterations
---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
test_snappy_framed[Mark.Twain-Tom.Sawyer.txt-cramjam]            34.5460 (1.0)             64.0360 (1.0)             39.3083 (1.0)            2.1469 (1.35)            38.9790 (1.0)            2.6720 (7.07)     1965;171  25,439.9180 (1.0)        7372           1
test_snappy_framed[Mark.Twain-Tom.Sawyer.txt-snappy]             74.2590 (2.15)           105.3040 (1.64)            75.1430 (1.91)           1.5901 (1.0)             74.7080 (1.92)           0.4141 (1.10)      361;933  13,307.9565 (0.52)       6986           1
test_snappy_framed[alice29.txt-cramjam]                         435.8150 (12.62)          523.1360 (8.17)           457.6633 (11.64)         12.5979 (7.92)           452.4115 (11.61)         19.1250 (50.59)      363;19   2,185.0126 (0.09)       1784           1
test_snappy_framed[alice29.txt-snappy]                          927.7140 (26.85)        1,456.8070 (22.75)          980.6164 (24.95)         45.5170 (28.62)          968.9630 (24.86)         66.2975 (175.36)      273;6   1,019.7667 (0.04)        905           1
test_snappy_framed[asyoulik.txt-cramjam]                        379.6710 (10.99)          473.0270 (7.39)           414.8673 (10.55)         12.1976 (7.67)           414.3355 (10.63)         13.0820 (34.60)      711;65   2,410.4092 (0.09)       2234           1
test_snappy_framed[asyoulik.txt-snappy]                         782.9360 (22.66)          938.6060 (14.66)          807.4152 (20.54)         22.1008 (13.90)          800.8410 (20.55)         28.6260 (75.72)      207;26   1,238.5201 (0.05)       1061           1
test_snappy_framed[fifty-four-mb-random-cramjam]             50,448.9600 (>1000.0)     60,230.9990 (940.58)      53,169.9146 (>1000.0)    3,170.3685 (>1000.0)     51,968.2390 (>1000.0)    1,977.7771 (>1000.0)       3;3      18.8076 (0.00)         18           1
test_snappy_framed[fifty-four-mb-random-snappy]           3,164,490.5070 (>1000.0)  5,119,895.9580 (>1000.0)  4,685,685.8194 (>1000.0)  850,989.5239 (>1000.0)  5,047,832.6710 (>1000.0)  507,628.8615 (>1000.0)       1;1       0.2134 (0.00)          5           1
test_snappy_framed[fifty-four-mb-repeating-cramjam]          36,469.7640 (>1000.0)     53,657.1800 (837.92)      41,827.1772 (>1000.0)    6,810.1733 (>1000.0)     37,215.6450 (954.76)    12,874.1443 (>1000.0)       7;0      23.9079 (0.00)         25           1
test_snappy_framed[fifty-four-mb-repeating-snappy]          272,525.3050 (>1000.0)    319,697.7690 (>1000.0)    295,036.7576 (>1000.0)   20,053.2981 (>1000.0)    291,268.1390 (>1000.0)   34,984.8552 (>1000.0)       2;0       3.3894 (0.00)          5           1
test_snappy_framed[fireworks.jpeg-cramjam]                       43.7250 (1.27)            82.4920 (1.29)            45.3223 (1.15)           2.3333 (1.47)            44.6160 (1.14)           0.3781 (1.0)     1459;2641  22,064.2048 (0.87)      13467           1
test_snappy_framed[fireworks.jpeg-snappy]                       418.9740 (12.13)          495.5910 (7.74)           422.4280 (10.75)          7.1744 (4.51)           420.3495 (10.78)          1.8170 (4.81)      172;298   2,367.2674 (0.09)       1964           1
test_snappy_framed[geo.protodata-cramjam]                       106.2290 (3.08)           156.0030 (2.44)           114.7162 (2.92)           5.6912 (3.58)           113.2460 (2.91)           5.6307 (14.89)    1898;293   8,717.1625 (0.34)       6317           1
test_snappy_framed[geo.protodata-snappy]                        459.6560 (13.31)          712.6420 (11.13)          490.9161 (12.49)         28.9804 (18.23)          480.5300 (12.33)         44.3642 (117.35)     417;16   2,037.0079 (0.08)       1761           1
test_snappy_framed[html-cramjam]                                114.4570 (3.31)           330.9220 (5.17)           124.1903 (3.16)           7.5444 (4.74)           122.0535 (3.13)           6.0660 (16.05)     767;233   8,052.1600 (0.32)       5358           1
test_snappy_framed[html-snappy]                                 419.1740 (12.13)          518.9120 (8.10)           440.4698 (11.21)         19.0325 (11.97)          432.7900 (11.10)         23.5960 (62.41)      425;63   2,270.3031 (0.09)       1874           1
test_snappy_framed[html_x_4-cramjam]                            498.1220 (14.42)          626.0840 (9.78)           538.0052 (13.69)         17.7006 (11.13)          533.8565 (13.70)         22.1410 (58.56)      294;39   1,858.7179 (0.07)       1406           1
test_snappy_framed[html_x_4-snappy]                           1,774.0160 (51.35)        2,062.4400 (32.21)        1,799.3159 (45.77)         30.9255 (19.45)        1,790.9070 (45.95)         20.1712 (53.35)       37;36     555.7668 (0.02)        463           1
test_snappy_framed[kppkn.gtb-cramjam]                           363.9480 (10.54)          445.5770 (6.96)           386.9350 (9.84)          12.8919 (8.11)           383.9880 (9.85)          13.1481 (34.78)      581;84   2,584.4136 (0.10)       1986           1
test_snappy_framed[kppkn.gtb-snappy]                            948.8640 (27.47)        1,818.3100 (28.40)          991.6895 (25.23)         69.9524 (43.99)          964.2670 (24.74)         58.9637 (155.96)      98;38   1,008.3801 (0.04)        851           1
test_snappy_framed[lcet10.txt-cramjam]                        1,211.4980 (35.07)        1,411.0670 (22.04)        1,266.7555 (32.23)         32.8859 (20.68)        1,269.7810 (32.58)         43.5995 (115.32)     201;22     789.4183 (0.03)        712           1
test_snappy_framed[lcet10.txt-snappy]                         2,746.5900 (79.51)        3,359.2020 (52.46)        2,815.2247 (71.62)         72.5785 (45.64)        2,801.3160 (71.87)         55.8608 (147.76)       16;9     355.2114 (0.01)        323           1
test_snappy_framed[paper-100k.pdf-cramjam]                       40.7860 (1.18)            71.1230 (1.11)            42.7810 (1.09)           2.1702 (1.36)            41.8275 (1.07)           1.8580 (4.91)     2532;810  23,374.8378 (0.92)      13638           1
test_snappy_framed[paper-100k.pdf-snappy]                       355.6370 (10.29)          417.0000 (6.51)           363.4989 (9.25)          11.6659 (7.34)           357.4060 (9.17)          11.1997 (29.62)     358;220   2,751.0397 (0.11)       2321           1
test_snappy_framed[plrabn12.txt-cramjam]                      1,619.8560 (46.89)        2,237.4490 (34.94)        1,774.7065 (45.15)         73.1317 (45.99)        1,758.2970 (45.11)         91.5220 (242.08)     145;11     563.4735 (0.02)        521           1
test_snappy_framed[plrabn12.txt-snappy]                       3,366.8110 (97.46)        4,179.6740 (65.27)        3,670.3920 (93.37)        221.7947 (139.48)       3,587.1925 (92.03)        398.7299 (>1000.0)      98;0     272.4505 (0.01)        246           1
test_snappy_framed[urls.10K-cramjam]                          1,572.6960 (45.52)        2,352.0380 (36.73)        1,782.1609 (45.34)        168.3364 (105.86)       1,712.2195 (43.93)        103.8710 (274.75)    109;108     561.1166 (0.02)        566           1
test_snappy_framed[urls.10K-snappy]                           3,668.1250 (106.18)       4,344.0750 (67.84)        3,754.5744 (95.52)        134.4371 (84.54)        3,698.2595 (94.88)         68.2920 (180.64)      30;33     266.3418 (0.01)        240           1
---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
```

`make bench-snappy-raw`

```bash
--------------------------------------------------------------------------------------------------------------- benchmark: 28 tests ----------------------------------------------------------------------------------------------------------------
Name (time in us)                                              Min                    Max                   Mean                StdDev                 Median                    IQR            Outliers           OPS            Rounds  Iterations
----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
test_snappy_raw[Mark.Twain-Tom.Sawyer.txt-cramjam]         28.0010 (4.79)         60.4390 (1.73)         31.9080 (4.53)         1.8590 (3.44)         31.7450 (4.87)          2.4550 (16.48)    3749;131   31,340.0822 (0.22)      12606           1
test_snappy_raw[Mark.Twain-Tom.Sawyer.txt-snappy]          26.9300 (4.61)         76.8340 (2.21)         28.9253 (4.11)         0.9898 (1.83)         28.7750 (4.41)          0.3340 (2.24)    1566;2560   34,571.8473 (0.24)      16034           1
test_snappy_raw[alice29.txt-cramjam]                      401.1650 (68.65)       480.1380 (13.78)       422.9262 (60.03)       11.5861 (21.45)       422.4815 (64.78)        11.9550 (80.23)      554;58    2,364.4788 (0.02)       1808           1
test_snappy_raw[alice29.txt-snappy]                       445.0010 (76.15)       572.9450 (16.45)       468.7200 (66.53)       10.7167 (19.84)       469.0450 (71.92)        11.3955 (76.47)      481;46    2,133.4697 (0.02)       1843           1
test_snappy_raw[asyoulik.txt-cramjam]                     357.2120 (61.12)       435.3620 (12.50)       377.4845 (53.58)        7.6349 (14.13)       377.0095 (57.81)         5.6040 (37.61)     466;346    2,649.1153 (0.02)       2260           1
test_snappy_raw[asyoulik.txt-snappy]                      396.0270 (67.77)       611.7280 (17.56)       419.0061 (59.47)       14.0578 (26.02)       417.9780 (64.09)        10.7302 (72.01)      222;67    2,386.5999 (0.02)       2161           1
test_snappy_raw[fifty-four-mb-random-cramjam]          35,430.8690 (>1000.0)  45,847.4250 (>1000.0)  37,358.7421 (>1000.0)  2,664.1931 (>1000.0)  36,391.3140 (>1000.0)   1,248.2977 (>1000.0)       3;3       26.7675 (0.00)         27           1
test_snappy_raw[fifty-four-mb-random-snappy]           31,321.3560 (>1000.0)  35,562.3420 (>1000.0)  33,133.2397 (>1000.0)  1,605.7256 (>1000.0)  32,645.6880 (>1000.0)   3,162.5080 (>1000.0)      14;0       30.1812 (0.00)         31           1
test_snappy_raw[fifty-four-mb-repeating-cramjam]       21,230.4320 (>1000.0)  35,662.5360 (>1000.0)  24,200.3345 (>1000.0)  4,277.1733 (>1000.0)  21,868.5330 (>1000.0)   5,020.0773 (>1000.0)       8;3       41.3217 (0.00)         43           1
test_snappy_raw[fifty-four-mb-repeating-snappy]        23,444.9620 (>1000.0)  43,529.4480 (>1000.0)  30,842.1981 (>1000.0)  6,956.1883 (>1000.0)  29,924.6840 (>1000.0)  12,560.6345 (>1000.0)      13;0       32.4231 (0.00)         35           1
test_snappy_raw[fireworks.jpeg-cramjam]                    10.1300 (1.73)         48.0650 (1.38)         10.9993 (1.56)         0.8872 (1.64)         10.8080 (1.66)          0.4860 (3.26)    1579;1765   90,915.0799 (0.64)      29916           1
test_snappy_raw[fireworks.jpeg-snappy]                      5.8440 (1.0)          34.8380 (1.0)           7.0455 (1.0)          1.9740 (3.65)          6.5220 (1.0)           0.2473 (1.66)    2521;4361  141,934.0808 (1.0)       36113           1
test_snappy_raw[geo.protodata-cramjam]                     78.4250 (13.42)       137.9500 (3.96)         86.2121 (12.24)        5.1533 (9.54)         84.6650 (12.98)         5.1345 (34.46)    1254;302   11,599.2970 (0.08)       5657           1
test_snappy_raw[geo.protodata-snappy]                      71.2520 (12.19)       117.4700 (3.37)         80.4148 (11.41)        4.3536 (8.06)         79.7810 (12.23)         6.0615 (40.68)     2014;77   12,435.5282 (0.09)       6831           1
test_snappy_raw[html-cramjam]                              89.5510 (15.32)       149.8580 (4.30)         98.8037 (14.02)        4.8620 (9.00)         97.5125 (14.95)         5.1255 (34.40)    1294;283   10,121.0744 (0.07)       5724           1
test_snappy_raw[html-snappy]                               85.5240 (14.63)       149.3820 (4.29)         94.9606 (13.48)        4.8372 (8.95)         94.4030 (14.47)         7.1725 (48.13)     2195;35   10,530.6790 (0.07)       6472           1
test_snappy_raw[html_x_4-cramjam]                         405.9860 (69.47)       702.9710 (20.18)       454.3526 (64.49)       34.8446 (64.50)       448.2530 (68.73)        21.3510 (143.28)      79;60    2,200.9337 (0.02)       1462           1
test_snappy_raw[html_x_4-snappy]                          422.9740 (72.38)       647.5400 (18.59)       464.3638 (65.91)       33.2042 (61.46)       455.7550 (69.88)        19.2223 (129.00)    160;114    2,153.4839 (0.02)       1561           1
test_snappy_raw[kppkn.gtb-cramjam]                        325.1210 (55.63)       426.9950 (12.26)       351.9489 (49.95)       16.5860 (30.70)       349.4085 (53.57)        26.1700 (175.62)     767;11    2,841.3220 (0.02)       1994           1
test_snappy_raw[kppkn.gtb-snappy]                         344.0450 (58.87)       425.1800 (12.20)       370.7409 (52.62)       10.3033 (19.07)       368.7160 (56.53)        12.7380 (85.48)      465;60    2,697.3012 (0.02)       2200           1
test_snappy_raw[lcet10.txt-cramjam]                     1,095.6750 (187.49)    1,362.0870 (39.10)     1,189.0433 (168.77)      41.1628 (76.20)     1,179.6520 (180.87)       56.2050 (377.19)     172;17      841.0123 (0.01)        754           1
test_snappy_raw[lcet10.txt-snappy]                      1,206.4470 (206.44)    1,584.6880 (45.49)     1,297.6861 (184.19)      47.3392 (87.63)     1,289.8420 (197.77)       38.5145 (258.47)     106;36      770.6024 (0.01)        704           1
test_snappy_raw[paper-100k.pdf-cramjam]                    12.2860 (2.10)         44.2950 (1.27)         13.1896 (1.87)         0.8487 (1.57)         13.0200 (2.00)          0.1490 (1.0)     1002;3367   75,817.5069 (0.53)      11359           1
test_snappy_raw[paper-100k.pdf-snappy]                     10.2890 (1.76)         35.7210 (1.03)         11.0991 (1.58)         0.5402 (1.0)          11.0640 (1.70)          0.3180 (2.13)    1434;1328   90,097.6017 (0.63)      27306           1
test_snappy_raw[plrabn12.txt-cramjam]                   1,512.8650 (258.88)    1,881.1250 (54.00)     1,609.2872 (228.41)      48.7215 (90.19)     1,591.1230 (243.96)       37.5485 (251.98)      66;46      621.3931 (0.00)        525           1
test_snappy_raw[plrabn12.txt-snappy]                    1,716.5850 (293.74)    2,070.5130 (59.43)     1,804.1379 (256.07)      68.1815 (126.21)    1,799.4450 (275.90)       57.6188 (386.67)      96;34      554.2814 (0.00)        499           1
test_snappy_raw[urls.10K-cramjam]                       1,357.1110 (232.22)    2,421.0440 (69.49)     1,634.3708 (231.97)     283.8191 (525.37)    1,483.5370 (227.47)      277.4218 (>1000.0)    120;68      611.8563 (0.00)        543           1
test_snappy_raw[urls.10K-snappy]                        1,333.5560 (228.19)    2,160.5230 (62.02)     1,579.1927 (224.14)     226.8612 (419.94)    1,461.6870 (224.12)      342.2740 (>1000.0)     138;0      633.2350 (0.00)        569           1
----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
```


##### Benchmarks for using `compress_into` and `decompress_into`
All variants implement a `compress_into` and `decompress_into`
This is where, if you have a Python numpy array pre-allocated to the size of the
de/compress data output, we can place the bytes directly into that buffer.

Again, since basically no variants implement similar functionality as we saw in the Gzip `output_len`
benchmarks, this benchmark is specific to `cramjam`

```bash
---------------------------------------------------------------------------------------------------------------------------- benchmark: 28 tests ----------------------------------------------------------------------------------------------------------------------------
Name (time in us)                                                                           Min                    Max                   Mean              StdDev                 Median                   IQR            Outliers          OPS            Rounds  Iterations
-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
test_cramjam_snappy_de_compress_into[Mark.Twain-Tom.Sawyer.txt-compress_into]           18.0310 (1.58)         54.7750 (1.18)         19.6129 (1.63)       1.4020 (2.01)         19.2360 (1.61)         1.0880 (8.00)    3383;1527  50,986.9037 (0.61)      23492           1
test_cramjam_snappy_de_compress_into[Mark.Twain-Tom.Sawyer.txt-decompress_into]         11.3850 (1.0)          50.0580 (1.08)         12.0015 (1.0)        0.6985 (1.0)          11.9760 (1.0)          0.5119 (3.76)    1709;1543  83,323.1549 (1.0)       20060           1
test_cramjam_snappy_de_compress_into[alice29.txt-compress_into]                        311.2630 (27.34)       369.1700 (7.97)        326.1066 (27.17)      9.4492 (13.53)       327.5030 (27.35)       13.9962 (102.89)    1094;44   3,066.4816 (0.04)       2891           1
test_cramjam_snappy_de_compress_into[alice29.txt-decompress_into]                      120.2330 (10.56)       192.3420 (4.15)        129.4531 (10.79)      5.1066 (7.31)        128.6950 (10.75)        5.0130 (36.85)    1505;261   7,724.8060 (0.09)       5758           1
test_cramjam_snappy_de_compress_into[asyoulik.txt-compress_into]                       271.2750 (23.83)       345.7670 (7.46)        289.7546 (24.14)      8.3428 (11.94)       290.9890 (24.30)       12.1738 (89.49)     1113;29   3,451.1967 (0.04)       3051           1
test_cramjam_snappy_de_compress_into[asyoulik.txt-decompress_into]                     107.4690 (9.44)        147.8530 (3.19)        113.0167 (9.42)       3.3989 (4.87)        113.0510 (9.44)         1.2400 (9.12)    2164;2575   8,848.2504 (0.11)       7605           1
test_cramjam_snappy_de_compress_into[fifty-four-mb-random-compress_into]            13,065.9060 (>1000.0)  13,720.7850 (296.08)   13,263.4344 (>1000.0)  137.4746 (196.80)   13,245.1840 (>1000.0)    180.3048 (>1000.0)      12;1      75.3953 (0.00)         41           1
test_cramjam_snappy_de_compress_into[fifty-four-mb-random-decompress_into]          10,884.4140 (956.03)   11,502.9810 (248.22)   11,023.9206 (918.55)   126.7868 (181.50)   10,993.2940 (917.94)     133.0880 (978.36)        7;2      90.7118 (0.00)         47           1
test_cramjam_snappy_de_compress_into[fifty-four-mb-repeating-compress_into]          9,793.2480 (860.19)   11,932.1500 (257.48)   10,499.6366 (874.86)   565.2263 (809.15)   10,233.8120 (854.53)   1,006.7380 (>1000.0)      34;0      95.2414 (0.00)         97           1
test_cramjam_snappy_de_compress_into[fifty-four-mb-repeating-decompress_into]       13,449.1500 (>1000.0)  13,798.8920 (297.76)   13,611.7777 (>1000.0)  100.3298 (143.63)   13,615.1300 (>1000.0)    173.1542 (>1000.0)      14;0      73.4658 (0.00)         37           1
test_cramjam_snappy_de_compress_into[fireworks.jpeg-compress_into]                      24.6690 (2.17)         60.5580 (1.31)         25.6230 (2.13)       1.2445 (1.78)         25.0670 (2.09)         1.0669 (7.84)    2634;1415  39,027.4796 (0.47)      29802           1
test_cramjam_snappy_de_compress_into[fireworks.jpeg-decompress_into]                    19.3420 (1.70)         46.3420 (1.0)          19.7981 (1.65)       0.8747 (1.25)         19.5900 (1.64)         0.1360 (1.0)     1500;2099  50,509.9939 (0.61)      19883           1
test_cramjam_snappy_de_compress_into[geo.protodata-compress_into]                       55.2620 (4.85)         97.6310 (2.11)         61.9433 (5.16)       3.1557 (4.52)         61.4180 (5.13)         3.8687 (28.44)    2492;238  16,143.8077 (0.19)      10229           1
test_cramjam_snappy_de_compress_into[geo.protodata-decompress_into]                     32.0750 (2.82)         66.2310 (1.43)         37.0492 (3.09)       3.3861 (4.85)         36.6340 (3.06)         4.0270 (29.60)    3967;490  26,991.1105 (0.32)      15932           1
test_cramjam_snappy_de_compress_into[html-compress_into]                                62.6890 (5.51)        111.9900 (2.42)         69.1820 (5.76)       3.4201 (4.90)         68.2530 (5.70)         4.1485 (30.50)    1793;260  14,454.6235 (0.17)       9785           1
test_cramjam_snappy_de_compress_into[html-decompress_into]                              31.9190 (2.80)         64.9500 (1.40)         33.0087 (2.75)       1.6430 (2.35)         32.2055 (2.69)         1.3650 (10.03)    1190;580  30,295.0795 (0.36)      12000           1
test_cramjam_snappy_de_compress_into[html_x_4-compress_into]                           304.8710 (26.78)       386.4630 (8.34)        327.8872 (27.32)     10.1213 (14.49)       324.7320 (27.12)        8.2359 (60.54)     505;312   3,049.8294 (0.04)       2897           1
test_cramjam_snappy_de_compress_into[html_x_4-decompress_into]                         146.0860 (12.83)       205.1390 (4.43)        155.8785 (12.99)      6.1026 (8.74)        154.2720 (12.88)        4.9608 (36.47)     899;264   6,415.2519 (0.08)       3619           1
test_cramjam_snappy_de_compress_into[kppkn.gtb-compress_into]                          233.1950 (20.48)       381.8900 (8.24)        248.9368 (20.74)     11.7801 (16.86)       245.4930 (20.50)       10.9696 (80.64)     819;173   4,017.0835 (0.05)       3484           1
test_cramjam_snappy_de_compress_into[kppkn.gtb-decompress_into]                        116.4330 (10.23)       163.3600 (3.53)        124.6151 (10.38)      4.9459 (7.08)        123.5830 (10.32)        3.6220 (26.63)    1697;640   8,024.7094 (0.10)       6924           1
test_cramjam_snappy_de_compress_into[lcet10.txt-compress_into]                         846.2590 (74.33)       975.1170 (21.04)       881.7123 (73.47)     21.7002 (31.06)       887.3200 (74.09)       37.6160 (276.52)      481;4   1,134.1568 (0.01)       1116           1
test_cramjam_snappy_de_compress_into[lcet10.txt-decompress_into]                       320.4570 (28.15)       516.7530 (11.15)       347.3935 (28.95)      9.6587 (13.83)       348.0740 (29.06)        3.3760 (24.82)     572;656   2,878.5801 (0.03)       2186           1
test_cramjam_snappy_de_compress_into[paper-100k.pdf-compress_into]                      22.2520 (1.95)         53.6130 (1.16)         23.0841 (1.92)       1.1055 (1.58)         22.5690 (1.88)         0.9760 (7.18)    2918;1451  43,319.7696 (0.52)      30512           1
test_cramjam_snappy_de_compress_into[paper-100k.pdf-decompress_into]                    18.8530 (1.66)         74.9160 (1.62)         19.4056 (1.62)       1.0193 (1.46)         19.0591 (1.59)         0.7590 (5.58)    2581;1475  51,531.4622 (0.62)      35150           1
test_cramjam_snappy_de_compress_into[plrabn12.txt-compress_into]                     1,159.5470 (101.85)    1,441.0510 (31.10)     1,224.4767 (102.03)    34.4923 (49.38)     1,221.3525 (101.98)      19.4960 (143.32)    182;167     816.6754 (0.01)        798           1
test_cramjam_snappy_de_compress_into[plrabn12.txt-decompress_into]                     446.9480 (39.26)       745.6270 (16.09)       491.7309 (40.97)     15.4399 (22.10)       490.1670 (40.93)       13.5438 (99.56)      227;97   2,033.6325 (0.02)       2083           1
test_cramjam_snappy_de_compress_into[urls.10K-compress_into]                           974.2860 (85.58)     1,125.5020 (24.29)     1,020.5675 (85.04)     27.0013 (38.65)     1,022.8860 (85.41)       37.7820 (277.74)     313;10     979.8470 (0.01)        918           1
test_cramjam_snappy_de_compress_into[urls.10K-decompress_into]                         412.4130 (36.22)       531.0850 (11.46)       426.7128 (35.56)     13.3330 (19.09)       423.2610 (35.34)       18.8139 (138.31)     152;29   2,343.4964 (0.03)       1548           1
-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
```

#### Lz4

`make bench-lz4`

```bash
----------------------------------------------------------------------------------------------------------------- benchmark: 56 tests ------------------------------------------------------------------------------------------------------------------
Name (time in us)                                                 Min                     Max                    Mean                StdDev                  Median                   IQR            Outliers          OPS            Rounds  Iterations
--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
test_lz4[Mark.Twain-Tom.Sawyer.txt-cramjam]                  125.9440 (11.01)        201.5990 (6.41)         138.2460 (10.93)        7.6495 (12.00)        136.9320 (10.89)       11.0935 (33.72)      831;30   7,233.4810 (0.09)       2817           1
test_lz4[Mark.Twain-Tom.Sawyer.txt-python-lz4]               107.1450 (9.37)         167.3470 (5.32)         118.1501 (9.34)         5.5486 (8.70)         116.6485 (9.28)         6.7630 (20.56)    1466;118   8,463.8073 (0.11)       5622           1
test_lz4[alice29.txt-cramjam]                              1,885.8810 (164.86)     2,087.5580 (66.39)      1,982.5844 (156.69)      37.5861 (58.96)      1,978.7290 (157.40)      48.9570 (148.81)      137;5     504.3921 (0.01)        433           1
test_lz4[alice29.txt-python-lz4]                           1,983.3180 (173.38)     2,214.2810 (70.42)      2,082.8597 (164.62)      33.4302 (52.44)      2,085.7810 (165.92)      48.5615 (147.61)      151;2     480.1091 (0.01)        468           1
test_lz4[asyoulik.txt-cramjam]                             1,594.3140 (139.38)     1,802.8770 (57.34)      1,696.2555 (134.06)      30.8526 (48.40)      1,693.4170 (134.71)      41.4690 (126.05)      166;6     589.5338 (0.01)        538           1
test_lz4[asyoulik.txt-python-lz4]                          1,727.6750 (151.03)     1,930.0050 (61.38)      1,804.6468 (142.63)      34.2418 (53.72)      1,802.1930 (143.36)      50.4698 (153.41)      188;1     554.1251 (0.01)        521           1
test_lz4[fifty-four-mb-random-cramjam]                   469,464.7980 (>1000.0)  478,719.4140 (>1000.0)  472,864.8178 (>1000.0)  3,619.3578 (>1000.0)  471,932.3960 (>1000.0)  4,572.3082 (>1000.0)       1;0       2.1148 (0.00)          5           1
test_lz4[fifty-four-mb-random-python-lz4]                921,533.9190 (>1000.0)  943,469.9070 (>1000.0)  927,403.4628 (>1000.0)  9,325.9997 (>1000.0)  922,199.7440 (>1000.0)  9,696.0720 (>1000.0)       1;0       1.0783 (0.00)          5           1
test_lz4[fifty-four-mb-repeating-cramjam]                 97,503.2970 (>1000.0)  100,140.5100 (>1000.0)   98,439.0015 (>1000.0)    762.2801 (>1000.0)   98,327.1300 (>1000.0)    717.2440 (>1000.0)       3;1      10.1586 (0.00)         10           1
test_lz4[fifty-four-mb-repeating-python-lz4]              79,055.6850 (>1000.0)   85,197.2530 (>1000.0)   81,777.3721 (>1000.0)  1,734.6742 (>1000.0)   81,735.7405 (>1000.0)  1,789.0415 (>1000.0)       4;1      12.2283 (0.00)         12           1
test_lz4[fireworks.jpeg-cramjam]                             881.6940 (77.08)      1,071.0420 (34.06)        936.1303 (73.99)       29.0636 (45.59)        932.5286 (74.18)       36.4660 (110.84)     271;15   1,068.2274 (0.01)        902           1
test_lz4[fireworks.jpeg-python-lz4]                        1,547.3600 (135.27)     1,932.0080 (61.45)      1,623.5895 (128.32)      36.0235 (56.51)      1,619.9770 (128.87)      38.5130 (117.06)     127;16     615.9192 (0.01)        584           1
test_lz4[geo.protodata-cramjam]                              412.7130 (36.08)        560.5040 (17.83)        456.0728 (36.05)       18.3228 (28.74)        453.0190 (36.04)       23.1310 (70.31)      487;48   2,192.6322 (0.03)       1783           1
test_lz4[geo.protodata-python-lz4]                           351.6870 (30.74)        440.7540 (14.02)        380.8956 (30.10)       10.7515 (16.87)        379.6055 (30.20)       13.9140 (42.29)      672;38   2,625.3914 (0.03)       2256           1
test_lz4[html-cramjam]                                       441.5590 (38.60)        722.5720 (22.98)        495.9693 (39.20)       21.9263 (34.40)        494.2480 (39.32)       28.4852 (86.58)      417;23   2,016.2540 (0.03)       1479           1
test_lz4[html-python-lz4]                                    404.3260 (35.35)        562.6990 (17.90)        436.2489 (34.48)       12.7749 (20.04)        434.8690 (34.59)       16.5522 (50.31)      668;21   2,292.2693 (0.03)       2023           1
test_lz4[html_x_4-cramjam]                                 2,036.0430 (177.99)     2,561.4560 (81.47)      2,217.6144 (175.27)     113.6074 (178.22)     2,175.9280 (173.09)     121.2330 (368.50)      96;24     450.9350 (0.01)        368           1
test_lz4[html_x_4-python-lz4]                              1,890.5030 (165.27)     2,539.0140 (80.75)      2,052.8446 (162.24)      92.0286 (144.37)     2,036.7405 (162.02)      60.5945 (184.18)      65;38     487.1289 (0.01)        468           1
test_lz4[kppkn.gtb-cramjam]                                2,279.1190 (199.24)     2,551.8590 (81.16)      2,425.6386 (191.71)      46.3382 (72.69)      2,426.7680 (193.05)      65.8842 (200.26)      120;3     412.2626 (0.01)        381           1
test_lz4[kppkn.gtb-python-lz4]                             2,223.8040 (194.41)     2,555.4450 (81.27)      2,371.9228 (187.46)      63.5021 (99.62)      2,361.3275 (187.84)      87.7710 (266.79)      146;3     421.5989 (0.01)        426           1
test_lz4[lcet10.txt-cramjam]                               5,326.5700 (465.65)     5,881.1520 (187.05)     5,502.5735 (434.89)      87.6411 (137.48)     5,506.4110 (438.03)      95.4195 (290.04)       61;3     181.7331 (0.00)        183           1
test_lz4[lcet10.txt-python-lz4]                            5,566.7540 (486.65)     6,082.4770 (193.45)     5,773.7842 (456.32)      79.5379 (124.77)     5,775.4660 (459.43)     116.3845 (353.76)       55;1     173.1966 (0.00)        175           1
test_lz4[paper-100k.pdf-cramjam]                             659.1950 (57.63)        984.2960 (31.31)        704.5409 (55.68)       18.3470 (28.78)        701.6630 (55.82)       20.1760 (61.33)      221;37   1,419.3641 (0.02)       1262           1
test_lz4[paper-100k.pdf-python-lz4]                        1,033.1420 (90.32)      1,149.5930 (36.56)      1,081.5924 (85.48)       19.6069 (30.76)      1,078.6270 (85.80)       26.7335 (81.26)      280;13     924.5627 (0.01)        889           1
test_lz4[plrabn12.txt-cramjam]                             7,277.7500 (636.22)     7,669.2110 (243.92)     7,540.0878 (595.92)      78.9117 (123.79)     7,559.9700 (601.38)      82.7795 (251.62)       31;6     132.6244 (0.00)        129           1
test_lz4[plrabn12.txt-python-lz4]                          7,743.8430 (676.97)     8,612.8020 (273.93)     8,105.4179 (640.60)     114.0547 (178.92)     8,119.9995 (645.93)      99.3470 (301.98)       27;8     123.3743 (0.00)        124           1
test_lz4[urls.10K-cramjam]                                 5,376.4820 (470.01)     6,053.2750 (192.52)     5,591.5178 (441.92)      91.2129 (143.09)     5,572.0190 (443.24)     107.4870 (326.72)       43;2     178.8423 (0.00)        169           1
test_lz4[urls.10K-python-lz4]                              6,001.4510 (524.65)     6,624.5310 (210.69)     6,179.6182 (488.40)      81.2789 (127.50)     6,173.7885 (491.11)      55.5270 (168.78)      24;15     161.8223 (0.00)        154           1
--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
```

#### LZ4 _block_ format

`make bench-lz4-block`

```bash
----------------------------------------------------------------------------------------------------------------- benchmark: 56 tests ------------------------------------------------------------------------------------------------------------------
Name (time in us)                                                 Min                     Max                    Mean                StdDev                  Median                   IQR            Outliers          OPS            Rounds  Iterations
--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
test_lz4_block[Mark.Twain-Tom.Sawyer.txt-cramjam]             19.0900 (1.67)          51.9429 (1.65)          21.0092 (1.66)         1.1547 (1.81)          20.9060 (1.66)         1.2360 (3.76)     3358;490  47,598.2661 (0.60)      19360           1
test_lz4_block[Mark.Twain-Tom.Sawyer.txt-python-lz4]          16.7090 (1.46)          31.4420 (1.0)           17.9088 (1.42)         0.6375 (1.0)           17.8340 (1.42)         0.5030 (1.53)     2304;872  55,838.5284 (0.71)      17721           1
test_lz4_block[alice29.txt-cramjam]                          319.3180 (27.91)        381.6720 (12.14)        336.4830 (26.59)        6.0062 (9.42)         335.2750 (26.67)        5.5785 (16.96)     600;165   2,971.9183 (0.04)       2571           1
test_lz4_block[alice29.txt-python-lz4]                       316.5320 (27.67)        433.7360 (13.79)        339.7019 (26.85)        8.0091 (12.56)        338.0800 (26.89)        9.2835 (28.22)      502;81   2,943.7578 (0.04)       2557           1
test_lz4_block[asyoulik.txt-cramjam]                         233.0440 (20.37)        468.6850 (14.91)        265.3226 (20.97)        9.8275 (15.42)        264.5825 (21.05)        5.0740 (15.42)     282;267   3,768.9961 (0.05)       3220           1
test_lz4_block[asyoulik.txt-python-lz4]                      251.7520 (22.01)        305.7670 (9.72)         264.3520 (20.89)        5.8861 (9.23)         263.0755 (20.93)        6.7730 (20.59)     658;114   3,782.8353 (0.05)       3146           1
test_lz4_block[fifty-four-mb-random-cramjam]              33,936.9620 (>1000.0)   39,619.5560 (>1000.0)   35,020.1372 (>1000.0)  1,505.6802 (>1000.0)   34,484.9795 (>1000.0)    643.3850 (>1000.0)       3;4      28.5550 (0.00)         26           1
test_lz4_block[fifty-four-mb-random-python-lz4]           68,777.0100 (>1000.0)   73,922.2570 (>1000.0)   69,746.8938 (>1000.0)  1,239.5380 (>1000.0)   69,442.6260 (>1000.0)    580.1038 (>1000.0)       1;1      14.3376 (0.00)         15           1
test_lz4_block[fifty-four-mb-repeating-cramjam]           19,954.6251 (>1000.0)   25,531.1310 (812.01)    20,492.9045 (>1000.0)    889.2840 (>1000.0)   20,244.1360 (>1000.0)    452.6035 (>1000.0)       2;3      48.7974 (0.00)         40           1
test_lz4_block[fifty-four-mb-repeating-python-lz4]        36,592.0160 (>1000.0)   46,017.1160 (>1000.0)   38,476.8488 (>1000.0)  2,577.5186 (>1000.0)   37,021.7970 (>1000.0)  3,577.2075 (>1000.0)       3;1      25.9897 (0.00)         21           1
test_lz4_block[fireworks.jpeg-cramjam]                        11.4390 (1.0)           36.9540 (1.18)          12.6528 (1.0)          0.7672 (1.20)          12.5710 (1.0)          0.3290 (1.0)     3205;3529  79,033.6115 (1.0)       28912           1
test_lz4_block[fireworks.jpeg-python-lz4]                     12.8610 (1.12)          44.1120 (1.40)          13.9412 (1.10)         0.9270 (1.45)          13.8190 (1.10)         0.3920 (1.19)    1652;1957  71,729.6495 (0.91)      28694           1
test_lz4_block[geo.protodata-cramjam]                         76.8950 (6.72)         115.9160 (3.69)          82.5451 (6.52)         3.2757 (5.14)          81.7310 (6.50)         3.1550 (9.59)     1578;488  12,114.5850 (0.15)       7139           1
test_lz4_block[geo.protodata-python-lz4]                      68.0240 (5.95)         140.0970 (4.46)          80.1480 (6.33)         4.9473 (7.76)          79.7030 (6.34)         7.4280 (22.58)     2367;28  12,476.9164 (0.16)       6880           1
test_lz4_block[html-cramjam]                                  79.2720 (6.93)         122.6350 (3.90)          83.5068 (6.60)         2.9809 (4.68)          82.9190 (6.60)         2.6020 (7.91)     1272;399  11,975.0698 (0.15)       6624           1
test_lz4_block[html-python-lz4]                               74.3250 (6.50)         122.1850 (3.89)          85.0801 (6.72)         4.8946 (7.68)          84.5900 (6.73)         7.8872 (23.97)     2512;19  11,753.6254 (0.15)       6913           1
test_lz4_block[html_x_4-cramjam]                             361.6360 (31.61)        604.3760 (19.22)        418.5955 (33.08)       36.3753 (57.06)        411.3950 (32.73)       13.0960 (39.81)     132;156   2,388.9413 (0.03)       1474           1
test_lz4_block[html_x_4-python-lz4]                          364.2090 (31.84)        672.1250 (21.38)        453.7803 (35.86)       78.2668 (122.78)       423.9325 (33.72)       33.9675 (103.25)    358;366   2,203.7097 (0.03)       2044           1
test_lz4_block[kppkn.gtb-cramjam]                            279.7870 (24.46)        426.6940 (13.57)        305.6761 (24.16)       14.1955 (22.27)        301.6310 (23.99)       20.3990 (62.01)      550;26   3,271.4365 (0.04)       2407           1
test_lz4_block[kppkn.gtb-python-lz4]                         293.6670 (25.67)        367.0160 (11.67)        304.3268 (24.05)       10.2700 (16.11)        301.3470 (23.97)       13.4405 (40.85)      293;56   3,285.9408 (0.04)       2504           1
test_lz4_block[lcet10.txt-cramjam]                           896.0210 (78.33)      1,182.3210 (37.60)        958.0824 (75.72)       32.5113 (51.00)        952.5710 (75.78)       14.2605 (43.35)       81;97   1,043.7516 (0.01)        999           1
test_lz4_block[lcet10.txt-python-lz4]                        931.2580 (81.41)      1,265.8670 (40.26)        995.4354 (78.67)       42.5147 (66.69)        986.1050 (78.44)       23.3195 (70.88)       53;48   1,004.5855 (0.01)        883           1
test_lz4_block[paper-100k.pdf-cramjam]                        31.9140 (2.79)          77.9020 (2.48)          35.5357 (2.81)         1.4271 (2.24)          35.3980 (2.82)         1.1760 (3.57)     2066;781  28,140.7087 (0.36)      13158           1
test_lz4_block[paper-100k.pdf-python-lz4]                     37.6930 (3.30)          75.8310 (2.41)          40.6318 (3.21)         1.5697 (2.46)          40.4160 (3.22)         1.3630 (4.14)     1679;505  24,611.2554 (0.31)      14017           1
test_lz4_block[plrabn12.txt-cramjam]                       1,036.5970 (90.62)      1,409.9260 (44.84)      1,103.4860 (87.21)       32.3477 (50.74)      1,097.3520 (87.29)       23.5150 (71.48)       66;28     906.2190 (0.01)        780           1
test_lz4_block[plrabn12.txt-python-lz4]                    1,047.2300 (91.55)      1,456.7000 (46.33)      1,139.6521 (90.07)       50.3670 (79.01)      1,130.3140 (89.91)       24.7840 (75.33)       57;56     877.4608 (0.01)        780           1
test_lz4_block[urls.10K-cramjam]                             990.4180 (86.58)      1,582.8770 (50.34)      1,069.2153 (84.50)       55.2439 (86.66)      1,062.6070 (84.53)       21.2860 (64.70)       25;34     935.2654 (0.01)        822           1
test_lz4_block[urls.10K-python-lz4]                        1,045.4830 (91.40)      1,756.4630 (55.86)      1,122.6357 (88.73)       68.5135 (107.48)     1,113.0090 (88.54)       26.6482 (81.00)       17;22     890.7609 (0.01)        645           1
--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
```

#### Brotli

`make bench-brotli`

```bash
------------------------------------------------------------------------------------------------ benchmark: 24 tests -------------------------------------------------------------------------------------------------
Name (time in ms)                                       Min                 Max                Mean            StdDev              Median               IQR            Outliers      OPS            Rounds  Iterations
----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
test_brotli[Mark.Twain-Tom.Sawyer.txt-brotli]       11.4190 (1.0)       12.9519 (1.0)       11.7689 (1.0)      0.1962 (1.0)       11.7535 (1.0)      0.1166 (1.0)          11;7  84.9697 (1.0)          83           1
test_brotli[Mark.Twain-Tom.Sawyer.txt-cramjam]      13.0522 (1.14)      14.6921 (1.13)      13.4430 (1.14)     0.3481 (1.77)      13.3178 (1.13)     0.2667 (2.29)         12;7  74.3880 (0.88)         74           1
test_brotli[alice29.txt-brotli]                    143.4769 (12.56)    156.1368 (12.06)    151.7042 (12.89)    4.5808 (23.34)    153.9301 (13.10)    5.9880 (51.36)         1;0   6.5918 (0.08)          7           1
test_brotli[alice29.txt-cramjam]                   152.9068 (13.39)    168.2758 (12.99)    160.7015 (13.65)    5.7192 (29.14)    161.0280 (13.70)    9.4915 (81.40)         3;0   6.2227 (0.07)          7           1
test_brotli[asyoulik.txt-brotli]                   112.6384 (9.86)     122.4138 (9.45)     116.6856 (9.91)     3.1558 (16.08)    116.6260 (9.92)     4.6635 (40.00)         3;0   8.5700 (0.10)          9           1
test_brotli[asyoulik.txt-cramjam]                  116.7427 (10.22)    127.8253 (9.87)     120.9304 (10.28)    3.6031 (18.36)    119.7438 (10.19)    3.6843 (31.60)         3;0   8.2692 (0.10)          8           1
test_brotli[fireworks.jpeg-brotli]                  34.3454 (3.01)      35.6485 (2.75)      34.8155 (2.96)     0.3415 (1.74)      34.7843 (2.96)     0.3527 (3.03)          9;1  28.7228 (0.34)         27           1
test_brotli[fireworks.jpeg-cramjam]                 42.6242 (3.73)      46.8781 (3.62)      43.7436 (3.72)     0.9247 (4.71)      43.4703 (3.70)     0.6432 (5.52)          5;3  22.8605 (0.27)         23           1
test_brotli[geo.protodata-brotli]                   70.6670 (6.19)      71.9949 (5.56)      71.0885 (6.04)     0.3661 (1.87)      71.0907 (6.05)     0.4194 (3.60)          4;1  14.0670 (0.17)         14           1
test_brotli[geo.protodata-cramjam]                  91.1827 (7.99)      92.7021 (7.16)      91.9072 (7.81)     0.5966 (3.04)      91.7817 (7.81)     1.1848 (10.16)         5;0  10.8805 (0.13)         11           1
test_brotli[html-brotli]                            76.3931 (6.69)      77.4682 (5.98)      76.9553 (6.54)     0.3230 (1.65)      76.9471 (6.55)     0.4636 (3.98)          5;0  12.9946 (0.15)         13           1
test_brotli[html-cramjam]                           99.2533 (8.69)     100.1915 (7.74)      99.7497 (8.48)     0.2519 (1.28)      99.7689 (8.49)     0.2664 (2.28)          2;0  10.0251 (0.12)         10           1
test_brotli[html_x_4-brotli]                        90.8162 (7.95)      91.7338 (7.08)      91.3030 (7.76)     0.3260 (1.66)      91.2223 (7.76)     0.5525 (4.74)          5;0  10.9525 (0.13)         11           1
test_brotli[html_x_4-cramjam]                      109.2463 (9.57)     112.8147 (8.71)     110.5038 (9.39)     1.2757 (6.50)     109.9550 (9.36)     2.1289 (18.26)         2;0   9.0495 (0.11)          9           1
test_brotli[kppkn.gtb-brotli]                      281.7638 (24.67)    283.8784 (21.92)    282.9393 (24.04)    0.8537 (4.35)     283.3318 (24.11)    1.2676 (10.87)         2;0   3.5343 (0.04)          5           1
test_brotli[kppkn.gtb-cramjam]                     338.8846 (29.68)    340.3086 (26.27)    339.5767 (28.85)    0.5379 (2.74)     339.5152 (28.89)    0.7513 (6.44)          2;0   2.9448 (0.03)          5           1
test_brotli[lcet10.txt-brotli]                     444.7350 (38.95)    446.8453 (34.50)    445.6186 (37.86)    0.8674 (4.42)     445.7035 (37.92)    1.3660 (11.72)         2;0   2.2441 (0.03)          5           1
test_brotli[lcet10.txt-cramjam]                    454.7244 (39.82)    462.5618 (35.71)    457.4089 (38.87)    3.0765 (15.68)    456.0511 (38.80)    3.3177 (28.45)         1;0   2.1862 (0.03)          5           1
test_brotli[paper-100k.pdf-brotli]                 272.0045 (23.82)    273.3697 (21.11)    272.8557 (23.18)    0.5674 (2.89)     272.9370 (23.22)    0.8819 (7.56)          1;0   3.6649 (0.04)          5           1
test_brotli[paper-100k.pdf-cramjam]                 92.7737 (8.12)      94.6250 (7.31)      93.6444 (7.96)     0.6740 (3.43)      93.9685 (7.99)     1.2357 (10.60)         4;0  10.6787 (0.13)         11           1
test_brotli[plrabn12.txt-brotli]                   490.8551 (42.99)    496.5078 (38.33)    493.5089 (41.93)    2.1778 (11.10)    493.9005 (42.02)    3.0938 (26.53)         2;0   2.0263 (0.02)          5           1
test_brotli[plrabn12.txt-cramjam]                  502.1304 (43.97)    510.4967 (39.41)    505.6520 (42.97)    3.4768 (17.72)    506.3175 (43.08)    5.4558 (46.79)         2;0   1.9776 (0.02)          5           1
test_brotli[urls.10K-brotli]                       759.7228 (66.53)    776.9392 (59.99)    765.7505 (65.07)    6.7197 (34.24)    762.9181 (64.91)    7.3713 (63.22)         1;0   1.3059 (0.02)          5           1
test_brotli[urls.10K-cramjam]                      747.8780 (65.49)    761.8195 (58.82)    751.5646 (63.86)    5.8166 (29.64)    749.6120 (63.78)    5.0277 (43.12)         1;1   1.3306 (0.02)          5           1
----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
```

#### Bzip2

`make bench-bzip2`

```bash
------------------------------------------------------------------------------------------------------- benchmark: 28 tests -------------------------------------------------------------------------------------------------------
Name (time in ms)                                        Min                    Max                  Mean              StdDev                Median                 IQR            Outliers       OPS            Rounds  Iterations
-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
test_bzip2[Mark.Twain-Tom.Sawyer.txt-bzip2]           1.2711 (1.00)          1.6461 (1.0)          1.3491 (1.01)       0.0385 (1.0)          1.3472 (1.02)       0.0512 (1.14)        204;5  741.2337 (0.99)        643           1
test_bzip2[Mark.Twain-Tom.Sawyer.txt-cramjam]         1.2670 (1.0)           1.8671 (1.13)         1.3389 (1.0)        0.0483 (1.26)         1.3258 (1.0)        0.0448 (1.0)         73;34  746.8702 (1.0)         630           1
test_bzip2[alice29.txt-bzip2]                        10.5259 (8.31)         11.9644 (7.27)        10.9831 (8.20)       0.2470 (6.42)        10.9820 (8.28)       0.2934 (6.55)         19;3   91.0490 (0.12)         90           1
test_bzip2[alice29.txt-cramjam]                      10.0803 (7.96)         11.6217 (7.06)        10.6278 (7.94)       0.3660 (9.51)        10.5032 (7.92)       0.5597 (12.49)        31;0   94.0926 (0.13)         94           1
test_bzip2[asyoulik.txt-bzip2]                        8.5538 (6.75)         10.2334 (6.22)         9.2801 (6.93)       0.4153 (10.79)        9.1760 (6.92)       0.6618 (14.76)        30;0  107.7571 (0.14)        106           1
test_bzip2[asyoulik.txt-cramjam]                      8.3083 (6.56)          9.0135 (5.48)         8.5585 (6.39)       0.1398 (3.63)         8.5525 (6.45)       0.1770 (3.95)         39;2  116.8429 (0.16)        116           1
test_bzip2[fifty-four-mb-random-bzip2]            5,798.4294 (>1000.0)   6,637.8179 (>1000.0)  6,010.0714 (>1000.0)  356.8759 (>1000.0)  5,825.5740 (>1000.0)  314.4360 (>1000.0)       1;1    0.1664 (0.00)          5           1
test_bzip2[fifty-four-mb-random-cramjam]          5,373.8872 (>1000.0)   5,395.6856 (>1000.0)  5,381.7519 (>1000.0)    9.1684 (238.21)   5,379.5500 (>1000.0)   14.1904 (316.55)        1;0    0.1858 (0.00)          5           1
test_bzip2[fifty-four-mb-repeating-bzip2]         9,565.3209 (>1000.0)  10,487.7696 (>1000.0)  9,907.5835 (>1000.0)  462.9329 (>1000.0)  9,575.6350 (>1000.0)  800.6694 (>1000.0)       1;0    0.1009 (0.00)          5           1
test_bzip2[fifty-four-mb-repeating-cramjam]       8,290.9926 (>1000.0)   8,564.3140 (>1000.0)  8,405.8197 (>1000.0)  100.4702 (>1000.0)  8,379.8931 (>1000.0)  105.7075 (>1000.0)       2;0    0.1190 (0.00)          5           1
test_bzip2[fireworks.jpeg-bzip2]                     11.4166 (9.01)         13.8160 (8.39)        11.9222 (8.90)       0.4338 (11.27)       11.8242 (8.92)       0.3818 (8.52)         10;5   83.8773 (0.11)         85           1
test_bzip2[fireworks.jpeg-cramjam]                   10.5497 (8.33)         12.2991 (7.47)        11.0318 (8.24)       0.2900 (7.53)        10.9575 (8.26)       0.2596 (5.79)         16;5   90.6472 (0.12)         92           1
test_bzip2[geo.protodata-bzip2]                       8.0638 (6.36)          9.1159 (5.54)         8.2840 (6.19)       0.1215 (3.16)         8.2872 (6.25)       0.1077 (2.40)         22;3  120.7146 (0.16)        118           1
test_bzip2[geo.protodata-cramjam]                     7.5771 (5.98)          8.8989 (5.41)         7.8477 (5.86)       0.1479 (3.84)         7.8153 (5.89)       0.1452 (3.24)         16;2  127.4257 (0.17)        127           1
test_bzip2[html-bzip2]                                6.4125 (5.06)          7.2505 (4.40)         6.6252 (4.95)       0.1029 (2.67)         6.6151 (4.99)       0.0885 (1.97)         22;7  150.9397 (0.20)        150           1
test_bzip2[html-cramjam]                              6.0190 (4.75)          6.6557 (4.04)         6.1884 (4.62)       0.0864 (2.25)         6.1636 (4.65)       0.1016 (2.27)         35;4  161.5915 (0.22)        158           1
test_bzip2[html_x_4-bzip2]                           31.2213 (24.64)        33.4331 (20.31)       31.5566 (23.57)      0.4104 (10.66)       31.4843 (23.75)      0.2987 (6.66)          1;1   31.6891 (0.04)         30           1
test_bzip2[html_x_4-cramjam]                         29.1961 (23.04)        31.7037 (19.26)       29.6869 (22.17)      0.4969 (12.91)       29.5098 (22.26)      0.4508 (10.06)         5;1   33.6849 (0.05)         32           1
test_bzip2[kppkn.gtb-bzip2]                           9.9215 (7.83)         11.7609 (7.14)        10.4842 (7.83)       0.3484 (9.05)        10.3495 (7.81)       0.3751 (8.37)         22;6   95.3820 (0.13)         94           1
test_bzip2[kppkn.gtb-cramjam]                         9.5839 (7.56)         10.6962 (6.50)         9.9063 (7.40)       0.2034 (5.28)         9.8640 (7.44)       0.2405 (5.36)         18;3  100.9457 (0.14)        101           1
test_bzip2[lcet10.txt-bzip2]                         28.7752 (22.71)        31.1098 (18.90)       29.2280 (21.83)      0.4010 (10.42)       29.1479 (21.99)      0.3664 (8.17)          2;1   34.2138 (0.05)         33           1
test_bzip2[lcet10.txt-cramjam]                       27.5969 (21.78)        29.8887 (18.16)       28.3710 (21.19)      0.6627 (17.22)       28.1787 (21.25)      0.8508 (18.98)         8;0   35.2472 (0.05)         34           1
test_bzip2[paper-100k.pdf-bzip2]                      8.5453 (6.74)          9.3542 (5.68)         8.8012 (6.57)       0.1380 (3.59)         8.7832 (6.62)       0.1122 (2.50)        25;11  113.6214 (0.15)        113           1
test_bzip2[paper-100k.pdf-cramjam]                    8.1157 (6.41)          9.4451 (5.74)         8.3298 (6.22)       0.1797 (4.67)         8.2959 (6.26)       0.0947 (2.11)         10;7  120.0510 (0.16)        119           1
test_bzip2[plrabn12.txt-bzip2]                       36.0196 (28.43)        38.5789 (23.44)       36.7931 (27.48)      0.7330 (19.05)       36.4790 (27.51)      0.9648 (21.52)         6;0   27.1790 (0.04)         26           1
test_bzip2[plrabn12.txt-cramjam]                     34.6344 (27.34)        37.5419 (22.81)       35.3211 (26.38)      0.5382 (13.98)       35.3134 (26.64)      0.4735 (10.56)         4;1   28.3117 (0.04)         27           1
test_bzip2[urls.10K-bzip2]                           51.0884 (40.32)        52.2594 (31.75)       51.6389 (38.57)      0.2879 (7.48)        51.6694 (38.97)      0.4628 (10.32)         5;0   19.3652 (0.03)         18           1
test_bzip2[urls.10K-cramjam]                         47.3159 (37.35)        48.2918 (29.34)       47.8151 (35.71)      0.2691 (6.99)        47.8155 (36.07)      0.3339 (7.45)          6;0   20.9139 (0.03)         19           1
-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
```


#### ZSTD

`make bench-zstd`

```bash 
----------------------------------------------------------------------------------------------------------- benchmark: 28 tests ------------------------------------------------------------------------------------------------------------
Name (time in us)                                        Min                    Max                   Mean                StdDev                 Median                   IQR            Outliers          OPS            Rounds  Iterations
--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
test_zstd[Mark.Twain-Tom.Sawyer.txt-cramjam]         70.5230 (1.15)        322.1980 (2.21)         83.2324 (1.23)        24.3582 (6.54)         77.1795 (1.15)         2.7310 (1.0)       104;305  12,014.5494 (0.81)       1952           1
test_zstd[Mark.Twain-Tom.Sawyer.txt-zstd]            61.4410 (1.0)         145.6300 (1.0)          67.5637 (1.0)          3.7238 (1.0)          67.0730 (1.0)          3.4307 (1.26)       434;79  14,800.8577 (1.0)        2845           1
test_zstd[alice29.txt-cramjam]                      678.0400 (11.04)     1,693.7420 (11.63)       884.3405 (13.09)      242.1637 (65.03)       787.4440 (11.74)       38.6440 (14.15)     132;180   1,130.7862 (0.08)        922           1
test_zstd[alice29.txt-zstd]                         711.2090 (11.58)     1,138.5870 (7.82)        864.7882 (12.80)      118.9756 (31.95)       795.0490 (11.85)      235.1320 (86.10)       351;0   1,156.3525 (0.08)        806           1
test_zstd[asyoulik.txt-cramjam]                     608.5230 (9.90)      1,528.0540 (10.49)       823.2820 (12.19)      240.6407 (64.62)       714.6660 (10.66)       64.3031 (23.55)     214;244   1,214.6507 (0.08)       1185           1
test_zstd[asyoulik.txt-zstd]                        655.8840 (10.68)     1,429.5070 (9.82)        697.0971 (10.32)       40.2685 (10.81)       691.1145 (10.30)       29.3460 (10.75)       73;49   1,434.5205 (0.10)        782           1
test_zstd[fifty-four-mb-random-cramjam]          53,400.1600 (869.13)   60,481.7220 (415.31)   56,141.9546 (830.95)   2,425.5086 (651.36)   54,846.9690 (817.72)   4,210.2271 (>1000.0)       5;0      17.8120 (0.00)         18           1
test_zstd[fifty-four-mb-random-zstd]             56,183.9560 (914.44)   61,756.0550 (424.06)   59,403.2061 (879.22)   1,776.4993 (477.07)   59,451.3920 (886.37)   3,177.9210 (>1000.0)       5;0      16.8341 (0.00)         17           1
test_zstd[fifty-four-mb-repeating-cramjam]       27,573.7410 (448.78)   28,531.4800 (195.92)   28,027.4104 (414.83)     231.2358 (62.10)    28,060.4760 (418.36)     310.7252 (113.78)        9;0      35.6794 (0.00)         31           1
test_zstd[fifty-four-mb-repeating-zstd]          35,470.3800 (577.31)   38,001.9480 (260.95)   36,584.3495 (541.48)     602.4920 (161.80)   36,570.5255 (545.23)     664.5960 (243.35)        7;2      27.3341 (0.00)         24           1
test_zstd[fireworks.jpeg-cramjam]                   112.2150 (1.83)        267.2100 (1.83)        122.2163 (1.81)        14.6080 (3.92)        118.9835 (1.77)         5.0770 (1.86)      124;275   8,182.2154 (0.55)       4158           1
test_zstd[fireworks.jpeg-zstd]                       97.1890 (1.58)        235.5780 (1.62)        122.0515 (1.81)        20.7609 (5.58)        110.0710 (1.64)        39.4670 (14.45)      2379;5   8,193.2630 (0.55)       6782           1
test_zstd[geo.protodata-cramjam]                    138.7520 (2.26)        418.6240 (2.87)        195.7750 (2.90)        74.8352 (20.10)       162.2900 (2.42)        16.4493 (6.02)      623;664   5,107.9044 (0.35)       3423           1
test_zstd[geo.protodata-zstd]                       128.5680 (2.09)        267.8980 (1.84)        153.2041 (2.27)        12.2514 (3.29)        150.9630 (2.25)        14.2765 (5.23)      820;110   6,527.2417 (0.44)       3015           1
test_zstd[html-cramjam]                             163.2600 (2.66)        467.2030 (3.21)        201.2071 (2.98)        44.9913 (12.08)       190.2790 (2.84)         9.6932 (3.55)      158;266   4,970.0030 (0.34)       3097           1
test_zstd[html-zstd]                                159.0910 (2.59)        279.1720 (1.92)        179.3438 (2.65)        11.5878 (3.11)        176.3290 (2.63)        14.4730 (5.30)     1031;111   5,575.8825 (0.38)       3973           1
test_zstd[html_x_4-cramjam]                         211.0290 (3.43)        648.3970 (4.45)        301.5790 (4.46)       122.9374 (33.01)       242.0545 (3.61)        33.0550 (12.10)     378;413   3,315.8807 (0.22)       1926           1
test_zstd[html_x_4-zstd]                            168.0120 (2.73)        520.4590 (3.57)        238.2982 (3.53)        78.2418 (21.01)       196.9880 (2.94)        52.9877 (19.40)     680;674   4,196.4231 (0.28)       2929           1
test_zstd[kppkn.gtb-cramjam]                        547.1670 (8.91)      1,565.6570 (10.75)       748.9480 (11.09)      233.7635 (62.78)       636.6600 (9.49)        55.8945 (20.47)     258;285   1,335.2062 (0.09)       1288           1
test_zstd[kppkn.gtb-zstd]                           565.8690 (9.21)        911.4790 (6.26)        653.9680 (9.68)        62.5809 (16.81)       626.3685 (9.34)        58.0240 (21.25)     285;140   1,529.1267 (0.10)       1074           1
test_zstd[lcet10.txt-cramjam]                     1,843.8050 (30.01)     4,286.3290 (29.43)     2,122.2706 (31.41)      356.2751 (95.68)     2,049.0970 (30.55)       42.2257 (15.46)       23;60     471.1935 (0.03)        439           1
test_zstd[lcet10.txt-zstd]                        1,913.1310 (31.14)     4,043.1030 (27.76)     2,042.1810 (30.23)      247.2250 (66.39)     2,008.1620 (29.94)       63.5270 (23.26)       14;16     489.6726 (0.03)        467           1
test_zstd[paper-100k.pdf-cramjam]                   217.1630 (3.53)        644.3760 (4.42)        255.1407 (3.78)        36.2405 (9.73)        249.5360 (3.72)         8.5870 (3.14)       51;216   3,919.4053 (0.26)       2910           1
test_zstd[paper-100k.pdf-zstd]                      267.2840 (4.35)        390.6240 (2.68)        286.3562 (4.24)         8.7586 (2.35)        285.4830 (4.26)         7.6720 (2.81)      612;219   3,492.1536 (0.24)       2816           1
test_zstd[plrabn12.txt-cramjam]                   2,557.3270 (41.62)     5,366.2910 (36.85)     2,787.0075 (41.25)      191.9163 (51.54)     2,766.6825 (41.25)       33.0600 (12.11)        7;33     358.8078 (0.02)        334           1
test_zstd[plrabn12.txt-zstd]                      2,464.5420 (40.11)     5,246.9490 (36.03)     2,703.3937 (40.01)      179.3986 (48.18)     2,689.2370 (40.09)       39.3315 (14.40)        6;26     369.9054 (0.02)        357           1
test_zstd[urls.10K-cramjam]                       2,552.0180 (41.54)     6,011.9580 (41.28)     2,728.0841 (40.38)      247.9977 (66.60)     2,705.6875 (40.34)       47.6180 (17.44)        2;25     366.5576 (0.02)        332           1
test_zstd[urls.10K-zstd]                          3,323.8390 (54.10)     5,461.0030 (37.50)     3,551.0236 (52.56)      229.1187 (61.53)     3,528.1250 (52.60)      122.4380 (44.83)         7;7     281.6089 (0.02)        243           1
--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
```
