<div align="center">

#  Python Super Collections

**Dictionaries as you dreamed them when you were a kid.**

Instantly Convert json and YAML files into objects with attributes.
</div>

```python
import json
from super_collections import SuperDict
with open('my_file.json', 'r') as file:
    data = json.load(file)
document = SuperDict(data)

print(document.author) # instead of document['author'] 
for document in document.blocks: # instead of document['blocks']
    ...
print(document.blocks[3].name) # instead of document['blocks'][3]['name'] -- eek! 🤢

```
________

<!-- To update, run the following command:
markdown-toc -i README.md 
-->

<!-- toc -->

- [Python Super Collections](#python-super-collections)
  - [What are SuperCollections?](#what-are-supercollections)
    - [What is the problem?](#what-is-the-problem)
    - [Definitions](#definitions)
  - [How it works](#how-it-works)
    - [Superdicts](#superdicts)
    - [Superlists](#superlists)
    - [Why Combining SuperDicts with SuperLists?](#why-combining-superdicts-with-superlists)
    - [SuperCollection](#supercollection)
    - [Factory function](#factory-function)
  - [Install](#install)
    - [From the repository](#from-the-repository)
  - [Usage](#usage)
  - [Export to JSON or Hjson](#export-to-json-or-hjson)
  - [Export to YAML](#export-to-yaml)
  - [Remarks](#remarks)
    - [Restrictions](#restrictions)
    - [Does it work?](#does-it-work)
    - [When are superdictionaries and superlists _not_ recommended?](#when-are-superdictionaries-and-superlists-not-recommended)
  - [Related data structures and ideas](#related-data-structures-and-ideas)
    - [Standard Python](#standard-python)
    - [Dot notation on dictionaries](#dot-notation-on-dictionaries)
    - [Using superlists to complement superdictionaries](#using-superlists-to-complement-superdictionaries)

<!-- tocstop -->

## What are SuperCollections?

### What is the problem?

There are several packages that quickly convert json or YAML files into 
dictionaries that contain dictionaries, lists etc.

If you want to properly use those data structures in Python, one solution is 
to create specific classes.

But sometimes, it is overkill. You just want your app to quickly load
structured data and navigate through them.

That's where the **super-collections** package comes handy.

### Definitions

- A **SuperCollection** is a nested data structure that can encode 
  any type of information (in the same way as a JSON file).
  It is essentially constituted of dictionaries and lists, where
  dictionaries can contain lists and vice-versa, and the root can
  be either a dictionary or a list.

- An **elementary datatype** is a non-mutable type: str, int, str,
  float in Python that is used to build classes.

- A **SuperList** is a list that can contain SuperCollections, or
  elementary datatypes.

- A **SuperDict** is a dict that can contain SuperCollections,
  or elementary datatypes. A key advantage of SuperDicts over dictionaries,
  is that its keys can be accessed as attributes (providing they are
  valid Python identifiers and they don't conflict with pre-existing attributes).
  
## How it works

### Superdicts


> 📝 **Definition** <br>  A **superdictionnary** is a dictionary
 whose keys (at least those that are valid identifiers) are automatically accessible as attributes, with the **dot notation*.

```python
d = SuperDict({'foo':5, 'bar': 'hello'})

# instead of writing d['foo']
d.foo = 7
```

> Several other languages, such as Javascript, LUA, Ruby, and PHP offer that **dot notation**
> in some form or other. However, implementing that idea is not as
> straightforward as it seems.
> The idea of superdictionaries in Python has been around for some time
> (see the [superdict](https://github.com/itdxer/superdict) packagage by Yuri 
> Shevchuk, 2015).



> 📝 **Property** <br> If a SuperDict object contains a value that is itself a dictionary, that dictionary is then converted in turn into a SuperDict.

If the object is not a dict or immediately compatible, it will try the following conversions:

- the methods `.asdict()`, `.dict()` or `dump()`, providing they actually generate a `dict`.
- if the object is a dataclass, it will apply the `asdict()` function on it.

### Superlists
A **superlist** is a list where all dictionary items have been
(automagically) converted to **superdictionnaries**.

> ⚠️ **Superlists are indispensable** <br> They were the missing piece of the jigsaw puzzle;
> without them, it is not possible to convert deep data structures into supercollections. 

### Why Combining SuperDicts with SuperLists?

The structure of JSON, YAML or HTML data is generally a deeply nested combination of dictionaries and lists. Using superdictionaries alone would not be sufficient, since lists within the data contained in a list would still contain regular (unconverted) dictionaries; this would require you to switch back to the standard dictionary access method. 

By combining superdictionaries and superlists, 
it is possible to ensure that all nested dictionaries within lists will also be converted to SuperDicts, allowing for a consistent dot notation throughout the entire data structure.

> 💡 **Deep conversion** <br> SuperLists objects, combined with SuperDicts make sure that the most complex
> datastructures (from json or YAML) can be recursively converted into 
> well-behaved Python  objects.

### SuperCollection

**SuperCollection** is an abstract class containing SuperList and SuperDict.

It means that SuperList and SuperDict objects are **instances** of 
SuperCollection, but they do not inherit from it.

```python
obj1 = SuperList([1, 3, 5])
assert isinstance(obj1, SuperCollection)

obj2 = SuperDict({'a': 5, 'b':7})
assert isinstance(obj2, SuperCollection)
```

### Factory function

You can use the `super_collect()` function to create a SuperCollection
(SuperList or SuperDict) from an object.



**It is particularly useful for converting Python data structures such as
JSON files.**

```python
import json
with open(FILENAME, 'r', encoding='utf-8') as f:
    data = json.load(f)
content = super_collect(data)
```

 `super_collect()` is designed to work on any list or dict, but it will also attempt
to process other types:

- all **sequences** (see [definition](https://docs.python.org/3/glossary.html#term-sequence)), 
  in other words objects whose class is registered as
  instance of `collections.abs.Sequence` 
  will be converted into lists. This applies to `tuple`, 
  `range`, `collections.UserList`, etc.
- Special types: `set`, `deque` as well as `ndarray` (Numpy or compatible)
  and `Series` (Pandas and others; your mileage may vary).
- Otherwise, will try to generate a SuperDict.

This is also available as a static method of the `SuperCollection` class:

```python
content = SuperCollection.collect(data)
```


## Install


### From the repository

```sh
pip install super-collections
```

## Usage

```python
from super_collections import SuperDict, SuperList

d = SuperDict({'foo':5, 'bar': 'hello'})
l = SuperList([5, 7, 'foo', {'foo': 5}])
```

You can cast any dictionary and list into its "Super" equivalent when you want, and you are off to the races. 

**The casting is recursive** i.e. in the case above, you can assert:

```python
l[-1].foo == 5
```

All methods of dict and list are available.


Those objects are self documented. `d.properties()` is a generator
that lists all keys that are available as attributes.

The `__dir__()` method (accessible with `dir()`) is properly updated with
those additional properties.

```python
list(d.properties())
> ['foo', 'bar']
dir(d)
> ['__class__', ..., 'bar', 'clear', 'copy', 'foo', 'fromkeys', 'get', 'items', 'keys', 'pop', 'popitem', 'properties', 'setdefault', 'to_hjson', 'to_json', 'update', 'values']
```

This means the **auto-complete feature** might be available
for the attributes of a SuperDict within a code editor (if the dictionary was
statically declared in the code); or in an advanced REPL
(such as [bpython](https://bpython-interpreter.org/)).

The methods `dict.update(other_dict)` and `list.extend(other_list)` 
automatically cast the contents into SuperDict and SuperList as needed.

## Export to JSON or Hjson

You can export a SuperDict or SuperList to JSON or [Hjson](https://hjson.github.io/), for debug purposes.
It is not guaranteed that it will preserve all the meaningful information you want, but all basic types
will be preserved. 

> Python DateTimes are converted into ISO Dates.

```python
print (d.to_json())
print (d.to_hjson())
```

The module also exports a `json_encode()` function, which will attempt to serialize
any object in Python to JSON (in an opinionated way).

## Export to YAML

If you wish to use PyYAML and guarantee the SuperDict and SuperList behave exactly as dict and list,
use the `yaml_support()` function.

This works with both `dump()` and `safedump()`


```python
from super_collections import SuperDict, SuperList, yaml_support
yaml_support()


d = SuperDict({"x": 1})
l = SuperList(["a", "b"])

dumped_dict = yaml.dump(d)
dumped_list = yaml.dump(l)
```

## Remarks

### Restrictions

1. In a SuperDict, **only keys that are valid Python identifiers
   can be accessed as attributes**. If 'bar' is a key of object `foo`,
   you can write `foo.bar`; but you can't
   write ~~`foo.hello world`~~ because 'hello world' is not a 
   valid Python identifier; 
   you will have to access that specific value with the "dictionary" notation: 
   `foo['hello world']`.
2. Similarly, you can't use pre-existing methods of the
   `dict` class: `keys`, `items`, `update`, etc. as properties; as well as the
   `properties` method itself (wich is specific to SuperDict).
   In that case again, use the dictionary notation to access
   the value (`d['items']`, etc.). Those keys that
   cannot be accessed as attributes are said to be **masked**.
   If you are uncertain which are available, just use `SuperDict.properties()`.
   method.
3. Updating a single element (`d['foo']` for a SuperDict and `l[5]`
    for a SuperList) does not perfom any casting. That's to avoid crazy
    recursive situations, while giving
    you fine grain control on what you want to do 
    (just cast with `SuperDict()` and `SuperList()`).


### Does it work?

Yes. It is tested with pytest. See the `test` directory for examples.

### When are superdictionaries and superlists _not_ recommended?

SuperDicts (and SuperLists) classes are most useful when the program you are
writing is consuming loosely structured data (json, YAML, HTML)
you have every reason to believe they
are sufficiently well-formed: typically data exported from existing APIs
or Web sources.

> ⚠️ **Caution** <br> super-collections may not be the best 
> tool when source data come from a source whose quality
> is unsufficiently guaranteed for your needs, or is untrusted.

If you want to impose strongly formatted data structures in your code, one solution is 
to create [dataclasses](https://docs.python.org/3/library/dataclasses.html); especially those of [Pydantic](https://docs.pydantic.dev/latest/concepts/dataclasses/), which make implicit and explicit
controls on the integrity of the source data.

## Related data structures and ideas

These projects contain ideas that inspired or predated super-collections.

### Standard Python

* `collections.namedtuple`: tuples with dot notation ([standard python class](https://docs.python.org/3/library/collections.html#collections.namedtuple))
* `types.SimpleNamespace`: objects with arbitrary attributes ([standard python class](https://docs.python.org/3/library/types.html#types.SimpleNamespace))
* All Python classes have a __dict__ attribute, used at the foundation to implement the dot notation in the language, with the relative standard methods (`__setattr__()`, etc.) and functions (`setattr()`, etc.).
* In modern Python, the `dict` class has ordered keys (by insertion order) and is subclassable.

### Dot notation on dictionaries

* [addict](https://github.com/mewwts/addict) (Github)
* [DotMap](https://github.com/drgrib/dotmap): subclasses and MutableMapping and OrderedDict (Github)
* [SuperDict](https://github.com/itdxer/superdict): subclasses `dict` (Github)
* [dotty_dict](https://github.com/pawelzny/dotty_dict): wrapper (Github)

### Using superlists to complement superdictionaries

* Packages that write to and read from files, such as [shelve](https://docs.python.org/3/library/shelve.html) (standard), json, YAML, [Beautifulsoup](https://code.launchpad.net/beautifulsoup/), etc. heavily rely
  on a **combination of dictionaries and lists**. BeautifulSoup in particular supports dot notation.
* In general, **the construction of any syntactic or semantic tree requires both dictionaries and lists**.