% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/request_develop.R
\name{request_develop}
\alias{request_develop}
\alias{request_build}
\title{Build a Google API request}
\usage{
request_develop(
  endpoint,
  params = list(),
  base_url = "https://www.googleapis.com"
)

request_build(
  method = "GET",
  path = "",
  params = list(),
  body = list(),
  token = NULL,
  key = NULL,
  base_url = "https://www.googleapis.com"
)
}
\arguments{
\item{endpoint}{List of information about the target endpoint or, in
Google's vocabulary, the target "method". Presumably prepared from the
\href{https://developers.google.com/discovery/v1/getting_started#background-resources}{Discovery Document}
for the target API.}

\item{params}{Named list. Values destined for URL substitution, the query,
or, for \code{request_develop()} only, the body. For \code{request_build()}, body
parameters must be passed via the \code{body} argument.}

\item{base_url}{Character.}

\item{method}{Character. An HTTP verb, such as \code{GET} or \code{POST}.}

\item{path}{Character. Path to the resource, not including the API's
\code{base_url}. Examples: \code{drive/v3/about} or \code{drive/v3/files/{fileId}}. The
\code{path} can be a template, i.e. it can include variables inside curly
brackets, such as \code{{fileId}} in the example. Such variables are substituted
by \code{request_build()}, using named parameters found in \code{params}.}

\item{body}{List. Values to send in the API request body.}

\item{token}{Token, ready for inclusion in a request, i.e. prepared with
\code{\link[httr:config]{httr::config()}}.}

\item{key}{API key. Needed for requests that don't contain a token. For more,
see Google's document Credentials, access, security, and identity
(\verb{https://support.google.com/googleapi/answer/6158857?hl=en&ref_topic=7013279}).
A key can be passed as a named component of \code{params}, but note that the
formal argument \code{key} will clobber it, if non-\code{NULL}.}
}
\value{
\code{request_develop()}: \code{list()} with components \code{method}, \code{path}, \code{params},
\code{body}, and \code{base_url}.

\code{request_build()}: \code{list()} with components \code{method}, \code{path}
(post-substitution), \code{query} (the input \code{params} not used in URL
substitution), \code{body}, \code{token}, \code{url} (the full URL, post-substitution,
including the query).
}
\description{
Intended primarily for internal use in client packages that provide
high-level wrappers for users. The \code{vignette("request-helper-functions")}
describes how one might use these functions inside a wrapper package.
}
\section{\code{request_develop()}}{


Combines user input (\code{params}) with information about an API endpoint.
\code{endpoint} should contain these components:
\itemize{
\item \code{path}: See documentation for argument.
\item \code{method}: See documentation for argument.
\item \code{parameters}: Compared with \code{params} supplied by user. An error is
thrown if user-supplied \code{params} aren't named in
\code{endpoint$parameters} or if user fails to supply all required
parameters. In the return value, body parameters are separated from
those destined for path substitution or the query.
}

The return value is typically used as input to \code{request_build()}.
}

\section{\code{request_build()}}{


Builds a request, in a purely mechanical sense. This function does nothing
specific to any particular Google API or endpoint.
\itemize{
\item Use with the output of \code{request_develop()} or with hand-crafted input.
\item \code{params} are used for variable substitution in \code{path}. Leftover \code{params}
that are not bound by the \code{path} template automatically become HTTP
query parameters.
\item Adds an API key to the query iff \code{token = NULL} and removes the API key
otherwise. Client packages should generally pass their own API key in, but
note that \code{\link[=gargle_api_key]{gargle_api_key()}} is available for small-scale experimentation.
}

See \code{googledrive::generate_request()} for an example of usage in a client
package. googledrive has an internal list of selected endpoints, derived from
the Drive API Discovery Document
(\verb{https://www.googleapis.com/discovery/v1/apis/drive/v3/rest}),
exposed via \code{googledrive::drive_endpoints()}. An element from such a list is
the expected input for \code{endpoint}. \code{googledrive::generate_request()} is a
wrapper around \code{request_develop()} and \code{request_build()} that inserts a
googledrive-managed API key and some logic about Team Drives. All user-facing
functions use \code{googledrive::generate_request()} under the hood.
}

\examples{
\dontrun{
## Example with a prepared endpoint
ept <- googledrive::drive_endpoints("drive.files.update")[[1]]
req <- request_develop(
  ept,
  params = list(
    fileId = "abc",
    addParents = "123",
    description = "Exciting File"
  )
)
req

req <- request_build(
  method = req$method,
  path = req$path,
  params = req$params,
  body = req$body,
  token = "PRETEND_I_AM_A_TOKEN"
)
req

## Example with no previous knowledge of the endpoint
## List a file's comments
## https://developers.google.com/drive/v3/reference/comments/list
req <- request_build(
  method = "GET",
  path = "drive/v3/files/{fileId}/comments",
  params = list(
    fileId = "your-file-id-goes-here",
    fields = "*"
  ),
  token = "PRETEND_I_AM_A_TOKEN"
)
req

# Example with no previous knowledge of the endpoint and no token
# use an API key for which the Places API is enabled!
API_KEY <- "1234567890"

# get restaurants close to a location in Vancouver, BC
req <- request_build(
  method = "GET",
  path = "maps/api/place/nearbysearch/json",
  params = list(
    location = "49.268682,-123.167117",
    radius = 100,
    type = "restaurant"
  ),
  key = API_KEY,
  base_url = "https://maps.googleapis.com"
)
resp <- request_make(req)
out <- response_process(resp)
vapply(out$results, function(x) x$name, character(1))
}
}
\seealso{
Other requests and responses: 
\code{\link{request_make}()},
\code{\link{response_process}()}
}
\concept{requests and responses}
