% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/docs_layer.R
\name{layer_geoms}
\alias{layer_geoms}
\title{Layer geometry display}
\description{
In ggplot2, a plot in constructed by adding layers to it. A layer consists
of two important parts: the geometry (geoms), and statistical transformations
(stats). The 'geom' part of a layer is important because it determines the
looks of the data. Geoms determine \emph{how} something is displayed, not \emph{what}
is displayed.
}
\section{Specifying geoms}{
There are five ways in which the 'geom' part of a layer can be specified.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# 1. The geom can have a layer constructor
geom_area()

# 2. A stat can default to a particular geom
stat_density() # has `geom = "area"` as default

# 3. It can be given to a stat as a string
stat_function(geom = "area")

# 4. The ggproto object of a geom can be given
stat_bin(geom = GeomArea)

# 5. It can be given to `layer()` directly
layer(
  geom = "area",
  stat = "smooth",
  position = "identity"
)
}\if{html}{\out{</div>}}

Many of these ways are absolutely equivalent. Using
\code{stat_density(geom = "line")} is identical to using
\code{geom_line(stat = "density")}. Note that for \code{\link[=layer]{layer()}}, you need to
provide the \code{"position"} argument as well. To give geoms as a string, take
the function name, and remove the \code{geom_} prefix, such that \code{geom_point}
becomes \code{"point"}.

Some of the more well known geoms that can be used for the \code{geom} argument
are: \code{\link[=geom_point]{"point"}}, \code{\link[=geom_line]{"line"}},
\code{\link[=geom_area]{"area"}}, \code{\link[=geom_bar]{"bar"}} and
\code{\link[=geom_polygon]{"polygon"}}.
}

\section{Graphical display}{
A ggplot is build on top of the \link[=grid-package]{grid} package. This package
understands various graphical primitives, such as points, lines, rectangles
and polygons and their \link[=aes_position]{positions}, as well as graphical
attributes, also termed aesthetics, such as
\link[=aes_colour_fill_alpha]{colours, fills},
\link[=aes_linetype_size_shape]{linewidths and linetypes}. The job of the geom part
of a layer, is to translate data to grid graphics that can be plotted.

To see how aesthetics are specified, run \code{vignette("ggplot2-specs")}. To see
what geom uses what aesthetics, you can find the \strong{Aesthetics} section in
their documentation, for example in \code{\link[=geom_line]{?geom_line}}.

While almost anything can be represented by polygons if you try hard enough,
it is not always convenient to do so manually. For this reason, the geoms
provide abstractions that take most of this hassle away. \code{\link[=geom_ribbon]{geom_ribbon()}}
for example is a special case of \code{\link[=geom_polygon]{geom_polygon()}}, where two sets of
y-positions have a shared x-position. In turn, \code{\link[=geom_area]{geom_area()}} is a special
case of a ribbon, where one of the two sets of y-positions is set at 0.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# A hassle to build a polygon
my_polygon <- data.frame(
  x = c(economics$date,    rev(economics$date)),
  y = c(economics$uempmed, rev(economics$psavert))
)
ggplot(my_polygon, aes(x, y)) +
  geom_polygon()

# More succinctly
ggplot(economics, aes(date)) +
  geom_ribbon(aes(ymin = uempmed, ymax = psavert))
}\if{html}{\out{</div>}}

In addition to abstraction, geoms sometimes also perform composition.
A boxplot is a particular arrangement of lines, rectangles and points that
people have agreed upon is a summary of some data, which is performed by
\code{\link[=geom_boxplot]{geom_boxplot()}}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{Boxplot data
value <- fivenum(rnorm(100))
df <- data.frame(
  min = value[1], lower = value[2], middle = value[3],
  upper = value[4], max = value[5]
)

# Drawing a boxplot manually
ggplot(df, aes(x = 1, xend = 1)) +
  geom_rect(
    aes(
      xmin = 0.55, xmax = 1.45,
      ymin = lower, ymax = upper
    ),
    colour = "black", fill = "white"
  ) +
  geom_segment(
    aes(
      x = 0.55, xend = 1.45,
      y = middle, yend = middle
    ),
    size = 1
  ) +
  geom_segment(aes(y = lower, yend = min)) +
  geom_segment(aes(y = upper, yend = max))

# More succinctly
ggplot(df, aes(x = 1)) +
  geom_boxplot(
    aes(ymin = min, ymax = max,
        lower = lower, upper = upper,
        middle = middle),
    stat = "identity"
  )
}\if{html}{\out{</div>}}
}

\section{Under the hood}{
Internally, geoms are represented as \code{\link[=ggproto]{ggproto}} classes that
occupy a slot in a layer. All these classes inherit from the parental
\code{\link{Geom}} ggproto object that orchestrates how geoms work. Briefly, geoms
are given the opportunity to draw the data of the layer as a whole,
a facet panel, or of individual groups. For more information on extending
geoms, see the \strong{Creating a new geom} section after running
\code{vignette("extending-ggplot2")}. Additionally, see the \strong{New geoms} section
of the \href{https://ggplot2-book.org/extensions.html#new-geoms}{online book}.
}

\seealso{
For an overview of all geom layers, see the
\href{https://ggplot2.tidyverse.org/reference/index.html#geoms}{online reference}.

Other layer documentation: 
\code{\link{layer}()},
\code{\link{layer_positions}},
\code{\link{layer_stats}}
}
\concept{layer documentation}
