/* SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Copyright:
 *   2020      Evan Nemerson <evan@nemerson.com>
 *   2020      Ashleigh Newman-Jones <ashnewman-jones@hotmail.co.uk>
 */

#define SIMDE_TEST_X86_AVX512_INSN madd

#include <test/x86/avx512/test-avx512.h>
#include <simde/x86/avx512/madd.h>

static int
test_simde_mm_mask_madd_epi16 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int32_t src[4];
    const simde__mmask8 k;
    const int16_t a[8];
    const int16_t b[8];
    const int32_t r[4];
  } test_vec[] = {
    { {  INT32_C(  2047177297), -INT32_C(  1370766858), -INT32_C(   673821164), -INT32_C(   542785296) },
      UINT8_C( 36),
      {  INT16_C( 23610), -INT16_C( 16119), -INT16_C( 12959), -INT16_C( 13952), -INT16_C(  9773),  INT16_C( 29746), -INT16_C( 16467),  INT16_C(  8133) },
      {  INT16_C( 16580), -INT16_C( 31467), -INT16_C( 15477), -INT16_C(  9830),  INT16_C( 29082),  INT16_C( 23242), -INT16_C( 22250),  INT16_C( 20606) },
      {  INT32_C(  2047177297), -INT32_C(  1370766858),  INT32_C(   407138146), -INT32_C(   542785296) } },
    { {  INT32_C(  1712424709),  INT32_C(   657429076),  INT32_C(   412901739), -INT32_C(   466067168) },
      UINT8_C(161),
      {  INT16_C( 27213),  INT16_C(  4396),  INT16_C(  1540),  INT16_C( 30123),  INT16_C(  1488),  INT16_C( 31116), -INT16_C(  9085),  INT16_C(  2943) },
      { -INT16_C(  6674), -INT16_C( 32673), -INT16_C( 30955),  INT16_C( 30443),  INT16_C(   803), -INT16_C( 31593),  INT16_C( 31547), -INT16_C( 30426) },
      { -INT32_C(   325250070),  INT32_C(   657429076),  INT32_C(   412901739), -INT32_C(   466067168) } },
    { { -INT32_C(   375762203),  INT32_C(   677332312), -INT32_C(   844960950), -INT32_C(  1244126777) },
      UINT8_C(  6),
      {  INT16_C( 13624), -INT16_C( 16613), -INT16_C( 28128),  INT16_C(  9442),  INT16_C( 26153), -INT16_C( 23457), -INT16_C(  6004), -INT16_C(  8310) },
      {  INT16_C( 29570), -INT16_C( 14537),  INT16_C( 24786), -INT16_C( 17135), -INT16_C(  8446),  INT16_C(  9093),  INT16_C( 15031), -INT16_C(  4311) },
      { -INT32_C(   375762203), -INT32_C(   858969278), -INT32_C(   434182739), -INT32_C(  1244126777) } },
    { { -INT32_C(  1867627152),  INT32_C(    11833559), -INT32_C(  2086398729),  INT32_C(  2137140988) },
      UINT8_C(162),
      {  INT16_C( 18074), -INT16_C(  1420),  INT16_C( 12888),  INT16_C( 14332),  INT16_C(  8119), -INT16_C(  3602), -INT16_C(  8632), -INT16_C( 29343) },
      { -INT16_C(  3444),  INT16_C(  7524),  INT16_C( 25766), -INT16_C( 17900), -INT16_C( 26871),  INT16_C( 14263),  INT16_C( 14074), -INT16_C( 27431) },
      { -INT32_C(  1867627152),  INT32_C(    75529408), -INT32_C(  2086398729),  INT32_C(  2137140988) } },
    { { -INT32_C(   728871300),  INT32_C(   923503232), -INT32_C(   248972631),  INT32_C(  1686080216) },
      UINT8_C(124),
      { -INT16_C( 32285),  INT16_C( 18466), -INT16_C(  8811),  INT16_C( 11601), -INT16_C( 30572), -INT16_C( 13785), -INT16_C( 17566), -INT16_C( 20410) },
      {  INT16_C(  6985), -INT16_C( 11472),  INT16_C( 26406),  INT16_C(  8316),  INT16_C( 28047),  INT16_C(  6648),  INT16_C( 24044), -INT16_C( 12139) },
      { -INT32_C(   728871300),  INT32_C(   923503232), -INT32_C(   949095564), -INT32_C(   174599914) } },
    { {  INT32_C(  1947777246),  INT32_C(   698444181),  INT32_C(  1408485617), -INT32_C(   872203901) },
      UINT8_C( 84),
      { -INT16_C( 24781), -INT16_C( 25989), -INT16_C( 25829), -INT16_C( 30678),  INT16_C( 17300), -INT16_C(  3723),  INT16_C( 17881), -INT16_C( 28209) },
      {  INT16_C( 17245), -INT16_C( 14810),  INT16_C( 20452), -INT16_C( 21321),  INT16_C(  2882),  INT16_C( 31535), -INT16_C(  1266),  INT16_C( 17104) },
      {  INT32_C(  1947777246),  INT32_C(   698444181), -INT32_C(    67546205), -INT32_C(   872203901) } },
    { { -INT32_C(  1243853926),  INT32_C(  2050885350),  INT32_C(   594260810),  INT32_C(  1437875192) },
      UINT8_C(126),
      {  INT16_C(  7130),  INT16_C( 10595),  INT16_C(  4050), -INT16_C(  8853), -INT16_C(  6593),  INT16_C( 15084),  INT16_C( 11958),  INT16_C(   469) },
      { -INT16_C( 30198),  INT16_C(  4584),  INT16_C( 25288),  INT16_C( 31579),  INT16_C( 32462),  INT16_C(  2419), -INT16_C( 14286),  INT16_C(  3207) },
      { -INT32_C(  1243853926), -INT32_C(   177152487), -INT32_C(   177533770), -INT32_C(   169327905) } },
    { { -INT32_C(  1237980445),  INT32_C(   965976314),  INT32_C(  1030979462), -INT32_C(  1203877715) },
      UINT8_C(211),
      { -INT16_C( 14042), -INT16_C( 30309),  INT16_C(  5924), -INT16_C( 23977),  INT16_C( 24714),  INT16_C( 21460), -INT16_C(  7961), -INT16_C( 11722) },
      { -INT16_C(  5099), -INT16_C( 18996),  INT16_C(  1408), -INT16_C(   197),  INT16_C( 30840), -INT16_C( 15955),  INT16_C( 26039), -INT16_C(  8812) },
      {  INT32_C(   647349922),  INT32_C(    13064461),  INT32_C(  1030979462), -INT32_C(  1203877715) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i src = simde_mm_loadu_epi32(test_vec[i].src);
    simde__m128i a = simde_mm_loadu_epi16(test_vec[i].a);
    simde__m128i b = simde_mm_loadu_epi16(test_vec[i].b);
    simde__m128i r = simde_mm_mask_madd_epi16(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i32x4(r, simde_mm_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m128i src = simde_test_x86_random_i32x4();
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m128i a = simde_test_x86_random_i16x8();
    simde__m128i b = simde_test_x86_random_i16x8();
    simde__m128i r = simde_mm_mask_madd_epi16(src, k, a, b);

    simde_test_x86_write_i32x4(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i16x8(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i16x8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x4(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm_maskz_madd_epi16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask8 k;
    const int16_t a[8];
    const int16_t b[8];
    const int32_t r[4];
  } test_vec[] = {
    { UINT8_C( 77),
      {  INT16_C( 19733),  INT16_C( 18694), -INT16_C(  2158), -INT16_C( 12949), -INT16_C( 22664), -INT16_C(  3805),  INT16_C( 30779),  INT16_C(  1600) },
      {  INT16_C( 11918),  INT16_C(  7268),  INT16_C( 32199),  INT16_C( 14806), -INT16_C( 14152), -INT16_C( 23986),  INT16_C( 16941),  INT16_C( 17135) },
      {  INT32_C(   371045886),  INT32_C(           0),  INT32_C(   412007658),  INT32_C(   548843039) } },
    { UINT8_C(143),
      { -INT16_C( 29707), -INT16_C(  5087), -INT16_C(  4361), -INT16_C( 24987),  INT16_C( 22033), -INT16_C( 30247), -INT16_C(  8041), -INT16_C( 15080) },
      {  INT16_C( 13380), -INT16_C( 15988), -INT16_C( 15094), -INT16_C( 11398),  INT16_C(  7187),  INT16_C( 21760),  INT16_C( 17163),  INT16_C(   228) },
      { -INT32_C(   316148704),  INT32_C(   350626760), -INT32_C(   499823549), -INT32_C(   141445923) } },
    { UINT8_C(206),
      { -INT16_C(  5115), -INT16_C(  3131),  INT16_C( 25681), -INT16_C( 22524), -INT16_C( 29123),  INT16_C(  7487),  INT16_C(  1190), -INT16_C(  9631) },
      {  INT16_C(  9105),  INT16_C( 22244), -INT16_C( 18531), -INT16_C( 18070), -INT16_C( 16456), -INT16_C(  1084), -INT16_C( 15196), -INT16_C( 22071) },
      {  INT32_C(           0), -INT32_C(    68885931),  INT32_C(   471132180),  INT32_C(   194482561) } },
    { UINT8_C(176),
      { -INT16_C( 25201), -INT16_C(  3326), -INT16_C( 21855),  INT16_C( 12080),  INT16_C( 20201), -INT16_C(  4651), -INT16_C( 20561), -INT16_C( 11650) },
      { -INT16_C( 10860),  INT16_C( 19311),  INT16_C( 10303), -INT16_C(   509), -INT16_C(   276), -INT16_C( 20318),  INT16_C( 19656),  INT16_C( 22369) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0) } },
    { UINT8_C(233),
      {  INT16_C( 19043),  INT16_C(  3466), -INT16_C( 17798), -INT16_C( 14090), -INT16_C(  7281),  INT16_C( 16248),  INT16_C( 19042),  INT16_C( 14291) },
      {  INT16_C(  7866), -INT16_C(  7562),  INT16_C( 29730),  INT16_C(  8399),  INT16_C( 32535),  INT16_C( 25576),  INT16_C( 16352),  INT16_C( 17228) },
      {  INT32_C(   123582346),  INT32_C(           0),  INT32_C(           0),  INT32_C(   557580132) } },
    { UINT8_C(137),
      {  INT16_C( 20694), -INT16_C( 28668), -INT16_C( 13242),  INT16_C( 10784),  INT16_C( 24388), -INT16_C( 28788), -INT16_C( 15566),  INT16_C( 20553) },
      {  INT16_C( 11065), -INT16_C( 21134), -INT16_C( 27654),  INT16_C( 31428),  INT16_C( 10107), -INT16_C( 17574), -INT16_C( 24973),  INT16_C( 19012) },
      {  INT32_C(   834848622),  INT32_C(           0),  INT32_C(           0),  INT32_C(   779483354) } },
    { UINT8_C(238),
      { -INT16_C(  9656),  INT16_C(  5429),  INT16_C( 24570),  INT16_C( 22873), -INT16_C(  5909), -INT16_C( 20853), -INT16_C(  9167),  INT16_C( 24039) },
      { -INT16_C( 27570), -INT16_C(  7849), -INT16_C( 11943), -INT16_C( 32675),  INT16_C(  6188), -INT16_C( 13580),  INT16_C( 15964), -INT16_C( 23112) },
      {  INT32_C(           0), -INT32_C(  1040814785),  INT32_C(   246618848), -INT32_C(   701931356) } },
    { UINT8_C( 24),
      { -INT16_C( 17683),  INT16_C( 19475),  INT16_C( 27667), -INT16_C(   969), -INT16_C(  6664), -INT16_C( 11219), -INT16_C( 30004),  INT16_C( 24866) },
      {  INT16_C(  1250), -INT16_C( 19526),  INT16_C( 14945),  INT16_C( 31199), -INT16_C( 22226),  INT16_C( 27861),  INT16_C( 31330),  INT16_C( 20357) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(   433828158) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i a = simde_x_mm_loadu_epi16(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epi16(test_vec[i].b);
    simde__m128i r = simde_mm_maskz_madd_epi16(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i32x4(r, simde_x_mm_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_mask_madd_epi16 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int32_t src[8];
    const simde__mmask8 k;
    const int16_t a[16];
    const int16_t b[16];
    const int32_t r[8];
  } test_vec[] = {
    { { -INT32_C(  1131792638),  INT32_C(  1157182518),  INT32_C(  1934490506),  INT32_C(   512519722), -INT32_C(   612345234),  INT32_C(  1466518124), -INT32_C(   871508367), -INT32_C(  2145305730) },
      UINT8_C(115),
      { -INT16_C( 22212),  INT16_C( 13791),  INT16_C( 27118),  INT16_C( 15152),  INT16_C( 23260),  INT16_C( 26794),  INT16_C(  6264), -INT16_C(  1598),
         INT16_C( 12275),  INT16_C( 23619), -INT16_C( 19322), -INT16_C( 27854), -INT16_C( 20352), -INT16_C( 24113),  INT16_C( 16944),  INT16_C( 27724) },
      {  INT16_C( 11243), -INT16_C(  9823), -INT16_C( 11628),  INT16_C( 28949), -INT16_C( 16596), -INT16_C( 23079), -INT16_C( 25385), -INT16_C( 13666),
        -INT16_C(  7733),  INT16_C( 20774),  INT16_C( 22677),  INT16_C(  5861), -INT16_C( 19448),  INT16_C( 14519),  INT16_C(  1270), -INT16_C(  7772) },
      { -INT32_C(   385198509),  INT32_C(   123307144),  INT32_C(  1934490506),  INT32_C(   512519722),  INT32_C(   395738531), -INT32_C(   601417288),  INT32_C(    45709049), -INT32_C(  2145305730) } },
    { { -INT32_C(   994360017),  INT32_C(  1144377367),  INT32_C(  1726549647),  INT32_C(  1966114730), -INT32_C(    37267864), -INT32_C(  1240224594),  INT32_C(  1458490208), -INT32_C(    29912369) },
      UINT8_C(215),
      { -INT16_C(  4158), -INT16_C(  2110),  INT16_C( 20787),  INT16_C(  7174), -INT16_C( 20297), -INT16_C(  6237),  INT16_C(  2854), -INT16_C(  4803),
        -INT16_C(  5368),  INT16_C(  7321), -INT16_C(  1631), -INT16_C( 28697), -INT16_C( 18865), -INT16_C( 31199), -INT16_C(  1612),  INT16_C( 30585) },
      {  INT16_C( 15336),  INT16_C(  7022),  INT16_C( 29837),  INT16_C( 17463), -INT16_C(  9691),  INT16_C( 19244),  INT16_C( 27109), -INT16_C(  4808),
        -INT16_C( 11947), -INT16_C(  2551), -INT16_C(  3894),  INT16_C(  6534), -INT16_C( 22618),  INT16_C( 23455),  INT16_C(  6304), -INT16_C( 30510) },
      { -INT32_C(    78583508),  INT32_C(   745501281),  INT32_C(    76673399),  INT32_C(  1966114730),  INT32_C(    45455625), -INT32_C(  1240224594), -INT32_C(   305083975), -INT32_C(   943310398) } },
    { { -INT32_C(   509394860), -INT32_C(   635053387), -INT32_C(  1725607500),  INT32_C(   277306811), -INT32_C(   133787602),  INT32_C(   655461505), -INT32_C(   729632716),  INT32_C(   492655817) },
      UINT8_C(149),
      {  INT16_C( 19198),  INT16_C(  9179), -INT16_C( 28892),  INT16_C( 18805),  INT16_C( 12329), -INT16_C( 20314), -INT16_C( 11200),  INT16_C( 17984),
        -INT16_C( 15924), -INT16_C(  8749),  INT16_C(  2025),  INT16_C( 27533),  INT16_C( 22235),  INT16_C( 14528),  INT16_C( 21875),  INT16_C( 28985) },
      {  INT16_C(  5279), -INT16_C( 15467),  INT16_C(  2723), -INT16_C( 13300), -INT16_C( 19910),  INT16_C( 31356), -INT16_C( 17018),  INT16_C( 21184),
        -INT16_C( 27778),  INT16_C( 26415), -INT16_C( 17254),  INT16_C( 30419), -INT16_C( 27885), -INT16_C( 31058), -INT16_C(  6168), -INT16_C( 30728) },
      { -INT32_C(    40625351), -INT32_C(   635053387), -INT32_C(   882436174),  INT32_C(   277306811),  INT32_C(   211232037),  INT32_C(   655461505), -INT32_C(   729632716), -INT32_C(  1025576080) } },
    { { -INT32_C(  1622503941), -INT32_C(   781494633), -INT32_C(  1907628024),  INT32_C(   601885605),  INT32_C(   965414815), -INT32_C(   558932277), -INT32_C(   647667983),  INT32_C(  1096834373) },
      UINT8_C(234),
      { -INT16_C( 32288),  INT16_C( 19200),  INT16_C(  2130), -INT16_C( 25293), -INT16_C( 10090),  INT16_C( 30376),  INT16_C( 18428), -INT16_C( 30843),
         INT16_C( 20609),  INT16_C( 12517), -INT16_C( 10705), -INT16_C( 27506), -INT16_C( 11089),  INT16_C(  4081), -INT16_C(  9451), -INT16_C(  2631) },
      { -INT16_C( 18084), -INT16_C( 20928),  INT16_C( 29889),  INT16_C( 22347), -INT16_C(  3252),  INT16_C( 18637),  INT16_C( 21051), -INT16_C( 17201),
        -INT16_C( 19294), -INT16_C( 11796),  INT16_C( 31626),  INT16_C( 14693),  INT16_C( 22095),  INT16_C( 25672),  INT16_C(   305), -INT16_C( 29351) },
      { -INT32_C(  1622503941), -INT32_C(   501559101), -INT32_C(  1907628024),  INT32_C(   918458271),  INT32_C(   965414815), -INT32_C(   742701988), -INT32_C(   140244023),  INT32_C(    74339926) } },
    { {  INT32_C(  2067503546),  INT32_C(  1523746317), -INT32_C(  1247633542), -INT32_C(  1804504335), -INT32_C(  1318757082),  INT32_C(   669699032),  INT32_C(  1401631521), -INT32_C(   270474188) },
      UINT8_C(126),
      { -INT16_C( 29846),  INT16_C( 15778),  INT16_C(  7397), -INT16_C( 30500), -INT16_C( 12591),  INT16_C( 17146),  INT16_C(  8290), -INT16_C( 14432),
         INT16_C( 30929), -INT16_C( 17262), -INT16_C( 19296),  INT16_C( 11247),  INT16_C(  8967), -INT16_C(  6384), -INT16_C( 29166),  INT16_C( 32003) },
      { -INT16_C( 23015), -INT16_C(    70), -INT16_C( 26942), -INT16_C( 27513), -INT16_C( 32412), -INT16_C( 14634),  INT16_C( 30369),  INT16_C( 29582),
         INT16_C(  8431), -INT16_C( 28881),  INT16_C(  7892), -INT16_C(  9286), -INT16_C( 13759),  INT16_C( 21699), -INT16_C( 14760),  INT16_C( 29393) },
      {  INT32_C(  2067503546),  INT32_C(   639856526),  INT32_C(   157184928), -INT32_C(   175168414),  INT32_C(   759306221), -INT32_C(   256723674), -INT32_C(   261903369), -INT32_C(   270474188) } },
    { {  INT32_C(   795970412), -INT32_C(  2033977311),  INT32_C(   441227641), -INT32_C(     7480816), -INT32_C(   812729093),  INT32_C(   480987354),  INT32_C(  1802530323), -INT32_C(  1579335372) },
      UINT8_C(204),
      { -INT16_C(  4656), -INT16_C( 27834), -INT16_C( 16525), -INT16_C( 16340),  INT16_C( 15578),  INT16_C( 26522), -INT16_C( 27333), -INT16_C( 14044),
        -INT16_C(   411),  INT16_C(  4114),  INT16_C(  9498), -INT16_C( 30082), -INT16_C( 19824),  INT16_C( 28363), -INT16_C( 26797),  INT16_C(  9148) },
      {  INT16_C(   901), -INT16_C(  1866), -INT16_C(  7230), -INT16_C( 25416),  INT16_C( 21279),  INT16_C( 23300),  INT16_C( 10472),  INT16_C( 19748),
         INT16_C( 13862),  INT16_C( 16733), -INT16_C(  9381), -INT16_C(  4917), -INT16_C( 26738), -INT16_C(  7846),  INT16_C(  5678), -INT16_C( 19707) },
      {  INT32_C(   795970412), -INT32_C(  2033977311),  INT32_C(   949446862), -INT32_C(   563572088), -INT32_C(   812729093),  INT32_C(   480987354),  INT32_C(   307518014), -INT32_C(   332433002) } },
    { { -INT32_C(   592659687), -INT32_C(  1099406178), -INT32_C(  1608942409), -INT32_C(   873644636), -INT32_C(   821277836), -INT32_C(  1262758106), -INT32_C(  1651108498),  INT32_C(  1162910508) },
      UINT8_C( 86),
      { -INT16_C(  2783), -INT16_C( 26015),  INT16_C(  6323), -INT16_C( 13290), -INT16_C( 17480), -INT16_C( 23031),  INT16_C( 32134), -INT16_C( 27919),
         INT16_C(  5965),  INT16_C(  2153), -INT16_C( 10036),  INT16_C( 25118),  INT16_C( 19061), -INT16_C( 14851),  INT16_C( 21391), -INT16_C( 20030) },
      {  INT16_C(  9032), -INT16_C(  1205),  INT16_C( 24891), -INT16_C(  2873), -INT16_C( 12004), -INT16_C( 23910), -INT16_C( 29874), -INT16_C( 25804),
        -INT16_C( 24926),  INT16_C( 28324), -INT16_C( 15754), -INT16_C(  5168), -INT16_C( 13044), -INT16_C( 25680),  INT16_C( 29217),  INT16_C( 26956) },
      { -INT32_C(   592659687),  INT32_C(   195567963),  INT32_C(   760501130), -INT32_C(   873644636), -INT32_C(    87702018), -INT32_C(  1262758106),  INT32_C(   132741996),  INT32_C(  1162910508) } },
    { { -INT32_C(   781871211),  INT32_C(   365243641),  INT32_C(  1287151613), -INT32_C(  1930957590),  INT32_C(    16485258),  INT32_C(  1508625229), -INT32_C(  1158308711), -INT32_C(  1541193458) },
      UINT8_C(217),
      { -INT16_C( 11659),  INT16_C( 15029), -INT16_C( 19737), -INT16_C( 24679), -INT16_C( 31746), -INT16_C(  6516),  INT16_C(  5647),  INT16_C(  2673),
        -INT16_C( 16617),  INT16_C(   726),  INT16_C( 28440),  INT16_C(  3486), -INT16_C( 21207),  INT16_C( 19535),  INT16_C( 10321), -INT16_C( 14635) },
      { -INT16_C( 29958), -INT16_C(  7936), -INT16_C( 26308),  INT16_C( 15233),  INT16_C(  3356),  INT16_C( 11041), -INT16_C( 28125),  INT16_C( 14902),
         INT16_C(  3153),  INT16_C( 27197), -INT16_C(  9349), -INT16_C( 23433), -INT16_C( 14712), -INT16_C(  9744), -INT16_C( 14866), -INT16_C(  5985) },
      {  INT32_C(   230010178),  INT32_C(   365243641),  INT32_C(  1287151613), -INT32_C(   118988829), -INT32_C(    32648379),  INT32_C(  1508625229),  INT32_C(   121648344), -INT32_C(    65841511) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i src = simde_x_mm256_loadu_epi16(test_vec[i].src);
    simde__m256i a = simde_x_mm256_loadu_epi16(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi16(test_vec[i].b);
    simde__m256i r = simde_mm256_mask_madd_epi16(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i16x16(r, simde_x_mm256_loadu_epi16(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m256i src = simde_test_x86_random_i32x8();
    simde__mmask8 k = simde_test_x86_random_mmask16();
    simde__m256i a = simde_test_x86_random_i16x16();
    simde__m256i b = simde_test_x86_random_i16x16();
    simde__m256i r = simde_mm256_mask_madd_epi16(src, k, a, b);

    simde_test_x86_write_i32x8(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i16x16(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i16x16(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm256_maskz_madd_epi16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask8 k;
    const int16_t a[16];
    const int16_t b[16];
    const int32_t r[8];
  } test_vec[] = {
    { UINT8_C(217),
      { -INT16_C( 16454), -INT16_C(  5745), -INT16_C( 10179),  INT16_C( 22530),  INT16_C( 26767), -INT16_C( 17839), -INT16_C( 21048), -INT16_C(   156),
         INT16_C(  7387),  INT16_C(  4894), -INT16_C(  9375),  INT16_C( 29558),  INT16_C(  5931), -INT16_C( 28270),  INT16_C(  6388), -INT16_C( 20885) },
      { -INT16_C(  1321),  INT16_C(  5271), -INT16_C( 26158),  INT16_C( 24940), -INT16_C( 17151), -INT16_C( 14053), -INT16_C( 32661),  INT16_C( 18121),
        -INT16_C(  6244), -INT16_C(   679), -INT16_C( 12350), -INT16_C(  4495),  INT16_C(   998), -INT16_C(  9601), -INT16_C(  5605), -INT16_C(  3448) },
      { -INT32_C(     8546161),  INT32_C(           0),  INT32_C(           0),  INT32_C(   684621852), -INT32_C(    49447454),  INT32_C(           0),  INT32_C(   277339408),  INT32_C(    36206740) } },
    { UINT8_C(228),
      {  INT16_C(  1568), -INT16_C( 17993),  INT16_C(  6259),  INT16_C( 12475), -INT16_C( 31692), -INT16_C( 19301), -INT16_C(  7859),  INT16_C( 13392),
         INT16_C( 20026),  INT16_C(  2807), -INT16_C(  6721), -INT16_C( 15632), -INT16_C( 13468),  INT16_C( 20445), -INT16_C( 12461),  INT16_C( 29491) },
      { -INT16_C(  5419),  INT16_C( 18477), -INT16_C(  6141),  INT16_C( 14201),  INT16_C(  5228), -INT16_C( 17685),  INT16_C( 15350),  INT16_C( 12526),
        -INT16_C(  6775),  INT16_C( 18490),  INT16_C( 11210),  INT16_C( 12042), -INT16_C(  6154),  INT16_C( 18814), -INT16_C( 20042), -INT16_C( 29507) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(   175652409),  INT32_C(           0),  INT32_C(           0), -INT32_C(   263582954),  INT32_C(   467534302), -INT32_C(   620447575) } },
    { UINT8_C(156),
      { -INT16_C( 11030), -INT16_C( 11617), -INT16_C( 10675),  INT16_C( 25150), -INT16_C(  1855), -INT16_C(   936), -INT16_C( 30489), -INT16_C( 13178),
        -INT16_C( 12605), -INT16_C(  4457), -INT16_C( 14631), -INT16_C( 16156),  INT16_C( 11588), -INT16_C(  2697),  INT16_C(  1002), -INT16_C( 11119) },
      {  INT16_C( 12503),  INT16_C(  9638), -INT16_C(  6906), -INT16_C( 14457), -INT16_C(  8227), -INT16_C( 15164),  INT16_C( 19047),  INT16_C( 10897),
         INT16_C( 10264), -INT16_C(  3816), -INT16_C(   786),  INT16_C( 12978),  INT16_C( 10538),  INT16_C(  5159), -INT16_C( 18132),  INT16_C(  1001) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(    29454589), -INT32_C(   724324649), -INT32_C(   112369808),  INT32_C(           0),  INT32_C(           0), -INT32_C(    29298383) } },
    { UINT8_C(233),
      {  INT16_C( 10383),  INT16_C( 29936), -INT16_C( 18513), -INT16_C( 29102),  INT16_C(  5755), -INT16_C( 14858),  INT16_C(  8359), -INT16_C( 12322),
        -INT16_C( 12487),  INT16_C( 13757), -INT16_C(  4223), -INT16_C( 21921),  INT16_C( 29719), -INT16_C( 12074), -INT16_C(  9635), -INT16_C(  4935) },
      { -INT16_C( 22270), -INT16_C( 19871), -INT16_C( 19615), -INT16_C(  9152),  INT16_C( 14025),  INT16_C( 29090), -INT16_C( 32681), -INT16_C( 28608),
        -INT16_C(   433), -INT16_C( 11835),  INT16_C(  9709),  INT16_C(  1147),  INT16_C( 21145), -INT16_C(  2348), -INT16_C( 29140),  INT16_C( 12002) },
      { -INT32_C(   826087666),  INT32_C(           0),  INT32_C(           0),  INT32_C(    79327297),  INT32_C(           0), -INT32_C(    66144494),  INT32_C(   656758007),  INT32_C(   221534030) } },
    { UINT8_C( 55),
      { -INT16_C(  8125), -INT16_C(  2408),  INT16_C( 29985),  INT16_C( 22464),  INT16_C( 12567), -INT16_C( 26706),  INT16_C( 15985),  INT16_C( 28646),
        -INT16_C( 18684),  INT16_C( 10589),  INT16_C( 24883), -INT16_C( 31294), -INT16_C( 18378), -INT16_C( 15183), -INT16_C(  8294), -INT16_C(  8453) },
      { -INT16_C( 27456), -INT16_C(  7724), -INT16_C( 27639),  INT16_C(  8248), -INT16_C(  6203),  INT16_C( 14263), -INT16_C( 25307),  INT16_C( 10662),
         INT16_C(   853), -INT16_C( 30638),  INT16_C(  5221), -INT16_C( 25843), -INT16_C( 16692),  INT16_C( 26463),  INT16_C( 23197),  INT16_C( 23877) },
      {  INT32_C(   241679392), -INT32_C(   643472343), -INT32_C(   458860779),  INT32_C(           0), -INT32_C(   340363234),  INT32_C(   938644985),  INT32_C(           0),  INT32_C(           0) } },
    { UINT8_C(238),
      {  INT16_C( 15897), -INT16_C( 20745),  INT16_C(  6007),  INT16_C( 24179), -INT16_C( 21810),  INT16_C( 27779), -INT16_C( 21167),  INT16_C( 21697),
         INT16_C( 18943),  INT16_C(  5305),  INT16_C( 21590),  INT16_C(  5344),  INT16_C( 18355),  INT16_C(  3761),  INT16_C(  3980), -INT16_C( 22788) },
      { -INT16_C(  2995), -INT16_C( 15276), -INT16_C( 14581), -INT16_C(  9694), -INT16_C( 22926), -INT16_C( 15546),  INT16_C(  1875),  INT16_C( 21015),
        -INT16_C( 11952), -INT16_C( 22938),  INT16_C( 18213), -INT16_C(  9798),  INT16_C( 27534),  INT16_C(  7143), -INT16_C(  7302), -INT16_C( 14143) },
      {  INT32_C(           0), -INT32_C(   321979293),  INT32_C(    68163726),  INT32_C(   416274330),  INT32_C(           0),  INT32_C(   340858158),  INT32_C(   532251393),  INT32_C(   293228724) } },
    { UINT8_C(215),
      { -INT16_C( 29675), -INT16_C(  8989), -INT16_C( 16977),  INT16_C( 21838),  INT16_C(  4355),  INT16_C(  2728), -INT16_C(  1495), -INT16_C(  1446),
         INT16_C(    97), -INT16_C( 22497), -INT16_C(  1862),  INT16_C(  9526),  INT16_C( 20959), -INT16_C( 15456),  INT16_C( 26642),  INT16_C( 10138) },
      {  INT16_C( 32244), -INT16_C( 23804),  INT16_C( 21050),  INT16_C( 15864), -INT16_C( 24476), -INT16_C( 29369), -INT16_C( 24165), -INT16_C(   889),
        -INT16_C( 22879),  INT16_C( 23460), -INT16_C(  9569),  INT16_C( 32385),  INT16_C(  8492),  INT16_C( 15937), -INT16_C(  9079),  INT16_C( 32102) },
      { -INT32_C(   742866544), -INT32_C(    10927818), -INT32_C(   186711612),  INT32_C(           0), -INT32_C(   529998883),  INT32_C(           0), -INT32_C(    68338444),  INT32_C(    83567358) } },
    { UINT8_C( 89),
      {  INT16_C(  8554), -INT16_C( 17260), -INT16_C( 12007), -INT16_C( 17888), -INT16_C( 21223), -INT16_C( 17835),  INT16_C( 20788), -INT16_C(  9380),
        -INT16_C( 18443), -INT16_C( 12422), -INT16_C(  1992),  INT16_C( 23035),  INT16_C( 14906),  INT16_C(  5858),  INT16_C( 24736),  INT16_C(  2671) },
      {  INT16_C(   897), -INT16_C( 25914), -INT16_C(  6187), -INT16_C(  4524), -INT16_C( 22124), -INT16_C( 13912),  INT16_C(  1274), -INT16_C(  4188),
         INT16_C(  7868), -INT16_C(  2881), -INT16_C( 17898),  INT16_C( 20558),  INT16_C( 12532), -INT16_C( 27546), -INT16_C( 10608),  INT16_C(  4510) },
      {  INT32_C(   454948578),  INT32_C(           0),  INT32_C(           0),  INT32_C(    65767352), -INT32_C(   109321742),  INT32_C(           0),  INT32_C(    25437524),  INT32_C(           0) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_mm256_loadu_epi16(test_vec[i].a);
    simde__m256i b = simde_mm256_loadu_epi16(test_vec[i].b);
    simde__m256i r = simde_mm256_maskz_madd_epi16(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i32x8(r, simde_mm256_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_madd_epi16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int16_t a[32];
    const int16_t b[32];
    const int32_t r[16];
  } test_vec[] = {
    { {  INT16_C( 28000),  INT16_C( 31877), -INT16_C( 10728),  INT16_C( 28413), -INT16_C(  7321), -INT16_C( 24759),  INT16_C( 27401),  INT16_C(  4695),
        -INT16_C(  5593), -INT16_C(  3334), -INT16_C( 10804),  INT16_C( 22250),  INT16_C( 24880), -INT16_C( 25050),  INT16_C(  4800),  INT16_C(  8447),
        -INT16_C( 31616), -INT16_C( 26467), -INT16_C( 26021), -INT16_C( 15866),  INT16_C( 20349), -INT16_C( 31135), -INT16_C( 18245), -INT16_C(  7528),
        -INT16_C( 27997),  INT16_C( 28629), -INT16_C( 16537), -INT16_C( 26427), -INT16_C(  5344), -INT16_C(  7882),  INT16_C( 13821),  INT16_C( 32001) },
      { -INT16_C( 24903),  INT16_C(  5142),  INT16_C(  7224), -INT16_C( 18986),  INT16_C( 14188),  INT16_C( 10043), -INT16_C( 11280), -INT16_C( 27895),
        -INT16_C(  8602), -INT16_C( 13054), -INT16_C( 14435), -INT16_C( 16795), -INT16_C( 25678), -INT16_C( 20577), -INT16_C( 24368), -INT16_C( 30163),
         INT16_C( 17215),  INT16_C( 30622),  INT16_C( 30047), -INT16_C( 13523),  INT16_C( 26796), -INT16_C( 25358), -INT16_C(   964), -INT16_C( 24017),
         INT16_C( 12762),  INT16_C( 30831), -INT16_C( 10760), -INT16_C( 21962), -INT16_C( 10896),  INT16_C( 16730), -INT16_C( 30859), -INT16_C( 19253) },
      { -INT32_C(   533372466), -INT32_C(   616948290), -INT32_C(   352524985), -INT32_C(   440050305),  INT32_C(    91633022), -INT32_C(   217733010), -INT32_C(   123414790), -INT32_C(   371753261),
        -INT32_C(  1354741914), -INT32_C(   567297069),  INT32_C(  1334793134),  INT32_C(   198388156),  INT32_C(   525362985),  INT32_C(   758327894), -INT32_C(    73637636), -INT32_C(  1042617492) } },
    { {  INT16_C( 27082),  INT16_C( 10540),  INT16_C( 23006), -INT16_C( 29707), -INT16_C(  6207), -INT16_C(   729),  INT16_C( 22499), -INT16_C( 16737),
         INT16_C(  3976), -INT16_C( 32458),  INT16_C( 27876),  INT16_C( 21547), -INT16_C( 31423), -INT16_C( 18795),  INT16_C( 24588), -INT16_C( 10645),
        -INT16_C( 26678), -INT16_C( 22528), -INT16_C(  2576), -INT16_C( 20173),  INT16_C( 23516), -INT16_C( 16209),  INT16_C( 20146),  INT16_C( 14974),
        -INT16_C( 19363),  INT16_C( 16827), -INT16_C(  6368),  INT16_C( 24982),  INT16_C( 11116),  INT16_C( 30999), -INT16_C( 32116),  INT16_C( 22095) },
      {  INT16_C( 20249),  INT16_C(  2558),  INT16_C( 12868),  INT16_C(  8635),  INT16_C( 27277),  INT16_C( 16353),  INT16_C( 24504),  INT16_C(  5753),
         INT16_C( 13587),  INT16_C( 13143), -INT16_C(  4836), -INT16_C( 30572), -INT16_C( 21735), -INT16_C( 23295),  INT16_C( 20782),  INT16_C( 18427),
        -INT16_C(  1632), -INT16_C(  6831),  INT16_C(  3115), -INT16_C( 18426), -INT16_C(  6282),  INT16_C( 12023),  INT16_C( 28998),  INT16_C( 22852),
        -INT16_C( 25434), -INT16_C( 15732),  INT16_C(  8329), -INT16_C( 23990),  INT16_C( 19659), -INT16_C(  1721),  INT16_C( 17053),  INT16_C( 15681) },
      {  INT32_C(   575344738),  INT32_C(    39521263), -INT32_C(   181229676),  INT32_C(   455027535), -INT32_C(   372573582), -INT32_C(   793543220),  INT32_C(  1120808430),  INT32_C(   314832401),
         INT32_C(   197427264),  INT32_C(   363683458), -INT32_C(   342608319),  INT32_C(   926379556),  INT32_C(   227756178), -INT32_C(   652357252),  INT32_C(   165180165), -INT32_C(   201202453) } },
    { { -INT16_C( 28100),  INT16_C( 26402),  INT16_C( 10398),  INT16_C(  5152),  INT16_C(  5903),  INT16_C( 21826), -INT16_C( 30840),  INT16_C( 11950),
         INT16_C( 14883), -INT16_C( 21264),  INT16_C( 15194),  INT16_C(  9807), -INT16_C( 27001),  INT16_C(  9247),  INT16_C( 24793),  INT16_C(  5473),
        -INT16_C( 31502), -INT16_C( 28548), -INT16_C( 25428), -INT16_C( 17244), -INT16_C(  6220),  INT16_C( 15377), -INT16_C( 16274), -INT16_C( 28309),
         INT16_C( 23546),  INT16_C( 21821), -INT16_C( 29546),  INT16_C(  7547), -INT16_C( 26077), -INT16_C(   959), -INT16_C( 23557), -INT16_C(  4847) },
      { -INT16_C( 29401), -INT16_C( 11394),  INT16_C(  8746), -INT16_C(  8561), -INT16_C( 24311),  INT16_C( 30490), -INT16_C( 31391),  INT16_C( 23304),
         INT16_C( 18145),  INT16_C( 30640),  INT16_C( 11218), -INT16_C(  2667), -INT16_C( 10554), -INT16_C( 15887),  INT16_C(   633), -INT16_C( 24402),
         INT16_C( 11408), -INT16_C( 17804),  INT16_C(   847),  INT16_C( 22680), -INT16_C( 19804),  INT16_C(  1488), -INT16_C( 10184),  INT16_C(  6497),
         INT16_C(  4382), -INT16_C(  3696),  INT16_C(  9533),  INT16_C(   998), -INT16_C(  9988),  INT16_C( 30148),  INT16_C( 29402),  INT16_C( 27158) },
      {  INT32_C(   525343712),  INT32_C(    46834636),  INT32_C(   521966907),  INT32_C(  1246581240), -INT32_C(   381476925),  INT32_C(   144291023),  INT32_C(   138061465), -INT32_C(   117858177),
         INT32_C(   148893776), -INT32_C(   412631436),  INT32_C(   146061856), -INT32_C(    18189157),  INT32_C(    22528156), -INT32_C(   274130112),  INT32_C(   231545144), -INT32_C(   824257740) } },
    { { -INT16_C( 30049), -INT16_C(  4572), -INT16_C( 17267),  INT16_C( 12870),  INT16_C(  5743), -INT16_C( 22729), -INT16_C( 26385),  INT16_C(  3520),
         INT16_C( 20650), -INT16_C(  6146), -INT16_C(  6794),  INT16_C( 29418), -INT16_C( 20803), -INT16_C( 26649), -INT16_C(   736), -INT16_C( 16638),
         INT16_C(  9863),  INT16_C(  5549), -INT16_C(  2845),  INT16_C( 21063),  INT16_C( 32266), -INT16_C(  1543), -INT16_C( 18153), -INT16_C( 16121),
         INT16_C(  1289),  INT16_C( 32680), -INT16_C( 27926), -INT16_C( 22543), -INT16_C(  9920),  INT16_C( 24639),  INT16_C( 16854),  INT16_C( 24096) },
      { -INT16_C( 12953),  INT16_C( 19059), -INT16_C( 17727), -INT16_C( 13156), -INT16_C( 27336),  INT16_C( 20421), -INT16_C( 13234),  INT16_C( 22544),
        -INT16_C( 18222), -INT16_C( 17193), -INT16_C( 14006), -INT16_C( 30108), -INT16_C( 23646),  INT16_C( 30955),  INT16_C(  3044),  INT16_C( 19414),
         INT16_C( 18904), -INT16_C( 25962),  INT16_C( 12803),  INT16_C( 15462),  INT16_C( 11208),  INT16_C(  5771), -INT16_C( 25352), -INT16_C( 13714),
         INT16_C( 18004), -INT16_C( 24698), -INT16_C(  5617), -INT16_C( 20183),  INT16_C(  5261),  INT16_C( 28969),  INT16_C(    31), -INT16_C(  1859) },
      {  INT32_C(   302086949),  INT32_C(   136774389), -INT32_C(   621139557),  INT32_C(   428533970), -INT32_C(   270616122), -INT32_C(   790560380), -INT32_C(   333012057), -INT32_C(   325250516),
         INT32_C(    42387014),  INT32_C(   289251571),  INT32_C(   352732675),  INT32_C(   681298250), -INT32_C(   783923484),  INT32_C(   611845711),  INT32_C(   661578071), -INT32_C(    44271990) } },
    { {  INT16_C( 21321),  INT16_C( 19858), -INT16_C(  1915),  INT16_C( 19849),  INT16_C(  5155),  INT16_C(  7012), -INT16_C( 11600),  INT16_C(  1509),
         INT16_C( 27672),  INT16_C( 10148), -INT16_C( 12970), -INT16_C(  6952),  INT16_C(   738),  INT16_C(   341),  INT16_C(  4610),  INT16_C( 19449),
        -INT16_C( 29851), -INT16_C(  5224),  INT16_C(  8579), -INT16_C( 22728), -INT16_C( 25546), -INT16_C(  6462), -INT16_C( 22417), -INT16_C( 30741),
        -INT16_C( 28908),  INT16_C( 27311), -INT16_C( 30883),  INT16_C( 16206), -INT16_C( 23415), -INT16_C( 29888),  INT16_C( 15030),  INT16_C(  7383) },
      {  INT16_C( 28613),  INT16_C( 18695),  INT16_C( 16273), -INT16_C( 14352), -INT16_C( 19748),  INT16_C( 19373), -INT16_C( 26278),  INT16_C( 28370),
        -INT16_C( 32472), -INT16_C( 31271),  INT16_C(  9993), -INT16_C( 27964),  INT16_C(  1483), -INT16_C( 32226), -INT16_C(  2753),  INT16_C(  1182),
        -INT16_C( 23196), -INT16_C(  2739),  INT16_C( 15844), -INT16_C( 16196),  INT16_C( 27376),  INT16_C( 18955), -INT16_C(  8701),  INT16_C( 11193),
        -INT16_C( 28065),  INT16_C( 26801),  INT16_C( 30137), -INT16_C( 31237),  INT16_C(  6522), -INT16_C( 18169), -INT16_C( 23282),  INT16_C( 29374) },
      {  INT32_C(   981303083), -INT32_C(   316035643),  INT32_C(    34042536),  INT32_C(   347635130), -INT32_C(  1215903292),  INT32_C(    64796518), -INT32_C(     9894612),  INT32_C(    10297388),
         INT32_C(   706732332),  INT32_C(   504028364), -INT32_C(   821834506), -INT32_C(   149033696),  INT32_C(  1543265131), -INT32_C(  1436947793),  INT32_C(   390322442), -INT32_C(   133060218) } },
    { {  INT16_C(  2890),  INT16_C( 11880),  INT16_C(  9289),  INT16_C( 14831), -INT16_C(  1394), -INT16_C( 28285),  INT16_C( 15576),  INT16_C( 14525),
         INT16_C( 28366), -INT16_C( 30560), -INT16_C( 25629),  INT16_C( 24077),  INT16_C(  5300), -INT16_C( 15849), -INT16_C( 10823),  INT16_C(   821),
        -INT16_C( 25119),  INT16_C( 10801),  INT16_C(  8385),  INT16_C( 20579), -INT16_C(  6629), -INT16_C(  3103), -INT16_C( 25053), -INT16_C(  3797),
        -INT16_C( 13300), -INT16_C(  3975), -INT16_C( 31129),  INT16_C(  7246),  INT16_C( 26010),  INT16_C( 21470),  INT16_C(  4923),  INT16_C(  7254) },
      { -INT16_C( 30544),  INT16_C( 29254), -INT16_C( 22104), -INT16_C( 15422), -INT16_C( 23665), -INT16_C( 19785), -INT16_C(  7614),  INT16_C( 20132),
         INT16_C(  7598),  INT16_C(  5694), -INT16_C( 29532),  INT16_C( 15922),  INT16_C(  4338),  INT16_C( 11666), -INT16_C(  6108), -INT16_C( 11191),
        -INT16_C( 28816),  INT16_C(  6470),  INT16_C(  2104), -INT16_C( 14372), -INT16_C( 27732), -INT16_C(  4486),  INT16_C(  7798),  INT16_C(  9276),
         INT16_C( 31547), -INT16_C(  8390),  INT16_C( 27655), -INT16_C(  1762), -INT16_C( 20355), -INT16_C( 24282),  INT16_C( 28568),  INT16_C(  2421) },
      {  INT32_C(   259265360), -INT32_C(   434047738),  INT32_C(   592607735),  INT32_C(   173821636),  INT32_C(    41516228),  INT32_C(  1140229622), -INT32_C(   161903034),  INT32_C(    56919073),
         INT32_C(   793711574), -INT32_C(   278119348),  INT32_C(   197755486), -INT32_C(   230584266), -INT32_C(   386224850), -INT32_C(   873639947), -INT32_C(  1050768090),  INT32_C(   158202198) } },
    { { -INT16_C( 17154),  INT16_C( 13858), -INT16_C(   316),  INT16_C( 28926),  INT16_C( 30866),  INT16_C(  2142), -INT16_C( 25706), -INT16_C( 11988),
         INT16_C( 26390),  INT16_C(  7601), -INT16_C( 12333),  INT16_C( 20503),  INT16_C( 15743),  INT16_C(  6129),  INT16_C( 26541), -INT16_C( 21728),
         INT16_C( 16931), -INT16_C(  6174), -INT16_C(  8127), -INT16_C( 11432), -INT16_C( 18856), -INT16_C(  4389),  INT16_C(  1873),  INT16_C( 26559),
         INT16_C( 28782),  INT16_C( 17029), -INT16_C( 25537), -INT16_C( 16750), -INT16_C( 31527), -INT16_C( 31018), -INT16_C(  2325),  INT16_C(  3634) },
      {  INT16_C(  5177),  INT16_C( 31477),  INT16_C( 19956),  INT16_C( 19533),  INT16_C( 10244),  INT16_C( 21818), -INT16_C(  1745), -INT16_C( 24899),
         INT16_C( 17002), -INT16_C( 22048),  INT16_C( 29406), -INT16_C( 18584),  INT16_C( 16118), -INT16_C(  7874),  INT16_C( 28724),  INT16_C( 28143),
        -INT16_C(  6780),  INT16_C( 30951),  INT16_C( 13362),  INT16_C( 14020), -INT16_C(   420), -INT16_C( 29556),  INT16_C( 18935),  INT16_C( 24874),
         INT16_C(  2699),  INT16_C( 26891),  INT16_C( 29564),  INT16_C( 29472),  INT16_C( 24241), -INT16_C(  6828),  INT16_C( 17614),  INT16_C( 21075) },
      {  INT32_C(   347402008),  INT32_C(   558705462),  INT32_C(   362925460),  INT32_C(   343346182),  INT32_C(   281095932), -INT32_C(   743691950),  INT32_C(   205485928),  INT32_C(   150872580),
        -INT32_C(   305883654), -INT32_C(   268869614),  INT32_C(   137640804),  INT32_C(   696093821),  INT32_C(   535609457), -INT32_C(  1248631868), -INT32_C(   552455103),  INT32_C(    35634000) } },
    { {  INT16_C( 14889),  INT16_C( 23498), -INT16_C( 29073), -INT16_C( 13422),  INT16_C(  7820), -INT16_C( 31657), -INT16_C( 32409), -INT16_C(  3355),
        -INT16_C(  3957),  INT16_C(  2139),  INT16_C( 31587),  INT16_C(  5243), -INT16_C( 12326), -INT16_C( 22278),  INT16_C( 19731),  INT16_C( 15611),
        -INT16_C( 14969), -INT16_C(  2408),  INT16_C( 10836), -INT16_C(  7998),  INT16_C(  6472), -INT16_C( 20636),  INT16_C( 19099),  INT16_C(  9889),
        -INT16_C(   966), -INT16_C( 25042), -INT16_C( 22153),  INT16_C( 20914), -INT16_C( 21383), -INT16_C( 29446), -INT16_C(  2567), -INT16_C( 32311) },
      {  INT16_C( 25018),  INT16_C(  3703),  INT16_C( 14731), -INT16_C( 11281),  INT16_C( 21331), -INT16_C(  4478),  INT16_C(  9117), -INT16_C( 10220),
         INT16_C( 17183), -INT16_C( 27018),  INT16_C( 10476),  INT16_C( 26088), -INT16_C(  7467), -INT16_C( 12558), -INT16_C( 17449), -INT16_C( 28337),
        -INT16_C( 14564), -INT16_C( 22624), -INT16_C( 28928),  INT16_C( 21370), -INT16_C(   798), -INT16_C( 32703),  INT16_C( 22047),  INT16_C( 15960),
        -INT16_C( 12647), -INT16_C( 31276), -INT16_C( 17162), -INT16_C( 13333), -INT16_C(  8802),  INT16_C( 30106), -INT16_C(  5736), -INT16_C( 19449) },
      {  INT32_C(   459506096), -INT32_C(   276860781),  INT32_C(   308568466), -INT32_C(   261184753), -INT32_C(   125784633),  INT32_C(   467684796),  INT32_C(   371805366), -INT32_C(   786655126),
         INT32_C(   272487108), -INT32_C(   484381068),  INT32_C(   669694452),  INT32_C(   578904093),  INT32_C(   795430594),  INT32_C(   101343424), -INT32_C(   698288110),  INT32_C(   643140951) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi16(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi16(test_vec[i].b);
    simde__m512i r = simde_mm512_madd_epi16(a, b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_mask_madd_epi16 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int32_t src[16];
    const simde__mmask16 k;
    const int16_t a[32];
    const int16_t b[32];
    const int32_t r[16];
  } test_vec[] = {
    { { -INT32_C(  1932878001),  INT32_C(  1422347064),  INT32_C(  2072045656), -INT32_C(   860440966),  INT32_C(  1027011522),  INT32_C(  1474407886),  INT32_C(  1647366516), -INT32_C(   431239017),
        -INT32_C(  1468918417), -INT32_C(  1191429792), -INT32_C(  1674347487), -INT32_C(   194451150), -INT32_C(  1422811428), -INT32_C(  1090383285),  INT32_C(  2065838820),  INT32_C(  1918987522) },
      UINT16_C(54402),
      { -INT16_C(  7654),  INT16_C(  5645),  INT16_C( 12186), -INT16_C( 12909), -INT16_C( 14901),  INT16_C( 13239), -INT16_C( 27718), -INT16_C(  5167),
         INT16_C(  7230),  INT16_C( 16638), -INT16_C(  7460), -INT16_C(   397),  INT16_C( 30045), -INT16_C( 16789), -INT16_C(  4633),  INT16_C(   402),
        -INT16_C( 24369),  INT16_C( 26904), -INT16_C( 21553), -INT16_C( 26058), -INT16_C(  4752),  INT16_C( 10957), -INT16_C( 24959), -INT16_C( 16618),
         INT16_C(  5306), -INT16_C( 27136),  INT16_C( 29686),  INT16_C( 21396), -INT16_C(    24), -INT16_C( 12271), -INT16_C( 23316), -INT16_C( 17455) },
      { -INT16_C(  5820),  INT16_C(  4900),  INT16_C( 23444),  INT16_C(  1453),  INT16_C( 31304), -INT16_C( 14033),  INT16_C( 17688), -INT16_C( 11639),
        -INT16_C( 30375),  INT16_C( 20329), -INT16_C(   516), -INT16_C(  7006), -INT16_C( 19203), -INT16_C(  5708), -INT16_C( 31144), -INT16_C( 25435),
        -INT16_C( 13969),  INT16_C(  1199),  INT16_C( 23588),  INT16_C( 27913),  INT16_C( 14550), -INT16_C(  4554), -INT16_C( 16514), -INT16_C( 10304),
         INT16_C( 10568),  INT16_C( 17447), -INT16_C( 14041),  INT16_C(  9257), -INT16_C(  8835), -INT16_C( 10995), -INT16_C( 19869), -INT16_C( 11407) },
      { -INT32_C(  1932878001),  INT32_C(   266931807),  INT32_C(  2072045656), -INT32_C(   860440966),  INT32_C(  1027011522),  INT32_C(  1474407886),  INT32_C(  1647366516),  INT32_C(   134065282),
        -INT32_C(  1468918417), -INT32_C(  1191429792), -INT32_C(   119039778), -INT32_C(   194451150), -INT32_C(   417367984), -INT32_C(  1090383285),  INT32_C(   135131685),  INT32_C(   662374789) } },
    { { -INT32_C(  1596514180),  INT32_C(  1376641148), -INT32_C(  1774173160),  INT32_C(  1282277635),  INT32_C(  1368429866), -INT32_C(   596264610), -INT32_C(    89029737), -INT32_C(  1311956171),
        -INT32_C(  1068325821), -INT32_C(  1659740284), -INT32_C(  1489808477),  INT32_C(  2129895764), -INT32_C(  1781497034), -INT32_C(   730774211), -INT32_C(    20045112), -INT32_C(  1984979899) },
      UINT16_C(  320),
      { -INT16_C( 15031),  INT16_C( 23393),  INT16_C(  1122), -INT16_C( 27218),  INT16_C(   683), -INT16_C( 25033),  INT16_C( 28033),  INT16_C( 20770),
         INT16_C( 24322),  INT16_C( 29590),  INT16_C( 24371),  INT16_C(   406), -INT16_C(  9379),  INT16_C(  3229), -INT16_C(  8604), -INT16_C( 21234),
         INT16_C( 28579),  INT16_C(  1289), -INT16_C( 18573),  INT16_C(  8090), -INT16_C( 11846),  INT16_C( 15293), -INT16_C(  8385),  INT16_C( 16780),
         INT16_C(  8766),  INT16_C( 29109),  INT16_C( 19329), -INT16_C(  8589),  INT16_C(  4134), -INT16_C( 29717), -INT16_C(  1554), -INT16_C( 28360) },
      {  INT16_C( 16744), -INT16_C(  9322),  INT16_C( 12793), -INT16_C( 19462), -INT16_C( 18430),  INT16_C( 16878),  INT16_C( 31383), -INT16_C( 10621),
         INT16_C( 14492),  INT16_C(  7751), -INT16_C( 17789), -INT16_C( 22020), -INT16_C(  6197), -INT16_C( 18124),  INT16_C( 28128),  INT16_C( 18507),
        -INT16_C(  7762), -INT16_C( 22748),  INT16_C(  7698),  INT16_C(  5466),  INT16_C( 18646),  INT16_C( 28246), -INT16_C(  9790),  INT16_C( 24388),
        -INT16_C( 29935), -INT16_C( 27523),  INT16_C( 31046),  INT16_C(  4414),  INT16_C( 29281),  INT16_C( 16842),  INT16_C(  5599), -INT16_C( 29046) },
      { -INT32_C(  1596514180),  INT32_C(  1376641148), -INT32_C(  1774173160),  INT32_C(  1282277635),  INT32_C(  1368429866), -INT32_C(   596264610), -INT32_C(      400733), -INT32_C(  1311956171),
        -INT32_C(   251152370), -INT32_C(  1659740284), -INT32_C(  1489808477),  INT32_C(  2129895764), -INT32_C(  1781497034), -INT32_C(   730774211), -INT32_C(    20045112), -INT32_C(  1984979899) } },
    { {  INT32_C(   154513143), -INT32_C(  1558277940), -INT32_C(  1693354536),  INT32_C(  1627018574),  INT32_C(   653555680),  INT32_C(  1362572016), -INT32_C(  2070740315),  INT32_C(   236068119),
        -INT32_C(  1760016181), -INT32_C(  1338362152), -INT32_C(    95728725), -INT32_C(  2124790368), -INT32_C(  1381544260),  INT32_C(   654237569), -INT32_C(   123039263),  INT32_C(  2030550446) },
      UINT16_C( 7941),
      { -INT16_C(  8943),  INT16_C( 19285),  INT16_C(   397), -INT16_C(  9833),  INT16_C( 14331),  INT16_C( 21790), -INT16_C(  9288),  INT16_C( 24739),
         INT16_C(  9352), -INT16_C( 31169),  INT16_C(  8266), -INT16_C(  2792), -INT16_C( 14824),  INT16_C(  8114), -INT16_C( 18624),  INT16_C( 20798),
        -INT16_C( 27500),  INT16_C(  8604),  INT16_C( 13205), -INT16_C( 28422),  INT16_C(  6507),  INT16_C(  9189), -INT16_C( 30476),  INT16_C( 31875),
        -INT16_C( 15699), -INT16_C(  2302),  INT16_C(  6882), -INT16_C(  1044), -INT16_C( 24863),  INT16_C(  8474),  INT16_C( 22869), -INT16_C(  5774) },
      {  INT16_C(  3821), -INT16_C( 32245),  INT16_C(  1346), -INT16_C( 21230), -INT16_C(  2274),  INT16_C(  4816),  INT16_C( 21631),  INT16_C( 11406),
        -INT16_C( 28394), -INT16_C(  1756),  INT16_C(  4267), -INT16_C( 29452),  INT16_C(  3759),  INT16_C(  1197),  INT16_C(  8039),  INT16_C( 21742),
        -INT16_C(  1746),  INT16_C( 28886), -INT16_C(  5890),  INT16_C(  7453), -INT16_C(  4641),  INT16_C( 24367), -INT16_C( 16831),  INT16_C( 22667),
        -INT16_C( 20657), -INT16_C(  1455),  INT16_C( 17856),  INT16_C( 28551),  INT16_C( 13395), -INT16_C( 17549),  INT16_C( 24916), -INT16_C( 32241) },
      { -INT32_C(   656016028), -INT32_C(  1558277940),  INT32_C(    72351946),  INT32_C(  1627018574),  INT32_C(   653555680),  INT32_C(  1362572016), -INT32_C(  2070740315),  INT32_C(   236068119),
         INT32_C(   296550144), -INT32_C(   289606616),  INT32_C(   193709376),  INT32_C(  1235452181),  INT32_C(   327643653),  INT32_C(   654237569), -INT32_C(   123039263),  INT32_C(  2030550446) } },
    { {  INT32_C(  1509090906), -INT32_C(  1367994418),  INT32_C(  1041081852), -INT32_C(  1298753437),  INT32_C(   145614664),  INT32_C(  2138518572), -INT32_C(  1136989592), -INT32_C(  1505867188),
        -INT32_C(    16830416),  INT32_C(  1017935167),  INT32_C(  2121971995), -INT32_C(  1708060590),  INT32_C(   597876471),  INT32_C(  2057443602),  INT32_C(  1329061123),  INT32_C(  1475769639) },
      UINT16_C(62886),
      { -INT16_C(  6827),  INT16_C(   619), -INT16_C( 31199), -INT16_C( 25669),  INT16_C(  3332),  INT16_C( 13739), -INT16_C( 23897),  INT16_C( 18707),
         INT16_C(  9669),  INT16_C( 26722),  INT16_C( 26272), -INT16_C( 10427),  INT16_C( 27829), -INT16_C( 21684), -INT16_C(  3389),  INT16_C(  6305),
         INT16_C(  3288), -INT16_C(  1766), -INT16_C( 10606), -INT16_C( 26987),  INT16_C( 16611), -INT16_C( 29748), -INT16_C(  8221), -INT16_C( 22316),
         INT16_C( 14085), -INT16_C( 23280),  INT16_C( 21917),  INT16_C( 21116), -INT16_C( 14143), -INT16_C( 31490), -INT16_C( 24645), -INT16_C( 27747) },
      { -INT16_C( 18517),  INT16_C( 15756),  INT16_C(  8589),  INT16_C( 29139), -INT16_C( 24734),  INT16_C( 17916), -INT16_C( 12161), -INT16_C( 31507),
        -INT16_C(   505), -INT16_C( 23511), -INT16_C( 23213),  INT16_C(  5623), -INT16_C(  2707),  INT16_C( 10393),  INT16_C( 13972),  INT16_C( 16315),
         INT16_C( 18670),  INT16_C( 31612),  INT16_C( 20329), -INT16_C( 13332), -INT16_C(  5905),  INT16_C( 28176), -INT16_C(   327), -INT16_C( 16142),
         INT16_C(  7164),  INT16_C( 20325),  INT16_C( 23744),  INT16_C( 11620), -INT16_C(   431), -INT16_C(  6826),  INT16_C(  4404),  INT16_C(  8740) },
      {  INT32_C(  1509090906), -INT32_C(  1015937202),  INT32_C(   163734236), -INT32_C(  1298753437),  INT32_C(   145614664), -INT32_C(   668482957), -INT32_C(  1136989592),  INT32_C(    55514967),
         INT32_C(     5560168),  INT32_C(  1017935167), -INT32_C(   936267603), -INT32_C(  1708060590), -INT32_C(   372261060),  INT32_C(   765765168),  INT32_C(   221046373), -INT32_C(   351045360) } },
    { { -INT32_C(  1013014439), -INT32_C(   561083665),  INT32_C(   743219059), -INT32_C(  1712570723),  INT32_C(   434655577), -INT32_C(    28881491),  INT32_C(  2145623371),  INT32_C(   144836526),
        -INT32_C(  1748287321),  INT32_C(  1031100874), -INT32_C(  1788230920),  INT32_C(  1512986112),  INT32_C(  1433606055), -INT32_C(  1353467292),  INT32_C(   103692119), -INT32_C(   435236802) },
      UINT16_C(55568),
      { -INT16_C(  9347), -INT16_C(  3534),  INT16_C( 11032), -INT16_C( 32076), -INT16_C( 19008), -INT16_C(  4136),  INT16_C( 32527), -INT16_C( 32250),
         INT16_C( 27348),  INT16_C( 10301), -INT16_C( 27623),  INT16_C( 18271), -INT16_C( 25190), -INT16_C( 22504),  INT16_C( 10371),  INT16_C(   129),
        -INT16_C( 19453),  INT16_C(  7411), -INT16_C( 22561), -INT16_C( 24674),  INT16_C( 30300),  INT16_C( 27534), -INT16_C( 27403), -INT16_C( 13586),
         INT16_C( 11262),  INT16_C(  6130),  INT16_C( 20927),  INT16_C( 23135),  INT16_C( 30702),  INT16_C( 29186), -INT16_C( 31585), -INT16_C( 23694) },
      {  INT16_C( 25912),  INT16_C(  6079),  INT16_C( 23821),  INT16_C( 27062),  INT16_C( 17875), -INT16_C( 14123), -INT16_C( 15399), -INT16_C( 10094),
        -INT16_C( 31506), -INT16_C( 21009),  INT16_C( 20181), -INT16_C( 15353),  INT16_C(  2757),  INT16_C( 25910), -INT16_C( 22386), -INT16_C( 14840),
        -INT16_C( 14578),  INT16_C(  7133), -INT16_C( 27868), -INT16_C(  2172),  INT16_C( 23000), -INT16_C( 19777),  INT16_C( 21020),  INT16_C(  2698),
         INT16_C( 31190), -INT16_C( 21320), -INT16_C( 16440), -INT16_C( 29328), -INT16_C( 22839),  INT16_C( 22514), -INT16_C(  1458),  INT16_C( 23581) },
      { -INT32_C(  1013014439), -INT32_C(   561083665),  INT32_C(   743219059), -INT32_C(  1712570723), -INT32_C(  1078039797), -INT32_C(    28881491),  INT32_C(  2145623371),  INT32_C(   144836526),
         INT32_C(   336448497),  INT32_C(  1031100874), -INT32_C(  1788230920), -INT32_C(   612666088),  INT32_C(   220570180), -INT32_C(  1353467292), -INT32_C(    44109374), -INT32_C(   512677284) } },
    { { -INT32_C(   445121855),  INT32_C(  1725758606),  INT32_C(  1914215509), -INT32_C(   998464786), -INT32_C(   462408676), -INT32_C(  1116610316), -INT32_C(   720018298),  INT32_C(   540095070),
        -INT32_C(  1157256915), -INT32_C(    98442587),  INT32_C(  1819032190), -INT32_C(   131012132),  INT32_C(   299671837),  INT32_C(   147803777),  INT32_C(   282977458),  INT32_C(  1127222806) },
      UINT16_C(14007),
      {  INT16_C( 23806),  INT16_C(  8216), -INT16_C( 27049), -INT16_C( 15526),  INT16_C( 13826),  INT16_C( 12972), -INT16_C( 13777),  INT16_C(  3027),
         INT16_C( 21979), -INT16_C( 21927),  INT16_C(  2909),  INT16_C( 14990), -INT16_C( 23268),  INT16_C( 19528),  INT16_C(   232), -INT16_C(  6270),
        -INT16_C( 26020), -INT16_C( 19705),  INT16_C( 24880),  INT16_C( 12919),  INT16_C(  9111), -INT16_C( 14747),  INT16_C( 14573), -INT16_C( 13870),
         INT16_C( 11149), -INT16_C(  5517),  INT16_C(   567),  INT16_C( 21284),  INT16_C( 28071), -INT16_C( 28769),  INT16_C(  8813), -INT16_C( 13962) },
      {  INT16_C( 32188), -INT16_C(  4739), -INT16_C(  2850),  INT16_C( 30239), -INT16_C( 31721),  INT16_C(  1340),  INT16_C(  3773),  INT16_C( 19150),
         INT16_C( 16698),  INT16_C( 28981),  INT16_C( 22851), -INT16_C(  5436),  INT16_C( 25542),  INT16_C( 13178), -INT16_C(  3963),  INT16_C( 17149),
         INT16_C( 31342),  INT16_C( 19503),  INT16_C( 20078), -INT16_C( 31294), -INT16_C(    45), -INT16_C( 28534),  INT16_C( 22541),  INT16_C( 18394),
         INT16_C(  3994), -INT16_C(  8776),  INT16_C( 31849),  INT16_C( 12232),  INT16_C( 17120),  INT16_C( 25955),  INT16_C( 24626), -INT16_C( 24409) },
      {  INT32_C(   727331904), -INT32_C(   392401064), -INT32_C(   421192066), -INT32_C(   998464786), -INT32_C(   268461045), -INT32_C(    15012081), -INT32_C(   720018298), -INT32_C(   108443646),
        -INT32_C(  1157256915),  INT32_C(    95253454),  INT32_C(   420380903), -INT32_C(   131012132),  INT32_C(    92946298),  INT32_C(   278404271),  INT32_C(   282977458),  INT32_C(  1127222806) } },
    { {  INT32_C(  1223546586), -INT32_C(   120738011), -INT32_C(  1131915090),  INT32_C(  1241735856), -INT32_C(   618087310),  INT32_C(   403370040),  INT32_C(  1686007090), -INT32_C(  1492834867),
         INT32_C(   569373436),  INT32_C(  1343864225), -INT32_C(   989028075),  INT32_C(  1963986691),  INT32_C(    72366283),  INT32_C(  1511807784), -INT32_C(  1765893432), -INT32_C(  1136802880) },
      UINT16_C(11701),
      {  INT16_C( 22493), -INT16_C(  2326), -INT16_C(    89), -INT16_C( 19561), -INT16_C( 25916), -INT16_C( 11070), -INT16_C( 29168),  INT16_C( 24588),
         INT16_C( 13458), -INT16_C( 20805), -INT16_C( 31602),  INT16_C( 19785),  INT16_C(  2330),  INT16_C( 22288), -INT16_C( 14651), -INT16_C( 23932),
         INT16_C( 28189), -INT16_C( 15208),  INT16_C( 12141),  INT16_C( 12919),  INT16_C( 14793), -INT16_C(  9978),  INT16_C(  5063),  INT16_C( 22842),
        -INT16_C(  2745), -INT16_C( 10744),  INT16_C( 20857), -INT16_C( 27869),  INT16_C( 13146),  INT16_C(  8171),  INT16_C( 28665),  INT16_C(  5825) },
      {  INT16_C( 23006),  INT16_C( 19418),  INT16_C( 20872),  INT16_C( 20861), -INT16_C( 31605),  INT16_C( 21035),  INT16_C( 26007), -INT16_C(  8532),
        -INT16_C( 19366), -INT16_C( 11084), -INT16_C( 10491),  INT16_C( 24423),  INT16_C( 21003),  INT16_C(  1150),  INT16_C( 16322), -INT16_C( 24549),
        -INT16_C(  2664),  INT16_C(  8427),  INT16_C( 26951), -INT16_C( 11663), -INT16_C( 25363), -INT16_C( 31708), -INT16_C( 12287),  INT16_C( 23650),
         INT16_C(  6020), -INT16_C( 30416), -INT16_C( 26642), -INT16_C(  1560),  INT16_C( 26346), -INT16_C( 21250),  INT16_C(  6565),  INT16_C( 15692) },
      {  INT32_C(   472307690), -INT32_C(   120738011),  INT32_C(   586217730),  INT32_C(  1241735856), -INT32_C(    30025008),  INT32_C(   814745637),  INT32_C(  1686007090),  INT32_C(   348373046),
        -INT32_C(   203253312),  INT32_C(  1343864225), -INT32_C(    58812435),  INT32_C(   478004219),  INT32_C(    72366283), -INT32_C(   512196554), -INT32_C(  1765893432), -INT32_C(  1136802880) } },
    { {  INT32_C(  1432172302), -INT32_C(  1926770784),  INT32_C(  1829850219), -INT32_C(  1580633060),  INT32_C(  2032859531),  INT32_C(  2054362000),  INT32_C(   522613113), -INT32_C(  1738771830),
         INT32_C(  1257159338), -INT32_C(   187165303),  INT32_C(  2120345953), -INT32_C(   400610723), -INT32_C(  1268627165), -INT32_C(   701573796), -INT32_C(   789228218),  INT32_C(  1902662087) },
      UINT16_C(22027),
      { -INT16_C( 27460), -INT16_C( 27540), -INT16_C( 12919), -INT16_C(  5507), -INT16_C(  9397),  INT16_C( 27157),  INT16_C( 14531),  INT16_C(  9652),
         INT16_C(  4332),  INT16_C(  7162),  INT16_C( 16614), -INT16_C(  9360),  INT16_C( 14096),  INT16_C( 31021),  INT16_C( 14505),  INT16_C( 26063),
         INT16_C( 15309),  INT16_C( 22265),  INT16_C( 30217),  INT16_C( 21568),  INT16_C( 21841),  INT16_C(  5567),  INT16_C( 29582),  INT16_C( 31290),
         INT16_C( 13699),  INT16_C( 27285),  INT16_C(  1397), -INT16_C( 31163),  INT16_C( 29245), -INT16_C(  6401), -INT16_C( 12629),  INT16_C( 30795) },
      {  INT16_C( 17418),  INT16_C(  5070),  INT16_C(  3770),  INT16_C(  3175),  INT16_C(  9828), -INT16_C(  3551),  INT16_C( 23449),  INT16_C(  7532),
         INT16_C(   656),  INT16_C(  1671), -INT16_C( 13305),  INT16_C( 17548), -INT16_C( 29889), -INT16_C(  5590),  INT16_C( 30041),  INT16_C( 25442),
         INT16_C( 12473),  INT16_C( 29814), -INT16_C(  8642), -INT16_C( 23936), -INT16_C( 24316), -INT16_C( 24940),  INT16_C(   508), -INT16_C( 29253),
         INT16_C( 16899),  INT16_C(  2707),  INT16_C(  7950),  INT16_C( 19791),  INT16_C( 31146),  INT16_C(   823), -INT16_C( 26129), -INT16_C( 22425) },
      { -INT32_C(   617926080), -INT32_C(    66189355),  INT32_C(  1829850219),  INT32_C(   413436283),  INT32_C(  2032859531),  INT32_C(  2054362000),  INT32_C(   522613113), -INT32_C(  1738771830),
         INT32_C(  1257159338), -INT32_C(   777386962), -INT32_C(   669926736), -INT32_C(   400610723),  INT32_C(   305359896), -INT32_C(   701573796),  INT32_C(   905596747),  INT32_C(  1902662087) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i src = simde_mm512_loadu_epi16(test_vec[i].src);
    simde__m512i a = simde_mm512_loadu_epi16(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi16(test_vec[i].b);
    simde__m512i r = simde_mm512_mask_madd_epi16(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i16x32(r, simde_mm512_loadu_epi16(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512i src = simde_test_x86_random_i32x16();
    simde__mmask16 k = simde_test_x86_random_mmask16();
    simde__m512i a = simde_test_x86_random_i16x32();
    simde__m512i b = simde_test_x86_random_i16x32();
    simde__m512i r = simde_mm512_mask_madd_epi16(src, k, a, b);

    simde_test_x86_write_i32x16(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask16(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i16x32(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i16x32(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_maskz_madd_epi16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask16 k;
    const int16_t a[32];
    const int16_t b[32];
    const int32_t r[16];
  } test_vec[] = {
    { UINT16_C(49133),
      { -INT16_C( 12436),  INT16_C(  8143),  INT16_C(  3509), -INT16_C(  8650),  INT16_C( 19172), -INT16_C( 17007), -INT16_C( 26437),  INT16_C( 27681),
        -INT16_C(  6686), -INT16_C( 11200), -INT16_C( 18005), -INT16_C(  6237), -INT16_C( 26567), -INT16_C(  3727),  INT16_C( 24180), -INT16_C(  7760),
         INT16_C( 32558), -INT16_C(  7424),  INT16_C( 14220),  INT16_C( 29121),  INT16_C( 21121),  INT16_C( 15406),  INT16_C( 20459), -INT16_C( 12887),
        -INT16_C(  5836), -INT16_C(  8031),  INT16_C( 17826), -INT16_C(  9273),  INT16_C( 14557),  INT16_C( 20940),  INT16_C( 31894), -INT16_C( 15310) },
      {  INT16_C( 13308), -INT16_C( 30553),  INT16_C( 26986), -INT16_C(  5127),  INT16_C( 10171), -INT16_C( 23000), -INT16_C( 11913), -INT16_C( 21645),
         INT16_C(  5562),  INT16_C( 23947),  INT16_C( 21082),  INT16_C( 14136),  INT16_C(  1418),  INT16_C(  8584), -INT16_C( 17535),  INT16_C( 32229),
        -INT16_C( 29202),  INT16_C( 22534), -INT16_C(    10), -INT16_C( 20157),  INT16_C( 27431), -INT16_C( 25000), -INT16_C( 13508), -INT16_C(  2231),
        -INT16_C( 10784),  INT16_C( 14932), -INT16_C( 29657), -INT16_C( 19855), -INT16_C(  1391),  INT16_C(  5075), -INT16_C( 18251), -INT16_C( 23664) },
      { -INT32_C(   414291367),  INT32_C(           0),  INT32_C(   586159412), -INT32_C(   284211264),  INT32_C(           0), -INT32_C(   467747642), -INT32_C(    69664574), -INT32_C(   674093340),
        -INT32_C(  1118051132), -INT32_C(   587134197),  INT32_C(   194220151), -INT32_C(   247609275), -INT32_C(    56983468), -INT32_C(   344550267),  INT32_C(           0), -INT32_C(   219801554) } },
    { UINT16_C(38469),
      {  INT16_C( 15355),  INT16_C( 16022), -INT16_C( 16915),  INT16_C( 17834), -INT16_C(  6565), -INT16_C( 23536), -INT16_C(  3619),  INT16_C( 12665),
        -INT16_C( 24277), -INT16_C( 25154),  INT16_C( 20307),  INT16_C(  9879),  INT16_C( 19554), -INT16_C(  3106),  INT16_C(  9455), -INT16_C(  5495),
         INT16_C(  8031),  INT16_C( 19496), -INT16_C( 11556),  INT16_C( 14225), -INT16_C( 23879), -INT16_C( 26916),  INT16_C( 21907), -INT16_C( 16696),
        -INT16_C( 30986),  INT16_C( 18779), -INT16_C(  3371),  INT16_C( 14447),  INT16_C( 20030),  INT16_C( 11563), -INT16_C( 19342), -INT16_C( 12009) },
      {  INT16_C( 16596), -INT16_C( 20450), -INT16_C( 20718), -INT16_C( 13336), -INT16_C( 15279), -INT16_C(  7070),  INT16_C( 10777),  INT16_C(  4259),
        -INT16_C(   336), -INT16_C( 31399), -INT16_C( 13839),  INT16_C( 12221), -INT16_C(  6121), -INT16_C( 30371),  INT16_C( 29853),  INT16_C( 29018),
         INT16_C( 30900), -INT16_C( 14559),  INT16_C(  2344),  INT16_C( 31122), -INT16_C(  2867), -INT16_C(  6306),  INT16_C(   286), -INT16_C( 12553),
         INT16_C( 20735), -INT16_C(  4012),  INT16_C(  4377),  INT16_C( 12320),  INT16_C( 32250), -INT16_C( 26695),  INT16_C(  5361), -INT16_C( 23032) },
      { -INT32_C(    72818320),  INT32_C(           0),  INT32_C(   266706155),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(    25357708),  INT32_C(           0),
         INT32_C(           0),  INT32_C(   415623186),  INT32_C(   238193389),  INT32_C(           0), -INT32_C(   717836058),  INT32_C(           0),  INT32_C(           0),  INT32_C(   172898826) } },
    { UINT16_C(10636),
      { -INT16_C( 19347), -INT16_C(   205),  INT16_C(    46), -INT16_C( 29452),  INT16_C(  4839), -INT16_C(  8563), -INT16_C( 29471),  INT16_C( 13615),
         INT16_C( 18557), -INT16_C( 25274),  INT16_C( 16505),  INT16_C( 12826),  INT16_C(  3031), -INT16_C(  8378), -INT16_C( 11343),  INT16_C(  7689),
         INT16_C( 15495), -INT16_C( 19170),  INT16_C(  4668),  INT16_C(  9281), -INT16_C( 12764),  INT16_C(  1282),  INT16_C( 12635), -INT16_C( 10182),
        -INT16_C( 32390), -INT16_C(  3211), -INT16_C( 28735), -INT16_C( 26331),  INT16_C( 27802),  INT16_C( 19576), -INT16_C( 32449), -INT16_C( 14742) },
      { -INT16_C( 30531), -INT16_C(  1412), -INT16_C( 16998), -INT16_C( 16610),  INT16_C(  8332), -INT16_C(  6204), -INT16_C(   174), -INT16_C( 13121),
         INT16_C( 13440),  INT16_C( 16831), -INT16_C(  6973),  INT16_C( 24026),  INT16_C( 21328), -INT16_C( 28759),  INT16_C(  5332), -INT16_C( 28074),
        -INT16_C( 11620),  INT16_C( 14220), -INT16_C( 21873),  INT16_C(  7158), -INT16_C( 17718),  INT16_C(  7170), -INT16_C( 15943),  INT16_C( 14824),
        -INT16_C( 22539), -INT16_C( 18309),  INT16_C( 21900), -INT16_C(  9194), -INT16_C( 16472),  INT16_C( 32108), -INT16_C( 15661),  INT16_C( 28687) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(    93443400), -INT32_C(   173514461),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(   276341862),
        -INT32_C(   452649300),  INT32_C(           0),  INT32_C(           0), -INT32_C(   352377773),  INT32_C(           0), -INT32_C(   387209286),  INT32_C(           0),  INT32_C(           0) } },
    { UINT16_C(39828),
      {  INT16_C(  9127), -INT16_C( 25275),  INT16_C(  3903),  INT16_C( 16727),  INT16_C(  4396),  INT16_C(  5123), -INT16_C(  1974), -INT16_C( 14916),
         INT16_C( 18609), -INT16_C( 14565), -INT16_C( 15580), -INT16_C( 28538),  INT16_C( 23104),  INT16_C( 20306), -INT16_C(  6454),  INT16_C( 29162),
         INT16_C( 12042),  INT16_C( 18702),  INT16_C( 25919),  INT16_C( 27530), -INT16_C( 29322), -INT16_C( 16001),  INT16_C( 15238),  INT16_C( 14214),
        -INT16_C( 24189), -INT16_C( 22274), -INT16_C( 31643), -INT16_C( 23240), -INT16_C( 29730), -INT16_C( 22283), -INT16_C(  8335),  INT16_C( 31513) },
      {  INT16_C(  9999),  INT16_C( 20164),  INT16_C( 20365),  INT16_C(   953),  INT16_C( 14556),  INT16_C( 25284),  INT16_C( 19316), -INT16_C(  2151),
        -INT16_C( 26644),  INT16_C( 20895), -INT16_C( 10212), -INT16_C(  1289), -INT16_C(  5021), -INT16_C( 11101), -INT16_C( 17205), -INT16_C(  9648),
         INT16_C(  5348),  INT16_C( 28968), -INT16_C(  7837),  INT16_C( 16500),  INT16_C( 14618), -INT16_C( 29022),  INT16_C( 15492),  INT16_C( 28805),
         INT16_C(  9683), -INT16_C(  4158), -INT16_C( 17923),  INT16_C( 24810), -INT16_C( 29275),  INT16_C( 28724), -INT16_C( 31671),  INT16_C( 11595) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(   193518108),  INT32_C(           0), -INT32_C(   800153871),  INT32_C(           0),  INT32_C(           0), -INT32_C(   170313906),
         INT32_C(   606160152),  INT32_C(   251117797),  INT32_C(           0),  INT32_C(   645501366), -INT32_C(   141606795),  INT32_C(           0),  INT32_C(           0),  INT32_C(   629371020) } },
    { UINT16_C(29593),
      { -INT16_C(   866),  INT16_C(  4949),  INT16_C( 28476), -INT16_C(  8372), -INT16_C( 12035), -INT16_C( 32229), -INT16_C(  4544),  INT16_C(   679),
        -INT16_C( 23330), -INT16_C( 14149),  INT16_C( 24580),  INT16_C( 14677), -INT16_C( 24879),  INT16_C(  7357),  INT16_C( 22220),  INT16_C( 27279),
        -INT16_C(  7085), -INT16_C( 28803), -INT16_C( 13997),  INT16_C( 20590), -INT16_C( 30311), -INT16_C(  9517),  INT16_C( 31352),  INT16_C( 22236),
        -INT16_C( 26593),  INT16_C(  8990),  INT16_C( 29688), -INT16_C( 13988),  INT16_C(  6673), -INT16_C(  8731),  INT16_C( 30064), -INT16_C( 15544) },
      { -INT16_C( 15015), -INT16_C( 21165), -INT16_C( 15985),  INT16_C( 10493), -INT16_C( 12213), -INT16_C( 15614), -INT16_C(  8373),  INT16_C( 27161),
         INT16_C( 14199),  INT16_C( 28557), -INT16_C(  5462), -INT16_C( 17607),  INT16_C(  7684),  INT16_C( 29849), -INT16_C(  7789), -INT16_C(  4808),
        -INT16_C( 29786),  INT16_C( 13722), -INT16_C( 26804), -INT16_C( 26786),  INT16_C( 24680), -INT16_C( 19622),  INT16_C( 29503), -INT16_C( 18915),
        -INT16_C( 21846),  INT16_C( 21542),  INT16_C( 24468), -INT16_C( 26608), -INT16_C( 22147),  INT16_C(  4365),  INT16_C( 17802),  INT16_C( 12542) },
      { -INT32_C(    91742595),  INT32_C(           0),  INT32_C(           0),  INT32_C(    56489231), -INT32_C(   735315663),  INT32_C(           0),  INT32_C(           0), -INT32_C(   304229012),
        -INT32_C(   184200956), -INT32_C(   176348152),  INT32_C(           0),  INT32_C(           0),  INT32_C(   774613258),  INT32_C(  1098598688), -INT32_C(   185897746),  INT32_C(           0) } },
    { UINT16_C(39120),
      {  INT16_C(  7270), -INT16_C( 15313), -INT16_C( 26700),  INT16_C(  3620),  INT16_C( 25674),  INT16_C( 26498),  INT16_C( 11290),  INT16_C( 16402),
        -INT16_C( 22911), -INT16_C( 28257),  INT16_C(  7487),  INT16_C( 19514), -INT16_C( 15314),  INT16_C( 11409),  INT16_C( 25076),  INT16_C( 23236),
        -INT16_C(  3203),  INT16_C( 12574),  INT16_C( 17291), -INT16_C( 10944), -INT16_C( 15705), -INT16_C( 16067),  INT16_C( 20462),  INT16_C( 28418),
        -INT16_C( 24075),  INT16_C( 13312),  INT16_C( 15038), -INT16_C(  4992),  INT16_C(  4606), -INT16_C(  3304), -INT16_C(  9102), -INT16_C(  4019) },
      {  INT16_C( 27856),  INT16_C( 23329),  INT16_C( 25007),  INT16_C( 22064),  INT16_C( 27939),  INT16_C(  4631),  INT16_C(  6588), -INT16_C( 19839),
        -INT16_C( 32069),  INT16_C( 31206),  INT16_C( 26556), -INT16_C( 17562),  INT16_C( 32376), -INT16_C(  5202), -INT16_C(  1189),  INT16_C( 11227),
        -INT16_C(   921),  INT16_C(  5766), -INT16_C( 18850), -INT16_C( 32404), -INT16_C( 31708), -INT16_C(  8045),  INT16_C(  5533),  INT16_C( 22674),
         INT16_C( 31127),  INT16_C( 21458),  INT16_C( 14560),  INT16_C( 22542), -INT16_C( 17226),  INT16_C(  4419),  INT16_C(  7864),  INT16_C(  7996) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(   147055083),  INT32_C(           0), -INT32_C(   555155682),  INT32_C(   231055208),
         INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(   757565978), -INT32_C(   463733629),  INT32_C(           0),  INT32_C(           0), -INT32_C(   103714052) } },
    { UINT16_C(49691),
      {  INT16_C( 31030), -INT16_C( 23943), -INT16_C( 25094), -INT16_C( 29146), -INT16_C( 15235),  INT16_C(  4259),  INT16_C( 14876), -INT16_C(  4471),
         INT16_C( 27021), -INT16_C( 25562), -INT16_C(  8767),  INT16_C(  1368),  INT16_C(  4334),  INT16_C( 11043),  INT16_C( 15920),  INT16_C( 26349),
         INT16_C( 26295), -INT16_C( 19960),  INT16_C( 12035), -INT16_C( 32448), -INT16_C(  7181),  INT16_C(  3985),  INT16_C(  6685), -INT16_C( 21762),
         INT16_C(  9347),  INT16_C( 17478), -INT16_C( 24831), -INT16_C(  4023),  INT16_C( 28079), -INT16_C(  8421),  INT16_C(  2219),  INT16_C( 25413) },
      {  INT16_C( 20079),  INT16_C( 29205),  INT16_C( 21885),  INT16_C( 28915), -INT16_C( 31688),  INT16_C( 21887),  INT16_C( 32158),  INT16_C(  8703),
         INT16_C( 18082), -INT16_C( 23706), -INT16_C( 20507), -INT16_C( 27501), -INT16_C( 20964), -INT16_C( 14220), -INT16_C( 17993),  INT16_C(  9771),
         INT16_C( 16391), -INT16_C( 31592), -INT16_C( 29547), -INT16_C( 12812),  INT16_C( 29712), -INT16_C( 20702),  INT16_C(  8689), -INT16_C( 27696),
         INT16_C( 13927),  INT16_C( 19511), -INT16_C( 13594),  INT16_C(   737),  INT16_C( 21881),  INT16_C( 12490), -INT16_C(  2802),  INT16_C(  5718) },
      { -INT32_C(    76203945), -INT32_C(  1391938780),  INT32_C(           0),  INT32_C(   439471295),  INT32_C(  1094566494),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),
         INT32_C(           0),  INT32_C(    60125631),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(   509218309),  INT32_C(   139093896) } },
    { UINT16_C(60981),
      { -INT16_C( 13670), -INT16_C( 28806), -INT16_C( 29801), -INT16_C( 18173), -INT16_C(  3014),  INT16_C(  2779),  INT16_C( 17032), -INT16_C( 16575),
         INT16_C( 10127),  INT16_C( 28809),  INT16_C(   553), -INT16_C(  2875), -INT16_C( 11470), -INT16_C( 30487),  INT16_C(  8169), -INT16_C( 31625),
        -INT16_C(  3607), -INT16_C( 32493),  INT16_C(  5756), -INT16_C( 18886),  INT16_C(  5386), -INT16_C( 27967),  INT16_C(   600), -INT16_C(  6319),
        -INT16_C(  9431),  INT16_C( 21079),  INT16_C(  7389),  INT16_C(  4166),  INT16_C( 12527), -INT16_C(  9832),  INT16_C(  3919),  INT16_C( 14429) },
      {  INT16_C( 28673),  INT16_C( 32185), -INT16_C(  2938), -INT16_C( 28620), -INT16_C(  2807),  INT16_C( 24867),  INT16_C( 29943),  INT16_C(  8264),
        -INT16_C( 24753),  INT16_C( 11634), -INT16_C( 17989), -INT16_C( 21699), -INT16_C( 10775),  INT16_C( 14468), -INT16_C(  7707), -INT16_C(  6544),
         INT16_C( 10833), -INT16_C( 10397), -INT16_C( 26850),  INT16_C( 10087), -INT16_C( 30068), -INT16_C( 31863), -INT16_C( 11777),  INT16_C( 20131),
         INT16_C(  5745),  INT16_C( 11387), -INT16_C( 18225), -INT16_C( 18217),  INT16_C( 23438),  INT16_C( 29680),  INT16_C( 24636), -INT16_C( 29351) },
      { -INT32_C(  1319081020),  INT32_C(           0),  INT32_C(    77565691),  INT32_C(           0),  INT32_C(    84490275),  INT32_C(    52436708),  INT32_C(           0),  INT32_C(           0),
         INT32_C(           0), -INT32_C(   345051682),  INT32_C(   729166273), -INT32_C(   134273989),  INT32_C(           0), -INT32_C(   210556547),  INT32_C(     1794066), -INT32_C(   326957095) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi16(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi16(test_vec[i].b);
    simde__m512i r = simde_mm512_maskz_madd_epi16(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

SIMDE_TEST_FUNC_LIST_BEGIN
  SIMDE_TEST_FUNC_LIST_ENTRY(mm_mask_madd_epi16)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm_maskz_madd_epi16)

  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_mask_madd_epi16)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_maskz_madd_epi16)

  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_madd_epi16)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_madd_epi16)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_madd_epi16)
SIMDE_TEST_FUNC_LIST_END

#include <test/x86/avx512/test-avx512-footer.h>
