/*
 * @(#)BarrelU.c
 *
 * Copyright 2003 - 2008  David A. Bagley, bagleyd@tux.org
 *
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the author not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * This program is distributed in the hope that it will be "useful",
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */

/* Undo algorithm for Barrel */

#include "BarrelP.h"

static int startSpace[PLUNGERS_NUMBER];
int *startPosition = NULL;

static void
newStack(BarrelStack *s)
{
	if (s->lastMove != NULL || s->firstMove != NULL)
		return;
	if (!(s->lastMove = (MoveStack *) malloc(sizeof (MoveStack)))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}
	if (!(s->firstMove = (MoveStack *) malloc(sizeof (MoveStack)))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}
	s->firstMove->previous = s->lastMove->next = NULL;
	s->firstMove->next = s->lastMove;
	s->lastMove->previous = s->firstMove;
	s->count = 0;
}

static void
pushStack(BarrelStack *s, moveRecord **move)
{
	if (!(s->currMove = (MoveStack *) malloc(sizeof (MoveStack)))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}
	s->lastMove->previous->next = s->currMove;
	s->currMove->previous = s->lastMove->previous;
	s->currMove->next = s->lastMove;
	s->lastMove->previous = s->currMove;
	*move = &(s->currMove->move);
	s->count++;
}

static void
popStack(BarrelStack *s)
{
	s->currMove = s->lastMove->previous;
	s->lastMove->previous->previous->next = s->lastMove;
	s->lastMove->previous = s->lastMove->previous->previous;
	free(s->currMove);
	s->count--;
}

static moveRecord *
topStack(BarrelStack *s)
{
	return &(s->lastMove->previous->move);
}

static int
emptyStack(BarrelStack *s)
{
	return (s->lastMove->previous == s->firstMove);
}

static void
flushStack(BarrelStack *s)
{
	while (s->lastMove->previous != s->firstMove) {
		s->currMove = s->lastMove->previous;
		s->lastMove->previous->previous->next = s->lastMove;
		s->lastMove->previous = s->lastMove->previous->previous;
		free(s->currMove);
	}
	s->count = 0;
}

static void
deleteStack(BarrelStack *s)
{
	flushStack(s);
	if (s->firstMove) {
		free(s->firstMove);
		s->firstMove = NULL;
	}
	if (s->lastMove) {
		free(s->lastMove);
		s->lastMove = NULL;
	}
}

/**********************************/

void
newMoves(BarrelStack *s)
{
	newStack(s);
}

void
deleteMoves(BarrelStack *s)
{
	deleteStack(s);
}

static void
writeMove(moveRecord *move, int direction, int tile, int control)
{
#if 0
	move->direction = direction;
	move->control = control;
#endif
	move->packed = ((control & 0xF) << 4) + (direction & 0xF);
	move->tile = tile;
}

static void
readMove(moveRecord *move, int *direction, int *tile, int *control)
{
#if 0
	*direction = move->direction;
	*control = move->control;
#endif
	*direction = (int) (move->packed & 0xF);
	*control = (int) ((move->packed >> 4) & 0xF);
	*tile = move->tile;
}

void
setMove(BarrelStack *s, int direction, int tile, int control)
{
	moveRecord *move;

	pushStack(s, &move);
	writeMove(move, direction, tile, control);
}

void
getMove(BarrelStack *s, int *direction, int *tile, int *control)
{
	readMove(topStack(s), direction, tile, control);
	popStack(s);
}

int
madeMoves(BarrelStack *s)
{
	return !emptyStack(s);
}

void
flushMoves(BarrelWidget w, BarrelStack *s, Boolean undo)
{
	int i;

	flushStack(s);
	if (undo) {
		for (i = 0; i < PLUNGERS_NUMBER; i++)
			startSpace[i] = w->barrel.spacePosition[i];
		startPosition[w->barrel.tileFaces - 1] = 0;
		for (i = 0; i < w->barrel.tileFaces; i++)
			startPosition[i] = w->barrel.tileOfPosition[i];
	}
}

int
numMoves(BarrelStack *s)
{
	return s->count;
}

void
scanMoves(FILE *fp, BarrelWidget w, int moves)
{
	int direction, tile, shift, l, c;

	for (l = 0; l < moves; l++) {
		while ((c = getc(fp)) != EOF && c != SYMBOL);
		(void) fscanf(fp, "%d %d %d", &direction, &tile, &shift);
		if (!movePuzzle(w, direction, tile, shift, False, INSTANT))
			(void) fprintf(stderr,
				"%d move in direction %d, tile %d, shift %d, can not be made.",
				l, direction, tile, shift);
	}
}

void
printMoves(FILE *fp, BarrelStack *s)
{
	int direction, tile, control, counter = 0;

	s->currMove = s->firstMove->next;
	(void) fprintf(fp, "moves\tdir\ttile\tshift\n");
	while (s->currMove != s->lastMove) {
		readMove(&(s->currMove->move), &direction, &tile, &control);
		(void) fprintf(fp, "%d%c\t%d\t%d\t%d\n",
			++counter, SYMBOL, direction, tile, control);
		s->currMove = s->currMove->next;
	}
}

void
scanStartPosition(FILE *fp, BarrelWidget w)
{
	int i, num, c;

	while ((c = getc(fp)) != EOF && c != SYMBOL);
	for (i = 0; i < w->barrel.tileFaces; i++) {
		(void) fscanf(fp, "%d ", &num);
		startPosition[i] = num;
		if (num < 0)
			startSpace[num + PLUNGERS_NUMBER] = i;
	}
}

void
printStartPosition(FILE *fp, BarrelWidget w)
{
	int i;

	(void) fprintf(fp, "\nstartingPosition%c\n", SYMBOL);
	for (i = 0; i < w->barrel.tileFaces; i++) {
		(void) fprintf(fp, "%3d ", startPosition[i]);
		if (!((i + 1) % w->barrel.tiles))
			(void) fprintf(fp, "\n");
	}
	(void) fprintf(fp, "\n");
}

void
setStartPosition(BarrelWidget w)
{
	int i;

	for (i = 0; i < PLUNGERS_NUMBER; i++)
			w->barrel.spacePosition[i] = startSpace[i];
	for (i = 0; i < w->barrel.tileFaces; i++)
		w->barrel.tileOfPosition[i] = startPosition[i];
	drawAllTiles(w);
}
