/******************************************************************************
 *
 *  Copyright 2022 Google LLC
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at:
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 ******************************************************************************/

#include "tables.h"


/**
 * Twiddles FFT 3 points
 *
 * T[0..N-1] =
 *   { cos(-2Pi *  i/N) + j sin(-2Pi *  i/N),
 *     cos(-2Pi * 2i/N) + j sin(-2Pi * 2i/N) } , N=15, 45
 */

static const struct lc3_fft_bf3_twiddles fft_twiddles_15 = {
    .n3 = 15/3, .t = (const struct lc3_complex [][2]){
        { {  1.0000000e+0, -0.0000000e+0 }, {  1.0000000e+0, -0.0000000e+0 } },
        { {  9.1354546e-1, -4.0673664e-1 }, {  6.6913061e-1, -7.4314483e-1 } },
        { {  6.6913061e-1, -7.4314483e-1 }, { -1.0452846e-1, -9.9452190e-1 } },
        { {  3.0901699e-1, -9.5105652e-1 }, { -8.0901699e-1, -5.8778525e-1 } },
        { { -1.0452846e-1, -9.9452190e-1 }, { -9.7814760e-1,  2.0791169e-1 } },
        { { -5.0000000e-1, -8.6602540e-1 }, { -5.0000000e-1,  8.6602540e-1 } },
        { { -8.0901699e-1, -5.8778525e-1 }, {  3.0901699e-1,  9.5105652e-1 } },
        { { -9.7814760e-1, -2.0791169e-1 }, {  9.1354546e-1,  4.0673664e-1 } },
        { { -9.7814760e-1,  2.0791169e-1 }, {  9.1354546e-1, -4.0673664e-1 } },
        { { -8.0901699e-1,  5.8778525e-1 }, {  3.0901699e-1, -9.5105652e-1 } },
        { { -5.0000000e-1,  8.6602540e-1 }, { -5.0000000e-1, -8.6602540e-1 } },
        { { -1.0452846e-1,  9.9452190e-1 }, { -9.7814760e-1, -2.0791169e-1 } },
        { {  3.0901699e-1,  9.5105652e-1 }, { -8.0901699e-1,  5.8778525e-1 } },
        { {  6.6913061e-1,  7.4314483e-1 }, { -1.0452846e-1,  9.9452190e-1 } },
        { {  9.1354546e-1,  4.0673664e-1 }, {  6.6913061e-1,  7.4314483e-1 } },
    }
};

static const struct lc3_fft_bf3_twiddles fft_twiddles_45 = {
    .n3 = 45/3, .t = (const struct lc3_complex [][2]){
        { {  1.0000000e+0, -0.0000000e+0 }, {  1.0000000e+0, -0.0000000e+0 } },
        { {  9.9026807e-1, -1.3917310e-1 }, {  9.6126170e-1, -2.7563736e-1 } },
        { {  9.6126170e-1, -2.7563736e-1 }, {  8.4804810e-1, -5.2991926e-1 } },
        { {  9.1354546e-1, -4.0673664e-1 }, {  6.6913061e-1, -7.4314483e-1 } },
        { {  8.4804810e-1, -5.2991926e-1 }, {  4.3837115e-1, -8.9879405e-1 } },
        { {  7.6604444e-1, -6.4278761e-1 }, {  1.7364818e-1, -9.8480775e-1 } },
        { {  6.6913061e-1, -7.4314483e-1 }, { -1.0452846e-1, -9.9452190e-1 } },
        { {  5.5919290e-1, -8.2903757e-1 }, { -3.7460659e-1, -9.2718385e-1 } },
        { {  4.3837115e-1, -8.9879405e-1 }, { -6.1566148e-1, -7.8801075e-1 } },
        { {  3.0901699e-1, -9.5105652e-1 }, { -8.0901699e-1, -5.8778525e-1 } },
        { {  1.7364818e-1, -9.8480775e-1 }, { -9.3969262e-1, -3.4202014e-1 } },
        { {  3.4899497e-2, -9.9939083e-1 }, { -9.9756405e-1, -6.9756474e-2 } },
        { { -1.0452846e-1, -9.9452190e-1 }, { -9.7814760e-1,  2.0791169e-1 } },
        { { -2.4192190e-1, -9.7029573e-1 }, { -8.8294759e-1,  4.6947156e-1 } },
        { { -3.7460659e-1, -9.2718385e-1 }, { -7.1933980e-1,  6.9465837e-1 } },
        { { -5.0000000e-1, -8.6602540e-1 }, { -5.0000000e-1,  8.6602540e-1 } },
        { { -6.1566148e-1, -7.8801075e-1 }, { -2.4192190e-1,  9.7029573e-1 } },
        { { -7.1933980e-1, -6.9465837e-1 }, {  3.4899497e-2,  9.9939083e-1 } },
        { { -8.0901699e-1, -5.8778525e-1 }, {  3.0901699e-1,  9.5105652e-1 } },
        { { -8.8294759e-1, -4.6947156e-1 }, {  5.5919290e-1,  8.2903757e-1 } },
        { { -9.3969262e-1, -3.4202014e-1 }, {  7.6604444e-1,  6.4278761e-1 } },
        { { -9.7814760e-1, -2.0791169e-1 }, {  9.1354546e-1,  4.0673664e-1 } },
        { { -9.9756405e-1, -6.9756474e-2 }, {  9.9026807e-1,  1.3917310e-1 } },
        { { -9.9756405e-1,  6.9756474e-2 }, {  9.9026807e-1, -1.3917310e-1 } },
        { { -9.7814760e-1,  2.0791169e-1 }, {  9.1354546e-1, -4.0673664e-1 } },
        { { -9.3969262e-1,  3.4202014e-1 }, {  7.6604444e-1, -6.4278761e-1 } },
        { { -8.8294759e-1,  4.6947156e-1 }, {  5.5919290e-1, -8.2903757e-1 } },
        { { -8.0901699e-1,  5.8778525e-1 }, {  3.0901699e-1, -9.5105652e-1 } },
        { { -7.1933980e-1,  6.9465837e-1 }, {  3.4899497e-2, -9.9939083e-1 } },
        { { -6.1566148e-1,  7.8801075e-1 }, { -2.4192190e-1, -9.7029573e-1 } },
        { { -5.0000000e-1,  8.6602540e-1 }, { -5.0000000e-1, -8.6602540e-1 } },
        { { -3.7460659e-1,  9.2718385e-1 }, { -7.1933980e-1, -6.9465837e-1 } },
        { { -2.4192190e-1,  9.7029573e-1 }, { -8.8294759e-1, -4.6947156e-1 } },
        { { -1.0452846e-1,  9.9452190e-1 }, { -9.7814760e-1, -2.0791169e-1 } },
        { {  3.4899497e-2,  9.9939083e-1 }, { -9.9756405e-1,  6.9756474e-2 } },
        { {  1.7364818e-1,  9.8480775e-1 }, { -9.3969262e-1,  3.4202014e-1 } },
        { {  3.0901699e-1,  9.5105652e-1 }, { -8.0901699e-1,  5.8778525e-1 } },
        { {  4.3837115e-1,  8.9879405e-1 }, { -6.1566148e-1,  7.8801075e-1 } },
        { {  5.5919290e-1,  8.2903757e-1 }, { -3.7460659e-1,  9.2718385e-1 } },
        { {  6.6913061e-1,  7.4314483e-1 }, { -1.0452846e-1,  9.9452190e-1 } },
        { {  7.6604444e-1,  6.4278761e-1 }, {  1.7364818e-1,  9.8480775e-1 } },
        { {  8.4804810e-1,  5.2991926e-1 }, {  4.3837115e-1,  8.9879405e-1 } },
        { {  9.1354546e-1,  4.0673664e-1 }, {  6.6913061e-1,  7.4314483e-1 } },
        { {  9.6126170e-1,  2.7563736e-1 }, {  8.4804810e-1,  5.2991926e-1 } },
        { {  9.9026807e-1,  1.3917310e-1 }, {  9.6126170e-1,  2.7563736e-1 } },
    }
};

const struct lc3_fft_bf3_twiddles *lc3_fft_twiddles_bf3[] =
    { &fft_twiddles_15, &fft_twiddles_45 };


/**
 * Twiddles FFT 2 points
 *
 * T[0..N/2-1] =
 *   cos(-2Pi * i/N) + j sin(-2Pi * i/N) , N=10, 20, ...
 */

static const struct lc3_fft_bf2_twiddles fft_twiddles_10 = {
    .n2 = 10/2, .t = (const struct lc3_complex []){
        {  1.0000000e+00, -0.0000000e+00 }, {  8.0901699e-01, -5.8778525e-01 },
        {  3.0901699e-01, -9.5105652e-01 }, { -3.0901699e-01, -9.5105652e-01 },
        { -8.0901699e-01, -5.8778525e-01 },
    }
};

static const struct lc3_fft_bf2_twiddles fft_twiddles_20 = {
    .n2 = 20/2, .t = (const struct lc3_complex []){
        {  1.0000000e+00, -0.0000000e+00 }, {  9.5105652e-01, -3.0901699e-01 },
        {  8.0901699e-01, -5.8778525e-01 }, {  5.8778525e-01, -8.0901699e-01 },
        {  3.0901699e-01, -9.5105652e-01 }, {  6.1232340e-17, -1.0000000e+00 },
        { -3.0901699e-01, -9.5105652e-01 }, { -5.8778525e-01, -8.0901699e-01 },
        { -8.0901699e-01, -5.8778525e-01 }, { -9.5105652e-01, -3.0901699e-01 },
    }
};

static const struct lc3_fft_bf2_twiddles fft_twiddles_30 = {
    .n2 = 30/2, .t = (const struct lc3_complex []){
        {  1.0000000e+00, -0.0000000e+00 }, {  9.7814760e-01, -2.0791169e-01 },
        {  9.1354546e-01, -4.0673664e-01 }, {  8.0901699e-01, -5.8778525e-01 },
        {  6.6913061e-01, -7.4314483e-01 }, {  5.0000000e-01, -8.6602540e-01 },
        {  3.0901699e-01, -9.5105652e-01 }, {  1.0452846e-01, -9.9452190e-01 },
        { -1.0452846e-01, -9.9452190e-01 }, { -3.0901699e-01, -9.5105652e-01 },
        { -5.0000000e-01, -8.6602540e-01 }, { -6.6913061e-01, -7.4314483e-01 },
        { -8.0901699e-01, -5.8778525e-01 }, { -9.1354546e-01, -4.0673664e-01 },
        { -9.7814760e-01, -2.0791169e-01 },
    }
};

static const struct lc3_fft_bf2_twiddles fft_twiddles_40 = {
    .n2 = 40/2, .t = (const struct lc3_complex []){
        {  1.0000000e+00, -0.0000000e+00 }, {  9.8768834e-01, -1.5643447e-01 },
        {  9.5105652e-01, -3.0901699e-01 }, {  8.9100652e-01, -4.5399050e-01 },
        {  8.0901699e-01, -5.8778525e-01 }, {  7.0710678e-01, -7.0710678e-01 },
        {  5.8778525e-01, -8.0901699e-01 }, {  4.5399050e-01, -8.9100652e-01 },
        {  3.0901699e-01, -9.5105652e-01 }, {  1.5643447e-01, -9.8768834e-01 },
        {  6.1232340e-17, -1.0000000e+00 }, { -1.5643447e-01, -9.8768834e-01 },
        { -3.0901699e-01, -9.5105652e-01 }, { -4.5399050e-01, -8.9100652e-01 },
        { -5.8778525e-01, -8.0901699e-01 }, { -7.0710678e-01, -7.0710678e-01 },
        { -8.0901699e-01, -5.8778525e-01 }, { -8.9100652e-01, -4.5399050e-01 },
        { -9.5105652e-01, -3.0901699e-01 }, { -9.8768834e-01, -1.5643447e-01 },
    }
};

static const struct lc3_fft_bf2_twiddles fft_twiddles_60 = {
    .n2 = 60/2, .t = (const struct lc3_complex []){
        {  1.0000000e+00, -0.0000000e+00 }, {  9.9452190e-01, -1.0452846e-01 },
        {  9.7814760e-01, -2.0791169e-01 }, {  9.5105652e-01, -3.0901699e-01 },
        {  9.1354546e-01, -4.0673664e-01 }, {  8.6602540e-01, -5.0000000e-01 },
        {  8.0901699e-01, -5.8778525e-01 }, {  7.4314483e-01, -6.6913061e-01 },
        {  6.6913061e-01, -7.4314483e-01 }, {  5.8778525e-01, -8.0901699e-01 },
        {  5.0000000e-01, -8.6602540e-01 }, {  4.0673664e-01, -9.1354546e-01 },
        {  3.0901699e-01, -9.5105652e-01 }, {  2.0791169e-01, -9.7814760e-01 },
        {  1.0452846e-01, -9.9452190e-01 }, {  2.8327694e-16, -1.0000000e+00 },
        { -1.0452846e-01, -9.9452190e-01 }, { -2.0791169e-01, -9.7814760e-01 },
        { -3.0901699e-01, -9.5105652e-01 }, { -4.0673664e-01, -9.1354546e-01 },
        { -5.0000000e-01, -8.6602540e-01 }, { -5.8778525e-01, -8.0901699e-01 },
        { -6.6913061e-01, -7.4314483e-01 }, { -7.4314483e-01, -6.6913061e-01 },
        { -8.0901699e-01, -5.8778525e-01 }, { -8.6602540e-01, -5.0000000e-01 },
        { -9.1354546e-01, -4.0673664e-01 }, { -9.5105652e-01, -3.0901699e-01 },
        { -9.7814760e-01, -2.0791169e-01 }, { -9.9452190e-01, -1.0452846e-01 },
    }
};

static const struct lc3_fft_bf2_twiddles fft_twiddles_80 = {
    .n2 = 80/2, .t = (const struct lc3_complex []){
        {  1.0000000e+00, -0.0000000e+00 }, {  9.9691733e-01, -7.8459096e-02 },
        {  9.8768834e-01, -1.5643447e-01 }, {  9.7236992e-01, -2.3344536e-01 },
        {  9.5105652e-01, -3.0901699e-01 }, {  9.2387953e-01, -3.8268343e-01 },
        {  8.9100652e-01, -4.5399050e-01 }, {  8.5264016e-01, -5.2249856e-01 },
        {  8.0901699e-01, -5.8778525e-01 }, {  7.6040597e-01, -6.4944805e-01 },
        {  7.0710678e-01, -7.0710678e-01 }, {  6.4944805e-01, -7.6040597e-01 },
        {  5.8778525e-01, -8.0901699e-01 }, {  5.2249856e-01, -8.5264016e-01 },
        {  4.5399050e-01, -8.9100652e-01 }, {  3.8268343e-01, -9.2387953e-01 },
        {  3.0901699e-01, -9.5105652e-01 }, {  2.3344536e-01, -9.7236992e-01 },
        {  1.5643447e-01, -9.8768834e-01 }, {  7.8459096e-02, -9.9691733e-01 },
        {  6.1232340e-17, -1.0000000e+00 }, { -7.8459096e-02, -9.9691733e-01 },
        { -1.5643447e-01, -9.8768834e-01 }, { -2.3344536e-01, -9.7236992e-01 },
        { -3.0901699e-01, -9.5105652e-01 }, { -3.8268343e-01, -9.2387953e-01 },
        { -4.5399050e-01, -8.9100652e-01 }, { -5.2249856e-01, -8.5264016e-01 },
        { -5.8778525e-01, -8.0901699e-01 }, { -6.4944805e-01, -7.6040597e-01 },
        { -7.0710678e-01, -7.0710678e-01 }, { -7.6040597e-01, -6.4944805e-01 },
        { -8.0901699e-01, -5.8778525e-01 }, { -8.5264016e-01, -5.2249856e-01 },
        { -8.9100652e-01, -4.5399050e-01 }, { -9.2387953e-01, -3.8268343e-01 },
        { -9.5105652e-01, -3.0901699e-01 }, { -9.7236992e-01, -2.3344536e-01 },
        { -9.8768834e-01, -1.5643447e-01 }, { -9.9691733e-01, -7.8459096e-02 },
    }
};

static const struct lc3_fft_bf2_twiddles fft_twiddles_90 = {
    .n2 = 90/2, .t = (const struct lc3_complex []){
        {  1.0000000e+00, -0.0000000e+00 }, {  9.9756405e-01, -6.9756474e-02 },
        {  9.9026807e-01, -1.3917310e-01 }, {  9.7814760e-01, -2.0791169e-01 },
        {  9.6126170e-01, -2.7563736e-01 }, {  9.3969262e-01, -3.4202014e-01 },
        {  9.1354546e-01, -4.0673664e-01 }, {  8.8294759e-01, -4.6947156e-01 },
        {  8.4804810e-01, -5.2991926e-01 }, {  8.0901699e-01, -5.8778525e-01 },
        {  7.6604444e-01, -6.4278761e-01 }, {  7.1933980e-01, -6.9465837e-01 },
        {  6.6913061e-01, -7.4314483e-01 }, {  6.1566148e-01, -7.8801075e-01 },
        {  5.5919290e-01, -8.2903757e-01 }, {  5.0000000e-01, -8.6602540e-01 },
        {  4.3837115e-01, -8.9879405e-01 }, {  3.7460659e-01, -9.2718385e-01 },
        {  3.0901699e-01, -9.5105652e-01 }, {  2.4192190e-01, -9.7029573e-01 },
        {  1.7364818e-01, -9.8480775e-01 }, {  1.0452846e-01, -9.9452190e-01 },
        {  3.4899497e-02, -9.9939083e-01 }, { -3.4899497e-02, -9.9939083e-01 },
        { -1.0452846e-01, -9.9452190e-01 }, { -1.7364818e-01, -9.8480775e-01 },
        { -2.4192190e-01, -9.7029573e-01 }, { -3.0901699e-01, -9.5105652e-01 },
        { -3.7460659e-01, -9.2718385e-01 }, { -4.3837115e-01, -8.9879405e-01 },
        { -5.0000000e-01, -8.6602540e-01 }, { -5.5919290e-01, -8.2903757e-01 },
        { -6.1566148e-01, -7.8801075e-01 }, { -6.6913061e-01, -7.4314483e-01 },
        { -7.1933980e-01, -6.9465837e-01 }, { -7.6604444e-01, -6.4278761e-01 },
        { -8.0901699e-01, -5.8778525e-01 }, { -8.4804810e-01, -5.2991926e-01 },
        { -8.8294759e-01, -4.6947156e-01 }, { -9.1354546e-01, -4.0673664e-01 },
        { -9.3969262e-01, -3.4202014e-01 }, { -9.6126170e-01, -2.7563736e-01 },
        { -9.7814760e-01, -2.0791169e-01 }, { -9.9026807e-01, -1.3917310e-01 },
        { -9.9756405e-01, -6.9756474e-02 },
    }
};

static const struct lc3_fft_bf2_twiddles fft_twiddles_120 = {
    .n2 = 120/2, .t = (const struct lc3_complex []){
        {  1.0000000e+00, -0.0000000e+00 }, {  9.9862953e-01, -5.2335956e-02 },
        {  9.9452190e-01, -1.0452846e-01 }, {  9.8768834e-01, -1.5643447e-01 },
        {  9.7814760e-01, -2.0791169e-01 }, {  9.6592583e-01, -2.5881905e-01 },
        {  9.5105652e-01, -3.0901699e-01 }, {  9.3358043e-01, -3.5836795e-01 },
        {  9.1354546e-01, -4.0673664e-01 }, {  8.9100652e-01, -4.5399050e-01 },
        {  8.6602540e-01, -5.0000000e-01 }, {  8.3867057e-01, -5.4463904e-01 },
        {  8.0901699e-01, -5.8778525e-01 }, {  7.7714596e-01, -6.2932039e-01 },
        {  7.4314483e-01, -6.6913061e-01 }, {  7.0710678e-01, -7.0710678e-01 },
        {  6.6913061e-01, -7.4314483e-01 }, {  6.2932039e-01, -7.7714596e-01 },
        {  5.8778525e-01, -8.0901699e-01 }, {  5.4463904e-01, -8.3867057e-01 },
        {  5.0000000e-01, -8.6602540e-01 }, {  4.5399050e-01, -8.9100652e-01 },
        {  4.0673664e-01, -9.1354546e-01 }, {  3.5836795e-01, -9.3358043e-01 },
        {  3.0901699e-01, -9.5105652e-01 }, {  2.5881905e-01, -9.6592583e-01 },
        {  2.0791169e-01, -9.7814760e-01 }, {  1.5643447e-01, -9.8768834e-01 },
        {  1.0452846e-01, -9.9452190e-01 }, {  5.2335956e-02, -9.9862953e-01 },
        {  2.8327694e-16, -1.0000000e+00 }, { -5.2335956e-02, -9.9862953e-01 },
        { -1.0452846e-01, -9.9452190e-01 }, { -1.5643447e-01, -9.8768834e-01 },
        { -2.0791169e-01, -9.7814760e-01 }, { -2.5881905e-01, -9.6592583e-01 },
        { -3.0901699e-01, -9.5105652e-01 }, { -3.5836795e-01, -9.3358043e-01 },
        { -4.0673664e-01, -9.1354546e-01 }, { -4.5399050e-01, -8.9100652e-01 },
        { -5.0000000e-01, -8.6602540e-01 }, { -5.4463904e-01, -8.3867057e-01 },
        { -5.8778525e-01, -8.0901699e-01 }, { -6.2932039e-01, -7.7714596e-01 },
        { -6.6913061e-01, -7.4314483e-01 }, { -7.0710678e-01, -7.0710678e-01 },
        { -7.4314483e-01, -6.6913061e-01 }, { -7.7714596e-01, -6.2932039e-01 },
        { -8.0901699e-01, -5.8778525e-01 }, { -8.3867057e-01, -5.4463904e-01 },
        { -8.6602540e-01, -5.0000000e-01 }, { -8.9100652e-01, -4.5399050e-01 },
        { -9.1354546e-01, -4.0673664e-01 }, { -9.3358043e-01, -3.5836795e-01 },
        { -9.5105652e-01, -3.0901699e-01 }, { -9.6592583e-01, -2.5881905e-01 },
        { -9.7814760e-01, -2.0791169e-01 }, { -9.8768834e-01, -1.5643447e-01 },
        { -9.9452190e-01, -1.0452846e-01 }, { -9.9862953e-01, -5.2335956e-02 },
    }
};

static const struct lc3_fft_bf2_twiddles fft_twiddles_160 = {
    .n2 = 160/2, .t = (const struct lc3_complex []){
        {  1.0000000e+00, -0.0000000e+00 }, {  9.9922904e-01, -3.9259816e-02 },
        {  9.9691733e-01, -7.8459096e-02 }, {  9.9306846e-01, -1.1753740e-01 },
        {  9.8768834e-01, -1.5643447e-01 }, {  9.8078528e-01, -1.9509032e-01 },
        {  9.7236992e-01, -2.3344536e-01 }, {  9.6245524e-01, -2.7144045e-01 },
        {  9.5105652e-01, -3.0901699e-01 }, {  9.3819134e-01, -3.4611706e-01 },
        {  9.2387953e-01, -3.8268343e-01 }, {  9.0814317e-01, -4.1865974e-01 },
        {  8.9100652e-01, -4.5399050e-01 }, {  8.7249601e-01, -4.8862124e-01 },
        {  8.5264016e-01, -5.2249856e-01 }, {  8.3146961e-01, -5.5557023e-01 },
        {  8.0901699e-01, -5.8778525e-01 }, {  7.8531693e-01, -6.1909395e-01 },
        {  7.6040597e-01, -6.4944805e-01 }, {  7.3432251e-01, -6.7880075e-01 },
        {  7.0710678e-01, -7.0710678e-01 }, {  6.7880075e-01, -7.3432251e-01 },
        {  6.4944805e-01, -7.6040597e-01 }, {  6.1909395e-01, -7.8531693e-01 },
        {  5.8778525e-01, -8.0901699e-01 }, {  5.5557023e-01, -8.3146961e-01 },
        {  5.2249856e-01, -8.5264016e-01 }, {  4.8862124e-01, -8.7249601e-01 },
        {  4.5399050e-01, -8.9100652e-01 }, {  4.1865974e-01, -9.0814317e-01 },
        {  3.8268343e-01, -9.2387953e-01 }, {  3.4611706e-01, -9.3819134e-01 },
        {  3.0901699e-01, -9.5105652e-01 }, {  2.7144045e-01, -9.6245524e-01 },
        {  2.3344536e-01, -9.7236992e-01 }, {  1.9509032e-01, -9.8078528e-01 },
        {  1.5643447e-01, -9.8768834e-01 }, {  1.1753740e-01, -9.9306846e-01 },
        {  7.8459096e-02, -9.9691733e-01 }, {  3.9259816e-02, -9.9922904e-01 },
        {  6.1232340e-17, -1.0000000e+00 }, { -3.9259816e-02, -9.9922904e-01 },
        { -7.8459096e-02, -9.9691733e-01 }, { -1.1753740e-01, -9.9306846e-01 },
        { -1.5643447e-01, -9.8768834e-01 }, { -1.9509032e-01, -9.8078528e-01 },
        { -2.3344536e-01, -9.7236992e-01 }, { -2.7144045e-01, -9.6245524e-01 },
        { -3.0901699e-01, -9.5105652e-01 }, { -3.4611706e-01, -9.3819134e-01 },
        { -3.8268343e-01, -9.2387953e-01 }, { -4.1865974e-01, -9.0814317e-01 },
        { -4.5399050e-01, -8.9100652e-01 }, { -4.8862124e-01, -8.7249601e-01 },
        { -5.2249856e-01, -8.5264016e-01 }, { -5.5557023e-01, -8.3146961e-01 },
        { -5.8778525e-01, -8.0901699e-01 }, { -6.1909395e-01, -7.8531693e-01 },
        { -6.4944805e-01, -7.6040597e-01 }, { -6.7880075e-01, -7.3432251e-01 },
        { -7.0710678e-01, -7.0710678e-01 }, { -7.3432251e-01, -6.7880075e-01 },
        { -7.6040597e-01, -6.4944805e-01 }, { -7.8531693e-01, -6.1909395e-01 },
        { -8.0901699e-01, -5.8778525e-01 }, { -8.3146961e-01, -5.5557023e-01 },
        { -8.5264016e-01, -5.2249856e-01 }, { -8.7249601e-01, -4.8862124e-01 },
        { -8.9100652e-01, -4.5399050e-01 }, { -9.0814317e-01, -4.1865974e-01 },
        { -9.2387953e-01, -3.8268343e-01 }, { -9.3819134e-01, -3.4611706e-01 },
        { -9.5105652e-01, -3.0901699e-01 }, { -9.6245524e-01, -2.7144045e-01 },
        { -9.7236992e-01, -2.3344536e-01 }, { -9.8078528e-01, -1.9509032e-01 },
        { -9.8768834e-01, -1.5643447e-01 }, { -9.9306846e-01, -1.1753740e-01 },
        { -9.9691733e-01, -7.8459096e-02 }, { -9.9922904e-01, -3.9259816e-02 },
    }
};

static const struct lc3_fft_bf2_twiddles fft_twiddles_180 = {
    .n2 = 180/2, .t = (const struct lc3_complex []){
        {  1.0000000e+00, -0.0000000e+00 }, {  9.9939083e-01, -3.4899497e-02 },
        {  9.9756405e-01, -6.9756474e-02 }, {  9.9452190e-01, -1.0452846e-01 },
        {  9.9026807e-01, -1.3917310e-01 }, {  9.8480775e-01, -1.7364818e-01 },
        {  9.7814760e-01, -2.0791169e-01 }, {  9.7029573e-01, -2.4192190e-01 },
        {  9.6126170e-01, -2.7563736e-01 }, {  9.5105652e-01, -3.0901699e-01 },
        {  9.3969262e-01, -3.4202014e-01 }, {  9.2718385e-01, -3.7460659e-01 },
        {  9.1354546e-01, -4.0673664e-01 }, {  8.9879405e-01, -4.3837115e-01 },
        {  8.8294759e-01, -4.6947156e-01 }, {  8.6602540e-01, -5.0000000e-01 },
        {  8.4804810e-01, -5.2991926e-01 }, {  8.2903757e-01, -5.5919290e-01 },
        {  8.0901699e-01, -5.8778525e-01 }, {  7.8801075e-01, -6.1566148e-01 },
        {  7.6604444e-01, -6.4278761e-01 }, {  7.4314483e-01, -6.6913061e-01 },
        {  7.1933980e-01, -6.9465837e-01 }, {  6.9465837e-01, -7.1933980e-01 },
        {  6.6913061e-01, -7.4314483e-01 }, {  6.4278761e-01, -7.6604444e-01 },
        {  6.1566148e-01, -7.8801075e-01 }, {  5.8778525e-01, -8.0901699e-01 },
        {  5.5919290e-01, -8.2903757e-01 }, {  5.2991926e-01, -8.4804810e-01 },
        {  5.0000000e-01, -8.6602540e-01 }, {  4.6947156e-01, -8.8294759e-01 },
        {  4.3837115e-01, -8.9879405e-01 }, {  4.0673664e-01, -9.1354546e-01 },
        {  3.7460659e-01, -9.2718385e-01 }, {  3.4202014e-01, -9.3969262e-01 },
        {  3.0901699e-01, -9.5105652e-01 }, {  2.7563736e-01, -9.6126170e-01 },
        {  2.4192190e-01, -9.7029573e-01 }, {  2.0791169e-01, -9.7814760e-01 },
        {  1.7364818e-01, -9.8480775e-01 }, {  1.3917310e-01, -9.9026807e-01 },
        {  1.0452846e-01, -9.9452190e-01 }, {  6.9756474e-02, -9.9756405e-01 },
        {  3.4899497e-02, -9.9939083e-01 }, {  6.1232340e-17, -1.0000000e+00 },
        { -3.4899497e-02, -9.9939083e-01 }, { -6.9756474e-02, -9.9756405e-01 },
        { -1.0452846e-01, -9.9452190e-01 }, { -1.3917310e-01, -9.9026807e-01 },
        { -1.7364818e-01, -9.8480775e-01 }, { -2.0791169e-01, -9.7814760e-01 },
        { -2.4192190e-01, -9.7029573e-01 }, { -2.7563736e-01, -9.6126170e-01 },
        { -3.0901699e-01, -9.5105652e-01 }, { -3.4202014e-01, -9.3969262e-01 },
        { -3.7460659e-01, -9.2718385e-01 }, { -4.0673664e-01, -9.1354546e-01 },
        { -4.3837115e-01, -8.9879405e-01 }, { -4.6947156e-01, -8.8294759e-01 },
        { -5.0000000e-01, -8.6602540e-01 }, { -5.2991926e-01, -8.4804810e-01 },
        { -5.5919290e-01, -8.2903757e-01 }, { -5.8778525e-01, -8.0901699e-01 },
        { -6.1566148e-01, -7.8801075e-01 }, { -6.4278761e-01, -7.6604444e-01 },
        { -6.6913061e-01, -7.4314483e-01 }, { -6.9465837e-01, -7.1933980e-01 },
        { -7.1933980e-01, -6.9465837e-01 }, { -7.4314483e-01, -6.6913061e-01 },
        { -7.6604444e-01, -6.4278761e-01 }, { -7.8801075e-01, -6.1566148e-01 },
        { -8.0901699e-01, -5.8778525e-01 }, { -8.2903757e-01, -5.5919290e-01 },
        { -8.4804810e-01, -5.2991926e-01 }, { -8.6602540e-01, -5.0000000e-01 },
        { -8.8294759e-01, -4.6947156e-01 }, { -8.9879405e-01, -4.3837115e-01 },
        { -9.1354546e-01, -4.0673664e-01 }, { -9.2718385e-01, -3.7460659e-01 },
        { -9.3969262e-01, -3.4202014e-01 }, { -9.5105652e-01, -3.0901699e-01 },
        { -9.6126170e-01, -2.7563736e-01 }, { -9.7029573e-01, -2.4192190e-01 },
        { -9.7814760e-01, -2.0791169e-01 }, { -9.8480775e-01, -1.7364818e-01 },
        { -9.9026807e-01, -1.3917310e-01 }, { -9.9452190e-01, -1.0452846e-01 },
        { -9.9756405e-01, -6.9756474e-02 }, { -9.9939083e-01, -3.4899497e-02 },
    }
};

static const struct lc3_fft_bf2_twiddles fft_twiddles_240 = {
    .n2 = 240/2, .t = (const struct lc3_complex []){
        {  1.0000000e+00, -0.0000000e+00 }, {  9.9965732e-01, -2.6176948e-02 },
        {  9.9862953e-01, -5.2335956e-02 }, {  9.9691733e-01, -7.8459096e-02 },
        {  9.9452190e-01, -1.0452846e-01 }, {  9.9144486e-01, -1.3052619e-01 },
        {  9.8768834e-01, -1.5643447e-01 }, {  9.8325491e-01, -1.8223553e-01 },
        {  9.7814760e-01, -2.0791169e-01 }, {  9.7236992e-01, -2.3344536e-01 },
        {  9.6592583e-01, -2.5881905e-01 }, {  9.5881973e-01, -2.8401534e-01 },
        {  9.5105652e-01, -3.0901699e-01 }, {  9.4264149e-01, -3.3380686e-01 },
        {  9.3358043e-01, -3.5836795e-01 }, {  9.2387953e-01, -3.8268343e-01 },
        {  9.1354546e-01, -4.0673664e-01 }, {  9.0258528e-01, -4.3051110e-01 },
        {  8.9100652e-01, -4.5399050e-01 }, {  8.7881711e-01, -4.7715876e-01 },
        {  8.6602540e-01, -5.0000000e-01 }, {  8.5264016e-01, -5.2249856e-01 },
        {  8.3867057e-01, -5.4463904e-01 }, {  8.2412619e-01, -5.6640624e-01 },
        {  8.0901699e-01, -5.8778525e-01 }, {  7.9335334e-01, -6.0876143e-01 },
        {  7.7714596e-01, -6.2932039e-01 }, {  7.6040597e-01, -6.4944805e-01 },
        {  7.4314483e-01, -6.6913061e-01 }, {  7.2537437e-01, -6.8835458e-01 },
        {  7.0710678e-01, -7.0710678e-01 }, {  6.8835458e-01, -7.2537437e-01 },
        {  6.6913061e-01, -7.4314483e-01 }, {  6.4944805e-01, -7.6040597e-01 },
        {  6.2932039e-01, -7.7714596e-01 }, {  6.0876143e-01, -7.9335334e-01 },
        {  5.8778525e-01, -8.0901699e-01 }, {  5.6640624e-01, -8.2412619e-01 },
        {  5.4463904e-01, -8.3867057e-01 }, {  5.2249856e-01, -8.5264016e-01 },
        {  5.0000000e-01, -8.6602540e-01 }, {  4.7715876e-01, -8.7881711e-01 },
        {  4.5399050e-01, -8.9100652e-01 }, {  4.3051110e-01, -9.0258528e-01 },
        {  4.0673664e-01, -9.1354546e-01 }, {  3.8268343e-01, -9.2387953e-01 },
        {  3.5836795e-01, -9.3358043e-01 }, {  3.3380686e-01, -9.4264149e-01 },
        {  3.0901699e-01, -9.5105652e-01 }, {  2.8401534e-01, -9.5881973e-01 },
        {  2.5881905e-01, -9.6592583e-01 }, {  2.3344536e-01, -9.7236992e-01 },
        {  2.0791169e-01, -9.7814760e-01 }, {  1.8223553e-01, -9.8325491e-01 },
        {  1.5643447e-01, -9.8768834e-01 }, {  1.3052619e-01, -9.9144486e-01 },
        {  1.0452846e-01, -9.9452190e-01 }, {  7.8459096e-02, -9.9691733e-01 },
        {  5.2335956e-02, -9.9862953e-01 }, {  2.6176948e-02, -9.9965732e-01 },
        {  2.8327694e-16, -1.0000000e+00 }, { -2.6176948e-02, -9.9965732e-01 },
        { -5.2335956e-02, -9.9862953e-01 }, { -7.8459096e-02, -9.9691733e-01 },
        { -1.0452846e-01, -9.9452190e-01 }, { -1.3052619e-01, -9.9144486e-01 },
        { -1.5643447e-01, -9.8768834e-01 }, { -1.8223553e-01, -9.8325491e-01 },
        { -2.0791169e-01, -9.7814760e-01 }, { -2.3344536e-01, -9.7236992e-01 },
        { -2.5881905e-01, -9.6592583e-01 }, { -2.8401534e-01, -9.5881973e-01 },
        { -3.0901699e-01, -9.5105652e-01 }, { -3.3380686e-01, -9.4264149e-01 },
        { -3.5836795e-01, -9.3358043e-01 }, { -3.8268343e-01, -9.2387953e-01 },
        { -4.0673664e-01, -9.1354546e-01 }, { -4.3051110e-01, -9.0258528e-01 },
        { -4.5399050e-01, -8.9100652e-01 }, { -4.7715876e-01, -8.7881711e-01 },
        { -5.0000000e-01, -8.6602540e-01 }, { -5.2249856e-01, -8.5264016e-01 },
        { -5.4463904e-01, -8.3867057e-01 }, { -5.6640624e-01, -8.2412619e-01 },
        { -5.8778525e-01, -8.0901699e-01 }, { -6.0876143e-01, -7.9335334e-01 },
        { -6.2932039e-01, -7.7714596e-01 }, { -6.4944805e-01, -7.6040597e-01 },
        { -6.6913061e-01, -7.4314483e-01 }, { -6.8835458e-01, -7.2537437e-01 },
        { -7.0710678e-01, -7.0710678e-01 }, { -7.2537437e-01, -6.8835458e-01 },
        { -7.4314483e-01, -6.6913061e-01 }, { -7.6040597e-01, -6.4944805e-01 },
        { -7.7714596e-01, -6.2932039e-01 }, { -7.9335334e-01, -6.0876143e-01 },
        { -8.0901699e-01, -5.8778525e-01 }, { -8.2412619e-01, -5.6640624e-01 },
        { -8.3867057e-01, -5.4463904e-01 }, { -8.5264016e-01, -5.2249856e-01 },
        { -8.6602540e-01, -5.0000000e-01 }, { -8.7881711e-01, -4.7715876e-01 },
        { -8.9100652e-01, -4.5399050e-01 }, { -9.0258528e-01, -4.3051110e-01 },
        { -9.1354546e-01, -4.0673664e-01 }, { -9.2387953e-01, -3.8268343e-01 },
        { -9.3358043e-01, -3.5836795e-01 }, { -9.4264149e-01, -3.3380686e-01 },
        { -9.5105652e-01, -3.0901699e-01 }, { -9.5881973e-01, -2.8401534e-01 },
        { -9.6592583e-01, -2.5881905e-01 }, { -9.7236992e-01, -2.3344536e-01 },
        { -9.7814760e-01, -2.0791169e-01 }, { -9.8325491e-01, -1.8223553e-01 },
        { -9.8768834e-01, -1.5643447e-01 }, { -9.9144486e-01, -1.3052619e-01 },
        { -9.9452190e-01, -1.0452846e-01 }, { -9.9691733e-01, -7.8459096e-02 },
        { -9.9862953e-01, -5.2335956e-02 }, { -9.9965732e-01, -2.6176948e-02 },
    }
};

const struct lc3_fft_bf2_twiddles *lc3_fft_twiddles_bf2[][3] = {
    { &fft_twiddles_10 , &fft_twiddles_30 , &fft_twiddles_90  },
    { &fft_twiddles_20 , &fft_twiddles_60 , &fft_twiddles_180 },
    { &fft_twiddles_40 , &fft_twiddles_120 },
    { &fft_twiddles_80 , &fft_twiddles_240 },
    { &fft_twiddles_160  }
};


/**
 * MDCT Rotation twiddles
 *
 *            2Pi (n + 1/8) / N
 *   W[n] = e                   * sqrt( sqrt( 4/N ) ), n = [0..N/4-1]
 */

static const struct lc3_mdct_rot_def mdct_rot_120 = {
    .n4 = 120/4, .w = (const struct lc3_complex []){
        { 4.2727785e-01, 2.7965670e-03 }, { 4.2654592e-01, 2.5154729e-02 },
        { 4.2464486e-01, 4.7443945e-02 }, { 4.2157988e-01, 6.9603119e-02 },
        { 4.1735937e-01, 9.1571516e-02 }, { 4.1199491e-01, 1.1328892e-01 },
        { 4.0550120e-01, 1.3469581e-01 }, { 3.9789604e-01, 1.5573351e-01 },
        { 3.8920028e-01, 1.7634435e-01 }, { 3.7943774e-01, 1.9647185e-01 },
        { 3.6863519e-01, 2.1606083e-01 }, { 3.5682224e-01, 2.3505760e-01 },
        { 3.4403126e-01, 2.5341009e-01 }, { 3.3029732e-01, 2.7106801e-01 },
        { 3.1565806e-01, 2.8798294e-01 }, { 3.0015360e-01, 3.0410854e-01 },
        { 2.8382644e-01, 3.1940060e-01 }, { 2.6672133e-01, 3.3381720e-01 },
        { 2.4888515e-01, 3.4731883e-01 }, { 2.3036680e-01, 3.5986848e-01 },
        { 2.1121703e-01, 3.7143176e-01 }, { 1.9148833e-01, 3.8197697e-01 },
        { 1.7123477e-01, 3.9147521e-01 }, { 1.5051187e-01, 3.9990044e-01 },
        { 1.2937643e-01, 4.0722957e-01 }, { 1.0788637e-01, 4.1344252e-01 },
        { 8.6100606e-02, 4.1852225e-01 }, { 6.4078846e-02, 4.2245483e-01 },
        { 4.1881450e-02, 4.2522950e-01 }, { 1.9569261e-02, 4.2683865e-01 },
    }
};

static const struct lc3_mdct_rot_def mdct_rot_160 = {
    .n4 = 160/4, .w = (const struct lc3_complex []){
        { 3.9763057e-01, 1.9518802e-03 }, { 3.9724738e-01, 1.7561278e-02 },
        { 3.9625167e-01, 3.3143598e-02 }, { 3.9464496e-01, 4.8674813e-02 },
        { 3.9242974e-01, 6.4130975e-02 }, { 3.8960942e-01, 7.9488252e-02 },
        { 3.8618835e-01, 9.4722964e-02 }, { 3.8217181e-01, 1.0981162e-01 },
        { 3.7756598e-01, 1.2473095e-01 }, { 3.7237798e-01, 1.3945796e-01 },
        { 3.6661580e-01, 1.5396993e-01 }, { 3.6028832e-01, 1.6824450e-01 },
        { 3.5340530e-01, 1.8225964e-01 }, { 3.4597736e-01, 1.9599375e-01 },
        { 3.3801594e-01, 2.0942566e-01 }, { 3.2953333e-01, 2.2253464e-01 },
        { 3.2054261e-01, 2.3530049e-01 }, { 3.1105762e-01, 2.4770353e-01 },
        { 3.0109302e-01, 2.5972462e-01 }, { 2.9066414e-01, 2.7134524e-01 },
        { 2.7978709e-01, 2.8254746e-01 }, { 2.6847862e-01, 2.9331402e-01 },
        { 2.5675618e-01, 3.0362831e-01 }, { 2.4463784e-01, 3.1347442e-01 },
        { 2.3214228e-01, 3.2283718e-01 }, { 2.1928878e-01, 3.3170215e-01 },
        { 2.0609715e-01, 3.4005565e-01 }, { 1.9258774e-01, 3.4788482e-01 },
        { 1.7878136e-01, 3.5517757e-01 }, { 1.6469932e-01, 3.6192266e-01 },
        { 1.5036333e-01, 3.6810970e-01 }, { 1.3579549e-01, 3.7372914e-01 },
        { 1.2101826e-01, 3.7877231e-01 }, { 1.0605442e-01, 3.8323145e-01 },
        { 9.0927064e-02, 3.8709967e-01 }, { 7.5659501e-02, 3.9037101e-01 },
        { 6.0275277e-02, 3.9304042e-01 }, { 4.4798112e-02, 3.9510380e-01 },
        { 2.9251872e-02, 3.9655795e-01 }, { 1.3660528e-02, 3.9740065e-01 },
    }
};

static const struct lc3_mdct_rot_def mdct_rot_240 = {
    .n4 = 240/4, .w = (const struct lc3_complex []){
        { 3.5930219e-01, 1.1758179e-03 }, { 3.5914828e-01, 1.0580850e-02 },
        { 3.5874824e-01, 1.9978630e-02 }, { 3.5810233e-01, 2.9362718e-02 },
        { 3.5721099e-01, 3.8726682e-02 }, { 3.5607483e-01, 4.8064105e-02 },
        { 3.5469464e-01, 5.7368587e-02 }, { 3.5307136e-01, 6.6633752e-02 },
        { 3.5120611e-01, 7.5853249e-02 }, { 3.4910015e-01, 8.5020760e-02 },
        { 3.4675494e-01, 9.4130002e-02 }, { 3.4417208e-01, 1.0317473e-01 },
        { 3.4135334e-01, 1.1214875e-01 }, { 3.3830065e-01, 1.2104591e-01 },
        { 3.3501611e-01, 1.2986011e-01 }, { 3.3150197e-01, 1.3858531e-01 },
        { 3.2776063e-01, 1.4721553e-01 }, { 3.2379466e-01, 1.5574485e-01 },
        { 3.1960678e-01, 1.6416744e-01 }, { 3.1519986e-01, 1.7247752e-01 },
        { 3.1057691e-01, 1.8066938e-01 }, { 3.0574111e-01, 1.8873743e-01 },
        { 3.0069577e-01, 1.9667612e-01 }, { 2.9544435e-01, 2.0448002e-01 },
        { 2.8999045e-01, 2.1214378e-01 }, { 2.8433780e-01, 2.1966215e-01 },
        { 2.7849028e-01, 2.2702998e-01 }, { 2.7245189e-01, 2.3424220e-01 },
        { 2.6622679e-01, 2.4129389e-01 }, { 2.5981922e-01, 2.4818021e-01 },
        { 2.5323358e-01, 2.5489644e-01 }, { 2.4647440e-01, 2.6143798e-01 },
        { 2.3954629e-01, 2.6780034e-01 }, { 2.3245401e-01, 2.7397916e-01 },
        { 2.2520241e-01, 2.7997021e-01 }, { 2.1779647e-01, 2.8576938e-01 },
        { 2.1024127e-01, 2.9137270e-01 }, { 2.0254198e-01, 2.9677633e-01 },
        { 1.9470387e-01, 3.0197657e-01 }, { 1.8673233e-01, 3.0696984e-01 },
        { 1.7863281e-01, 3.1175273e-01 }, { 1.7041086e-01, 3.1632196e-01 },
        { 1.6207212e-01, 3.2067440e-01 }, { 1.5362230e-01, 3.2480707e-01 },
        { 1.4506720e-01, 3.2871713e-01 }, { 1.3641268e-01, 3.3240190e-01 },
        { 1.2766467e-01, 3.3585887e-01 }, { 1.1882916e-01, 3.3908565e-01 },
        { 1.0991221e-01, 3.4208003e-01 }, { 1.0091994e-01, 3.4483998e-01 },
        { 9.1858496e-02, 3.4736359e-01 }, { 8.2734100e-02, 3.4964913e-01 },
        { 7.3553002e-02, 3.5169504e-01 }, { 6.4321494e-02, 3.5349992e-01 },
        { 5.5045904e-02, 3.5506252e-01 }, { 4.5732588e-02, 3.5638178e-01 },
        { 3.6387929e-02, 3.5745680e-01 }, { 2.7018332e-02, 3.5828683e-01 },
        { 1.7630217e-02, 3.5887131e-01 }, { 8.2300199e-03, 3.5920984e-01 },
    }
};

static const struct lc3_mdct_rot_def mdct_rot_320 = {
    .n4 = 320/4, .w = (const struct lc3_complex []){
        { 3.3436915e-01, 8.2066700e-04 }, { 3.3428858e-01, 7.3854098e-03 },
        { 3.3407914e-01, 1.3947305e-02 }, { 3.3374091e-01, 2.0503824e-02 },
        { 3.3327401e-01, 2.7052438e-02 }, { 3.3267863e-01, 3.3590623e-02 },
        { 3.3195499e-01, 4.0115858e-02 }, { 3.3110338e-01, 4.6625627e-02 },
        { 3.3012413e-01, 5.3117422e-02 }, { 3.2901760e-01, 5.9588738e-02 },
        { 3.2778423e-01, 6.6037082e-02 }, { 3.2642450e-01, 7.2459968e-02 },
        { 3.2493892e-01, 7.8854919e-02 }, { 3.2332807e-01, 8.5219469e-02 },
        { 3.2159257e-01, 9.1551166e-02 }, { 3.1973310e-01, 9.7847569e-02 },
        { 3.1775035e-01, 1.0410625e-01 }, { 3.1564512e-01, 1.1032479e-01 },
        { 3.1341819e-01, 1.1650081e-01 }, { 3.1107043e-01, 1.2263191e-01 },
        { 3.0860275e-01, 1.2871573e-01 }, { 3.0601610e-01, 1.3474993e-01 },
        { 3.0331148e-01, 1.4073218e-01 }, { 3.0048992e-01, 1.4666018e-01 },
        { 2.9755251e-01, 1.5253164e-01 }, { 2.9450040e-01, 1.5834429e-01 },
        { 2.9133475e-01, 1.6409590e-01 }, { 2.8805678e-01, 1.6978424e-01 },
        { 2.8466777e-01, 1.7540713e-01 }, { 2.8116900e-01, 1.8096240e-01 },
        { 2.7756185e-01, 1.8644790e-01 }, { 2.7384768e-01, 1.9186153e-01 },
        { 2.7002795e-01, 1.9720119e-01 }, { 2.6610411e-01, 2.0246482e-01 },
        { 2.6207768e-01, 2.0765040e-01 }, { 2.5795022e-01, 2.1275592e-01 },
        { 2.5372331e-01, 2.1777943e-01 }, { 2.4939859e-01, 2.2271898e-01 },
        { 2.4497772e-01, 2.2757266e-01 }, { 2.4046241e-01, 2.3233861e-01 },
        { 2.3585439e-01, 2.3701499e-01 }, { 2.3115545e-01, 2.4159999e-01 },
        { 2.2636739e-01, 2.4609186e-01 }, { 2.2149206e-01, 2.5048885e-01 },
        { 2.1653135e-01, 2.5478927e-01 }, { 2.1148716e-01, 2.5899147e-01 },
        { 2.0636143e-01, 2.6309382e-01 }, { 2.0115615e-01, 2.6709474e-01 },
        { 1.9587332e-01, 2.7099270e-01 }, { 1.9051498e-01, 2.7478618e-01 },
        { 1.8508318e-01, 2.7847372e-01 }, { 1.7958004e-01, 2.8205391e-01 },
        { 1.7400766e-01, 2.8552536e-01 }, { 1.6836821e-01, 2.8888674e-01 },
        { 1.6266384e-01, 2.9213674e-01 }, { 1.5689676e-01, 2.9527412e-01 },
        { 1.5106920e-01, 2.9829767e-01 }, { 1.4518339e-01, 3.0120621e-01 },
        { 1.3924162e-01, 3.0399864e-01 }, { 1.3324616e-01, 3.0667387e-01 },
        { 1.2719933e-01, 3.0923087e-01 }, { 1.2110347e-01, 3.1166865e-01 },
        { 1.1496092e-01, 3.1398628e-01 }, { 1.0877405e-01, 3.1618287e-01 },
        { 1.0254525e-01, 3.1825755e-01 }, { 9.6276910e-02, 3.2020955e-01 },
        { 8.9971456e-02, 3.2203810e-01 }, { 8.3631316e-02, 3.2374249e-01 },
        { 7.7258935e-02, 3.2532208e-01 }, { 7.0856769e-02, 3.2677625e-01 },
        { 6.4427286e-02, 3.2810444e-01 }, { 5.7972965e-02, 3.2930614e-01 },
        { 5.1496295e-02, 3.3038089e-01 }, { 4.4999772e-02, 3.3132827e-01 },
        { 3.8485901e-02, 3.3214791e-01 }, { 3.1957192e-02, 3.3283951e-01 },
        { 2.5416164e-02, 3.3340279e-01 }, { 1.8865337e-02, 3.3383753e-01 },
        { 1.2307237e-02, 3.3414358e-01 }, { 5.7443922e-03, 3.3432081e-01 },
    }
};

static const struct lc3_mdct_rot_def mdct_rot_360 = {
    .n4 = 360/4, .w = (const struct lc3_complex []){
        { 3.2466714e-01, 7.0831495e-04 }, { 3.2460533e-01, 6.3744300e-03 },
        { 3.2444464e-01, 1.2038603e-02 }, { 3.2418513e-01, 1.7699110e-02 },
        { 3.2382686e-01, 2.3354225e-02 }, { 3.2336995e-01, 2.9002226e-02 },
        { 3.2281454e-01, 3.4641392e-02 }, { 3.2216080e-01, 4.0270007e-02 },
        { 3.2140893e-01, 4.5886355e-02 }, { 3.2055915e-01, 5.1488725e-02 },
        { 3.1961172e-01, 5.7075412e-02 }, { 3.1856694e-01, 6.2644713e-02 },
        { 3.1742512e-01, 6.8194931e-02 }, { 3.1618661e-01, 7.3724377e-02 },
        { 3.1485178e-01, 7.9231366e-02 }, { 3.1342105e-01, 8.4714220e-02 },
        { 3.1189485e-01, 9.0171269e-02 }, { 3.1027364e-01, 9.5600851e-02 },
        { 3.0855792e-01, 1.0100131e-01 }, { 3.0674821e-01, 1.0637101e-01 },
        { 3.0484506e-01, 1.1170830e-01 }, { 3.0284905e-01, 1.1701157e-01 },
        { 3.0076079e-01, 1.2227919e-01 }, { 2.9858092e-01, 1.2750957e-01 },
        { 2.9631010e-01, 1.3270110e-01 }, { 2.9394901e-01, 1.3785221e-01 },
        { 2.9149839e-01, 1.4296134e-01 }, { 2.8895897e-01, 1.4802691e-01 },
        { 2.8633154e-01, 1.5304740e-01 }, { 2.8361688e-01, 1.5802126e-01 },
        { 2.8081584e-01, 1.6294699e-01 }, { 2.7792925e-01, 1.6782308e-01 },
        { 2.7495800e-01, 1.7264806e-01 }, { 2.7190300e-01, 1.7742044e-01 },
        { 2.6876518e-01, 1.8213878e-01 }, { 2.6554548e-01, 1.8680164e-01 },
        { 2.6224490e-01, 1.9140760e-01 }, { 2.5886443e-01, 1.9595525e-01 },
        { 2.5540512e-01, 2.0044321e-01 }, { 2.5186800e-01, 2.0487012e-01 },
        { 2.4825416e-01, 2.0923462e-01 }, { 2.4456471e-01, 2.1353538e-01 },
        { 2.4080075e-01, 2.1777110e-01 }, { 2.3696345e-01, 2.2194049e-01 },
        { 2.3305396e-01, 2.2604227e-01 }, { 2.2907348e-01, 2.3007519e-01 },
        { 2.2502323e-01, 2.3403803e-01 }, { 2.2090443e-01, 2.3792959e-01 },
        { 2.1671834e-01, 2.4174866e-01 }, { 2.1246624e-01, 2.4549410e-01 },
        { 2.0814942e-01, 2.4916476e-01 }, { 2.0376919e-01, 2.5275952e-01 },
        { 1.9932689e-01, 2.5627728e-01 }, { 1.9482388e-01, 2.5971698e-01 },
        { 1.9026152e-01, 2.6307757e-01 }, { 1.8564121e-01, 2.6635803e-01 },
        { 1.8096434e-01, 2.6955734e-01 }, { 1.7623236e-01, 2.7267455e-01 },
        { 1.7144669e-01, 2.7570870e-01 }, { 1.6660880e-01, 2.7865887e-01 },
        { 1.6172015e-01, 2.8152415e-01 }, { 1.5678225e-01, 2.8430368e-01 },
        { 1.5179659e-01, 2.8699661e-01 }, { 1.4676469e-01, 2.8960211e-01 },
        { 1.4168808e-01, 2.9211940e-01 }, { 1.3656831e-01, 2.9454771e-01 },
        { 1.3140695e-01, 2.9688629e-01 }, { 1.2620555e-01, 2.9913444e-01 },
        { 1.2096571e-01, 3.0129147e-01 }, { 1.1568903e-01, 3.0335673e-01 },
        { 1.1037710e-01, 3.0532958e-01 }, { 1.0503156e-01, 3.0720942e-01 },
        { 9.9654017e-02, 3.0899568e-01 }, { 9.4246121e-02, 3.1068782e-01 },
        { 8.8809517e-02, 3.1228533e-01 }, { 8.3345860e-02, 3.1378770e-01 },
        { 7.7856816e-02, 3.1519450e-01 }, { 7.2344055e-02, 3.1650528e-01 },
        { 6.6809258e-02, 3.1771965e-01 }, { 6.1254110e-02, 3.1883725e-01 },
        { 5.5680304e-02, 3.1985772e-01 }, { 5.0089536e-02, 3.2078076e-01 },
        { 4.4483511e-02, 3.2160608e-01 }, { 3.8863936e-02, 3.2233345e-01 },
        { 3.3232523e-02, 3.2296262e-01 }, { 2.7590986e-02, 3.2349342e-01 },
        { 2.1941045e-02, 3.2392568e-01 }, { 1.6284421e-02, 3.2425927e-01 },
        { 1.0622836e-02, 3.2449408e-01 }, { 4.9580159e-03, 3.2463006e-01 },
    }
};

static const struct lc3_mdct_rot_def mdct_rot_480 = {
    .n4 = 480/4, .w = (const struct lc3_complex []){
        { 3.0213714e-01, 4.9437117e-04 }, { 3.0210478e-01, 4.4491817e-03 },
        { 3.0202066e-01, 8.4032299e-03 }, { 3.0188479e-01, 1.2355838e-02 },
        { 3.0169719e-01, 1.6306330e-02 }, { 3.0145790e-01, 2.0254027e-02 },
        { 3.0116696e-01, 2.4198254e-02 }, { 3.0082441e-01, 2.8138334e-02 },
        { 3.0043032e-01, 3.2073593e-02 }, { 2.9998475e-01, 3.6003357e-02 },
        { 2.9948778e-01, 3.9926952e-02 }, { 2.9893950e-01, 4.3843705e-02 },
        { 2.9833999e-01, 4.7752946e-02 }, { 2.9768936e-01, 5.1654004e-02 },
        { 2.9698773e-01, 5.5546213e-02 }, { 2.9623521e-01, 5.9428903e-02 },
        { 2.9543193e-01, 6.3301411e-02 }, { 2.9457803e-01, 6.7163072e-02 },
        { 2.9367365e-01, 7.1013225e-02 }, { 2.9271896e-01, 7.4851211e-02 },
        { 2.9171411e-01, 7.8676371e-02 }, { 2.9065928e-01, 8.2488050e-02 },
        { 2.8955464e-01, 8.6285595e-02 }, { 2.8840039e-01, 9.0068356e-02 },
        { 2.8719672e-01, 9.3835684e-02 }, { 2.8594385e-01, 9.7586934e-02 },
        { 2.8464198e-01, 1.0132146e-01 }, { 2.8329133e-01, 1.0503863e-01 },
        { 2.8189215e-01, 1.0873780e-01 }, { 2.8044466e-01, 1.1241834e-01 },
        { 2.7894913e-01, 1.1607962e-01 }, { 2.7740579e-01, 1.1972100e-01 },
        { 2.7581493e-01, 1.2334187e-01 }, { 2.7417680e-01, 1.2694161e-01 },
        { 2.7249170e-01, 1.3051960e-01 }, { 2.7075991e-01, 1.3407523e-01 },
        { 2.6898172e-01, 1.3760788e-01 }, { 2.6715744e-01, 1.4111695e-01 },
        { 2.6528739e-01, 1.4460184e-01 }, { 2.6337188e-01, 1.4806196e-01 },
        { 2.6141125e-01, 1.5149671e-01 }, { 2.5940582e-01, 1.5490549e-01 },
        { 2.5735595e-01, 1.5828774e-01 }, { 2.5526198e-01, 1.6164286e-01 },
        { 2.5312427e-01, 1.6497029e-01 }, { 2.5094319e-01, 1.6826945e-01 },
        { 2.4871911e-01, 1.7153978e-01 }, { 2.4645242e-01, 1.7478072e-01 },
        { 2.4414349e-01, 1.7799171e-01 }, { 2.4179274e-01, 1.8117220e-01 },
        { 2.3940055e-01, 1.8432165e-01 }, { 2.3696735e-01, 1.8743951e-01 },
        { 2.3449354e-01, 1.9052526e-01 }, { 2.3197955e-01, 1.9357836e-01 },
        { 2.2942581e-01, 1.9659830e-01 }, { 2.2683276e-01, 1.9958454e-01 },
        { 2.2420085e-01, 2.0253659e-01 }, { 2.2153052e-01, 2.0545394e-01 },
        { 2.1882223e-01, 2.0833608e-01 }, { 2.1607645e-01, 2.1118253e-01 },
        { 2.1329364e-01, 2.1399279e-01 }, { 2.1047429e-01, 2.1676638e-01 },
        { 2.0761888e-01, 2.1950284e-01 }, { 2.0472788e-01, 2.2220168e-01 },
        { 2.0180182e-01, 2.2486245e-01 }, { 1.9884117e-01, 2.2748469e-01 },
        { 1.9584645e-01, 2.3006795e-01 }, { 1.9281818e-01, 2.3261179e-01 },
        { 1.8975686e-01, 2.3511577e-01 }, { 1.8666303e-01, 2.3757947e-01 },
        { 1.8353722e-01, 2.4000246e-01 }, { 1.8037996e-01, 2.4238433e-01 },
        { 1.7719180e-01, 2.4472466e-01 }, { 1.7397327e-01, 2.4702306e-01 },
        { 1.7072493e-01, 2.4927914e-01 }, { 1.6744734e-01, 2.5149250e-01 },
        { 1.6414106e-01, 2.5366278e-01 }, { 1.6080666e-01, 2.5578958e-01 },
        { 1.5744470e-01, 2.5787256e-01 }, { 1.5405576e-01, 2.5991136e-01 },
        { 1.5064043e-01, 2.6190562e-01 }, { 1.4719929e-01, 2.6385500e-01 },
        { 1.4373292e-01, 2.6575918e-01 }, { 1.4024192e-01, 2.6761782e-01 },
        { 1.3672690e-01, 2.6943060e-01 }, { 1.3318845e-01, 2.7119722e-01 },
        { 1.2962718e-01, 2.7291736e-01 }, { 1.2604369e-01, 2.7459075e-01 },
        { 1.2243861e-01, 2.7621709e-01 }, { 1.1881255e-01, 2.7779609e-01 },
        { 1.1516614e-01, 2.7932750e-01 }, { 1.1149999e-01, 2.8081105e-01 },
        { 1.0781473e-01, 2.8224648e-01 }, { 1.0411100e-01, 2.8363355e-01 },
        { 1.0038943e-01, 2.8497202e-01 }, { 9.6650664e-02, 2.8626167e-01 },
        { 9.2895335e-02, 2.8750226e-01 }, { 8.9124088e-02, 2.8869359e-01 },
        { 8.5337570e-02, 2.8983546e-01 }, { 8.1536430e-02, 2.9092766e-01 },
        { 7.7721319e-02, 2.9197001e-01 }, { 7.3892891e-02, 2.9296234e-01 },
        { 7.0051802e-02, 2.9390447e-01 }, { 6.6198710e-02, 2.9479624e-01 },
        { 6.2334275e-02, 2.9563750e-01 }, { 5.8459159e-02, 2.9642810e-01 },
        { 5.4574027e-02, 2.9716791e-01 }, { 5.0679543e-02, 2.9785681e-01 },
        { 4.6776376e-02, 2.9849466e-01 }, { 4.2865195e-02, 2.9908137e-01 },
        { 3.8946668e-02, 2.9961684e-01 }, { 3.5021468e-02, 3.0010097e-01 },
        { 3.1090267e-02, 3.0053367e-01 }, { 2.7153740e-02, 3.0091488e-01 },
        { 2.3212559e-02, 3.0124454e-01 }, { 1.9267401e-02, 3.0152257e-01 },
        { 1.5318942e-02, 3.0174894e-01 }, { 1.1367858e-02, 3.0192361e-01 },
        { 7.4148264e-03, 3.0204654e-01 }, { 3.4605241e-03, 3.0211772e-01 },
    }
};

static const struct lc3_mdct_rot_def mdct_rot_640 = {
    .n4 = 640/4, .w = (const struct lc3_complex []){
        { 2.8117045e-01, 3.4504823e-04 }, { 2.8115351e-01, 3.1053717e-03 },
        { 2.8110948e-01, 5.8653959e-03 }, { 2.8103835e-01, 8.6248547e-03 },
        { 2.8094013e-01, 1.1383482e-02 }, { 2.8081484e-01, 1.4141013e-02 },
        { 2.8066248e-01, 1.6897180e-02 }, { 2.8048307e-01, 1.9651719e-02 },
        { 2.8027662e-01, 2.2404364e-02 }, { 2.8004317e-01, 2.5154849e-02 },
        { 2.7978272e-01, 2.7902910e-02 }, { 2.7949530e-01, 3.0648282e-02 },
        { 2.7918095e-01, 3.3390700e-02 }, { 2.7883969e-01, 3.6129899e-02 },
        { 2.7847155e-01, 3.8865616e-02 }, { 2.7807658e-01, 4.1597587e-02 },
        { 2.7765480e-01, 4.4325549e-02 }, { 2.7720626e-01, 4.7049239e-02 },
        { 2.7673100e-01, 4.9768394e-02 }, { 2.7622908e-01, 5.2482752e-02 },
        { 2.7570052e-01, 5.5192052e-02 }, { 2.7514540e-01, 5.7896032e-02 },
        { 2.7456376e-01, 6.0594433e-02 }, { 2.7395565e-01, 6.3286992e-02 },
        { 2.7332114e-01, 6.5973453e-02 }, { 2.7266028e-01, 6.8653554e-02 },
        { 2.7197315e-01, 7.1327039e-02 }, { 2.7125980e-01, 7.3993649e-02 },
        { 2.7052031e-01, 7.6653127e-02 }, { 2.6975475e-01, 7.9305217e-02 },
        { 2.6896318e-01, 8.1949664e-02 }, { 2.6814570e-01, 8.4586212e-02 },
        { 2.6730236e-01, 8.7214608e-02 }, { 2.6643327e-01, 8.9834598e-02 },
        { 2.6553849e-01, 9.2445929e-02 }, { 2.6461813e-01, 9.5048350e-02 },
        { 2.6367225e-01, 9.7641610e-02 }, { 2.6270097e-01, 1.0022546e-01 },
        { 2.6170436e-01, 1.0279965e-01 }, { 2.6068253e-01, 1.0536393e-01 },
        { 2.5963558e-01, 1.0791806e-01 }, { 2.5856360e-01, 1.1046178e-01 },
        { 2.5746670e-01, 1.1299486e-01 }, { 2.5634499e-01, 1.1551705e-01 },
        { 2.5519857e-01, 1.1802810e-01 }, { 2.5402755e-01, 1.2052778e-01 },
        { 2.5283205e-01, 1.2301584e-01 }, { 2.5161218e-01, 1.2549204e-01 },
        { 2.5036806e-01, 1.2795615e-01 }, { 2.4909981e-01, 1.3040793e-01 },
        { 2.4780754e-01, 1.3284714e-01 }, { 2.4649140e-01, 1.3527354e-01 },
        { 2.4515150e-01, 1.3768691e-01 }, { 2.4378797e-01, 1.4008700e-01 },
        { 2.4240094e-01, 1.4247360e-01 }, { 2.4099055e-01, 1.4484646e-01 },
        { 2.3955693e-01, 1.4720536e-01 }, { 2.3810023e-01, 1.4955007e-01 },
        { 2.3662057e-01, 1.5188037e-01 }, { 2.3511811e-01, 1.5419603e-01 },
        { 2.3359299e-01, 1.5649683e-01 }, { 2.3204535e-01, 1.5878255e-01 },
        { 2.3047535e-01, 1.6105296e-01 }, { 2.2888313e-01, 1.6330785e-01 },
        { 2.2726886e-01, 1.6554699e-01 }, { 2.2563268e-01, 1.6777019e-01 },
        { 2.2397475e-01, 1.6997721e-01 }, { 2.2229524e-01, 1.7216785e-01 },
        { 2.2059430e-01, 1.7434190e-01 }, { 2.1887210e-01, 1.7649914e-01 },
        { 2.1712880e-01, 1.7863937e-01 }, { 2.1536458e-01, 1.8076239e-01 },
        { 2.1357960e-01, 1.8286798e-01 }, { 2.1177403e-01, 1.8495594e-01 },
        { 2.0994805e-01, 1.8702608e-01 }, { 2.0810184e-01, 1.8907820e-01 },
        { 2.0623557e-01, 1.9111209e-01 }, { 2.0434942e-01, 1.9312756e-01 },
        { 2.0244358e-01, 1.9512442e-01 }, { 2.0051823e-01, 1.9710247e-01 },
        { 1.9857355e-01, 1.9906152e-01 }, { 1.9660973e-01, 2.0100139e-01 },
        { 1.9462696e-01, 2.0292188e-01 }, { 1.9262543e-01, 2.0482282e-01 },
        { 1.9060533e-01, 2.0670401e-01 }, { 1.8856687e-01, 2.0856528e-01 },
        { 1.8651023e-01, 2.1040645e-01 }, { 1.8443562e-01, 2.1222734e-01 },
        { 1.8234322e-01, 2.1402778e-01 }, { 1.8023326e-01, 2.1580759e-01 },
        { 1.7810592e-01, 2.1756659e-01 }, { 1.7596142e-01, 2.1930463e-01 },
        { 1.7379995e-01, 2.2102153e-01 }, { 1.7162174e-01, 2.2271713e-01 },
        { 1.6942698e-01, 2.2439126e-01 }, { 1.6721590e-01, 2.2604377e-01 },
        { 1.6498869e-01, 2.2767449e-01 }, { 1.6274559e-01, 2.2928326e-01 },
        { 1.6048680e-01, 2.3086994e-01 }, { 1.5821254e-01, 2.3243436e-01 },
        { 1.5592304e-01, 2.3397638e-01 }, { 1.5361850e-01, 2.3549585e-01 },
        { 1.5129916e-01, 2.3699263e-01 }, { 1.4896524e-01, 2.3846656e-01 },
        { 1.4661696e-01, 2.3991751e-01 }, { 1.4425454e-01, 2.4134533e-01 },
        { 1.4187823e-01, 2.4274989e-01 }, { 1.3948824e-01, 2.4413106e-01 },
        { 1.3708480e-01, 2.4548869e-01 }, { 1.3466815e-01, 2.4682267e-01 },
        { 1.3223853e-01, 2.4813285e-01 }, { 1.2979616e-01, 2.4941912e-01 },
        { 1.2734127e-01, 2.5068135e-01 }, { 1.2487412e-01, 2.5191942e-01 },
        { 1.2239493e-01, 2.5313321e-01 }, { 1.1990394e-01, 2.5432260e-01 },
        { 1.1740139e-01, 2.5548748e-01 }, { 1.1488753e-01, 2.5662774e-01 },
        { 1.1236260e-01, 2.5774326e-01 }, { 1.0982684e-01, 2.5883394e-01 },
        { 1.0728049e-01, 2.5989967e-01 }, { 1.0472380e-01, 2.6094035e-01 },
        { 1.0215702e-01, 2.6195588e-01 }, { 9.9580393e-02, 2.6294617e-01 },
        { 9.6994168e-02, 2.6391111e-01 }, { 9.4398594e-02, 2.6485061e-01 },
        { 9.1793922e-02, 2.6576459e-01 }, { 8.9180402e-02, 2.6665295e-01 },
        { 8.6558287e-02, 2.6751562e-01 }, { 8.3927830e-02, 2.6835249e-01 },
        { 8.1289283e-02, 2.6916351e-01 }, { 7.8642901e-02, 2.6994858e-01 },
        { 7.5988940e-02, 2.7070763e-01 }, { 7.3327655e-02, 2.7144059e-01 },
        { 7.0659302e-02, 2.7214739e-01 }, { 6.7984139e-02, 2.7282796e-01 },
        { 6.5302424e-02, 2.7348224e-01 }, { 6.2614414e-02, 2.7411015e-01 },
        { 5.9920370e-02, 2.7471165e-01 }, { 5.7220550e-02, 2.7528667e-01 },
        { 5.4515216e-02, 2.7583516e-01 }, { 5.1804627e-02, 2.7635706e-01 },
        { 4.9089045e-02, 2.7685232e-01 }, { 4.6368731e-02, 2.7732090e-01 },
        { 4.3643949e-02, 2.7776275e-01 }, { 4.0914960e-02, 2.7817783e-01 },
        { 3.8182028e-02, 2.7856610e-01 }, { 3.5445415e-02, 2.7892752e-01 },
        { 3.2705387e-02, 2.7926206e-01 }, { 2.9962206e-02, 2.7956968e-01 },
        { 2.7216137e-02, 2.7985036e-01 }, { 2.4467445e-02, 2.8010406e-01 },
        { 2.1716395e-02, 2.8033077e-01 }, { 1.8963252e-02, 2.8053046e-01 },
        { 1.6208281e-02, 2.8070310e-01 }, { 1.3451748e-02, 2.8084870e-01 },
        { 1.0693918e-02, 2.8096723e-01 }, { 7.9350576e-03, 2.8105867e-01 },
        { 5.1754324e-03, 2.8112303e-01 }, { 2.4153085e-03, 2.8116029e-01 },
    }
};

static const struct lc3_mdct_rot_def mdct_rot_720 = {
    .n4 = 720/4, .w = (const struct lc3_complex []){
        { 2.7301192e-01, 2.9780993e-04 }, { 2.7299893e-01, 2.6802468e-03 },
        { 2.7296515e-01, 5.0624796e-03 }, { 2.7291057e-01, 7.4443269e-03 },
        { 2.7283522e-01, 9.8256072e-03 }, { 2.7273909e-01, 1.2206139e-02 },
        { 2.7262218e-01, 1.4585742e-02 }, { 2.7248452e-01, 1.6964234e-02 },
        { 2.7232611e-01, 1.9341434e-02 }, { 2.7214695e-01, 2.1717161e-02 },
        { 2.7194708e-01, 2.4091234e-02 }, { 2.7172649e-01, 2.6463472e-02 },
        { 2.7148521e-01, 2.8833695e-02 }, { 2.7122325e-01, 3.1201723e-02 },
        { 2.7094064e-01, 3.3567374e-02 }, { 2.7063740e-01, 3.5930469e-02 },
        { 2.7031354e-01, 3.8290828e-02 }, { 2.6996910e-01, 4.0648270e-02 },
        { 2.6960411e-01, 4.3002618e-02 }, { 2.6921858e-01, 4.5353690e-02 },
        { 2.6881255e-01, 4.7701309e-02 }, { 2.6838604e-01, 5.0045294e-02 },
        { 2.6793910e-01, 5.2385469e-02 }, { 2.6747176e-01, 5.4721655e-02 },
        { 2.6698404e-01, 5.7053673e-02 }, { 2.6647599e-01, 5.9381346e-02 },
        { 2.6594765e-01, 6.1704497e-02 }, { 2.6539906e-01, 6.4022949e-02 },
        { 2.6483026e-01, 6.6336526e-02 }, { 2.6424128e-01, 6.8645051e-02 },
        { 2.6363219e-01, 7.0948348e-02 }, { 2.6300302e-01, 7.3246242e-02 },
        { 2.6235382e-01, 7.5538558e-02 }, { 2.6168464e-01, 7.7825122e-02 },
        { 2.6099553e-01, 8.0105759e-02 }, { 2.6028655e-01, 8.2380295e-02 },
        { 2.5955774e-01, 8.4648558e-02 }, { 2.5880917e-01, 8.6910375e-02 },
        { 2.5804089e-01, 8.9165573e-02 }, { 2.5725296e-01, 9.1413981e-02 },
        { 2.5644543e-01, 9.3655427e-02 }, { 2.5561838e-01, 9.5889741e-02 },
        { 2.5477186e-01, 9.8116753e-02 }, { 2.5390594e-01, 1.0033629e-01 },
        { 2.5302069e-01, 1.0254819e-01 }, { 2.5211616e-01, 1.0475228e-01 },
        { 2.5119244e-01, 1.0694839e-01 }, { 2.5024958e-01, 1.0913636e-01 },
        { 2.4928767e-01, 1.1131602e-01 }, { 2.4830678e-01, 1.1348720e-01 },
        { 2.4730697e-01, 1.1564973e-01 }, { 2.4628833e-01, 1.1780346e-01 },
        { 2.4525094e-01, 1.1994822e-01 }, { 2.4419487e-01, 1.2208384e-01 },
        { 2.4312020e-01, 1.2421017e-01 }, { 2.4202702e-01, 1.2632704e-01 },
        { 2.4091541e-01, 1.2843429e-01 }, { 2.3978545e-01, 1.3053175e-01 },
        { 2.3863723e-01, 1.3261928e-01 }, { 2.3747083e-01, 1.3469670e-01 },
        { 2.3628636e-01, 1.3676387e-01 }, { 2.3508388e-01, 1.3882063e-01 },
        { 2.3386351e-01, 1.4086681e-01 }, { 2.3262533e-01, 1.4290226e-01 },
        { 2.3136943e-01, 1.4492683e-01 }, { 2.3009591e-01, 1.4694037e-01 },
        { 2.2880487e-01, 1.4894272e-01 }, { 2.2749640e-01, 1.5093372e-01 },
        { 2.2617061e-01, 1.5291323e-01 }, { 2.2482759e-01, 1.5488109e-01 },
        { 2.2346746e-01, 1.5683716e-01 }, { 2.2209030e-01, 1.5878128e-01 },
        { 2.2069624e-01, 1.6071332e-01 }, { 2.1928536e-01, 1.6263311e-01 },
        { 2.1785779e-01, 1.6454052e-01 }, { 2.1641363e-01, 1.6643540e-01 },
        { 2.1495298e-01, 1.6831760e-01 }, { 2.1347597e-01, 1.7018699e-01 },
        { 2.1198270e-01, 1.7204341e-01 }, { 2.1047328e-01, 1.7388674e-01 },
        { 2.0894784e-01, 1.7571682e-01 }, { 2.0740648e-01, 1.7753352e-01 },
        { 2.0584933e-01, 1.7933670e-01 }, { 2.0427651e-01, 1.8112622e-01 },
        { 2.0268812e-01, 1.8290195e-01 }, { 2.0108431e-01, 1.8466375e-01 },
        { 1.9946518e-01, 1.8641149e-01 }, { 1.9783085e-01, 1.8814503e-01 },
        { 1.9618147e-01, 1.8986424e-01 }, { 1.9451714e-01, 1.9156900e-01 },
        { 1.9283800e-01, 1.9325917e-01 }, { 1.9114417e-01, 1.9493462e-01 },
        { 1.8943579e-01, 1.9659522e-01 }, { 1.8771298e-01, 1.9824085e-01 },
        { 1.8597588e-01, 1.9987139e-01 }, { 1.8422461e-01, 2.0148670e-01 },
        { 1.8245932e-01, 2.0308667e-01 }, { 1.8068013e-01, 2.0467118e-01 },
        { 1.7888718e-01, 2.0624010e-01 }, { 1.7708060e-01, 2.0779331e-01 },
        { 1.7526055e-01, 2.0933070e-01 }, { 1.7342714e-01, 2.1085214e-01 },
        { 1.7158053e-01, 2.1235753e-01 }, { 1.6972085e-01, 2.1384675e-01 },
        { 1.6784825e-01, 2.1531968e-01 }, { 1.6596286e-01, 2.1677622e-01 },
        { 1.6406484e-01, 2.1821624e-01 }, { 1.6215432e-01, 2.1963965e-01 },
        { 1.6023145e-01, 2.2104633e-01 }, { 1.5829638e-01, 2.2243618e-01 },
        { 1.5634925e-01, 2.2380909e-01 }, { 1.5439022e-01, 2.2516496e-01 },
        { 1.5241943e-01, 2.2650368e-01 }, { 1.5043704e-01, 2.2782514e-01 },
        { 1.4844319e-01, 2.2912926e-01 }, { 1.4643803e-01, 2.3041593e-01 },
        { 1.4442172e-01, 2.3168506e-01 }, { 1.4239441e-01, 2.3293654e-01 },
        { 1.4035626e-01, 2.3417028e-01 }, { 1.3830742e-01, 2.3538618e-01 },
        { 1.3624805e-01, 2.3658417e-01 }, { 1.3417830e-01, 2.3776413e-01 },
        { 1.3209834e-01, 2.3892599e-01 }, { 1.3000831e-01, 2.4006965e-01 },
        { 1.2790838e-01, 2.4119503e-01 }, { 1.2579872e-01, 2.4230205e-01 },
        { 1.2367947e-01, 2.4339061e-01 }, { 1.2155080e-01, 2.4446063e-01 },
        { 1.1941288e-01, 2.4551204e-01 }, { 1.1726586e-01, 2.4654476e-01 },
        { 1.1510992e-01, 2.4755869e-01 }, { 1.1294520e-01, 2.4855378e-01 },
        { 1.1077189e-01, 2.4952993e-01 }, { 1.0859014e-01, 2.5048709e-01 },
        { 1.0640012e-01, 2.5142516e-01 }, { 1.0420200e-01, 2.5234410e-01 },
        { 1.0199594e-01, 2.5324381e-01 }, { 9.9782117e-02, 2.5412424e-01 },
        { 9.7560694e-02, 2.5498531e-01 }, { 9.5331841e-02, 2.5582697e-01 },
        { 9.3095728e-02, 2.5664915e-01 }, { 9.0852525e-02, 2.5745178e-01 },
        { 8.8602403e-02, 2.5823480e-01 }, { 8.6345534e-02, 2.5899816e-01 },
        { 8.4082090e-02, 2.5974180e-01 }, { 8.1812242e-02, 2.6046565e-01 },
        { 7.9536165e-02, 2.6116967e-01 }, { 7.7254030e-02, 2.6185380e-01 },
        { 7.4966012e-02, 2.6251799e-01 }, { 7.2672284e-02, 2.6316219e-01 },
        { 7.0373023e-02, 2.6378635e-01 }, { 6.8068403e-02, 2.6439042e-01 },
        { 6.5758598e-02, 2.6497435e-01 }, { 6.3443786e-02, 2.6553810e-01 },
        { 6.1124143e-02, 2.6608164e-01 }, { 5.8799845e-02, 2.6660491e-01 },
        { 5.6471069e-02, 2.6710788e-01 }, { 5.4137992e-02, 2.6759050e-01 },
        { 5.1800793e-02, 2.6805275e-01 }, { 4.9459648e-02, 2.6849459e-01 },
        { 4.7114738e-02, 2.6891597e-01 }, { 4.4766239e-02, 2.6931688e-01 },
        { 4.2414331e-02, 2.6969728e-01 }, { 4.0059193e-02, 2.7005714e-01 },
        { 3.7701004e-02, 2.7039644e-01 }, { 3.5339945e-02, 2.7071514e-01 },
        { 3.2976194e-02, 2.7101323e-01 }, { 3.0609932e-02, 2.7129068e-01 },
        { 2.8241338e-02, 2.7154747e-01 }, { 2.5870594e-02, 2.7178357e-01 },
        { 2.3497880e-02, 2.7199899e-01 }, { 2.1123377e-02, 2.7219369e-01 },
        { 1.8747265e-02, 2.7236765e-01 }, { 1.6369725e-02, 2.7252088e-01 },
        { 1.3990938e-02, 2.7265336e-01 }, { 1.1611086e-02, 2.7276507e-01 },
        { 9.2303502e-03, 2.7285601e-01 }, { 6.8489111e-03, 2.7292617e-01 },
        { 4.4669505e-03, 2.7297554e-01 }, { 2.0846497e-03, 2.7300413e-01 },
    }
};

static const struct lc3_mdct_rot_def mdct_rot_960 = {
    .n4 = 960/4, .w = (const struct lc3_complex []){
        { 2.5406629e-01, 2.0785754e-04 }, { 2.5405949e-01, 1.8707012e-03 },
        { 2.5404180e-01, 3.5334647e-03 }, { 2.5401323e-01, 5.1960769e-03 },
        { 2.5397379e-01, 6.8584664e-03 }, { 2.5392346e-01, 8.5205622e-03 },
        { 2.5386225e-01, 1.0182293e-02 }, { 2.5379017e-01, 1.1843588e-02 },
        { 2.5370722e-01, 1.3504375e-02 }, { 2.5361340e-01, 1.5164584e-02 },
        { 2.5350872e-01, 1.6824143e-02 }, { 2.5339318e-01, 1.8482981e-02 },
        { 2.5326678e-01, 2.0141028e-02 }, { 2.5312953e-01, 2.1798212e-02 },
        { 2.5298144e-01, 2.3454462e-02 }, { 2.5282252e-01, 2.5109708e-02 },
        { 2.5265276e-01, 2.6763878e-02 }, { 2.5247218e-01, 2.8416901e-02 },
        { 2.5228079e-01, 3.0068707e-02 }, { 2.5207859e-01, 3.1719225e-02 },
        { 2.5186559e-01, 3.3368385e-02 }, { 2.5164180e-01, 3.5016115e-02 },
        { 2.5140723e-01, 3.6662344e-02 }, { 2.5116189e-01, 3.8307004e-02 },
        { 2.5090580e-01, 3.9950022e-02 }, { 2.5063895e-01, 4.1591330e-02 },
        { 2.5036137e-01, 4.3230855e-02 }, { 2.5007306e-01, 4.4868529e-02 },
        { 2.4977405e-01, 4.6504281e-02 }, { 2.4946433e-01, 4.8138040e-02 },
        { 2.4914393e-01, 4.9769738e-02 }, { 2.4881285e-01, 5.1399303e-02 },
        { 2.4847112e-01, 5.3026667e-02 }, { 2.4811874e-01, 5.4651759e-02 },
        { 2.4775573e-01, 5.6274511e-02 }, { 2.4738211e-01, 5.7894851e-02 },
        { 2.4699789e-01, 5.9512712e-02 }, { 2.4660310e-01, 6.1128023e-02 },
        { 2.4619774e-01, 6.2740716e-02 }, { 2.4578183e-01, 6.4350721e-02 },
        { 2.4535539e-01, 6.5957969e-02 }, { 2.4491845e-01, 6.7562392e-02 },
        { 2.4447101e-01, 6.9163921e-02 }, { 2.4401310e-01, 7.0762488e-02 },
        { 2.4354474e-01, 7.2358023e-02 }, { 2.4306594e-01, 7.3950458e-02 },
        { 2.4257673e-01, 7.5539726e-02 }, { 2.4207714e-01, 7.7125757e-02 },
        { 2.4156717e-01, 7.8708485e-02 }, { 2.4104685e-01, 8.0287842e-02 },
        { 2.4051621e-01, 8.1863759e-02 }, { 2.3997527e-01, 8.3436169e-02 },
        { 2.3942404e-01, 8.5005005e-02 }, { 2.3886256e-01, 8.6570200e-02 },
        { 2.3829085e-01, 8.8131686e-02 }, { 2.3770893e-01, 8.9689398e-02 },
        { 2.3711683e-01, 9.1243267e-02 }, { 2.3651456e-01, 9.2793227e-02 },
        { 2.3590217e-01, 9.4339213e-02 }, { 2.3527968e-01, 9.5881158e-02 },
        { 2.3464710e-01, 9.7418995e-02 }, { 2.3400447e-01, 9.8952659e-02 },
        { 2.3335182e-01, 1.0048208e-01 }, { 2.3268918e-01, 1.0200721e-01 },
        { 2.3201656e-01, 1.0352796e-01 }, { 2.3133401e-01, 1.0504427e-01 },
        { 2.3064154e-01, 1.0655609e-01 }, { 2.2993920e-01, 1.0806334e-01 },
        { 2.2922701e-01, 1.0956597e-01 }, { 2.2850500e-01, 1.1106390e-01 },
        { 2.2777320e-01, 1.1255707e-01 }, { 2.2703164e-01, 1.1404542e-01 },
        { 2.2628036e-01, 1.1552888e-01 }, { 2.2551938e-01, 1.1700740e-01 },
        { 2.2474874e-01, 1.1848090e-01 }, { 2.2396848e-01, 1.1994933e-01 },
        { 2.2317862e-01, 1.2141262e-01 }, { 2.2237920e-01, 1.2287071e-01 },
        { 2.2157026e-01, 1.2432354e-01 }, { 2.2075182e-01, 1.2577104e-01 },
        { 2.1992393e-01, 1.2721315e-01 }, { 2.1908662e-01, 1.2864982e-01 },
        { 2.1823992e-01, 1.3008097e-01 }, { 2.1738388e-01, 1.3150655e-01 },
        { 2.1651852e-01, 1.3292650e-01 }, { 2.1564388e-01, 1.3434075e-01 },
        { 2.1476001e-01, 1.3574925e-01 }, { 2.1386694e-01, 1.3715193e-01 },
        { 2.1296471e-01, 1.3854874e-01 }, { 2.1205336e-01, 1.3993962e-01 },
        { 2.1113292e-01, 1.4132449e-01 }, { 2.1020344e-01, 1.4270332e-01 },
        { 2.0926495e-01, 1.4407603e-01 }, { 2.0831750e-01, 1.4544257e-01 },
        { 2.0736113e-01, 1.4680288e-01 }, { 2.0639587e-01, 1.4815690e-01 },
        { 2.0542177e-01, 1.4950458e-01 }, { 2.0443887e-01, 1.5084585e-01 },
        { 2.0344722e-01, 1.5218066e-01 }, { 2.0244685e-01, 1.5350895e-01 },
        { 2.0143780e-01, 1.5483066e-01 }, { 2.0042013e-01, 1.5614574e-01 },
        { 1.9939388e-01, 1.5745414e-01 }, { 1.9835908e-01, 1.5875578e-01 },
        { 1.9731578e-01, 1.6005063e-01 }, { 1.9626403e-01, 1.6133862e-01 },
        { 1.9520388e-01, 1.6261970e-01 }, { 1.9413536e-01, 1.6389382e-01 },
        { 1.9305853e-01, 1.6516091e-01 }, { 1.9197343e-01, 1.6642093e-01 },
        { 1.9088010e-01, 1.6767382e-01 }, { 1.8977860e-01, 1.6891953e-01 },
        { 1.8866896e-01, 1.7015800e-01 }, { 1.8755125e-01, 1.7138918e-01 },
        { 1.8642550e-01, 1.7261302e-01 }, { 1.8529177e-01, 1.7382947e-01 },
        { 1.8415009e-01, 1.7503847e-01 }, { 1.8300053e-01, 1.7623997e-01 },
        { 1.8184314e-01, 1.7743392e-01 }, { 1.8067795e-01, 1.7862027e-01 },
        { 1.7950502e-01, 1.7979897e-01 }, { 1.7832440e-01, 1.8096997e-01 },
        { 1.7713614e-01, 1.8213322e-01 }, { 1.7594030e-01, 1.8328866e-01 },
        { 1.7473692e-01, 1.8443625e-01 }, { 1.7352605e-01, 1.8557595e-01 },
        { 1.7230775e-01, 1.8670769e-01 }, { 1.7108207e-01, 1.8783143e-01 },
        { 1.6984906e-01, 1.8894713e-01 }, { 1.6860878e-01, 1.9005474e-01 },
        { 1.6736127e-01, 1.9115420e-01 }, { 1.6610659e-01, 1.9224547e-01 },
        { 1.6484480e-01, 1.9332851e-01 }, { 1.6357595e-01, 1.9440327e-01 },
        { 1.6230008e-01, 1.9546970e-01 }, { 1.6101727e-01, 1.9652776e-01 },
        { 1.5972756e-01, 1.9757740e-01 }, { 1.5843101e-01, 1.9861857e-01 },
        { 1.5712767e-01, 1.9965124e-01 }, { 1.5581760e-01, 2.0067536e-01 },
        { 1.5450085e-01, 2.0169087e-01 }, { 1.5317749e-01, 2.0269775e-01 },
        { 1.5184756e-01, 2.0369595e-01 }, { 1.5051113e-01, 2.0468542e-01 },
        { 1.4916826e-01, 2.0566612e-01 }, { 1.4781899e-01, 2.0663801e-01 },
        { 1.4646339e-01, 2.0760105e-01 }, { 1.4510152e-01, 2.0855520e-01 },
        { 1.4373343e-01, 2.0950041e-01 }, { 1.4235918e-01, 2.1043665e-01 },
        { 1.4097884e-01, 2.1136388e-01 }, { 1.3959246e-01, 2.1228205e-01 },
        { 1.3820009e-01, 2.1319113e-01 }, { 1.3680181e-01, 2.1409107e-01 },
        { 1.3539767e-01, 2.1498185e-01 }, { 1.3398773e-01, 2.1586341e-01 },
        { 1.3257204e-01, 2.1673573e-01 }, { 1.3115068e-01, 2.1759876e-01 },
        { 1.2972370e-01, 2.1845247e-01 }, { 1.2829117e-01, 2.1929683e-01 },
        { 1.2685313e-01, 2.2013179e-01 }, { 1.2540967e-01, 2.2095732e-01 },
        { 1.2396083e-01, 2.2177339e-01 }, { 1.2250668e-01, 2.2257995e-01 },
        { 1.2104729e-01, 2.2337698e-01 }, { 1.1958271e-01, 2.2416445e-01 },
        { 1.1811300e-01, 2.2494231e-01 }, { 1.1663824e-01, 2.2571053e-01 },
        { 1.1515848e-01, 2.2646909e-01 }, { 1.1367379e-01, 2.2721794e-01 },
        { 1.1218422e-01, 2.2795706e-01 }, { 1.1068986e-01, 2.2868642e-01 },
        { 1.0919075e-01, 2.2940598e-01 }, { 1.0768696e-01, 2.3011571e-01 },
        { 1.0617856e-01, 2.3081559e-01 }, { 1.0466561e-01, 2.3150558e-01 },
        { 1.0314818e-01, 2.3218565e-01 }, { 1.0162633e-01, 2.3285577e-01 },
        { 1.0010013e-01, 2.3351592e-01 }, { 9.8569638e-02, 2.3416607e-01 },
        { 9.7034924e-02, 2.3480619e-01 }, { 9.5496054e-02, 2.3543625e-01 },
        { 9.3953093e-02, 2.3605622e-01 }, { 9.2406107e-02, 2.3666608e-01 },
        { 9.0855163e-02, 2.3726580e-01 }, { 8.9300327e-02, 2.3785536e-01 },
        { 8.7741666e-02, 2.3843473e-01 }, { 8.6179246e-02, 2.3900389e-01 },
        { 8.4613135e-02, 2.3956281e-01 }, { 8.3043399e-02, 2.4011147e-01 },
        { 8.1470106e-02, 2.4064984e-01 }, { 7.9893322e-02, 2.4117790e-01 },
        { 7.8313117e-02, 2.4169563e-01 }, { 7.6729556e-02, 2.4220301e-01 },
        { 7.5142709e-02, 2.4270001e-01 }, { 7.3552643e-02, 2.4318662e-01 },
        { 7.1959427e-02, 2.4366281e-01 }, { 7.0363128e-02, 2.4412856e-01 },
        { 6.8763814e-02, 2.4458385e-01 }, { 6.7161555e-02, 2.4502867e-01 },
        { 6.5556419e-02, 2.4546299e-01 }, { 6.3948475e-02, 2.4588679e-01 },
        { 6.2337792e-02, 2.4630007e-01 }, { 6.0724438e-02, 2.4670279e-01 },
        { 5.9108483e-02, 2.4709494e-01 }, { 5.7489996e-02, 2.4747651e-01 },
        { 5.5869046e-02, 2.4784748e-01 }, { 5.4245703e-02, 2.4820783e-01 },
        { 5.2620036e-02, 2.4855755e-01 }, { 5.0992116e-02, 2.4889662e-01 },
        { 4.9362011e-02, 2.4922503e-01 }, { 4.7729791e-02, 2.4954276e-01 },
        { 4.6095527e-02, 2.4984980e-01 }, { 4.4459288e-02, 2.5014615e-01 },
        { 4.2821145e-02, 2.5043177e-01 }, { 4.1181167e-02, 2.5070667e-01 },
        { 3.9539426e-02, 2.5097083e-01 }, { 3.7895990e-02, 2.5122424e-01 },
        { 3.6250931e-02, 2.5146688e-01 }, { 3.4604320e-02, 2.5169876e-01 },
        { 3.2956226e-02, 2.5191985e-01 }, { 3.1306720e-02, 2.5213015e-01 },
        { 2.9655874e-02, 2.5232965e-01 }, { 2.8003757e-02, 2.5251834e-01 },
        { 2.6350440e-02, 2.5269621e-01 }, { 2.4695994e-02, 2.5286326e-01 },
        { 2.3040491e-02, 2.5301948e-01 }, { 2.1384001e-02, 2.5316486e-01 },
        { 1.9726595e-02, 2.5329940e-01 }, { 1.8068343e-02, 2.5342308e-01 },
        { 1.6409318e-02, 2.5353591e-01 }, { 1.4749590e-02, 2.5363788e-01 },
        { 1.3089230e-02, 2.5372898e-01 }, { 1.1428309e-02, 2.5380921e-01 },
        { 9.7668984e-03, 2.5387857e-01 }, { 8.1050697e-03, 2.5393706e-01 },
        { 6.4428938e-03, 2.5398467e-01 }, { 4.7804419e-03, 2.5402140e-01 },
        { 3.1177852e-03, 2.5404724e-01 }, { 1.4549950e-03, 2.5406221e-01 },
    }
};

const struct lc3_mdct_rot_def * lc3_mdct_rot[LC3_NUM_DT][LC3_NUM_SRATE] = {
    [LC3_DT_7M5] = { &mdct_rot_120, &mdct_rot_240, &mdct_rot_360,
                     &mdct_rot_480, &mdct_rot_720                },
    [LC3_DT_10M] = { &mdct_rot_160, &mdct_rot_320, &mdct_rot_480,
                     &mdct_rot_640, &mdct_rot_960                }
};


/**
 * Low delay MDCT windows (cf. 3.7.3)
 */

static const float mdct_win_10m_80[80+50] = {
    -7.07854671e-04, -2.09819773e-03, -4.52519808e-03, -8.23397633e-03,
    -1.33771310e-02, -1.99972156e-02, -2.80090946e-02, -3.72150208e-02,
    -4.73176826e-02, -5.79465483e-02, -6.86760675e-02, -7.90464744e-02,
    -8.85970547e-02, -9.68830362e-02, -1.03496124e-01, -1.08076646e-01,
    -1.10324226e-01, -1.09980985e-01, -1.06817214e-01, -1.00619042e-01,
    -9.11645251e-02, -7.82061748e-02, -6.14668812e-02, -4.06336286e-02,
    -1.53632952e-02,  1.47015507e-02,  4.98973651e-02,  9.05036926e-02,
     1.36691102e-01,  1.88468639e-01,  2.45645680e-01,  3.07778908e-01,
     3.74164237e-01,  4.43811480e-01,  5.15473546e-01,  5.87666172e-01,
     6.58761977e-01,  7.27057670e-01,  7.90875299e-01,  8.48664336e-01,
     8.99132024e-01,  9.41334815e-01,  9.74763483e-01,  9.99411473e-01,
     1.01576037e+00,  1.02473616e+00,  1.02763429e+00,  1.02599149e+00,
     1.02142721e+00,  1.01543986e+00,  1.00936693e+00,  1.00350816e+00,
     9.98889821e-01,  9.95313390e-01,  9.92594392e-01,  9.90577196e-01,
     9.89137162e-01,  9.88179075e-01,  9.87624927e-01,  9.87405628e-01,
     9.87452485e-01,  9.87695113e-01,  9.88064062e-01,  9.88492687e-01,
     9.88923003e-01,  9.89307497e-01,  9.89614633e-01,  9.89831927e-01,
     9.89969310e-01,  9.90060335e-01,  9.90157502e-01,  9.90325529e-01,
     9.90630379e-01,  9.91129889e-01,  9.91866549e-01,  9.92861973e-01,
     9.94115607e-01,  9.95603378e-01,  9.97279311e-01,  9.99078484e-01,
     1.00092237e+00,  1.00272811e+00,  1.00441604e+00,  1.00591922e+00,
     1.00718935e+00,  1.00820015e+00,  1.00894949e+00,  1.00945824e+00,
     1.00976898e+00,  1.00994034e+00,  1.01003945e+00,  1.01013232e+00,
     1.01027252e+00,  1.01049435e+00,  1.01080807e+00,  1.01120107e+00,
     1.01164127e+00,  1.01208013e+00,  1.01245818e+00,  1.01270696e+00,
     1.01275501e+00,  1.01253013e+00,  1.01196233e+00,  1.01098214e+00,
     1.00951244e+00,  1.00746086e+00,  1.00470868e+00,  1.00111141e+00,
     9.96504102e-01,  9.90720000e-01,  9.82376587e-01,  9.70882175e-01,
     9.54673298e-01,  9.32155386e-01,  9.01800368e-01,  8.62398408e-01,
     8.13281737e-01,  7.54455197e-01,  6.86658072e-01,  6.11348804e-01,
     5.30618165e-01,  4.47130985e-01,  3.63911468e-01,  2.84164703e-01,
     2.11020945e-01,  1.47228797e-01,  9.48266535e-02,  5.48243661e-02,
     2.70146141e-02,  9.99674359e-03,
};

static const float mdct_win_10m_160[160+100] = {
    -4.61989875e-04, -9.74716672e-04, -1.66447310e-03, -2.59710692e-03,
    -3.80628516e-03, -5.32460872e-03, -7.17588528e-03, -9.38248086e-03,
    -1.19527030e-02, -1.48952816e-02, -1.82066640e-02, -2.18757093e-02,
    -2.58847194e-02, -3.02086274e-02, -3.48159779e-02, -3.96706799e-02,
    -4.47269805e-02, -4.99422586e-02, -5.52633479e-02, -6.06371724e-02,
    -6.60096152e-02, -7.13196627e-02, -7.65117823e-02, -8.15296401e-02,
    -8.63113754e-02, -9.08041129e-02, -9.49537776e-02, -9.87073651e-02,
    -1.02020268e-01, -1.04843883e-01, -1.07138231e-01, -1.08869014e-01,
    -1.09996966e-01, -1.10489847e-01, -1.10322584e-01, -1.09462175e-01,
    -1.07883429e-01, -1.05561251e-01, -1.02465016e-01, -9.85701457e-02,
    -9.38468492e-02, -8.82630999e-02, -8.17879272e-02, -7.43878560e-02,
    -6.60218980e-02, -5.66565564e-02, -4.62445689e-02, -3.47458578e-02,
    -2.21158161e-02, -8.31042570e-03,  6.71769764e-03,  2.30064206e-02,
     4.06010646e-02,  5.95323909e-02,  7.98335419e-02,  1.01523314e-01,
     1.24617139e-01,  1.49115252e-01,  1.75006740e-01,  2.02269985e-01,
     2.30865538e-01,  2.60736512e-01,  2.91814469e-01,  3.24009570e-01,
     3.57217518e-01,  3.91314689e-01,  4.26157164e-01,  4.61592545e-01,
     4.97447159e-01,  5.33532682e-01,  5.69654673e-01,  6.05608382e-01,
     6.41183084e-01,  6.76165350e-01,  7.10340055e-01,  7.43494372e-01,
     7.75428189e-01,  8.05943723e-01,  8.34858937e-01,  8.62010834e-01,
     8.87259971e-01,  9.10486312e-01,  9.31596250e-01,  9.50522086e-01,
     9.67236671e-01,  9.81739750e-01,  9.94055718e-01,  1.00424751e+00,
     1.01240743e+00,  1.01865099e+00,  1.02311884e+00,  1.02597245e+00,
     1.02739752e+00,  1.02758583e+00,  1.02673867e+00,  1.02506178e+00,
     1.02275651e+00,  1.02000914e+00,  1.01699650e+00,  1.01391595e+00,
     1.01104487e+00,  1.00777386e+00,  1.00484875e+00,  1.00224501e+00,
     9.99939317e-01,  9.97905542e-01,  9.96120338e-01,  9.94559753e-01,
     9.93203161e-01,  9.92029727e-01,  9.91023065e-01,  9.90166895e-01,
     9.89448837e-01,  9.88855636e-01,  9.88377852e-01,  9.88005163e-01,
     9.87729546e-01,  9.87541274e-01,  9.87432981e-01,  9.87394992e-01,
     9.87419705e-01,  9.87497321e-01,  9.87620124e-01,  9.87778192e-01,
     9.87963798e-01,  9.88167801e-01,  9.88383520e-01,  9.88602222e-01,
     9.88818277e-01,  9.89024798e-01,  9.89217866e-01,  9.89392368e-01,
     9.89546334e-01,  9.89677201e-01,  9.89785920e-01,  9.89872536e-01,
     9.89941079e-01,  9.89994556e-01,  9.90039402e-01,  9.90081472e-01,
     9.90129379e-01,  9.90190227e-01,  9.90273445e-01,  9.90386228e-01,
     9.90537983e-01,  9.90734883e-01,  9.90984259e-01,  9.91290512e-01,
     9.91658694e-01,  9.92090615e-01,  9.92588721e-01,  9.93151653e-01,
     9.93779087e-01,  9.94466818e-01,  9.95211663e-01,  9.96006862e-01,
     9.96846133e-01,  9.97720337e-01,  9.98621352e-01,  9.99538258e-01,
     1.00046196e+00,  1.00138055e+00,  1.00228487e+00,  1.00316385e+00,
     1.00400915e+00,  1.00481138e+00,  1.00556397e+00,  1.00625986e+00,
     1.00689557e+00,  1.00746662e+00,  1.00797244e+00,  1.00841147e+00,
     1.00878601e+00,  1.00909776e+00,  1.00935176e+00,  1.00955240e+00,
     1.00970709e+00,  1.00982209e+00,  1.00990696e+00,  1.00996902e+00,
     1.01001789e+00,  1.01006081e+00,  1.01010656e+00,  1.01016113e+00,
     1.01023108e+00,  1.01031948e+00,  1.01043047e+00,  1.01056410e+00,
     1.01072136e+00,  1.01089966e+00,  1.01109699e+00,  1.01130817e+00,
     1.01152919e+00,  1.01175301e+00,  1.01197388e+00,  1.01218284e+00,
     1.01237303e+00,  1.01253506e+00,  1.01266098e+00,  1.01274058e+00,
     1.01276592e+00,  1.01272696e+00,  1.01261590e+00,  1.01242289e+00,
     1.01214046e+00,  1.01175881e+00,  1.01126996e+00,  1.01066368e+00,
     1.00993075e+00,  1.00905825e+00,  1.00803431e+00,  1.00684335e+00,
     1.00547001e+00,  1.00389477e+00,  1.00209885e+00,  1.00006069e+00,
     9.97760020e-01,  9.95174643e-01,  9.92286108e-01,  9.89075787e-01,
     9.84736245e-01,  9.79861353e-01,  9.74137862e-01,  9.67333198e-01,
     9.59253976e-01,  9.49698408e-01,  9.38463416e-01,  9.25356797e-01,
     9.10198679e-01,  8.92833832e-01,  8.73143784e-01,  8.51042044e-01,
     8.26483991e-01,  7.99468149e-01,  7.70043128e-01,  7.38302860e-01,
     7.04381434e-01,  6.68461648e-01,  6.30775533e-01,  5.91579959e-01,
     5.51170316e-01,  5.09891542e-01,  4.68101711e-01,  4.26177297e-01,
     3.84517234e-01,  3.43522867e-01,  3.03600465e-01,  2.65143468e-01,
     2.28528397e-01,  1.94102191e-01,  1.62173542e-01,  1.33001524e-01,
     1.06784043e-01,  8.36505724e-02,  6.36518811e-02,  4.67653841e-02,
     3.28807275e-02,  2.18305756e-02,  1.33638143e-02,  6.75812489e-03,
};

static const float mdct_win_10m_240[240+150] = {
    -3.61349642e-04, -7.07854671e-04, -1.07444364e-03, -1.53347854e-03,
    -2.09819773e-03, -2.77842087e-03, -3.58412992e-03, -4.52519808e-03,
    -5.60932724e-03, -6.84323454e-03, -8.23397633e-03, -9.78531476e-03,
    -1.14988030e-02, -1.33771310e-02, -1.54218168e-02, -1.76297991e-02,
    -1.99972156e-02, -2.25208056e-02, -2.51940630e-02, -2.80090946e-02,
    -3.09576509e-02, -3.40299627e-02, -3.72150208e-02, -4.05005325e-02,
    -4.38721922e-02, -4.73176826e-02, -5.08232534e-02, -5.43716664e-02,
    -5.79465483e-02, -6.15342620e-02, -6.51170816e-02, -6.86760675e-02,
    -7.21944781e-02, -7.56569598e-02, -7.90464744e-02, -8.23444256e-02,
    -8.55332458e-02, -8.85970547e-02, -9.15209110e-02, -9.42884745e-02,
    -9.68830362e-02, -9.92912326e-02, -1.01500847e-01, -1.03496124e-01,
    -1.05263700e-01, -1.06793998e-01, -1.08076646e-01, -1.09099730e-01,
    -1.09852449e-01, -1.10324226e-01, -1.10508462e-01, -1.10397741e-01,
    -1.09980985e-01, -1.09249277e-01, -1.08197423e-01, -1.06817214e-01,
    -1.05099580e-01, -1.03036011e-01, -1.00619042e-01, -9.78412002e-02,
    -9.46930422e-02, -9.11645251e-02, -8.72464453e-02, -8.29304391e-02,
    -7.82061748e-02, -7.30614243e-02, -6.74846818e-02, -6.14668812e-02,
    -5.49949726e-02, -4.80544442e-02, -4.06336286e-02, -3.27204559e-02,
    -2.43012258e-02, -1.53632952e-02, -5.89143427e-03,  4.12659586e-03,
     1.47015507e-02,  2.58473819e-02,  3.75765277e-02,  4.98973651e-02,
     6.28203403e-02,  7.63539773e-02,  9.05036926e-02,  1.05274712e-01,
     1.20670347e-01,  1.36691102e-01,  1.53334389e-01,  1.70595471e-01,
     1.88468639e-01,  2.06944996e-01,  2.26009300e-01,  2.45645680e-01,
     2.65834602e-01,  2.86554381e-01,  3.07778908e-01,  3.29476944e-01,
     3.51617148e-01,  3.74164237e-01,  3.97073959e-01,  4.20304305e-01,
     4.43811480e-01,  4.67544229e-01,  4.91449863e-01,  5.15473546e-01,
     5.39555764e-01,  5.63639982e-01,  5.87666172e-01,  6.11569531e-01,
     6.35289059e-01,  6.58761977e-01,  6.81923097e-01,  7.04709282e-01,
     7.27057670e-01,  7.48906896e-01,  7.70199019e-01,  7.90875299e-01,
     8.10878869e-01,  8.30157914e-01,  8.48664336e-01,  8.66354816e-01,
     8.83189685e-01,  8.99132024e-01,  9.14154056e-01,  9.28228255e-01,
     9.41334815e-01,  9.53461939e-01,  9.64604825e-01,  9.74763483e-01,
     9.83943539e-01,  9.92152910e-01,  9.99411473e-01,  1.00574608e+00,
     1.01118397e+00,  1.01576037e+00,  1.01951507e+00,  1.02249094e+00,
     1.02473616e+00,  1.02630410e+00,  1.02725098e+00,  1.02763429e+00,
     1.02751106e+00,  1.02694280e+00,  1.02599149e+00,  1.02471615e+00,
     1.02317598e+00,  1.02142721e+00,  1.01952157e+00,  1.01751012e+00,
     1.01543986e+00,  1.01346092e+00,  1.01165490e+00,  1.00936693e+00,
     1.00726318e+00,  1.00531319e+00,  1.00350816e+00,  1.00184079e+00,
     1.00030393e+00,  9.98889821e-01,  9.97591528e-01,  9.96401528e-01,
     9.95313390e-01,  9.94320108e-01,  9.93415896e-01,  9.92594392e-01,
     9.91851028e-01,  9.91179799e-01,  9.90577196e-01,  9.90038105e-01,
     9.89559439e-01,  9.89137162e-01,  9.88768437e-01,  9.88449792e-01,
     9.88179075e-01,  9.87952836e-01,  9.87769137e-01,  9.87624927e-01,
     9.87517995e-01,  9.87445813e-01,  9.87405628e-01,  9.87395112e-01,
     9.87411537e-01,  9.87452485e-01,  9.87514989e-01,  9.87596889e-01,
     9.87695113e-01,  9.87807582e-01,  9.87931200e-01,  9.88064062e-01,
     9.88203257e-01,  9.88347108e-01,  9.88492687e-01,  9.88638659e-01,
     9.88782558e-01,  9.88923003e-01,  9.89058172e-01,  9.89186767e-01,
     9.89307497e-01,  9.89419640e-01,  9.89522076e-01,  9.89614633e-01,
     9.89697035e-01,  9.89769260e-01,  9.89831927e-01,  9.89885257e-01,
     9.89930764e-01,  9.89969310e-01,  9.90002569e-01,  9.90032156e-01,
     9.90060335e-01,  9.90088981e-01,  9.90120659e-01,  9.90157502e-01,
     9.90202395e-01,  9.90257541e-01,  9.90325529e-01,  9.90408791e-01,
     9.90509649e-01,  9.90630379e-01,  9.90772711e-01,  9.90938744e-01,
     9.91129889e-01,  9.91347632e-01,  9.91592856e-01,  9.91866549e-01,
     9.92169132e-01,  9.92501085e-01,  9.92861973e-01,  9.93251918e-01,
     9.93670021e-01,  9.94115607e-01,  9.94587315e-01,  9.95083740e-01,
     9.95603378e-01,  9.96143992e-01,  9.96703453e-01,  9.97279311e-01,
     9.97869086e-01,  9.98469709e-01,  9.99078484e-01,  9.99691901e-01,
     1.00030819e+00,  1.00092237e+00,  1.00153264e+00,  1.00213546e+00,
     1.00272811e+00,  1.00330745e+00,  1.00387093e+00,  1.00441604e+00,
     1.00494055e+00,  1.00544214e+00,  1.00591922e+00,  1.00637030e+00,
     1.00679393e+00,  1.00718935e+00,  1.00755557e+00,  1.00789267e+00,
     1.00820015e+00,  1.00847842e+00,  1.00872788e+00,  1.00894949e+00,
     1.00914411e+00,  1.00931322e+00,  1.00945824e+00,  1.00958128e+00,
     1.00968409e+00,  1.00976898e+00,  1.00983831e+00,  1.00989455e+00,
     1.00994034e+00,  1.00997792e+00,  1.01001023e+00,  1.01003945e+00,
     1.01006820e+00,  1.01009839e+00,  1.01013232e+00,  1.01017166e+00,
     1.01021810e+00,  1.01027252e+00,  1.01033649e+00,  1.01041022e+00,
     1.01049435e+00,  1.01058887e+00,  1.01069350e+00,  1.01080807e+00,
     1.01093144e+00,  1.01106288e+00,  1.01120107e+00,  1.01134470e+00,
     1.01149190e+00,  1.01164127e+00,  1.01179028e+00,  1.01193757e+00,
     1.01208013e+00,  1.01221624e+00,  1.01234291e+00,  1.01245818e+00,
     1.01255888e+00,  1.01264286e+00,  1.01270696e+00,  1.01274895e+00,
     1.01276580e+00,  1.01275501e+00,  1.01271380e+00,  1.01263978e+00,
     1.01253013e+00,  1.01238231e+00,  1.01219407e+00,  1.01196233e+00,
     1.01168517e+00,  1.01135914e+00,  1.01098214e+00,  1.01055072e+00,
     1.01006213e+00,  1.00951244e+00,  1.00889869e+00,  1.00821592e+00,
     1.00746086e+00,  1.00662774e+00,  1.00571234e+00,  1.00470868e+00,
     1.00361147e+00,  1.00241429e+00,  1.00111141e+00,  9.99696165e-01,
     9.98162595e-01,  9.96504102e-01,  9.94714888e-01,  9.92789191e-01,
     9.90720000e-01,  9.88479371e-01,  9.85534766e-01,  9.82376587e-01,
     9.78974733e-01,  9.75162381e-01,  9.70882175e-01,  9.66080552e-01,
     9.60697640e-01,  9.54673298e-01,  9.47947935e-01,  9.40460905e-01,
     9.32155386e-01,  9.22977548e-01,  9.12874535e-01,  9.01800368e-01,
     8.89716328e-01,  8.76590897e-01,  8.62398408e-01,  8.47120080e-01,
     8.30747973e-01,  8.13281737e-01,  7.94729145e-01,  7.75110884e-01,
     7.54455197e-01,  7.32796355e-01,  7.10179084e-01,  6.86658072e-01,
     6.62296243e-01,  6.37168412e-01,  6.11348804e-01,  5.84920660e-01,
     5.57974743e-01,  5.30618165e-01,  5.02952396e-01,  4.75086883e-01,
     4.47130985e-01,  4.19204992e-01,  3.91425291e-01,  3.63911468e-01,
     3.36783777e-01,  3.10162784e-01,  2.84164703e-01,  2.58903371e-01,
     2.34488060e-01,  2.11020945e-01,  1.88599764e-01,  1.67310081e-01,
     1.47228797e-01,  1.28422307e-01,  1.10942255e-01,  9.48266535e-02,
     8.00991437e-02,  6.67676585e-02,  5.48243661e-02,  4.42458885e-02,
     3.49936100e-02,  2.70146141e-02,  2.02437018e-02,  1.46079676e-02,
     9.99674359e-03,  5.30523510e-03,
};

static const float mdct_win_10m_320[320+200] = {
    -3.02115349e-04, -5.86773749e-04, -8.36650400e-04, -1.12663536e-03,
    -1.47049294e-03, -1.87347339e-03, -2.33929236e-03, -2.87200807e-03,
    -3.47625639e-03, -4.15596382e-03, -4.91456379e-03, -5.75517250e-03,
    -6.68062338e-03, -7.69381692e-03, -8.79676075e-03, -9.99050307e-03,
    -1.12757412e-02, -1.26533415e-02, -1.41243899e-02, -1.56888962e-02,
    -1.73451209e-02, -1.90909737e-02, -2.09254671e-02, -2.28468479e-02,
    -2.48520772e-02, -2.69374670e-02, -2.90995249e-02, -3.13350463e-02,
    -3.36396073e-02, -3.60082097e-02, -3.84360174e-02, -4.09174603e-02,
    -4.34465489e-02, -4.60178672e-02, -4.86259851e-02, -5.12647420e-02,
    -5.39264475e-02, -5.66038431e-02, -5.92911675e-02, -6.19826820e-02,
    -6.46702555e-02, -6.73454222e-02, -7.00009902e-02, -7.26305701e-02,
    -7.52278496e-02, -7.77852594e-02, -8.02948025e-02, -8.27492454e-02,
    -8.51412546e-02, -8.74637912e-02, -8.97106934e-02, -9.18756408e-02,
    -9.39517698e-02, -9.59313774e-02, -9.78084326e-02, -9.95785130e-02,
    -1.01236117e-01, -1.02774104e-01, -1.04186122e-01, -1.05468025e-01,
    -1.06616088e-01, -1.07625538e-01, -1.08491230e-01, -1.09208742e-01,
    -1.09773615e-01, -1.10180886e-01, -1.10427188e-01, -1.10510836e-01,
    -1.10428147e-01, -1.10173922e-01, -1.09743736e-01, -1.09135313e-01,
    -1.08346734e-01, -1.07373994e-01, -1.06213016e-01, -1.04860615e-01,
    -1.03313240e-01, -1.01567316e-01, -9.96200551e-02, -9.74680323e-02,
    -9.51072362e-02, -9.25330338e-02, -8.97412522e-02, -8.67287769e-02,
    -8.34921384e-02, -8.00263990e-02, -7.63267954e-02, -7.23880616e-02,
    -6.82057680e-02, -6.37761143e-02, -5.90938600e-02, -5.41531632e-02,
    -4.89481272e-02, -4.34734711e-02, -3.77246130e-02, -3.16958761e-02,
    -2.53817983e-02, -1.87768910e-02, -1.18746138e-02, -4.66909925e-03,
     2.84409675e-03,  1.06697612e-02,  1.88135595e-02,  2.72815601e-02,
     3.60781047e-02,  4.52070276e-02,  5.46723880e-02,  6.44786605e-02,
     7.46286220e-02,  8.51249057e-02,  9.59698399e-02,  1.07165078e-01,
     1.18711585e-01,  1.30610107e-01,  1.42859645e-01,  1.55458473e-01,
     1.68404161e-01,  1.81694789e-01,  1.95327388e-01,  2.09296321e-01,
     2.23594564e-01,  2.38216022e-01,  2.53152972e-01,  2.68396157e-01,
     2.83936139e-01,  2.99762426e-01,  3.15861908e-01,  3.32221055e-01,
     3.48826468e-01,  3.65664038e-01,  3.82715297e-01,  3.99961186e-01,
     4.17384327e-01,  4.34966962e-01,  4.52687640e-01,  4.70524201e-01,
     4.88453925e-01,  5.06454555e-01,  5.24500675e-01,  5.42567437e-01,
     5.60631204e-01,  5.78667265e-01,  5.96647704e-01,  6.14545890e-01,
     6.32336194e-01,  6.49992632e-01,  6.67487403e-01,  6.84793267e-01,
     7.01883546e-01,  7.18732254e-01,  7.35312821e-01,  7.51600199e-01,
     7.67569925e-01,  7.83197457e-01,  7.98458386e-01,  8.13329535e-01,
     8.27789227e-01,  8.41817856e-01,  8.55396130e-01,  8.68506898e-01,
     8.81133444e-01,  8.93259678e-01,  9.04874884e-01,  9.15965761e-01,
     9.26521530e-01,  9.36533999e-01,  9.45997703e-01,  9.54908841e-01,
     9.63265812e-01,  9.71068890e-01,  9.78320416e-01,  9.85022676e-01,
     9.91179208e-01,  9.96798994e-01,  1.00189402e+00,  1.00647434e+00,
     1.01055206e+00,  1.01414254e+00,  1.01726259e+00,  1.01992884e+00,
     1.02215987e+00,  1.02397632e+00,  1.02540073e+00,  1.02645534e+00,
     1.02716451e+00,  1.02755273e+00,  1.02764446e+00,  1.02746325e+00,
     1.02703590e+00,  1.02638907e+00,  1.02554820e+00,  1.02453713e+00,
     1.02338080e+00,  1.02210370e+00,  1.02072836e+00,  1.01927533e+00,
     1.01776518e+00,  1.01621736e+00,  1.01466531e+00,  1.01324907e+00,
     1.01194801e+00,  1.01018909e+00,  1.00855796e+00,  1.00701129e+00,
     1.00554876e+00,  1.00416842e+00,  1.00286727e+00,  1.00164177e+00,
     1.00048907e+00,  9.99406080e-01,  9.98389887e-01,  9.97437085e-01,
     9.96544484e-01,  9.95709855e-01,  9.94930241e-01,  9.94202405e-01,
     9.93524160e-01,  9.92893043e-01,  9.92306810e-01,  9.91763378e-01,
     9.91259764e-01,  9.90795450e-01,  9.90367789e-01,  9.89975161e-01,
     9.89616034e-01,  9.89289016e-01,  9.88992851e-01,  9.88726033e-01,
     9.88486872e-01,  9.88275104e-01,  9.88089217e-01,  9.87927711e-01,
     9.87789826e-01,  9.87674344e-01,  9.87580750e-01,  9.87507202e-01,
     9.87452945e-01,  9.87416974e-01,  9.87398469e-01,  9.87395830e-01,
     9.87408003e-01,  9.87434340e-01,  9.87473624e-01,  9.87524314e-01,
     9.87585620e-01,  9.87656379e-01,  9.87735892e-01,  9.87822558e-01,
     9.87915097e-01,  9.88013273e-01,  9.88115695e-01,  9.88221131e-01,
     9.88328903e-01,  9.88437831e-01,  9.88547679e-01,  9.88656841e-01,
     9.88764587e-01,  9.88870854e-01,  9.88974432e-01,  9.89074727e-01,
     9.89171004e-01,  9.89263102e-01,  9.89350722e-01,  9.89433065e-01,
     9.89509692e-01,  9.89581081e-01,  9.89646747e-01,  9.89706737e-01,
     9.89760693e-01,  9.89809448e-01,  9.89853013e-01,  9.89891471e-01,
     9.89925419e-01,  9.89955420e-01,  9.89982449e-01,  9.90006512e-01,
     9.90028481e-01,  9.90049748e-01,  9.90070956e-01,  9.90092836e-01,
     9.90116392e-01,  9.90142748e-01,  9.90173428e-01,  9.90208733e-01,
     9.90249864e-01,  9.90298369e-01,  9.90354850e-01,  9.90420508e-01,
     9.90495930e-01,  9.90582515e-01,  9.90681257e-01,  9.90792209e-01,
     9.90916546e-01,  9.91055074e-01,  9.91208461e-01,  9.91376861e-01,
     9.91560583e-01,  9.91760421e-01,  9.91976718e-01,  9.92209110e-01,
     9.92457914e-01,  9.92723123e-01,  9.93004954e-01,  9.93302728e-01,
     9.93616108e-01,  9.93945371e-01,  9.94289515e-01,  9.94648168e-01,
     9.95020303e-01,  9.95405817e-01,  9.95803871e-01,  9.96213027e-01,
     9.96632469e-01,  9.97061531e-01,  9.97499058e-01,  9.97943743e-01,
     9.98394057e-01,  9.98849312e-01,  9.99308343e-01,  9.99768922e-01,
     1.00023113e+00,  1.00069214e+00,  1.00115201e+00,  1.00160853e+00,
     1.00206049e+00,  1.00250721e+00,  1.00294713e+00,  1.00337891e+00,
     1.00380137e+00,  1.00421381e+00,  1.00461539e+00,  1.00500462e+00,
     1.00538063e+00,  1.00574328e+00,  1.00609151e+00,  1.00642491e+00,
     1.00674243e+00,  1.00704432e+00,  1.00733022e+00,  1.00759940e+00,
     1.00785206e+00,  1.00808818e+00,  1.00830803e+00,  1.00851125e+00,
     1.00869814e+00,  1.00886952e+00,  1.00902566e+00,  1.00916672e+00,
     1.00929336e+00,  1.00940640e+00,  1.00950702e+00,  1.00959526e+00,
     1.00967215e+00,  1.00973908e+00,  1.00979668e+00,  1.00984614e+00,
     1.00988808e+00,  1.00992409e+00,  1.00995538e+00,  1.00998227e+00,
     1.01000630e+00,  1.01002862e+00,  1.01005025e+00,  1.01007195e+00,
     1.01009437e+00,  1.01011892e+00,  1.01014650e+00,  1.01017711e+00,
     1.01021176e+00,  1.01025100e+00,  1.01029547e+00,  1.01034523e+00,
     1.01040032e+00,  1.01046156e+00,  1.01052862e+00,  1.01060152e+00,
     1.01067979e+00,  1.01076391e+00,  1.01085343e+00,  1.01094755e+00,
     1.01104595e+00,  1.01114849e+00,  1.01125440e+00,  1.01136308e+00,
     1.01147330e+00,  1.01158500e+00,  1.01169742e+00,  1.01180892e+00,
     1.01191926e+00,  1.01202724e+00,  1.01213215e+00,  1.01223273e+00,
     1.01232756e+00,  1.01241638e+00,  1.01249789e+00,  1.01257043e+00,
     1.01263330e+00,  1.01268528e+00,  1.01272556e+00,  1.01275258e+00,
     1.01276506e+00,  1.01276236e+00,  1.01274338e+00,  1.01270648e+00,
     1.01265084e+00,  1.01257543e+00,  1.01247947e+00,  1.01236111e+00,
     1.01221981e+00,  1.01205436e+00,  1.01186400e+00,  1.01164722e+00,
     1.01140252e+00,  1.01112965e+00,  1.01082695e+00,  1.01049292e+00,
     1.01012635e+00,  1.00972589e+00,  1.00929006e+00,  1.00881730e+00,
     1.00830503e+00,  1.00775283e+00,  1.00715783e+00,  1.00651805e+00,
     1.00583140e+00,  1.00509559e+00,  1.00430863e+00,  1.00346750e+00,
     1.00256950e+00,  1.00161271e+00,  1.00059427e+00,  9.99511170e-01,
     9.98360922e-01,  9.97140929e-01,  9.95848886e-01,  9.94481854e-01,
     9.93037528e-01,  9.91514656e-01,  9.89913680e-01,  9.88193062e-01,
     9.85942259e-01,  9.83566790e-01,  9.81142303e-01,  9.78521444e-01,
     9.75663604e-01,  9.72545344e-01,  9.69145663e-01,  9.65440618e-01,
     9.61404362e-01,  9.57011307e-01,  9.52236767e-01,  9.47054884e-01,
     9.41440374e-01,  9.35369161e-01,  9.28819009e-01,  9.21766289e-01,
     9.14189628e-01,  9.06069468e-01,  8.97389168e-01,  8.88133200e-01,
     8.78289389e-01,  8.67846957e-01,  8.56797064e-01,  8.45133465e-01,
     8.32854281e-01,  8.19959478e-01,  8.06451101e-01,  7.92334648e-01,
     7.77620449e-01,  7.62320618e-01,  7.46448649e-01,  7.30020573e-01,
     7.13056738e-01,  6.95580544e-01,  6.77617323e-01,  6.59195531e-01,
     6.40348643e-01,  6.21107220e-01,  6.01504928e-01,  5.81578761e-01,
     5.61367451e-01,  5.40918863e-01,  5.20273683e-01,  4.99478073e-01,
     4.78577418e-01,  4.57617260e-01,  4.36649021e-01,  4.15722146e-01,
     3.94885659e-01,  3.74190319e-01,  3.53686890e-01,  3.33426002e-01,
     3.13458647e-01,  2.93833790e-01,  2.74599264e-01,  2.55803064e-01,
     2.37490219e-01,  2.19703603e-01,  2.02485542e-01,  1.85874992e-01,
     1.69906780e-01,  1.54613227e-01,  1.40023821e-01,  1.26163740e-01,
     1.13053443e-01,  1.00708497e-01,  8.91402439e-02,  7.83561210e-02,
     6.83582123e-02,  5.91421154e-02,  5.06989301e-02,  4.30171776e-02,
     3.60802073e-02,  2.98631634e-02,  2.43372266e-02,  1.94767524e-02,
     1.52571017e-02,  1.16378749e-02,  8.43308778e-03,  4.44966900e-03,
};

static const float mdct_win_10m_480[480+300] = {
    -2.35303215e-04, -4.61989875e-04, -6.26293154e-04, -7.92918043e-04,
    -9.74716672e-04, -1.18025689e-03, -1.40920904e-03, -1.66447310e-03,
    -1.94659161e-03, -2.25708173e-03, -2.59710692e-03, -2.96760762e-03,
    -3.37045488e-03, -3.80628516e-03, -4.27687377e-03, -4.78246990e-03,
    -5.32460872e-03, -5.90340381e-03, -6.52041973e-03, -7.17588528e-03,
    -7.87142282e-03, -8.60658604e-03, -9.38248086e-03, -1.01982718e-02,
    -1.10552055e-02, -1.19527030e-02, -1.28920591e-02, -1.38726348e-02,
    -1.48952816e-02, -1.59585662e-02, -1.70628856e-02, -1.82066640e-02,
    -1.93906598e-02, -2.06135542e-02, -2.18757093e-02, -2.31752632e-02,
    -2.45122745e-02, -2.58847194e-02, -2.72926374e-02, -2.87339090e-02,
    -3.02086274e-02, -3.17144037e-02, -3.32509886e-02, -3.48159779e-02,
    -3.64089241e-02, -3.80274232e-02, -3.96706799e-02, -4.13357542e-02,
    -4.30220337e-02, -4.47269805e-02, -4.64502229e-02, -4.81889149e-02,
    -4.99422586e-02, -5.17069080e-02, -5.34816204e-02, -5.52633479e-02,
    -5.70512315e-02, -5.88427175e-02, -6.06371724e-02, -6.24310403e-02,
    -6.42230355e-02, -6.60096152e-02, -6.77896227e-02, -6.95599687e-02,
    -7.13196627e-02, -7.30658127e-02, -7.47975891e-02, -7.65117823e-02,
    -7.82071142e-02, -7.98801069e-02, -8.15296401e-02, -8.31523735e-02,
    -8.47472895e-02, -8.63113754e-02, -8.78437445e-02, -8.93416436e-02,
    -9.08041129e-02, -9.22279576e-02, -9.36123287e-02, -9.49537776e-02,
    -9.62515531e-02, -9.75028462e-02, -9.87073651e-02, -9.98627129e-02,
    -1.00968022e-01, -1.02020268e-01, -1.03018380e-01, -1.03959636e-01,
    -1.04843883e-01, -1.05668684e-01, -1.06434282e-01, -1.07138231e-01,
    -1.07779996e-01, -1.08357063e-01, -1.08869014e-01, -1.09313559e-01,
    -1.09690356e-01, -1.09996966e-01, -1.10233226e-01, -1.10397281e-01,
    -1.10489847e-01, -1.10508642e-01, -1.10453743e-01, -1.10322584e-01,
    -1.10114583e-01, -1.09827693e-01, -1.09462175e-01, -1.09016396e-01,
    -1.08490885e-01, -1.07883429e-01, -1.07193718e-01, -1.06419636e-01,
    -1.05561251e-01, -1.04616281e-01, -1.03584904e-01, -1.02465016e-01,
    -1.01256900e-01, -9.99586457e-02, -9.85701457e-02, -9.70891114e-02,
    -9.55154582e-02, -9.38468492e-02, -9.20830006e-02, -9.02217102e-02,
    -8.82630999e-02, -8.62049382e-02, -8.40474215e-02, -8.17879272e-02,
    -7.94262503e-02, -7.69598078e-02, -7.43878560e-02, -7.17079700e-02,
    -6.89199478e-02, -6.60218980e-02, -6.30134942e-02, -5.98919191e-02,
    -5.66565564e-02, -5.33040616e-02, -4.98342724e-02, -4.62445689e-02,
    -4.25345569e-02, -3.87019577e-02, -3.47458578e-02, -3.06634152e-02,
    -2.64542508e-02, -2.21158161e-02, -1.76474054e-02, -1.30458136e-02,
    -8.31042570e-03, -3.43826866e-03,  1.57031548e-03,  6.71769764e-03,
     1.20047702e-02,  1.74339832e-02,  2.30064206e-02,  2.87248142e-02,
     3.45889635e-02,  4.06010646e-02,  4.67610292e-02,  5.30713391e-02,
     5.95323909e-02,  6.61464781e-02,  7.29129318e-02,  7.98335419e-02,
     8.69080741e-02,  9.41381377e-02,  1.01523314e-01,  1.09065152e-01,
     1.16762655e-01,  1.24617139e-01,  1.32627295e-01,  1.40793819e-01,
     1.49115252e-01,  1.57592141e-01,  1.66222480e-01,  1.75006740e-01,
     1.83943194e-01,  1.93031818e-01,  2.02269985e-01,  2.11656743e-01,
     2.21188852e-01,  2.30865538e-01,  2.40683799e-01,  2.50642064e-01,
     2.60736512e-01,  2.70965907e-01,  2.81325902e-01,  2.91814469e-01,
     3.02427028e-01,  3.13160350e-01,  3.24009570e-01,  3.34971959e-01,
     3.46042294e-01,  3.57217518e-01,  3.68491565e-01,  3.79859512e-01,
     3.91314689e-01,  4.02853287e-01,  4.14468833e-01,  4.26157164e-01,
     4.37911390e-01,  4.49725632e-01,  4.61592545e-01,  4.73506703e-01,
     4.85460018e-01,  4.97447159e-01,  5.09459723e-01,  5.21490984e-01,
     5.33532682e-01,  5.45578981e-01,  5.57621716e-01,  5.69654673e-01,
     5.81668558e-01,  5.93656062e-01,  6.05608382e-01,  6.17519206e-01,
     6.29379661e-01,  6.41183084e-01,  6.52920354e-01,  6.64584079e-01,
     6.76165350e-01,  6.87657395e-01,  6.99051154e-01,  7.10340055e-01,
     7.21514933e-01,  7.32569177e-01,  7.43494372e-01,  7.54284633e-01,
     7.64931365e-01,  7.75428189e-01,  7.85767017e-01,  7.95941465e-01,
     8.05943723e-01,  8.15768707e-01,  8.25408622e-01,  8.34858937e-01,
     8.44112583e-01,  8.53165119e-01,  8.62010834e-01,  8.70645634e-01,
     8.79063156e-01,  8.87259971e-01,  8.95231329e-01,  9.02975168e-01,
     9.10486312e-01,  9.17762555e-01,  9.24799743e-01,  9.31596250e-01,
     9.38149486e-01,  9.44458839e-01,  9.50522086e-01,  9.56340292e-01,
     9.61911452e-01,  9.67236671e-01,  9.72315664e-01,  9.77150119e-01,
     9.81739750e-01,  9.86086587e-01,  9.90190638e-01,  9.94055718e-01,
     9.97684240e-01,  1.00108096e+00,  1.00424751e+00,  1.00718858e+00,
     1.00990665e+00,  1.01240743e+00,  1.01469470e+00,  1.01677466e+00,
     1.01865099e+00,  1.02033046e+00,  1.02181733e+00,  1.02311884e+00,
     1.02424026e+00,  1.02518972e+00,  1.02597245e+00,  1.02659694e+00,
     1.02706918e+00,  1.02739752e+00,  1.02758790e+00,  1.02764895e+00,
     1.02758583e+00,  1.02740852e+00,  1.02712299e+00,  1.02673867e+00,
     1.02626166e+00,  1.02570100e+00,  1.02506178e+00,  1.02435398e+00,
     1.02358239e+00,  1.02275651e+00,  1.02188060e+00,  1.02096387e+00,
     1.02000914e+00,  1.01902729e+00,  1.01801944e+00,  1.01699650e+00,
     1.01595743e+00,  1.01492344e+00,  1.01391595e+00,  1.01304757e+00,
     1.01221613e+00,  1.01104487e+00,  1.00991459e+00,  1.00882489e+00,
     1.00777386e+00,  1.00676170e+00,  1.00578665e+00,  1.00484875e+00,
     1.00394608e+00,  1.00307885e+00,  1.00224501e+00,  1.00144473e+00,
     1.00067619e+00,  9.99939317e-01,  9.99232085e-01,  9.98554813e-01,
     9.97905542e-01,  9.97284268e-01,  9.96689095e-01,  9.96120338e-01,
     9.95576126e-01,  9.95056572e-01,  9.94559753e-01,  9.94086038e-01,
     9.93633779e-01,  9.93203161e-01,  9.92792187e-01,  9.92401518e-01,
     9.92029727e-01,  9.91676778e-01,  9.91340877e-01,  9.91023065e-01,
     9.90721643e-01,  9.90436680e-01,  9.90166895e-01,  9.89913101e-01,
     9.89673564e-01,  9.89448837e-01,  9.89237484e-01,  9.89040193e-01,
     9.88855636e-01,  9.88684347e-01,  9.88524761e-01,  9.88377852e-01,
     9.88242327e-01,  9.88118564e-01,  9.88005163e-01,  9.87903202e-01,
     9.87811174e-01,  9.87729546e-01,  9.87657198e-01,  9.87594984e-01,
     9.87541274e-01,  9.87496906e-01,  9.87460625e-01,  9.87432981e-01,
     9.87412641e-01,  9.87400475e-01,  9.87394992e-01,  9.87396916e-01,
     9.87404906e-01,  9.87419705e-01,  9.87439972e-01,  9.87466328e-01,
     9.87497321e-01,  9.87533893e-01,  9.87574654e-01,  9.87620124e-01,
     9.87668980e-01,  9.87722156e-01,  9.87778192e-01,  9.87837649e-01,
     9.87899199e-01,  9.87963798e-01,  9.88030030e-01,  9.88098468e-01,
     9.88167801e-01,  9.88239030e-01,  9.88310769e-01,  9.88383520e-01,
     9.88456016e-01,  9.88529420e-01,  9.88602222e-01,  9.88674940e-01,
     9.88746626e-01,  9.88818277e-01,  9.88888248e-01,  9.88957438e-01,
     9.89024798e-01,  9.89091125e-01,  9.89155170e-01,  9.89217866e-01,
     9.89277956e-01,  9.89336519e-01,  9.89392368e-01,  9.89446283e-01,
     9.89497212e-01,  9.89546334e-01,  9.89592362e-01,  9.89636265e-01,
     9.89677201e-01,  9.89716220e-01,  9.89752029e-01,  9.89785920e-01,
     9.89817027e-01,  9.89846207e-01,  9.89872536e-01,  9.89897514e-01,
     9.89920005e-01,  9.89941079e-01,  9.89960061e-01,  9.89978226e-01,
     9.89994556e-01,  9.90010350e-01,  9.90024832e-01,  9.90039402e-01,
     9.90053211e-01,  9.90067475e-01,  9.90081472e-01,  9.90096693e-01,
     9.90112245e-01,  9.90129379e-01,  9.90147465e-01,  9.90168060e-01,
     9.90190227e-01,  9.90215190e-01,  9.90242442e-01,  9.90273445e-01,
     9.90307127e-01,  9.90344891e-01,  9.90386228e-01,  9.90432448e-01,
     9.90482565e-01,  9.90537983e-01,  9.90598060e-01,  9.90664037e-01,
     9.90734883e-01,  9.90812038e-01,  9.90894786e-01,  9.90984259e-01,
     9.91079525e-01,  9.91181924e-01,  9.91290512e-01,  9.91406471e-01,
     9.91528801e-01,  9.91658694e-01,  9.91795272e-01,  9.91939622e-01,
     9.92090615e-01,  9.92249503e-01,  9.92415240e-01,  9.92588721e-01,
     9.92768871e-01,  9.92956911e-01,  9.93151653e-01,  9.93353924e-01,
     9.93562689e-01,  9.93779087e-01,  9.94001643e-01,  9.94231202e-01,
     9.94466818e-01,  9.94709344e-01,  9.94957285e-01,  9.95211663e-01,
     9.95471264e-01,  9.95736795e-01,  9.96006862e-01,  9.96282303e-01,
     9.96561799e-01,  9.96846133e-01,  9.97133827e-01,  9.97425669e-01,
     9.97720337e-01,  9.98018509e-01,  9.98318587e-01,  9.98621352e-01,
     9.98925543e-01,  9.99231731e-01,  9.99538258e-01,  9.99846116e-01,
     1.00015391e+00,  1.00046196e+00,  1.00076886e+00,  1.00107561e+00,
     1.00138055e+00,  1.00168424e+00,  1.00198543e+00,  1.00228487e+00,
     1.00258098e+00,  1.00287441e+00,  1.00316385e+00,  1.00345006e+00,
     1.00373157e+00,  1.00400915e+00,  1.00428146e+00,  1.00454934e+00,
     1.00481138e+00,  1.00506827e+00,  1.00531880e+00,  1.00556397e+00,
     1.00580227e+00,  1.00603455e+00,  1.00625986e+00,  1.00647902e+00,
     1.00669054e+00,  1.00689557e+00,  1.00709305e+00,  1.00728380e+00,
     1.00746662e+00,  1.00764273e+00,  1.00781104e+00,  1.00797244e+00,
     1.00812588e+00,  1.00827260e+00,  1.00841147e+00,  1.00854357e+00,
     1.00866802e+00,  1.00878601e+00,  1.00889653e+00,  1.00900077e+00,
     1.00909776e+00,  1.00918888e+00,  1.00927316e+00,  1.00935176e+00,
     1.00942394e+00,  1.00949118e+00,  1.00955240e+00,  1.00960889e+00,
     1.00965997e+00,  1.00970709e+00,  1.00974924e+00,  1.00978774e+00,
     1.00982209e+00,  1.00985371e+00,  1.00988150e+00,  1.00990696e+00,
     1.00992957e+00,  1.00995057e+00,  1.00996902e+00,  1.00998650e+00,
     1.01000236e+00,  1.01001789e+00,  1.01003217e+00,  1.01004672e+00,
     1.01006081e+00,  1.01007567e+00,  1.01009045e+00,  1.01010656e+00,
     1.01012323e+00,  1.01014176e+00,  1.01016113e+00,  1.01018264e+00,
     1.01020559e+00,  1.01023108e+00,  1.01025795e+00,  1.01028773e+00,
     1.01031948e+00,  1.01035408e+00,  1.01039064e+00,  1.01043047e+00,
     1.01047227e+00,  1.01051710e+00,  1.01056410e+00,  1.01061427e+00,
     1.01066629e+00,  1.01072136e+00,  1.01077842e+00,  1.01083825e+00,
     1.01089966e+00,  1.01096373e+00,  1.01102919e+00,  1.01109699e+00,
     1.01116586e+00,  1.01123661e+00,  1.01130817e+00,  1.01138145e+00,
     1.01145479e+00,  1.01152919e+00,  1.01160368e+00,  1.01167880e+00,
     1.01175301e+00,  1.01182748e+00,  1.01190094e+00,  1.01197388e+00,
     1.01204489e+00,  1.01211499e+00,  1.01218284e+00,  1.01224902e+00,
     1.01231210e+00,  1.01237303e+00,  1.01243046e+00,  1.01248497e+00,
     1.01253506e+00,  1.01258168e+00,  1.01262347e+00,  1.01266098e+00,
     1.01269276e+00,  1.01271979e+00,  1.01274058e+00,  1.01275575e+00,
     1.01276395e+00,  1.01276592e+00,  1.01276030e+00,  1.01274782e+00,
     1.01272696e+00,  1.01269861e+00,  1.01266140e+00,  1.01261590e+00,
     1.01256083e+00,  1.01249705e+00,  1.01242289e+00,  1.01233923e+00,
     1.01224492e+00,  1.01214046e+00,  1.01202430e+00,  1.01189756e+00,
     1.01175881e+00,  1.01160845e+00,  1.01144516e+00,  1.01126996e+00,
     1.01108126e+00,  1.01087961e+00,  1.01066368e+00,  1.01043418e+00,
     1.01018968e+00,  1.00993075e+00,  1.00965566e+00,  1.00936525e+00,
     1.00905825e+00,  1.00873476e+00,  1.00839308e+00,  1.00803431e+00,
     1.00765666e+00,  1.00726014e+00,  1.00684335e+00,  1.00640701e+00,
     1.00594915e+00,  1.00547001e+00,  1.00496799e+00,  1.00444353e+00,
     1.00389477e+00,  1.00332190e+00,  1.00272313e+00,  1.00209885e+00,
     1.00144728e+00,  1.00076851e+00,  1.00006069e+00,  9.99324268e-01,
     9.98557350e-01,  9.97760020e-01,  9.96930604e-01,  9.96069427e-01,
     9.95174643e-01,  9.94246644e-01,  9.93283713e-01,  9.92286108e-01,
     9.91252309e-01,  9.90182742e-01,  9.89075787e-01,  9.87931302e-01,
     9.86355322e-01,  9.84736245e-01,  9.83175095e-01,  9.81558334e-01,
     9.79861353e-01,  9.78061749e-01,  9.76157432e-01,  9.74137862e-01,
     9.71999011e-01,  9.69732741e-01,  9.67333198e-01,  9.64791512e-01,
     9.62101150e-01,  9.59253976e-01,  9.56242718e-01,  9.53060091e-01,
     9.49698408e-01,  9.46149812e-01,  9.42407161e-01,  9.38463416e-01,
     9.34311297e-01,  9.29944987e-01,  9.25356797e-01,  9.20540463e-01,
     9.15489628e-01,  9.10198679e-01,  9.04662060e-01,  8.98875519e-01,
     8.92833832e-01,  8.86533719e-01,  8.79971272e-01,  8.73143784e-01,
     8.66047653e-01,  8.58681252e-01,  8.51042044e-01,  8.43129723e-01,
     8.34943514e-01,  8.26483991e-01,  8.17750537e-01,  8.08744982e-01,
     7.99468149e-01,  7.89923516e-01,  7.80113773e-01,  7.70043128e-01,
     7.59714574e-01,  7.49133097e-01,  7.38302860e-01,  7.27229876e-01,
     7.15920192e-01,  7.04381434e-01,  6.92619693e-01,  6.80643883e-01,
     6.68461648e-01,  6.56083014e-01,  6.43517927e-01,  6.30775533e-01,
     6.17864165e-01,  6.04795463e-01,  5.91579959e-01,  5.78228937e-01,
     5.64753589e-01,  5.51170316e-01,  5.37490509e-01,  5.23726350e-01,
     5.09891542e-01,  4.96000807e-01,  4.82066294e-01,  4.68101711e-01,
     4.54121700e-01,  4.40142182e-01,  4.26177297e-01,  4.12241789e-01,
     3.98349961e-01,  3.84517234e-01,  3.70758372e-01,  3.57088679e-01,
     3.43522867e-01,  3.30076376e-01,  3.16764033e-01,  3.03600465e-01,
     2.90599616e-01,  2.77775850e-01,  2.65143468e-01,  2.52716188e-01,
     2.40506985e-01,  2.28528397e-01,  2.16793343e-01,  2.05313990e-01,
     1.94102191e-01,  1.83168087e-01,  1.72522195e-01,  1.62173542e-01,
     1.52132068e-01,  1.42405280e-01,  1.33001524e-01,  1.23926066e-01,
     1.15185830e-01,  1.06784043e-01,  9.87263751e-02,  9.10137900e-02,
     8.36505724e-02,  7.66350831e-02,  6.99703341e-02,  6.36518811e-02,
     5.76817602e-02,  5.20524422e-02,  4.67653841e-02,  4.18095054e-02,
     3.71864025e-02,  3.28807275e-02,  2.88954850e-02,  2.52098057e-02,
     2.18305756e-02,  1.87289619e-02,  1.59212782e-02,  1.33638143e-02,
     1.10855888e-02,  8.94347419e-03,  6.75812489e-03,  3.50443813e-03,
};

static const float mdct_win_7m5_60[60+46] = {
     2.95060859e-03,  7.17541132e-03,  1.37695374e-02,  2.30953556e-02,
     3.54036230e-02,  5.08289304e-02,  6.94696293e-02,  9.13884278e-02,
     1.16604575e-01,  1.45073546e-01,  1.76711174e-01,  2.11342953e-01,
     2.48768614e-01,  2.88701102e-01,  3.30823871e-01,  3.74814544e-01,
     4.20308013e-01,  4.66904918e-01,  5.14185341e-01,  5.61710041e-01,
     6.09026346e-01,  6.55671016e-01,  7.01218384e-01,  7.45240679e-01,
     7.87369206e-01,  8.27223833e-01,  8.64513675e-01,  8.98977415e-01,
     9.30407518e-01,  9.58599937e-01,  9.83447719e-01,  1.00488283e+00,
     1.02285381e+00,  1.03740495e+00,  1.04859791e+00,  1.05656184e+00,
     1.06149371e+00,  1.06362578e+00,  1.06325973e+00,  1.06074505e+00,
     1.05643590e+00,  1.05069500e+00,  1.04392435e+00,  1.03647725e+00,
     1.02872867e+00,  1.02106486e+00,  1.01400658e+00,  1.00727455e+00,
     1.00172250e+00,  9.97309592e-01,  9.93985158e-01,  9.91683335e-01,
     9.90325325e-01,  9.89822613e-01,  9.90074734e-01,  9.90975314e-01,
     9.92412851e-01,  9.94273149e-01,  9.96439157e-01,  9.98791616e-01,
     1.00120985e+00,  1.00357357e+00,  1.00575984e+00,  1.00764515e+00,
     1.00910687e+00,  1.01002476e+00,  1.01028203e+00,  1.00976919e+00,
     1.00838641e+00,  1.00605124e+00,  1.00269767e+00,  9.98280464e-01,
     9.92777987e-01,  9.86186892e-01,  9.77634164e-01,  9.67447270e-01,
     9.55129725e-01,  9.40389877e-01,  9.22959280e-01,  9.02607350e-01,
     8.79202689e-01,  8.52641750e-01,  8.22881272e-01,  7.89971715e-01,
     7.54030328e-01,  7.15255742e-01,  6.73936911e-01,  6.30414716e-01,
     5.85078858e-01,  5.38398518e-01,  4.90833753e-01,  4.42885823e-01,
     3.95091024e-01,  3.48004343e-01,  3.02196710e-01,  2.58227431e-01,
     2.16641416e-01,  1.77922122e-01,  1.42480547e-01,  1.10652194e-01,
     8.26995967e-02,  5.88334516e-02,  3.92030848e-02,  2.38629107e-02,
     1.26976223e-02,  5.35665361e-03,
};

static const float mdct_win_7m5_120[120+92] = {
     2.20824874e-03,  3.81014420e-03,  5.91552473e-03,  8.58361457e-03,
     1.18759723e-02,  1.58335301e-02,  2.04918652e-02,  2.58883593e-02,
     3.20415894e-02,  3.89616721e-02,  4.66742169e-02,  5.51849337e-02,
     6.45038384e-02,  7.46411071e-02,  8.56000162e-02,  9.73846703e-02,
     1.09993603e-01,  1.23419277e-01,  1.37655457e-01,  1.52690437e-01,
     1.68513363e-01,  1.85093105e-01,  2.02410419e-01,  2.20450365e-01,
     2.39167941e-01,  2.58526168e-01,  2.78498539e-01,  2.99038432e-01,
     3.20104862e-01,  3.41658622e-01,  3.63660034e-01,  3.86062695e-01,
     4.08815272e-01,  4.31871046e-01,  4.55176988e-01,  4.78676593e-01,
     5.02324813e-01,  5.26060916e-01,  5.49831283e-01,  5.73576883e-01,
     5.97241338e-01,  6.20770242e-01,  6.44099662e-01,  6.67176382e-01,
     6.89958854e-01,  7.12379980e-01,  7.34396372e-01,  7.55966688e-01,
     7.77036981e-01,  7.97558114e-01,  8.17490856e-01,  8.36796950e-01,
     8.55447310e-01,  8.73400798e-01,  8.90635719e-01,  9.07128770e-01,
     9.22848784e-01,  9.37763323e-01,  9.51860206e-01,  9.65130600e-01,
     9.77556541e-01,  9.89126209e-01,  9.99846919e-01,  1.00970073e+00,
     1.01868229e+00,  1.02681455e+00,  1.03408981e+00,  1.04051196e+00,
     1.04610837e+00,  1.05088565e+00,  1.05486289e+00,  1.05807221e+00,
     1.06053414e+00,  1.06227662e+00,  1.06333815e+00,  1.06375557e+00,
     1.06356632e+00,  1.06282156e+00,  1.06155996e+00,  1.05981709e+00,
     1.05765876e+00,  1.05512006e+00,  1.05223985e+00,  1.04908779e+00,
     1.04569860e+00,  1.04210831e+00,  1.03838099e+00,  1.03455276e+00,
     1.03067200e+00,  1.02679167e+00,  1.02295558e+00,  1.01920733e+00,
     1.01587289e+00,  1.01221017e+00,  1.00884559e+00,  1.00577851e+00,
     1.00300262e+00,  1.00051460e+00,  9.98309229e-01,  9.96378601e-01,
     9.94718132e-01,  9.93316216e-01,  9.92166957e-01,  9.91258603e-01,
     9.90581104e-01,  9.90123118e-01,  9.89873712e-01,  9.89818707e-01,
     9.89946800e-01,  9.90243175e-01,  9.90695564e-01,  9.91288540e-01,
     9.92009469e-01,  9.92842693e-01,  9.93775067e-01,  9.94790398e-01,
     9.95875534e-01,  9.97014367e-01,  9.98192871e-01,  9.99394506e-01,
     1.00060586e+00,  1.00181040e+00,  1.00299457e+00,  1.00414155e+00,
     1.00523688e+00,  1.00626393e+00,  1.00720890e+00,  1.00805489e+00,
     1.00878802e+00,  1.00939182e+00,  1.00985296e+00,  1.01015529e+00,
     1.01028602e+00,  1.01022988e+00,  1.00997541e+00,  1.00950846e+00,
     1.00881848e+00,  1.00789488e+00,  1.00672876e+00,  1.00530991e+00,
     1.00363456e+00,  1.00169363e+00,  9.99485663e-01,  9.97006370e-01,
     9.94254687e-01,  9.91231967e-01,  9.87937115e-01,  9.84375125e-01,
     9.79890963e-01,  9.75269879e-01,  9.70180498e-01,  9.64580027e-01,
     9.58425534e-01,  9.51684014e-01,  9.44320232e-01,  9.36290624e-01,
     9.27580507e-01,  9.18153414e-01,  9.07976524e-01,  8.97050058e-01,
     8.85351360e-01,  8.72857927e-01,  8.59579819e-01,  8.45502615e-01,
     8.30619943e-01,  8.14946648e-01,  7.98489378e-01,  7.81262450e-01,
     7.63291769e-01,  7.44590843e-01,  7.25199287e-01,  7.05153668e-01,
     6.84490545e-01,  6.63245210e-01,  6.41477162e-01,  6.19235334e-01,
     5.96559133e-01,  5.73519989e-01,  5.50173851e-01,  5.26568538e-01,
     5.02781159e-01,  4.78860889e-01,  4.54877894e-01,  4.30898123e-01,
     4.06993964e-01,  3.83234031e-01,  3.59680098e-01,  3.36408100e-01,
     3.13496418e-01,  2.91010565e-01,  2.69019585e-01,  2.47584348e-01,
     2.26788433e-01,  2.06677771e-01,  1.87310343e-01,  1.68739644e-01,
     1.51012382e-01,  1.34171842e-01,  1.18254662e-01,  1.03290734e-01,
     8.93117360e-02,  7.63429787e-02,  6.44077291e-02,  5.35243715e-02,
     4.37084453e-02,  3.49667099e-02,  2.72984629e-02,  2.06895808e-02,
     1.51125125e-02,  1.05228754e-02,  6.85547314e-03,  4.02351119e-03,
};

static const float mdct_win_7m5_180[180+138] = {
     1.97084908e-03,  2.95060859e-03,  4.12447721e-03,  5.52688664e-03,
     7.17541132e-03,  9.08757730e-03,  1.12819105e-02,  1.37695374e-02,
     1.65600266e-02,  1.96650895e-02,  2.30953556e-02,  2.68612894e-02,
     3.09632560e-02,  3.54036230e-02,  4.01915610e-02,  4.53331403e-02,
     5.08289304e-02,  5.66815448e-02,  6.28935304e-02,  6.94696293e-02,
     7.64106314e-02,  8.37160016e-02,  9.13884278e-02,  9.94294008e-02,
     1.07834725e-01,  1.16604575e-01,  1.25736503e-01,  1.35226811e-01,
     1.45073546e-01,  1.55273819e-01,  1.65822194e-01,  1.76711174e-01,
     1.87928776e-01,  1.99473180e-01,  2.11342953e-01,  2.23524554e-01,
     2.36003100e-01,  2.48768614e-01,  2.61813811e-01,  2.75129161e-01,
     2.88701102e-01,  3.02514034e-01,  3.16558805e-01,  3.30823871e-01,
     3.45295567e-01,  3.59963992e-01,  3.74814544e-01,  3.89831817e-01,
     4.05001010e-01,  4.20308013e-01,  4.35739515e-01,  4.51277817e-01,
     4.66904918e-01,  4.82609041e-01,  4.98375466e-01,  5.14185341e-01,
     5.30021478e-01,  5.45869352e-01,  5.61710041e-01,  5.77528151e-01,
     5.93304696e-01,  6.09026346e-01,  6.24674189e-01,  6.40227555e-01,
     6.55671016e-01,  6.70995935e-01,  6.86184559e-01,  7.01218384e-01,
     7.16078449e-01,  7.30756084e-01,  7.45240679e-01,  7.59515122e-01,
     7.73561955e-01,  7.87369206e-01,  8.00923138e-01,  8.14211386e-01,
     8.27223833e-01,  8.39952374e-01,  8.52386102e-01,  8.64513675e-01,
     8.76324079e-01,  8.87814288e-01,  8.98977415e-01,  9.09803319e-01,
     9.20284312e-01,  9.30407518e-01,  9.40169652e-01,  9.49567795e-01,
     9.58599937e-01,  9.67260260e-01,  9.75545166e-01,  9.83447719e-01,
     9.90971957e-01,  9.98119269e-01,  1.00488283e+00,  1.01125773e+00,
     1.01724436e+00,  1.02285381e+00,  1.02808734e+00,  1.03293706e+00,
     1.03740495e+00,  1.04150164e+00,  1.04523236e+00,  1.04859791e+00,
     1.05160340e+00,  1.05425505e+00,  1.05656184e+00,  1.05853400e+00,
     1.06017414e+00,  1.06149371e+00,  1.06249943e+00,  1.06320577e+00,
     1.06362578e+00,  1.06376487e+00,  1.06363778e+00,  1.06325973e+00,
     1.06264695e+00,  1.06180496e+00,  1.06074505e+00,  1.05948492e+00,
     1.05804533e+00,  1.05643590e+00,  1.05466218e+00,  1.05274047e+00,
     1.05069500e+00,  1.04853894e+00,  1.04627898e+00,  1.04392435e+00,
     1.04149540e+00,  1.03901003e+00,  1.03647725e+00,  1.03390793e+00,
     1.03131989e+00,  1.02872867e+00,  1.02614832e+00,  1.02358988e+00,
     1.02106486e+00,  1.01856262e+00,  1.01655770e+00,  1.01400658e+00,
     1.01162953e+00,  1.00938590e+00,  1.00727455e+00,  1.00529616e+00,
     1.00344526e+00,  1.00172250e+00,  1.00012792e+00,  9.98657533e-01,
     9.97309592e-01,  9.96083571e-01,  9.94976569e-01,  9.93985158e-01,
     9.93107530e-01,  9.92341305e-01,  9.91683335e-01,  9.91130070e-01,
     9.90678325e-01,  9.90325325e-01,  9.90067562e-01,  9.89901282e-01,
     9.89822613e-01,  9.89827845e-01,  9.89913241e-01,  9.90074734e-01,
     9.90308256e-01,  9.90609852e-01,  9.90975314e-01,  9.91400330e-01,
     9.91880966e-01,  9.92412851e-01,  9.92991779e-01,  9.93613381e-01,
     9.94273149e-01,  9.94966958e-01,  9.95690370e-01,  9.96439157e-01,
     9.97208572e-01,  9.97994275e-01,  9.98791616e-01,  9.99596062e-01,
     1.00040410e+00,  1.00120985e+00,  1.00200976e+00,  1.00279924e+00,
     1.00357357e+00,  1.00432828e+00,  1.00505850e+00,  1.00575984e+00,
     1.00642767e+00,  1.00705768e+00,  1.00764515e+00,  1.00818549e+00,
     1.00867427e+00,  1.00910687e+00,  1.00947916e+00,  1.00978659e+00,
     1.01002476e+00,  1.01018954e+00,  1.01027669e+00,  1.01028203e+00,
     1.01020174e+00,  1.01003208e+00,  1.00976919e+00,  1.00940939e+00,
     1.00894931e+00,  1.00838641e+00,  1.00771780e+00,  1.00694031e+00,
     1.00605124e+00,  1.00504879e+00,  1.00393183e+00,  1.00269767e+00,
     1.00134427e+00,  9.99872092e-01,  9.98280464e-01,  9.96566569e-01,
     9.94731737e-01,  9.92777987e-01,  9.90701374e-01,  9.88504165e-01,
     9.86186892e-01,  9.83711989e-01,  9.80584643e-01,  9.77634164e-01,
     9.74455033e-01,  9.71062916e-01,  9.67447270e-01,  9.63593926e-01,
     9.59491398e-01,  9.55129725e-01,  9.50501326e-01,  9.45592810e-01,
     9.40389877e-01,  9.34886760e-01,  9.29080559e-01,  9.22959280e-01,
     9.16509579e-01,  9.09724456e-01,  9.02607350e-01,  8.95155084e-01,
     8.87356154e-01,  8.79202689e-01,  8.70699698e-01,  8.61847424e-01,
     8.52641750e-01,  8.43077833e-01,  8.33154905e-01,  8.22881272e-01,
     8.12257597e-01,  8.01285439e-01,  7.89971715e-01,  7.78318177e-01,
     7.66337710e-01,  7.54030328e-01,  7.41407991e-01,  7.28477501e-01,
     7.15255742e-01,  7.01751739e-01,  6.87975632e-01,  6.73936911e-01,
     6.59652573e-01,  6.45139489e-01,  6.30414716e-01,  6.15483622e-01,
     6.00365852e-01,  5.85078858e-01,  5.69649536e-01,  5.54084810e-01,
     5.38398518e-01,  5.22614738e-01,  5.06756805e-01,  4.90833753e-01,
     4.74866033e-01,  4.58876566e-01,  4.42885823e-01,  4.26906539e-01,
     4.10970973e-01,  3.95091024e-01,  3.79291327e-01,  3.63587417e-01,
     3.48004343e-01,  3.32563201e-01,  3.17287485e-01,  3.02196710e-01,
     2.87309403e-01,  2.72643992e-01,  2.58227431e-01,  2.44072856e-01,
     2.30208977e-01,  2.16641416e-01,  2.03398481e-01,  1.90486162e-01,
     1.77922122e-01,  1.65726674e-01,  1.53906397e-01,  1.42480547e-01,
     1.31453980e-01,  1.20841778e-01,  1.10652194e-01,  1.00891734e-01,
     9.15718851e-02,  8.26995967e-02,  7.42815529e-02,  6.63242382e-02,
     5.88334516e-02,  5.18140676e-02,  4.52698346e-02,  3.92030848e-02,
     3.36144159e-02,  2.85023308e-02,  2.38629107e-02,  1.96894227e-02,
     1.59720527e-02,  1.26976223e-02,  9.84937739e-03,  7.40724463e-03,
     5.35665361e-03,  3.83226552e-03,
};

static const float mdct_win_7m5_240[240+184] = {
     1.84833037e-03,  2.56481839e-03,  3.36762118e-03,  4.28736617e-03,
     5.33830143e-03,  6.52679223e-03,  7.86112587e-03,  9.34628179e-03,
     1.09916868e-02,  1.28011172e-02,  1.47805911e-02,  1.69307043e-02,
     1.92592307e-02,  2.17696937e-02,  2.44685983e-02,  2.73556543e-02,
     3.04319230e-02,  3.36980464e-02,  3.71583577e-02,  4.08148180e-02,
     4.46708068e-02,  4.87262995e-02,  5.29820633e-02,  5.74382470e-02,
     6.20968580e-02,  6.69609767e-02,  7.20298364e-02,  7.73039146e-02,
     8.27825574e-02,  8.84682102e-02,  9.43607566e-02,  1.00460272e-01,
     1.06763824e-01,  1.13273679e-01,  1.19986420e-01,  1.26903521e-01,
     1.34020853e-01,  1.41339557e-01,  1.48857211e-01,  1.56573685e-01,
     1.64484622e-01,  1.72589077e-01,  1.80879090e-01,  1.89354320e-01,
     1.98012244e-01,  2.06854141e-01,  2.15875319e-01,  2.25068672e-01,
     2.34427407e-01,  2.43948314e-01,  2.53627993e-01,  2.63464061e-01,
     2.73450494e-01,  2.83582189e-01,  2.93853469e-01,  3.04257373e-01,
     3.14790914e-01,  3.25449123e-01,  3.36227410e-01,  3.47118760e-01,
     3.58120177e-01,  3.69224663e-01,  3.80427793e-01,  3.91720023e-01,
     4.03097022e-01,  4.14551955e-01,  4.26081719e-01,  4.37676318e-01,
     4.49330196e-01,  4.61034855e-01,  4.72786043e-01,  4.84576777e-01,
     4.96401707e-01,  5.08252458e-01,  5.20122078e-01,  5.32002077e-01,
     5.43888090e-01,  5.55771601e-01,  5.67645739e-01,  5.79502786e-01,
     5.91335035e-01,  6.03138367e-01,  6.14904172e-01,  6.26623941e-01,
     6.38288834e-01,  6.49893375e-01,  6.61432360e-01,  6.72902514e-01,
     6.84293750e-01,  6.95600460e-01,  7.06811784e-01,  7.17923425e-01,
     7.28931386e-01,  7.39832773e-01,  7.50618982e-01,  7.61284053e-01,
     7.71818919e-01,  7.82220992e-01,  7.92481330e-01,  8.02599448e-01,
     8.12565230e-01,  8.22377129e-01,  8.32030518e-01,  8.41523208e-01,
     8.50848313e-01,  8.60002412e-01,  8.68979881e-01,  8.77778347e-01,
     8.86395904e-01,  8.94829421e-01,  9.03077626e-01,  9.11132652e-01,
     9.18993585e-01,  9.26652937e-01,  9.34111420e-01,  9.41364344e-01,
     9.48412967e-01,  9.55255630e-01,  9.61892013e-01,  9.68316363e-01,
     9.74530156e-01,  9.80528338e-01,  9.86313928e-01,  9.91886049e-01,
     9.97246345e-01,  1.00239190e+00,  1.00731946e+00,  1.01202707e+00,
     1.01651654e+00,  1.02079430e+00,  1.02486082e+00,  1.02871471e+00,
     1.03235170e+00,  1.03577375e+00,  1.03898432e+00,  1.04198786e+00,
     1.04478564e+00,  1.04737818e+00,  1.04976743e+00,  1.05195405e+00,
     1.05394290e+00,  1.05573463e+00,  1.05734177e+00,  1.05875726e+00,
     1.05998674e+00,  1.06103672e+00,  1.06190651e+00,  1.06260369e+00,
     1.06313289e+00,  1.06350237e+00,  1.06370981e+00,  1.06376322e+00,
     1.06366765e+00,  1.06343012e+00,  1.06305656e+00,  1.06255421e+00,
     1.06192235e+00,  1.06116702e+00,  1.06029469e+00,  1.05931469e+00,
     1.05823465e+00,  1.05705891e+00,  1.05578948e+00,  1.05442979e+00,
     1.05298793e+00,  1.05147505e+00,  1.04989930e+00,  1.04826213e+00,
     1.04656691e+00,  1.04481699e+00,  1.04302125e+00,  1.04118768e+00,
     1.03932339e+00,  1.03743168e+00,  1.03551757e+00,  1.03358511e+00,
     1.03164371e+00,  1.02969955e+00,  1.02775944e+00,  1.02582719e+00,
     1.02390791e+00,  1.02200805e+00,  1.02013910e+00,  1.01826310e+00,
     1.01687901e+00,  1.01492195e+00,  1.01309662e+00,  1.01134205e+00,
     1.00965912e+00,  1.00805036e+00,  1.00651754e+00,  1.00505799e+00,
     1.00366956e+00,  1.00235327e+00,  1.00110981e+00,  9.99937523e-01,
     9.98834524e-01,  9.97800606e-01,  9.96835756e-01,  9.95938881e-01,
     9.95108459e-01,  9.94343411e-01,  9.93642921e-01,  9.93005832e-01,
     9.92430984e-01,  9.91917493e-01,  9.91463898e-01,  9.91068214e-01,
     9.90729218e-01,  9.90446225e-01,  9.90217819e-01,  9.90041963e-01,
     9.89917085e-01,  9.89841975e-01,  9.89815048e-01,  9.89834329e-01,
     9.89898211e-01,  9.90005403e-01,  9.90154189e-01,  9.90342427e-01,
     9.90568459e-01,  9.90830953e-01,  9.91128038e-01,  9.91457566e-01,
     9.91817881e-01,  9.92207559e-01,  9.92624757e-01,  9.93067358e-01,
     9.93533398e-01,  9.94021410e-01,  9.94529685e-01,  9.95055964e-01,
     9.95598351e-01,  9.96155580e-01,  9.96725627e-01,  9.97306092e-01,
     9.97895214e-01,  9.98491441e-01,  9.99092890e-01,  9.99697063e-01,
     1.00030303e+00,  1.00090793e+00,  1.00151084e+00,  1.00210923e+00,
     1.00270118e+00,  1.00328513e+00,  1.00385926e+00,  1.00442111e+00,
     1.00496860e+00,  1.00550040e+00,  1.00601455e+00,  1.00650869e+00,
     1.00698104e+00,  1.00743004e+00,  1.00785364e+00,  1.00824962e+00,
     1.00861604e+00,  1.00895138e+00,  1.00925390e+00,  1.00952134e+00,
     1.00975175e+00,  1.00994371e+00,  1.01009550e+00,  1.01020488e+00,
     1.01027007e+00,  1.01028975e+00,  1.01026227e+00,  1.01018562e+00,
     1.01005820e+00,  1.00987882e+00,  1.00964593e+00,  1.00935753e+00,
     1.00901228e+00,  1.00860959e+00,  1.00814837e+00,  1.00762674e+00,
     1.00704343e+00,  1.00639775e+00,  1.00568877e+00,  1.00491559e+00,
     1.00407768e+00,  1.00317429e+00,  1.00220424e+00,  1.00116684e+00,
     1.00006248e+00,  9.98891422e-01,  9.97652252e-01,  9.96343856e-01,
     9.94967462e-01,  9.93524663e-01,  9.92013927e-01,  9.90433283e-01,
     9.88785147e-01,  9.87072681e-01,  9.85297443e-01,  9.83401161e-01,
     9.80949418e-01,  9.78782729e-01,  9.76468238e-01,  9.74042850e-01,
     9.71498848e-01,  9.68829968e-01,  9.66030974e-01,  9.63095104e-01,
     9.60018198e-01,  9.56795738e-01,  9.53426267e-01,  9.49903482e-01,
     9.46222115e-01,  9.42375820e-01,  9.38361702e-01,  9.34177798e-01,
     9.29823124e-01,  9.25292320e-01,  9.20580120e-01,  9.15679793e-01,
     9.10590604e-01,  9.05315030e-01,  8.99852756e-01,  8.94199497e-01,
     8.88350152e-01,  8.82301631e-01,  8.76054874e-01,  8.69612385e-01,
     8.62972799e-01,  8.56135198e-01,  8.49098179e-01,  8.41857024e-01,
     8.34414055e-01,  8.26774617e-01,  8.18939244e-01,  8.10904891e-01,
     8.02675318e-01,  7.94253751e-01,  7.85641662e-01,  7.76838609e-01,
     7.67853193e-01,  7.58685181e-01,  7.49330658e-01,  7.39809171e-01,
     7.30109944e-01,  7.20247781e-01,  7.10224161e-01,  7.00044326e-01,
     6.89711890e-01,  6.79231154e-01,  6.68608179e-01,  6.57850997e-01,
     6.46965718e-01,  6.35959617e-01,  6.24840336e-01,  6.13603503e-01,
     6.02265091e-01,  5.90829083e-01,  5.79309408e-01,  5.67711124e-01,
     5.56037416e-01,  5.44293664e-01,  5.32489768e-01,  5.20636084e-01,
     5.08743273e-01,  4.96811166e-01,  4.84849881e-01,  4.72868107e-01,
     4.60875918e-01,  4.48881081e-01,  4.36891039e-01,  4.24912022e-01,
     4.12960603e-01,  4.01035896e-01,  3.89157867e-01,  3.77322199e-01,
     3.65543767e-01,  3.53832356e-01,  3.42196115e-01,  3.30644820e-01,
     3.19187559e-01,  3.07833309e-01,  2.96588182e-01,  2.85463717e-01,
     2.74462409e-01,  2.63609584e-01,  2.52883101e-01,  2.42323489e-01,
     2.31925746e-01,  2.21690837e-01,  2.11638058e-01,  2.01766920e-01,
     1.92082236e-01,  1.82589160e-01,  1.73305997e-01,  1.64229200e-01,
     1.55362654e-01,  1.46717079e-01,  1.38299391e-01,  1.30105078e-01,
     1.22145310e-01,  1.14423458e-01,  1.06941076e-01,  9.97025893e-02,
     9.27124283e-02,  8.59737427e-02,  7.94893311e-02,  7.32616579e-02,
     6.72934102e-02,  6.15874081e-02,  5.61458003e-02,  5.09700747e-02,
     4.60617047e-02,  4.14220117e-02,  3.70514189e-02,  3.29494666e-02,
     2.91153327e-02,  2.55476401e-02,  2.22437711e-02,  1.92000659e-02,
     1.64122205e-02,  1.38747611e-02,  1.15806353e-02,  9.52213664e-03,
     7.69137380e-03,  6.07207833e-03,  4.62581217e-03,  3.60685164e-03,
};

static const float mdct_win_7m5_360[360+276] = {
     1.72152668e-03,  2.20824874e-03,  2.68901752e-03,  3.22613342e-03,
     3.81014420e-03,  4.45371932e-03,  5.15369240e-03,  5.91552473e-03,
     6.73869158e-03,  7.62861841e-03,  8.58361457e-03,  9.60938437e-03,
     1.07060753e-02,  1.18759723e-02,  1.31190130e-02,  1.44390108e-02,
     1.58335301e-02,  1.73063081e-02,  1.88584711e-02,  2.04918652e-02,
     2.22061476e-02,  2.40057166e-02,  2.58883593e-02,  2.78552326e-02,
     2.99059145e-02,  3.20415894e-02,  3.42610013e-02,  3.65680973e-02,
     3.89616721e-02,  4.14435824e-02,  4.40140796e-02,  4.66742169e-02,
     4.94214625e-02,  5.22588489e-02,  5.51849337e-02,  5.82005143e-02,
     6.13059845e-02,  6.45038384e-02,  6.77913923e-02,  7.11707833e-02,
     7.46411071e-02,  7.82028053e-02,  8.18549521e-02,  8.56000162e-02,
     8.94357617e-02,  9.33642589e-02,  9.73846703e-02,  1.01496718e-01,
     1.05698760e-01,  1.09993603e-01,  1.14378287e-01,  1.18853508e-01,
     1.23419277e-01,  1.28075997e-01,  1.32820581e-01,  1.37655457e-01,
     1.42578648e-01,  1.47590522e-01,  1.52690437e-01,  1.57878853e-01,
     1.63152529e-01,  1.68513363e-01,  1.73957969e-01,  1.79484737e-01,
     1.85093105e-01,  1.90784835e-01,  1.96556497e-01,  2.02410419e-01,
     2.08345433e-01,  2.14359825e-01,  2.20450365e-01,  2.26617296e-01,
     2.32856279e-01,  2.39167941e-01,  2.45550642e-01,  2.52003951e-01,
     2.58526168e-01,  2.65118408e-01,  2.71775911e-01,  2.78498539e-01,
     2.85284606e-01,  2.92132459e-01,  2.99038432e-01,  3.06004256e-01,
     3.13026529e-01,  3.20104862e-01,  3.27237324e-01,  3.34423210e-01,
     3.41658622e-01,  3.48944976e-01,  3.56279252e-01,  3.63660034e-01,
     3.71085146e-01,  3.78554327e-01,  3.86062695e-01,  3.93610554e-01,
     4.01195225e-01,  4.08815272e-01,  4.16468460e-01,  4.24155411e-01,
     4.31871046e-01,  4.39614744e-01,  4.47384019e-01,  4.55176988e-01,
     4.62990138e-01,  4.70824619e-01,  4.78676593e-01,  4.86545433e-01,
     4.94428714e-01,  5.02324813e-01,  5.10229471e-01,  5.18142927e-01,
     5.26060916e-01,  5.33982818e-01,  5.41906817e-01,  5.49831283e-01,
     5.57751234e-01,  5.65667636e-01,  5.73576883e-01,  5.81476666e-01,
     5.89364661e-01,  5.97241338e-01,  6.05102013e-01,  6.12946170e-01,
     6.20770242e-01,  6.28572094e-01,  6.36348526e-01,  6.44099662e-01,
     6.51820973e-01,  6.59513822e-01,  6.67176382e-01,  6.74806795e-01,
     6.82400711e-01,  6.89958854e-01,  6.97475722e-01,  7.04950145e-01,
     7.12379980e-01,  7.19765434e-01,  7.27103833e-01,  7.34396372e-01,
     7.41638561e-01,  7.48829639e-01,  7.55966688e-01,  7.63049259e-01,
     7.70072273e-01,  7.77036981e-01,  7.83941108e-01,  7.90781257e-01,
     7.97558114e-01,  8.04271381e-01,  8.10914901e-01,  8.17490856e-01,
     8.23997094e-01,  8.30432785e-01,  8.36796950e-01,  8.43089298e-01,
     8.49305847e-01,  8.55447310e-01,  8.61511037e-01,  8.67496281e-01,
     8.73400798e-01,  8.79227518e-01,  8.84972438e-01,  8.90635719e-01,
     8.96217173e-01,  9.01716414e-01,  9.07128770e-01,  9.12456578e-01,
     9.17697261e-01,  9.22848784e-01,  9.27909917e-01,  9.32882596e-01,
     9.37763323e-01,  9.42553356e-01,  9.47252428e-01,  9.51860206e-01,
     9.56376060e-01,  9.60800602e-01,  9.65130600e-01,  9.69366689e-01,
     9.73508812e-01,  9.77556541e-01,  9.81507226e-01,  9.85364580e-01,
     9.89126209e-01,  9.92794201e-01,  9.96367545e-01,  9.99846919e-01,
     1.00322812e+00,  1.00651341e+00,  1.00970073e+00,  1.01279029e+00,
     1.01578293e+00,  1.01868229e+00,  1.02148657e+00,  1.02419772e+00,
     1.02681455e+00,  1.02933598e+00,  1.03176043e+00,  1.03408981e+00,
     1.03632326e+00,  1.03846361e+00,  1.04051196e+00,  1.04246831e+00,
     1.04433331e+00,  1.04610837e+00,  1.04779018e+00,  1.04938334e+00,
     1.05088565e+00,  1.05229923e+00,  1.05362522e+00,  1.05486289e+00,
     1.05601521e+00,  1.05708746e+00,  1.05807221e+00,  1.05897524e+00,
     1.05979447e+00,  1.06053414e+00,  1.06119412e+00,  1.06177366e+00,
     1.06227662e+00,  1.06270324e+00,  1.06305569e+00,  1.06333815e+00,
     1.06354800e+00,  1.06368607e+00,  1.06375557e+00,  1.06375743e+00,
     1.06369358e+00,  1.06356632e+00,  1.06337707e+00,  1.06312782e+00,
     1.06282156e+00,  1.06245782e+00,  1.06203634e+00,  1.06155996e+00,
     1.06102951e+00,  1.06044797e+00,  1.05981709e+00,  1.05914163e+00,
     1.05842136e+00,  1.05765876e+00,  1.05685377e+00,  1.05600761e+00,
     1.05512006e+00,  1.05419505e+00,  1.05323346e+00,  1.05223985e+00,
     1.05121668e+00,  1.05016637e+00,  1.04908779e+00,  1.04798366e+00,
     1.04685334e+00,  1.04569860e+00,  1.04452056e+00,  1.04332348e+00,
     1.04210831e+00,  1.04087907e+00,  1.03963603e+00,  1.03838099e+00,
     1.03711403e+00,  1.03583813e+00,  1.03455276e+00,  1.03326200e+00,
     1.03196750e+00,  1.03067200e+00,  1.02937564e+00,  1.02808244e+00,
     1.02679167e+00,  1.02550635e+00,  1.02422655e+00,  1.02295558e+00,
     1.02169299e+00,  1.02044475e+00,  1.01920733e+00,  1.01799992e+00,
     1.01716022e+00,  1.01587289e+00,  1.01461783e+00,  1.01339738e+00,
     1.01221017e+00,  1.01105652e+00,  1.00993444e+00,  1.00884559e+00,
     1.00778956e+00,  1.00676790e+00,  1.00577851e+00,  1.00482173e+00,
     1.00389592e+00,  1.00300262e+00,  1.00214091e+00,  1.00131213e+00,
     1.00051460e+00,  9.99748988e-01,  9.99013486e-01,  9.98309229e-01,
     9.97634934e-01,  9.96991885e-01,  9.96378601e-01,  9.95795982e-01,
     9.95242217e-01,  9.94718132e-01,  9.94222122e-01,  9.93755313e-01,
     9.93316216e-01,  9.92905809e-01,  9.92522422e-01,  9.92166957e-01,
     9.91837704e-01,  9.91535508e-01,  9.91258603e-01,  9.91007878e-01,
     9.90781723e-01,  9.90581104e-01,  9.90404336e-01,  9.90252267e-01,
     9.90123118e-01,  9.90017726e-01,  9.89934325e-01,  9.89873712e-01,
     9.89834110e-01,  9.89816359e-01,  9.89818707e-01,  9.89841998e-01,
     9.89884438e-01,  9.89946800e-01,  9.90027287e-01,  9.90126680e-01,
     9.90243175e-01,  9.90377594e-01,  9.90528134e-01,  9.90695564e-01,
     9.90878043e-01,  9.91076302e-01,  9.91288540e-01,  9.91515602e-01,
     9.91755666e-01,  9.92009469e-01,  9.92275155e-01,  9.92553486e-01,
     9.92842693e-01,  9.93143533e-01,  9.93454080e-01,  9.93775067e-01,
     9.94104689e-01,  9.94443742e-01,  9.94790398e-01,  9.95145361e-01,
     9.95506800e-01,  9.95875534e-01,  9.96249681e-01,  9.96629919e-01,
     9.97014367e-01,  9.97403799e-01,  9.97796404e-01,  9.98192871e-01,
     9.98591286e-01,  9.98992436e-01,  9.99394506e-01,  9.99798247e-01,
     1.00020179e+00,  1.00060586e+00,  1.00100858e+00,  1.00141070e+00,
     1.00181040e+00,  1.00220846e+00,  1.00260296e+00,  1.00299457e+00,
     1.00338148e+00,  1.00376444e+00,  1.00414155e+00,  1.00451348e+00,
     1.00487832e+00,  1.00523688e+00,  1.00558730e+00,  1.00593027e+00,
     1.00626393e+00,  1.00658905e+00,  1.00690380e+00,  1.00720890e+00,
     1.00750238e+00,  1.00778498e+00,  1.00805489e+00,  1.00831287e+00,
     1.00855700e+00,  1.00878802e+00,  1.00900405e+00,  1.00920593e+00,
     1.00939182e+00,  1.00956244e+00,  1.00971590e+00,  1.00985296e+00,
     1.00997177e+00,  1.01007317e+00,  1.01015529e+00,  1.01021893e+00,
     1.01026225e+00,  1.01028602e+00,  1.01028842e+00,  1.01027030e+00,
     1.01022988e+00,  1.01016802e+00,  1.01008292e+00,  1.00997541e+00,
     1.00984369e+00,  1.00968863e+00,  1.00950846e+00,  1.00930404e+00,
     1.00907371e+00,  1.00881848e+00,  1.00853675e+00,  1.00822947e+00,
     1.00789488e+00,  1.00753391e+00,  1.00714488e+00,  1.00672876e+00,
     1.00628393e+00,  1.00581146e+00,  1.00530991e+00,  1.00478053e+00,
     1.00422177e+00,  1.00363456e+00,  1.00301719e+00,  1.00237067e+00,
     1.00169363e+00,  1.00098749e+00,  1.00025108e+00,  9.99485663e-01,
     9.98689592e-01,  9.97863666e-01,  9.97006370e-01,  9.96119199e-01,
     9.95201404e-01,  9.94254687e-01,  9.93277595e-01,  9.92270651e-01,
     9.91231967e-01,  9.90163286e-01,  9.89064394e-01,  9.87937115e-01,
     9.86779736e-01,  9.85592773e-01,  9.84375125e-01,  9.83129288e-01,
     9.81348463e-01,  9.79890963e-01,  9.78400459e-01,  9.76860435e-01,
     9.75269879e-01,  9.73627353e-01,  9.71931341e-01,  9.70180498e-01,
     9.68372652e-01,  9.66506952e-01,  9.64580027e-01,  9.62592318e-01,
     9.60540986e-01,  9.58425534e-01,  9.56244393e-01,  9.53998416e-01,
     9.51684014e-01,  9.49301185e-01,  9.46846884e-01,  9.44320232e-01,
     9.41718404e-01,  9.39042580e-01,  9.36290624e-01,  9.33464050e-01,
     9.30560854e-01,  9.27580507e-01,  9.24519592e-01,  9.21378471e-01,
     9.18153414e-01,  9.14844696e-01,  9.11451652e-01,  9.07976524e-01,
     9.04417545e-01,  9.00776308e-01,  8.97050058e-01,  8.93238398e-01,
     8.89338681e-01,  8.85351360e-01,  8.81274023e-01,  8.77109638e-01,
     8.72857927e-01,  8.68519505e-01,  8.64092796e-01,  8.59579819e-01,
     8.54976007e-01,  8.50285220e-01,  8.45502615e-01,  8.40630470e-01,
     8.35667925e-01,  8.30619943e-01,  8.25482007e-01,  8.20258909e-01,
     8.14946648e-01,  8.09546696e-01,  8.04059978e-01,  7.98489378e-01,
     7.92831417e-01,  7.87090668e-01,  7.81262450e-01,  7.75353947e-01,
     7.69363613e-01,  7.63291769e-01,  7.57139016e-01,  7.50901711e-01,
     7.44590843e-01,  7.38205136e-01,  7.31738075e-01,  7.25199287e-01,
     7.18588225e-01,  7.11905687e-01,  7.05153668e-01,  6.98332634e-01,
     6.91444101e-01,  6.84490545e-01,  6.77470119e-01,  6.70388375e-01,
     6.63245210e-01,  6.56045780e-01,  6.48788627e-01,  6.41477162e-01,
     6.34114323e-01,  6.26702000e-01,  6.19235334e-01,  6.11720596e-01,
     6.04161612e-01,  5.96559133e-01,  5.88914401e-01,  5.81234783e-01,
     5.73519989e-01,  5.65770616e-01,  5.57988067e-01,  5.50173851e-01,
     5.42330194e-01,  5.34460798e-01,  5.26568538e-01,  5.18656324e-01,
     5.10728813e-01,  5.02781159e-01,  4.94819491e-01,  4.86845139e-01,
     4.78860889e-01,  4.70869928e-01,  4.62875144e-01,  4.54877894e-01,
     4.46882512e-01,  4.38889325e-01,  4.30898123e-01,  4.22918322e-01,
     4.14950878e-01,  4.06993964e-01,  3.99052648e-01,  3.91134614e-01,
     3.83234031e-01,  3.75354653e-01,  3.67502060e-01,  3.59680098e-01,
     3.51887312e-01,  3.44130166e-01,  3.36408100e-01,  3.28728966e-01,
     3.21090505e-01,  3.13496418e-01,  3.05951565e-01,  2.98454319e-01,
     2.91010565e-01,  2.83621109e-01,  2.76285415e-01,  2.69019585e-01,
     2.61812445e-01,  2.54659232e-01,  2.47584348e-01,  2.40578694e-01,
     2.33647009e-01,  2.26788433e-01,  2.20001992e-01,  2.13301325e-01,
     2.06677771e-01,  2.00140409e-01,  1.93683630e-01,  1.87310343e-01,
     1.81027384e-01,  1.74839476e-01,  1.68739644e-01,  1.62737273e-01,
     1.56825277e-01,  1.51012382e-01,  1.45298230e-01,  1.39687469e-01,
     1.34171842e-01,  1.28762544e-01,  1.23455562e-01,  1.18254662e-01,
     1.13159677e-01,  1.08171439e-01,  1.03290734e-01,  9.85202978e-02,
     9.38600023e-02,  8.93117360e-02,  8.48752103e-02,  8.05523737e-02,
     7.63429787e-02,  7.22489246e-02,  6.82699120e-02,  6.44077291e-02,
     6.06620003e-02,  5.70343711e-02,  5.35243715e-02,  5.01334690e-02,
     4.68610790e-02,  4.37084453e-02,  4.06748365e-02,  3.77612269e-02,
     3.49667099e-02,  3.22919275e-02,  2.97357669e-02,  2.72984629e-02,
     2.49787186e-02,  2.27762542e-02,  2.06895808e-02,  1.87178169e-02,
     1.68593418e-02,  1.51125125e-02,  1.34757094e-02,  1.19462709e-02,
     1.05228754e-02,  9.20130941e-03,  7.98124316e-03,  6.85547314e-03,
     5.82657334e-03,  4.87838525e-03,  4.02351119e-03,  3.15418663e-03,
};

const float *lc3_mdct_win[LC3_NUM_DT][LC3_NUM_SRATE] = {

    [LC3_DT_7M5] = {
        [LC3_SRATE_8K ] = mdct_win_7m5_60,
        [LC3_SRATE_16K] = mdct_win_7m5_120,
        [LC3_SRATE_24K] = mdct_win_7m5_180,
        [LC3_SRATE_32K] = mdct_win_7m5_240,
        [LC3_SRATE_48K] = mdct_win_7m5_360,
    },

    [LC3_DT_10M] = {
        [LC3_SRATE_8K ] = mdct_win_10m_80,
        [LC3_SRATE_16K] = mdct_win_10m_160,
        [LC3_SRATE_24K] = mdct_win_10m_240,
        [LC3_SRATE_32K] = mdct_win_10m_320,
        [LC3_SRATE_48K] = mdct_win_10m_480,
    },
};


/**
 * Bands limits (cf. 3.7.1-2)
 */

const int lc3_band_lim[LC3_NUM_DT][LC3_NUM_SRATE][LC3_NUM_BANDS+1] = {

    [LC3_DT_7M5] = {

        [LC3_SRATE_8K ] = {
              0,   1,   2,   3,   4,   5,   6,   7,   8,   9,
             10,  11,  12,  13,  14,  15,  16,  17,  18,  19,
             20,  21,  22,  23,  24,  25,  26,  27,  28,  29,
             30,  31,  32,  33,  34,  35,  36,  37,  38,  39,
             40,  41,  42,  43,  44,  45,  46,  47,  48,  49,
             50,  51,  52,  53,  54,  55,  56,  57,  58,  59,
             60,  60,  60,  60,  60                          },

        [LC3_SRATE_16K] = {
              0,   1,   2,   3,   4,   5,   6,   7,   8,   9,
             10,  11,  12,  13,  14,  15,  16,  17,  18,  19,
             20,  21,  22,  23,  24,  25,  26,  27,  28,  29,
             30,  31,  32,  33,  34,  36,  38,  40,  42,  44,
             46,  48,  50,  52,  54,  56,  58,  60,  62,  65,
             68,  71,  74,  77,  80,  83,  86,  90,  94,  98,
            102, 106, 110, 115, 120                          },

        [LC3_SRATE_24K] = {
              0,   1,   2,   3,   4,   5,   6,   7,   8,   9,
             10,  11,  12,  13,  14,  15,  16,  17,  18,  19,
             20,  21,  22,  23,  24,  25,  26,  27,  29,  31,
             33,  35,  37,  39,  41,  43,  45,  47,  49,  52,
             55,  58,  61,  64,  67,  70,  74,  78,  82,  86,
             90,  95, 100, 105, 110, 115, 121, 127, 134, 141,
            148, 155, 163, 171, 180                          },

        [LC3_SRATE_32K] = {
              0,   1,   2,   3,   4,   5,   6,   7,   8,   9,
             10,  11,  12,  13,  14,  15,  16,  17,  18,  19,
             20,  21,  22,  23,  24,  26,  28,  30,  32,  34,
             36,  38,  40,  42,  45,  48,  51,  54,  57,  60,
             63,  67,  71,  75,  79,  84,  89,  94,  99, 105,
            111, 117, 124, 131, 138, 146, 154, 163, 172, 182,
            192, 203, 215, 227, 240                          },

        [LC3_SRATE_48K] = {
              0,   1,   2,   3,   4,   5,   6,   7,   8,   9,
             10,  11,  12,  13,  14,  15,  16,  17,  18,  19,
             20,  21,  22,  24,  26,  28,  30,  32,  34,  36,
             38,  40,  43,  46,  49,  52,  55,  59,  63,  67,
             71,  75,  80,  85,  90,  96, 102, 108, 115, 122,
            129, 137, 146, 155, 165, 175, 186, 197, 209, 222,
            236, 251, 266, 283, 300                          },
    },

    [LC3_DT_10M] = {

        [LC3_SRATE_8K ] = {
              0,   1,   2,   3,   4,   5,   6,   7,   8,   9,
             10,  11,  12,  13,  14,  15,  16,  17,  18,  19,
             20,  21,  22,  23,  24,  25,  26,  27,  28,  29,
             30,  31,  32,  33,  34,  35,  36,  37,  38,  39,
             40,  41,  42,  43,  44,  45,  46,  47,  48,  49,
             51,  53,  55,  57,  59,  61,  63,  65,  67,  69,
             71,  73,  75,  77,  80                          },

        [LC3_SRATE_16K] = {
              0,   1,   2,   3,   4,   5,   6,   7,   8,   9,
             10,  11,  12,  13,  14,  15,  16,  17,  18,  19,
             20,  21,  22,  23,  24,  25,  26,  27,  28,  30,
             32,  34,  36,  38,  40,  42,  44,  46,  48,  50,
             52,  55,  58,  61,  64,  67,  70,  73,  76,  80,
             84,  88,  92,  96, 101, 106, 111, 116, 121, 127,
            133, 139, 146, 153, 160                          },

        [LC3_SRATE_24K] = {
              0,   1,   2,   3,   4,   5,   6,   7,   8,   9,
             10,  11,  12,  13,  14,  15,  16,  17,  18,  19,
             20,  21,  22,  23,  25,  27,  29,  31,  33,  35,
             37,  39,  41,  43,  46,  49,  52,  55,  58,  61,
             64,  68,  72,  76,  80,  85,  90,  95, 100, 106,
            112, 118, 125, 132, 139, 147, 155, 164, 173, 183,
            193, 204, 215, 227, 240                          },

        [LC3_SRATE_32K] = {
              0,   1,   2,   3,   4,   5,   6,   7,   8,   9,
             10,  11,  12,  13,  14,  15,  16,  17,  18,  19,
             20,  22,  24,  26,  28,  30,  32,  34,  36,  38,
             41,  44,  47,  50,  53,  56,  60,  64,  68,  72,
             76,  81,  86,  91,  97, 103, 109, 116, 123, 131,
            139, 148, 157, 166, 176, 187, 199, 211, 224, 238,
            252, 268, 284, 302, 320                          },

        [LC3_SRATE_48K] = {
              0,   1,   2,   3,   4,   5,   6,   7,   8,   9,
             10,  11,  12,  13,  14,  15,  16,  17,  18,  20,
             22,  24,  26,  28,  30,  32,  34,  36,  39,  42,
             45,  48,  51,  55,  59,  63,  67,  71,  76,  81,
             86,  92,  98, 105, 112, 119, 127, 135, 144, 154,
            164, 175, 186, 198, 211, 225, 240, 256, 273, 291,
            310, 330, 352, 375, 400                          },
    }
};


/**
 * SNS Quantization (cf. 3.7.4)
 */

const float lc3_sns_lfcb[32][8] = {

    {  2.26283366e+00,  8.13311269e-01, -5.30193495e-01, -1.35664836e+00,
      -1.59952177e+00, -1.44098768e+00, -1.14381648e+00, -7.55203768e-01 },

    {  2.94516479e+00,  2.41143318e+00,  9.60455106e-01, -4.43226488e-01,
      -1.22913612e+00, -1.55590039e+00, -1.49688656e+00, -1.11689987e+00 },

    { -2.18610707e+00, -1.97152136e+00, -1.78718620e+00, -1.91865896e+00,
      -1.79399122e+00, -1.35738404e+00, -7.05444279e-01, -4.78172945e-02 },

    {  6.93688237e-01,  9.55609857e-01,  5.75230787e-01, -1.14603419e-01,
      -6.46050637e-01, -9.52351370e-01, -1.07405247e+00, -7.58087707e-01 },

    { -1.29752132e+00, -7.40369057e-01, -3.45372484e-01, -3.13285696e-01,
      -4.02977243e-01, -3.72020853e-01, -7.83414177e-02,  9.70441304e-02 },

    {  9.14652038e-01,  1.74293043e+00,  1.90906627e+00,  1.54408484e+00,
       1.09344961e+00,  6.47479550e-01,  3.61790752e-02, -2.97092807e-01 },

    { -2.51428813e+00, -2.89175271e+00, -2.00450667e+00, -7.50912274e-01,
       4.41202105e-01,  1.20190988e+00,  1.32742857e+00,  1.22049081e+00 },

    { -9.22188405e-01,  6.32495141e-01,  1.08736431e+00,  6.08628625e-01,
       1.31174568e-01, -2.96149158e-01, -2.07013517e-01,  1.34924917e-01 },

    {  7.90322288e-01,  6.28401262e-01,  3.93117924e-01,  4.80007711e-01,
       4.47815138e-01,  2.09734215e-01,  6.56691996e-03, -8.61242342e-02 },

    {  1.44775580e+00,  2.72399952e+00,  2.31083269e+00,  9.35051270e-01,
      -2.74743911e-01, -9.02077697e-01, -9.40681512e-01, -6.33697039e-01 },

    {  7.93354526e-01,  1.43931186e-02, -5.67834845e-01, -6.54760468e-01,
      -4.79458998e-01, -1.73894662e-01,  6.80162706e-02,  2.95125948e-01 },

    {  2.72425347e+00,  2.95947572e+00,  1.84953559e+00,  5.63284922e-01,
       1.39917088e-01,  3.59641093e-01,  6.89461355e-01,  6.39790177e-01 },

    { -5.30830198e-01, -2.12690683e-01,  5.76613628e-03,  4.24871484e-01,
       4.73128952e-01,  8.58894199e-01,  1.19111161e+00,  9.96189670e-01 },

    {  1.68728411e+00,  2.43614509e+00,  2.33019429e+00,  1.77983778e+00,
       1.44411295e+00,  1.51995177e+00,  1.47199394e+00,  9.77682474e-01 },

    { -2.95183273e+00, -1.59393497e+00, -1.09918773e-01,  3.88609073e-01,
       5.12932650e-01,  6.28112597e-01,  8.22621796e-01,  8.75891425e-01 },

    {  1.01878343e-01,  5.89857324e-01,  6.19047647e-01,  1.26731314e+00,
       2.41961048e+00,  2.25174253e+00,  5.26537031e-01, -3.96591513e-01 },

    {  2.68254575e+00,  1.32738011e+00,  1.30185274e-01, -3.38533089e-01,
      -3.68219236e-01, -1.91689947e-01, -1.54782377e-01, -2.34207178e-01 },

    {  4.82697924e+00,  3.11947804e+00,  1.39513671e+00,  2.50295316e-01,
      -3.93613839e-01, -6.43458173e-01, -6.42570737e-01, -7.23193223e-01 },

    {  8.78419936e-02, -5.69586840e-01, -1.14506016e+00, -1.66968488e+00,
      -1.84534418e+00, -1.56468027e+00, -1.11746759e+00, -5.33981663e-01 },

    {  1.39102308e+00,  1.98146479e+00,  1.11265796e+00, -2.20107509e-01,
      -7.74965612e-01, -5.94063874e-01,  1.36937681e-01,  8.18242891e-01 },

    {  3.84585894e-01, -1.60588786e-01, -5.39366810e-01, -5.29309079e-01,
       1.90433547e-01,  2.56062918e+00,  2.81896398e+00,  6.56670876e-01 },

    {  1.93227399e+00,  3.01030180e+00,  3.06543894e+00,  2.50110161e+00,
       1.93089593e+00,  5.72153811e-01, -8.11741794e-01, -1.17641811e+00 },

    {  1.75080463e-01, -7.50522832e-01, -1.03943893e+00, -1.13577509e+00,
      -1.04197904e+00, -1.52060099e-02,  2.07048392e+00,  3.42948918e+00 },

    { -1.18817020e+00,  3.66792874e-01,  1.30957830e+00,  1.68330687e+00,
       1.25100924e+00,  9.42375752e-01,  8.26250483e-01,  4.39952741e-01 },

    {  2.53322203e+00,  2.11274643e+00,  1.26288412e+00,  7.61513512e-01,
       5.22117938e-01,  1.18680070e-01, -4.52346828e-01, -7.00352426e-01 },

    {  3.99889837e+00,  4.07901751e+00,  2.82285661e+00,  1.72607213e+00,
       6.47144377e-01, -3.31148521e-01, -8.84042571e-01, -1.12697341e+00 },

    {  5.07902593e-01,  1.58838450e+00,  1.72899024e+00,  1.00692230e+00,
       3.77121232e-01,  4.76370767e-01,  1.08754740e+00,  1.08756266e+00 },

    {  3.16856825e+00,  3.25853458e+00,  2.42230591e+00,  1.79446078e+00,
       1.52177911e+00,  1.17196707e+00,  4.89394597e-01, -6.22795716e-02 },

    {  1.89414767e+00,  1.25108695e+00,  5.90451211e-01,  6.08358583e-01,
       8.78171010e-01,  1.11912511e+00,  1.01857662e+00,  6.20453891e-01 },

    {  9.48880605e-01,  2.13239439e+00,  2.72345350e+00,  2.76986077e+00,
       2.54286973e+00,  2.02046264e+00,  8.30045859e-01, -2.75569174e-02 },

    { -1.88026757e+00, -1.26431073e+00,  3.11424977e-01,  1.83670210e+00,
       2.25634192e+00,  2.04818998e+00,  2.19526837e+00,  2.02659614e+00 },

    {  2.46375746e-01,  9.55621773e-01,  1.52046777e+00,  1.97647400e+00,
       1.94043867e+00,  2.23375847e+00,  1.98835978e+00,  1.27232673e+00 },

};

const float lc3_sns_hfcb[32][8] = {

    {  2.32028419e-01, -1.00890271e+00, -2.14223503e+00, -2.37533814e+00,
      -2.23041933e+00, -2.17595881e+00, -2.29065914e+00, -2.53286398e+00 },

    { -1.29503937e+00, -1.79929965e+00, -1.88703148e+00, -1.80991660e+00,
      -1.76340038e+00, -1.83418428e+00, -1.80480981e+00, -1.73679545e+00 },

    {  1.39285716e-01, -2.58185126e-01, -6.50804573e-01, -1.06815732e+00,
      -1.61928742e+00, -2.18762566e+00, -2.63757587e+00, -2.97897750e+00 },

    { -3.16513102e-01, -4.77747657e-01, -5.51162076e-01, -4.84788283e-01,
      -2.38388394e-01, -1.43024507e-01,  6.83186674e-02,  8.83061717e-02 },

    {  8.79518405e-01,  2.98340096e-01, -9.15386396e-01, -2.20645975e+00,
      -2.74142181e+00, -2.86139074e+00, -2.88841597e+00, -2.95182608e+00 },

    { -2.96701922e-01, -9.75004919e-01, -1.35857500e+00, -9.83721106e-01,
      -6.52956939e-01, -9.89986993e-01, -1.61467225e+00, -2.40712302e+00 },

    {  3.40981100e-01,  2.68899789e-01,  5.63335685e-02,  4.99114047e-02,
      -9.54130727e-02, -7.60166146e-01, -2.32758120e+00, -3.77155485e+00 },

    { -1.41229759e+00, -1.48522119e+00, -1.18603580e+00, -6.25001634e-01,
       1.53902497e-01,  5.76386498e-01,  7.95092604e-01,  5.96564632e-01 },

    { -2.28839512e-01, -3.33719070e-01, -8.09321359e-01, -1.63587877e+00,
      -1.88486397e+00, -1.64496691e+00, -1.40515778e+00, -1.46666471e+00 },

    { -1.07148629e+00, -1.41767015e+00, -1.54891762e+00, -1.45296062e+00,
      -1.03182970e+00, -6.90642640e-01, -4.28843805e-01, -4.94960215e-01 },

    { -5.90988511e-01, -7.11737759e-02,  3.45719523e-01,  3.00549461e-01,
      -1.11865218e+00, -2.44089151e+00, -2.22854732e+00, -1.89509228e+00 },

    { -8.48434099e-01, -5.83226811e-01,  9.00423688e-02,  8.45025008e-01,
       1.06572385e+00,  7.37582999e-01,  2.56590452e-01, -4.91963360e-01 },

    {  1.14069146e+00,  9.64016892e-01,  3.81461206e-01, -4.82849341e-01,
      -1.81632721e+00, -2.80279513e+00, -3.23385725e+00, -3.45908714e+00 },

    { -3.76283238e-01,  4.25675462e-02,  5.16547697e-01,  2.51716882e-01,
      -2.16179968e-01, -5.34074091e-01, -6.40786096e-01, -8.69745032e-01 },

    {  6.65004121e-01,  1.09790765e+00,  1.38342667e+00,  1.34327359e+00,
       8.22978837e-01,  2.15876799e-01, -4.04925753e-01, -1.07025606e+00 },

    { -8.26265954e-01, -6.71181233e-01, -2.28495593e-01,  5.18980853e-01,
       1.36721896e+00,  2.18023038e+00,  2.53596093e+00,  2.20121099e+00 },

    {  1.41008327e+00,  7.54441908e-01, -1.30550585e+00, -1.87133711e+00,
      -1.24008685e+00, -1.26712925e+00, -2.03670813e+00, -2.89685162e+00 },

    {  3.61386818e-01, -2.19991705e-02, -5.79368834e-01, -8.79427961e-01,
      -8.50685023e-01, -7.79397050e-01, -7.32182927e-01, -8.88348515e-01 },

    {  4.37469239e-01,  3.05440420e-01, -7.38786566e-03, -4.95649855e-01,
      -8.06651271e-01, -1.22431892e+00, -1.70157770e+00, -2.24491914e+00 },

    {  6.48100319e-01,  6.82299134e-01,  2.53247464e-01,  7.35842144e-02,
       3.14216709e-01,  2.34729881e-01,  1.44600134e-01, -6.82120179e-02 },

    {  1.11919833e+00,  1.23465533e+00,  5.89170238e-01, -1.37192460e+00,
      -2.37095707e+00, -2.00779783e+00, -1.66688540e+00, -1.92631846e+00 },

    {  1.41847497e-01, -1.10660071e-01, -2.82824593e-01, -6.59813475e-03,
       2.85929280e-01,  4.60445530e-02, -6.02596416e-01, -2.26568729e+00 },

    {  5.04046955e-01,  8.26982163e-01,  1.11981236e+00,  1.17914044e+00,
       1.07987429e+00,  6.97536239e-01, -9.12548817e-01, -3.57684747e+00 },

    { -5.01076050e-01, -3.25678006e-01,  2.80798195e-02,  2.62054555e-01,
       3.60590806e-01,  6.35623722e-01,  9.59012467e-01,  1.30745157e+00 },

    {  3.74970983e+00,  1.52342612e+00, -4.57715662e-01, -7.98711008e-01,
      -3.86819329e-01, -3.75901062e-01, -6.57836900e-01, -1.28163964e+00 },

    { -1.15258991e+00, -1.10800886e+00, -5.62615117e-01, -2.20562124e-01,
      -3.49842880e-01, -7.53432770e-01, -9.88596593e-01, -1.28790472e+00 },

    {  1.02827246e+00,  1.09770519e+00,  7.68645546e-01,  2.06081978e-01,
      -3.42805735e-01, -7.54939405e-01, -1.04196178e+00, -1.50335653e+00 },

    {  1.28831972e-01,  6.89439395e-01,  1.12346905e+00,  1.30934523e+00,
       1.35511965e+00,  1.42311381e+00,  1.15706449e+00,  4.06319438e-01 },

    {  1.34033030e+00,  1.38996825e+00,  1.04467922e+00,  6.35822746e-01,
      -2.74733756e-01, -1.54923372e+00, -2.44239710e+00, -3.02457607e+00 },

    {  2.13843105e+00,  4.24711267e+00,  2.89734110e+00,  9.32730658e-01,
      -2.92822250e-01, -8.10404297e-01, -7.88868099e-01, -9.35353149e-01 },

    {  5.64830487e-01,  1.59184978e+00,  2.39771699e+00,  3.03697344e+00,
       2.66424350e+00,  1.39304485e+00,  4.03834024e-01, -6.56270971e-01 },

    { -4.22460548e-01,  3.26149625e-01,  1.39171313e+00,  2.23146615e+00,
       2.61179442e+00,  2.66540340e+00,  2.40103554e+00,  1.75920380e+00 },

};

const struct lc3_sns_vq_gains lc3_sns_vq_gains[4] = {

    { 2, (const float []){
             8915.f / 4096, 12054.f / 4096 } },

    { 4, (const float []){
             6245.f / 4096, 15043.f / 4096, 17861.f / 4096, 21014.f / 4096 } },

    { 4, (const float []){
             7099.f / 4096,  9132.f / 4096, 11253.f / 4096, 14808.f / 4096 } },

    { 8, (const float []){
             4336.f / 4096,  5067.f / 4096,  5895.f / 4096,  8149.f / 4096,
            10235.f / 4096, 12825.f / 4096, 16868.f / 4096, 19882.f / 4096 } }
};

const int32_t lc3_sns_mpvq_offsets[][11] = {
    { 0, 1,  1,   1,    1,     1,      1,      1,      1,       1,       1 },
    { 0, 1,  3,   5,    7,     9,     11,     13,     15,      17,      19 },
    { 0, 1,  5,  13,   25,    41,     61,     85,    113,     145,     181 },
    { 0, 1,  7,  25,   63,   129,    231,    377,    575,     833,    1159 },
    { 0, 1,  9,  41,  129,   321,    681,   1289,   2241,    3649,    5641 },
    { 0, 1, 11,  61,  231,   681,   1683,   3653,   7183,   13073  , 22363 },
    { 0, 1, 13,  85,  377,  1289,   3653,   8989,  19825,   40081,   75517 },
    { 0, 1, 15, 113,  575,  2241,   7183,  19825,  48639,  108545,  224143 },
    { 0, 1, 17, 145,  833,  3649,  13073,  40081, 108545,  265729,  598417 },
    { 0, 1, 19, 181, 1159,  5641,  22363,  75517, 224143,  598417, 1462563 },
    { 0, 1, 21, 221, 1561,  8361,  36365, 134245, 433905, 1256465, 3317445 },
    { 0, 1, 23, 265, 2047, 11969,  56695, 227305, 795455, 2485825, 7059735 },
    { 0, 1, 25, 313, 2625, 16641,  85305, 369305,1392065, 4673345,14218905 },
    { 0, 1, 27, 365, 3303, 22569, 124515, 579125,2340495, 8405905,27298155 },
    { 0, 1, 29, 421, 4089, 29961, 177045, 880685,3800305,14546705,50250765 },
    { 0, 1, 31, 481, 4991, 39041, 246047,1303777,5984767,24331777,89129247 },
};


/**
 * TNS Arithmetic Coding (cf. 3.7.5)
 * The number of bits are given at 2048th of bits
 */

const struct lc3_ac_model lc3_tns_order_models[] = {

    { { {    0,   3 }, {    3,   9 }, {   12,  23 }, {   35,  54 },
        {   89, 111 }, {  200, 190 }, {  390, 268 }, {  658, 366 },
        { 1024,   0 }, { 1024,   0 }, { 1024,   0 }, { 1024,   0 },
        { 1024,   0 }, { 1024,   0 }, { 1024,   0 }, { 1024,   0 },
        { 1024,   0 } } },

    { { {    0,  14 }, {   14,  42 }, {   56, 100 }, {  156, 157 },
        {  313, 181 }, {  494, 178 }, {  672, 167 }, {  839, 185 },
        { 1024,   0 }, { 1024,   0 }, { 1024,   0 }, { 1024,   0 },
        { 1024,   0 }, { 1024,   0 }, { 1024,   0 }, { 1024,   0 },
        { 1024,   0 } } },
};

const uint16_t lc3_tns_order_bits[][8] = {
    { 17234, 13988, 11216, 8694, 6566, 4977, 3961, 3040 },
    { 12683,  9437,  6874, 5541, 5121, 5170, 5359, 5056 }
};

const struct lc3_ac_model lc3_tns_coeffs_models[] = {

    { { {    0,   1 }, {    1,   5 }, {    6,  15 }, {   21,  31 },
        {   52,  54 }, {  106,  86 }, {  192,  97 }, {  289, 120 },
        {  409, 159 }, {  568, 152 }, {  720, 111 }, {  831, 104 },
        {  935,  59 }, {  994,  22 }, { 1016,   6 }, { 1022,   1 },
        { 1023,   1 } } },

    { { {    0,   1 }, {    1,   1 }, {    2,   1 }, {    3,   1 },
        {    4,  13 }, {   17,  43 }, {   60,  94 }, {  154, 139 },
        {  293, 173 }, {  466, 160 }, {  626, 154 }, {  780, 131 },
        {  911,  78 }, {  989,  27 }, { 1016,   6 }, { 1022,   1 },
        { 1023,   1 } } },

    { { {    0,   1 }, {    1,   1 }, {    2,   1 }, {    3,   1 },
        {    4,   9 }, {   13,  43 }, {   56, 106 }, {  162, 199 },
        {  361, 217 }, {  578, 210 }, {  788, 141 }, {  929,  74 },
        { 1003,  17 }, { 1020,   1 }, { 1021,   1 }, { 1022,   1 },
        { 1023,   1 } } },

    { { {    0,   1 }, {    1,   1 }, {    2,   1 }, {    3,   1 },
        {    4,   2 }, {    6,  11 }, {   17,  49 }, {   66, 204 },
        {  270, 285 }, {  555, 297 }, {  852, 120 }, {  972,  39 },
        { 1011,   9 }, { 1020,   1 }, { 1021,   1 }, { 1022,   1 },
        { 1023,   1 } } },

    { { {    0,   1 }, {    1,   1 }, {    2,   1 }, {    3,   1 },
        {    4,   1 }, {    5,   7 }, {   12,  42 }, {   54, 241 },
        {  295, 341 }, {  636, 314 }, {  950,  58 }, { 1008,   9 },
        { 1017,   3 }, { 1020,   1 }, { 1021,   1 }, { 1022,   1 },
        { 1023,   1 } } },

    { { {    0,   1 }, {    1,   1 }, {    2,   1 }, {    3,   1 },
        {    4,   1 }, {    5,   1 }, {    6,  13 }, {   19, 205 },
        {  224, 366 }, {  590, 377 }, {  967,  47 }, { 1014,   5 },
        { 1019,   1 }, { 1020,   1 }, { 1021,   1 }, { 1022,   1 },
        { 1023,   1 } } },

    { { {    0,   1 }, {    1,   1 }, {    2,   1 }, {    3,   1 },
        {    4,   1 }, {    5,   1 }, {    6,  13 }, {   19, 281 },
        {  300, 330 }, {  630, 371 }, { 1001,  17 }, { 1018,   1 },
        { 1019,   1 }, { 1020,   1 }, { 1021,   1 }, { 1022,   1 },
        { 1023,   1 } } },

    { { {    0,   1 }, {    1,   1 }, {    2,   1 }, {    3,   1 },
        {    4,   1 }, {    5,   1 }, {    6,   5 }, {   11, 297 },
        {  308,   1 }, {  309, 682 }, {  991,  26 }, { 1017,   2 },
        { 1019,   1 }, { 1020,   1 }, { 1021,   1 }, { 1022,   1 },
        { 1023,   1 } } },

};

const uint16_t lc3_tns_coeffs_bits[][17] = {

    { 20480, 15725, 12479, 10334,  8694,  7320,  6964,  6335,
       5504,  5637,  6566,  6758,  8433, 11348, 15186, 20480, 20480 },

    { 20480, 20480, 20480, 20480, 12902,  9368,  7057,  5901,
       5254,  5485,  5598,  6076,  7608, 10742, 15186, 20480, 20480 },

    { 20480, 20480, 20480, 20480, 13988,  9368,  6702,  4841,
       4585,  4682,  5859,  7764, 12109, 20480, 20480, 20480, 20480 },

    { 20480, 20480, 20480, 20480, 18432, 13396,  8982,  4767,
       3779,  3658,  6335,  9656, 13988, 20480, 20480, 20480, 20480 },

    { 20480, 20480, 20480, 20480, 20480, 14731,  9437,  4275,
       3249,  3493,  8483, 13988, 17234, 20480, 20480, 20480, 20480 },

    { 20480, 20480, 20480, 20480, 20480, 20480, 12902,  4753,
       3040,  2953,  9105, 15725, 20480, 20480, 20480, 20480, 20480 },

    { 20480, 20480, 20480, 20480, 20480, 20480, 12902,  3821,
       3346,  3000, 12109, 20480, 20480, 20480, 20480, 20480, 20480 },

    { 20480, 20480, 20480, 20480, 20480, 20480, 15725,  3658,
      20480,  1201, 10854, 18432, 20480, 20480, 20480, 20480, 20480 }

};


/**
 * Long Term Postfilter Synthesis (cf. 3.7.6)
 * with - addition of a 0 for num coefficients
 *      - remove of first 0 den coefficients
 */

const float *lc3_ltpf_cnum[LC3_NUM_SRATE][4] = {

    [LC3_SRATE_8K] = {
        (const float []){
           6.02361821e-01,  4.19760926e-01, -1.88342453e-02,  0. },
        (const float []){
           5.99476858e-01,  4.19760926e-01, -1.59492828e-02,  0. },
        (const float []){
           5.96776466e-01,  4.19760926e-01, -1.32488910e-02,  0. },
        (const float []){
           5.94241012e-01,  4.19760926e-01, -1.07134366e-02,  0. },
    },

    [LC3_SRATE_16K] = {
        (const float []){
           6.02361821e-01,  4.19760926e-01, -1.88342453e-02,  0. },
        (const float []){
           5.99476858e-01,  4.19760926e-01, -1.59492828e-02,  0. },
        (const float []){
           5.96776466e-01,  4.19760926e-01, -1.32488910e-02,  0. },
        (const float []){
           5.94241012e-01,  4.19760926e-01, -1.07134366e-02,  0. },
    },

    [LC3_SRATE_24K] = {
        (const float []){
           3.98969559e-01,  5.14250861e-01,  1.00438297e-01, -1.27889396e-02,
          -1.57228008e-03,  0.                                               },
        (const float []){
           3.94863491e-01,  5.12381921e-01,  1.04319493e-01, -1.09199996e-02,
          -1.34740833e-03,  0.                                               },
        (const float []){
           3.90984448e-01,  5.10605352e-01,  1.07983252e-01, -9.14343107e-03,
          -1.13212462e-03,  0.                                               },
        (const float []){
           3.87309389e-01,  5.08912208e-01,  1.11451738e-01, -7.45028713e-03,
          -9.25551405e-04,  0.                                               },
    },

    [LC3_SRATE_32K] = {
        (const float []){
           2.98237945e-01,  4.65280920e-01,  2.10599743e-01,  3.76678038e-02,
          -1.01569616e-02, -2.53588100e-03, -3.18294617e-04,  0.             },
        (const float []){
           2.94383415e-01,  4.61929400e-01,  2.12946577e-01,  4.06617500e-02,
          -8.69327230e-03, -2.17830711e-03, -2.74288806e-04,  0.             },
        (const float []){
           2.90743921e-01,  4.58746191e-01,  2.15145697e-01,  4.35010477e-02,
          -7.29549535e-03, -1.83439564e-03, -2.31692019e-04,  0.             },
        (const float []){
           2.87297585e-01,  4.55714889e-01,  2.17212695e-01,  4.62008888e-02,
          -5.95746380e-03, -1.50293428e-03, -1.90385191e-04,  0.             },
    },

    [LC3_SRATE_48K] = {
        (const float []){
           1.98136374e-01,  3.52449490e-01,  2.51369527e-01,  1.42414624e-01,
           5.70473102e-02,  9.29336624e-03, -7.22602537e-03, -3.17267989e-03,
          -1.12183596e-03, -2.90295724e-04, -4.27081559e-05,  0.             },
        (const float []){
           1.95070943e-01,  3.48466041e-01,  2.50998846e-01,  1.44116741e-01,
           5.92894732e-02,  1.10892383e-02, -6.19290811e-03, -2.72670551e-03,
          -9.66712583e-04, -2.50810092e-04, -3.69993877e-05,  0.             },
        (const float []){
           1.92181006e-01,  3.44694556e-01,  2.50622009e-01,  1.45710245e-01,
           6.14113213e-02,  1.27994140e-02, -5.20372109e-03, -2.29732451e-03,
          -8.16560813e-04, -2.12385575e-04, -3.14127133e-05,  0.             },
        (const float []){
           1.89448531e-01,  3.41113925e-01,  2.50240688e-01,  1.47206563e-01,
           6.34247723e-02,  1.44320343e-02, -4.25444914e-03, -1.88308147e-03,
          -6.70961906e-04, -1.74936334e-04, -2.59386474e-05,  0.             },
    }
};

const float *lc3_ltpf_cden[LC3_NUM_SRATE][4] = {

    [LC3_SRATE_8K] = {
        (const float []){
           2.09880463e-01,  5.83527575e-01,  2.09880463e-01,  0.00000000e+00 },
        (const float []){
           1.06999186e-01,  5.50075002e-01,  3.35690625e-01,  6.69885837e-03 },
        (const float []){
           3.96711478e-02,  4.59220930e-01,  4.59220930e-01,  3.96711478e-02 },
        (const float []){
           6.69885837e-03,  3.35690625e-01,  5.50075002e-01,  1.06999186e-01 },
    },

    [LC3_SRATE_16K] = {
        (const float []){
           2.09880463e-01,  5.83527575e-01,  2.09880463e-01,  0.00000000e+00 },
        (const float []){
           1.06999186e-01,  5.50075002e-01,  3.35690625e-01,  6.69885837e-03 },
        (const float []){
           3.96711478e-02,  4.59220930e-01,  4.59220930e-01,  3.96711478e-02 },
        (const float []){
           6.69885837e-03,  3.35690625e-01,  5.50075002e-01,  1.06999186e-01 },
    },

    [LC3_SRATE_24K] = {
        (const float []){
           6.32223163e-02,  2.50730961e-01,  3.71390943e-01,  2.50730961e-01,
           6.32223163e-02,  0.00000000e+00                                   },
        (const float []){
           3.45927217e-02,  1.98651560e-01,  3.62641173e-01,  2.98675055e-01,
           1.01309287e-01,  4.26354371e-03                                   },
        (const float []){
           1.53574678e-02,  1.47434488e-01,  3.37425955e-01,  3.37425955e-01,
           1.47434488e-01,  1.53574678e-02                                   },
        (const float []){
           4.26354371e-03,  1.01309287e-01,  2.98675055e-01,  3.62641173e-01,
           1.98651560e-01,  3.45927217e-02                                   },
    },

    [LC3_SRATE_32K] = {
        (const float []){
           2.90040188e-02,  1.12985742e-01,  2.21202403e-01,  2.72390947e-01,
           2.21202403e-01,  1.12985742e-01,  2.90040188e-02,  0.00000000e+00 },
        (const float []){
           1.70315342e-02,  8.72250379e-02,  1.96140776e-01,  2.68923798e-01,
           2.42499910e-01,  1.40577336e-01,  4.47487717e-02,  3.12703024e-03 },
        (const float []){
           8.56367375e-03,  6.42622294e-02,  1.68767671e-01,  2.58744594e-01,
           2.58744594e-01,  1.68767671e-01,  6.42622294e-02,  8.56367375e-03 },
        (const float []){
           3.12703024e-03,  4.47487717e-02,  1.40577336e-01,  2.42499910e-01,
           2.68923798e-01,  1.96140776e-01,  8.72250379e-02,  1.70315342e-02 },
    },

    [LC3_SRATE_48K] = {
        (const float []){
           1.08235939e-02,  3.60896922e-02,  7.67640147e-02,  1.24153058e-01,
           1.62759644e-01,  1.77677142e-01,  1.62759644e-01,  1.24153058e-01,
           7.67640147e-02,  3.60896922e-02,  1.08235939e-02,  0.00000000e+00 },
        (const float []){
           7.04140493e-03,  2.81970232e-02,  6.54704494e-02,  1.12464799e-01,
           1.54841896e-01,  1.76712238e-01,  1.69150721e-01,  1.35290158e-01,
           8.85142501e-02,  4.49935385e-02,  1.55761371e-02,  2.03972196e-03 },
        (const float []){
           4.14699847e-03,  2.13575731e-02,  5.48273558e-02,  1.00497144e-01,
           1.45606034e-01,  1.73843984e-01,  1.73843984e-01,  1.45606034e-01,
           1.00497144e-01,  5.48273558e-02,  2.13575731e-02,  4.14699847e-03 },
        (const float []){
           2.03972196e-03,  1.55761371e-02,  4.49935385e-02,  8.85142501e-02,
           1.35290158e-01,  1.69150721e-01,  1.76712238e-01,  1.54841896e-01,
           1.12464799e-01,  6.54704494e-02,  2.81970232e-02,  7.04140493e-03 },
    }
};


/**
 * Spectral Data Arithmetic Coding (cf. 3.7.7)
 * The number of bits are given at 2048th of bits
 *
 * The dimensions of the lookup table are set as following :
 *   1: Rate selection
 *   2: Half spectrum selection (1st half / 2nd half)
 *   3: State of the arithmetic coder
 *   4: Number of msb bits (significant - 2), limited to 3
 *
 * table[r][h][s][k] = table(normative)[s + h*256 + r*512 + k*1024]
 */

const uint8_t lc3_spectrum_lookup[2][2][256][4] = {

    { { {  1,13, 0, 0 }, { 39,13, 0, 0 }, {  7,13, 0, 0 }, { 25,13, 0, 0 },
        { 22,13, 0, 0 }, { 22,13, 0, 0 }, { 28,13, 0, 0 }, { 22,13, 0, 0 },
        { 22,60, 0, 0 }, { 22,60, 0, 0 }, { 22,60, 0, 0 }, { 28,60, 0, 0 },
        { 28,60, 0, 0 }, { 28,60,13, 0 }, { 34,60,13, 0 }, { 31,16,13, 0 },
        { 31,16,13, 0 }, { 40, 0, 0, 0 }, { 43, 0, 0, 0 }, { 46, 0, 0, 0 },
        { 49, 0, 0, 0 }, { 52, 0, 0, 0 }, { 14, 0, 0, 0 }, { 17, 0, 0, 0 },
        { 36, 0, 0, 0 }, { 36, 0, 0, 0 }, { 36, 0, 0, 0 }, { 38, 0, 0, 0 },
        {  0, 0, 0, 0 }, { 57, 0, 0, 0 }, { 38,13, 0, 0 }, { 22,60, 0, 0 },
        {  0, 0, 0, 0 }, {  8, 0, 0, 0 }, {  9, 0, 0, 0 }, { 11, 0, 0, 0 },
        { 47, 0, 0, 0 }, { 14, 0, 0, 0 }, { 14, 0, 0, 0 }, { 17, 0, 0, 0 },
        { 36, 0, 0, 0 }, { 36, 0, 0, 0 }, { 36, 0, 0, 0 }, { 38, 0, 0, 0 },
        { 59, 0, 0, 0 }, { 59, 0, 0, 0 }, { 38,13, 0, 0 }, { 22,60, 0, 0 },
        { 22,60, 0, 0 }, { 26, 0, 0, 0 }, { 46, 0, 0, 0 }, { 29, 0, 0, 0 },
        { 30, 0, 0, 0 }, { 32, 0, 0, 0 }, { 33, 0, 0, 0 }, { 35, 0, 0, 0 },
        { 36, 0, 0, 0 }, { 36, 0, 0, 0 }, { 36, 0, 0, 0 }, { 38, 0, 0, 0 },
        {  0,13, 0, 0 }, { 59,13, 0, 0 }, { 23,13, 0, 0 }, { 22,60, 0, 0 },
        { 46,60, 0, 0 }, { 46, 0, 0, 0 }, { 45, 0, 0, 0 }, { 47, 0, 0, 0 },
        { 48, 0, 0, 0 }, { 50, 0, 0, 0 }, { 50, 0, 0, 0 }, { 18, 0, 0, 0 },
        { 54, 0, 0, 0 }, { 54, 0, 0, 0 }, { 54, 0, 0, 0 }, { 38, 0, 0, 0 },
        { 59,13, 0, 0 }, { 59,13, 0, 0 }, { 59,13, 0, 0 }, { 22,60, 0, 0 },
        {  0,60, 0, 0 }, { 62, 0, 0, 0 }, { 63, 0, 0, 0 }, {  3, 0, 0, 0 },
        { 33, 0, 0, 0 }, {  2, 0, 0, 0 }, {  2, 0, 0, 0 }, { 61, 0, 0, 0 },
        { 20, 0, 0, 0 }, { 20, 0, 0, 0 }, { 20,13, 0, 0 }, { 21,13, 0, 0 },
        { 59,13, 0, 0 }, { 59,13, 0, 0 }, { 39,13, 0, 0 }, { 28,60, 0, 0 },
        { 28,60, 0, 0 }, { 63, 0, 0, 0 }, { 63, 0, 0, 0 }, {  3, 0, 0, 0 },
        { 33, 0, 0, 0 }, {  2, 0, 0, 0 }, {  2, 0, 0, 0 }, { 61, 0, 0, 0 },
        { 38, 0, 0, 0 }, { 38, 0, 0, 0 }, { 38,13, 0, 0 }, { 21,13, 0, 0 },
        { 59,13, 0, 0 }, { 59,13, 0, 0 }, { 39,13, 0, 0 }, { 28,60, 0, 0 },
        { 28,60, 0, 0 }, {  6, 0, 0, 0 }, {  6, 0, 0, 0 }, {  6, 0, 0, 0 },
        {  2, 0, 0, 0 }, { 18, 0, 0, 0 }, { 61, 0, 0, 0 }, { 20, 0, 0, 0 },
        { 21, 0, 0, 0 }, { 21, 0, 0, 0 }, { 21,13, 0, 0 }, { 59,13, 0, 0 },
        { 39,13, 0, 0 }, { 39,13, 0, 0 }, {  7,13, 0, 0 }, { 34,60,13, 0 },
        { 34,60,13, 0 }, { 34,60,13, 0 }, { 34,60,13, 0 }, { 34,60,13, 0 },
        { 34,60,13, 0 }, { 34,60,13, 0 }, { 34,60,13, 0 }, { 34,60,13, 0 },
        { 34,60,13, 0 }, { 34,60,13, 0 }, { 34,60,13, 0 }, { 34,60,13, 0 },
        { 34,60,13, 0 }, { 34,60,13, 0 }, { 34,60,13, 0 }, { 34,60,13, 0 },
        { 34,60,13, 0 }, { 51, 0, 0, 0 }, { 51, 0, 0, 0 }, { 51, 0, 0, 0 },
        { 53, 0, 0, 0 }, { 54, 0, 0, 0 }, { 20, 0, 0, 0 }, { 38, 0, 0, 0 },
        { 38, 0, 0, 0 }, { 57, 0, 0, 0 }, { 39,13, 0, 0 }, { 39,13, 0, 0 },
        { 39,13, 0, 0 }, {  7,13, 0, 0 }, { 24,13, 0, 0 }, { 34,60,13, 0 },
        {  4,60, 0, 0 }, {  4,60, 0, 0 }, {  4,60, 0, 0 }, {  4,60, 0, 0 },
        {  4,60, 0, 0 }, {  4,60, 0, 0 }, {  4,60, 0, 0 }, {  4,60, 0, 0 },
        {  4,60, 0, 0 }, {  4,60, 0, 0 }, {  4,60, 0, 0 }, {  4,60, 0, 0 },
        {  4,60, 0, 0 }, {  4,60, 0, 0 }, {  4,60, 0, 0 }, {  4,60, 0, 0 },
        {  4,60, 0, 0 }, {  4, 0, 0, 0 }, {  4, 0, 0, 0 }, {  4, 0, 0, 0 },
        {  4, 0, 0, 0 }, { 56, 0, 0, 0 }, { 38, 0, 0, 0 }, { 57, 0, 0, 0 },
        { 57,13, 0, 0 }, { 59,13, 0, 0 }, {  7,13, 0, 0 }, {  7,13, 0, 0 },
        {  7,13, 0, 0 }, { 42,13, 0, 0 }, { 42,13, 0, 0 }, { 34,60,13, 0 },
        {  0,60,13, 0 }, {  0,60,13, 0 }, {  0,60,13, 0 }, {  0,60,13, 0 },
        {  0,60,13, 0 }, {  0,60,13, 0 }, {  0,60,13, 0 }, {  0,60,13, 0 },
        {  0,60,13, 0 }, {  0,60,13, 0 }, {  0,60,13, 0 }, {  0,60,13, 0 },
        {  0,60,13, 0 }, {  0,60,13, 0 }, {  0,60,13, 0 }, {  0,60,13, 0 },
        {  0,60,13, 0 }, {  5, 0, 0, 0 }, {  4, 0, 0, 0 }, {  4, 0, 0, 0 },
        {  5, 0, 0, 0 }, { 21, 0, 0, 0 }, { 21, 0, 0, 0 }, { 59,13, 0, 0 },
        {  7,13, 0, 0 }, {  7,13, 0, 0 }, {  7,13, 0, 0 }, {  7,13, 0, 0 },
        { 25,13, 0, 0 }, { 25,13, 0, 0 }, { 25,13, 0, 0 }, { 34,60,13, 0 },
        {  4,13, 0, 0 }, {  4,13, 0, 0 }, {  4,13, 0, 0 }, {  4,13, 0, 0 },
        {  5,13, 0, 0 }, { 23,13, 0, 0 }, { 23,13, 0, 0 }, { 39,13, 0, 0 },
        {  7,13, 0, 0 }, {  7,13, 0, 0 }, {  7,13, 0, 0 }, { 42,13, 0, 0 },
        { 25,13, 0, 0 }, { 25,13, 0, 0 }, { 22,13, 0, 0 }, { 31,60,13, 0 },
        { 31,60,13, 0 }, { 39,60, 0, 0 }, { 39,60, 0, 0 }, { 39,60, 0, 0 },
        { 39,60, 0, 0 }, {  7,60, 0, 0 }, {  7,60, 0, 0 }, { 42,60, 0, 0 },
        {  0,60, 0, 0 }, { 25,60, 0, 0 }, { 22,60, 0, 0 }, { 22,60, 0, 0 },
        { 22,60, 0, 0 }, { 28,60, 0, 0 }, { 34,60, 0, 0 }, { 31,16,13, 0 } },

      { { 55, 0,13, 0 }, { 55, 0,13, 0 }, { 55, 0,13, 0 }, { 55, 0,13, 0 },
        { 55, 0,13, 0 }, { 55, 0,13, 0 }, { 55, 0,13, 0 }, { 55, 0,13, 0 },
        { 55, 0,13, 0 }, { 55, 0,13, 0 }, { 55, 0,13, 0 }, { 55, 0,13, 0 },
        { 55, 0,13, 0 }, { 55, 0,13, 0 }, { 55, 0,13, 0 }, { 55, 0,13, 0 },
        { 55, 0,13, 0 }, { 55, 0, 0, 0 }, { 40, 0, 0, 0 }, {  8, 0, 0, 0 },
        {  9, 0, 0, 0 }, { 49, 0, 0, 0 }, { 49, 0, 0, 0 }, { 52, 0, 0, 0 },
        { 17, 0, 0, 0 }, { 17, 0, 0, 0 }, { 17, 0, 0, 0 }, {  4,13, 0, 0 },
        {  0,13, 0, 0 }, { 20,13, 0, 0 }, { 17, 0, 0, 0 }, { 60,13,60,13 },
        { 40, 0, 0,13 }, { 40, 0, 0, 0 }, {  8, 0, 0, 0 }, { 43, 0, 0, 0 },
        { 27, 0, 0, 0 }, { 49, 0, 0, 0 }, { 49, 0, 0, 0 }, { 14, 0, 0, 0 },
        { 17, 0, 0, 0 }, { 17, 0, 0, 0 }, { 17, 0, 0, 0 }, { 36, 0, 0, 0 },
        { 42,13, 0, 0 }, { 42,13, 0, 0 }, { 17, 0, 0, 0 }, { 57,60,13, 0 },
        { 57, 0,13, 0 }, { 40, 0, 0, 0 }, {  8, 0, 0, 0 }, { 26, 0, 0, 0 },
        { 27, 0, 0, 0 }, { 49, 0, 0, 0 }, { 12, 0, 0, 0 }, { 14, 0, 0, 0 },
        { 17, 0, 0, 0 }, { 17, 0, 0, 0 }, { 17, 0, 0, 0 }, { 36, 0, 0, 0 },
        {  0, 0,13, 0 }, { 38, 0,13, 0 }, { 36,13, 0, 0 }, {  1,60, 0, 0 },
        {  8,60, 0, 0 }, {  8, 0, 0, 0 }, { 43, 0, 0, 0 }, {  9, 0, 0, 0 },
        { 11, 0, 0, 0 }, { 49, 0, 0, 0 }, { 12, 0, 0, 0 }, { 14, 0, 0, 0 },
        { 14, 0,13, 0 }, { 33, 0,13, 0 }, { 50, 0,13, 0 }, { 50, 0, 0, 0 },
        { 50, 0,13, 0 }, { 61, 0,13, 0 }, { 36,13, 0, 0 }, { 39,60, 0, 0 },
        {  8,60, 0, 0 }, {  8, 0, 0, 0 }, { 43, 0, 0, 0 }, { 46, 0, 0, 0 },
        { 49, 0, 0, 0 }, { 52, 0, 0, 0 }, { 30, 0, 0, 0 }, { 14, 0, 0, 0 },
        { 14, 0,13, 0 }, { 33, 0,13, 0 }, { 50, 0,13, 0 }, { 50, 0,13, 0 },
        { 50,13,13, 0 }, { 50,13, 0, 0 }, { 18,13,13, 0 }, { 25,60,13, 0 },
        {  8,60,13,13 }, {  8, 0, 0,13 }, { 43, 0, 0,13 }, { 46, 0, 0,13 },
        { 49, 0, 0,13 }, { 52, 0, 0, 0 }, { 30, 0, 0, 0 }, { 14, 0, 0, 0 },
        { 14, 0, 0, 0 }, { 18, 0,60, 0 }, {  5, 0, 0,13 }, {  5, 0, 0,13 },
        {  5, 0, 0,13 }, { 61,13, 0,13 }, { 18,13,13, 0 }, { 23,13,60, 0 },
        { 43,13, 0,13 }, { 43, 0, 0,13 }, { 43, 0, 0,13 }, {  9, 0, 0,13 },
        { 49, 0, 0,13 }, { 52, 0, 0, 0 }, {  3, 0, 0, 0 }, { 14, 0, 0, 0 },
        { 14, 0, 0, 0 }, { 50, 0, 0, 0 }, { 50,13,13, 0 }, { 50,13,13, 0 },
        { 50,13,13, 0 }, { 61, 0, 0, 0 }, { 17,13,13, 0 }, { 24,60,13, 0 },
        { 43,60,13, 0 }, { 43,60,13, 0 }, { 43,60,13, 0 }, { 43,60,13, 0 },
        { 43,60,13, 0 }, { 43,60,13, 0 }, { 43,60,13, 0 }, { 43,60,13, 0 },
        { 43,60,13, 0 }, { 43,60,13, 0 }, { 43,60,13, 0 }, { 43,60,13, 0 },
        { 43,60,13, 0 }, { 43,60,13, 0 }, { 43,60,13, 0 }, { 43,60,13, 0 },
        { 43,60,13, 0 }, { 43, 0, 0, 0 }, { 43, 0,19, 0 }, {  9, 0, 0, 0 },
        { 11, 0, 0, 0 }, { 52, 0, 0, 0 }, { 52, 0, 0, 0 }, { 14, 0, 0, 0 },
        { 14, 0, 0, 0 }, { 17, 0, 0, 0 }, { 61,13, 0, 0 }, { 61,13, 0, 0 },
        { 61,13, 0, 0 }, { 54, 0, 0, 0 }, { 17, 0,13,13 }, { 39,13,13, 0 },
        { 45,13,13, 0 }, { 45,13,13, 0 }, { 45,13,13, 0 }, { 45,13,13, 0 },
        { 45,13,13, 0 }, { 45,13,13, 0 }, { 45,13,13, 0 }, { 45,13,13, 0 },
        { 45,13,13, 0 }, { 45,13,13, 0 }, { 45,13,13, 0 }, { 45,13,13, 0 },
        { 45,13,13, 0 }, { 45,13,13, 0 }, { 45,13,13, 0 }, { 45,13,13, 0 },
        { 45,13,13, 0 }, { 45, 0,13, 0 }, { 44, 0,13, 0 }, { 27, 0, 0, 0 },
        { 29, 0, 0, 0 }, { 52, 0, 0, 0 }, { 48, 0, 0, 0 }, { 52, 0, 0, 0 },
        { 52, 0, 0, 0 }, { 17, 0, 0, 0 }, { 17, 0, 0, 0 }, { 17, 0,19, 0 },
        { 17, 0,13, 0 }, {  2, 0,13, 0 }, { 17, 0,13, 0 }, {  7,13, 0, 0 },
        { 27, 0, 0,13 }, { 27, 0, 0,13 }, { 27, 0, 0,13 }, { 27, 0, 0,13 },
        { 27, 0, 0,13 }, { 27, 0, 0,13 }, { 27, 0, 0,13 }, { 27, 0, 0,13 },
        { 27, 0, 0,13 }, { 27, 0, 0,13 }, { 27, 0, 0,13 }, { 27, 0, 0,13 },
        { 27, 0, 0,13 }, { 27, 0, 0,13 }, { 27, 0, 0,13 }, { 27, 0, 0,13 },
        { 27, 0, 0,13 }, { 27, 0, 0,13 }, {  9, 0, 0,13 }, { 27, 0, 0,13 },
        { 27, 0, 0,13 }, { 12, 0, 0,13 }, { 52, 0, 0,13 }, { 14, 0, 0,13 },
        { 14, 0, 0,13 }, { 58, 0, 0,13 }, { 41, 0, 0,13 }, { 41, 0, 0,13 },
        { 41, 0, 0,13 }, {  6, 0, 0,13 }, { 17,60, 0,13 }, { 37, 0,19,13 },
        {  9, 0, 0,13 }, {  9,16, 0,13 }, {  9, 0, 0,13 }, { 27, 0, 0,13 },
        { 11, 0, 0,13 }, { 49, 0, 0, 0 }, { 12, 0, 0, 0 }, { 52, 0, 0, 0 },
        { 14, 0, 0, 0 }, { 14, 0, 0, 0 }, { 14, 0, 0, 0 }, { 50, 0, 0, 0 },
        {  0, 0, 0,13 }, { 53, 0, 0,13 }, { 17, 0, 0,13 }, { 28, 0,13, 0 },
        { 52, 0,13, 0 }, { 52, 0,13, 0 }, { 49, 0,13, 0 }, { 52, 0, 0, 0 },
        { 12, 0, 0, 0 }, { 52, 0, 0, 0 }, { 30, 0, 0, 0 }, { 14, 0, 0, 0 },
        { 14, 0, 0, 0 }, { 17, 0, 0, 0 }, {  2, 0, 0, 0 }, {  2, 0, 0, 0 },
        {  2, 0, 0, 0 }, { 38, 0, 0, 0 }, { 38, 0, 0, 0 }, { 34, 0, 0, 0 } } },

    { { { 31,16,60,13 }, { 34,16,13, 0 }, { 34,16,13, 0 }, { 31,16,13, 0 },
        { 31,16,13, 0 }, { 31,16,13, 0 }, { 31,16,13, 0 }, { 19,16,60, 0 },
        { 19,16,60, 0 }, { 19,16,60, 0 }, { 19,16,60, 0 }, { 19,16,60, 0 },
        { 19,16,60, 0 }, { 19,16,60, 0 }, { 31,16,60,13 }, { 19,37,16,60 },
        { 44, 0, 0,60 }, { 44, 0, 0, 0 }, { 62, 0, 0, 0 }, { 30, 0, 0, 0 },
        { 32, 0, 0, 0 }, { 58, 0, 0, 0 }, { 35, 0, 0, 0 }, { 36, 0, 0, 0 },
        { 36, 0, 0, 0 }, { 38,13, 0, 0 }, {  0,13, 0, 0 }, { 59,13, 0, 0 },
        {  7,13, 0, 0 }, {  7,13, 0, 0 }, { 39,13, 0, 0 }, { 34,60,13, 0 },
        { 34, 0,13, 0 }, { 45, 0, 0, 0 }, { 47, 0, 0, 0 }, { 48, 0, 0, 0 },
        { 33, 0, 0, 0 }, { 35, 0, 0, 0 }, { 35, 0, 0, 0 }, { 36, 0, 0, 0 },
        { 38,13, 0, 0 }, { 38,13, 0, 0 }, { 38,13, 0, 0 }, { 59,13, 0, 0 },
        {  7,13, 0, 0 }, {  7,13, 0, 0 }, { 39,13, 0, 0 }, { 34,60,13, 0 },
        { 34, 0,13, 0 }, { 62, 0, 0, 0 }, { 30, 0, 0, 0 }, { 15, 0, 0, 0 },
        { 50, 0, 0, 0 }, { 53, 0, 0, 0 }, { 53, 0, 0, 0 }, { 54,13, 0, 0 },
        { 21,13, 0, 0 }, { 21,13, 0, 0 }, { 21,13, 0, 0 }, { 59,13, 0, 0 },
        {  7,13, 0, 0 }, {  7,13, 0, 0 }, {  7,13, 0, 0 }, { 34,60,13, 0 },
        { 30, 0,13, 0 }, { 30, 0, 0, 0 }, { 48, 0, 0, 0 }, { 33, 0, 0, 0 },
        { 58, 0, 0, 0 }, { 18, 0, 0, 0 }, { 18, 0, 0, 0 }, { 56,13, 0, 0 },
        { 23,13, 0, 0 }, { 23,13, 0, 0 }, { 23,13, 0, 0 }, { 59,13, 0, 0 },
        {  7,13, 0, 0 }, {  7,13, 0, 0 }, { 24,13, 0, 0 }, { 34,60,13, 0 },
        { 34, 0,13, 0 }, {  6, 0, 0, 0 }, {  6, 0, 0, 0 }, { 58, 0, 0, 0 },
        { 53, 0, 0, 0 }, { 54, 0, 0, 0 }, { 54, 0, 0, 0 }, { 21,13, 0, 0 },
        { 59,13, 0, 0 }, { 59,13, 0, 0 }, { 59,13, 0, 0 }, { 39,13, 0, 0 },
        {  7,13, 0, 0 }, {  7,13, 0, 0 }, { 42,60, 0, 0 }, { 34,16,13, 0 },
        {  6, 0,13, 0 }, {  6, 0, 0, 0 }, { 33, 0, 0, 0 }, { 58, 0, 0, 0 },
        { 53, 0, 0, 0 }, { 54, 0, 0, 0 }, { 61, 0, 0, 0 }, { 21,13, 0, 0 },
        { 59,13, 0, 0 }, { 59,13, 0, 0 }, { 59,13, 0, 0 }, { 39,13, 0, 0 },
        {  7,13, 0, 0 }, {  7,13, 0, 0 }, { 42,60, 0, 0 }, { 34,16,13, 0 },
        { 34, 0,13, 0 }, { 51, 0, 0, 0 }, { 51, 0, 0, 0 }, { 53, 0, 0, 0 },
        { 54, 0, 0, 0 }, { 56,13, 0, 0 }, { 56,13, 0, 0 }, { 57,13, 0, 0 },
        { 39,13, 0, 0 }, { 39,13, 0, 0 }, { 39,13, 0, 0 }, {  7,13, 0, 0 },
        { 42,13, 0, 0 }, { 42,13, 0, 0 }, { 25,60, 0, 0 }, { 31,16,13, 0 },
        { 31, 0,13, 0 }, { 31, 0,13, 0 }, { 31, 0,13, 0 }, { 31, 0,13, 0 },
        { 31, 0,13, 0 }, { 31, 0,13, 0 }, { 31, 0,13, 0 }, { 31, 0,13, 0 },
        { 31, 0,13, 0 }, { 31, 0,13, 0 }, { 31, 0,13, 0 }, { 31, 0,13, 0 },
        { 31, 0,13, 0 }, { 31, 0,13, 0 }, { 31, 0,13, 0 }, { 31, 0,13, 0 },
        { 31, 0,13, 0 }, {  4, 0, 0, 0 }, {  4, 0, 0, 0 }, {  4, 0, 0, 0 },
        {  5,13, 0, 0 }, { 23,13, 0, 0 }, { 23,13, 0, 0 }, { 39,13, 0, 0 },
        {  7,13, 0, 0 }, {  7,13, 0, 0 }, {  7,13, 0, 0 }, { 42,13, 0, 0 },
        { 25,13, 0, 0 }, { 25,13, 0, 0 }, { 22,60, 0, 0 }, { 31,16,60, 0 },
        { 31,13, 0, 0 }, { 31,13, 0, 0 }, { 31,13, 0, 0 }, { 31,13, 0, 0 },
        { 31,13, 0, 0 }, { 31,13, 0, 0 }, { 31,13, 0, 0 }, { 31,13, 0, 0 },
        { 31,13, 0, 0 }, { 31,13, 0, 0 }, { 31,13, 0, 0 }, { 31,13, 0, 0 },
        { 31,13, 0, 0 }, { 31,13, 0, 0 }, { 31,13, 0, 0 }, { 31,13, 0, 0 },
        { 31,13, 0, 0 }, {  5,13, 0, 0 }, {  5,13, 0, 0 }, {  5,13, 0, 0 },
        {  5,13, 0, 0 }, { 57,13, 0, 0 }, { 57,13, 0, 0 }, { 39,13, 0, 0 },
        { 24,13, 0, 0 }, { 24,13, 0, 0 }, { 24,13, 0, 0 }, { 42,13, 0, 0 },
        { 22,13, 0, 0 }, { 22,60, 0, 0 }, { 28,60,13, 0 }, { 31,16,60, 0 },
        { 31,13, 0, 0 }, { 31,13, 0, 0 }, { 31,13, 0, 0 }, { 31,13, 0, 0 },
        { 31,13, 0, 0 }, { 31,13, 0, 0 }, { 31,13, 0, 0 }, { 31,13, 0, 0 },
        { 31,13, 0, 0 }, { 31,13, 0, 0 }, { 31,13, 0, 0 }, { 31,13, 0, 0 },
        { 31,13, 0, 0 }, { 31,13, 0, 0 }, { 31,13, 0, 0 }, { 31,13, 0, 0 },
        { 31,13, 0, 0 }, { 41,13, 0, 0 }, { 41,13, 0, 0 }, { 41,13, 0, 0 },
        { 41,13, 0, 0 }, { 39,13, 0, 0 }, { 39,13, 0, 0 }, {  7,13, 0, 0 },
        { 42,13, 0, 0 }, { 42,13, 0, 0 }, { 42,13, 0, 0 }, { 25,13, 0, 0 },
        { 28,13, 0, 0 }, { 28,60, 0, 0 }, { 28,60,13, 0 }, { 31,16,60,13 },
        { 31,13, 0, 0 }, { 41,13, 0, 0 }, { 41,13, 0, 0 }, { 41,13, 0, 0 },
        { 41,13, 0, 0 }, { 39,13, 0, 0 }, { 39,13, 0, 0 }, { 24,13, 0, 0 },
        { 25,60, 0, 0 }, { 25,60, 0, 0 }, { 25,60, 0, 0 }, { 22,60, 0, 0 },
        { 28,60, 0, 0 }, { 28,60, 0, 0 }, { 34,60,13, 0 }, { 31,16,60,13 },
        { 31,60,13,13 }, { 10,60,13, 0 }, { 10,60,13, 0 }, { 10,60,13, 0 },
        { 10,60,13, 0 }, { 10,60,13, 0 }, { 10,60,13, 0 }, { 28,60,13, 0 },
        { 34,60,13, 0 }, { 34,60,13, 0 }, { 34,16,13, 0 }, { 34,16,13, 0 },
        { 34,16,60, 0 }, { 34,16,60, 0 }, { 31,16,60, 0 }, { 19,37,16,13 } },

      { {  8, 0,16, 0 }, {  8, 0,16, 0 }, {  8, 0,16, 0 }, {  8, 0,16, 0 },
        {  8, 0,16, 0 }, {  8, 0,16, 0 }, {  8, 0,16, 0 }, {  8, 0,16, 0 },
        {  8, 0,16, 0 }, {  8, 0,16, 0 }, {  8, 0,16, 0 }, {  8, 0,16, 0 },
        {  8, 0,16, 0 }, {  8, 0,16, 0 }, {  8, 0,16, 0 }, {  8, 0,16, 0 },
        {  8, 0,16, 0 }, {  8, 0, 0, 0 }, {  9, 0, 0, 0 }, { 11, 0, 0, 0 },
        { 47, 0, 0, 0 }, { 32, 0, 0, 0 }, { 50, 0, 0, 0 }, { 18, 0, 0, 0 },
        { 18, 0, 0, 0 }, { 20, 0, 0, 0 }, { 21, 0, 0, 0 }, { 21, 0, 0, 0 },
        { 21,13, 0, 0 }, { 39,13, 0, 0 }, { 59,13, 0, 0 }, { 34,16,60, 0 },
        { 26, 0, 0, 0 }, { 26, 0, 0, 0 }, { 27, 0, 0, 0 }, { 29, 0, 0, 0 },
        { 30, 0, 0, 0 }, { 33, 0, 0, 0 }, { 50, 0, 0, 0 }, { 18, 0, 0, 0 },
        { 18, 0, 0, 0 }, { 20, 0, 0, 0 }, { 57, 0, 0, 0 }, { 57,13, 0, 0 },
        { 57,13, 0, 0 }, { 59,13, 0, 0 }, { 59,13, 0, 0 }, { 34,16,60, 0 },
        { 27, 0, 0, 0 }, { 27, 0, 0, 0 }, { 11, 0, 0, 0 }, { 12, 0, 0, 0 },
        { 48, 0, 0, 0 }, { 50, 0, 0, 0 }, { 58, 0, 0, 0 }, { 61, 0, 0, 0 },
        { 61, 0, 0, 0 }, { 56, 0, 0, 0 }, { 57,13, 0, 0 }, { 57,13, 0, 0 },
        { 57,13, 0, 0 }, { 59,13, 0, 0 }, { 39,13, 0, 0 }, { 34,16,60, 0 },
        { 45, 0, 0, 0 }, { 45, 0, 0, 0 }, { 12, 0, 0, 0 }, { 30, 0, 0, 0 },
        { 32, 0, 0, 0 }, {  2, 0, 0, 0 }, {  2, 0, 0, 0 }, { 61, 0, 0, 0 },
        { 38, 0, 0, 0 }, { 38, 0, 0, 0 }, { 38,13, 0, 0 }, { 57,13, 0, 0 },
        {  0,13, 0, 0 }, { 59,13, 0, 0 }, { 39,13, 0, 0 }, { 34,16,60, 0 },
        { 63, 0, 0, 0 }, { 63, 0, 0, 0 }, {  3, 0, 0, 0 }, { 32, 0, 0, 0 },
        { 58, 0, 0, 0 }, { 18, 0, 0, 0 }, { 18, 0, 0, 0 }, { 20, 0, 0, 0 },
        { 21, 0, 0, 0 }, { 21, 0, 0, 0 }, { 21,13, 0, 0 }, { 59,13, 0, 0 },
        { 39,13, 0, 0 }, { 39,13, 0, 0 }, {  7,13,13, 0 }, { 31,16,60, 0 },
        { 31, 0, 0, 0 }, {  3, 0, 0, 0 }, {  3, 0, 0, 0 }, { 33, 0, 0, 0 },
        { 58, 0, 0, 0 }, { 18, 0, 0, 0 }, { 18, 0, 0, 0 }, { 20, 0, 0, 0 },
        { 21, 0, 0, 0 }, { 21, 0, 0, 0 }, { 21,13, 0, 0 }, { 59,13, 0, 0 },
        {  7,13, 0, 0 }, {  7,13, 0, 0 }, {  7,13,13, 0 }, { 31,16,60, 0 },
        {  6, 0, 0, 0 }, {  6, 0, 0, 0 }, { 51, 0, 0, 0 }, { 51, 0, 0, 0 },
        { 53, 0, 0, 0 }, { 54, 0, 0, 0 }, { 54, 0, 0, 0 }, { 38, 0, 0, 0 },
        { 57,13, 0, 0 }, { 57,13, 0, 0 }, { 57,13, 0, 0 }, { 39,13, 0, 0 },
        {  7,13, 0, 0 }, {  7,13, 0, 0 }, { 42,60,13, 0 }, { 31,16,60, 0 },
        { 31, 0, 0, 0 }, { 31, 0, 0, 0 }, { 31, 0, 0, 0 }, { 31, 0, 0, 0 },
        { 31, 0, 0, 0 }, { 31, 0, 0, 0 }, { 31, 0, 0, 0 }, { 31, 0, 0, 0 },
        { 31, 0, 0, 0 }, { 31, 0, 0, 0 }, { 31, 0, 0, 0 }, { 31, 0, 0, 0 },
        { 31, 0, 0, 0 }, { 31, 0, 0, 0 }, { 31, 0, 0, 0 }, { 31, 0, 0, 0 },
        { 31, 0, 0, 0 }, { 51, 0, 0, 0 }, { 53, 0, 0, 0 }, { 53, 0, 0, 0 },
        { 54, 0, 0, 0 }, { 56, 0, 0, 0 }, { 56, 0, 0, 0 }, { 57,13, 0, 0 },
        { 59,13, 0, 0 }, { 59,13, 0, 0 }, { 59,13, 0, 0 }, {  7,13, 0, 0 },
        { 24,13, 0, 0 }, { 24,13, 0, 0 }, { 25,60,13, 0 }, { 31,16,60, 0 },
        { 31, 0, 0, 0 }, { 31, 0, 0, 0 }, { 31, 0, 0, 0 }, { 31, 0, 0, 0 },
        { 31, 0, 0, 0 }, { 31, 0, 0, 0 }, { 31, 0, 0, 0 }, { 31, 0, 0, 0 },
        { 31, 0, 0, 0 }, { 31, 0, 0, 0 }, { 31, 0, 0, 0 }, { 31, 0, 0, 0 },
        { 31, 0, 0, 0 }, { 31, 0, 0, 0 }, { 31, 0, 0, 0 }, { 31, 0, 0, 0 },
        { 31, 0, 0, 0 }, {  4, 0, 0, 0 }, {  4, 0, 0, 0 }, {  4, 0, 0, 0 },
        { 54, 0, 0, 0 }, { 21,13, 0, 0 }, { 21, 0, 0, 0 }, { 57,13, 0, 0 },
        { 39,13, 0, 0 }, { 39,13, 0, 0 }, { 39,13, 0, 0 }, {  7,13, 0, 0 },
        { 42,13,13, 0 }, { 42,13,13, 0 }, { 22,60,13, 0 }, { 31,16,60, 0 },
        { 31,16, 0, 0 }, { 31,16, 0, 0 }, { 31,16, 0, 0 }, { 31,16, 0, 0 },
        { 31,16, 0, 0 }, { 31,16, 0, 0 }, { 31,16, 0, 0 }, { 31,16, 0, 0 },
        { 31,16, 0, 0 }, { 31,16, 0, 0 }, { 31,16, 0, 0 }, { 31,16, 0, 0 },
        { 31,16, 0, 0 }, { 31,16, 0, 0 }, { 31,16, 0, 0 }, { 31,16, 0, 0 },
        { 31,16, 0, 0 }, {  5, 0, 0, 0 }, {  5, 0, 0, 0 }, {  5, 0, 0, 0 },
        {  5,13, 0, 0 }, { 23,13, 0, 0 }, { 23,13, 0, 0 }, { 59,13, 0, 0 },
        {  7,13, 0, 0 }, {  7,13, 0, 0 }, {  7,13,13, 0 }, { 42,13,13, 0 },
        { 22,60,13, 0 }, { 22,60,13, 0 }, { 28,60,13, 0 }, { 31,16,60, 0 },
        { 31,13, 0, 0 }, {  4,13, 0, 0 }, {  4,13, 0, 0 }, {  4,13, 0, 0 },
        {  5,13, 0, 0 }, { 23,13, 0, 0 }, { 23,13, 0, 0 }, { 39,13,13, 0 },
        { 24,60,13, 0 }, { 24,60,13, 0 }, { 24,60,13, 0 }, { 25,60,13, 0 },
        { 28,60,13, 0 }, { 28,60,13, 0 }, { 34,16,13, 0 }, { 31,16,60, 0 },
        { 31,16,13, 0 }, { 10,16,13, 0 }, { 10,16,13, 0 }, { 10,16,13, 0 },
        { 10,16,13, 0 }, { 10,16,60, 0 }, { 10,16,60, 0 }, { 28,16,60, 0 },
        { 34,16,60, 0 }, { 34,16,60, 0 }, { 34,16,60, 0 }, { 31,16,60, 0 },
        { 31,16,60, 0 }, { 31,16,60, 0 }, { 31,16,60, 0 }, { 19,37,60, 0 } } }
};

const struct lc3_ac_model lc3_spectrum_models[] = {

    { { {    0,    1 }, {    1,    1 }, {    2,  175 }, {  177,   48 },
        {  225,    1 }, {  226,    1 }, {  227,  109 }, {  336,   36 },
        {  372,  171 }, {  543,  109 }, {  652,   47 }, {  699,   20 },
        {  719,   49 }, {  768,   36 }, {  804,   20 }, {  824,   10 },
        {  834,  190 } } },

    { { {    0,   18 }, {   18,   26 }, {   44,   17 }, {   61,   10 },
        {   71,   27 }, {   98,   37 }, {  135,   24 }, {  159,   16 },
        {  175,   22 }, {  197,   32 }, {  229,   22 }, {  251,   14 },
        {  265,   17 }, {  282,   26 }, {  308,   20 }, {  328,   13 },
        {  341,  683 } } },

    { { {    0,   71 }, {   71,   92 }, {  163,   49 }, {  212,   25 },
        {  237,   81 }, {  318,  102 }, {  420,   61 }, {  481,   33 },
        {  514,   42 }, {  556,   57 }, {  613,   39 }, {  652,   23 },
        {  675,   22 }, {  697,   30 }, {  727,   22 }, {  749,   15 },
        {  764,  260 } } },

    { { {    0,  160 }, {  160,  130 }, {  290,   46 }, {  336,   18 },
        {  354,  121 }, {  475,  123 }, {  598,   55 }, {  653,   24 },
        {  677,   45 }, {  722,   55 }, {  777,   31 }, {  808,   15 },
        {  823,   19 }, {  842,   24 }, {  866,   15 }, {  881,    9 },
        {  890,  134 } } },

    { { {    0,   71 }, {   71,   73 }, {  144,   33 }, {  177,   18 },
        {  195,   71 }, {  266,   76 }, {  342,   43 }, {  385,   26 },
        {  411,   34 }, {  445,   44 }, {  489,   30 }, {  519,   20 },
        {  539,   20 }, {  559,   27 }, {  586,   21 }, {  607,   15 },
        {  622,  402 } } },

    { { {    0,   48 }, {   48,   60 }, {  108,   32 }, {  140,   19 },
        {  159,   58 }, {  217,   68 }, {  285,   42 }, {  327,   27 },
        {  354,   31 }, {  385,   42 }, {  427,   30 }, {  457,   21 },
        {  478,   19 }, {  497,   27 }, {  524,   21 }, {  545,   16 },
        {  561,  463 } } },

    { { {    0,  138 }, {  138,  109 }, {  247,   43 }, {  290,   18 },
        {  308,  111 }, {  419,  112 }, {  531,   53 }, {  584,   25 },
        {  609,   46 }, {  655,   55 }, {  710,   32 }, {  742,   17 },
        {  759,   21 }, {  780,   27 }, {  807,   18 }, {  825,   11 },
        {  836,  188 } } },

    { { {    0,   16 }, {   16,   24 }, {   40,   22 }, {   62,   17 },
        {   79,   24 }, {  103,   36 }, {  139,   31 }, {  170,   25 },
        {  195,   20 }, {  215,   30 }, {  245,   25 }, {  270,   20 },
        {  290,   15 }, {  305,   22 }, {  327,   19 }, {  346,   16 },
        {  362,  662 } } },

    { { {    0,  579 }, {  579,  150 }, {  729,   12 }, {  741,    2 },
        {  743,  154 }, {  897,   73 }, {  970,   10 }, {  980,    2 },
        {  982,   14 }, {  996,   11 }, { 1007,    3 }, { 1010,    1 },
        { 1011,    3 }, { 1014,    3 }, { 1017,    1 }, { 1018,    1 },
        { 1019,    5 } } },

    { { {    0,  398 }, {  398,  184 }, {  582,   25 }, {  607,    5 },
        {  612,  176 }, {  788,  114 }, {  902,   23 }, {  925,    6 },
        {  931,   25 }, {  956,   23 }, {  979,    8 }, {  987,    3 },
        {  990,    6 }, {  996,    6 }, { 1002,    3 }, { 1005,    2 },
        { 1007,   17 } } },

    { { {    0,   13 }, {   13,   21 }, {   34,   18 }, {   52,   11 },
        {   63,   20 }, {   83,   29 }, {  112,   22 }, {  134,   15 },
        {  149,   14 }, {  163,   20 }, {  183,   16 }, {  199,   12 },
        {  211,   10 }, {  221,   14 }, {  235,   12 }, {  247,   10 },
        {  257,  767 } } },

    { { {    0,  281 }, {  281,  183 }, {  464,   37 }, {  501,    9 },
        {  510,  171 }, {  681,  139 }, {  820,   37 }, {  857,   10 },
        {  867,   35 }, {  902,   36 }, {  938,   15 }, {  953,    6 },
        {  959,    9 }, {  968,   10 }, {  978,    6 }, {  984,    3 },
        {  987,   37 } } },

    { { {    0,  198 }, {  198,  164 }, {  362,   46 }, {  408,   13 },
        {  421,  154 }, {  575,  147 }, {  722,   51 }, {  773,   16 },
        {  789,   43 }, {  832,   49 }, {  881,   24 }, {  905,   10 },
        {  915,   13 }, {  928,   16 }, {  944,   10 }, {  954,    5 },
        {  959,   65 } } },

    { { {    0,    1 }, {    1,    1 }, {    2,   93 }, {   95,   44 },
        {  139,    1 }, {  140,    1 }, {  141,   72 }, {  213,   38 },
        {  251,   86 }, {  337,   70 }, {  407,   43 }, {  450,   25 },
        {  475,   40 }, {  515,   36 }, {  551,   25 }, {  576,   16 },
        {  592,  432 } } },

    { { {    0,  133 }, {  133,  141 }, {  274,   64 }, {  338,   28 },
        {  366,  117 }, {  483,  122 }, {  605,   59 }, {  664,   27 },
        {  691,   39 }, {  730,   48 }, {  778,   29 }, {  807,   15 },
        {  822,   15 }, {  837,   20 }, {  857,   13 }, {  870,    8 },
        {  878,  146 } } },

    { { {    0,  128 }, {  128,  125 }, {  253,   49 }, {  302,   18 },
        {  320,  123 }, {  443,  134 }, {  577,   59 }, {  636,   23 },
        {  659,   49 }, {  708,   59 }, {  767,   32 }, {  799,   15 },
        {  814,   19 }, {  833,   24 }, {  857,   15 }, {  872,    9 },
        {  881,  143 } } },

    { { {    0,    1 }, {    1,    1 }, {    2,   23 }, {   25,   17 },
        {   42,    1 }, {   43,    1 }, {   44,   23 }, {   67,   18 },
        {   85,   20 }, {  105,   21 }, {  126,   18 }, {  144,   15 },
        {  159,   15 }, {  174,   17 }, {  191,   14 }, {  205,   12 },
        {  217,  807 } } },

    { { {    0,   70 }, {   70,   96 }, {  166,   63 }, {  229,   38 },
        {  267,   89 }, {  356,  112 }, {  468,   65 }, {  533,   36 },
        {  569,   37 }, {  606,   47 }, {  653,   32 }, {  685,   20 },
        {  705,   17 }, {  722,   23 }, {  745,   17 }, {  762,   12 },
        {  774,  250 } } },

    { { {    0,   55 }, {   55,   75 }, {  130,   45 }, {  175,   25 },
        {  200,   68 }, {  268,   90 }, {  358,   58 }, {  416,   33 },
        {  449,   39 }, {  488,   54 }, {  542,   39 }, {  581,   25 },
        {  606,   22 }, {  628,   31 }, {  659,   24 }, {  683,   16 },
        {  699,  325 } } },

    { { {    0,    1 }, {    1,    2 }, {    3,    2 }, {    5,    2 },
        {    7,    2 }, {    9,    2 }, {   11,    2 }, {   13,    2 },
        {   15,    2 }, {   17,    2 }, {   19,    2 }, {   21,    2 },
        {   23,    2 }, {   25,    2 }, {   27,    2 }, {   29,    2 },
        {   31,  993 } } },

    { { {    0,   34 }, {   34,   51 }, {   85,   38 }, {  123,   24 },
        {  147,   49 }, {  196,   69 }, {  265,   52 }, {  317,   35 },
        {  352,   34 }, {  386,   47 }, {  433,   37 }, {  470,   27 },
        {  497,   21 }, {  518,   31 }, {  549,   25 }, {  574,   19 },
        {  593,  431 } } },

    { { {    0,   30 }, {   30,   43 }, {   73,   32 }, {  105,   22 },
        {  127,   43 }, {  170,   59 }, {  229,   45 }, {  274,   31 },
        {  305,   30 }, {  335,   42 }, {  377,   34 }, {  411,   25 },
        {  436,   19 }, {  455,   28 }, {  483,   23 }, {  506,   18 },
        {  524,  500 } } },

    { { {    0,    9 }, {    9,   15 }, {   24,   14 }, {   38,   13 },
        {   51,   14 }, {   65,   22 }, {   87,   21 }, {  108,   18 },
        {  126,   13 }, {  139,   20 }, {  159,   18 }, {  177,   16 },
        {  193,   11 }, {  204,   17 }, {  221,   15 }, {  236,   14 },
        {  250,  774 } } },

    { { {    0,   30 }, {   30,   44 }, {   74,   31 }, {  105,   20 },
        {  125,   41 }, {  166,   58 }, {  224,   42 }, {  266,   28 },
        {  294,   28 }, {  322,   39 }, {  361,   30 }, {  391,   22 },
        {  413,   18 }, {  431,   26 }, {  457,   21 }, {  478,   16 },
        {  494,  530 } } },

    { { {    0,   15 }, {   15,   23 }, {   38,   20 }, {   58,   15 },
        {   73,   22 }, {   95,   33 }, {  128,   28 }, {  156,   22 },
        {  178,   18 }, {  196,   26 }, {  222,   23 }, {  245,   18 },
        {  263,   13 }, {  276,   20 }, {  296,   18 }, {  314,   15 },
        {  329,  695 } } },

    { { {    0,   11 }, {   11,   17 }, {   28,   16 }, {   44,   13 },
        {   57,   17 }, {   74,   26 }, {  100,   23 }, {  123,   19 },
        {  142,   15 }, {  157,   22 }, {  179,   20 }, {  199,   17 },
        {  216,   12 }, {  228,   18 }, {  246,   16 }, {  262,   14 },
        {  276,  748 } } },

    { { {    0,  448 }, {  448,  171 }, {  619,   20 }, {  639,    4 },
        {  643,  178 }, {  821,  105 }, {  926,   18 }, {  944,    4 },
        {  948,   23 }, {  971,   20 }, {  991,    7 }, {  998,    2 },
        { 1000,    5 }, { 1005,    5 }, { 1010,    2 }, { 1012,    1 },
        { 1013,   11 } } },

    { { {    0,  332 }, {  332,  188 }, {  520,   29 }, {  549,    6 },
        {  555,  186 }, {  741,  133 }, {  874,   29 }, {  903,    7 },
        {  910,   30 }, {  940,   30 }, {  970,   11 }, {  981,    4 },
        {  985,    6 }, {  991,    7 }, {  998,    4 }, { 1002,    2 },
        { 1004,   20 } } },

    { { {    0,    8 }, {    8,   13 }, {   21,   13 }, {   34,   11 },
        {   45,   13 }, {   58,   20 }, {   78,   18 }, {   96,   16 },
        {  112,   12 }, {  124,   17 }, {  141,   16 }, {  157,   13 },
        {  170,   10 }, {  180,   14 }, {  194,   13 }, {  207,   12 },
        {  219,  805 } } },

    { { {    0,  239 }, {  239,  176 }, {  415,   42 }, {  457,   11 },
        {  468,  163 }, {  631,  145 }, {  776,   44 }, {  820,   13 },
        {  833,   39 }, {  872,   42 }, {  914,   19 }, {  933,    7 },
        {  940,   11 }, {  951,   13 }, {  964,    7 }, {  971,    4 },
        {  975,   49 } } },

    { { {    0,  165 }, {  165,  145 }, {  310,   49 }, {  359,   16 },
        {  375,  138 }, {  513,  139 }, {  652,   55 }, {  707,   20 },
        {  727,   47 }, {  774,   54 }, {  828,   28 }, {  856,   12 },
        {  868,   16 }, {  884,   20 }, {  904,   12 }, {  916,    7 },
        {  923,  101 } } },

    { { {    0,    3 }, {    3,    5 }, {    8,    5 }, {   13,    5 },
        {   18,    5 }, {   23,    7 }, {   30,    7 }, {   37,    7 },
        {   44,    4 }, {   48,    7 }, {   55,    7 }, {   62,    6 },
        {   68,    4 }, {   72,    6 }, {   78,    6 }, {   84,    6 },
        {   90,  934 } } },

    { { {    0,  115 }, {  115,  122 }, {  237,   52 }, {  289,   22 },
        {  311,  111 }, {  422,  125 }, {  547,   61 }, {  608,   27 },
        {  635,   45 }, {  680,   57 }, {  737,   34 }, {  771,   17 },
        {  788,   19 }, {  807,   25 }, {  832,   17 }, {  849,   10 },
        {  859,  165 } } },

    { { {    0,  107 }, {  107,  114 }, {  221,   51 }, {  272,   21 },
        {  293,  106 }, {  399,  122 }, {  521,   61 }, {  582,   28 },
        {  610,   46 }, {  656,   58 }, {  714,   35 }, {  749,   18 },
        {  767,   20 }, {  787,   26 }, {  813,   18 }, {  831,   11 },
        {  842,  182 } } },

    { { {    0,    6 }, {    6,   10 }, {   16,   10 }, {   26,    9 },
        {   35,   10 }, {   45,   15 }, {   60,   15 }, {   75,   14 },
        {   89,    9 }, {   98,   14 }, {  112,   13 }, {  125,   12 },
        {  137,    8 }, {  145,   12 }, {  157,   11 }, {  168,   10 },
        {  178,  846 } } },

    { { {    0,   72 }, {   72,   88 }, {  160,   50 }, {  210,   26 },
        {  236,   84 }, {  320,  102 }, {  422,   60 }, {  482,   32 },
        {  514,   41 }, {  555,   53 }, {  608,   36 }, {  644,   21 },
        {  665,   20 }, {  685,   27 }, {  712,   20 }, {  732,   13 },
        {  745,  279 } } },

    { { {    0,   45 }, {   45,   63 }, {  108,   45 }, {  153,   30 },
        {  183,   61 }, {  244,   83 }, {  327,   58 }, {  385,   36 },
        {  421,   34 }, {  455,   47 }, {  502,   34 }, {  536,   23 },
        {  559,   19 }, {  578,   27 }, {  605,   21 }, {  626,   15 },
        {  641,  383 } } },

    { { {    0,    1 }, {    1,    1 }, {    2,    7 }, {    9,    7 },
        {   16,    1 }, {   17,    1 }, {   18,    8 }, {   26,    8 },
        {   34,    6 }, {   40,    8 }, {   48,    7 }, {   55,    7 },
        {   62,    6 }, {   68,    7 }, {   75,    7 }, {   82,    6 },
        {   88,  936 } } },

    { { {    0,   29 }, {   29,   44 }, {   73,   35 }, {  108,   24 },
        {  132,   42 }, {  174,   62 }, {  236,   48 }, {  284,   34 },
        {  318,   30 }, {  348,   43 }, {  391,   35 }, {  426,   26 },
        {  452,   19 }, {  471,   29 }, {  500,   24 }, {  524,   19 },
        {  543,  481 } } },

    { { {    0,   20 }, {   20,   31 }, {   51,   25 }, {   76,   17 },
        {   93,   30 }, {  123,   43 }, {  166,   34 }, {  200,   25 },
        {  225,   22 }, {  247,   32 }, {  279,   26 }, {  305,   21 },
        {  326,   16 }, {  342,   23 }, {  365,   20 }, {  385,   16 },
        {  401,  623 } } },

    { { {    0,  742 }, {  742,  103 }, {  845,    5 }, {  850,    1 },
        {  851,  108 }, {  959,   38 }, {  997,    4 }, { 1001,    1 },
        { 1002,    7 }, { 1009,    5 }, { 1014,    2 }, { 1016,    1 },
        { 1017,    2 }, { 1019,    1 }, { 1020,    1 }, { 1021,    1 },
        { 1022,    2 } } },

    { { {    0,   42 }, {   42,   52 }, {   94,   27 }, {  121,   16 },
        {  137,   49 }, {  186,   58 }, {  244,   36 }, {  280,   23 },
        {  303,   27 }, {  330,   36 }, {  366,   26 }, {  392,   18 },
        {  410,   17 }, {  427,   24 }, {  451,   19 }, {  470,   14 },
        {  484,  540 } } },

    { { {    0,   13 }, {   13,   20 }, {   33,   18 }, {   51,   15 },
        {   66,   19 }, {   85,   29 }, {  114,   26 }, {  140,   21 },
        {  161,   17 }, {  178,   25 }, {  203,   22 }, {  225,   18 },
        {  243,   13 }, {  256,   19 }, {  275,   17 }, {  292,   15 },
        {  307,  717 } } },

    { { {    0,  501 }, {  501,  169 }, {  670,   19 }, {  689,    4 },
        {  693,  155 }, {  848,   88 }, {  936,   16 }, {  952,    4 },
        {  956,   19 }, {  975,   16 }, {  991,    6 }, {  997,    2 },
        {  999,    5 }, { 1004,    4 }, { 1008,    2 }, { 1010,    1 },
        { 1011,   13 } } },

    { { {    0,  445 }, {  445,  136 }, {  581,   22 }, {  603,    6 },
        {  609,  158 }, {  767,   98 }, {  865,   23 }, {  888,    7 },
        {  895,   31 }, {  926,   28 }, {  954,   10 }, {  964,    4 },
        {  968,    9 }, {  977,    9 }, {  986,    5 }, {  991,    2 },
        {  993,   31 } } },

    { { {    0,  285 }, {  285,  157 }, {  442,   37 }, {  479,   10 },
        {  489,  161 }, {  650,  129 }, {  779,   39 }, {  818,   12 },
        {  830,   40 }, {  870,   42 }, {  912,   18 }, {  930,    7 },
        {  937,   12 }, {  949,   14 }, {  963,    8 }, {  971,    4 },
        {  975,   49 } } },

    { { {    0,  349 }, {  349,  179 }, {  528,   33 }, {  561,    8 },
        {  569,  162 }, {  731,  121 }, {  852,   31 }, {  883,    9 },
        {  892,   31 }, {  923,   30 }, {  953,   12 }, {  965,    5 },
        {  970,    8 }, {  978,    9 }, {  987,    5 }, {  992,    2 },
        {  994,   30 } } },

    { { {    0,  199 }, {  199,  156 }, {  355,   47 }, {  402,   15 },
        {  417,  146 }, {  563,  137 }, {  700,   50 }, {  750,   17 },
        {  767,   44 }, {  811,   49 }, {  860,   24 }, {  884,   10 },
        {  894,   15 }, {  909,   17 }, {  926,   10 }, {  936,    6 },
        {  942,   82 } } },

    { { {    0,  141 }, {  141,  134 }, {  275,   50 }, {  325,   18 },
        {  343,  128 }, {  471,  135 }, {  606,   58 }, {  664,   22 },
        {  686,   48 }, {  734,   57 }, {  791,   31 }, {  822,   14 },
        {  836,   18 }, {  854,   23 }, {  877,   14 }, {  891,    8 },
        {  899,  125 } } },

    { { {    0,  243 }, {  243,  194 }, {  437,   56 }, {  493,   17 },
        {  510,  139 }, {  649,  126 }, {  775,   45 }, {  820,   16 },
        {  836,   33 }, {  869,   36 }, {  905,   18 }, {  923,    8 },
        {  931,   10 }, {  941,   12 }, {  953,    7 }, {  960,    4 },
        {  964,   60 } } },

    { { {    0,   91 }, {   91,  106 }, {  197,   51 }, {  248,   23 },
        {  271,   99 }, {  370,  117 }, {  487,   63 }, {  550,   30 },
        {  580,   45 }, {  625,   59 }, {  684,   37 }, {  721,   20 },
        {  741,   20 }, {  761,   27 }, {  788,   19 }, {  807,   12 },
        {  819,  205 } } },

    { { {    0,  107 }, {  107,   94 }, {  201,   41 }, {  242,   20 },
        {  262,   92 }, {  354,   97 }, {  451,   52 }, {  503,   28 },
        {  531,   42 }, {  573,   53 }, {  626,   34 }, {  660,   20 },
        {  680,   21 }, {  701,   29 }, {  730,   21 }, {  751,   14 },
        {  765,  259 } } },

    { { {    0,  168 }, {  168,  171 }, {  339,   68 }, {  407,   25 },
        {  432,  121 }, {  553,  123 }, {  676,   55 }, {  731,   24 },
        {  755,   34 }, {  789,   41 }, {  830,   24 }, {  854,   12 },
        {  866,   13 }, {  879,   16 }, {  895,   11 }, {  906,    6 },
        {  912,  112 } } },

    { { {    0,   67 }, {   67,   80 }, {  147,   44 }, {  191,   23 },
        {  214,   76 }, {  290,   94 }, {  384,   57 }, {  441,   31 },
        {  472,   41 }, {  513,   54 }, {  567,   37 }, {  604,   23 },
        {  627,   21 }, {  648,   30 }, {  678,   22 }, {  700,   15 },
        {  715,  309 } } },

    { { {    0,   46 }, {   46,   63 }, {  109,   39 }, {  148,   23 },
        {  171,   58 }, {  229,   78 }, {  307,   52 }, {  359,   32 },
        {  391,   36 }, {  427,   49 }, {  476,   37 }, {  513,   24 },
        {  537,   21 }, {  558,   30 }, {  588,   24 }, {  612,   17 },
        {  629,  395 } } },

    { { {    0,  848 }, {  848,   70 }, {  918,    2 }, {  920,    1 },
        {  921,   75 }, {  996,   16 }, { 1012,    1 }, { 1013,    1 },
        { 1014,    2 }, { 1016,    1 }, { 1017,    1 }, { 1018,    1 },
        { 1019,    1 }, { 1020,    1 }, { 1021,    1 }, { 1022,    1 },
        { 1023,    1 } } },

    { { {    0,   36 }, {   36,   52 }, {   88,   35 }, {  123,   22 },
        {  145,   48 }, {  193,   67 }, {  260,   48 }, {  308,   32 },
        {  340,   32 }, {  372,   45 }, {  417,   35 }, {  452,   24 },
        {  476,   20 }, {  496,   29 }, {  525,   23 }, {  548,   17 },
        {  565,  459 } } },

    { { {    0,   24 }, {   24,   37 }, {   61,   29 }, {   90,   20 },
        {  110,   35 }, {  145,   51 }, {  196,   41 }, {  237,   29 },
        {  266,   26 }, {  292,   38 }, {  330,   31 }, {  361,   24 },
        {  385,   18 }, {  403,   27 }, {  430,   23 }, {  453,   18 },
        {  471,  553 } } },

    { { {    0,   85 }, {   85,   97 }, {  182,   48 }, {  230,   23 },
        {  253,   91 }, {  344,  110 }, {  454,   61 }, {  515,   30 },
        {  545,   45 }, {  590,   58 }, {  648,   37 }, {  685,   21 },
        {  706,   21 }, {  727,   29 }, {  756,   20 }, {  776,   13 },
        {  789,  235 } } },

    { { {    0,   22 }, {   22,   33 }, {   55,   27 }, {   82,   20 },
        {  102,   33 }, {  135,   48 }, {  183,   39 }, {  222,   30 },
        {  252,   26 }, {  278,   37 }, {  315,   30 }, {  345,   23 },
        {  368,   17 }, {  385,   25 }, {  410,   21 }, {  431,   17 },
        {  448,  576 } } },

    { { {    0,    1 }, {    1,    1 }, {    2,   54 }, {   56,   33 },
        {   89,    1 }, {   90,    1 }, {   91,   49 }, {  140,   32 },
        {  172,   49 }, {  221,   47 }, {  268,   35 }, {  303,   25 },
        {  328,   30 }, {  358,   30 }, {  388,   24 }, {  412,   18 },
        {  430,  594 } } },

    { { {    0,   45 }, {   45,   64 }, {  109,   43 }, {  152,   25 },
        {  177,   62 }, {  239,   81 }, {  320,   56 }, {  376,   35 },
        {  411,   37 }, {  448,   51 }, {  499,   38 }, {  537,   26 },
        {  563,   22 }, {  585,   31 }, {  616,   24 }, {  640,   18 },
        {  658,  366 } } },

    { { {    0,  247 }, {  247,  148 }, {  395,   38 }, {  433,   12 },
        {  445,  154 }, {  599,  130 }, {  729,   42 }, {  771,   14 },
        {  785,   44 }, {  829,   46 }, {  875,   21 }, {  896,    9 },
        {  905,   15 }, {  920,   17 }, {  937,    9 }, {  946,    5 },
        {  951,   73 } } },

    { { {    0,  231 }, {  231,  136 }, {  367,   41 }, {  408,   15 },
        {  423,  134 }, {  557,  119 }, {  676,   47 }, {  723,   19 },
        {  742,   44 }, {  786,   49 }, {  835,   25 }, {  860,   12 },
        {  872,   17 }, {  889,   20 }, {  909,   12 }, {  921,    7 },
        {  928,   96 } } }

};

const uint16_t lc3_spectrum_bits[][17] = {

    { 20480, 20480,  5220,  9042, 20480, 20480,  6619,  9892,
       5289,  6619,  9105, 11629,  8982,  9892, 11629, 13677,  4977 },

    { 11940, 10854, 12109, 13677, 10742,  9812, 11090, 12288,
      11348, 10240, 11348, 12683, 12109, 10854, 11629, 12902,  1197 },

    {  7886,  7120,  8982, 10970,  7496,  6815,  8334, 10150,
       9437,  8535,  9656, 11216, 11348, 10431, 11348, 12479,  4051 },

    {  5485,  6099,  9168, 11940,  6311,  6262,  8640, 11090,
       9233,  8640, 10334, 12479, 11781, 11090, 12479, 13988,  6009 },

    {  7886,  7804, 10150, 11940,  7886,  7685,  9368, 10854,
      10061,  9300, 10431, 11629, 11629, 10742, 11485, 12479,  2763 },

    {  9042,  8383, 10240, 11781,  8483,  8013,  9437, 10742,
      10334,  9437, 10431, 11485, 11781, 10742, 11485, 12288,  2346 },

    {  5922,  6619,  9368, 11940,  6566,  6539,  8750, 10970,
       9168,  8640, 10240, 12109, 11485, 10742, 11940, 13396,  5009 },

    { 12288, 11090, 11348, 12109, 11090,  9892, 10334, 10970,
      11629, 10431, 10970, 11629, 12479, 11348, 11781, 12288,  1289 },

    {  1685,  5676, 13138, 18432,  5598,  7804, 13677, 18432,
      12683, 13396, 17234, 20480, 17234, 17234, 20480, 20480, 15725 },

    {  2793,  5072, 10970, 15725,  5204,  6487, 11216, 15186,
      10970, 11216, 14336, 17234, 15186, 15186, 17234, 18432, 12109 },

    { 12902, 11485, 11940, 13396, 11629, 10531, 11348, 12479,
      12683, 11629, 12288, 13138, 13677, 12683, 13138, 13677,   854 },

    {  3821,  5088,  9812, 13988,  5289,  5901,  9812, 13677,
       9976,  9892, 12479, 15186, 13988, 13677, 15186, 17234,  9812 },

    {  4856,  5412,  9168, 12902,  5598,  5736,  8863, 12288,
       9368,  8982, 11090, 13677, 12902, 12288, 13677, 15725,  8147 },

    { 20480, 20480,  7088,  9300, 20480, 20480,  7844,  9733,
       7320,  7928,  9368, 10970,  9581,  9892, 10970, 12288,  2550 },

    {  6031,  5859,  8192, 10635,  6410,  6286,  8433, 10742,
       9656,  9042, 10531, 12479, 12479, 11629, 12902, 14336,  5756 },

    {  6144,  6215,  8982, 11940,  6262,  6009,  8433, 11216,
       8982,  8433, 10240, 12479, 11781, 11090, 12479, 13988,  5817 },

    { 20480, 20480, 11216, 12109, 20480, 20480, 11216, 11940,
      11629, 11485, 11940, 12479, 12479, 12109, 12683, 13138,   704 },

    {  7928,  6994,  8239,  9733,  7218,  6539,  8147,  9892,
       9812,  9105, 10240, 11629, 12109, 11216, 12109, 13138,  4167 },

    {  8640,  7724,  9233, 10970,  8013,  7185,  8483, 10150,
       9656,  8694,  9656, 10970, 11348, 10334, 11090, 12288,  3391 },

    { 20480, 18432, 18432, 18432, 18432, 18432, 18432, 18432,
      18432, 18432, 18432, 18432, 18432, 18432, 18432, 18432,    91 },

    { 10061,  8863,  9733, 11090,  8982,  7970,  8806,  9976,
      10061,  9105,  9812, 10742, 11485, 10334, 10970, 11781,  2557 },

    { 10431,  9368, 10240, 11348,  9368,  8433,  9233, 10334,
      10431,  9437, 10061, 10970, 11781, 10635, 11216, 11940,  2119 },

    { 13988, 12479, 12683, 12902, 12683, 11348, 11485, 11940,
      12902, 11629, 11940, 12288, 13396, 12109, 12479, 12683,   828 },

    { 10431,  9300, 10334, 11629,  9508,  8483,  9437, 10635,
      10635,  9656, 10431, 11348, 11940, 10854, 11485, 12288,  1946 },

    { 12479, 11216, 11629, 12479, 11348, 10150, 10635, 11348,
      11940, 10854, 11216, 11940, 12902, 11629, 11940, 12479,  1146 },

    { 13396, 12109, 12288, 12902, 12109, 10854, 11216, 11781,
      12479, 11348, 11629, 12109, 13138, 11940, 12288, 12683,   928 },

    {  2443,  5289, 11629, 16384,  5170,  6730, 11940, 16384,
      11216, 11629, 14731, 18432, 15725, 15725, 18432, 20480, 13396 },

    {  3328,  5009, 10531, 15186,  5040,  6031, 10531, 14731,
      10431, 10431, 13396, 16384, 15186, 14731, 16384, 18432, 11629 },

    { 14336, 12902, 12902, 13396, 12902, 11629, 11940, 12288,
      13138, 12109, 12288, 12902, 13677, 12683, 12902, 13138,   711 },

    {  4300,  5204,  9437, 13396,  5430,  5776,  9300, 12902,
       9656,  9437, 11781, 14731, 13396, 12902, 14731, 16384,  8982 },

    {  5394,  5776,  8982, 12288,  5922,  5901,  8640, 11629,
       9105,  8694, 10635, 13138, 12288, 11629, 13138, 14731,  6844 },

    { 17234, 15725, 15725, 15725, 15725, 14731, 14731, 14731,
      16384, 14731, 14731, 15186, 16384, 15186, 15186, 15186,   272 },

    {  6461,  6286,  8806, 11348,  6566,  6215,  8334, 10742,
       9233,  8535, 10061, 12109, 11781, 10970, 12109, 13677,  5394 },

    {  6674,  6487,  8863, 11485,  6702,  6286,  8334, 10635,
       9168,  8483,  9976, 11940, 11629, 10854, 11940, 13396,  5105 },

    { 15186, 13677, 13677, 13988, 13677, 12479, 12479, 12683,
      13988, 12683, 12902, 13138, 14336, 13138, 13396, 13677,   565 },

    {  7844,  7252,  8922, 10854,  7389,  6815,  8383, 10240,
       9508,  8750,  9892, 11485, 11629, 10742, 11629, 12902,  3842 },

    {  9233,  8239,  9233, 10431,  8334,  7424,  8483,  9892,
      10061,  9105, 10061, 11216, 11781, 10742, 11485, 12479,  2906 },

    { 20480, 20480, 14731, 14731, 20480, 20480, 14336, 14336,
      15186, 14336, 14731, 14731, 15186, 14731, 14731, 15186,   266 },

    { 10531,  9300,  9976, 11090,  9437,  8286,  9042, 10061,
      10431,  9368,  9976, 10854, 11781, 10531, 11090, 11781,  2233 },

    { 11629, 10334, 10970, 12109, 10431,  9368, 10061, 10970,
      11348, 10240, 10854, 11485, 12288, 11216, 11629, 12288,  1469 },

    {   952,  6787, 15725, 20480,  6646,  9733, 16384, 20480,
      14731, 15725, 18432, 20480, 18432, 20480, 20480, 20480, 18432 },

    {  9437,  8806, 10742, 12288,  8982,  8483,  9892, 11216,
      10742,  9892, 10854, 11940, 12109, 11090, 11781, 12683,  1891 },

    { 12902, 11629, 11940, 12479, 11781, 10531, 10854, 11485,
      12109, 10970, 11348, 11940, 12902, 11781, 12109, 12479,  1054 },

    {  2113,  5323, 11781, 16384,  5579,  7252, 12288, 16384,
      11781, 12288, 15186, 18432, 15725, 16384, 18432, 20480, 12902 },

    {  2463,  5965, 11348, 15186,  5522,  6934, 11216, 14731,
      10334, 10635, 13677, 16384, 13988, 13988, 15725, 18432, 10334 },

    {  3779,  5541,  9812, 13677,  5467,  6122,  9656, 13138,
       9581,  9437, 11940, 14731, 13138, 12683, 14336, 16384,  8982 },

    {  3181,  5154, 10150, 14336,  5448,  6311, 10334, 13988,
      10334, 10431, 13138, 15725, 14336, 13988, 15725, 18432, 10431 },

    {  4841,  5560,  9105, 12479,  5756,  5944,  8922, 12109,
       9300,  8982, 11090, 13677, 12479, 12109, 13677, 15186,  7460 },

    {  5859,  6009,  8922, 11940,  6144,  5987,  8483, 11348,
       9042,  8535, 10334, 12683, 11940, 11216, 12683, 14336,  6215 },

    {  4250,  4916,  8587, 12109,  5901,  6191,  9233, 12288,
      10150,  9892, 11940, 14336, 13677, 13138, 14731, 16384,  8383 },

    {  7153,  6702,  8863, 11216,  6904,  6410,  8239, 10431,
       9233,  8433,  9812, 11629, 11629, 10742, 11781, 13138,  4753 },

    {  6674,  7057,  9508, 11629,  7120,  6964,  8806, 10635,
       9437,  8750, 10061, 11629, 11485, 10531, 11485, 12683,  4062 },

    {  5341,  5289,  8013, 10970,  6311,  6262,  8640, 11090,
      10061,  9508, 11090, 13138, 12902, 12288, 13396, 15186,  6539 },

    {  8057,  7533,  9300, 11216,  7685,  7057,  8535, 10334,
       9508,  8694,  9812, 11216, 11485, 10431, 11348, 12479,  3541 },

    {  9168,  8239,  9656, 11216,  8483,  7608,  8806, 10240,
       9892,  8982,  9812, 11090, 11485, 10431, 11090, 12109,  2815 },

    {   558,  7928, 18432, 20480,  7724, 12288, 20480, 20480,
      18432, 20480, 20480, 20480, 20480, 20480, 20480, 20480, 20480 },

    {  9892,  8806,  9976, 11348,  9042,  8057,  9042, 10240,
      10240,  9233,  9976, 11090, 11629, 10531, 11216, 12109,  2371 },

    { 11090,  9812, 10531, 11629,  9976,  8863,  9508, 10531,
      10854,  9733, 10334, 11090, 11940, 10742, 11216, 11940,  1821 },

    {  7354,  6964,  9042, 11216,  7153,  6592,  8334, 10431,
       9233,  8483,  9812, 11485, 11485, 10531, 11629, 12902,  4349 },

    { 11348, 10150, 10742, 11629, 10150,  9042,  9656, 10431,
      10854,  9812, 10431, 11216, 12109, 10970, 11485, 12109,  1700 },

    { 20480, 20480,  8694, 10150, 20480, 20480,  8982, 10240,
       8982,  9105,  9976, 10970, 10431, 10431, 11090, 11940,  1610 },

    {  9233,  8192,  9368, 10970,  8286,  7496,  8587,  9976,
       9812,  8863,  9733, 10854, 11348, 10334, 11090, 11940,  3040 },

    {  4202,  5716,  9733, 13138,  5598,  6099,  9437, 12683,
       9300,  9168, 11485, 13988, 12479, 12109, 13988, 15725,  7804 },

    {  4400,  5965,  9508, 12479,  6009,  6360,  9105, 11781,
       9300,  8982, 10970, 13138, 12109, 11629, 13138, 14731,  6994 }

};
