!{\src2tex{textfont=tt}}
!!****f* ABINIT/mklocl_realspace
!! NAME
!! mklocl_realspace
!!
!! FUNCTION
!! This method is equivalent to mklocl_recipspace except that
!! it uses real space pseudo-potentials. It is usefull for isolated
!! systems. Then the option 3 and 4 are not available for this
!! implementation.
!!
!! Optionally compute :
!!  option=1 : local ionic potential throughout unit cell
!!  option=2 : contribution of local ionic potential to E gradient wrt xred
!!
!! COPYRIGHT
!! Copyright (C) 1998-2007 ABINIT group (DCA, XG, GMR)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!  dtset <type(dataset_type)>=all input variables in this dataset
!!  mgfft=maximum size of 1D FFTs
!!  mpi_enreg=informations about MPI parallelization
!!  natom=number of atoms in unit cell.
!!  nattyp(ntypat)=number of atoms of each type in cell.
!!  nfft=(effective) number of FFT grid points (for this processor)
!!  ngfft(18)=contain all needed information about 3D FFT, see ~abinit/doc/input_variables/vargs.htm#ngfft
!!  nspden=number of spin-density components
!!  ntypat=number of types of atoms.
!!  option= (see above)
!!  ph1d(2,3*(2*mgfft+1)*natom)=1-dim structure factor phase information.
!!  psps <type(pseudopotential_type)>=variables related to pseudopotentials
!!  rhog(2,nfft)=electron density rho(G) (electrons/$\textrm{Bohr}^3$)
!!  rhor(nfft,nspden)=electron density in electrons/bohr**3.
!!    (needed if option==2 or if option==4)
!!  rmet(3,3)=real space metric (bohr**2)
!!  rprimd(3,3)=dimensional primitive translations in real space (bohr)
!!  ucvol=unit cell volume ($\textrm{Bohr}^3$).
!!  xred(3,natom)=reduced dimensionless atomic coordinates
!!
!! OUTPUT
!!  (if option==1) vpsp(nfft)=local crystal pseudopotential in real space.
!!  (if option==2) grtn(3,natom)=grads of Etot wrt tn. These gradients are in
!!                 reduced coordinates. Multiply them by rprimd to get
!!                 gradients in cartesian coordinates.
!!
!! SIDE EFFECTS
!!
!!
!! PARENTS
!!      mklocl
!!
!! CHILDREN
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine mklocl_realspace(dtset, grtn, mgfft, mpi_enreg, natom, nattyp, nfft, ngfft, &
                          & nspden, ntypat, option, ph1d, psps, rhog, rhor, rmet, &
                          & rprimd, ucvol, vpsp, xred)

 use defs_basis
  use defs_datatypes

!This section has been created automatically by the script Abilint (TD). Do not modify these by hand.
#ifdef HAVE_FORTRAN_INTERFACES
 use interfaces_01manage_mpi
 use interfaces_12ffts
 use interfaces_12geometry
#endif
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: mgfft,natom,nfft,nspden,ntypat,option
 real(dp),intent(in) :: ucvol
 type(MPI_type),intent(inout) :: mpi_enreg
 type(dataset_type),intent(in) :: dtset
 type(pseudopotential_type),intent(in) :: psps
!arrays
 integer,intent(in) :: nattyp(ntypat),ngfft(18)
 real(dp),intent(in) :: ph1d(2,3*(2*mgfft+1)*natom),rhog(2,nfft)
 real(dp),intent(in) :: rhor(nfft,nspden),rmet(3,3),rprimd(3,3)
 real(dp),intent(inout) :: xred(3,natom)
 real(dp),intent(out) :: grtn(3,natom),vpsp(nfft)

!Local variables-------------------------------
  !testing variables
!scalars
 integer,parameter :: nStep=2
 integer :: countParSeconde,i1,i2,i3,ia,ia1,ia2,igeo,ii,ind,itypat,ix,iy,iz,jj
 integer :: kk,me_fft,n1,n2,n3,n_interpol,nproc_fft,tpsStart,tpsStop
 real(dp),parameter :: min_rho_value=1.0d-12
 real(dp) :: aa,bb,cc,dd,delta,deltaV,dr,dr2div6,invdr,r,vol_interpol,x,y,z
 logical,parameter :: customRho=.false.,finiteDiff=.false.,testing=.false.
 logical :: doIt
 character(len=500) :: message
!arrays
 integer :: coordAtom(3,natom),ngfft_interpol(18)
 real(dp) :: coord(3),coordXYZ(3),refValue(3),tsec(2)
 real(dp),allocatable :: coordCart_interpol(:,:),coordRed_interpol(:,:)
 real(dp),allocatable :: grad_sum(:,:),gridcart(:,:),gridred(:,:)
 real(dp),allocatable :: grtn_cart_interpol(:,:),grtn_diff(:,:)
 real(dp),allocatable :: rhog_interpol(:,:),rhog_testing(:,:),rhor_interpol(:)
 real(dp),allocatable :: rhor_testing(:),rhor_work(:),xcart(:,:)

! *************************************************************************

  !Keep track of total time spent in mklocl
  if(option==2)then
    call timab(72,1,tsec)
  end if

  if (testing) then
    call system_clock(count_rate = countParSeconde)
    call system_clock(tpsStart, count_rate = countParSeconde)
  end if

  n1 = ngfft(1)
  n2 = ngfft(2)
  n3 = ngfft(3)
  me_fft = ngfft(11)
  nproc_fft = ngfft(10)

  if (nspden /= 1) then
    write(message, '(a,a,a,a)' ) ch10,&
      & ' mklocl_realspace : ERROR - ',ch10,&
      & '  real space computation is only possible without spin density'
    call wrtout(06,message,'COLL')
    call leave_new('COLL')
  end if

  ! Store xcart for each atom
  allocate(xcart(3, natom))
  call xredxcart(natom, 1, rprimd, xcart, xred)
  ! Store cartesian coordinates for each grid points
  allocate(gridred(3, nfft))
  allocate(gridcart(3, nfft))
  ii = 0
  do i3 = 1, n3, 1
    coord(3) = real(i3 - 1, dp) / real(n3, dp)
    do i2 = 1, n2, 1
      coord(2) = real(i2 - 1, dp) / real(n2, dp)
      do i1 = 1, n1, 1
        ii = ii + 1
        coord(1) = real(i1 - 1, dp) / real(n1, dp)
        gridred(:, ii) = coord(:)
      end do
    end do
  end do
  call xredxcart(nfft, 1, rprimd, gridcart, gridred)
  deallocate(gridred)

  ! dr is the r step in the sampling psps%vlspl
  dr = psps%qgrid_vl(2)
  invdr = 1._dp / dr
  dr2div6 = dr * dr / 6._dp

  if (option == 1) then
    ! Set 0 in vpsp before summing
    vpsp(:) = 0._dp
  else if (option == 2) then
    ! Allocate array to store cartesian gradient computed with
    ! an interpolation of rhor
    allocate(grtn_cart_interpol(3, natom))
    grtn_cart_interpol(:, :) = 0._dp

    n_interpol = nStep ** 3
    allocate(coordRed_interpol(3, nStep ** 3))
    allocate(coordCart_interpol(3, nStep ** 3))

    if (testing .and. customRho) then
      ! Use a custom rho instead of the self-consistent one.
      allocate(rhor_testing(nfft))
      allocate(rhog_testing(2, nfft))
    end if

    allocate(rhor_interpol(nfft * n_interpol))
    allocate(rhor_work(nfft * n_interpol))
    allocate(rhog_interpol(2, nfft * n_interpol))

    if (testing .and. customRho) then
      ! Testing only, changing rho with a centered gaussian
      do ii = 1, nfft, 1
        ! using the position of the first atom as center.
        r = (gridcart(1, ii) - xcart(1, 1)) ** 2 + &
          & (gridcart(2, ii) - xcart(2, 1)) ** 2 + &
          & (gridcart(3, ii) - xcart(3, 1)) ** 2
        rhor_testing(ii) = exp(-r/4._dp)
      end do
      ! Testing only, compute rhog_testing from rhor_testing
      call fourdp(1, rhog_testing, rhor_testing, -1, mpi_enreg, nfft, ngfft, 0)
    end if

    ! Compute the interpolation of rho, using a fourrier transform
    rhog_interpol(:, :) = 0._dp
    ii = 0
    do i3 = 1, n3, 1
      if (i3 <= n3 / 2) then
        iz = i3
      else
        iz = n3 * nStep - n3 + i3
      end if
      do i2 = 1, n2, 1
        if (i2 <= n2 / 2) then
          iy = i2
        else
          iy = n2 * nStep - n2 + i2
        end if
        do i1 = 1, n1, 1
          ii = ii + 1
          if (i1 <= n1 / 2) then
            ix = i1
          else
            ix = n1 * nStep - n1 + i1
          end if
          jj = (iz - 1) * n2 * n1 * nStep ** 2 + (iy - 1) * n3 * nStep + ix
          if (testing .and. customRho) then
            rhog_interpol(:, jj) = rhog_testing(:, ii)
          else
            rhog_interpol(:, jj) = rhog(:, ii)
          end if
        end do
      end do
    end do

    ! Compute the interpolation of rho from the Fourier transformation
    ngfft_interpol(:) = ngfft(:)
    ngfft_interpol(1:3) = (/ n1 * nStep, n2 * nStep, n3 * nStep /)
    ngfft_interpol(4:6) = (/ n1 * nStep + 1, n2 * nStep + 1, n3 * nStep /)
    call fourdp(1, rhog_interpol, rhor_work, 1, mpi_enreg, nfft * n_interpol, ngfft_interpol, 0)

    ! Reorder rhor_interpol to be able to read it linearly
    jj = 0
    do i3 = 1, n3, 1
      do i2 = 1, n2, 1
        do i1 = 1, n1, 1
          do iz = 1, nStep, 1
            do iy = 1, nStep, 1
              do ix = 1, nStep, 1
                jj = jj + 1
                kk = ((i3 - 1) * nStep + iz - 1) ! z coordinate in the interpolated grid
                kk = kk * n1 * n2 * nStep ** 2
                kk = kk + ((i2 - 1) * nStep + iy - 1) * n1 * nStep ! adding y coordinate
                kk = kk + (i1 - 1) * nStep + ix ! adding x coordinate
                rhor_interpol(jj) = rhor_work(kk)
              end do
            end do
          end do
        end do
      end do
    end do
    deallocate(rhor_work)

    ! Compute grid access in the interpolated volume
    ii = 0
    do iz = 1, nStep, 1
      z = real(iz - 1, dp) / real(nStep, dp)
      do iy = 1, nStep, 1
        y = real(iy - 1, dp) / real(nStep, dp)
        do ix = 1, nStep, 1
          x = real(ix - 1, dp) / real(nStep, dp)
          ii = ii + 1
          coordRed_interpol(:, ii) = (/ x, y, z /)
          ! Assuming orthogonal box (should be change later)
          coordCart_interpol(:, ii) = (/ x * rprimd(1, 1) / real(n1, dp), &
                                       & y * rprimd(2, 2) / real(n2, dp), &
                                       & z * rprimd(3, 3) / real(n3, dp) /)
        end do
      end do
    end do

    vol_interpol = 1._dp / real(nStep, dp) ** 3
    ! Compute the coordinates (integer) of each atom and deduce
    ! the max extens of the integral summation.
    do ia = 1, natom, 1
      coordAtom(1, ia) = int(xred(1, ia) * n1) + 1
      coordAtom(2, ia) = int(xred(2, ia) * n2) + 1
      coordAtom(3, ia) = int(xred(3, ia) * n3) + 1
    end do
  end if

  if (testing .and. option == 2) then
    call system_clock(tpsStop, count_rate = countParSeconde)
    write(*,*) "Tps : ", real(tpsStop - tpsStart) / real(countParSeconde)
  end if

  ia1=1
  do itypat = 1, ntypat, 1
    ! ia1,ia2 sets range of loop over atoms:
    ia2 = ia1 + nattyp(itypat) - 1

    do ii = 1, nfft, 1
      do ia = ia1, ia2, 1
        if (option == 1) then
          ! Compute the potential
          ! r is the distance between grid point and atom
          r = sqrt((gridcart(1, ii) - xcart(1, ia)) ** 2 + &
                 & (gridcart(2, ii) - xcart(2, ia)) ** 2 + &
                 & (gridcart(3, ii) - xcart(3, ia)) ** 2)

          ! Coefficients needed to compute the spline.
          jj = int(r * invdr) + 1
          if (jj > psps%mqgrid_vl - 2) then
            write(message, '(a,a,a,a,a,a,i0,a,i0,a,a)' ) ch10,&
              & ' mklocl_realspace : ERROR - ',ch10,&
              & '  pseudo-potential local part sampling is not wide enough', ch10, &
              & '  want to access position ', jj, ' whereas mqgrid_vl = ', psps%mqgrid_vl, ch10, &
              & '  Action : no idea, contact developpers...'
            call wrtout(06,message,'COLL')
            call leave_new('COLL')
          end if
          delta = r - psps%qgrid_vl(jj)
          bb = delta * invdr
          aa = 1._dp - bb
          cc = aa * (aa ** 2 - 1._dp) * dr2div6
          dd = bb * (bb ** 2 - 1._dp) * dr2div6

          ! compute V(r) from the spline, jj and jj + 1 is braketting r in
          ! the sampling
          deltaV = aa * psps%vlspl(jj, 1, itypat) + bb * psps%vlspl(jj + 1, 1, itypat) + &
                 & cc * psps%vlspl(jj, 2, itypat) + dd * psps%vlspl(jj + 1, 2, itypat)
          ! Add on grid point ii the contribution of atom ia
          vpsp(ii) = vpsp(ii) + deltaV
        else if (option == 2) then
          ! Compute the forces, as gradient of energy (V(r).rho(r))

          ! Testing only - reference points
          if (.false.) then
            ! r is the distance between grid point and atom
            r = sqrt((gridcart(1, ii) - xcart(1, ia)) ** 2 + &
                    & (gridcart(2, ii) - xcart(2, ia)) ** 2 + &
                    & (gridcart(3, ii) - xcart(3, ia)) ** 2)

            ! Coefficients needed to compute the spline.
            jj = int(r * invdr) + 1
            delta = r - psps%qgrid_vl(jj)
            bb = delta * invdr
            aa = 1._dp - bb
            cc = aa * (aa ** 2 - 1._dp) * dr2div6
            dd = bb * (bb ** 2 - 1._dp) * dr2div6

            ! When mesh position is on a node, forces are null.
            if (r /= 0._dp) then
              ! This value deltaV is the first derivative of V(r) taken at r.
              deltaV = aa * psps%dvlspl(jj, 1, itypat) + bb * psps%dvlspl(jj + 1, 1, itypat) + &
                      & cc * psps%dvlspl(jj, 2, itypat) + dd * psps%dvlspl(jj + 1, 2, itypat)
              ! We multiply by rho(r) to have an energy.
              deltaV = deltaV * rhor(ii, 1) / r
              refValue(:) = - deltaV * (gridcart(:, ii) - xcart(:, ia))
              grtn_cart_interpol(:, ia) = grtn_cart_interpol(:, ia) + refValue(:)
            end if
          end if

          ! Compute the interpolation for the point ii
          ind = (ii - 1) * n_interpol
          do kk = 1, n_interpol, 1
            ind = ind + 1

            if (rhor_interpol(ind) > min_rho_value) then
              ! Assume orthogonal box...
              coordXYZ(1) = gridcart(1, ii) - xcart(1, ia) + coordCart_interpol(1, kk)
              coordXYZ(2) = gridcart(2, ii) - xcart(2, ia) + coordCart_interpol(2, kk)
              coordXYZ(3) = gridcart(3, ii) - xcart(3, ia) + coordCart_interpol(3, kk)
              r = coordXYZ(1) ** 2 + coordXYZ(2) ** 2 + coordXYZ(3) ** 2

              if (r /= 0._dp) then
                r = sqrt(r)
                ! Coefficients needed to compute the spline.
                jj = int(r * invdr) + 1
                delta = r - psps%qgrid_vl(jj)
                bb = delta * invdr
                aa = 1._dp - bb
                cc = aa * (aa ** 2 - 1._dp) * dr2div6
                dd = bb * (bb ** 2 - 1._dp) * dr2div6
                deltaV = aa * psps%dvlspl(jj, 1, itypat) + &
                        & bb * psps%dvlspl(jj + 1, 1, itypat) + &
                        & cc * psps%dvlspl(jj, 2, itypat) + &
                        & dd * psps%dvlspl(jj + 1, 2, itypat)
                deltaV = deltaV * rhor_interpol(ind) / r
                grtn_cart_interpol(1, ia) = grtn_cart_interpol(1, ia) - deltaV * coordXYZ(1)
                grtn_cart_interpol(2, ia) = grtn_cart_interpol(2, ia) - deltaV * coordXYZ(2)
                grtn_cart_interpol(3, ia) = grtn_cart_interpol(3, ia) - deltaV * coordXYZ(3)
                !do igeo = 1, 3, 1
                !  grtn_cart_interpol(igeo, ia) = grtn_cart_interpol(igeo, ia) - deltaV * coordXYZ(igeo)
                !end do
              end if
            end if
          end do

          !=============
          ! Testing only
          !=============
          ! use of finite differences
          if (finiteDiff) then
            do igeo = 1, 3, 1
              coord(:) = 0._dp
              coord(igeo) = dr / 2.0_dp
              r = sqrt((gridcart(1, ii) - xcart(1, ia) + coord(1)) ** 2 + &
                     & (gridcart(2, ii) - xcart(2, ia) + coord(2)) ** 2 + &
                     & (gridcart(3, ii) - xcart(3, ia) + coord(3)) ** 2)

              ! Coefficients needed to compute the spline.
              jj = int(r * invdr) + 1
              delta = r - psps%qgrid_vl(jj)
              bb = delta * invdr
              aa = 1._dp - bb
              cc = aa * (aa ** 2 - 1._dp) * dr2div6
              dd = bb * (bb ** 2 - 1._dp) * dr2div6

              deltaV = aa * psps%vlspl(jj, 1, itypat) + bb * psps%vlspl(jj + 1, 1, itypat) + &
                     & cc * psps%vlspl(jj, 2, itypat) + dd * psps%vlspl(jj + 1, 2, itypat)


              coord(:) = 0._dp
              coord(igeo) = -dr / 2.0_dp
              r = sqrt((gridcart(1, ii) - xcart(1, ia) + coord(1)) ** 2 + &
                     & (gridcart(2, ii) - xcart(2, ia) + coord(2)) ** 2 + &
                     & (gridcart(3, ii) - xcart(3, ia) + coord(3)) ** 2)

              ! Coefficients needed to compute the spline.
              jj = int(r * invdr) + 1
              delta = r - psps%qgrid_vl(jj)
              bb = delta * invdr
              aa = 1._dp - bb
              cc = aa * (aa ** 2 - 1._dp) * dr2div6
              dd = bb * (bb ** 2 - 1._dp) * dr2div6

              deltaV = deltaV - (aa * psps%vlspl(jj, 1, itypat) + &
                               & bb * psps%vlspl(jj + 1, 1, itypat) + &
                               & cc * psps%vlspl(jj, 2, itypat) + &
                               & dd * psps%vlspl(jj + 1, 2, itypat))
              grtn_diff(igeo, ia) = grtn_diff(igeo, ia) - deltaV * rhor(ii, 1) / dr
            end do
          end if
          !=============
          ! Testing only
          !=============

        end if
      end do
      ! End loop over atoms of type itypat
    end do
    ! End loop over real space grid points

    ia1 = ia2 + 1
  end do
  ! End loop over type of atoms

  deallocate(xcart)
  deallocate(gridcart)

  if(option==2)then
    ! multiply the forces by the volume of a single box mesh.
    grtn_cart_interpol(:, :) = grtn_cart_interpol(:, :) * &
      & ucvol / real(n1 * n2 * n3, dp) * vol_interpol
    ! Transform cartesian forces to reduce coordinates
    do ia = 1, natom, 1
      do igeo = 1, 3, 1
        grtn(igeo, ia) = rprimd(1, igeo) * grtn_cart_interpol(1, ia) + &
                       & rprimd(2, igeo) * grtn_cart_interpol(2, ia) + &
                       & rprimd(3, igeo) * grtn_cart_interpol(3, ia)
      end do
    end do
    deallocate(rhor_interpol)
    deallocate(rhog_interpol)
    deallocate(coordRed_interpol)
    deallocate(coordCart_interpol)
    if (testing .and. customRho) then
      deallocate(rhor_testing)
      deallocate(rhog_testing)
    end if

    call timab(72,2,tsec)

    if (testing) then
      call system_clock(tpsStop, count_rate = countParSeconde)
      write(*,*) "Tps : ", real(tpsStop - tpsStart) / real(countParSeconde)
      write(*,*) grtn_cart_interpol
      stop
    end if

  end if
end subroutine mklocl_realspace
!!***
