#file: technique.py

from bbqsql import settings
from bbqsql import utilities

import gevent
from gevent.event import AsyncResult,Event
from gevent.coros import Semaphore
from gevent.queue import Queue
from gevent.pool import Pool

from time import time
from copy import copy

__all__ = ['BooleanBlindTechnique','FrequencyTechnique']


#########################
# Binary Search Technique
#########################

class BlindCharacter(object):
    def __init__(self,row_index,char_index,queue,row_die):
        '''
            :row_index  - what row this character is a part of (for rendering our Query)
            :char_index - what character in the row is this (for rendering our Query)
            :queue      - what queue will we push to. this queue will receive tuples in the
                          form of:
                             item=(self.row_index,self.char_index,self.char_val,comparator,asr)
            :row_die    - gevent.event.AsyncResult that gets fired when the row needs to die. the
                          value passed to this ASR's set() should be the char_index in this row
                          after which all Character()s need to kill themselves
        '''
        #row_die is an AsyncResult. We link our die method to and store the 
        #event so the die method can know if it should die (based on char_index emmitted by row_die)
        self.row_die = row_die
        self.row_die.rawlink(self._die_callback)
        #run_gl will store the greenlet running the run() method
        self.run_gl = None
        self.q = queue

        self.row_index = row_index
        self.char_index = char_index
        self.char_val = settings.CHARSET[0]

        #these flags are used in computing the __str__, __repr__, and __eq__
        self.error = False
        self.working = False
        self.done = False
    
    @utilities.debug 
    def run(self):
        #make note of the current greenlet
        self.run_gl = gevent.getcurrent()

        low = 0
        high = settings.CHARSET_LEN
        self.working = True        
        #binary search unless we hit an error
        while not self.error and self.working:
            mid = (low+high)//2
            self.char_val = settings.CHARSET[mid]

            if low >= high:
                self.error = True
                self.row_die.set((self.char_index,AsyncResult()))
                break

            if self._test("<"):
                #print "data[%d][%d] > %s (%d)" % (self.row_index,self.char_index,settings.CHARSET[mid],ord(settings.CHARSET[mid]))
                high = mid
            elif self._test(">"):
                #print "data[%d][%d] < %s (%d)" % (self.row_index,self.char_index,settings.CHARSET[mid],ord(settings.CHARSET[mid]))
                low = mid + 1
            elif low < settings.CHARSET_LEN and self._test("="):
                #print "data[%d][%d] = %s (%d)" % (self.row_index,self.char_index,settings.CHARSET[mid],ord(settings.CHARSET[mid]))
                self.working = False
                break
            else:
                #if there isn't a value for the character we are working on, that means we went past the end of the row.
                #we set error and kill characters after us in the row.
                self.error = True
                self.row_die.set((self.char_index,AsyncResult()))
                break

            gevent.sleep(0)
            
        self.done = True
        self.working = False

        #clear the note regarding the running greenlet
        self.run_gl = None
    
    @utilities.debug 
    def get_status(self):
        if self.error: return "error"
        if self.working: return "working"
        if self.done: return "success"
        return "unknown"
    
    def _die_callback(self,event):
        #we do the next_event because the first event might be first for the last character. 
        #this way we can fire the die event multiple times if necessary
        die_index,next_event = self.row_die.get()
        if die_index  < self.char_index and self.run_gl:
            self.run_gl.kill(block=False)
            self.error = True
            self.done = True
            self.working = False
        else:
            self.row_die = next_event
            self.row_die.rawlink(self._die_callback)
    
    def _test(self,comparator):
        asr = AsyncResult()
        self.q.put(item=(self.row_index,self.char_index,self.char_val,comparator,asr))
        res = asr.get()
        return res

    def __eq__(self,y):
        if y == "error":
            return self.error
            
        if y == "working":
            return self.working and (not self.error)
        
        if y == "success":
            return self.done and (not self.error)
        
        if y.hasattr('char_val'):
            return self.char_val == y.char_val
        
        return self.char_val == y
        
    def __ne__(self,y):
        return not self.__eq__(y)

    def __str__(self):
        # if error or not started yet return ''
        if self.error or (not self.working and not self.done): return ""
        # else return char_val
        return self.char_val

    def __repr__(self):
        # if error or not started yet return ''
        if self.error or (not self.working and not self.done): return ""
        # else return char_val
        return self.char_val
    
    def __hash__(self):
        # objects that override __eq__ cannot be hashed (cannot be added to a lot of structures like set()....). 
        return id(self)


class BooleanBlindTechnique:
    def __init__(self, query, requester):
        self.query = query
        self.requester = requester
        self.rungl = None

    def _reset(self):
        '''
        reset all the variables used for keeping track of internal state
        '''
        #an list of Character()s 
        self.results = []
        #an list of strings
        self.str_results = []
        #character generators take care of building the Character objects. we need one per row
        self.char_gens = []
        #a queue for communications between Character()s and request_makers
        self.q = Queue()
        #"threads" that run the Character()s
        self.character_pool = Pool(self.concurrency)
        #"threads" that make requests
        self.request_makers = [gevent.spawn(self._request_maker) for i in range(self.concurrency)]
        #fire this event when shutting down
        self.shutting_down = Event()
        #do we need to add more rows?
        self.need_more_rows = True
        #use this as a lock to know when not to mess with self.results        
        self.results_lock = Semaphore(1)
        #request_count is the number of requests made on the current run
        self.request_count = 0
        #failure_count is the number of requests made on the current run
        self.failure_count = 0

    def _request_maker(self):
        '''
        this runs in a gevent "thread". It is a worker
        '''
        #keep going until we shut down the technique
        while not self.shutting_down.is_set():
            #pull the info needed to make a request from the queue
            row_index,char_index,char_val,comparator,char_asyncresult = self.q.get()

            #build out our query object
            query = copy(self.query)
            query.set_option('row_index',str(row_index))
            query.set_option('char_index',str(char_index))
            query.set_option('char_val',str(ord(char_val)))
            query.set_option('comparator',comparator)
            query_string = query.render()

            self.request_count += 1

            count = 0
            response = None
            while response == None:
                try:
                    response = self.requester.make_request(query_string)
                except utilities.SendRequestFailed:
                    self.failure_count += 1
                    response = None
                    gevent.sleep(.01 * 2 ** count)                    
                    if count == 10: raise SendRequestFailed('cant request')
                count += 1

            char_asyncresult.set(response)

    def _character_generator(self,row_index):
        '''
        creates a Character object for us. this generator is useful just because it keeps track of the char_index
        '''
        char_index = 1
        row_die_event = AsyncResult()
        while not self.shutting_down.is_set():
            c = BlindCharacter(\
                row_index   = row_index,\
                char_index  = char_index,\
                queue       = self.q,\
                row_die     = row_die_event)
            char_index += 1
            #fire off the Character within our Pool.
            self.character_pool.spawn(c.run)
            yield c

    def _adjust_row_lengths(self):
        ''' 
        if a row is full of "success", but we havent reached the end yet (the last elt isnt "error")
        then we need to increase the row_len.
        '''
        while not self.shutting_down.is_set():
            self.results_lock.acquire()

            if self.row_len is not None:
                unused_threads = self.concurrency - reduce(lambda x,row: x + row.count('working'),self.results,0)
                rows_working = len(filter(lambda row: 'working' in row,self.results))
                if rows_working == 0:
                    add_to_rows = self.row_len
                else:
                    add_to_rows = unused_threads//rows_working
                    add_to_rows = [add_to_rows,1][add_to_rows==0]
            else:
                add_to_rows = 1

            for row_index in range(len(self.results)):
                #if the row isn't finished or hasn't been started yet, we add Character()s to the row
                if 'error' not in self.results[row_index]:
                    self.results[row_index] += [self.char_gens[row_index].next() for i in range(add_to_rows)]
            self.results_lock.release()
            gevent.sleep(.3)

    def _add_rows(self):
        '''
        look at how many gevent "threads" are being used and add more rows to correct this
        '''
        if self.row_len is not None:
            rows_to_work_on = self.concurrency // self.row_len
        else:
            rows_to_work_on = self.concurrency
        rows_to_work_on = [rows_to_work_on,1][rows_to_work_on == 0]

        row_index = 0

        # keep adding new rows until we dont need any more
        while self.need_more_rows:
            working_rows = len(filter(lambda row: 'working' in row,self.results))
            for row in range(rows_to_work_on - working_rows):
                self.char_gens.append(self._character_generator(row_index))
                self.results.append([])
                row_index += 1

            gevent.sleep(.3)
            self.need_more_rows = not(len(self.results) and filter(lambda row: len(row) and row[0] == 'error',self.results))
        
        # delete any extra rows.
        while not self.shutting_down.is_set():
            self.results_lock.acquire()
            # delete any rows that shouldn't have been added in the first place
            errored = filter(lambda ri: len(self.results[ri]) and self.results[ri][0] == 'error',range(len(self.results)))
            if errored:
                end = min(errored)
                for ri in xrange(len(self.results)-1,end-1,-1):
                    del(self.results[ri])

            self.results_lock.release()    
            #if there aren't going to be any more rows in need of deletion we can stop this nonsense
            if self.results and self.results[-1][0] == 'success':
                break
            gevent.sleep(.3)

    def _keep_going(self):
        '''
        Look at the results gathered so far and determine if we should keep going. we want to keep going until we have an empty row
        '''
        # chill out until we don't need any more rows
        while self.need_more_rows:
            gevent.sleep(1)

        # chill out untill all the rows have finished working
        while filter(lambda row:'error' not in row or 'working' in row[:row.index('error')],self.results):
            gevent.sleep(.5)
        
        # call it quits
        self.shutting_down.set()

    def _run(self):
        self.kg_gl = gevent.spawn(self._keep_going)
        self.ar_gl = gevent.spawn(self._add_rows)
        self.arl_gl = gevent.spawn(self._adjust_row_lengths)

        self.kg_gl.join()
        self.ar_gl.join()
        self.arl_gl.join()
    
        self.character_pool.join()
        gevent.killall(self.request_makers)
        gevent.joinall(self.request_makers)

    @utilities.debug 
    def run(self,row_len=None,concurrency=20):
        '''
        run the exploit. returns the data retreived.
            :concurrency    how many gevent "threads" to use. This is useful for throttling the attack.
            :row_len        An estimated starting point for the length of rows. This will get adjusted as the attack goes on.
        '''
        self.run_start_time = time()

        self.row_len = row_len
        self.concurrency = concurrency

        #start fresh
        self._reset()

        self.rungl = gevent.spawn(self._run)
        
        return self.rungl

    @utilities.debug 
    def get_results(self,color=False):
        if not color:
            return filter(lambda row: row != '',[''.join([str(x) for x in row]) for row in self.results])
        
        rval = []
        running_status = "unknown"

        for row in self.results:
            if len(row):
                    running_status = "unknown"
                    row_string = ""
                    for c in row:
                        cstatus = c.get_status()
                        if cstatus != running_status:
                            row_string += settings.COLORS[cstatus]
                            running_status = cstatus
                        row_string += str(c)
                    rval.append(row_string + settings.COLORS['endc'])
        return rval
            

        #return filter(lambda row: row != '',[''.join([settings.COLORS[x.get_status()] + str(x) + settings.COLORS['endc'] for x in row]) for row in self.results])        

    @utilities.debug 
    def get_status(self):
        status = ""
        status += "requests: %d\t" % self.request_count
        status += "failures: %d\t" % self.failure_count
        status += "rows: %d\t" % reduce(lambda x,row: ('success' in row)+x,self.results,0)
        status += "working threads: %d\t" %  reduce(lambda x,row: x + row.count('working'),self.results,0)
        
        chars = reduce(lambda x,row: row.count('success') + x,self.results,0)
        status += "chars: %d\t" % chars

        if self.run_start_time:
            run_time = time() - self.run_start_time
            status += "time: %f\t" % run_time
            status += "char/sec: %f\t" % (chars/run_time)

        if chars: rc = float(self.request_count) / chars
        else: rc = 0.0
        status += "req/char: %f\t" % rc

        return status


###########################
# Frequency Based Technique
###########################

diagraphs_english = {'\n': ['t', 's', 'a', 'h', 'w', '\n', 'c', 'b', 'f', 'o', 'i', 'm', 'p', 'd', '"', 'l', 'r', 'e', 'n', 'g', 'T', 'y', 'u', 'M', 'S', 'I', 'A', 'v', 'H', 'k', 'B', 'E', 'W', '&', 'P', 'q', 'C', 'j', 'F', 'D', 'G', 'L', 'R', "'", 'J', 'O', 'N', 'Y', 'K', 'V', ' ', '(', 'U', ',', '.', 'Z', '-', 'Q', '_', ';', '1', '2', '!', '3', '4', '6', '?', '[', '\t', '\r', '#', '$', '%', ')', '*', '+', '/', '0', '5', '7', '8', '9', ':', '<', '=', '>', '@', 'X', '\\', ']', '^', '`', 'x', 'z', '{', '|', '}', '~'], '!': [' ', '&', '"', "'", '\n', '-', ')', 'A', 'H', 'T', 'B', 'I', 'S', 'W', ',', 'D', 'O', 'R', '_', '!', '>', 'F', 'N', 'P', 'U', 'V', 'Y', '\t', '\r', '#', '$', '%', '(', '*', '+', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '?', '@', 'C', 'E', 'G', 'J', 'K', 'L', 'M', 'Q', 'X', 'Z', '[', '\\', ']', '^', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], ' ': ['t', 'a', 'h', 's', 'w', 'o', 'i', 'b', 'm', 'f', 'c', 'l', 'd', 'p', 'n', 'y', 'I', 'r', 'g', 'e', 'T', 'M', 'u', ' ', 'S', 'A', '-', 'k', 'H', 'v', 'B', '&', 'W', 'L', 'q', 'P', 'j', 'E', 'D', 'J', 'G', "'", 'C', '"', 'R', 'Y', 'O', 'N', 'F', 'K', 'V', '(', '_', 'U', '.', '\n', 'Z', '1', 'Q', 'z', '4', '3', '2', '7', '!', '5', '8', '6', '?', '\t', '\r', '#', '$', '%', ')', '*', '+', ',', '/', '0', '9', ':', ';', '<', '=', '>', '@', 'X', '[', '\\', ']', '^', '`', 'x', '{', '|', '}', '~'], '#': ['1', '\t', '\n', '\r', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '"': ['\n', ' ', 'I', 'W', 'Y', 'T', 'A', 'S', 'M', 'D', 'H', 'N', 'O', 't', 'L', 'B', 'i', 'G', 'E', "'", 'w', 'F', 'P', 'a', 'C', 'y', 'b', '_', 'm', 's', '-', 'J', 'V', 'h', 'R', 'U', 'o', 'e', 'f', 'n', '?', 'K', 'Q', 'g', ';', 'd', 'k', 'l', 'r', '\t', '\r', '!', '"', '#', '$', '%', '&', '(', ')', '*', '+', ',', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', '<', '=', '>', '@', 'X', 'Z', '[', '\\', ']', '^', '`', 'c', 'j', 'p', 'q', 'u', 'v', 'x', 'z', '{', '|', '}', '~'], "'": ['s', 't', ' ', 'l', 'v', "'", 'm', 'r', 'd', 'I', 'W', 'Y', 'e', 'T', 'A', '\n', 'u', 'N', 'H', 'O', 'c', 'S', 'D', 'n', 'M', 'B', 'a', 'C', 'P', 'L', ',', 'y', 'G', '&', '-', 'i', 'w', 'o', 'E', 'R', 'b', 'F', 'h', 'p', 'J', '1', 'g', '.', 'V', '"', 'f', 'Q', ';', '?', 'K', 'U', '!', ')', '(', '>', 'j', 'q', '\t', '\r', '#', '$', '%', '*', '+', '/', '0', '2', '3', '4', '5', '6', '7', '8', '9', ':', '<', '=', '@', 'X', 'Z', '[', '\\', ']', '^', '_', '`', 'k', 'x', 'z', '{', '|', '}', '~'], '&': ['q', 'A', 'a', 'o', 'b', '#', 'n', 'c', 'l', 's', 'u', '\t', '\n', '\r', ' ', '!', '"', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'm', 'p', 'r', 't', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], ')': [' ', ',', '\n', '.', ';', '&', '-', ':', 'S', '\t', '\r', '!', '"', '#', '$', '%', "'", '(', ')', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '(': ['a', 'i', 'w', 'h', 't', 's', 'I', 'f', 'M', 'o', 'p', 'm', 'n', 'L', 'T', 'c', 'N', 'P', 'V', 'b', 'l', 'y', 'A', 'C', 'B', 'E', 'D', 'H', 'O', 'S', '_', 'e', 'd', 'r', 'v', '\t', '\n', '\r', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'F', 'G', 'J', 'K', 'Q', 'R', 'U', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '`', 'g', 'j', 'k', 'q', 'u', 'x', 'z', '{', '|', '}', '~'], '*': ["'", '\t', '\n', '\r', ' ', '!', '"', '#', '$', '%', '&', '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '-': [' ', '-', 'a', 'b', 's', 't', 'c', 'm', 'w', 'l', 'h', 'r', 'p', 'd', 'f', 'e', 'g', 'o', 'i', 'n', '&', 'u', '\n', 'k', 'I', 'y', '"', "'", 'j', 'v', 'M', 'O', 'S', 'C', 'E', 'A', 'H', 'T', 'N', 'q', '!', '?', 'B', 'D', 'G', 'F', 'W', 'Z', '1', '>', 'L', 'P', '\t', '\r', '#', '$', '%', '(', ')', '*', '+', ',', '.', '/', '0', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '@', 'J', 'K', 'Q', 'R', 'U', 'V', 'X', 'Y', '[', '\\', ']', '^', '_', '`', 'x', 'z', '{', '|', '}', '~'], ',': [' ', '\n', '&', "'", '"', '0', 'I', 'A', 't', 'w', 'y', ',', 'B', 'F', 'd', 's', 'E', 'O', 'W', 'Y', 'b', 'h', 'l', 'o', '\t', '\r', '!', '#', '$', '%', '(', ')', '*', '+', '-', '.', '/', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'C', 'D', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'X', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'c', 'e', 'f', 'g', 'i', 'j', 'k', 'm', 'n', 'p', 'q', 'r', 'u', 'v', 'x', 'z', '{', '|', '}', '~'], '.': [' ', '&', '\n', "'", '"', 'T', 'A', 'S', 'B', 'H', 'I', 'M', 'W', '>', 'P', 'N', 'O', '.', ',', 'R', 'Y', 'J', 'D', 'F', 'L', ')', 'K', 'C', 'V', 'U', 'E', '-', 'G', '2', '?', 'p', '*', '1', 'Q', 'h', '\t', '\r', '!', '#', '$', '%', '(', '+', '/', '0', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '@', 'X', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '1': ['9', '8', '4', '6', '5', '7', ' ', '.', '3', '\n', '1', '0', '\t', '\r', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '/', '2', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '0': ['0', ' ', ',', '\n', '2', '5', 't', '6', ';', '\t', '\r', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '.', '/', '1', '3', '4', '7', '8', '9', ':', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '3': ['0', ' ', ',', '.', '4', '5', '\t', '\n', '\r', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '1', '2', '3', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '2': [' ', ',', '0', '.', '5', '9', '\t', '\n', '\r', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '1', '2', '3', '4', '6', '7', '8', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '5': [' ', ',', '-', '.', '1', '0', '5', 't', ':', '\t', '\n', '\r', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '/', '2', '3', '4', '6', '7', '8', '9', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '4': ['0', ' ', '7', '.', '6', ';', '\n', ',', '\t', '\r', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '1', '2', '3', '4', '5', '8', '9', ':', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '7': [',', '7', ';', ' ', '.', '0', '2', '5', '6', '\t', '\n', '\r', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '1', '3', '4', '8', '9', ':', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '6': [',', ' ', ';', '9', '0', '\t', '\n', '\r', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '.', '/', '1', '2', '3', '4', '5', '6', '7', '8', ':', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '9': ['1', ' ', '2', '0', ',', '\t', '\n', '\r', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '.', '/', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '8': ['7', ',', '0', ' ', '.', '3', 't', '8', '\t', '\n', '\r', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '1', '2', '4', '5', '6', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], ';': [' ', '&', 'I', 'W', 'T', 'Y', 'A', 'O', 'N', 'B', 'M', 'H', '\n', 'S', 'D', 't', 'L', 'a', 'P', 'i', 'C', 'G', 'w', 'y', 'b', 'R', 's', 'F', 'J', 'h', 'E', 'K', '.', 'f', ';', ',', 'V', 'd', 'c', 'm', 'n', 'l', 'p', 'g', '>', 'e', 'r', 'U', 'k', 'u', 'o', 'Q', "'", '-', 'v', 'j', ')', '4', 'q', '!', '(', '2', 'Z', '_', '\t', '\r', '"', '#', '$', '%', '*', '+', '/', '0', '1', '3', '5', '6', '7', '8', '9', ':', '<', '=', '?', '@', 'X', '[', '\\', ']', '^', '`', 'x', 'z', '{', '|', '}', '~'], ':': [' ', '&', '\n', "'", 'I', '-', 'T', 'D', 'E', 'G', 'H', 'K', 'u', '\t', '\r', '!', '"', '#', '$', '%', '(', ')', '*', '+', ',', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'F', 'J', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '?': [' ', '&', '"', "'", '\n', '-', 'T', 'S', 'I', 'W', 'A', 'B', 'D', 'F', 'M', 'N', 'P', '_', ')', 'G', 'H', 'J', 'L', 'O', 'R', 'Y', 'd', '\t', '\r', '!', '#', '$', '%', '(', '*', '+', ',', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'C', 'E', 'K', 'Q', 'U', 'V', 'X', 'Z', '[', '\\', ']', '^', '`', 'a', 'b', 'c', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '>': ['\n', '\t', '\r', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'A': ['n', 'l', ' ', 'u', 't', 's', 'h', 'r', 'f', 'm', 'y', 'g', 'N', 'L', 'p', 'R', 'b', '\n', 'T', 'c', 'd', 'S', 'w', 'V', 'D', '.', 'M', 'B', 'K', 'P', 'C', 'G', 'F', 'U', 'W', "'", 'I', 'Y', ',', '1', 'a', 'i', 'j', 'v', 'z', '\t', '\r', '!', '"', '#', '$', '%', '&', '(', ')', '*', '+', '-', '/', '0', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'E', 'H', 'J', 'O', 'Q', 'X', 'Z', '[', '\\', ']', '^', '_', '`', 'e', 'k', 'o', 'q', 'x', '{', '|', '}', '~'], 'C': ['o', 'a', 'h', 'e', 'l', 'r', 'A', 'O', 'E', 'i', 'u', ' ', 'I', '3', 'T', 'H', 'R', '2', 'y', '\n', '&', ',', '.', 'K', 'L', 'S', 'U', 'Y', '\t', '\r', '!', '"', '#', '$', '%', "'", '(', ')', '*', '+', '-', '/', '0', '1', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'B', 'C', 'D', 'F', 'G', 'J', 'M', 'N', 'P', 'Q', 'V', 'W', 'X', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'j', 'k', 'm', 'n', 'p', 'q', 's', 't', 'v', 'w', 'x', 'z', '{', '|', '}', '~'], 'B': ['u', 'a', 'e', 'o', 'r', 'i', 'l', 'y', 'E', 'L', 'U', '.', 'S', ' ', 'O', 'R', 'Y', ',', '\t', '\n', '\r', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'F', 'G', 'H', 'I', 'J', 'K', 'M', 'N', 'P', 'Q', 'T', 'V', 'W', 'X', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'm', 'n', 'p', 'q', 's', 't', 'v', 'w', 'x', 'z', '{', '|', '}', '~'], 'E': ['m', 'n', 'v', ' ', 'l', 'u', 'R', 'a', 'S', 'D', 'L', 'N', 'E', 'h', 'x', 'A', 'V', '.', 'g', 'M', 'd', '!', ',', 'i', 'I', 't', 'T', 'Y', 's', '\n', '?', 'C', 'r', 'f', 'O', 'W', 'X', 'B', 'G', 'F', 'K', 'P', 'c', 'q', 'p', 'y', '\t', '\r', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '@', 'H', 'J', 'Q', 'U', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'e', 'j', 'k', 'o', 'w', 'z', '{', '|', '}', '~'], 'D': ['o', 'e', 'a', 'i', 'r', ' ', 'u', 'O', 'E', 'I', '.', 'N', ',', 'A', 'S', '\n', "'", 'D', 'R', 'U', 'y', 'L', '\t', '\r', '!', '"', '#', '$', '%', '&', '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'B', 'C', 'F', 'G', 'H', 'J', 'K', 'M', 'P', 'Q', 'T', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 's', 't', 'v', 'w', 'x', 'z', '{', '|', '}', '~'], 'G': ['l', 'o', 'e', 'r', 'u', 'i', 'a', '.', ' ', 'E', 'H', '!', ',', 'h', 'A', 'I', 'O', 'G', 'N', 'S', 'R', 'U', '\t', '\n', '\r', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'B', 'C', 'D', 'F', 'J', 'K', 'L', 'M', 'P', 'Q', 'T', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'j', 'k', 'm', 'n', 'p', 'q', 's', 't', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'F': ['r', 'o', 'a', 'i', 'l', 'e', 'u', 'E', 'O', 'I', ' ', 'A', 'R', 'U', 'T', 'F', '\n', '.', '\t', '\r', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'B', 'C', 'D', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'P', 'Q', 'S', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'm', 'n', 'p', 'q', 's', 't', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'I': [' ', 't', "'", 'n', '\n', 'f', 's', 'N', '?', 'S', ',', 'T', 'd', 'r', 'w', 'c', 'm', 'E', 'D', 'V', 'L', 'R', '.', 'l', 'C', 'G', '_', 'k', 'O', 'a', 'g', 'h', 'F', '!', 'A', 'B', 'M', 'b', 'v', '-', 'I', 'P', 'o', ';', 'K', 'Q', 'X', 'e', '\t', '\r', '"', '#', '$', '%', '&', '(', ')', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', '<', '=', '>', '@', 'H', 'J', 'U', 'W', 'Y', 'Z', '[', '\\', ']', '^', '`', 'i', 'j', 'p', 'q', 'u', 'x', 'y', 'z', '{', '|', '}', '~'], 'H': ['e', 'a', 'o', 'i', 'E', 'u', 'A', 'I', 'O', 'T', ' ', 'y', 'R', 'm', 'Y', '.', 'Q', 'U', '\t', '\n', '\r', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'B', 'C', 'D', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'P', 'S', 'V', 'W', 'X', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'l', 'n', 'p', 'q', 'r', 's', 't', 'v', 'w', 'x', 'z', '{', '|', '}', '~'], 'K': ['a', 'e', 'i', 'N', 'E', 'n', ' ', 'I', 'r', '.', ',', 'S', '\t', '\n', '\r', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'O', 'P', 'Q', 'R', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'l', 'm', 'o', 'p', 'q', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'J': ['a', 'e', 'u', 'o', 'i', '.', 'A', 'U', '\t', '\n', '\r', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 'r', 's', 't', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'M': ['r', 'a', 'i', 'o', 'y', 'u', 'e', 'E', 'A', ' ', 'U', 'O', '.', 'I', 'P', 'L', 'B', 'Y', '-', ',', 'S', 'c', 'm', '\t', '\n', '\r', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'C', 'D', 'F', 'G', 'H', 'J', 'K', 'M', 'N', 'Q', 'R', 'T', 'V', 'W', 'X', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'd', 'f', 'g', 'h', 'j', 'k', 'l', 'n', 'p', 'q', 's', 't', 'v', 'w', 'x', 'z', '{', '|', '}', '~'], 'L': ['u', 'o', 'a', 'e', 'i', 'L', ' ', 'I', 'E', 'O', 'D', 'Y', ',', 'M', 'l', 'F', 'U', 'T', '.', 'A', 'S', 'C', '\n', '!', "'", '?', 'K', 'W', '\t', '\r', '"', '#', '$', '%', '&', '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '@', 'B', 'G', 'H', 'J', 'N', 'P', 'Q', 'R', 'V', 'X', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'm', 'n', 'p', 'q', 'r', 's', 't', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'O': ['h', 'n', 'g', 'f', 'r', 'U', 'u', 'N', 'l', ' ', 'T', 'W', 'R', 'M', 't', 'L', 'x', 'c', 'V', '!', 'v', 'E', 'S', 'O', 'b', 'w', '\n', '.', 'K', 'p', ',', 'B', 'P', 'd', 'F', 'I', 'A', 'G', 'Y', 's', '\t', '\r', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'C', 'D', 'H', 'J', 'Q', 'X', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'e', 'i', 'j', 'k', 'm', 'o', 'q', 'y', 'z', '{', '|', '}', '~'], 'N': ['o', 'e', 'a', 'G', 'O', 'E', "'", 'T', ' ', 'D', 'i', 'C', 'S', '.', ',', 'A', 'I', 'Y', '\n', 'N', 'V', '!', 'K', '?', 'B', 'R', 'W', 'u', '\t', '\r', '"', '#', '$', '%', '&', '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '@', 'F', 'H', 'J', 'L', 'M', 'P', 'Q', 'U', 'X', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 'r', 's', 't', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'Q': ['u', 'U', 'c', '\t', '\n', '\r', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'P': ['h', 'u', 'e', 'o', 'a', 'r', 'l', 'i', 's', 'E', 'I', 'O', '.', 'P', 'L', 'y', 'A', 'U', ' ', 'H', 'S', 'R', '\t', '\n', '\r', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'B', 'C', 'D', 'F', 'G', 'J', 'K', 'M', 'N', 'Q', 'T', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'j', 'k', 'm', 'n', 'p', 'q', 't', 'v', 'w', 'x', 'z', '{', '|', '}', '~'], 'S': ['h', 'o', 't', 'u', 'a', 'i', 'y', 'e', ' ', 'c', 'p', 'T', 'E', 'S', 'O', 'w', 'H', '.', 'l', 'I', 'P', 'A', 'm', '\n', ',', 'C', 'N', 'U', 'q', 'L', 'k', 'n', 'W', '?', 'Y', '\t', '\r', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '@', 'B', 'D', 'F', 'G', 'J', 'K', 'M', 'Q', 'R', 'V', 'X', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'd', 'f', 'g', 'j', 'r', 's', 'v', 'x', 'z', '{', '|', '}', '~'], 'R': ['i', 'o', 'e', 'E', ' ', 'a', 'I', 'S', 'u', 'Y', 'A', 'T', 'h', '.', 'D', 'M', 'O', 'C', ',', 'K', 'R', '!', '?', 'F', 'N', 'U', 'V', '\t', '\n', '\r', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '@', 'B', 'G', 'H', 'J', 'L', 'P', 'Q', 'W', 'X', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 'r', 's', 't', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'U': ['n', 'p', 'L', 'S', 'T', 'R', 'N', ' ', 'I', ',', 'A', 's', 'C', 'G', 'l', '.', 'P', 'g', 'r', 't', "'", 'E', 'D', 'M', 'O', '\t', '\n', '\r', '!', '"', '#', '$', '%', '&', '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'B', 'F', 'H', 'J', 'K', 'Q', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'h', 'i', 'j', 'k', 'm', 'o', 'q', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'T': ['h', 'o', 'u', ' ', 'r', 'e', 'H', 'a', 'i', 'w', 'E', 'I', '.', '\n', 'O', ',', 'R', '!', 'W', 'Y', 'N', 'S', 'y', '&', '?', 'A', 'L', 'U', 'T', 'c', 's', '\t', '\r', '"', '#', '$', '%', "'", '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '@', 'B', 'C', 'D', 'F', 'G', 'J', 'K', 'M', 'P', 'Q', 'V', 'X', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'd', 'f', 'g', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 't', 'v', 'x', 'z', '{', '|', '}', '~'], 'W': ['h', 'e', 'a', 'i', 'o', 'y', 'A', 'O', ' ', 'r', 'E', 'I', 'H', '.', ',', 'F', 'N', 'R', '?', '\t', '\n', '\r', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '@', 'B', 'C', 'D', 'G', 'J', 'K', 'L', 'M', 'P', 'Q', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 's', 't', 'u', 'v', 'w', 'x', 'z', '{', '|', '}', '~'], 'V': ['i', 'e', 'E', 'o', 'a', 'I', 'A', 'O', '.', '-', '\t', '\n', '\r', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'B', 'C', 'D', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'Y': ['o', 'e', ' ', 'a', 'O', 'T', "'", ',', 'S', 'E', 'I', '\n', 'M', '.', 'u', '\t', '\r', '!', '"', '#', '$', '%', '&', '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'F', 'G', 'H', 'J', 'K', 'L', 'N', 'P', 'Q', 'R', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 'r', 's', 't', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'X': ['E', '!', '\t', '\n', '\r', ' ', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '[': ['G', '\t', '\n', '\r', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'Z': ['e', 'a', 'o', 'u', 'z', '\t', '\n', '\r', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 'r', 's', 't', 'v', 'w', 'x', 'y', '{', '|', '}', '~'], ']': [' ', '\t', '\n', '\r', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '_': [' ', '.', 'y', 'h', 'S', 'm', 't', '\n', ',', 'I', 'w', 'i', 's', 'H', 'a', '"', 'N', 'T', 'W', 'Y', 'd', 'n', 'p', 'u', '?', 'C', 'B', 'E', 'M', 'c', 'b', 'o', '\t', '\r', '!', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '@', 'A', 'D', 'F', 'G', 'J', 'K', 'L', 'O', 'P', 'Q', 'R', 'U', 'V', 'X', 'Z', '[', '\\', ']', '^', '_', '`', 'e', 'f', 'g', 'j', 'k', 'l', 'q', 'r', 'v', 'x', 'z', '{', '|', '}', '~'], 'a': ['n', 't', 's', 'r', 'l', ' ', 'd', 'i', 'c', 'y', 'g', 'v', 'm', 'b', 'k', 'p', 'u', 'w', '\n', 'f', 'z', ',', '.', "'", 'x', '-', 'h', 'j', '!', 'e', 'o', '?', 'a', ':', ';', 'q', '&', ')', '"', 'T', '\t', '\r', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', '{', '|', '}', '~'], 'c': ['o', 'h', 'e', 'a', 'k', 't', 'i', 'l', 'r', 'u', 'y', 'c', ' ', 'q', 's', '.', ',', '\n', '-', ';', '!', ')', "'", '?', 'd', 'I', '&', ':', 'm', '\t', '\r', '"', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'f', 'g', 'j', 'n', 'p', 'v', 'w', 'x', 'z', '{', '|', '}', '~'], 'b': ['e', 'l', 'o', 'u', 'a', 'r', 'y', 'i', 's', 'b', 't', 'j', ' ', ',', 'm', '.', 'd', "'", '-', 'v', '\n', '?', 'n', ';', ':', 'h', '!', '&', ')', 'f', 'w', '\t', '\r', '"', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'c', 'g', 'k', 'p', 'q', 'x', 'z', '{', '|', '}', '~'], 'e': [' ', 'r', 'n', 'd', 's', 'a', 'l', 'e', 't', '\n', ',', 'm', '.', 'v', 'c', 'y', 'p', 'f', 'i', 'w', 'x', "'", 'g', '-', 'o', '?', ';', 'h', 'k', '!', 'b', 'q', ':', 'u', 'j', 'z', '&', '_', ')', 'I', '"', 'B', 'J', 'P', 'T', '\t', '\r', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'C', 'D', 'E', 'F', 'G', 'H', 'K', 'L', 'M', 'N', 'O', 'Q', 'R', 'S', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '`', '{', '|', '}', '~'], 'd': [' ', 'e', 'i', 'o', ',', '\n', '.', 'a', 's', 'r', 'y', 'd', 'n', 'l', 'u', '-', 'g', ';', "'", 'm', 'v', ':', '?', '!', 'f', 'w', 'h', 'b', 'j', ')', 'p', 'k', 't', '&', 'c', '_', 'q', 'I', 'T', '\t', '\r', '"', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '`', 'x', 'z', '{', '|', '}', '~'], 'g': [' ', 'h', 'e', 'o', 'i', 'a', 'g', 'r', ',', 'l', 's', '.', '\n', 'u', 'n', '-', "'", 'y', ';', 't', '?', ':', 'm', '!', '&', 'p', 'f', '"', ')', 'b', 'w', 'T', '\t', '\r', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'c', 'd', 'j', 'k', 'q', 'v', 'x', 'z', '{', '|', '}', '~'], 'f': [' ', 'o', 'e', 'a', 'i', 'r', 'u', 'f', 't', '\n', 'l', ',', '.', '-', 'y', 's', ';', '?', ':', 'p', '!', 'h', '&', 'm', 'n', "'", ')', 'w', 'I', 'T', '_', 'b', '\t', '\r', '"', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '`', 'c', 'd', 'g', 'j', 'k', 'q', 'v', 'x', 'z', '{', '|', '}', '~'], 'i': ['n', 't', 's', 'l', 'd', 'e', 'c', 'm', 'o', 'r', 'g', 'f', 'v', 'k', 'a', 'p', 'b', 'z', 'x', "'", 'u', ' ', 'q', '-', ',', 'h', '.', '\n', '&', '!', ')', ':', '?', '_', 'j', '\t', '\r', '"', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ';', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '`', 'i', 'w', 'y', '{', '|', '}', '~'], 'h': ['e', 'a', 'i', ' ', 'o', 't', ',', 'u', 'r', 'y', '\n', '.', 'n', 's', '-', 'l', 'm', '!', 'b', "'", '?', ';', 'f', 'w', ':', 'd', 'h', 'q', 'c', '&', ')', 'p', 'I', '\t', '\r', '"', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'g', 'j', 'k', 'v', 'x', 'z', '{', '|', '}', '~'], 'k': ['e', ' ', 'i', 'n', ',', 's', '.', '\n', 'y', 'l', '-', 'a', "'", '?', 'f', ';', 'w', '!', 'o', 'm', 'r', ':', 'h', 'c', 'g', 'u', 'b', 't', 'p', '&', 'd', '\t', '\r', '"', '#', '$', '%', '(', ')', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'j', 'k', 'q', 'v', 'x', 'z', '{', '|', '}', '~'], 'j': ['u', 'e', 'o', 'a', 'i', '\t', '\n', '\r', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 'r', 's', 't', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'm': ['e', 'a', ' ', 'o', 'i', 'y', 'p', 'u', ',', 's', '.', 'b', 'm', '\n', "'", 'n', ';', '-', '?', 'f', 'l', '!', ':', 'r', 't', '&', '_', 'c', ')', 'w', 'h', 'v', '\t', '\r', '"', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '`', 'd', 'g', 'j', 'k', 'q', 'x', 'z', '{', '|', '}', '~'], 'l': ['e', 'l', 'i', ' ', 'y', 'o', 'd', 'a', 'f', ',', 'u', 't', 's', '.', 'k', '\n', 'm', 'w', 'v', 'p', '-', 'b', 'c', 'r', 'n', '?', ';', "'", '!', 'g', ':', 'h', '&', ')', '_', 'I', 'P', 'T', '\t', '\r', '"', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'O', 'Q', 'R', 'S', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '`', 'j', 'q', 'x', 'z', '{', '|', '}', '~'], 'o': ['u', 't', 'n', ' ', 'r', 'f', 'm', 'w', 'o', 'l', 's', 'k', 'v', 'd', 'p', 'i', 'c', '\n', 'b', ',', 'a', 'g', "'", '.', 'y', 'e', '-', 'x', 'h', '?', '!', ';', 'z', 'q', ':', 'j', '_', '&', '"', 'C', 'I', '\t', '\r', '#', '$', '%', '(', ')', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'D', 'E', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '`', '{', '|', '}', '~'], 'n': [' ', 'd', 'g', 'e', 't', 'o', 'c', 's', "'", 'i', ',', 'a', '\n', '.', 'y', 'k', 'l', 'n', 'f', 'u', 'v', '-', ';', '?', 'w', 'j', 'q', 'r', 'b', '!', 'x', 'm', 'h', ':', 'p', 'z', '&', ')', 'I', '"', 'N', 'T', '_', '\t', '\r', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'O', 'P', 'Q', 'R', 'S', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '`', '{', '|', '}', '~'], 'q': ['u', '.', '\t', '\n', '\r', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'p': ['e', 'o', 'a', 'r', 'l', ' ', 'i', 'p', 'u', 't', 's', 'h', ',', '.', 'y', '\n', "'", '-', ';', 'w', '?', '!', 'b', 'n', 'm', ':', 'k', '&', 'f', 'c', '"', 'T', '\t', '\r', '#', '$', '%', '(', ')', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'd', 'g', 'j', 'q', 'v', 'x', 'z', '{', '|', '}', '~'], 's': [' ', 't', 'e', 'h', 'a', 'o', 'i', 's', ',', '.', 'u', '\n', 'p', 'c', 'l', 'm', 'k', 'n', 'w', 'y', ';', '?', 'b', '!', "'", ':', '-', 'f', 'g', 'q', '&', 'd', ')', 'r', '_', 'I', 'j', '"', 'A', 'M', 'v', '\t', '\r', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'J', 'K', 'L', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '`', 'x', 'z', '{', '|', '}', '~'], 'r': ['e', ' ', 'o', 'i', 'a', 's', 't', 'y', 'd', '.', ',', 'r', 'n', '\n', 'u', 'l', 'm', 'k', 'c', 'g', 'v', 'p', 'f', "'", '-', 'h', ';', '?', 'b', 'w', '!', ':', '&', '_', ')', 'j', '"', 'C', 'I', 'q', 'x', '\t', '\r', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'D', 'E', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '`', 'z', '{', '|', '}', '~'], 'u': ['o', 't', 's', 'l', 'r', 'n', ' ', 'g', 'p', 'c', 'e', 'i', 'm', 'd', 'a', 'b', "'", ',', '.', 'f', '\n', '?', 'k', 'y', 'z', ';', 'x', '!', '-', '_', 'v', 'w', ':', '&', 'h', 'q', 'I', '\t', '\r', '"', '#', '$', '%', '(', ')', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '`', 'j', 'u', '{', '|', '}', '~'], 't': ['h', ' ', 'o', 'e', ';', 'i', 'a', 'r', 't', ',', '.', '\n', 's', 'l', 'u', 'y', "'", 'w', 'c', '-', '?', 'n', 'm', '!', 'f', ':', 'b', '&', 'g', 'z', 'p', 'I', 'd', ')', '_', 'k', '"', '(', 'T', ']', '\t', '\r', '#', '$', '%', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', '^', '`', 'j', 'q', 'v', 'x', '{', '|', '}', '~'], 'w': ['a', 'i', 'h', 'e', 'o', ' ', 'n', ',', 'r', 's', '.', 'l', '\n', '-', '?', 'f', ';', 'd', 'y', 'k', '!', "'", 'u', ':', 't', 'b', 'c', ')', '&', '_', 'm', 'I', 'q', 'p', '\t', '\r', '"', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '`', 'g', 'j', 'v', 'w', 'x', 'z', '{', '|', '}', '~'], 'v': ['e', 'i', 'a', 'o', 'y', 'u', ' ', 'r', 'b', '.', 'v', '\n', 's', '\t', '\r', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 't', 'w', 'x', 'z', '{', '|', '}', '~'], 'y': [' ', 'o', ',', '.', 'e', '\n', 's', 'i', 't', "'", '-', ';', 'b', '?', 'm', 'a', '!', ':', 'p', 'd', 'l', 'v', 'f', 'w', 'n', 'c', 'r', 'h', '&', ')', 'g', '_', 'z', 'y', 'k', '"', 'I', 'O', 'j', 'x', '\t', '\r', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '`', 'q', 'u', '{', '|', '}', '~'], 'x': ['p', 'c', 't', 'i', 'e', 'a', ' ', ',', 'h', 'u', '.', '-', 'o', 'q', 'f', 'y', '\n', '?', 'g', '!', "'", ';', 'n', '\t', '\r', '"', '#', '$', '%', '&', '(', ')', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'd', 'j', 'k', 'l', 'm', 'r', 's', 'v', 'w', 'x', 'z', '{', '|', '}', '~'], 'z': ['e', 'z', 'i', ' ', 'l', 'y', ',', 'a', 'o', '.', "'", '\n', 'u', '?', ';', ':', 's', '!', 't', '\t', '\r', '"', '#', '$', '%', '&', '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'm', 'n', 'p', 'q', 'r', 'v', 'w', 'x', '{', '|', '}', '~']}
characters_by_freq_english = [' ', 'e', 't', 'o', 'a', 'n', 'i', 'h', 's', 'r', 'l', 'd', 'u', 'm', 'w', 'g', 'f', 'c', 'y', ',', '\n', 'p', '.', 'b', 'v', 'k', ';', "'", 'q', '&', 'I', '-', 'T', 'M', 'S', 'A', '"', 'H', '?', 'B', 'W', 'x', 'E', 'Y', '!', 'L', 'j', 'D', 'N', 'O', 'P', 'G', 'C', 'J', ':', 'R', 'z', 'F', 'K', 'V', '_', 'U', ')', '(', '>', '1', '0', 'Q', 'Z', '9', '4', '7', '2', '8', '3', '5', '6', '#', 'X', '*', '[', ']']

diagraphs_english_no_nl = {'!': [' ', '&', '"', "'", '-', ')', 'A', 'H', 'T', 'B', 'I', 'S', 'W', ',', 'D', 'O', 'R', '_', '!', '>', 'F', 'N', 'P', 'U', 'V', 'Y', '#', '$', '%', '(', '*', '+', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '?', '@', 'C', 'E', 'G', 'J', 'K', 'L', 'M', 'Q', 'X', 'Z', '[', '\\', ']', '^', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], ' ': ['t', 'a', 'h', 's', 'w', 'o', 'i', 'b', 'm', 'f', 'c', 'l', 'd', 'p', 'n', 'y', 'I', 'r', 'g', 'e', 'T', 'M', 'u', ' ', 'S', 'A', '-', 'k', 'H', 'v', 'B', '&', 'W', 'L', 'q', 'P', 'j', 'E', 'D', 'J', 'G', "'", 'C', '"', 'R', 'Y', 'O', 'N', 'F', 'K', 'V', '(', '_', 'U', '.', 'Z', '1', 'Q', 'z', '4', '3', '2', '7', '!', '5', '8', '6', '?', '#', '$', '%', ')', '*', '+', ',', '/', '0', '9', ':', ';', '<', '=', '>', '@', 'X', '[', '\\', ']', '^', '`', 'x', '{', '|', '}', '~'], '#': ['1', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '"': ['\n', ' ', 'I', 'W', 'Y', 'T', 'A', 'S', 'M', 'D', 'H', 'N', 'O', 't', 'L', 'B', 'i', 'G', 'E', "'", 'w', 'F', 'P', 'a', 'C', 'y', 'b', '_', 'm', 's', '-', 'J', 'V', 'h', 'R', 'U', 'o', 'e', 'f', 'n', '?', 'K', 'Q', 'g', ';', 'd', 'k', 'l', 'r', '!', '"', '#', '$', '%', '&', '(', ')', '*', '+', ',', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', '<', '=', '>', '@', 'X', 'Z', '[', '\\', ']', '^', '`', 'c', 'j', 'p', 'q', 'u', 'v', 'x', 'z', '{', '|', '}', '~'], "'": ['s', 't', ' ', 'l', 'v', "'", 'm', 'r', 'd', 'I', 'W', 'Y', 'e', 'T', 'A', 'u', 'N', 'H', 'O', 'c', 'S', 'D', 'n', 'M', 'B', 'a', 'C', 'P', 'L', ',', 'y', 'G', '&', '-', 'i', 'w', 'o', 'E', 'R', 'b', 'F', 'h', 'p', 'J', '1', 'g', '.', 'V', '"', 'f', 'Q', ';', '?', 'K', 'U', '!', ')', '(', '>', 'j', 'q', '#', '$', '%', '*', '+', '/', '0', '2', '3', '4', '5', '6', '7', '8', '9', ':', '<', '=', '@', 'X', 'Z', '[', '\\', ']', '^', '_', '`', 'k', 'x', 'z', '{', '|', '}', '~'], '&': ['q', 'A', 'a', 'o', 'b', '#', 'n', 'c', 'l', 's', 'u', ' ', '!', '"', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'm', 'p', 'r', 't', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], ')': [' ', ',', '.', ';', '&', '-', ':', 'S', '!', '"', '#', '$', '%', "'", '(', ')', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '(': ['a', 'i', 'w', 'h', 't', 's', 'I', 'f', 'M', 'o', 'p', 'm', 'n', 'L', 'T', 'c', 'N', 'P', 'V', 'b', 'l', 'y', 'A', 'C', 'B', 'E', 'D', 'H', 'O', 'S', '_', 'e', 'd', 'r', 'v', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'F', 'G', 'J', 'K', 'Q', 'R', 'U', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '`', 'g', 'j', 'k', 'q', 'u', 'x', 'z', '{', '|', '}', '~'], '*': ["'", ' ', '!', '"', '#', '$', '%', '&', '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '-': [' ', '-', 'a', 'b', 's', 't', 'c', 'm', 'w', 'l', 'h', 'r', 'p', 'd', 'f', 'e', 'g', 'o', 'i', 'n', '&', 'u', 'k', 'I', 'y', '"', "'", 'j', 'v', 'M', 'O', 'S', 'C', 'E', 'A', 'H', 'T', 'N', 'q', '!', '?', 'B', 'D', 'G', 'F', 'W', 'Z', '1', '>', 'L', 'P', '#', '$', '%', '(', ')', '*', '+', ',', '.', '/', '0', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '@', 'J', 'K', 'Q', 'R', 'U', 'V', 'X', 'Y', '[', '\\', ']', '^', '_', '`', 'x', 'z', '{', '|', '}', '~'], ',': [' ', '&', "'", '"', '0', 'I', 'A', 't', 'w', 'y', ',', 'B', 'F', 'd', 's', 'E', 'O', 'W', 'Y', 'b', 'h', 'l', 'o', '!', '#', '$', '%', '(', ')', '*', '+', '-', '.', '/', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'C', 'D', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'X', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'c', 'e', 'f', 'g', 'i', 'j', 'k', 'm', 'n', 'p', 'q', 'r', 'u', 'v', 'x', 'z', '{', '|', '}', '~'], '.': [' ', '&', "'", '"', 'T', 'A', 'S', 'B', 'H', 'I', 'M', 'W', '>', 'P', 'N', 'O', '.', ',', 'R', 'Y', 'J', 'D', 'F', 'L', ')', 'K', 'C', 'V', 'U', 'E', '-', 'G', '2', '?', 'p', '*', '1', 'Q', 'h', '!', '#', '$', '%', '(', '+', '/', '0', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '@', 'X', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '1': ['9', '8', '4', '6', '5', '7', ' ', '.', '3', '1', '0', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '/', '2', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '0': ['0', ' ', ',', '2', '5', 't', '6', ';', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '.', '/', '1', '3', '4', '7', '8', '9', ':', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '3': ['0', ' ', ',', '.', '4', '5', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '1', '2', '3', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '2': [' ', ',', '0', '.', '5', '9', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '1', '2', '3', '4', '6', '7', '8', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '5': [' ', ',', '-', '.', '1', '0', '5', 't', ':', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '/', '2', '3', '4', '6', '7', '8', '9', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '4': ['0', ' ', '7', '.', '6', ';', ',', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '1', '2', '3', '4', '5', '8', '9', ':', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '7': [',', '7', ';', ' ', '.', '0', '2', '5', '6', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '1', '3', '4', '8', '9', ':', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '6': [',', ' ', ';', '9', '0', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '.', '/', '1', '2', '3', '4', '5', '6', '7', '8', ':', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '9': ['1', ' ', '2', '0', ',', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '.', '/', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '8': ['7', ',', '0', ' ', '.', '3', 't', '8', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '1', '2', '4', '5', '6', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], ';': [' ', '&', 'I', 'W', 'T', 'Y', 'A', 'O', 'N', 'B', 'M', 'H', 'S', 'D', 't', 'L', 'a', 'P', 'i', 'C', 'G', 'w', 'y', 'b', 'R', 's', 'F', 'J', 'h', 'E', 'K', '.', 'f', ';', ',', 'V', 'd', 'c', 'm', 'n', 'l', 'p', 'g', '>', 'e', 'r', 'U', 'k', 'u', 'o', 'Q', "'", '-', 'v', 'j', ')', '4', 'q', '!', '(', '2', 'Z', '_', '"', '#', '$', '%', '*', '+', '/', '0', '1', '3', '5', '6', '7', '8', '9', ':', '<', '=', '?', '@', 'X', '[', '\\', ']', '^', '`', 'x', 'z', '{', '|', '}', '~'], ':': [' ', '&', "'", 'I', '-', 'T', 'D', 'E', 'G', 'H', 'K', 'u', '!', '"', '#', '$', '%', '(', ')', '*', '+', ',', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'F', 'J', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '?': [' ', '&', '"', "'", '-', 'T', 'S', 'I', 'W', 'A', 'B', 'D', 'F', 'M', 'N', 'P', '_', ')', 'G', 'H', 'J', 'L', 'O', 'R', 'Y', 'd', '!', '#', '$', '%', '(', '*', '+', ',', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'C', 'E', 'K', 'Q', 'U', 'V', 'X', 'Z', '[', '\\', ']', '^', '`', 'a', 'b', 'c', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '>': ['\n', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'A': ['n', 'l', ' ', 'u', 't', 's', 'h', 'r', 'f', 'm', 'y', 'g', 'N', 'L', 'p', 'R', 'b', 'T', 'c', 'd', 'S', 'w', 'V', 'D', '.', 'M', 'B', 'K', 'P', 'C', 'G', 'F', 'U', 'W', "'", 'I', 'Y', ',', '1', 'a', 'i', 'j', 'v', 'z', '!', '"', '#', '$', '%', '&', '(', ')', '*', '+', '-', '/', '0', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'E', 'H', 'J', 'O', 'Q', 'X', 'Z', '[', '\\', ']', '^', '_', '`', 'e', 'k', 'o', 'q', 'x', '{', '|', '}', '~'], 'C': ['o', 'a', 'h', 'e', 'l', 'r', 'A', 'O', 'E', 'i', 'u', ' ', 'I', '3', 'T', 'H', 'R', '2', 'y', '&', ',', '.', 'K', 'L', 'S', 'U', 'Y', '!', '"', '#', '$', '%', "'", '(', ')', '*', '+', '-', '/', '0', '1', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'B', 'C', 'D', 'F', 'G', 'J', 'M', 'N', 'P', 'Q', 'V', 'W', 'X', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'j', 'k', 'm', 'n', 'p', 'q', 's', 't', 'v', 'w', 'x', 'z', '{', '|', '}', '~'], 'B': ['u', 'a', 'e', 'o', 'r', 'i', 'l', 'y', 'E', 'L', 'U', '.', 'S', ' ', 'O', 'R', 'Y', ',', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'F', 'G', 'H', 'I', 'J', 'K', 'M', 'N', 'P', 'Q', 'T', 'V', 'W', 'X', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'm', 'n', 'p', 'q', 's', 't', 'v', 'w', 'x', 'z', '{', '|', '}', '~'], 'E': ['m', 'n', 'v', ' ', 'l', 'u', 'R', 'a', 'S', 'D', 'L', 'N', 'E', 'h', 'x', 'A', 'V', '.', 'g', 'M', 'd', '!', ',', 'i', 'I', 't', 'T', 'Y', 's', '?', 'C', 'r', 'f', 'O', 'W', 'X', 'B', 'G', 'F', 'K', 'P', 'c', 'q', 'p', 'y', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '@', 'H', 'J', 'Q', 'U', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'e', 'j', 'k', 'o', 'w', 'z', '{', '|', '}', '~'], 'D': ['o', 'e', 'a', 'i', 'r', ' ', 'u', 'O', 'E', 'I', '.', 'N', ',', 'A', 'S', "'", 'D', 'R', 'U', 'y', 'L', '!', '"', '#', '$', '%', '&', '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'B', 'C', 'F', 'G', 'H', 'J', 'K', 'M', 'P', 'Q', 'T', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 's', 't', 'v', 'w', 'x', 'z', '{', '|', '}', '~'], 'G': ['l', 'o', 'e', 'r', 'u', 'i', 'a', '.', ' ', 'E', 'H', '!', ',', 'h', 'A', 'I', 'O', 'G', 'N', 'S', 'R', 'U', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'B', 'C', 'D', 'F', 'J', 'K', 'L', 'M', 'P', 'Q', 'T', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'j', 'k', 'm', 'n', 'p', 'q', 's', 't', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'F': ['r', 'o', 'a', 'i', 'l', 'e', 'u', 'E', 'O', 'I', ' ', 'A', 'R', 'U', 'T', 'F', '.', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'B', 'C', 'D', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'P', 'Q', 'S', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'm', 'n', 'p', 'q', 's', 't', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'I': [' ', 't', "'", 'n', 'f', 's', 'N', '?', 'S', ',', 'T', 'd', 'r', 'w', 'c', 'm', 'E', 'D', 'V', 'L', 'R', '.', 'l', 'C', 'G', '_', 'k', 'O', 'a', 'g', 'h', 'F', '!', 'A', 'B', 'M', 'b', 'v', '-', 'I', 'P', 'o', ';', 'K', 'Q', 'X', 'e', '"', '#', '$', '%', '&', '(', ')', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', '<', '=', '>', '@', 'H', 'J', 'U', 'W', 'Y', 'Z', '[', '\\', ']', '^', '`', 'i', 'j', 'p', 'q', 'u', 'x', 'y', 'z', '{', '|', '}', '~'], 'H': ['e', 'a', 'o', 'i', 'E', 'u', 'A', 'I', 'O', 'T', ' ', 'y', 'R', 'm', 'Y', '.', 'Q', 'U', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'B', 'C', 'D', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'P', 'S', 'V', 'W', 'X', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'l', 'n', 'p', 'q', 'r', 's', 't', 'v', 'w', 'x', 'z', '{', '|', '}', '~'], 'K': ['a', 'e', 'i', 'N', 'E', 'n', ' ', 'I', 'r', '.', ',', 'S', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'O', 'P', 'Q', 'R', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'l', 'm', 'o', 'p', 'q', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'J': ['a', 'e', 'u', 'o', 'i', '.', 'A', 'U', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 'r', 's', 't', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'M': ['r', 'a', 'i', 'o', 'y', 'u', 'e', 'E', 'A', ' ', 'U', 'O', '.', 'I', 'P', 'L', 'B', 'Y', '-', ',', 'S', 'c', 'm', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'C', 'D', 'F', 'G', 'H', 'J', 'K', 'M', 'N', 'Q', 'R', 'T', 'V', 'W', 'X', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'd', 'f', 'g', 'h', 'j', 'k', 'l', 'n', 'p', 'q', 's', 't', 'v', 'w', 'x', 'z', '{', '|', '}', '~'], 'L': ['u', 'o', 'a', 'e', 'i', 'L', ' ', 'I', 'E', 'O', 'D', 'Y', ',', 'M', 'l', 'F', 'U', 'T', '.', 'A', 'S', 'C', '!', "'", '?', 'K', 'W', '"', '#', '$', '%', '&', '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '@', 'B', 'G', 'H', 'J', 'N', 'P', 'Q', 'R', 'V', 'X', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'm', 'n', 'p', 'q', 'r', 's', 't', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'O': ['h', 'n', 'g', 'f', 'r', 'U', 'u', 'N', 'l', ' ', 'T', 'W', 'R', 'M', 't', 'L', 'x', 'c', 'V', '!', 'v', 'E', 'S', 'O', 'b', 'w', '.', 'K', 'p', ',', 'B', 'P', 'd', 'F', 'I', 'A', 'G', 'Y', 's', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'C', 'D', 'H', 'J', 'Q', 'X', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'e', 'i', 'j', 'k', 'm', 'o', 'q', 'y', 'z', '{', '|', '}', '~'], 'N': ['o', 'e', 'a', 'G', 'O', 'E', "'", 'T', ' ', 'D', 'i', 'C', 'S', '.', ',', 'A', 'I', 'Y', 'N', 'V', '!', 'K', '?', 'B', 'R', 'W', 'u', '"', '#', '$', '%', '&', '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '@', 'F', 'H', 'J', 'L', 'M', 'P', 'Q', 'U', 'X', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 'r', 's', 't', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'Q': ['u', 'U', 'c', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'P': ['h', 'u', 'e', 'o', 'a', 'r', 'l', 'i', 's', 'E', 'I', 'O', '.', 'P', 'L', 'y', 'A', 'U', ' ', 'H', 'S', 'R', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'B', 'C', 'D', 'F', 'G', 'J', 'K', 'M', 'N', 'Q', 'T', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'j', 'k', 'm', 'n', 'p', 'q', 't', 'v', 'w', 'x', 'z', '{', '|', '}', '~'], 'S': ['h', 'o', 't', 'u', 'a', 'i', 'y', 'e', ' ', 'c', 'p', 'T', 'E', 'S', 'O', 'w', 'H', '.', 'l', 'I', 'P', 'A', 'm', ',', 'C', 'N', 'U', 'q', 'L', 'k', 'n', 'W', '?', 'Y', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '@', 'B', 'D', 'F', 'G', 'J', 'K', 'M', 'Q', 'R', 'V', 'X', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'd', 'f', 'g', 'j', 'r', 's', 'v', 'x', 'z', '{', '|', '}', '~'], 'R': ['i', 'o', 'e', 'E', ' ', 'a', 'I', 'S', 'u', 'Y', 'A', 'T', 'h', '.', 'D', 'M', 'O', 'C', ',', 'K', 'R', '!', '?', 'F', 'N', 'U', 'V', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '@', 'B', 'G', 'H', 'J', 'L', 'P', 'Q', 'W', 'X', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 'r', 's', 't', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'U': ['n', 'p', 'L', 'S', 'T', 'R', 'N', ' ', 'I', ',', 'A', 's', 'C', 'G', 'l', '.', 'P', 'g', 'r', 't', "'", 'E', 'D', 'M', 'O', '!', '"', '#', '$', '%', '&', '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'B', 'F', 'H', 'J', 'K', 'Q', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'h', 'i', 'j', 'k', 'm', 'o', 'q', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'T': ['h', 'o', 'u', ' ', 'r', 'e', 'H', 'a', 'i', 'w', 'E', 'I', '.', 'O', ',', 'R', '!', 'W', 'Y', 'N', 'S', 'y', '&', '?', 'A', 'L', 'U', 'T', 'c', 's', '"', '#', '$', '%', "'", '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '@', 'B', 'C', 'D', 'F', 'G', 'J', 'K', 'M', 'P', 'Q', 'V', 'X', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'd', 'f', 'g', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 't', 'v', 'x', 'z', '{', '|', '}', '~'], 'W': ['h', 'e', 'a', 'i', 'o', 'y', 'A', 'O', ' ', 'r', 'E', 'I', 'H', '.', ',', 'F', 'N', 'R', '?', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '@', 'B', 'C', 'D', 'G', 'J', 'K', 'L', 'M', 'P', 'Q', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 's', 't', 'u', 'v', 'w', 'x', 'z', '{', '|', '}', '~'], 'V': ['i', 'e', 'E', 'o', 'a', 'I', 'A', 'O', '.', '-', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'B', 'C', 'D', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'Y': ['o', 'e', ' ', 'a', 'O', 'T', "'", ',', 'S', 'E', 'I', 'M', '.', 'u', '!', '"', '#', '$', '%', '&', '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'F', 'G', 'H', 'J', 'K', 'L', 'N', 'P', 'Q', 'R', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 'r', 's', 't', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'X': ['E', '!', ' ', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '[': ['G', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'Z': ['e', 'a', 'o', 'u', 'z', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 'r', 's', 't', 'v', 'w', 'x', 'y', '{', '|', '}', '~'], ']': [' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '_': [' ', '.', 'y', 'h', 'S', 'm', 't', ',', 'I', 'w', 'i', 's', 'H', 'a', '"', 'N', 'T', 'W', 'Y', 'd', 'n', 'p', 'u', '?', 'C', 'B', 'E', 'M', 'c', 'b', 'o', '!', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '@', 'A', 'D', 'F', 'G', 'J', 'K', 'L', 'O', 'P', 'Q', 'R', 'U', 'V', 'X', 'Z', '[', '\\', ']', '^', '_', '`', 'e', 'f', 'g', 'j', 'k', 'l', 'q', 'r', 'v', 'x', 'z', '{', '|', '}', '~'], 'a': ['n', 't', 's', 'r', 'l', ' ', 'd', 'i', 'c', 'y', 'g', 'v', 'm', 'b', 'k', 'p', 'u', 'w', 'f', 'z', ',', '.', "'", 'x', '-', 'h', 'j', '!', 'e', 'o', '?', 'a', ':', ';', 'q', '&', ')', '"', 'T', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', '{', '|', '}', '~'], 'c': ['o', 'h', 'e', 'a', 'k', 't', 'i', 'l', 'r', 'u', 'y', 'c', ' ', 'q', 's', '.', ',', '-', ';', '!', ')', "'", '?', 'd', 'I', '&', ':', 'm', '"', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'f', 'g', 'j', 'n', 'p', 'v', 'w', 'x', 'z', '{', '|', '}', '~'], 'b': ['e', 'l', 'o', 'u', 'a', 'r', 'y', 'i', 's', 'b', 't', 'j', ' ', ',', 'm', '.', 'd', "'", '-', 'v', '?', 'n', ';', ':', 'h', '!', '&', ')', 'f', 'w', '"', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'c', 'g', 'k', 'p', 'q', 'x', 'z', '{', '|', '}', '~'], 'e': [' ', 'r', 'n', 'd', 's', 'a', 'l', 'e', 't', ',', 'm', '.', 'v', 'c', 'y', 'p', 'f', 'i', 'w', 'x', "'", 'g', '-', 'o', '?', ';', 'h', 'k', '!', 'b', 'q', ':', 'u', 'j', 'z', '&', '_', ')', 'I', '"', 'B', 'J', 'P', 'T', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'C', 'D', 'E', 'F', 'G', 'H', 'K', 'L', 'M', 'N', 'O', 'Q', 'R', 'S', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '`', '{', '|', '}', '~'], 'd': [' ', 'e', 'i', 'o', ',', '.', 'a', 's', 'r', 'y', 'd', 'n', 'l', 'u', '-', 'g', ';', "'", 'm', 'v', ':', '?', '!', 'f', 'w', 'h', 'b', 'j', ')', 'p', 'k', 't', '&', 'c', '_', 'q', 'I', 'T', '"', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '`', 'x', 'z', '{', '|', '}', '~'], 'g': [' ', 'h', 'e', 'o', 'i', 'a', 'g', 'r', ',', 'l', 's', '.', 'u', 'n', '-', "'", 'y', ';', 't', '?', ':', 'm', '!', '&', 'p', 'f', '"', ')', 'b', 'w', 'T', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'c', 'd', 'j', 'k', 'q', 'v', 'x', 'z', '{', '|', '}', '~'], 'f': [' ', 'o', 'e', 'a', 'i', 'r', 'u', 'f', 't', 'l', ',', '.', '-', 'y', 's', ';', '?', ':', 'p', '!', 'h', '&', 'm', 'n', "'", ')', 'w', 'I', 'T', '_', 'b', '"', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '`', 'c', 'd', 'g', 'j', 'k', 'q', 'v', 'x', 'z', '{', '|', '}', '~'], 'i': ['n', 't', 's', 'l', 'd', 'e', 'c', 'm', 'o', 'r', 'g', 'f', 'v', 'k', 'a', 'p', 'b', 'z', 'x', "'", 'u', ' ', 'q', '-', ',', 'h', '.', '&', '!', ')', ':', '?', '_', 'j', '"', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ';', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '`', 'i', 'w', 'y', '{', '|', '}', '~'], 'h': ['e', 'a', 'i', ' ', 'o', 't', ',', 'u', 'r', 'y', '.', 'n', 's', '-', 'l', 'm', '!', 'b', "'", '?', ';', 'f', 'w', ':', 'd', 'h', 'q', 'c', '&', ')', 'p', 'I', '"', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'g', 'j', 'k', 'v', 'x', 'z', '{', '|', '}', '~'], 'k': ['e', ' ', 'i', 'n', ',', 's', '.', 'y', 'l', '-', 'a', "'", '?', 'f', ';', 'w', '!', 'o', 'm', 'r', ':', 'h', 'c', 'g', 'u', 'b', 't', 'p', '&', 'd', '"', '#', '$', '%', '(', ')', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'j', 'k', 'q', 'v', 'x', 'z', '{', '|', '}', '~'], 'j': ['u', 'e', 'o', 'a', 'i', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 'r', 's', 't', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'm': ['e', 'a', ' ', 'o', 'i', 'y', 'p', 'u', ',', 's', '.', 'b', 'm', "'", 'n', ';', '-', '?', 'f', 'l', '!', ':', 'r', 't', '&', '_', 'c', ')', 'w', 'h', 'v', '"', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '`', 'd', 'g', 'j', 'k', 'q', 'x', 'z', '{', '|', '}', '~'], 'l': ['e', 'l', 'i', ' ', 'y', 'o', 'd', 'a', 'f', ',', 'u', 't', 's', '.', 'k', 'm', 'w', 'v', 'p', '-', 'b', 'c', 'r', 'n', '?', ';', "'", '!', 'g', ':', 'h', '&', ')', '_', 'I', 'P', 'T', '"', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'O', 'Q', 'R', 'S', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '`', 'j', 'q', 'x', 'z', '{', '|', '}', '~'], 'o': ['u', 't', 'n', ' ', 'r', 'f', 'm', 'w', 'o', 'l', 's', 'k', 'v', 'd', 'p', 'i', 'c', 'b', ',', 'a', 'g', "'", '.', 'y', 'e', '-', 'x', 'h', '?', '!', ';', 'z', 'q', ':', 'j', '_', '&', '"', 'C', 'I', '#', '$', '%', '(', ')', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'D', 'E', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '`', '{', '|', '}', '~'], 'n': [' ', 'd', 'g', 'e', 't', 'o', 'c', 's', "'", 'i', ',', 'a', '.', 'y', 'k', 'l', 'n', 'f', 'u', 'v', '-', ';', '?', 'w', 'j', 'q', 'r', 'b', '!', 'x', 'm', 'h', ':', 'p', 'z', '&', ')', 'I', '"', 'N', 'T', '_', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'O', 'P', 'Q', 'R', 'S', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '`', '{', '|', '}', '~'], 'q': ['u', '.', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'p': ['e', 'o', 'a', 'r', 'l', ' ', 'i', 'p', 'u', 't', 's', 'h', ',', '.', 'y', "'", '-', ';', 'w', '?', '!', 'b', 'n', 'm', ':', 'k', '&', 'f', 'c', '"', 'T', '#', '$', '%', '(', ')', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'd', 'g', 'j', 'q', 'v', 'x', 'z', '{', '|', '}', '~'], 's': [' ', 't', 'e', 'h', 'a', 'o', 'i', 's', ',', '.', 'u', 'p', 'c', 'l', 'm', 'k', 'n', 'w', 'y', ';', '?', 'b', '!', "'", ':', '-', 'f', 'g', 'q', '&', 'd', ')', 'r', '_', 'I', 'j', '"', 'A', 'M', 'v', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'J', 'K', 'L', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '`', 'x', 'z', '{', '|', '}', '~'], 'r': ['e', ' ', 'o', 'i', 'a', 's', 't', 'y', 'd', '.', ',', 'r', 'n', 'u', 'l', 'm', 'k', 'c', 'g', 'v', 'p', 'f', "'", '-', 'h', ';', '?', 'b', 'w', '!', ':', '&', '_', ')', 'j', '"', 'C', 'I', 'q', 'x', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'D', 'E', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '`', 'z', '{', '|', '}', '~'], 'u': ['o', 't', 's', 'l', 'r', 'n', ' ', 'g', 'p', 'c', 'e', 'i', 'm', 'd', 'a', 'b', "'", ',', '.', 'f', '?', 'k', 'y', 'z', ';', 'x', '!', '-', '_', 'v', 'w', ':', '&', 'h', 'q', 'I', '"', '#', '$', '%', '(', ')', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '`', 'j', 'u', '{', '|', '}', '~'], 't': ['h', ' ', 'o', 'e', ';', 'i', 'a', 'r', 't', ',', '.', 's', 'l', 'u', 'y', "'", 'w', 'c', '-', '?', 'n', 'm', '!', 'f', ':', 'b', '&', 'g', 'z', 'p', 'I', 'd', ')', '_', 'k', '"', '(', 'T', ']', '#', '$', '%', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', '^', '`', 'j', 'q', 'v', 'x', '{', '|', '}', '~'], 'w': ['a', 'i', 'h', 'e', 'o', ' ', 'n', ',', 'r', 's', '.', 'l', '-', '?', 'f', ';', 'd', 'y', 'k', '!', "'", 'u', ':', 't', 'b', 'c', ')', '&', '_', 'm', 'I', 'q', 'p', '"', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '`', 'g', 'j', 'v', 'w', 'x', 'z', '{', '|', '}', '~'], 'v': ['e', 'i', 'a', 'o', 'y', 'u', ' ', 'r', 'b', '.', 'v', 's', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 't', 'w', 'x', 'z', '{', '|', '}', '~'], 'y': [' ', 'o', ',', '.', 'e', 's', 'i', 't', "'", '-', ';', 'b', '?', 'm', 'a', '!', ':', 'p', 'd', 'l', 'v', 'f', 'w', 'n', 'c', 'r', 'h', '&', ')', 'g', '_', 'z', 'y', 'k', '"', 'I', 'O', 'j', 'x', '#', '$', '%', '(', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '`', 'q', 'u', '{', '|', '}', '~'], 'x': ['p', 'c', 't', 'i', 'e', 'a', ' ', ',', 'h', 'u', '.', '-', 'o', 'q', 'f', 'y', '?', 'g', '!', "'", ';', 'n', '"', '#', '$', '%', '&', '(', ')', '*', '+', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'd', 'j', 'k', 'l', 'm', 'r', 's', 'v', 'w', 'x', 'z', '{', '|', '}', '~'], 'z': ['e', 'z', 'i', ' ', 'l', 'y', ',', 'a', 'o', '.', "'", 'u', '?', ';', ':', 's', '!', 't', '"', '#', '$', '%', '&', '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'm', 'n', 'p', 'q', 'r', 'v', 'w', 'x', '{', '|', '}', '~']}
characters_by_freq_english_no_nl = [' ', 'e', 't', 'o', 'a', 'n', 'i', 'h', 's', 'r', 'l', 'd', 'u', 'm', 'w', 'g', 'f', 'c', 'y', ',', 'p', '.', 'b', 'v', 'k', ';', "'", 'q', '&', 'I', '-', 'T', 'M', 'S', 'A', '"', 'H', '?', 'B', 'W', 'x', 'E', 'Y', '!', 'L', 'j', 'D', 'N', 'O', 'P', 'G', 'C', 'J', ':', 'R', 'z', 'F', 'K', 'V', '_', 'U', ')', '(', '>', '1', '0', 'Q', 'Z', '9', '4', '7', '2', '8', '3', '5', '6', '#', 'X', '*', '[', ']']

diagraphs_test_server = {'!': [' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], ' ': ['t', 'a', 'o', 's', 'd', 'w', 'b', 'i', 'f', 'I', 'm', 'y', 'l', 'g', 'r', 'c', 'h', 'n', 'W', 'T', 'e', 'A', 'p', 'D', 'u', 'S', '-', 'B', 'k', 'j', 'E', 'F', 'M', 'O', 'N', '2', '4', '6', '9', 'C', 'G', 'H', 'K', 'L', 'Y', 'q', 'v', 'z', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '.', '/', '0', '1', '3', '5', '7', '8', ':', ';', '<', '=', '>', '?', '@', 'J', 'P', 'Q', 'R', 'U', 'V', 'X', 'Z', '[', '\\', ']', '^', '_', '`', 'x', '{', '|', '}', '~'], '%': [' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '-': [' ', '-', 'r', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], ',': [' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '.': [' ', '.', 's', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '3': ['%', ' ', '!', '"', '#', '$', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '2': ['%', ' ', '!', '"', '#', '$', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '4': ['%', ' ', '!', '"', '#', '$', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '6': ['%', ' ', '!', '"', '#', '$', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '9': ['3', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], ';': [' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], '?': [' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'A': ['n', 'P', 'm', ',', 'l', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'C': ['i', 'E', 'O', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'B': ['i', 'u', 'o', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 'r', 's', 't', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'E': [' ', 'N', 'a', 'l', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'D': ['a', 'i', 'o', 'w', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 'r', 's', 't', 'u', 'v', 'x', 'y', 'z', '{', '|', '}', '~'], 'G': ['I', 'o', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'F': ['i', 'O', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'I': ['m', ' ', 'f', 'n', 't', 'E', 'N', 'Z', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'g', 'h', 'i', 'j', 'k', 'l', 'o', 'p', 'q', 'r', 's', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'H': ['e', 'E', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'K': ['.', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'M': ['o', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'L': ['O', 'o', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'O': ['G', 'L', 'o', 'N', 'R', 'n', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'H', 'I', 'J', 'K', 'M', 'O', 'P', 'Q', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'N': ['C', 'I', 'e', 'o', 'V', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'D', 'E', 'F', 'G', 'H', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'P': ['O', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'S': ['p', 'u', 't', 'o', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'q', 'r', 's', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'R': [' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'T': ['h', 'o', 'H', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'i', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'W': ['e', 'i', 'a', 'E', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'V': ['E', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'Y': ['o', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'Z': ['E', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'a': ['t', 'n', 'l', 'r', ' ', 's', 'f', 'p', 'c', 'd', 'v', 'y', 'i', 'k', ',', '.', '?', 'b', 'g', 'm', 'q', 'u', 'w', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'e', 'h', 'j', 'o', 'x', 'z', '{', '|', '}', '~'], 'c': ['a', 'e', 'h', 'k', 'o', ' ', 'i', 't', 'l', 'r', 'u', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'j', 'm', 'n', 'p', 'q', 's', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'b': ['e', 'a', 'i', 'u', 'o', 'r', '!', ' ', 'l', 't', 'y', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'm', 'n', 'p', 'q', 's', 'v', 'w', 'x', 'z', '{', '|', '}', '~'], 'e': [' ', 'r', 'n', 's', 'a', '.', 'm', 't', 'v', 'c', 'e', 'd', ',', 'l', 'f', 'i', 'x', 'o', 'p', 'y', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'g', 'h', 'j', 'k', 'q', 'u', 'w', 'z', '{', '|', '}', '~'], 'd': [' ', 'o', 'e', 'a', 's', 'i', '.', 'y', 'd', 'j', ',', 'r', 'v', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'f', 'g', 'h', 'k', 'l', 'm', 'n', 'p', 'q', 't', 'u', 'w', 'x', 'z', '{', '|', '}', '~'], 'g': [' ', 'e', 'h', 's', 'o', 'r', 'i', '.', 'g', ',', 'u', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'f', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 't', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'f': [' ', 'o', 'r', 'e', 'f', 'i', 'a', 'u', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'g', 'h', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 's', 't', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'i': ['n', 't', 's', 'c', 'v', 'g', 'e', 'd', 'l', 'r', 'o', 'b', 'm', 'p', 'a', 'f', '.', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'h', 'i', 'j', 'k', 'q', 'u', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'h': ['e', 'a', 'i', ' ', 't', 'r', 'o', '.', 'u', '?', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 's', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'k': [' ', 'e', 'i', 'y', 'n', '.', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'l', 'm', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'z', '{', '|', '}', '~'], 'j': ['u', 'a', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'm': ['e', ' ', 'a', 'i', 'b', ',', 'o', '.', 'u', 'y', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 'r', 's', 't', 'v', 'w', 'x', 'z', '{', '|', '}', '~'], 'l': ['l', 'e', ' ', 'i', 'o', ',', 'r', 'y', 'a', 'd', 's', '.', 'v', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'f', 'g', 'h', 'j', 'k', 'm', 'n', 'p', 'q', 't', 'u', 'w', 'x', 'z', '{', '|', '}', '~'], 'o': ['u', 'r', 'n', ' ', 'm', 't', 'f', 'o', 'w', 'd', 'i', 'k', 'p', 's', 'v', '.', 'c', 'b', 'g', 'h', 'l', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'e', 'j', 'q', 'x', 'y', 'z', '{', '|', '}', '~'], 'n': ['g', ' ', 'd', 'o', 't', 'e', 's', ',', 'i', 'k', '.', 'y', '?', 'a', 'c', 'm', 'l', 'u', 'v', 'n', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'f', 'h', 'j', 'p', 'q', 'r', 'w', 'x', 'z', '{', '|', '}', '~'], 'q': ['u', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'p': ['o', 'e', 'a', 'p', 'i', 'l', ' ', 'r', 't', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'm', 'n', 'q', 's', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 's': [' ', 't', 'e', ',', 'i', 'h', 'o', 'p', 'a', 's', 'u', 'k', 'c', '.', 'w', 'y', ';', '?', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'd', 'f', 'g', 'j', 'l', 'm', 'n', 'q', 'r', 'v', 'x', 'z', '{', '|', '}', '~'], 'r': ['e', ' ', 'i', 'a', 'd', 'o', 'y', 'k', 's', 'c', 't', 'g', 'f', '-', 'l', 'n', 'r', 'u', '.', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'h', 'j', 'm', 'p', 'q', 'v', 'w', 'x', 'z', '{', '|', '}', '~'], 'u': [' ', 't', 's', 'n', 'r', 'd', 'a', 'g', 'c', 'b', 'e', 'i', 'h', 'l', 'p', ',', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'f', 'j', 'k', 'm', 'o', 'q', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 't': ['h', ' ', 'o', 'e', 'i', 's', 'a', 'r', '.', 't', 'l', 'u', 'y', ',', 'c', '?', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'd', 'f', 'g', 'j', 'k', 'm', 'n', 'p', 'q', 'v', 'w', 'x', 'z', '{', '|', '}', '~'], 'w': ['i', 'h', 'e', 'o', 'a', 's', ' ', ',', 'l', '.', 'r', 'v', 'n', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'j', 'k', 'm', 'p', 'q', 't', 'u', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'v': ['e', 'i', 'a', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'b', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'y': ['o', ' ', ',', 's', 'a', 'i', '.', 'b', ';', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', '-', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'c', 'd', 'e', 'f', 'g', 'h', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 'r', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'x': ['c', 'e', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'd', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~'], 'z': ['o', ' ', '!', '"', '#', '$', '%', '&', "'", '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~']}
characters_by_freq_test_server = [' ', 'e', 't', 'o', 'a', 'n', 'r', 'i', 's', 'h', 'd', 'l', 'u', 'g', 'm', 'y', '.', 'f', 'c', ',', 'w', 'b', 'p', 'v', 'k', 'I', 'W', 'E', 'T', '-', 'A', '?', 'O', 'N', 'D', '%', 'S', 'j', 'C', 'B', ';', 'G', 'F', 'H', 'M', 'L', 'q', 'x', '!', '3', '2', '4', '6', '9', 'K', 'P', 'R', 'V', 'Y', 'Z', 'z']

diagraphs_credit_cards = {'1': ['6', '4', '3', '5', '7', '0', '1', '8', '2', '9'], '0': ['0', '7', '2', '4', '5', '3', '1', '9', '8', '6'], '3': ['4', '7', '2', '9', '5', '3', '1', '8', '6', '0'], '2': ['4', '9', '3', '5', '0', '2', '6', '1', '7', '8'], '5': ['3', '5', '4', '1', '2', '6', '8', '7', '9', '0'], '4': ['5', '9', '0', '4', '8', '7', '3', '1', '6', '2'], '7': ['1', '4', '3', '5', '0', '2', '6', '8', '7', '9'], '6': ['4', '3', '5', '7', '9', '6', '8', '0', '1', '2'], '9': ['2', '1', '4', '3', '5', '7', '8', '0', '9', '6'], '8': ['4', '6', '3', '5', '9', '7', '2', '1', '0', '8']}
characters_by_freq_credit_cards = ['4','5','3','7','2','9','6','1','0','8']

diagraphs_python = {' ': [' ', 't', 's', 'i', 'a', '=', 'd', 'f', 'r', 'c', "'", 'o', '(', 'e', '"', 'p', 'n', 'u', 'm', '#', 'w', 'b', 'l', 'v', '%', 'T', 'g', '_', 'h', 'N', 'C', 'S', '[', 'F', 'I', '1', 'A', '{', 'G', 'P', 'D', 'M', 'O', '-', '*', 'R', '+', 'k', '2', '/', '`', 'E', '}', '0', 'U', 'L', 'q', 'y', 'W', 'j', '<', 'B', 'H', '3', ':', 'V', '>', 'x', ')', '4', '.', '&', '@', '5', '\n', '!', 'J', 'z', 'K', '|', '\\', '8', 'Y', '6', '9', ']', '7', 'Q', 'X', 'Z', ',', '$', ';', '^', '?', '~'], '$': ["'", '%', ')', '"', ' ', '\\', 'd', 'J', '-', 'P', 's', 'f', 'h', '&', 'p', '.'], '(': ["'", 's', ')', 'r', '"', 'f', 'p', 'c', 'o', 't', 'u', 'm', 'a', 'l', 'i', 'v', 'd', 'w', '\n', 'n', 'e', '1', '[', '2', 'T', '(', 'g', 'b', '?', '*', '0', '_', 'k', '3', 'F', 'G', 'C', '5', 'S', '{', '4', 'B', 'N', 'h', '%', 'A', '-', 'V', 'E', '8', 'x', 'L', 'P', 'I', 'M', 'q', '6', 'D', 'O', '9', 'U', '\\', ' ', 'y', 'R', 'W', 'j', 'H', '`', '#', 'z', '7', '.', '+', 'K', 'Q', 'Y', '|', '/', 'J', '&', '^', '<', 'X', '>', '!', '='], ',': [' ', '\n', "'", ')', '2', '1', '"', '0', '3', 'A', 'P', '(', '5', '-', '6', 'U', 'S', '4', 'D', 'b', '%', 'i', '\\', 'v', '8', 'k', 'm', 't', 'G', 'j', 'p', '}', '9', 'L', 'n', ',', '.', 'N', 'T', 'a', 'e', '|', 'J', 'u', '*', '7', 'E', '[', 'c', 'd', 'l', 'q', 's', 'w', '{', '!', ':', 'B', 'V', 'Y', ']', 'y'], '0': ['0', '1', ',', ')', ']', "'", ' ', '4', ':', '.', '"', '-', '2', '8', '5', '3', '\n', '7', '6', '9', 'd', '\\', 'a', '/', '<', 'N', 'n', 'x', 'A', 'c', 'e', '}', 'm', '=', '%', '(', 'f', '_', '*', 'M', 'k', 't', 'b', 'E', '+', 'D', 'F', 'Z', 'p', 'r', '{', 'X', '`', 'C', 'B', '^', 'l', 'o', 'w', '|'], '4': ['0', ',', ' ', "'", ')', '2', '"', '3', '.', '5', ':', '4', '1', '8', '(', '\n', '<', '_', '7', '-', '9', '6', ']', '/', 'd', '\\', 'e', '}', 'x', '*', '=', 'c', 'a', '%', 'T', '[', 's', 'f', '`', 'm', 't', '>', 'F', 'M', 'b', 'h'], '8': [',', "'", '0', ' ', '.', '2', '3', '4', '"', '1', '9', ')', 'n', 'a', '5', '8', '7', '6', '<', '/', 'N', '-', '\n', '\\', '_', 'h', '(', 'c', ':', ']', 'b', 'd', 'o', 'D', 'f', '}', '%', 'C', '`', 't', 'x'], '<': ['/', 'o', 'l', 'i', 's', ' ', 't', 'p', 'u', 'a', 'b', '=', 'h', 'd', 'm', 'e', 'f', '"', 'y', '%', 'A', 'B', '1', 'c', '5', 'r', '-', '?', 'n', '>', 'v', '|', '!', '2', 'D', 'F', 'M', 'P', 'w', "'", '0', '<', 'E', 'I', 'N', 'R', 'g', '&', 'L', 'O', 'T', 'W', '`', 'j', 'x'], '@': ['r', 'p', 'e', 's', 'c', 'f', 'w', 'n', "'", '(', 'o', '|', 'd', 'm', 'l', 'v', 'b', ' ', '#', '"', '%', '+', '[', 'x'], 'D': ['e', 'o', 'E', 'a', 'i', 'A', 'I', 'j', ' ', "'", '_', 'r', 'B', 'D', 'S', 'L', 'O', '(', 'R', '\n', 'u', 'U', ')', ',', 'J', '=', '8', 'N', '[', '.', 'C', 'F', 'M', '5', 'y', '"', '2', 'T', 'P', 's', '4', ':', 'K', '\\', 'v', '1', 'G', 'V', 'l', '%', 'H', 'W', 'g', 'p', 'x'], 'H': ['T', 'O', 'a', 'E', 't', 'A', 'e', 'o', "'", ' ', 'i', 'S', 'I', '_', ':', '.', 'U', 'u', 'P', 'R', 'r', 'H', 'v', '\n', 'M', ',', 'B', 'y', '%', '=', 'N', ']', '"', ')', 'C', 'D', 'G', 'K', 'l', 'n', 'p'], 'L': ['E', 'i', 'I', ' ', 'o', 'A', 'a', 'O', '_', 'T', 'L', 'e', "'", 'U', 'Y', 'D', ':', 'C', '\n', '1', 'S', '.', ')', ',', 'P', 'B', 'H', 'u', 's', '"', '(', 'c', 'R', 'F', '-', ']', 'G', 'j', 'N', 'W', '=', 'n', '/', ';', 'V', '}', 'K', 'J', 'M', 'X', '`', 'l', 't'], 'P': ['o', 'a', 'O', 'A', 'y', 'r', 'S', 'R', 'e', 'T', '<', "'", '_', 'E', ' ', 'L', 'P', 'I', 'v', 'U', 'l', 'M', 'i', 'u', '"', 'H', 'B', ')', 'K', 'Y', '\n', 'G', 'd', ',', 'D', 'N', '.', ':', 'C', 'h', 't', '(', ';', 'V', 's', '-', '1', '=', 'F', 'W', 'Z'], 'T': ['e', 'r', 'h', 'E', 'y', '_', 'I', 'M', 'A', ' ', 'H', 'R', 'T', 'i', 'Y', "'", 'o', 'O', 'S', 'a', 'P', '(', ',', 'u', '\n', 'U', 'Z', 'F', ')', '.', '[', 'C', 'D', 'N', 'B', 'w', 'W', '"', '*', ':', 'L', 's', '1', 'X', 'x', '/', 'G', ']', '0', 'V', 'l', 'v', '|', '+', '-', '7', '6', '=', '`', 'z'], 'X': [' ', 'T', '_', 'X', "'", 'M', 'E', '-', 'C', 'I', ',', '"', 'V', '\n', ')', '(', '.', 'A', ':', 'S', ']', 'H', 'P', 'Y', 't', 'D', 'F', 'O', 'R', '\\', 'u'], '\\': ['n', 'u', 'd', '\\', 'x', '\n', "'", 's', '"', 't', '.', 'w', ']', 'r', '[', '+', '0', ':', ')', '(', '/', '?', '*', 'b', '{', ' ', '}', '^', 'f', '$', '1', '|', '-', 'S', 'g', '%', '&', 'C', 'E', 'P', 'T', 'W', 'e', 'o', 'v'], '`': ['`', ' ', '.', ',', '\n', 'c', 's', 'f', 't', 'g', 'o', ')', 'a', 'm', '{', 'd', 'N', 'e', 'n', 'r', 'F', 'D', 'T', 'u', 'i', 'w', 'p', 'l', '(', 'S', '<', '"', 'C', 'U', 'L', 'O', '_', 'G', 'M', 'v', 'H', ']', 'b', 'I', '\\', '%', "'", '/', ':', '=', '>', 'E', 'W', '[', 'h', '}', '!', '#', '&', '-', ';', 'A', 'B', 'P', 'R', 'y', 'x', 'z'], 'd': ['e', ' ', 'i', 'a', '_', '(', 'o', 'j', 's', '.', 'd', 'u', ',', 'l', '\n', ')', "'", 'r', 'b', '=', 'g', '"', 'm', ':', 'E', 'y', 't', '/', '1', 'n', 'D', 'F', 'T', '-', '[', 'M', ']', 'f', '>', 'S', 'z', '+', 'O', '2', 'x', 'c', 'C', 'N', '\\', '`', '{', 'p', 'H', ';', 'w', '5', 'I', '3', 'v', 'W', 'P', '4', 'A', 'k', 'R', '0', '<', 'h', 'U', '&', '|', 'Q', '6', '8', 'G', '*', 'B', '}', '?', '%', '@', '$', 'L', 'V', '9', '#', '7', 'K', '!'], 'h': ['e', 'a', 'o', 'i', ' ', 't', '.', '_', '/', 'r', '(', ')', ',', "'", 'u', 's', 'm', '\n', 'n', '>', 'l', '=', '"', 'd', ':', 'y', '1', 'b', '2', '-', '[', 'p', 'F', ']', '0', 'h', 'M', 'V', '`', '<', 'G', 'H', 'A', 'c', 'f', 'j', '}', '3', ';', 'D', 'I', 'N', 'Q', 'R', 'w', '$', 'E', 'T', '+', '?', 'L', 'g', 'k'], 'l': ['f', 'e', 'i', 'a', 'u', 'd', 's', 'o', 'l', 't', '(', ' ', 'y', '_', '.', "'", ',', ')', '>', 'p', '\n', '=', '"', 'v', '[', 'b', 'g', 'c', 'r', ':', 'n', 'R', '-', 'm', '/', '1', 'h', 'k', 'w', ']', 'j', 'B', 'F', 'T', '<', '2', 'A', '`', 'C', 'I', '#', 'M', 'H', ';', 'N', 'P', '0', 'D', '\\', 'E', '*', 'q', '|', '%', 'z', 'U', '3', 'V', '4', '@', 'K', 'L', 'x', '}', '!', '$', '5', '6', '?', 'G', 'S', 'W'], 'p': ['e', 'a', 'o', 't', 'l', 'r', 'p', 'u', 'i', ' ', '_', '(', 's', 'y', '.', 'k', 'd', ':', ',', '>', 'h', 'n', ')', ';', '=', '\n', "'", 'f', 'v', '[', 'R', ']', 'g', 'C', 'c', '1', 'T', '4', '2', 'F', '/', 'E', '"', 'm', 'x', 'I', 'b', 'D', 'S', '3', 'M', '-', 'w', 'U', '`', '+', '0', '}', 'A', 'N', '&', '<', 'G', 'P', '%', '@', 'O', '\\', 'q', '|'], 't': ['e', 'h', ' ', 'i', 'r', 'a', 'o', '_', 'u', 's', 't', '(', 'y', '.', 'E', ',', ')', 'p', '\n', "'", '"', 'c', 'l', 'C', 'T', '/', ':', 'm', '=', '[', 'R', 'F', 'd', 'H', 'I', 'f', '-', 'w', '>', 'M', ';', 'z', 'N', 'U', '0', 'S', '1', 'A', '2', 'D', 'b', '`', 'n', 'L', 'v', 'P', ']', 'x', 'V', 'G', 'O', 'g', '<', '@', '3', 'Q', '+', '\\', 'W', '4', '|', '*', 'j', '}', 'B', '%', 'k', '&', '6', 'Z', '5', 'X', 'Y', '8', '7', '?', '!', '9', 'K', '$', '#', '{', 'q'], 'x': ['t', 'c', 'p', 'a', 'i', ' ', '_', 'e', ',', ')', 'E', '(', '.', '"', 'm', '=', 'r', 'y', '&', ':', '\n', "'", 'd', 'f', '<', ']', '>', '[', 'h', '-', '1', 'l', 'U', '8', 'k', '0', '3', '2', 'o', 'F', 'I', 'L', '+', 'b', '/', 'O', 'R', 'g', 's', 'v', 'x', 'P', 'S', 'Y', 'X', '7', '9', 'M', 'V', '`', 'D', ';', 'N', 'T', 'u', '}', '4', 'C', 'H', '\\'], '|': [' ', '\\', 'c', 'l', 'f', 't', '\n', 's', '!', 'u', 'd', 'p', 'e', 'r', '"', '=', '<', 'a', 'j', '|', '(', '~', '^', '>', 'w', "'", '-', '$', ',', 'm', '&', ';', 'A', '@', '[', 'i', ')', 'M', 'G', 'L', '%', '.', '1', ':', '?', 'P', 'W', ']', 'o', 'v'], '#': [' ', '#', '\n', '1', '3', '.', 'h', '7', 'd', '8', 'l', '5', 'e', '-', '/', 's', '!', '4', '"', '0', '9', "'", '2', ':', 'G', 'T', 'c', 't', 'w', 'a', 'b', 'f', 'p', 'r', '$', 'I', 'R', '\\', '}', '(', '6', ';', 'F', 'S', 'i', 'n', '%', ')', 'A', '@', 'D', 'H', 'N', 'W', 'V', '`', 'm', 'o'], "'": [',', ')', ':', ' ', ']', 's', '\n', 't', 'c', 'f', '/', 'a', 'd', "'", '%', 'm', 'P', 'S', 'r', '.', 'p', '1', 'n', 'i', 'T', 'C', 'G', 'M', 'e', '<', 'l', '2', 'N', 'b', 'R', 'A', 'o', 'g', 'h', '\\', 'B', 'D', 'L', '"', '-', '{', 'k', 'v', 'I', '}', '0', 'K', 'V', 'u', 'E', '^', '_', 'F', 'J', 'w', 'O', '3', 'U', '4', 'W', 'H', '5', 'j', '6', 'Z', '7', '8', 'y', '(', '9', 'z', 'Y', '[', '#', '+', 'X', 'x', '*', 'q', '$', ';', '?', 'Q', '=', '&', '|', '>', '@', '`', '!', '~'], '+': [' ', '=', ')', '-', '\n', '1', '"', '+', '\\', '|', '(', '0', "'", '/', 'l', '.', '$', 'd', 's', '4', ']', '2', 'a', 'e', 'n', 'p', 'x', '3', '[', 'i', 'k', '#', ',', '?', 'b', 'f', 't', '5', '6', 'E', 'N', 'c', 'h', 'u', 'v', 'y'], '/': ['/', 't', 'o', 'p', '>', 's', 'l', "'", 'j', 'c', 'a', 'd', '%', 'b', 'm', 'e', 'f', '"', '2', ' ', 'u', 'w', 'r', '$', '1', '(', '0', 'g', 'h', 'v', 'n', 'i', '?', '\\', '\n', 'x', '^', '3', '[', ',', '=', 'P', '#', '*', 'L', '<', 'O', '4', 'C', ')', 'A', ']', '|', 'E', 'D', 'H', '8', 'F', 'M', 'R', '`', '&', '+', '.', '5', '6', '9', ';', 'B', 'G', 'I', 'S', 'W', 'Y', 'k', 'q', 'y', '~', '-', ':', 'K', 'N', 'U', '{'], '3': ['0', '3', ',', '2', "'", '"', '.', ')', '4', ' ', ':', '1', '7', '5', '6', '8', '<', ']', '9', 'd', '\n', '-', '_', 'D', '(', ';', '/', 'C', '}', '%', '&', 'E', 'c', 'r', '>', '\\', 'b', 'e', 'f', 'F', 'T', 'a', '|', '*', 'k', 'l', '+', '@', 'L', 'R', 'g', 'h'], '7': [',', '4', '.', '0', '8', "'", '2', '"', '1', 'e', ' ', ')', '7', '5', '6', '9', '3', '\\', 'd', '<', '-', ':', '/', ']', '_', '\n', '(', 'a', 'b', 'g', 'f', 's', '!', 'F', 'c', '}', 'T', 'r', 't', 'v'], ';': [' ', '"', '\n', 'y', "'", 'b', '\\', 's', 'a', '}', '&', '1', 'q', 't', '%', '/', '<', '`', 'x', 'e', 'S', '|', '#', ')', 'h', 'm', 'l', ';', 'P', 'd', 'f', '$', ',', '.', 'A', 'U', ']', 'c', 'g', 'k', 'o', 'p', 'v'], '?': ['P', ':', '\n', "'", 'x', '"', '!', ' ', ')', 'p', '\\', 'c', 't', '/', '?', '$', 'n', '|', '>', 'f', 'm', '(', '=', '<', 'u', '*', '^', 's', '#', '-', '.', 'L', '[', 'i'], 'C': ['o', 'a', 'h', 'O', 'l', 'E', 'H', 'A', 'S', 'r', 'T', ' ', 'K', "'", 'L', 'u', '_', 'e', 'I', 'i', 'R', 'C', 'D', 'G', 'F', 'U', 'y', 's', 'N', '(', '3', '\n', 'V', 'Z', 'm', ')', ':', '/', 'M', 'z', '%', '.', '0', '4', 'P', 'Y', '\\', 'c', 'b', 't', '"', '+', '-', ',', '2', '5', '<', 'B', 'J', 'x'], 'G': ['e', 'E', 'R', 'I', 'o', "'", 'D', '"', 'S', 'r', 'O', 'U', ' ', 'i', 'T', 'N', 'C', 'L', 'M', '_', 'P', 'a', ':', '=', '(', 'H', 'u', 'A', 'l', '\n', 'B', 'G', 'x', ',', 'J', 'w', ')', 'F', 'Z', 'z', '|', '!', 'd', 'V', 'j', 'm', 'y'], 'K': ['e', 'E', 'T', 'a', 'B', 'I', '_', 'o', "'", 'r', 'M', 'L', ' ', 'i', 'u', 'U', 'D', 'A', 'S', ')', '(', ':', 'R', '\n', 'Y', 'n', ',', '.', 'C', 'h', 'y', 'N', 'l', '"', '/', 'F', 'H', 'K', 'Z', 'j', 's', '=', 'G', 'X', 'w'], 'O': ['S', 'R', 'G', 'N', 'I', 'T', 'F', 'b', 'M', 'C', 'D', 'p', 'O', 'L', 'K', 'r', '_', 'U', 'n', 'u', "'", ' ', 't', 'f', 'P', 'V', 'J', 'W', 'v', '(', 'A', '\n', 'E', 'c', ':', ',', '.', 'l', 's', 'B', 'X', 'H', 'k', 'Y', 'i', 'h', '2', '=', '\\', 'g', 'o', 'x', ')', '-', 'Q', 'a', 'd', 'm'], 'S': ['e', ' ', 't', 'E', 'T', 'o', 'i', 'G', '_', "'", 'p', 'I', 'u', 'S', 'a', 'R', 'Q', 'O', 'y', '\n', 'C', 'P', '[', 'h', 'F', 'A', 'H', 'k', ')', ',', ':', 'c', 'L', 'U', 'm', 'W', '"', '.', 'l', 'K', '(', '8', 'D', 'V', 'B', 'M', ']', 'v', '/', '=', '-', 'r', '1', 'Y', 'w', '|', ';', '`', 's', 'N', '2', '5', 'n', '!', '7', 'J', 'Z'], 'W': ['i', 'e', 'K', 'a', 'A', 'h', 'S', 'r', 'G', 'H', 'O', 'o', 'W', 'I', 'E', "'", '_', 'D', 'V', 'B', ' ', 'Y', 'N', '\n', '/', 'L', 'R', ',', '3', 'y', '%', ')', ':', 'C', 'U', '[', 'k', 'u'], '[': ["'", '0', '"', ']', '1', ':', 'i', 'u', 's', 'f', '(', 'c', '-', 'm', 'a', '\n', 'o', 'n', '2', 'l', 'k', '^', 'p', 'd', 't', 'N', 'e', 'r', '\\', '{', '%', 'G', 'v', '3', '[', 'L', ' ', 'T', 'b', 'R', '4', 'g', 'j', 'x', 'A', 'C', 'D', 'J', '5', 'O', '7', '.', 'h', 'z', '_', 'M', 'P', 'X', '|', '/', '6', '9', 'B', 'q', 'w', '+', '*', '8', 'E', 'F', 'I', 'Z', 'y'], '_': ['c', '_', 's', 'f', 't', 'm', 'n', 'd', 'p', 'i', 'l', 'r', 'a', '(', 'o', 'u', 'v', 'e', 'g', 'h', 'w', 'b', 'k', 'C', 'F', 'P', 'D', 'A', ' ', 'T', 'S', 'q', 'N', 'U', 'I', '1', 'M', '"', ')', 'L', 'E', '\n', "'", 'y', 'R', '2', 'j', ',', 'B', 'V', '.', 'G', '0', 'O', 'x', 'H', '4', 'K', 'W', '3', '%', 'z', '*', '[', 'J', ']', '8', '`', 'Z', ':', 'X', '=', '\\', '-', '6', 'Y', '5'], 'c': ['o', 't', 'e', 'h', 'l', 'a', 'k', 'r', 'u', 'i', 's', '_', 'c', ' ', '.', '=', '(', 'T', ')', "'", ',', 'm', 'f', 'y', '\n', '2', 'z', '1', 'd', 'g', 'n', '/', '3', '-', '4', '"', 'F', 'I', '[', 'R', ':', 'w', '5', 'E', '\\', 'x', '0', 'b', 'q', '>', 'N', 'S', 'C', 'P', ']', 'p', '`', '%', 'M', '7', '6', '8', 'H', 'O', ';', 'j', '9', 'A', 'v'], 'g': ['e', ' ', 's', 'o', 'i', 'r', 'u', 'n', '.', 'a', ')', '_', 'g', 'l', 't', '(', ',', 'h', '\n', "'", '=', 'd', ':', '"', '/', 'I', 'y', '1', 'f', 'N', 'v', '[', 'm', '2', '>', 'L', 'c', 'D', ']', 'z', '-', '\\', 'R', 'T', '0', '`', 'M', 'b', '<', 'F', 'P', 'U', 'k', 'p', '}', '!', '?', 'C', 'S', '|', ';', 'x', 'A', 'E', '#', 'V', 'j', 'w', '$', '&', 'G', 'H', 'O', 'X', 'q'], 'k': ['e', 'i', ' ', '_', 'w', 'a', 's', 't', 'u', 'b', "'", 'l', 'o', '.', '(', '\n', ')', 'n', '=', ',', 'm', 'r', 'd', ']', ':', 'k', 'F', 'y', '"', 'j', '-', '/', '[', 'f', 'c', 'g', 'h', '1', 'E', 'L', '>', 'S', '2', 'N', '`', 'v', 'T', '}', 'I', 'M', '!', '+', 'C', 'P', '\\', '6', ';', 'A', 'K', 'R', 'V', 'x'], 'o': ['n', 'r', 'm', 'd', 'u', 't', 'p', ' ', 'l', 'b', 's', 'f', 'i', 'o', 'c', 'w', '.', 'k', '_', 'a', 'v', 'g', 'e', "'", '/', 'x', '\n', 'j', '"', '-', ',', 'h', ')', '(', 'I', '=', 'y', '<', ':', 'A', '[', '\\', 'S', 'z', 'M', 'F', 'R', '@', 'D', '&', 'J', 'C', ']', '`', '>', 'T', 'V', '+', 'P', 'O', 'q', 'N', 'H', 'W', '%', '2', 'E', 'K', 'Q', 'U', '#', '$', '1', '3', 'B', 'L', '{', '|'], 's': ['e', 't', ' ', 's', '.', 'i', '_', 'u', 'p', 'a', '(', 'o', ')', ',', '\n', 'h', "'", 'c', ':', 'k', '"', '[', 'r', '=', 'l', 'y', 'g', '/', 'q', 'w', '-', 'n', ']', '1', 'f', 'm', '2', 'N', 'z', '3', '\\', 'd', '`', 'C', 'F', '%', '4', 'T', '<', '*', 'M', '>', 'v', 'I', ';', 'j', '0', 'B', 'H', '+', 'S', 'E', 'U', 'b', 'O', 'R', '?', '$', 'V', 'D', '}', '!', 'A', '&', 'L', 'P', '|', '{', 'x', 'G', 'W', '8', '@', '#', 'Z', '5'], 'w': ['i', 'a', 'e', 'o', 'h', 's', ' ', '_', 'r', '.', 'k', 'n', '(', 'w', ')', "'", ',', 'l', '\n', '/', 'd', '1', '"', '[', 'T', '3', 'y', 'M', '=', 'u', '-', 'L', '+', ':', ']', '2', 'A', '4', 'I', 'c', 'Q', '`', 'D', 'H', 'b', '5', '7', '6', '9', '8', 'f', '|', '*', 'V', 'E', 'N', '\\', 'v', '}', 'F', 'g', 't', 'x'], '{': ['%', "'", ' ', '{', '\n', '}', '"', '1', '4', 'y', 'm', 'x', '0', 'f', '|', '#', 'O', 'N', ',', '3', '7', 'M', 'T', '\\', '`', 's', 'u', '(', '-', '2', 'A', 'C', 'P', '[', 'b', 'i', 'o'], '\n': [' ', '\n', 'f', '#', '<', 'c', 'd', 'i', '"', '@', 'g', ')', 'T', 'e', 't', '_', 'S', 's', 'A', 'a', 'C', 'D', 'R', 'r', 'G', 'u', 'p', 'F', 'P', 'w', 'U', 'h', 'E', 'l', 'M', 'm', '}', 'L', 'n', 'I', 'O', 'N', 'W', 'o', 'b', 'H', 'B', 'v', ']', '%', 'Y', '(', '>', '`', 'Q', 'V', "'", '*', '1', 'J', 'X', '-', '.', '3', '2', 'K', '[', 'j'], '"': ['"', '\n', ' ', ')', ',', '>', 't', 'a', ':', 'R', 's', 'T', 'c', ']', '%', '/', '2', '1', 'i', 'E', 'h', '<', "'", 'C', 'p', 'b', 'f', 'd', 'n', 'S', 'r', 'A', 'e', 'm', 'G', 'P', 'D', '.', 'I', '{', 'N', '3', 'x', 'L', 'K', '\\', 'F', 'J', '0', '4', 'W', 'H', 'M', 'U', 'o', '}', '5', '_', 'u', 'V', '&', '8', 'l', '7', '6', '9', 'O', 'v', 'j', 'g', '(', 'B', '-', 'w', ';', 'k', 'Y', '|', 'y', '#', '[', '*', 'q', '^', 'X', '`', '$', '?', '!', '=', '+', 'Z', 'z', '~', '@', 'Q'], '&': ['a', ' ', 'b', 'q', 'g', 'l', 'y', '#', '"', '\\', '&', 'c', 'p', "'", 'm', '<', 'n', '|', '%', ',', '=', 's', '(', '*', '.', 'A', 'f', 'x'], '*': ['*', ' ', 'k', 'a', '-', 'o', 's', '\n', "'", ')', '(', 'f', '/', '?', '{', 'n', 't', '=', 'd', '.', '"', '\\', 'm', ',', 'T', 'e', 'i', '6', 'c', 'p', '$', '2', '|', '>', 'F', 'r', 'u', 'v', '%', 'b', '~', '+', '8', ':', 'N', '_', '`', 'l', '1', '3', '7', '[', 'q', 'w', '9', ';', '@', 'G', 'h'], '.': ['a', '\n', 'c', 'g', 's', '_', ' ', 'p', 'r', 'f', 'm', 't', 'o', 'd', 'e', 'u', '"', 'i', 'n', 'w', 'l', 'h', "'", '.', 'j', 'v', 'b', '0', 'q', 'T', '1', '2', 'M', 'D', '%', 'S', 'P', 'L', '5', 'C', 'k', '4', 'x', '3', 'I', 'V', 'A', 'O', 'U', ',', '\\', 'F', 'y', '<', 'H', '6', ')', 'G', 'E', '7', '*', 'N', '{', 'B', '8', '9', 'R', '/', '[', ']', 'z', 'W', '(', '|', 'Z', ':', 'Q', 'Y', '+', '>', 'X', ';', '@', 'J'], '2': ['0', ',', "'", '3', '1', ' ', '.', '5', ')', '2', '"', '6', ':', '4', '7', 'm', ']', '8', '-', '9', '\n', '<', '(', '_', '/', '}', '>', 'e', 'a', 'n', 'f', '[', 'b', 'd', '=', 'F', '*', 'c', '\\', '&', 'S', '@', 'D', '|', '%', ';', 'M', 'P', 'p', 'A', 'B', 'O', 'L', 'T', 'X', '^', 'i', 'o', 'r', 'x'], '6': ['0', ',', '1', ' ', '4', "'", '3', '2', ')', '"', '7', '_', '5', '9', '.', '6', ':', '<', 'c', '\n', '-', '8', '\\', '(', '/', ']', '}', '=', 'b', '%', ';', 'a', 'f', '*', 'e', 'x', '>', '`', 'd'], ':': ['\n', ' ', '/', ':', ']', '3', '0', '"', "'", '%', '2', 'f', '-', '\\', '<', '1', '5', 'p', 'k', 'i', '6', 'b', '4', 'n', 'e', '[', 'j', '(', '8', 'a', ')', 'l', 'g', '|', '.', 'm', 's', '7', 'M', 't', 'v', 'd', 'u', '{', '=', 'Y', '9', 'B', 'o', 'z', 'c', '#', '$', '*', ';', 'A', 'C', 'G', 'H', 'L', 'N', 'P', '_', '^', '`', 'h', 'q', 'r', 'w', 'y'], '>': ['\n', '<', ' ', "'", '>', '"', '\\', '2', '=', '1', '3', 'R', '%', 'C', 'J', 'N', 'V', 'T', 'I', 'y', 'O', 'E', 'x', 'A', 'h', '-', '.', '4', 'F', 'P', 'S', 'D', 'G', '[', 'p', 'M', 'a', 'e', 's', '0', '5', '{', '|', 'f', 'U', '7', '6', '9', '8', 'Y', '`', 'b', 'n', '&', ')', 'm', '(', ']', 'j', '!', '*', '/', '_', 't', 'v', '$', ',', ':', '?', 'B', 'K', 'L', 'Z', 'd', 'o'], 'B': ['a', 'o', 'L', 'U', 'y', 'A', '_', 'r', "'", 'e', ' ', 'u', 'i', 'R', 'E', 'l', 'O', 'Y', 'C', 'I', 'S', 'K', '\\', '\n', '"', '.', 'M', 'N', 'T', ',', 'B', 'W', ')', '(', '3', 'D', 'F', 'J', '-', 'G', 'Z', 'c', ':', 'H', 'Q', 'P', '/', '8', '<', 'V', 'd', 'j', 'x', 'z', '|'], 'F': ['a', 'i', 'o', 'O', 'I', 'u', 'A', 'e', 'T', 'r', ' ', 'l', '_', 'F', 'R', '-', 'L', "'", 'E', 'C', 'K', 'U', '2', '\n', '*', 'S', '0', 's', '8', 'D', ':', '1', 'Q', 'n', '"', 'M', ']', ')', '/', '.', '3', '<', 'G', 'V', 'c', 'j', 'w', 'x'], 'J': ['o', 'S', "'", 'a', 'u', 'O', '"', 'A', 'e', 's', 'C', 'E', 'I', 'P', 'y', ' ', '.', 'K', 'H', 'U', 'T', 'B', 'G', 'M', 'V', 'i', 'h', 't', 'w'], 'N': ['o', 'T', 'G', 'a', 'D', '_', ' ', 'O', 'E', 'A', 'S', "'", 'I', 'e', 'U', 'C', 'u', 'V', 'K', 'F', 'P', 'L', '\n', 'Y', ',', 'N', ')', '.', 'i', '(', '-', 'B', ':', 'M', 'W', 'H', '[', '`', 'J', '"', 'Q', 'Z', '/', 'R', '|', '4', '=', ']', 'g', 'k', 'y'], 'R': ['e', 'E', 'a', 'I', 'O', ' ', 'L', 'G', 'S', '_', 'M', "'", 'u', 'A', 'i', 'R', 'T', 'D', 'F', 'o', '\n', '=', '[', '"', '(', 'Y', 'N', ',', 'U', ')', ':', 'C', ']', '.', 'V', 'K', 's', 'H', '2', 'h', 'B', 'J', 'P', 'W', '-', '/', ';', 'w', '!', 'j', 'r', 'y', 'z'], 'V': ['a', 'i', 'A', 'E', 'e', 'I', "'", 'o', 'N', 'S', 'r', 'T', 'M', 'R', ' ', 'L', 'O', 'C', 'K', 'u', ',', 'B', 'D', 'J', 'U', 'V', '_', '\n', 'G', 'H', 'P', 'W', 'Y', 'X', 'l', 'y'], 'Z': ["'", ' ', 'E', 'a', 'i', ']', 'A', 'g', 'G', 'o', '\n', 'O', '0', 'C', '(', ':', 'e', 'u', 'D', 'H', 'S', 'r', '"', 'V', 'd', 'l', '!', ')', '.', '2', ';', 'I', 'K', 'M', 'L', 'R', 'Z', '\\', '`', 'v', '|'], '^': ['(', '\\', '"', 's', 'c', "'", 'n', '/', '[', 'a', 'm', 'i', 't', ' ', '$', 'd', 'h', '`', 'r', '|', '%', '@', 'e', 'o', 'p', '>', 'u', 'v', ':', '=', '_', 'l', '&', '3', 'E', 'N', 'S', 'b', 'f'], 'b': ['e', 'j', 'l', 'a', 'o', 'u', 'y', 'i', '.', 'r', 's', '_', ' ', '"', 'c', 'd', ')', "'", 'm', 't', '|', '(', '2', 'g', 'b', '\n', '>', ',', 'p', ']', '-', 'q', 'f', 'n', '1', '3', 'k', '6', '/', '=', ':', '\\', 'z', 'G', '[', 'v', 'M', '5', '`', '&', '8', 'C', 'L', 'w', '0', '7', '9', '<', 'N', 'X', '}'], 'f': ['.', ' ', 'i', 'o', 'r', ')', 'e', ',', 'a', 'u', 'f', 'l', 't', '_', '\n', 'y', 's', '=', 'c', 'n', '(', '-', "'", 'k', ':', 'g', 'd', '[', 'm', 'p', '2', 'j', ']', '/', 'S', 'P', '"', '1', '8', '3', '>', 'N', 'T', '`', 'M', '\\', 'h', '0', 'q', '%', 'E', 'V', '}', '7', '+', '9', 'A', 'C', 'D', 'O', 'b', '!', '5', '6', ';', '?', '@', 'R'], 'j': ['a', 'e', 'o', 's', 'u', '.', ',', ' ', ')', "'", '_', ':', 'i', '(', '\n', '=', ']', 'n', '4', 'd', '[', 'c', '1', '2', 'w', '"', 'h', 'k', 'p', 'y', '3', 'S', '\\', 'q', '/', 'Q', '`', 'r'], 'n': [' ', 't', 'g', 'd', 's', 'e', 'a', 'o', 'c', 'i', '(', '_', 'u', "'", 'n', 'f', '.', 'p', ',', 'v', 'k', '\n', 'l', 'y', ')', '>', '"', '=', '-', ':', 'E', 'm', '/', '<', 'r', 'j', '\\', 'h', 'M', 'W', '[', 'D', 'T', '2', 'S', 'F', 'x', 'b', 'R', 'I', '%', 'w', '`', 'z', '3', 'K', ']', 'B', 'P', '1', ';', 'C', 'A', '0', 'N', '!', '&', '@', '#', '?', 'Y', 'G', 'U', 'X', 'q', 'H', 'Z', '|', '*', '4', 'L', '+', '9', 'O'], 'r': ['e', 't', 'o', ' ', 'i', 'a', 'n', 's', 'm', 'y', '(', 'r', 'g', 'u', '.', '_', 'd', 'l', ',', 'c', "'", ')', '\n', 'k', ':', 'b', 'v', 'f', '"', '=', '-', '/', 'D', 'p', 'w', 'F', 'z', '>', '1', 'h', '`', 'R', 'T', 'M', '2', 'j', '[', '\\', 'L', '<', 'C', 'B', 'E', ']', '0', 'N', 'W', '|', '3', '4', 'H', 'O', '{', 'A', '*', '!', 'P', '}', ';', '?', '@', 'X', '5', 'I', '%', '&', 'G', '#', '6', 'Q', 'U', 'V', '+', 'K'], 'v': ['a', 'e', 'i', 'o', ' ', '6', ')', '4', '_', '[', 's', '.', "'", 'c', 'n', '\n', ',', '/', '>', 'r', ':', '\\', 'u', '=', 'l', 't', '(', 'd', 'g', '1', '2', 'I', '"', ']', 'j', 'm', '-', 'E', 'k', '&', '3', '<', 'C', 'Y', 'h', 'q', 'p', 'w'], 'z': ['e', 'o', 'y', 'i', 'a', ' ', "'", ')', 'k', '(', '_', 'c', 'n', 'z', ':', '"', 'l', '\n', ']', 'h', '.', 't', ',', '2', '=', 'A', 'd', 'f', 'u', 'b', 'g', 's', '/', '`', 'm', 'w', '}', '$', '-', '0', ';', 'j', '>'], '~': ['|', '~', "'", '=', ' ', '"', '0', '!', 'f', ']', 'o', '/'], '!': ['=', '"', '~', '\n', "'", ' ', '\\', '/', '[', '<', '|', '#', '!', '%', ')', '$', '*', '-', '.', '?', '@', 'p'], '%': [' ', 's', '}', 'd', '(', 'r', '\n', 'Y', 'm', 'M', '%', 'H', 'S', 'p', 'I', 'b', 'B', 'C', 'y', '0', '2', '9', 'A', '"', '3', '8', 'E', 'w', "'", '.', '\\', 'f', 'i', '&', ')', ',', '[', '^', '`'], ')': ['\n', ':', ',', ')', ' ', '.', ']', "'", '}', 's', '[', '"', '(', '/', '?', ';', '\\', '+', '`', '|', '*', '-', '>', '$', 'd', 'f', '^', '<', '#', '&', '!', '%', '@', 'E', 'L', 'P', 'R', 'e', 'i', 'r', '{'], '-': ['-', ' ', '1', 's', 't', 'c', '2', '0', '8', 'v', 'n', 'l', 'd', 'm', 'e', 'i', '*', 'b', 'a', '%', 'r', '\n', '9', 'T', 'p', 'u', 'f', 'M', '+', 'I', "'", '3', 'o', 'D', 'C', 'O', '=', 'A', 'L', 'k', 'z', '4', '\\', 'w', 'Z', '<', 'g', '>', 'h', 'S', 'B', '5', 'q', 'F', 'P', '6', 'E', ']', '(', '#', '"', 'H', 'j', 'x', 'W', 'R', 'V', ')', 'G', '_', 'J', '?', 'K', 'N', 'U', '7', '.', '~', '!', '@', 'X', '`', '|', '/', '[', 'y'], '1': ['0', ',', ')', '2', ' ', "'", '.', '"', '3', '1', ']', '6', ':', '7', '4', '\n', '8', '9', '-', '<', '5', '/', '_', '\\', '(', 'f', '>', 'e', '[', '&', '=', 'c', 'F', 'a', 'k', ';', 'b', 'i', '|', 'Q', 'd', '+', 'P', 'n', 't', 'r', '}', 'E', 'o', 'p', 's', '{', '%', '*', 'A', '@', 'g', 'h', 'x'], '5': [' ', ',', "'", '0', ')', '2', '1', '5', '"', '6', '.', '3', '4', '7', '<', '9', '\n', ':', 'D', 'b', '/', '8', '_', ']', '(', '\\', '-', ';', 'd', '+', '*', 'e', 's', 'P', 'a', '%', '?', 'C', 'M', 'X', 'l', 'p'], '9': ['9', '2', '0', '8', ',', '"', "'", '7', ' ', '4', ')', '1', ']', '.', '6', '<', '-', '5', '3', '_', '/', ';', '\n', ':', 'I', 'a', '(', 'A', 'c', 'b', 'd', '+', 'C', '!', '#', '%', 'T', '\\', 'e', 'f'], '=': [' ', '"', '=', "'", 'N', 'T', 'F', '1', 's', '{', '2', '(', '[', '0', 'u', 'c', 'd', '%', '3', 'e', 'D', 'v', 'f', 'g', 'l', 'p', '5', 'a', 'm', 'i', 'o', '|', '4', '&', 'n', 't', 'b', 'r', '\n', '-', 'M', 'A', 'L', '/', '6', 'W', '7', 'C', 'S', '\\', 'P', '8', 'B', 'G', 'R', 'U', 'h', 'k', 'q', 'w', '+', '.', 'V', '`', '<', 'E', '_', '>', 'y', '?', 'I', 'H', 'J', ',', ';', ':', 'Q', 'z'], 'A': ['L', 'T', 'r', ' ', 'S', 'R', 'N', 'd', 'U', 'u', 'l', 'M', 'C', 't', 'P', 'G', 'n', "'", 'D', 'B', 's', 'p', 'I', 'X', 'c', 'g', '.', '(', '-', 'Y', '_', 'm', 'b', 'V', 'K', 'v', 'k', 'J', '1', '[', ',', 'f', 'A', 'Z', '4', ':', 'i', 'j', '\n', 'F', 'W', 'y', '0', '2', 'E', 'Q', 'o', '"', ')', 'O', 'w', 'z', '3', '5', '<', '\\', 'a', '`', 'e', 'q', 'x'], 'E': ['q', 'r', 'x', 'O', 'R', 'S', '_', 'T', 'n', 'N', ' ', 'P', 'M', "'", 'D', 'F', 'L', 'X', 'C', 'm', 'A', 'W', ')', ',', ']', 'B', 'l', 'Q', 'v', 'G', '\n', 'a', 'Y', ':', 's', '.', '=', '"', 'V', '(', '-', 'g', 'E', '>', 'u', '6', 'i', ';', 'd', '2', '+', '4', 'U', '\\', '`', 'c', 'e', 'h', 't', '!', '1', '3', 'I', 'K', '[', 'Z', 'p', 'y'], 'I': ['n', 'N', 'f', 'm', 'O', 'T', 'D', 'P', 'C', 'L', 'R', 'A', 'S', 't', 'E', ' ', 'M', 's', "'", 'G', 'F', 'I', 'd', 'X', 'Z', 'B', '_', '1', 'V', ':', 'g', 'l', ')', 'c', '-', '\n', '.', 'r', ',', 'H', 'Q', 'i', 'v', 'y', '/', 'a', '&', 'U', 'W', 'b', 'o', 'w', 'z', '"', 'Y', 'k'], 'M': ['a', 'L', 'o', 'E', 'e', 'u', 'A', 'i', 'y', 'P', 'I', 'S', 'O', 's', '_', "'", ' ', 'U', 'M', 'B', '[', 'T', ':', '"', 'D', 'N', '.', 'Z', '-', '\n', 'C', 'H', 'K', '%', '2', 'G', 'R', ')', '/', 'V', 'Y', 'j', 'p'], 'Q': ['L', 'u', 'U', "'", 'a', '-', 'C', 'G', 'D', '.', ')', '(', '\n', '1', '2', 'V', ','], 'U': ['s', 'R', 'T', 'n', 'L', 'p', 'S', 'N', 'M', 'E', 'G', "'", 'A', 'P', 'I', 't', 'r', 'D', 'C', 'B', 'O', 'K', 'Z', 'u', ' ', ',', 'F', 'V', '_', 'd', 'l', '"', ')', '(', '.', ':', 'a', 'c', 'g', 'i', 'k', 'm'], 'Y': [' ', 'o', '_', "'", '[', 'P', '"', 'G', '-', 'Y', 'T', ',', 'a', 'e', '/', 'S', 'W', '\n', 'A', 'E', '(', 'C', 'I', 'M', 'R', 'l', ')', ']', 'u', '%', ':', 'i', '.', '=', 'J', 'N', 'U', 'Z', 'p', 'v'], ']': ['\n', ')', ',', ' ', '.', ']', '[', ':', "'", '}', '+', '*', '/', '|', '"', '\\', '(', '-', '>', '{', '?', 'H', 'b', '=', '`', 't'], 'a': ['t', 'l', 's', 'n', 'r', 'm', ' ', 'g', 'c', 'd', 'i', 'p', 'b', 'u', 'v', 'y', "'", 'k', 'f', '.', 'x', '_', '"', ')', 'z', '\n', 'w', '(', '=', '|', ',', ':', 'a', '[', 'h', 'j', '&', '1', '>', 'S', '-', 'e', '/', '2', '0', '3', '\\', '4', '5', 'T', '<', 'N', 'L', '*', '6', 'U', 'W', 'o', 'q', 'F', 'M', '`', '%', '+', '7', '9', '8', 'C', 'H', '}', ';', '#', '?', '@', 'Z', ']'], 'e': ['l', 'r', ' ', 's', 't', 'n', 'd', 'c', 'x', 'f', '(', '_', 'a', 'm', '.', ',', ')', '\n', '=', 'p', 'o', ':', "'", 'w', 'q', 'g', 'e', 'y', 'v', 'E', '"', 'i', 'F', '/', 'S', 'b', '-', ']', '[', 'C', 'D', 'T', '`', 'U', 'z', '2', 'u', 'h', 'k', 'M', '1', 'R', 'N', 'L', 'j', '\\', 'V', '<', '0', 'I', '6', '>', '8', 'P', 'W', '3', '|', 'H', '}', 'Q', 'A', ';', '4', '?', 'O', '@', 'G', 'B', '7', '&', '!', 'K', '5', 'Z', 'X', '+', '{', '*', '%', 'J', '9'], 'i': ['n', 's', 'o', 't', 'e', 'l', 'm', 'f', 'c', 'd', 'a', 'r', 'p', 'g', 'v', 'b', "'", 'z', 'x', '>', ' ', '.', 'k', 'q', '"', ']', ')', '_', ',', 'i', 'P', 'j', '1', ':', '\n', 'V', '(', 'W', '-', '\\', 'L', 'h', 'u', 'y', '/', '+', '2', 'w', '<', 'N', '}', '|', 'A', 'G', 'J', '=', '`', 'D', '[', '%', '&', '3', 'I', 'M'], 'm': ['e', 'p', 'a', 'o', ' ', 's', 'i', 'm', '_', 'b', 'u', 'l', '.', "'", '(', 'y', ',', '/', 'n', ')', '2', 'E', '\n', 'S', 't', '-', 'T', '=', '"', ':', 'c', '1', 'd', 'f', 'g', '>', 'W', ']', 'F', 'L', 'Q', 'M', '[', 'k', '`', 'C', 'N', 'z', '3', '?', 'A', 'r', 'O', 'R', 'P', 'U', 'v', '<', '@', 'h', '\\', 'D', 'q', 'x', '%', '0', '}', '$', '+', ';', 'G', 'I', 'H', 'V', 'w', '|'], 'q': ['u', 'l', '_', 'n', '.', 's', ')', ' ', '1', '2', '(', '\n', '=', ',', 'd', "'", '>', '[', 'r', '-', '0', ']', 'i', '|', '"', '\\', 'c', 'f', '}'], 'u': ['r', 'e', 't', 'l', 'n', 's', 'a', "'", 'p', 'm', 'i', 'b', 'g', '"', 'c', 'd', '0', 'o', ' ', 'f', 'u', '.', 'k', 'h', 'v', '\\', '1', '2', 'z', 'j', 'y', '\n', ']', 'x', '%', '(', '-', 'w', ')', '3', 'X', '_', ',', '&', '/', '4', '?', 'E', 'O', 'q'], 'y': [' ', 'p', 's', '_', '.', ':', '(', 't', '\n', ',', 'e', ')', "'", 'l', 'o', 'i', '=', 'g', '"', 'n', 'm', 'w', 'd', 'W', 'a', 'r', 'C', 'F', 'c', '-', 'T', 'E', '<', '/', 'M', 'S', ']', '[', 'b', 'k', 'R', 'O', 'y', 'f', '1', '`', 'z', '>', 'P', '2', 'G', 'I', 'h', 'D', '0', 'U', 'V', ';', 'j', 'B', 'u', '!', '4', 'v', '}', '*', 'L', '3', '?', 'A', 'N', 'Q', '&', '+', 'H', '%', '$', '@', 'Y', 'Z', '\\', '|'], '}': ['\n', '}', ')', ',', "'", '{', ' ', '"', '.', '`', ']', '<', '|', 'F', ';', ':', 'L', '(', 'N', '[', '/', '>', '#', '$', 'H', '\\', 'i', 'o']}
characters_by_freq_python = [' ', 'e', 't', 's', 'a', 'r', 'o', 'i', 'n', 'l', '\n', 'd', 'f', 'u', 'c', 'm', 'p', '_', '.', "'", '(', ')', ',', 'h', 'g', '"', '=', ':', 'y', 'b', 'v', 'w', '#', 'E', 'x', 'k', 'T', '/', '0', '1', 'S', 'j', 'R', '-', '2', 'N', 'q', 'I', 'A', '%', ']', '[', 'C', 'O', 'F', '>', 'L', '<', 'D', 'M', 'P', 'G', '3', '{', '}', 'U', '4', '`', 'z', '*', '\\', 'H', '5', 'B', '6', '8', 'V', 'W', '+', '7', '9', 'K', '|', 'Y', ';', '&', 'J', '@', '?', 'X', 'Q', '!', '^', 'Z', '$', '~']


diagraphs = diagraphs_english_no_nl
characters_by_freq = characters_by_freq_english_no_nl

class FrequencyCharacter(BlindCharacter):
    def __init__(self,previous_char,*args,**kwargs):
        self.previous_char = previous_char
        super(FrequencyCharacter,self).__init__(*args,**kwargs)

    @utilities.debug 
    def run(self):
        #make note of the current greenlet
        self.run_gl = gevent.getcurrent()

        self.working = True        

        tried = []
        chars_to_try = copy(characters_by_freq)
        previous_char_finished = False

        success = False

        while not success and len(chars_to_try):
            if not previous_char_finished and self.previous_char and self.previous_char == "success":
                chars_to_try = filter(lambda c: c not in tried,diagraphs[self.previous_char.char_val])
                previous_char_finished = True

            self.char_val = chars_to_try.pop(0)

            if self._test("="):
                success = True

            tried.append(self.char_val)

            gevent.sleep(0)

        if not success:
            self.error = True
            self.row_die.set((self.char_index,AsyncResult()))
            
        self.done = True
        self.working = False

        #clear the note regarding the running greenlet
        self.run_gl = None


class FrequencyTechnique(BooleanBlindTechnique):
    def _character_generator(self,row_index):
        '''
        creates a Character object for us. this generator is useful just because it keeps track of the char_index
        '''
        char_index = 1
        row_die_event = AsyncResult()

        previous_char = None

        while True:
            c = FrequencyCharacter(\
                row_index     = row_index,\
                char_index    = char_index,\
                queue         = self.q,\
                row_die       = row_die_event,\
                previous_char = previous_char)
            # note the previous char
            previous_char = c
            #increment our char_index
            char_index += 1
            #fire off the Character within our Pool.
            self.character_pool.spawn(c.run)
            yield c

    def _adjust_row_lengths(self):
        ''' 
        if a row is full of "success", but we havent reached the end yet (the last elt isnt "error")
        then we need to increase the row_len.
        '''
        while not self.shutting_down.is_set():
            self.results_lock.acquire()

            for row_index in range(len(self.results)):
                #if the row isn't finished or hasn't been started yet, we add Character()s to the row
                if not len(self.results[row_index]) or ('error' not in self.results[row_index] and self.results[row_index][-1] == "success"):
                    self.results[row_index].append(self.char_gens[row_index].next())

            self.results_lock.release()
            gevent.sleep(.3)

    def _add_rows(self):
        '''
        look at how many gevent "threads" are being used and add more rows to correct this
        '''
        row_index = 0
        while self.need_more_rows:
            # add rows until we realize that we are at the end of rows
            working_rows = len(filter(lambda row: 'working' in row,self.results))
            for row in range(self.concurrency - working_rows):
                self.char_gens.append(self._character_generator(row_index))
                self.results.append([])
                row_index += 1

            gevent.sleep(.3)
            self.need_more_rows = not(len(self.results) and filter(lambda row: len(row) and row[0] == 'error',self.results))
        
        while not self.shutting_down.is_set():
            self.results_lock.acquire()
            # delete any rows that shouldn't have been added in the first place
            errored = filter(lambda ri: len(self.results[ri]) and self.results[ri][0] == 'error',range(len(self.results)))
            if errored:
                end = min(errored)
                for ri in xrange(len(self.results)-1,end-1,-1):
                    del(self.results[ri])

            self.results_lock.release()    
            #if there aren't going to be any more rows in need of deletion we can stop this nonsense
            if self.results and self.results[-1][0] == 'success':
                break
            gevent.sleep(.3)