/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file server.c
 * \brief Server handling
 */

#include <ffgtk.h>

#ifndef G_OS_WIN32
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#else
#include <winsock2.h>
typedef int in_addr_t;
#endif

#define BUFFER_LENGTH	1024

/**
 * \brief Read transmission and process it
 * \param nFile file id
 * \param nSock socket id
 * \param pnFile temporary file name
 */
static void readSocket( int nFile, int nSock, char *pnFile ) {
	char anBuffer[ BUFFER_LENGTH ];
	ssize_t nReadResult;
	ssize_t nWriteResult;
	ssize_t nWritten;

	while ( ( nReadResult = read( nSock, anBuffer, BUFFER_LENGTH ) ) > 0 ||
		( nReadResult == -1 && errno == EINTR ) ) {
		if ( nReadResult > 0 ) {
			nWritten = 0;
			do {
				nWriteResult = write( nFile, anBuffer + nWritten, nReadResult );
				if ( nWriteResult > 0 ) {
					nWritten += nWriteResult;
					nReadResult -= nWriteResult;
				}
			} while ( nReadResult && ( nWriteResult != -1 || errno == EINTR ) );
		}
	}

	if ( !nReadResult ) {
		Debug( KERN_DEBUG, _( "Print job received on socket\n" ) );

		ffgtkLock();
		processFax( pnFile );
		ffgtkUnlock();
	} else {
		Debug( KERN_WARNING, _( "Error in receiving print job on socket\n" ) );
	}
}

/**
 * \brief Initialize server and wait for incoming transmissions
 * \param data data pointer
 * \return error code
 */
gpointer server( gpointer data ) {
	int nSock;
	struct sockaddr_in sAddress;
	struct sockaddr_in sConnectAddress;
	in_addr_t sLocalHost;
	socklen_t nClientLen = sizeof( sConnectAddress );
	char anTest[ 256 ];

	if ( ( nSock = socket( AF_INET, SOCK_STREAM, 0 ) ) < 0 ) {
		Debug( KERN_WARNING, _( "could not open socket, abort\n" ) );
		return NULL;
	}

	memset( &sAddress, 0, sizeof( sAddress ) );

	sAddress.sin_family = AF_INET;
	sAddress.sin_addr.s_addr = htonl( INADDR_ANY );
	sLocalHost = inet_addr( "127.0.0.1" );
	if ( sLocalHost != ( in_addr_t )( -1 ) ) {
		sAddress.sin_addr.s_addr = sLocalHost;
	}

	sAddress.sin_port = htons( atoi( "9901" ) );

	if ( ( bind( nSock, ( struct sockaddr * ) &sAddress, sizeof( sAddress ) ) ) < 0 ) {
		Debug( KERN_WARNING, _( "could not bind socket, abort\n" ) );
		return NULL;
	}

	if ( ( listen( nSock, 5 ) ) < 0 ) {
		Debug( KERN_WARNING, _( "could not listen on socket, abort\n" ) );
		return NULL;
	}

	Debug( KERN_DEBUG, _( "Fax Server running on port 9901\n" ) );

	memset( &sConnectAddress, 0, sizeof( sConnectAddress ) );

	while ( 1 ) {
		int nConnectSock = accept( nSock, ( struct sockaddr * ) &sConnectAddress, &nClientLen );

		if ( nConnectSock >= 0 ) {
			snprintf( anTest, sizeof( anTest ), "%s/fax-XXXXXX", g_get_tmp_dir() );
			int nFileId = g_mkstemp( anTest );
			if ( nFileId == -1 ) {
				Debug( KERN_WARNING, _( "Can't open temporary file\n" ) );
			} else {
				readSocket( nFileId, nConnectSock, anTest );
				while ( close( nFileId ) == -1 && errno == EINTR );
			}
			while ( close( nConnectSock ) == -1 && errno == EINTR );
		} else {
			Debug( KERN_WARNING, _( "There is a problem with receiving a connection on the socket server\n" ) );
		}
	}

	return NULL;
}

/**
 * \brief Initialize print server thread
 */
void PrintServerInit( void ) {
	CREATE_THREAD( "printserver", server, NULL );
}
