package tests::RecordsTest;

use strict;

use base qw/Lire::Test::TestCase tests::TestStoreFixture/;

use Lire::Records;
use Lire::ReportSpec;
use Lire::Report::Group;
use Lire::Report::TableInfo;
use Lire::DlfQuery;

sub new {
    my $self = shift()->SUPER::new( @_ );

    $self->init();

    return $self;
}

sub set_up {
    my $self = shift->SUPER::set_up();

    $self->set_up_test_schema();

    $self->set_up_tz( 'UTC' );

    $self->{'spec'} = new Lire::ReportSpec();
    $self->{'spec'}->superservice( 'test' );
    $self->{'spec'}->id( 'records-test' );

    $self->{'records'} = new Lire::Records( 'report_spec' => $self->{'spec'},
                                            'fields' => [ 'time_start', 'user',
                                                          'file_size', 'time_taken' ],
                                          );
    $self->{'spec'}->calc_spec( $self->{'records'} );

    return $self;
}

sub tear_down {
    my $self = $_[0];
    $self->SUPER::tear_down();

    return;
}

sub test_ops {
    my $self = $_[0];

    my $records = $self->{'records'};
    $self->assert_deep_equals( [], $records->ops() );

    $self->assert_died( sub { $records->ops( [] ) },
                        qr/records cannot contain any children/ );
}

sub test_name {
    my $self = $_[0];

    $self->assert_equals( 'records:4',
                          $self->{'records'}->name() )
}

sub test_create_categorical_info {
    my $self = $_[0];

    my $info = new Lire::Report::TableInfo();
    $self->{'records'}->create_categorical_info( $info );

    my @fields = $info->children();
    $self->assert_equals( 4, scalar @fields );

    my $infos = [ [ 'time_start', 'timestamp', "Start Time" ],
                  [ 'user', 'string', "Username" ],
                  [ 'file_size', 'bytes', 'Size' ],
                  [ 'time_taken', 'duration', 'Duration' ] ];
    for ( my $i=0; $i< @$infos; $i++ ) {
        my $col = $fields[$i];
        $self->assert_equals( 'categorical', $col->class() );
        $self->assert_equals( $infos->[$i][0], $col->name() );
        $self->assert_equals( $infos->[$i][1], $col->type() );
        $self->assert_equals( $infos->[$i][2], $col->label() );
    }
}

sub test_build_query {
    my $self = $_[0];

    my $e_query = new Lire::DlfQuery( 'test' );
    $e_query->add_aggr_field( '_lr_nrecords', 'count(*)' );
    foreach my $f ( qw/time_start user file_size time_taken/ ) {
        $e_query->add_field( $f );
    }
    $e_query->set_sort_spec( 'time_start' );

    my $query = new Lire::DlfQuery( 'test' );
    $self->{'records'}->build_query( $query );
    $self->assert_deep_equals( $e_query, $query );
}

sub test_create_entry {
    my $self = $_[0];

    my $group = new Lire::Report::Group( bless( {}, 'Lire::Report::Entry' ),
                                         $self->{'spec'}->create_table_info() );

    my $dlf = {
               'time_start' => 1065124732,
               'user' => 'flacoste',
               'file_size' => '2048',
               'time_taken' => '15',
              };

    $self->{'cfg'}{'lr_scale_numbers'} = 0;
    $self->{'cfg'}{'lr_scale_bytes'} = 1;
    $self->{'cfg'}{'lr_scale_seconds'} = 1;

    my $entry = $self->{'records'}->create_entry( $group, $dlf );
    $self->assert_not_null( $entry, "create_entry() returned undef" );

    my @e_data = ( ['2003-10-02 20:58:52',  1065124732 ],
                   ['flacoste', 'flacoste' ],
                   [ '2k', 2048 ],
                   [ '15s', 15 ],
                 );
    my @data = $entry->data();
    $self->assert_equals( 4, scalar @data );
    for ( my $i=0; $i < @data; $i++ ) {
        my $c = $data[$i];
        $self->assert_equals( 'name', $c->{'type'} );
        $self->assert_equals( $e_data[$i][0], $c->{'content'} );
        $self->assert_equals( $e_data[$i][1], $c->{'value'} );
    }
}

1;
