/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <errno.h>
#include <kstat.h>
#include <strings.h>

static kstat_t* get_next_ksp(kstat_t* ksp);
static int assign_dst(struct sa_disk* dst, kstat_t* ksp) SA_NONNULL;

__thread kstat_ctl_t* disk_kc;

int sa_open_disk() {
	disk_kc = kstat_open();
	if (disk_kc == NULL)
		return ENOSYS;

	return 0;
}

int sa_close_disk() {
	if (disk_kc != NULL)
		kstat_close(disk_kc);

	return 0;
}

int sa_count_disks(uint16_t* number) {
	if (number == NULL)
		return EINVAL;

	if (kstat_chain_update(disk_kc) == -1)
		return errno;

	kstat_t* ksp = NULL;
	*number = 0;

	while ((ksp = get_next_ksp(ksp)) != NULL)
		++(*number);

	return 0;
}

int sa_reset_disks() {
	if(kstat_chain_update(disk_kc) == -1)
		return errno;

	return 0;
}

int sa_get_disks_ids(char* dst, uint16_t dst_size, uint16_t* written) {
	if (dst == NULL || dst_size == 0 || written == NULL)
		return EINVAL;

	uint16_t i;
	kstat_t* ksp = NULL;
	*written = 0;
	for (i = 0; (ksp = get_next_ksp(ksp)) != NULL; ++i) {
		if (i == dst_size)
			return ENOMEM;

		strncpy(&dst[i * SA_DISK_NAME], ksp->ks_name, SA_DISK_NAME);
		++(*written);
	}
	return 0;
}

int sa_get_disk(char* name, struct sa_disk* dst) {
	if (name == NULL || dst == NULL)
		return EINVAL;

	kstat_t* ksp = NULL;
	while ((ksp = get_next_ksp(ksp)) != NULL) {
		if (strncmp(ksp->ks_name, name, sizeof ksp->ks_name) != 0)
			continue;

		if (assign_dst(dst, ksp) != 0)
			return ENOSYS;
		return 0;
	}
	return ENODEV;
}

int sa_get_disks(struct sa_disk* dst, uint16_t dst_size, uint16_t* written) {
	if (dst == NULL || dst_size == 0 || written == NULL)
		return EINVAL;

	uint16_t i;
	kstat_t* ksp = NULL;
	*written = 0;
	for (i = 0; (ksp = get_next_ksp(ksp)) != NULL; ++i) {
		if (i == dst_size)
			return ENOMEM;

		if (assign_dst(&dst[i], ksp) != 0)
			return ENOSYS;
		++(*written);
	}
	return 0;
}

static kstat_t* get_next_ksp(kstat_t* ksp) {
	do {
		if (ksp == NULL)
			ksp = disk_kc->kc_chain;
		else
			ksp = ksp->ks_next;

		if (ksp == NULL)
			return ksp;
	} while (ksp->ks_type != KSTAT_TYPE_IO);

	return ksp;
}

static int assign_dst(struct sa_disk* dst, kstat_t* ksp) {
	kstat_io_t kio;
	if (kstat_read(disk_kc, ksp, &kio) == -1)
		return errno;

	strncpy(dst->name, ksp->ks_name, sizeof dst->name);
	dst->reads = kio.reads;
	dst->writes = kio.writes;
	dst->bytes_read = kio.nread;
	dst->bytes_written = kio.nwritten;
	return 0;
}
