/*
 * This file is part of Flowplayer, http://flowplayer.org
 *
 * By: Daniel Rossi <electroteque@gmail.com>, Anssi Piirainen <api@iki.fi> Flowplayer Oy
 * Copyright (c) 2009, 2010 Electroteque Multimedia, Flowplayer Oy
 *
 * Released under the MIT License:
 * http://www.opensource.org/licenses/mit-license.php
 */

package org.flowplayer.bwcheck {
import de.betriebsraum.video.BufferCalculator;

import flash.display.DisplayObject;
import flash.events.NetStatusEvent;
import flash.net.NetStream;
import flash.utils.Dictionary;

import fp.HDSymbol;

import org.flowplayer.bwcheck.net.OsmfLoggerFactory;
import org.flowplayer.bwcheck.net.OsmfNetStreamClient;
import org.flowplayer.net.BitrateItem;
import org.flowplayer.net.StreamSelectionManager;
import org.flowplayer.net.IStreamSelectionManager;
import org.flowplayer.net.StreamSwitchManager;
import org.flowplayer.net.BitrateResource;

import org.flowplayer.bwcheck.config.Config;
import org.flowplayer.bwcheck.detect.BandwidthDetectEvent;
import org.flowplayer.bwcheck.detect.BandwidthDetector;
import org.flowplayer.bwcheck.net.ScreenSizeRule;
import org.flowplayer.bwcheck.net.BWStreamSelectionManager;


import org.flowplayer.controller.ClipURLResolver;
import org.flowplayer.controller.NetStreamClient;
import org.flowplayer.controller.StreamProvider;
import org.flowplayer.model.Clip;
import org.flowplayer.model.ClipEvent;
import org.flowplayer.model.PlayerEvent;
import org.flowplayer.model.Plugin;
import org.flowplayer.model.PluginEventType;
import org.flowplayer.model.PluginModel;
import org.flowplayer.util.PropertyBinder;
import org.flowplayer.view.AbstractSprite;
import org.flowplayer.view.Flowplayer;


import org.osmf.net.DynamicStreamingItem;
import org.osmf.net.DynamicStreamingResource;
import org.osmf.net.NetStreamSwitchManager;
import org.osmf.net.NetStreamMetricsBase;
import org.osmf.net.SwitchingRuleBase;

CONFIG::LOGGING
{
    import org.osmf.logging.Log;
}

CONFIG::enableRtmpMetrics {
    import org.osmf.net.rtmpstreaming.RTMPNetStreamMetrics;
    import org.osmf.net.rtmpstreaming.DroppedFramesRule;
    import org.osmf.net.rtmpstreaming.InsufficientBandwidthRule;
    import org.osmf.net.rtmpstreaming.InsufficientBufferRule;
    import org.osmf.net.rtmpstreaming.SufficientBandwidthRule;
}

CONFIG::enableHttpMetrics {
    import org.osmf.net.rtmpstreaming.DroppedFramesRule;
    import org.osmf.net.httpstreaming.HTTPNetStreamMetrics;
    import org.osmf.net.httpstreaming.DownloadRatioRule;
    import org.osmf.net.httpstreaming.HTTPNetStream;
}

public class BwCheckProvider extends AbstractSprite implements ClipURLResolver, Plugin {

    private var _config:Config;
    private var _netStream:NetStream;
    private var _resolveSuccessListener:Function;
    private var _failureListener:Function;
    private var _clip:Clip;
    private var _hasDetectedBW:Boolean = false;
    private var _start:Number = 0;
    private var _model:PluginModel;
    private var _player:Flowplayer;
    private var _resolving:Boolean;
    private var _playButton:DisplayObject;
    private var _provider:StreamProvider;
    private var _bitrateStorage:BitrateStorage;
    private var _detector:BandwidthDetector;

    private var _switchManager:NetStreamSwitchManager;
    private var _netStreamMetrics:NetStreamMetricsBase;
    private var dsResource:DynamicStreamingResource;

    private var _streamSelectionManager:IStreamSelectionManager;
    private var _streamSwitchManager:StreamSwitchManager;


    public function onConfig(model:PluginModel):void {
        log.debug("onConfig(_)");

        CONFIG::LOGGING
        {
        Log.loggerFactory = new OsmfLoggerFactory();
        }

        _config = new PropertyBinder(new Config()).copyProperties(model.config) as Config;

        _model = model;
        _bitrateStorage = new BitrateStorage(_config.bitrateProfileName, "/");
        _bitrateStorage.expiry = _config.cacheExpiry;

        log.debug("onConfig(), dynamic " + _config.dynamic);
    }

    private function applyForClip(clip:Clip):Boolean {
        var bw:Object = clip.getCustomProperty("bwcheck");
        log.debug("applyForClip() ? " + bw);
        if (bw is Boolean && ! bw) return false;
        return true;
    }

    private function canSwitchOnFullscreen():Boolean {
        //disabling fullscreen switching on dynamic for now  #336
        return _config.switchOnFullscreen && _player.streamProvider.type != "http" && !_config.dynamic;
    }

    public function onLoad(player:Flowplayer):void {
        log.debug("onLoad()");
        _player = player;
        lookupProvider(player.pluginRegistry.providers);

        _detector = new BandwidthDetector(_model, _config, _player.playlist);
        _detector.addEventListener(BandwidthDetectEvent.DETECT_COMPLETE, onDetectorComplete);
        _detector.addEventListener(BandwidthDetectEvent.CLUSTER_FAILED, onClusterFailed);

        if (canSwitchOnFullscreen()) {
            _player.onFullscreen(onFullscreen);
            _player.onFullscreenExit(onFullscreen);
        }

        _player.playlist.onSwitch(function(event:ClipEvent):void {
            //fixes for #336 provide correction bitrateitem information when using dynamic stream switching
            //fixes for #352 for wowza streams with secure names that return the real name from the server, return the item from the metrics index instead.
            var newItem:DynamicStreamingItem = _streamSelectionManager.fromName(String(event.info));
            log.info("new item is " + newItem + ", (" + event.info + "), current " + _streamSelectionManager.currentBitrateItem);
            _model.dispatch(PluginEventType.PLUGIN_EVENT, "onStreamSwitchBegin", newItem, _streamSelectionManager.currentBitrateItem);
        });

        _player.playlist.onSwitchFailed(function(event:ClipEvent):void {
            log.info("Transition failed with error " + event.info2.toString());
            _model.dispatch(PluginEventType.PLUGIN_EVENT, "onStreamSwitchFailed", "Transition failed with error " + event.info2.toString());
        });

        _player.playlist.onSwitchComplete(function(event:ClipEvent):void {
            log.info("Stream switch completed with item " + _streamSelectionManager.currentBitrateItem);
            _model.dispatch(PluginEventType.PLUGIN_EVENT, "onStreamSwitch", _streamSelectionManager.currentBitrateItem);
        });


        _player.playlist.onStart(function(event:ClipEvent):void {
            log.debug("onStart() clip == " + clip);
            var clip:Clip = event.target as Clip;

            if (alreadyResolved(clip)) {

                init(clip.getNetStream(), clip);

                //only setup dynamic stream switching in these conditions
                if (_config.dynamic && _streamSelectionManager is BWStreamSelectionManager) {
                    initQoS(clip.getNetStream(), clip);
                }
            }
        }, applyForClip);

        var autoSwitch:Function = function(enable:Boolean):Function {
            return function(event:ClipEvent):void {
                if (! _switchManager) return;
                var newVal:Boolean = _config.dynamic && enable;
                log.info("setting QOS state to " + newVal);
                _switchManager.autoSwitch = newVal;

            }
        };

        _player.playlist.onPause(autoSwitch(false), applyForClip);
        _player.playlist.onStop(autoSwitch(false), applyForClip);
        _player.playlist.onStart(autoSwitch(true), applyForClip);
        _player.playlist.onResume(autoSwitch(true)),applyForClip;
        _player.playlist.onFinish(autoSwitch(false), applyForClip);


        _model.dispatchOnLoad();
    }

    private function onFullscreen(event:PlayerEvent):void {
        log.debug("onFullscreen(), checking bandwidth and switching stream");
        checkBandwidthIfNotDetectedYet();
    }

    private function alreadyResolved(clip:Clip):Boolean {
        return clip.getCustomProperty("bwcheckResolvedUrl") != null;
    }

    protected function hasDetectedBW():Boolean {
        if (! _config.rememberBitrate) return false;
        if (_hasDetectedBW) return true;
        if (isRememberedBitrateValid()) return true;
        return false;
    }

    public function set onFailure(listener:Function):void {
        _failureListener = listener;
    }

    public function handeNetStatusEvent(event:NetStatusEvent):Boolean {
        return true;
    }

    private function detect():void {
        log.debug("connectServer()");
        _detector.detect();
    }

    private function onClusterFailed(event:BandwidthDetectEvent):void {
        log.error("onClusterFailed(), will use default bitrate");
        useDefaultBitrate();
    }

    private function onDetectorComplete(event:BandwidthDetectEvent):void {
        log.debug("onDetectorComplete()");
        event.stopPropagation();
        log.info("\n\n kbit Down: " + event.info.kbitDown + " Delta Down: " + event.info.deltaDown + " Delta Time: " + event.info.deltaTime + " Latency: " + event.info.latency);
        _hasDetectedBW = true;

        // Set the detected bandwidth
        var bandwidth:Number = event.info.kbitDown;
        var mappedBitrate:BitrateItem = getMappedBitrate(bandwidth);
        log.debug("bandwidth (kbitDown) " + bandwidth);
        log.info("mapped to bitrate " + mappedBitrate.bitrate);
        rememberBandwidth(bandwidth);

        selectBitrate(mappedBitrate, bandwidth);
    }

    private function getMappedBitrate(bandwidth:Number = -1):BitrateItem {
        return _streamSelectionManager.getStream(bandwidth) as BitrateItem;
    }

    private function dynamicBuffering(mappedBitrate:Number, detectedBitrate:Number):void {
        if (_config.dynamicBuffer) {
            _clip.onMetaData(function(event:ClipEvent):void {
                _clip.bufferLength = BufferCalculator.calculate(_clip.metaData.duration, mappedBitrate, detectedBitrate);
                log.info("Dynamically setting buffer time to " + _clip.bufferLength + "s");
            });

        }
    }

    private function selectBitrate(mappedBitrate:BitrateItem, detectedBitrate:Number = -1):void {
        log.debug("selectBitrate()");

        dynamicBuffering(mappedBitrate.bitrate, detectedBitrate);

        if (_playButton && _playButton.hasOwnProperty("stopBuffering")) {
            _playButton["stopBuffering"]();
        }

        //move this event up to give it time before onStreamSwitchBegin is called.
        log.info("dispatching onBwDone, mapped bitrate: " + mappedBitrate.bitrate + " detected bitrate " + detectedBitrate + " url: " + mappedBitrate.streamName);
        _model.dispatch(PluginEventType.PLUGIN_EVENT, "onBwDone", mappedBitrate, detectedBitrate);


        if (_resolving) {
            _streamSelectionManager.changeStreamNames(mappedBitrate);
            _resolveSuccessListener(_clip);
            _resolving = false;
        } else if (_netStream && (_player.isPlaying() || _player.isPaused())) {
            switchStream(mappedBitrate);
        } else {
            _streamSelectionManager.changeStreamNames(mappedBitrate);
        }


    }

    private function switchStream(mappedBitrate:BitrateItem):void {
        _streamSwitchManager.switchStream(mappedBitrate);
    }

    private function addRule(prop:String, rules:Vector.<SwitchingRuleBase>, rule:SwitchingRuleBase):void {
        if (_config.qos[prop]) {
            log.info("using QoS switching rules " + rule);
            rules.push(rule);
        }
    }

    [External]
    public function get bitrateItems():Vector.<DynamicStreamingItem> {
        return Vector.<DynamicStreamingItem>(_streamSelectionManager.bitrates);
    }

    /**
     * Store the detection and chosen bitrate if the rememberBitrate config property is set.
     */
    protected function rememberBandwidth(bw:int):void {
        if (_config.rememberBitrate) {
            _bitrateStorage.bandwidth = bw;
            log.debug("stored bandwidth " + bw);
        }

    }


    private function isRememberedBitrateValid():Boolean {
        log.debug("isRememberedBitrateValid()");

        if (! _bitrateStorage.bandwidth) {
            log.debug("bandwidth not in SO");
            return false;
        }

        var expired:Boolean = _bitrateStorage.isExpired;
        log.info("is remembered bitrate expired?: " + expired + (expired ? ", age is " + _bitrateStorage.age : ""));

        return ! expired;
    }

    public function resolve(provider:StreamProvider, clip:Clip, successListener:Function):void {
        log.debug("resolve " + clip);

        if (!clip.getCustomProperty("bitrates") && !clip.getCustomProperty("bitrateItems")) {
            log.info("Bitrates configuration not enabled for this clip");
            successListener(clip);
            return;
        }

        if (alreadyResolved(clip)) {
            log.info("resolve(): bandwidth already resolved for clip " + clip + ", will not detect again");
            successListener(clip);
            return;
        }

        _provider = provider;
        _resolving = true;
        _resolveSuccessListener = successListener;

        init(provider.netStream, clip);
        checkBandwidthIfNotDetectedYet();
    }


    private function useDefaultBitrate():void {
        selectBitrate(getMappedBitrate(), -1);
    }

    private function useStoredBitrate():void {
        var mappedBitrate:BitrateItem = getMappedBitrate(_bitrateStorage.bandwidth);
        log.info("using remembered bandwidth " + _bitrateStorage.bandwidth + ", maps to bitrate " + mappedBitrate.bitrate);
        selectBitrate(mappedBitrate, _bitrateStorage.bandwidth);
    }

    private function checkBandwidthIfNotDetectedYet():void {
        if (! applyForClip(_player.playlist.current)) return;
        if (hasDetectedBW()) {
            useStoredBitrate();
        } else if (_config.checkOnStart) {
            log.info("not using remembered bandwidth, detecting now");
            detect();
        } else {
            log.info("using dynamic switching with default bitrate ");
            useDefaultBitrate();
        }
    }

    private function init(netStream:NetStream, clip:Clip):void {
        log.debug("init(), netStream == " + netStream + ", clip == " + clip);
        _netStream = netStream;
        _clip = clip;
        _start = netStream ? netStream.time : 0;

       /* if (_resolving && _clip.getCustomProperty("streamSelectionManager")
                && !_clip.getCustomProperty("streamSelectionManager") is BWStreamSelectionManager) {
            _clip.setCustomProperty("streamSelectionManager", null);
        }  */

        if (!_clip.getCustomProperty("streamSelectionManager")) {
            _streamSelectionManager = new BWStreamSelectionManager(new BitrateResource(), _player, this, _config);
            _clip.setCustomProperty("streamSelectionManager",_streamSelectionManager);
        } else {
            _streamSelectionManager = _clip.getCustomProperty("streamSelectionManager") as IStreamSelectionManager;
        }

        initSwitchManager();
    }


    private function initSwitchManager():void {
        _streamSwitchManager = new StreamSwitchManager(_netStream, _streamSelectionManager, _player);
    }

    private function initQoS(netStream:NetStream, clip:Clip):void {
        log.info("initQoS(), netStream == " + netStream + ", host == " + _detector.host);

        import org.osmf.net.StreamType;

        //save the streaming resource and load for each clip in the playlist
        if (clip.getCustomProperty("urlResource")) {
            dsResource = clip.getCustomProperty("urlResource") as DynamicStreamingResource;
            dsResource.initialIndex = _streamSelectionManager.currentIndex;
        } else {
            dsResource = new DynamicStreamingResource(_detector.host);

            dsResource.streamItems = bitrateItems;
            dsResource.initialIndex = _streamSelectionManager.currentIndex;
            dsResource.streamType = _config.live ? StreamType.LIVE : StreamType.RECORDED;

            //#369 set clip start time for adding to play2 arguments.
            //#547 don't set the start property unless set, causes problems for live streams.
            if (clip.start) dsResource.clipStartTime = clip.start;
            clip.setCustomProperty("urlResource", dsResource);
        }
        //#500 setup osmf netclient when configuring metrics.
        if (netStream && ! (netStream.client is OsmfNetStreamClient)) {
            var netStreamClient:OsmfNetStreamClient = new OsmfNetStreamClient(NetStreamClient(netStream.client));
            netStream.client = netStreamClient;
        }

        setupMetrics(netStream);
    }

    private function setupMetrics(netStream:NetStream):void {
        CONFIG::enableRtmpMetrics {

            //#500 fix for netstream metrics, do not set to the clip properties as it causes errors with event callbacks.
            if (_provider.type == "rtmp") {
                _netStreamMetrics = new RTMPNetStreamMetrics(netStream);
                _netStreamMetrics.resource = dsResource;
                _switchManager = new NetStreamSwitchManager(_provider.netConnection, netStream, dsResource, _netStreamMetrics, getRTMPSwitchingRules(_netStreamMetrics as RTMPNetStreamMetrics));
                log.debug("using switch manager " + _switchManager);
                _netStreamMetrics.startMeasurements();

            }
        }

        CONFIG::enableHttpMetrics {

            if (_provider.type == "httpstreaming") {

                _netStreamMetrics = new HTTPNetStreamMetrics(netStream as HTTPNetStream);
                _netStreamMetrics.resource = dsResource;
                _switchManager = new NetStreamSwitchManager(_provider.netConnection, netStream, dsResource, _netStreamMetrics, getHTTPSwitchingRules(_netStreamMetrics as HTTPNetStreamMetrics));
                log.debug("using switch manager " + _switchManager);
                _netStreamMetrics.startMeasurements();
            }

        }

        if (_streamSelectionManager is BWStreamSelectionManager) BWStreamSelectionManager(_streamSelectionManager).qosSwitchManager = _switchManager;

        //set the clear failed count interval to the clip duration in milliseconds times a prescision value
        _switchManager.clearFailedCountInterval = _config.qos.clearFailedCountInterval * (Math.round(_player.playlist.current.duration) * 1000);
        _switchManager.maxUpSwitchesPerStream = _config.qos.maxUpSwitchesPerStream;
        _switchManager.ruleCheckInterval = _config.qos.ruleCheckInterval;
        _switchManager.waitDurationAfterDownSwitch = _config.qos.waitDurationAfterDownSwitch;

    }

    CONFIG::enableRtmpMetrics {
        private function getRTMPSwitchingRules(metrics:RTMPNetStreamMetrics):Vector.<SwitchingRuleBase> {
            var rules:Vector.<SwitchingRuleBase> = new Vector.<SwitchingRuleBase>();
            addRule("bwUp", rules, new SufficientBandwidthRule(metrics, _config.qos.bitrateSafety, _config.qos.minDroppedFrames));
            addRule("bwDown", rules, new InsufficientBandwidthRule(metrics, _config.qos.bitrateSafety));
            addRule("frames", rules, new DroppedFramesRule(metrics, _config.qos.framesToOne, _config.qos.framesToTwo, _config.qos.framesToLow));
            addRule("buffer", rules, new InsufficientBufferRule(metrics, _config.qos.minBufferLength));
            addRule("screen", rules, new ScreenSizeRule(metrics, _streamSelectionManager, _player, _config));
            return rules;
        }
    }

    CONFIG::enableHttpMetrics {
        private function getHTTPSwitchingRules(metrics:HTTPNetStreamMetrics):Vector.<SwitchingRuleBase> {
            var rules:Vector.<SwitchingRuleBase> = new Vector.<SwitchingRuleBase>();
            addRule("ratio", rules, new DownloadRatioRule(metrics));
            addRule("frames", rules, new DroppedFramesRule(metrics));
            addRule("screen", rules, new ScreenSizeRule(metrics, _streamSelectionManager, _player, _config));
            return rules;
        }
    }

    [External]
    public function currentItem():BitrateItem {
        log.info("currentItem(), _switchManager.currentIndex: " + (_switchManager ? _switchManager.currentIndex : "no switch manager") + ", streamSelectionManager.currentIndex: " + _streamSelectionManager.currentIndex );
        return BitrateItem(_streamSelectionManager.streamItems[_switchManager ? _switchManager.currentIndex : _streamSelectionManager.currentIndex]);
    }

    protected function getClipUrl(clip:Clip, mappedBitrate:BitrateItem):String {
        log.info("Resolved stream url: " + mappedBitrate.url);
        return mappedBitrate.url;
    }

    private function checkCurrentClip():Boolean {
        var clip:Clip = _player.playlist.current;
        if (_clip == clip) return true;

        if (clip.urlResolvers && clip.urlResolvers.indexOf(_model.name) < 0) {
            return false;
        }
        _clip = clip;
        return true;
    }

    [External]
    public function checkBandwidth():void {
        log.debug("checkBandwidth");
        if (! checkCurrentClip()) return;
        _start = _provider ? _provider.time : 0;
        _hasDetectedBW = false;
        _bitrateStorage.clear();
        detect();
    }

    [External]
    public function setBitrate(bitrate:Number):void {
        log.debug("set bitrate()");
        if (! checkCurrentClip()) return;

        try {
            if (_player.isPlaying() || _player.isPaused()) {

                switchStream(getMappedBitrate(bitrate));
                _config.dynamic = false;
                if (_switchManager) {
                    _switchManager.autoSwitch = false;
                }
            }
        } catch (e:Error) {
            log.error("error when switching streams " + e);
        }
    }

    [External]
    public function enableDynamic(enabled:Boolean):void {
        log.debug("set dynamic(), currently " + _config.dynamic + ", new value " + enabled);
        if (_config.dynamic == enabled) return;
        _config.dynamic = enabled;

        if (enabled) {
            if (! _switchManager) {
                var clip:Clip = _player.playlist.current;
                initQoS(clip.getNetStream(), clip);
            }
            _switchManager.autoSwitch = true;
        } else {
            if (_switchManager) {
                _switchManager.autoSwitch = false;
            }
        }
    }

    [External]
    public function get labels():Object {
        var labels:Object = {};
        for (var i:int = 0; i < bitrateItems.length; i++) {
            var item:BitrateItem = bitrateItems[i] as BitrateItem;
            if (item.label) {
                labels[item.bitrate] = item.label;
            }
        }
        return labels;
    }

    /**
     * Gets the current bitrate. The returned value is the bitrate in use after the latest bitrate transition has been completed. If
     * a transition is in progress the value reflects the bitrate right now being used, not the one we are changing to.
     * @return
     */
    [External]
    public function get bitrate():Number {

        log.debug("get bitrate()");
        if (! checkCurrentClip()) return undefined;

        if (_config.rememberBitrate && _bitrateStorage.bandwidth >= 0) {
            log.debug("get bitrate(), returning remembered bandwidth");
            var mappedBitrate:BitrateItem = getMappedBitrate(_bitrateStorage.bandwidth);
            return mappedBitrate.bitrate;
        }

        log.debug("get bitrate(), returning current bitrate");
        return currentItem().bitrate;
    }

    /**
     * Get the netstream metrics for external measurement monitoring.
     */
    [External]
    public function get netStreamMetrics():NetStreamMetricsBase
    {
        return _netStreamMetrics;
    }

    public function getDefaultConfig():Object {
        return {
            top: "45%",
            left: "50%",
            opacity: 1,
            borderRadius: 15,
            border: 'none',
            width: "80%",
            height: "80%"
        };
    }

    private function lookupProvider(providers:Dictionary):void {
        log.debug("lookupProvider() " + providers);
        if (_config.provider) {
            var model:PluginModel = _player.pluginRegistry.getPlugin(_config.provider) as PluginModel;
            if (! model) throw new Error("Failed to find plugin '" + _config.provider + "'");
            if (! (model.pluginObject is StreamProvider)) throw new Error("The specified provider is not a StreamProvider");
            _provider = StreamProvider(model.pluginObject);
            return;
        }
        for each (model in providers) {
            log.debug(model.name);
            if (model.name == "rtmp") {
                _provider = StreamProvider(model.pluginObject);
                return;
            }
            if (["http", "httpInstream"].indexOf(model.name) < 0 && model.pluginObject is StreamProvider) {
                _provider = StreamProvider(model.pluginObject);
            }
        }
    }
}
}
