/*  BEGIN software license
 *
 *  msXpertSuite - mass spectrometry software suite
 *  -----------------------------------------------
 *  Copyright(C) 2009, 2017 Filippo Rusconi
 *
 *  http://www.msxpertsuite.org
 *
 *  This file is part of the msXpertSuite project.
 *
 *  The msXpertSuite project is the successor of the massXpert project. This
 *  project now includes various independent modules:
 *  
 *  - massXpert, model polymer chemistries and simulate mass spectrometric data;
 *  - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * END software license
 */



/////////////////////// Qt includes
#include <QDebug>
#include <QScriptEngine>


/////////////////////// Local includes
#include <libmass/TraceJsPrototype.hpp>
#include <libmass/Trace.hpp>
#include <libmass/DataPointJs.hpp>

#include <globals/globals.hpp>

namespace msXpSlibmass
{


	TraceJsPrototype::TraceJsPrototype(QObject *parent)
		: QObject(parent)
	{
	}


	TraceJsPrototype::~TraceJsPrototype()
	{
	}


	Trace *
		TraceJsPrototype::thisTrace() const
		{
			Trace *trace = qscriptvalue_cast<Trace*>(thisObject().data());

			if(trace == Q_NULLPTR)
				qFatal("Fatal error at %s@%d -- %s. "
						"thisTrace() would return Q_NULLPTR."
						"Program aborted.",
						__FILE__, __LINE__, __FUNCTION__);

			return  trace;
		}


	int 
		TraceJsPrototype::initialize(const QString &xyFormatString)
		{
			qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()" 
				<< "title:" << thisTrace()->title();

			return thisTrace()->Trace::initialize(xyFormatString);
		}


	int 
		TraceJsPrototype::initialize(const QList<double> &keyList, const QList<double> &valList)
		{
			return thisTrace()->Trace::initialize(keyList, valList);
		}


	int 
		TraceJsPrototype::initializeRange(const QList<double> &keyList, const QList<double> &valList,
				double keyStart, double keyEnd)
		{
			return thisTrace()->Trace::initialize(keyList, valList, keyStart, keyEnd);
		}


	QList<double> 
		TraceJsPrototype::keyArray() const
		{
			return thisTrace()->keyList();
		}


	QList<double> 
		TraceJsPrototype::valArray() const
		{
			return thisTrace()->valList();
		}


	QString 
		TraceJsPrototype::asText()
		{
			QScriptContext *ctx = context();
			int argCount = ctx->argumentCount();

			double rangeStart = -1;
			double rangeEnd = -1;

			if(argCount != 0 && argCount != 2)
			{
				ctx->throwError("Syntax error: function asText takes either none or two numbers as index arguments.");

				return QString();
			}

			if(argCount == 2)
			{
				QScriptValue arg = ctx->argument(0);

				if(!arg.isNumber())
				{
					ctx->throwError("Syntax error: function asText() takes zero or two numbers as index arguments.");
					return QString();
				}

				rangeStart = qvariant_cast<double>(arg.data().toVariant());

				arg = ctx->argument(1);

				if(!arg.isNumber())
				{
					ctx->throwError("Syntax error: function combine() takes two numbers as index arguments.");
					return QString();
				}

				rangeEnd = qvariant_cast<double>(arg.data().toVariant());
			}

			// Finally, do the work.

			QString *text = thisTrace()->asText(rangeStart, rangeEnd);

			QString result = *text;
			delete text;

			return result;
		}


	bool
		TraceJsPrototype::exportToFile(const QString &fileName)
		{

			return thisTrace()->exportToFile(fileName);
		}


	double 
		TraceJsPrototype::valSum() const
		{
			// Either there is no argument, or there must be two arguments:
			// keyRangeStart and keyRangeEnd

			QScriptContext *ctx = context();
			int argCount = ctx->argumentCount();

			if(argCount == 0)
				return thisTrace()->valSum();

			if(argCount == 2)
			{
				QScriptValue arg = ctx->argument(0);

				if(!arg.isNumber())
				{
					ctx->throwError("Syntax error: function tic() takes two numbers as arguments.");

					return -1;
				}

				double keyRangeStart = qvariant_cast<double>(arg.data().toVariant());

				arg = ctx->argument(1);

				if(!arg.isNumber())
				{
					ctx->throwError("Syntax error: function tic() takes two numbers as arguments.");

					return -1;
				}

				double keyRangeEnd = qvariant_cast<double>(arg.data().toVariant());

				return thisTrace()->valSum(keyRangeStart, keyRangeEnd);
			}

			ctx->throwError("Syntax error: function tic() takes either no argument "
					"or two numbers as arguments.");

			return -1;
		}


	int 
		TraceJsPrototype::combine()
		{
			QScriptContext *ctx = context();
			int argCount = ctx->argumentCount();

			if(argCount == 1)
			{

				// The single argument can be either DataPoint or Trace

				QScriptValue arg = ctx->argument(0);

				QVariant variant = arg.data().toVariant();

				if(!variant.isValid())
				{
					qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()" 
						<< "Variant is not valid";

					ctx->throwError("Syntax error: function combine takes one DataPoint object "
							"or one Trace object or two numbers (key & i) as argument.");

					return -1;
				}

				// Test if DataPoint

				if(variant.userType() == QMetaType::type("msXpSlibmass::DataPoint"))
				{
					DataPoint dp(qscriptvalue_cast<DataPoint>(arg));

					if(!dp.isValid())
					{
						ctx->throwError("Function combine takes one initialized "
								"DataPoint object as argument.");

						return -1;
					}

					return thisTrace()->combine(dp);
				}

				// Test if Trace

				if(variant.userType() == QMetaType::type("msXpSlibmass::Trace"))
				{
					Trace ms(qscriptvalue_cast<Trace>(arg));

					return thisTrace()->combine(ms);
				}	
			}
			else if(argCount == 2)
			{
				QScriptValue arg = ctx->argument(0);

				if(!arg.isNumber())
				{
					ctx->throwError("Syntax error: function combine() takes two numbers as arguments.");
					return -1;
				}

				double key = qvariant_cast<double>(arg.data().toVariant());

				arg = ctx->argument(1);

				if(!arg.isNumber())
				{
					ctx->throwError("Syntax error: function combine() takes two numbers as arguments.");
					return -1;
				}

				double i = qvariant_cast<double>(arg.data().toVariant());

				return thisTrace()->combine(DataPoint(key, i));
			}

			ctx->throwError("Syntax error: function combine takes one DataPoint "
					"or one Trace object or two numbers (key & i) as argument.");

			return -1;
		}


	int 
		TraceJsPrototype::subtract()
		{
			QScriptContext *ctx = context();
			int argCount = ctx->argumentCount();

			if(argCount == 1)
			{

				// The single argument can be either DataPoint or Trace

				QScriptValue arg = ctx->argument(0);

				QVariant variant = arg.data().toVariant();

				if(!variant.isValid())
				{
					qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()" 
						<< "Variant is not valid";

					ctx->throwError("Syntax error: function subtract takes one DataPoint object "
							"or one Trace object or two numbers (key & i) as argument.");

					return -1;
				}

				// Test if DataPoint

				if(variant.userType() == QMetaType::type("msXpSlibmass::DataPoint"))
				{
					DataPoint mp(qscriptvalue_cast<DataPoint>(arg));

					if(!mp.isValid())
					{
						ctx->throwError("Function subtract takes one initialized "
								"DataPoint object as argument.");

						return -1;
					}

					return thisTrace()->subtract(mp);
				}

				// Test if Trace

				if(variant.userType() == QMetaType::type("msXpSlibmass::Trace"))
				{
					Trace ms(qscriptvalue_cast<Trace>(arg));

					return thisTrace()->subtract(ms);
				}	
			}
			else if(argCount == 2)
			{
				QScriptValue arg = ctx->argument(0);

				if(!arg.isNumber())
				{
					ctx->throwError("Syntax error: function subtract() takes two numbers as arguments.");
					return -1;
				}

				double key = qvariant_cast<double>(arg.data().toVariant());

				arg = ctx->argument(1);

				if(!arg.isNumber())
				{
					ctx->throwError("Syntax error: function subtract() takes two numbers as arguments.");
					return -1;
				}

				double i = qvariant_cast<double>(arg.data().toVariant());

				return thisTrace()->subtract(DataPoint(key, i));
			}

			ctx->throwError("Syntax error: function subtract takes one DataPoint object "
					"or one Trace object or two numbers (key & i) as argument.");

			return -1;
		}


	QScriptValue 
		TraceJsPrototype::valueOf() const
		{
			return thisObject().data();
		}

} // namespace msXpSlibmass
