///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file DataChannel.h
 * \brief Contains the definition of the AtomViz::DataChannel class.
 */

#ifndef __ATOMTYPE_DATA_CHANNEL_H
#define __ATOMTYPE_DATA_CHANNEL_H

#include <atomviz/AtomViz.h>
#include "DataChannel.h"
#include <atomviz/atoms/AtomType.h>

namespace AtomViz {

/**
 * \brief A data channel that stores the atom types.
 *
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT AtomTypeDataChannel : public DataChannel
{
	Q_OBJECT

public:

	/// \brief Serialization constructor that creates an empty data channel.
	///
	/// \note This constructor is only used when a data channel is loaded from a scene file.
	///       It must not be used to create a new data channel.
	AtomTypeDataChannel(bool isLoading = false);

	/// \brief Constructor that creates a standard data channel.
	/// \param which Specifies which standard data channel should be created.
	///              This must not be DataChannelIdentifier::UserDataChannel.
	///
	/// Data type, component count and name are automatically chosen by this constructor.
	AtomTypeDataChannel(DataChannelIdentifier which);

	/// Creates an AtomType at the given index or returns the existing one.
	AtomType* createAtomType(int index = -1);

	/// Inserts an atom type into this object. The atom type
	/// must not be owned by another AtomsObject.
	void insertAtomType(const AtomType::SmartPtr& atype, int index = -1);

	/// Returns the array of atom types.
	const AtomTypeList& atomTypes() const { return _atomTypes; }

	/// Returns the first atom type with the given name or NULL if there is
	/// no such atom type with the given name.
	AtomType* findAtomTypeByName(const QString& name) const;

	/// Returns the index of the first atom type with the given name or -1 if there is
	/// no such atom type with the given name.
	int findAtomTypeIndexByName(const QString& name) const;

	/// Removed a single atom type from this object.
	void removeAtomType(AtomType* atomType);

protected:

	/// Contains the atom types.
	VectorReferenceField<AtomType> _atomTypes;

private:

	DECLARE_SERIALIZABLE_PLUGIN_CLASS(AtomTypeDataChannel)
	DECLARE_VECTOR_REFERENCE_FIELD(_atomTypes)
};


/**
 * \brief A properties editor for the AtomTypeDataChannel class.
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT AtomTypeDataChannelEditor : public PropertiesEditor
{
protected:

	/// Creates the user interface controls for the editor.
	virtual void createUI(const RolloutInsertionParameters& rolloutParams);

private:
	Q_OBJECT
	DECLARE_PLUGIN_CLASS(AtomTypeDataChannelEditor)
};

};	// End of namespace AtomViz

#endif // __ATOMTYPE_DATA_CHANNEL_H
