///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/actions/ViewportModeAction.h>
#include <core/actions/ActionProxy.h>

namespace Core {

// Gives the class run-time type information.
IMPLEMENT_PLUGIN_CLASS(ViewportModeAction, Action)

/******************************************************************************
* Initializes the action object.
******************************************************************************/
ViewportModeAction::ViewportModeAction(const QString& id, const ViewportInputHandler::SmartPtr& inputHandler, const QColor& highlightColor)
	: Action(id), _inputHandler(inputHandler), _highlightColor(highlightColor)
{
	CHECK_OBJECT_POINTER(inputHandler);
}

/******************************************************************************
* This is called by the system after the action object has been bound to its ActionProxy.
******************************************************************************/
void ViewportModeAction::initializeAction(ActionProxy* proxy)
{
	Action::initializeAction(proxy);

	proxy->setCheckable(true);
	proxy->setChecked(VIEWPORT_INPUT_MANAGER.currentHandler() == _inputHandler);

	connect(&VIEWPORT_INPUT_MANAGER, SIGNAL(inputModeChanged(ViewportInputHandler*, ViewportInputHandler*)), this, SLOT(onInputModeChanged(ViewportInputHandler*, ViewportInputHandler*)));
	connect(proxy, SIGNAL(triggered(bool)), this, SLOT(onActionTriggered(bool)));
}

/******************************************************************************
* Is called when the user has triggered the action's state.
******************************************************************************/
void ViewportModeAction::onActionTriggered(bool checked)
{
	// Activate/deactivate the input mode.
	CHECK_OBJECT_POINTER(_inputHandler);
	if(checked && VIEWPORT_INPUT_MANAGER.currentHandler() != _inputHandler) {
		VIEWPORT_INPUT_MANAGER.pushInputHandler(_inputHandler);
	}
	else if(!checked) {
		if(_inputHandler->handlerActivationType() != ViewportInputHandler::EXCLUSIVE)
        	VIEWPORT_INPUT_MANAGER.removeInputHandler(_inputHandler.get());
		else
			// Make sure that an exclusive input mode cannot be deactivated by the user.
			proxy()->setChecked(true);
	}
}

/******************************************************************************
* Is called when the active viewport input handler has changed.
******************************************************************************/
void ViewportModeAction::onInputModeChanged(ViewportInputHandler* oldMode, ViewportInputHandler* newMode)
{
	if(oldMode == _inputHandler || newMode == _inputHandler) {
		proxy()->setChecked(VIEWPORT_INPUT_MANAGER.currentHandler() == _inputHandler);
	}
}

/******************************************************************************
* This function is called whenever the action is added to a
* container widget that supports custom widgets.
******************************************************************************/
QWidget* ViewportModeAction::createWidget(QWidget* parent)
{
	QWidget* w = Action::createWidget(parent);
#if 0
	if(highlightColor.isValid() && OVITO_DYN_CAST(w, QToolButton)) {
		w->setStyleSheet(QString("QToolButton:checked { padding: 4px 2px 2px 4px; border: 1px solid dimgray; border-radius: 3px; background-color: rgb(%1,%2,%3) }").arg(highlightColor.red()).arg(highlightColor.green()).arg(highlightColor.blue()));
	}
#endif
	return w;
}

};
