///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/plugins/PluginClassDescriptor.h>
#include <core/plugins/NativePluginClassDescriptor.h>
#include <core/plugins/PluginClass.h>
#include <core/plugins/Plugin.h>
#include <core/plugins/PluginManager.h>
#include <core/reference/PropertyFieldDescriptor.h>
#include <core/reference/RefTarget.h>
#include <core/data/ObjectSaveStream.h>
#include <core/data/ObjectLoadStream.h>
#include <core/undo/UndoManager.h>

namespace Core {

/// The descriptor of the root class of all plugin classes.
/// This class is named "PluginClass".
PluginClassDescriptor& PluginClassDescriptor::rootClass = NativePluginClassDescriptor::nativeRootClass;

/******************************************************************************
* Constructor of the descriptor structure.
******************************************************************************/
PluginClassDescriptor::PluginClassDescriptor(const QString& name, PluginClassDescriptor* baseClass, Plugin* plugin, const QDomElement& classNode, bool isAbstract, bool isSerializable) :
	_name(name), _plugin(plugin), _classNode(classNode), _isAbstract(isAbstract), _baseClass(baseClass),
	_isSerializable(isSerializable), firstChild(NULL), _firstPropertyField(NULL)
{
	OVITO_ASSERT(baseClass != NULL || name == "PluginClass");

	// Insert into the global list of classes.
	next = rootClass.next;
	rootClass.next = this;

	if(baseClass) {
		// Insert into linked list of base class.
		nextSibling = baseClass->firstChild;
		baseClass->firstChild = this;

		// Inherit serializable attribute.
		if(!baseClass->isSerializable())
			_isSerializable = false;
	}
}

/******************************************************************************
* Destructor of the descriptor structure.
******************************************************************************/
PluginClassDescriptor::~PluginClassDescriptor()
{
	if(baseClass()) { CHECK_POINTER(baseClass()); }

	// Delete derived classes.
	while(firstChild != NULL) {
        CHECK_POINTER(firstChild);
		PluginClassDescriptor* nextChild = firstChild->nextSibling;
		delete firstChild;
		firstChild = nextChild;
	}
}

/******************************************************************************
* Creates an object of the appropriate kind.
* Throws an exception if the containing plugin failed to load.
******************************************************************************/
intrusive_ptr<PluginClass> PluginClassDescriptor::createInstance(bool isLoading)
{
	CHECK_POINTER(plugin());

	if(!plugin()->isLoaded()) {
		// Load plugin first.
		try {
			plugin()->loadPlugin();
		}
		catch(Exception& ex) {
			throw ex.prependGeneralMessage(Plugin::tr("Failed to load plugin '%1'").arg(plugin()->pluginId()));
		}
	}
	if(isAbstract())
		throw Exception(Plugin::tr("Cannot instantiate abstract plugin class '%1'.").arg(name()));

	OVITO_ASSERT_MSG(isLoading == false || isSerializable(), "PluginClassDescriptor::createInstance", "Tried to load class that is not serializable.");

	return createInstanceImpl(isLoading);
}

/******************************************************************************
* Returns the display name of this plugin class.
******************************************************************************/
QString PluginClassDescriptor::schematicTitle() const
{
	QDomElement el = getMetaData("Schematic-Title");
	if(!el.isNull()) return el.text();
	return name();
}

/******************************************************************************
* Writes a class descriptor to the stream. This is for internal use of the core only.
******************************************************************************/
void PluginClassDescriptor::saveRTTI(ObjectSaveStream& stream, const PluginClassDescriptor* descriptor)
{
	CHECK_POINTER(descriptor);

	stream.beginChunk(0x10000000);
	stream << descriptor->plugin()->pluginId();
	stream << descriptor->name();
	stream.endChunk();
}

/******************************************************************************
* Loads a class descriptor from the stream. This is for internal use of the core only.
* Throws an exception if the class is not defined or the required plugin is not installed.
******************************************************************************/
PluginClassDescriptor* PluginClassDescriptor::loadRTTI(ObjectLoadStream& stream)
{
	QString pluginId, className;
	stream.expectChunk(0x10000000);
	stream >> pluginId;
	stream >> className;
	stream.closeChunk();

	// Lookup class descriptor.
	Plugin* plugin = PLUGIN_MANAGER.plugin(pluginId);
	if(plugin == NULL)
		throw Exception(Plugin::tr("Required plugin is not installed: %1").arg(pluginId));
	CHECK_POINTER(plugin);
	PluginClassDescriptor* descriptor = plugin->findClass(className);
	if(descriptor == NULL)
		throw Exception(Plugin::tr("Required class %1 not found in plugin %2.").arg(className, pluginId));

	return descriptor;
}

/******************************************************************************
* Searches for a property field defined in this class.
******************************************************************************/
const PropertyFieldDescriptor* PluginClassDescriptor::findPropertyField(const char* identifier) const
{
	for(const PropertyFieldDescriptor* field = firstPropertyField(); field; field = field->next())
		if(qstrcmp(field->identifier(), identifier) == 0) return field;
	return NULL;
}

};
