/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment media rendering library
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author(s): Loïc Molinari <loic@fluendo.com>
 *            Florian Boucault <florian@fluendo.com>
 *
 * The inverse 4x4 matrix function comes from the Mesa 3D Graphics Library
 * and is a contribution by Jacques Leroy <jle@star.be>.
 */

/**
 * SECTION:pgmlinearalgebra
 * @short_description: Various linear algebra objects and utility functions.
 *
 * <refsect2>
 * <para>
 * Various linear algebra objects and utility functions simplifying the
 * use of matrice and vector types.
 * </para>
 * <title>Notation</title>
 * <para>
 * In this documentation, we use the column vector notation for matrices and
 * matrix operations. That is also the notation used by the *_to_string()
 * functions, with the exception of pgm_vec3_to_string() and
 * pgm_vec4_to_string() for practical reasons.
 * </para>
 * <title>Memory storage convention</title>
 * <para>
 * Pigment matrices are stored in row-major order. This means that rows are
 * stored one after the other. The difference between addresses of array
 * cells in increasing rows is larger than addresses of cells in increasing
 * columns.
 * </para>
 * </refsect2>
 *
 * Last reviewed on 2008-01-15 (0.3.2)
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <gst/gst.h> /* debug */
#include <string.h>  /* memcpy */

#include "pgmlinearalgebra.h"
#include "pgmcommon.h"

/* Horizontal flipping 4x4 matrix */
#define MAT4X4_FLIP_HORIZ_SCALARS \
  -1.0f, 0.0f, 0.0f, 1.0f,        \
   0.0f, 1.0f, 0.0f, 0.0f,        \
   0.0f, 0.0f, 1.0f, 0.0f,        \
   0.0f, 0.0f, 0.0f, 1.0f

/* Vertical flipping 4x4 matrix */
#define MAT4X4_FLIP_VERT_SCALARS  \
   1.0f, 0.0f, 0.0f, 0.0f,        \
   0.0f,-1.0f, 0.0f, 1.0f,        \
   0.0f, 0.0f, 1.0f, 0.0f,        \
   0.0f, 0.0f, 0.0f, 1.0f

/* Clockwise rotating 4x4 matrix */
#define MAT4X4_ROTATE_CW_SCALARS  \
   0.0f, 1.0f, 0.0f, 0.0f,        \
  -1.0f, 0.0f, 0.0f, 1.0f,        \
   0.0f, 0.0f, 1.0f, 0.0f,        \
   0.0f, 0.0f, 0.0f, 1.0f

/* Counter-clockwise rotating 4x4 matrix */
#define MAT4X4_ROTATE_CCW_SCALARS \
   0.0f,-1.0f, 0.0f, 1.0f,        \
   1.0f, 0.0f, 0.0f, 0.0f,        \
   0.0f, 0.0f, 1.0f, 0.0f,        \
   0.0f, 0.0f, 0.0f, 1.0f

GST_DEBUG_CATEGORY_EXTERN (pgm_debug);
#define GST_CAT_DEFAULT pgm_debug

/* Private functions */

/* Multiply m1.m2 4x4 matrices putting the result in r */
static void
mat4x4_multiply_mat4x4_mat4x4 (PgmMat4x4 *r,
                               const PgmMat4x4 *m1,
                               const PgmMat4x4 *m2)
{
  r->m[0] = m1->m[0]*m2->m[0] + m1->m[1]*m2->m[4]
    + m1->m[2]*m2->m[8] + m1->m[3]*m2->m[12];
  r->m[1] = m1->m[0]*m2->m[1] + m1->m[1]*m2->m[5]
    + m1->m[2]*m2->m[9] + m1->m[3]*m2->m[13];
  r->m[2] = m1->m[0]*m2->m[2] + m1->m[1]*m2->m[6]
    + m1->m[2]*m2->m[10] + m1->m[3]*m2->m[14];
  r->m[3] = m1->m[0]*m2->m[3] + m1->m[1]*m2->m[7]
    + m1->m[2]*m2->m[11] + m1->m[3]*m2->m[15];

  r->m[4] = m1->m[4]*m2->m[0] + m1->m[5]*m2->m[4]
    + m1->m[6]*m2->m[8] + m1->m[7]*m2->m[12];
  r->m[5] = m1->m[4]*m2->m[1] + m1->m[5]*m2->m[5]
    + m1->m[6]*m2->m[9] + m1->m[7]*m2->m[13];
  r->m[6] = m1->m[4]*m2->m[2] + m1->m[5]*m2->m[6]
    + m1->m[6]*m2->m[10] + m1->m[7]*m2->m[14];
  r->m[7] = m1->m[4]*m2->m[3] + m1->m[5]*m2->m[7]
    + m1->m[6]*m2->m[11] + m1->m[7]*m2->m[15];

  r->m[8] = m1->m[8]*m2->m[0] + m1->m[9]*m2->m[4]
    + m1->m[10]*m2->m[8] + m1->m[11]*m2->m[12];
  r->m[9] = m1->m[8]*m2->m[1] + m1->m[9]*m2->m[5]
    + m1->m[10]*m2->m[9] + m1->m[11]*m2->m[13];
  r->m[10] = m1->m[8]*m2->m[2] + m1->m[9]*m2->m[6]
    + m1->m[10]*m2->m[10] + m1->m[11]*m2->m[14];
  r->m[11] = m1->m[8]*m2->m[3] + m1->m[9]*m2->m[7]
    + m1->m[10]*m2->m[11] + m1->m[11]*m2->m[15];

  r->m[12] = m1->m[12]*m2->m[0] + m1->m[13]*m2->m[4]
    + m1->m[14]*m2->m[8] + m1->m[15]*m2->m[12];
  r->m[13] = m1->m[12]*m2->m[1] + m1->m[13]*m2->m[5]
    + m1->m[14]*m2->m[9] + m1->m[15]*m2->m[13];
  r->m[14] = m1->m[12]*m2->m[2] + m1->m[13]*m2->m[6]
    + m1->m[14]*m2->m[10] + m1->m[15]*m2->m[14];
  r->m[15] = m1->m[12]*m2->m[3] + m1->m[13]*m2->m[7]
    + m1->m[14]*m2->m[11] + m1->m[15]*m2->m[15];
}

/* 3-components vector public methods */

GType
pgm_vec3_get_type (void)
{
  static GType our_type = 0;

  if (G_UNLIKELY (!our_type))
    our_type = g_boxed_type_register_static ("PgmVec3",
                                             (GBoxedCopyFunc) pgm_vec3_copy,
                                             (GBoxedFreeFunc) pgm_vec3_free);

  return our_type;
}

/**
 * pgm_vec3_new:
 *
 * Creates a new #PgmVec3 with components initialized at 0.0f.
 *
 * MT safe.
 *
 * Returns: a new #PgmVec3 instance.
 */
PgmVec3*
pgm_vec3_new (void)
{
  PgmVec3 *vec3;

  vec3 = g_slice_new0 (PgmVec3);

  GST_DEBUG ("created new vec3(%p:[%.1f, %.1f, %.1f])",
             vec3, vec3->v[0], vec3->v[1], vec3->v[2]);

  return vec3;
}

/**
 * pgm_vec3_new_from_scalars:
 * @v0: the 1st component.
 * @v1: the 2nd component.
 * @v2: the 3rd component.
 *
 * Creates a new #PgmVec3 with components initialized with the given scalars.
 *
 * MT safe.
 *
 * Returns: a new #PgmVec3 instance.
 */
PgmVec3*
pgm_vec3_new_from_scalars (gfloat v0,
                           gfloat v1,
                           gfloat v2)
{
  PgmVec3 *vec3;

  vec3 = g_slice_new (PgmVec3);

  vec3->v[0] = v0;
  vec3->v[1] = v1;
  vec3->v[2] = v2;

  GST_DEBUG ("created new vec3(%p:[%f, %f, %f])",
             vec3, vec3->v[0], vec3->v[1], vec3->v[2]);

  return vec3;
}

/**
 * pgm_vec3_copy:
 * @vec3: a #PgmVec3 object.
 *
 * Copies @vec3.
 *
 * MT safe.
 *
 * Returns: a newly allocated #PgmVec3.
 */
PgmVec3*
pgm_vec3_copy (PgmVec3 *vec3)
{
  PgmVec3 *new_vec3;

  g_return_val_if_fail (vec3 != NULL, NULL);

  new_vec3 = pgm_vec3_new ();
  memcpy (new_vec3, vec3, sizeof (PgmVec3));

  return new_vec3;
}

/**
 * pgm_vec3_free:
 * @vec3: A #PgmVec3 object.
 *
 * Frees all resources used by @vec3.
 *
 * MT safe.
 */
void
pgm_vec3_free (PgmVec3 *vec3)
{
  g_return_if_fail (vec3 != NULL);

  GST_DEBUG ("freeing vec3(%p:[%f, %f, %f])",
             vec3, vec3->v[0], vec3->v[1], vec3->v[2]);

  g_slice_free (PgmVec3, vec3);
}

/**
 * pgm_vec3_set_from_scalars:
 * @vec3: A #PgmVec3 object.
 * @v0: the 1st component.
 * @v1: the 2nd component.
 * @v2: the 3rd component.
 *
 * Sets the components of @vec3 with the given scalars.
 *
 * MT safe.
 */
void
pgm_vec3_set_from_scalars (PgmVec3 *vec3,
                           gfloat v0,
                           gfloat v1,
                           gfloat v2)
{
  g_return_if_fail (vec3 != NULL);

  vec3->v[0] = v0;
  vec3->v[1] = v1;
  vec3->v[2] = v2;
}

/**
 * pgm_vec3_set_from_vec3:
 * @vec3: A #PgmVec3 object.
 * @v: A #PgmVec3 object.
 *
 * Sets the components of @vec3 with the given vector.
 *
 * MT safe.
 */
void
pgm_vec3_set_from_vec3 (PgmVec3 *vec3,
                        const PgmVec3 *v)
{
  g_return_if_fail (vec3 != NULL);
  g_return_if_fail (v != NULL);

  memcpy (vec3, v, sizeof (PgmVec3));
}

/**
 * pgm_vec3_length:
 * @vec3: A #PgmVec3 object.
 *
 * Retrieves the length of the vector @vec3.
 *
 * MT safe.
 *
 * Returns: the length of @vec3.
 */
gfloat
pgm_vec3_length (PgmVec3 *vec3)
{
  g_return_val_if_fail (vec3 != NULL, -1.0f);

  return (gfloat) sqrt (vec3->v[0] * vec3->v[0] +
                        vec3->v[1] * vec3->v[1] +
                        vec3->v[2] * vec3->v[2]);
}

/**
 * pgm_vec3_normalize:
 * @vec3: A #PgmVec3 object.
 *
 * Retrieves the normalized @vec3 vector.
 *
 * MT safe.
 *
 * Returns: the normalized @vec3 vector, or NULL if @vec3 is the zero vector.
 */
PgmVec3*
pgm_vec3_normalize (PgmVec3 *vec3)
{
  gfloat length;

  g_return_val_if_fail (vec3 != NULL, NULL);

  length = pgm_vec3_length (vec3);

  if (length == 0.0f)
    return NULL;

  length = 1.0f / length;

  return pgm_vec3_new_from_scalars (vec3->v[0] * length,
                                    vec3->v[1] * length,
                                    vec3->v[2] * length);
}

/**
 * pgm_vec3_dot_product:
 * @vec3: A #PgmVec3 object.
 * @v: A #PgmVec3 object.
 *
 * Computes the dot product @vec3 ∙ @v.
 *
 * MT safe.
 *
 * Returns: the dot product @vec3 ∙ @v.
 */
gfloat
pgm_vec3_dot_product (PgmVec3 *vec3,
                      const PgmVec3 *v)
{
  g_return_val_if_fail (vec3 != NULL, -1.0f);
  g_return_val_if_fail (v != NULL, -1.0f);

  return (vec3->v[0] * v->v[0] +
          vec3->v[1] * v->v[1] +
          vec3->v[2] * v->v[2]);
}

/**
 * pgm_vec3_cross_product:
 * @vec3: A #PgmVec3 object.
 * @v: A #PgmVec3 object.
 *
 * Computes the dot product @vec3 × @v.
 *
 * MT safe.
 *
 * Returns: the cross product @vec3 × @v in a newly allocated #PgmVec3.
 */
PgmVec3*
pgm_vec3_cross_product (PgmVec3 *vec3,
                        const PgmVec3 *v)
{
  PgmVec3 *cross;

  g_return_val_if_fail (vec3 != NULL, NULL);
  g_return_val_if_fail (v != NULL, NULL);

  cross = pgm_vec3_new ();

  cross->v[0] = vec3->v[1] * v->v[2] - vec3->v[2] * v->v[1];
  cross->v[1] = vec3->v[2] * v->v[0] - vec3->v[0] * v->v[2];
  cross->v[2] = vec3->v[0] * v->v[1] - vec3->v[1] * v->v[0];

  return cross;
}

/**
 * pgm_vec3_add_scalar:
 * @vec3: A #PgmVec3 object.
 * @s: A scalar.
 *
 * Computes the sum @vec3 + @s.
 *
 * MT safe.
 *
 * Returns: the sum @vec3 + @s in a newly allocated #PgmVec3.
 */
PgmVec3*
pgm_vec3_add_scalar (PgmVec3 *vec3,
                     gfloat s)
{
  PgmVec3 *sum;

  g_return_val_if_fail (vec3 != NULL, NULL);

  sum = pgm_vec3_new ();

  sum->v[0] = vec3->v[0] + s;
  sum->v[1] = vec3->v[1] + s;
  sum->v[2] = vec3->v[2] + s;

  return sum;
}

/**
 * pgm_vec3_add_vec3:
 * @vec3: A #PgmVec3 object.
 * @v: A #PgmVec3 object.
 *
 * Computes the sum @vec3 + @v.
 *
 * MT safe.
 *
 * Returns: the sum @vec3 + @v in a newly allocated #PgmVec3.
 */
PgmVec3*
pgm_vec3_add_vec3 (PgmVec3 *vec3,
                   const PgmVec3 *v)
{
  PgmVec3 *sum;

  g_return_val_if_fail (vec3 != NULL, NULL);
  g_return_val_if_fail (v != NULL, NULL);

  sum = pgm_vec3_new ();

  sum->v[0] = vec3->v[0] + v->v[0];
  sum->v[1] = vec3->v[1] + v->v[1];
  sum->v[2] = vec3->v[2] + v->v[2];

  return sum;
}

/**
 * pgm_vec3_substract_scalar:
 * @vec3: A #PgmVec3 object.
 * @s: A scalar.
 *
 * Computes the substraction @vec3 - @s.
 *
 * MT safe.
 *
 * Returns: the substraction @vec3 - @s in a newly allocated #PgmVec3.
 */
PgmVec3*
pgm_vec3_substract_scalar (PgmVec3 *vec3,
                           gfloat s)
{
  PgmVec3 *sub;

  g_return_val_if_fail (vec3 != NULL, NULL);

  sub = pgm_vec3_new ();

  sub->v[0] = vec3->v[0] - s;
  sub->v[1] = vec3->v[1] - s;
  sub->v[2] = vec3->v[2] - s;

  return sub;
}

/**
 * pgm_vec3_substract_vec3:
 * @vec3: A #PgmVec3 object.
 * @v: A #PgmVec3 object.
 *
 * Computes the substraction @vec3 - @v.
 *
 * MT safe.
 *
 * Returns: the substraction @vec3 - @v in a newly allocated #PgmVec3.
 */
PgmVec3*
pgm_vec3_substract_vec3 (PgmVec3 *vec3,
                         const PgmVec3 *v)
{
  PgmVec3 *sub;

  g_return_val_if_fail (vec3 != NULL, NULL);
  g_return_val_if_fail (v != NULL, NULL);

  sub = pgm_vec3_new ();

  sub->v[0] = vec3->v[0] - v->v[0];
  sub->v[1] = vec3->v[1] - v->v[1];
  sub->v[2] = vec3->v[2] - v->v[2];

  return sub;
}

/**
 * pgm_vec3_multiply_scalar:
 * @vec3: A #PgmVec3 object.
 * @s: A scalar.
 *
 * Computes the multiplication @vec3 . @s.
 *
 * MT safe.
 *
 * Returns: the multiplication @vec3 . @s in a newly allocated #PgmVec3.
 */
PgmVec3*
pgm_vec3_multiply_scalar (PgmVec3 *vec3,
                          gfloat s)
{
  PgmVec3 *mult;

  g_return_val_if_fail (vec3 != NULL, NULL);

  mult = pgm_vec3_new ();

  mult->v[0] = s * vec3->v[0];
  mult->v[1] = s * vec3->v[1];
  mult->v[2] = s * vec3->v[2];

  return mult;
}

/**
 * pgm_vec3_multiply_vec3:
 * @vec3: A #PgmVec3 object.
 * @v: A #PgmVec3 object.
 *
 * Computes the component by component multiplication @vec3 . @v.
 *
 * MT safe.
 *
 * Returns: the component by component multiplication @vec3 . @v in a newly
 * allocated #PgmVec3.
 */
PgmVec3*
pgm_vec3_multiply_vec3 (PgmVec3 *vec3,
                        const PgmVec3 *v)
{
  PgmVec3 *mult;

  g_return_val_if_fail (vec3 != NULL, NULL);
  g_return_val_if_fail (v != NULL, NULL);

  mult = pgm_vec3_new ();

  mult->v[0] = vec3->v[0] * v->v[0];
  mult->v[1] = vec3->v[1] * v->v[1];
  mult->v[2] = vec3->v[2] * v->v[2];

  return mult;
}

/**
 * pgm_vec3_to_string:
 * @vec3: A #PgmVec3 object.
 *
 * Converts @vec3 to a string representation.
 *
 * This function, as well as pgm_vec4_to_string(), returns a representation
 * of @vec3 in row vector notation. This is practical for most usages since
 * it fits in one line.
 *
 * MT safe.
 *
 * Returns: a newly allocated string representing @vec3 as a row.
 */
gchar*
pgm_vec3_to_string (PgmVec3 *vec3)
{
  gchar *string;

  g_return_val_if_fail (vec3 != NULL, NULL);

  string = g_strdup_printf ("[%f %f %f]", vec3->v[0], vec3->v[1], vec3->v[2]);

  return string;
}

/* 4-components vector public methods */

GType
pgm_vec4_get_type (void)
{
  static GType our_type = 0;

  if (G_UNLIKELY (!our_type))
    our_type = g_boxed_type_register_static ("PgmVec4",
                                             (GBoxedCopyFunc) pgm_vec4_copy,
                                             (GBoxedFreeFunc) pgm_vec4_free);

  return our_type;
}

/**
 * pgm_vec4_new:
 *
 * Creates a new #PgmVec4 with components initilized at 0.0f.
 *
 * MT safe.
 *
 * Returns: a new #PgmVec4 instance.
 */
PgmVec4*
pgm_vec4_new (void)
{
  PgmVec4 *vec4;

  vec4 = g_slice_new0 (PgmVec4);

  GST_DEBUG ("created new vec4(%p:[%.1f, %.1f, %.1f, %.1f])",
             vec4, vec4->v[0], vec4->v[1], vec4->v[2], vec4->v[3]);

  return vec4;
}

/**
 * pgm_vec4_new_from_scalars:
 * @v0: the 1st component.
 * @v1: the 2nd component.
 * @v2: the 3rd component.
 * @v3: the 4th component.
 *
 * Creates a new #PgmVec4 with components initialized with the given scalars.
 *
 * MT safe.
 *
 * Returns: a new #PgmVec4 instance.
 */
PgmVec4*
pgm_vec4_new_from_scalars (gfloat v0,
                           gfloat v1,
                           gfloat v2,
                           gfloat v3)
{
  PgmVec4 *vec4;

  vec4 = g_slice_new (PgmVec4);

  vec4->v[0] = v0;
  vec4->v[1] = v1;
  vec4->v[2] = v2;
  vec4->v[3] = v3;

  GST_DEBUG ("created new vec4(%p:[%f, %f, %f, %f])",
             vec4, vec4->v[0], vec4->v[1], vec4->v[2], vec4->v[3]);

  return vec4;
}

/**
 * pgm_vec4_copy:
 * @vec4: a #PgmVec4 object.
 *
 * Copies @vec4.
 *
 * MT safe.
 *
 * Returns: a newly allocated #PgmVec4.
 */
PgmVec4*
pgm_vec4_copy (PgmVec4 *vec4)
{
  PgmVec4 *new_vec4;

  g_return_val_if_fail (vec4 != NULL, NULL);

  new_vec4 = pgm_vec4_new ();
  memcpy (new_vec4, vec4, sizeof (PgmVec4));

  return new_vec4;
}

/**
 * pgm_vec4_free:
 * @vec4: A #PgmVec4 object.
 *
 * Frees all resources used by @vec4.
 *
 * MT safe.
 */
void
pgm_vec4_free (PgmVec4 *vec4)
{
  g_return_if_fail (vec4 != NULL);

  GST_DEBUG ("freeing vec4(%p:[%f, %f, %f, %f])",
             vec4, vec4->v[0], vec4->v[1], vec4->v[2], vec4->v[3]);

  g_slice_free (PgmVec4, vec4);
}

/**
 * pgm_vec4_set_from_scalars:
 * @vec4: A #PgmVec4 object.
 * @v0: the 1st component.
 * @v1: the 2nd component.
 * @v2: the 3rd component.
 * @v3: the 4th component.
 *
 * Sets the components of @vec4 with the given scalars.
 *
 * MT safe.
 */
void
pgm_vec4_set_from_scalars (PgmVec4 *vec4,
                           gfloat v0,
                           gfloat v1,
                           gfloat v2,
                           gfloat v3)
{
  g_return_if_fail (vec4 != NULL);

  vec4->v[0] = v0;
  vec4->v[1] = v1;
  vec4->v[2] = v2;
  vec4->v[3] = v3;
}

/**
 * pgm_vec4_set_from_vec4:
 * @vec4: A #PgmVec4 object.
 * @v: A #PgmVec4 object.
 *
 * Sets the components of @vec4 with the given vector.
 *
 * MT safe.
 */
void
pgm_vec4_set_from_vec4 (PgmVec4 *vec4,
                        const PgmVec4 *v)
{
  g_return_if_fail (vec4 != NULL);
  g_return_if_fail (v != NULL);

  memcpy (vec4, v, sizeof (PgmVec4));
}

/**
 * pgm_vec4_length:
 * @vec4: A #PgmVec4 object.
 *
 * Retrieves the length of the vector @vec4.
 *
 * MT safe.
 *
 * Returns: the length of @vec4.
 */
gfloat
pgm_vec4_length (PgmVec4 *vec4)
{
  g_return_val_if_fail (vec4 != NULL, -1.0f);

  return (gfloat) sqrt (vec4->v[0] * vec4->v[0] +
                        vec4->v[1] * vec4->v[1] +
                        vec4->v[2] * vec4->v[2] +
                        vec4->v[3] * vec4->v[3]);
}

/**
 * pgm_vec4_normalize:
 * @vec4: A #PgmVec4 object.
 *
 * Retrieves the normalized @vec4 vector.
 *
 * MT safe.
 *
 * Returns: the normalized @vec4 vector, or NULL if @vec4 is the zero vector.
 */
PgmVec4*
pgm_vec4_normalize (PgmVec4 *vec4)
{
  gfloat length;

  g_return_val_if_fail (vec4 != NULL, NULL);

  length = pgm_vec4_length (vec4);

  if (length == 0.0f)
    return NULL;

  length = 1.0f / length;

  return pgm_vec4_new_from_scalars (vec4->v[0] * length,
                                    vec4->v[1] * length,
                                    vec4->v[2] * length,
                                    vec4->v[3] * length);
}

/**
 * pgm_vec4_add_scalar:
 * @vec4: A #PgmVec4 object.
 * @s: A scalar.
 *
 * Computes the sum @vec4 + @s.
 *
 * MT safe.
 *
 * Returns: the sum @vec4 + @s in a newly allocated #PgmVec4.
 */
PgmVec4*
pgm_vec4_add_scalar (PgmVec4 *vec4,
                     gfloat s)
{
  PgmVec4 *sum;

  g_return_val_if_fail (vec4 != NULL, NULL);

  sum = pgm_vec4_new ();

  sum->v[0] = vec4->v[0] + s;
  sum->v[1] = vec4->v[1] + s;
  sum->v[2] = vec4->v[2] + s;
  sum->v[3] = vec4->v[3] + s;

  return sum;
}

/**
 * pgm_vec4_add_vec4:
 * @vec4: A #PgmVec4 object.
 * @v: A #PgmVec4 object.
 *
 * Computes the sum @vec4 + @v.
 *
 * MT safe.
 *
 * Returns: the sum @vec4 + @v in a newly allocated #PgmVec4.
 */
PgmVec4*
pgm_vec4_add_vec4 (PgmVec4 *vec4,
                   const PgmVec4 *v)
{
  PgmVec4 *sum;

  g_return_val_if_fail (vec4 != NULL, NULL);
  g_return_val_if_fail (v != NULL, NULL);

  sum = pgm_vec4_new ();

  sum->v[0] = vec4->v[0] + v->v[0];
  sum->v[1] = vec4->v[1] + v->v[1];
  sum->v[2] = vec4->v[2] + v->v[2];
  sum->v[3] = vec4->v[3] + v->v[3];

  return sum;
}

/**
 * pgm_vec4_substract_scalar:
 * @vec4: A #PgmVec4 object.
 * @s: A scalar.
 *
 * Computes the substraction @vec4 - @s.
 *
 * MT safe.
 *
 * Returns: the substraction @vec4 - @s in a newly allocated #PgmVec4.
 */
PgmVec4*
pgm_vec4_substract_scalar (PgmVec4 *vec4,
                           gfloat s)
{
  PgmVec4 *sub;

  g_return_val_if_fail (vec4 != NULL, NULL);

  sub = pgm_vec4_new ();

  sub->v[0] = vec4->v[0] - s;
  sub->v[1] = vec4->v[1] - s;
  sub->v[2] = vec4->v[2] - s;
  sub->v[3] = vec4->v[3] - s;

  return sub;
}

/**
 * pgm_vec4_substract_vec4:
 * @vec4: A #PgmVec4 object.
 * @v: A #PgmVec4 object.
 *
 * Computes the substraction @vec4 - @v.
 *
 * MT safe.
 *
 * Returns: the substraction @vec4 - @v in a newly allocated #PgmVec4.
 */
PgmVec4*
pgm_vec4_substract_vec4 (PgmVec4 *vec4,
                         const PgmVec4 *v)
{
  PgmVec4 *sub;

  g_return_val_if_fail (vec4 != NULL, NULL);
  g_return_val_if_fail (v != NULL, NULL);

  sub = pgm_vec4_new ();

  sub->v[0] = vec4->v[0] - v->v[0];
  sub->v[1] = vec4->v[1] - v->v[1];
  sub->v[2] = vec4->v[2] - v->v[2];
  sub->v[3] = vec4->v[3] - v->v[3];

  return sub;
}

/**
 * pgm_vec4_multiply_scalar:
 * @vec4: A #PgmVec4 object.
 * @s: A scalar.
 *
 * Computes the multiplication @vec4 . @s.
 *
 * MT safe.
 *
 * Returns: the multiplication @vec4 . @s in a newly allocated #PgmVec4.
 */
PgmVec4*
pgm_vec4_multiply_scalar (PgmVec4 *vec4,
                          gfloat s)
{
  PgmVec4 *mult;

  g_return_val_if_fail (vec4 != NULL, NULL);

  mult = pgm_vec4_new ();

  mult->v[0] = s * vec4->v[0];
  mult->v[1] = s * vec4->v[1];
  mult->v[2] = s * vec4->v[2];
  mult->v[3] = s * vec4->v[3];

  return mult;
}

/**
 * pgm_vec4_multiply_vec4:
 * @vec4: A #PgmVec4 object.
 * @v: A #PgmVec4 object.
 *
 * Computes the component by component multiplication @vec4 . @v.
 *
 * MT safe.
 *
 * Returns: the component by component multiplication @vec4 . @v in a newly
 * allocated #PgmVec4.
 */
PgmVec4*
pgm_vec4_multiply_vec4 (PgmVec4 *vec4,
                        const PgmVec4 *v)
{
  PgmVec4 *mult;

  g_return_val_if_fail (vec4 != NULL, NULL);
  g_return_val_if_fail (v != NULL, NULL);

  mult = pgm_vec4_new ();

  mult->v[0] = vec4->v[0] * v->v[0];
  mult->v[1] = vec4->v[1] * v->v[1];
  mult->v[2] = vec4->v[2] * v->v[2];
  mult->v[3] = vec4->v[3] * v->v[3];

  return mult;
}

/**
 * pgm_vec4_to_string:
 * @vec4: A #PgmVec4 object.
 *
 * Converts @vec4 to a string representation.
 *
 * This function, as well as pgm_vec3_to_string(), returns a representation
 * of @vec4 in row vector notation. This is practical for most usages since
 * it fits in one line.
 *
 * MT safe.
 *
 * Returns: a newly allocated string representing @vec4 as a row.
 */
gchar*
pgm_vec4_to_string (PgmVec4 *vec4)
{
  gchar *string;

  g_return_val_if_fail (vec4 != NULL, NULL);

  string = g_strdup_printf ("[%f %f %f %f]", vec4->v[0], vec4->v[1],
                            vec4->v[2], vec4->v[3]);

  return string;
}

/* 3x3 matrix public methods */

GType
pgm_mat3x3_get_type (void)
{
  static GType our_type = 0;

  if (G_UNLIKELY (!our_type))
    our_type = g_boxed_type_register_static ("PgmMat3x3",
                                             (GBoxedCopyFunc) pgm_mat3x3_copy,
                                             (GBoxedFreeFunc) pgm_mat3x3_free);

  return our_type;
}

/**
 * pgm_mat3x3_new:
 *
 * Creates a new #PgmMat3x3 with components initilized at 0.0f.
 *
 * MT safe.
 *
 * Returns: a new #PgmMat3x3 instance.
 */
PgmMat3x3*
pgm_mat3x3_new (void)
{
  PgmMat3x3 *mat3x3;

  mat3x3 = g_slice_new0 (PgmMat3x3);

  GST_DEBUG ("created new mat3x3(%p:[%.1f, %.1f, %.1f], "
             "[%.1f, %.1f, %.1f], [%.1f, %.1f, %.1f])", mat3x3,
             mat3x3->m[0], mat3x3->m[1], mat3x3->m[2],
             mat3x3->m[3], mat3x3->m[4], mat3x3->m[5],
             mat3x3->m[6], mat3x3->m[7], mat3x3->m[8]);

  return mat3x3;
}

/**
 * pgm_mat3x3_new_from_scalars:
 * @v0: the 1st component of the 1st vector.
 * @v1: the 2nd component of the 1st vector.
 * @v2: the 3rd component of the 1st vector.
 * @v3: the 1st component of the 2nd vector.
 * @v4: the 2nd component of the 2nd vector.
 * @v5: the 3rd component of the 2nd vector.
 * @v6: the 1st component of the 3rd vector.
 * @v7: the 2nd component of the 3rd vector.
 * @v8: the 3rd component of the 3rd vector.
 *
 * Creates a new #PgmMat3x3 with components initialized with the given scalars.
 *
 * MT safe.
 *
 * Returns: a new #PgmMat3x3 instance.
 */
PgmMat3x3*
pgm_mat3x3_new_from_scalars (gfloat v0,
                             gfloat v1,
                             gfloat v2,
                             gfloat v3,
                             gfloat v4,
                             gfloat v5,
                             gfloat v6,
                             gfloat v7,
                             gfloat v8)
{
  PgmMat3x3 *mat3x3;

  mat3x3 = g_slice_new (PgmMat3x3);

  mat3x3->m[0]=v0; mat3x3->m[1]=v1; mat3x3->m[2]=v2;
  mat3x3->m[3]=v3; mat3x3->m[4]=v4; mat3x3->m[5]=v5;
  mat3x3->m[6]=v6; mat3x3->m[7]=v7; mat3x3->m[8]=v8;

  GST_DEBUG ("created new mat3x3(%p:[%f, %f, %f], [%f, %f, %f], [%f, %f, %f])",
             mat3x3, mat3x3->m[0], mat3x3->m[1], mat3x3->m[2],
             mat3x3->m[3], mat3x3->m[4], mat3x3->m[5],
             mat3x3->m[6], mat3x3->m[7], mat3x3->m[8]);

  return mat3x3;
}

/**
 * pgm_mat3x3_new_from_vec3:
 * @v0: the 1st vector.
 * @v1: the 2nd vector.
 * @v2: the 3rd vector.
 *
 * Creates a new #PgmMat3x3 with components initialized with the given vectors.
 *
 * MT safe.
 *
 * Returns: a new #PgmMat3x3 instance.
 */
PgmMat3x3*
pgm_mat3x3_new_from_vec3 (const PgmVec3 *v0,
                          const PgmVec3 *v1,
                          const PgmVec3 *v2)
{
  PgmMat3x3 *mat3x3;

  g_return_val_if_fail (v0 != NULL, NULL);
  g_return_val_if_fail (v1 != NULL, NULL);
  g_return_val_if_fail (v2 != NULL, NULL);

  mat3x3 = g_slice_new (PgmMat3x3);

  mat3x3->m[0]=v0->v[0]; mat3x3->m[1]=v0->v[1]; mat3x3->m[2]=v0->v[2];
  mat3x3->m[3]=v1->v[0]; mat3x3->m[4]=v1->v[1]; mat3x3->m[5]=v1->v[2];
  mat3x3->m[6]=v2->v[0]; mat3x3->m[7]=v2->v[1]; mat3x3->m[8]=v2->v[2];

  GST_DEBUG ("created new mat3x3(%p:[%f, %f, %f], [%f, %f, %f], [%f, %f, %f])",
             mat3x3, mat3x3->m[0], mat3x3->m[1], mat3x3->m[2],
             mat3x3->m[3], mat3x3->m[4], mat3x3->m[5],
             mat3x3->m[6], mat3x3->m[7], mat3x3->m[8]);

  return mat3x3;
}

/**
 * pgm_mat3x3_new_identity:
 *
 * Creates a new #PgmMat3x3 identity matrix.
 *
 * MT safe.
 *
 * Returns: a new #PgmMat3x3 instance.
 */
PgmMat3x3*
pgm_mat3x3_new_identity (void)
{
  PgmMat3x3 *mat3x3;

  mat3x3 = g_slice_new0 (PgmMat3x3);

  mat3x3->m[0] = 1.0f;
  mat3x3->m[4] = 1.0f;
  mat3x3->m[8] = 1.0f;

  GST_DEBUG ("created new mat3x3(%p:[%.1f, %.1f, %.1f], "
             "[%.1f, %.1f, %.1f], [%.1f, %.1f, %.1f])", mat3x3,
             mat3x3->m[0], mat3x3->m[1], mat3x3->m[2],
             mat3x3->m[3], mat3x3->m[4], mat3x3->m[5],
             mat3x3->m[6], mat3x3->m[7], mat3x3->m[8]);

  return mat3x3;
}

/**
 * pgm_mat3x3_copy:
 * @mat3x3: a #PgmMat3x3 object.
 *
 * Copies @mat3x3.
 *
 * MT safe.
 *
 * Returns: a newly allocated #PgmMat3x3.
 */
PgmMat3x3*
pgm_mat3x3_copy (PgmMat3x3 *mat3x3)
{
  PgmMat3x3 *new_mat3x3;

  g_return_val_if_fail (mat3x3 != NULL, NULL);

  new_mat3x3 = pgm_mat3x3_new ();
  memcpy (new_mat3x3, mat3x3, sizeof (PgmMat3x3));

  return new_mat3x3;
}

/**
 * pgm_mat3x3_free:
 * @mat3x3: A #PgmMat3x3 object.
 *
 * Frees all resources used by @mat3x3.
 *
 * MT safe.
 */
void
pgm_mat3x3_free (PgmMat3x3 *mat3x3)
{
  g_return_if_fail (mat3x3 != NULL);

  GST_DEBUG ("freeing mat3x3(%p:[%f, %f, %f], [%f, %f, %f], [%f, %f, %f])",
             mat3x3, mat3x3->m[0], mat3x3->m[1], mat3x3->m[2],
             mat3x3->m[3], mat3x3->m[4], mat3x3->m[5],
             mat3x3->m[6], mat3x3->m[7], mat3x3->m[8]);

  g_slice_free (PgmMat3x3, mat3x3);
}

/**
 * pgm_mat3x3_set_from_scalars:
 * @mat3x3: A #PgmMat3x3 object.
 * @v0: the 1st component of the 1st vector.
 * @v1: the 2nd component of the 1st vector.
 * @v2: the 3rd component of the 1st vector.
 * @v3: the 1st component of the 2nd vector.
 * @v4: the 2nd component of the 2nd vector.
 * @v5: the 3rd component of the 2nd vector.
 * @v6: the 1st component of the 3rd vector.
 * @v7: the 2nd component of the 3rd vector.
 * @v8: the 3rd component of the 3rd vector.
 *
 * Set the components of @mat3x3 with the given scalars.
 *
 * MT safe.
 */
void
pgm_mat3x3_set_from_scalars (PgmMat3x3 *mat3x3,
                             gfloat v0,
                             gfloat v1,
                             gfloat v2,
                             gfloat v3,
                             gfloat v4,
                             gfloat v5,
                             gfloat v6,
                             gfloat v7,
                             gfloat v8)
{
  g_return_if_fail (mat3x3 != NULL);

  mat3x3->m[0]=v0; mat3x3->m[1]=v1; mat3x3->m[2]=v2;
  mat3x3->m[3]=v3; mat3x3->m[4]=v4; mat3x3->m[5]=v5;
  mat3x3->m[6]=v6; mat3x3->m[7]=v7; mat3x3->m[8]=v8;
}

/**
 * pgm_mat3x3_set_from_vec3:
 * @mat3x3: A #PgmMat3x3 object.
 * @v0: the 1st vector.
 * @v1: the 1st vector.
 * @v2: the 1st vector.
 *
 * Set the components of @mat3x3 with the given vectors.
 *
 * MT safe.
 */
void
pgm_mat3x3_set_from_vec3 (PgmMat3x3 *mat3x3,
                          const PgmVec3 *v0,
                          const PgmVec3 *v1,
                          const PgmVec3 *v2)
{
  g_return_if_fail (mat3x3 != NULL);
  g_return_if_fail (v0 != NULL);
  g_return_if_fail (v1 != NULL);
  g_return_if_fail (v2 != NULL);

  mat3x3->m[0]=v0->v[0]; mat3x3->m[1]=v0->v[1]; mat3x3->m[2]=v0->v[2];
  mat3x3->m[3]=v1->v[0]; mat3x3->m[4]=v1->v[1]; mat3x3->m[5]=v1->v[2];
  mat3x3->m[6]=v2->v[0]; mat3x3->m[7]=v2->v[1]; mat3x3->m[8]=v2->v[2];
}

/**
 * pgm_mat3x3_set_from_mat3x3:
 * @mat3x3: A #PgmMat3x3 object.
 * @m: A #PgmMat3x3 object.
 *
 * Sets the components of @mat3x3 with the given matrix.
 *
 * MT safe.
 */
void
pgm_mat3x3_set_from_mat3x3 (PgmMat3x3 *mat3x3,
                            const PgmMat3x3 *m)
{
  g_return_if_fail (mat3x3 != NULL);
  g_return_if_fail (m != NULL);

  memcpy (mat3x3, m, sizeof (PgmMat3x3));
}

/**
 * pgm_mat3x3_is_identity:
 * @mat3x3: A #PgmMat3x3 object.
 *
 * Retrieves whether or not @mat3x3 is the identity matrix.
 *
 * MT safe.
 *
 * Returns: #TRUE if @mat3x3 is the identity matrix, #FALSE otherwise.
 */
gboolean
pgm_mat3x3_is_identity (PgmMat3x3 *mat3x3)
{
  g_return_val_if_fail (mat3x3 != NULL, FALSE);

  return (mat3x3->m[0]==1.0f && mat3x3->m[1]==0.0f && mat3x3->m[2]==0.0f &&
          mat3x3->m[3]==0.0f && mat3x3->m[4]==1.0f && mat3x3->m[5]==0.0f &&
          mat3x3->m[6]==0.0f && mat3x3->m[7]==0.0f && mat3x3->m[8]==1.0f);
}

/**
 * pgm_mat3x3_inverse:
 * @mat3x3: A #PgmMat3x3 object.
 *
 * Retrieves the inverted @mat3x3 matrix.
 *
 * MT safe.
 *
 * Returns: the inverted @mat3x3 in a newly allocated #PgmMat3x3.
 */
PgmMat3x3*
pgm_mat3x3_inverse (PgmMat3x3 *mat3x3)
{
  /* FIXME */
  return NULL;
}

/**
 * pgm_mat3x3_transpose:
 * @mat3x3: A #PgmMat3x3 object.
 *
 * Retrieves the transposed @mat3x3 matrix.
 *
 * MT safe.
 *
 * Returns: the transposed @mat3x3 in a newly allocated #PgmMat3x3.
 */
PgmMat3x3*
pgm_mat3x3_transpose (PgmMat3x3 *mat3x3)
{
  g_return_val_if_fail (mat3x3 != NULL, NULL);

  return pgm_mat3x3_new_from_scalars (mat3x3->m[0], mat3x3->m[3], mat3x3->m[6],
                                      mat3x3->m[1], mat3x3->m[4], mat3x3->m[7],
                                      mat3x3->m[2], mat3x3->m[5], mat3x3->m[8]);
}

/**
 * pgm_mat3x3_add_scalar:
 * @mat3x3: A #PgmMat3x3 object.
 * @s: A scalar.
 *
 * Computes the sum @mat3x3 + @s.
 *
 * MT safe.
 *
 * Returns: the sum @mat3x3 + @s in a newly allocated #PgmMat3x3.
 */
PgmMat3x3*
pgm_mat3x3_add_scalar (PgmMat3x3 *mat3x3,
                       gfloat s)
{
  PgmMat3x3 *sum;

  g_return_val_if_fail (mat3x3 != NULL, NULL);

  sum = pgm_mat3x3_new ();

  sum->m[0] = s + mat3x3->m[0];  sum->m[1] = s + mat3x3->m[1];
  sum->m[2] = s + mat3x3->m[2];  sum->m[3] = s + mat3x3->m[3];
  sum->m[4] = s + mat3x3->m[4];  sum->m[5] = s + mat3x3->m[5];
  sum->m[6] = s + mat3x3->m[6];  sum->m[7] = s + mat3x3->m[7];
  sum->m[8] = s + mat3x3->m[8];

  return sum;
}

/**
 * pgm_mat3x3_add_mat3x3:
 * @mat3x3: A #PgmMat3x3 object.
 * @m: A #PgmMat3x3 object.
 *
 * Computes the sum @mat3x3 + @m.
 *
 * MT safe.
 *
 * Returns: the sum @mat3x3 + @m in a newly allocated #PgmMat3x3.
 */
PgmMat3x3*
pgm_mat3x3_add_mat3x3 (PgmMat3x3 *mat3x3,
                       const PgmMat3x3 *m)
{
  PgmMat3x3 *sum;

  g_return_val_if_fail (mat3x3 != NULL, NULL);
  g_return_val_if_fail (m != NULL, NULL);

  sum = pgm_mat3x3_new ();

  sum->m[0] = m->m[0] + mat3x3->m[0];  sum->m[1] = m->m[1] + mat3x3->m[1];
  sum->m[2] = m->m[2] + mat3x3->m[2];  sum->m[3] = m->m[3] + mat3x3->m[3];
  sum->m[4] = m->m[4] + mat3x3->m[4];  sum->m[5] = m->m[5] + mat3x3->m[5];
  sum->m[6] = m->m[6] + mat3x3->m[6];  sum->m[7] = m->m[7] + mat3x3->m[7];
  sum->m[8] = m->m[8] + mat3x3->m[8];

  return sum;
}

/**
 * pgm_mat3x3_substract_scalar:
 * @mat3x3: A #PgmMat3x3 object.
 * @s: A scalar.
 *
 * Computes the substraction @mat3x3 - @s.
 *
 * MT safe.
 *
 * Returns: the substraction @mat3x3 - @s in a newly allocated #PgmMat3x3.
 */
PgmMat3x3*
pgm_mat3x3_substract_scalar (PgmMat3x3 *mat3x3,
                             gfloat s)
{
  PgmMat3x3 *sub;

  g_return_val_if_fail (mat3x3 != NULL, NULL);

  sub = pgm_mat3x3_new ();

  sub->m[0] = mat3x3->m[0] - s; sub->m[1] = mat3x3->m[1] - s;
  sub->m[2] = mat3x3->m[2] - s; sub->m[3] = mat3x3->m[3] - s;
  sub->m[4] = mat3x3->m[4] - s; sub->m[5] = mat3x3->m[5] - s;
  sub->m[6] = mat3x3->m[6] - s; sub->m[7] = mat3x3->m[7] - s;
  sub->m[8] = mat3x3->m[8] - s;

  return sub;
}

/**
 * pgm_mat3x3_substract_mat3x3:
 * @mat3x3: A #PgmMat3x3 object.
 * @m: A #PgmMat3x3 object.
 *
 * Computes the substraction @mat3x3 - @m.
 *
 * MT safe.
 *
 * Returns: the substraction @mat3x3 - @m in a newly allocated #PgmMat3x3.
 */
PgmMat3x3*
pgm_mat3x3_substract_mat3x3 (PgmMat3x3 *mat3x3,
                             const PgmMat3x3 *m)
{
  PgmMat3x3 *sub;

  g_return_val_if_fail (mat3x3 != NULL, NULL);
  g_return_val_if_fail (m != NULL, NULL);

  sub = pgm_mat3x3_new ();

  sub->m[0] = mat3x3->m[0] - m->m[0];
  sub->m[1] = mat3x3->m[1] - m->m[1];
  sub->m[2] = mat3x3->m[2] - m->m[2];
  sub->m[3] = mat3x3->m[3] - m->m[3];
  sub->m[4] = mat3x3->m[4] - m->m[4];
  sub->m[5] = mat3x3->m[5] - m->m[5];
  sub->m[6] = mat3x3->m[6] - m->m[6];
  sub->m[7] = mat3x3->m[7] - m->m[7];
  sub->m[8] = mat3x3->m[8] - m->m[8];

  return sub;
}

/**
 * pgm_mat3x3_multiply_scalar:
 * @mat3x3: A #PgmMat3x3 object.
 * @s: A scalar.
 *
 * Computes the multiplication @mat3x3 . @s.
 *
 * MT safe.
 *
 * Returns: the multiplication @mat3x3 . @s in a newly allocated #PgmMat3x3.
 */
PgmMat3x3*
pgm_mat3x3_multiply_scalar (PgmMat3x3 *mat3x3,
                            gfloat s)
{
  PgmMat3x3 *mult;

  g_return_val_if_fail (mat3x3 != NULL, NULL);

  mult = pgm_mat3x3_new ();

  mult->m[0] = s * mat3x3->m[0];  mult->m[1] = s * mat3x3->m[1];
  mult->m[2] = s * mat3x3->m[2];  mult->m[3] = s * mat3x3->m[3];
  mult->m[4] = s * mat3x3->m[4];  mult->m[5] = s * mat3x3->m[5];
  mult->m[6] = s * mat3x3->m[6];  mult->m[7] = s * mat3x3->m[7];
  mult->m[8] = s * mat3x3->m[8];

  return mult;
}

/**
 * pgm_mat3x3_multiply_vec3:
 * @mat3x3: A #PgmMat3x3 object.
 * @v: A #PgmVec3 object.
 *
 * Computes the multiplication @mat3x3 . @v.
 *
 * MT safe.
 *
 * Returns: the multiplication @mat3x3 . @v in a newly allocated #PgmVec3.
 */
PgmVec3*
pgm_mat3x3_multiply_vec3 (PgmMat3x3 *mat3x3,
                          const PgmVec3 *v)
{
  PgmMat3x3 *m;
  PgmVec3 *mult;

  g_return_val_if_fail (mat3x3 != NULL, NULL);
  g_return_val_if_fail (v != NULL, NULL);

  mult = pgm_vec3_new ();
  m = mat3x3;

  mult->v[0] = m->m[0]*v->v[0] + m->m[1]*v->v[1] + m->m[2]*v->v[2];
  mult->v[1] = m->m[3]*v->v[0] + m->m[4]*v->v[1] + m->m[5]*v->v[2];
  mult->v[2] = m->m[6]*v->v[0] + m->m[7]*v->v[1] + m->m[8]*v->v[2];

  return mult;
}

/**
 * pgm_mat3x3_multiply_mat3x3:
 * @mat3x3: A #PgmMat3x3 object.
 * @m: A #PgmMat3x3 object.
 *
 * Computes the multiplication @mat3x3 . @m. Note that it is a
 * post-multiplication of @mat3x3 by @m.
 *
 * MT safe.
 *
 * Returns: the multiplication @mat3x3 . @m in a newly allocated #PgmMat3x3.
 */
PgmMat3x3*
pgm_mat3x3_multiply_mat3x3 (PgmMat3x3 *mat3x3,
                            const PgmMat3x3 *m)
{
  PgmMat3x3 *mult, *m1;
  const PgmMat3x3 *m2;

  g_return_val_if_fail (mat3x3 != NULL, NULL);
  g_return_val_if_fail (m != NULL, NULL);

  mult = pgm_mat3x3_new ();
  m1 = mat3x3;
  m2 = m;

  mult->m[0] = m1->m[0]*m2->m[0] + m1->m[1]*m2->m[3] + m1->m[2]*m2->m[6];
  mult->m[1] = m1->m[0]*m2->m[1] + m1->m[1]*m2->m[4] + m1->m[2]*m2->m[7];
  mult->m[2] = m1->m[0]*m2->m[2] + m1->m[1]*m2->m[5] + m1->m[2]*m2->m[8];

  mult->m[3] = m1->m[3]*m2->m[0] + m1->m[4]*m2->m[3] + m1->m[5]*m2->m[6];
  mult->m[4] = m1->m[3]*m2->m[1] + m1->m[4]*m2->m[4] + m1->m[5]*m2->m[7];
  mult->m[5] = m1->m[3]*m2->m[2] + m1->m[4]*m2->m[5] + m1->m[5]*m2->m[8];

  mult->m[6] = m1->m[6]*m2->m[0] + m1->m[7]*m2->m[3] + m1->m[8]*m2->m[6];
  mult->m[7] = m1->m[6]*m2->m[1] + m1->m[7]*m2->m[4] + m1->m[8]*m2->m[7];
  mult->m[8] = m1->m[6]*m2->m[2] + m1->m[7]*m2->m[5] + m1->m[8]*m2->m[8];

  return mult;
}

/**
 * pgm_mat3x3_to_string:
 * @mat3x3: A #PgmMat3x3 object.
 *
 * Converts @mat3x3 to a string representation.
 *
 * MT safe.
 *
 * Returns: a newly allocated string representing @mat3x3.
 */
gchar*
pgm_mat3x3_to_string (PgmMat3x3 *mat3x3)
{
  gchar *string;

  g_return_val_if_fail (mat3x3 != NULL, NULL);

  string = g_strdup_printf ("[%f %f %f]\n"
                            "|%f %f %f|\n"
                            "[%f %f %f]",
                            mat3x3->m[0], mat3x3->m[1], mat3x3->m[2],
                            mat3x3->m[3], mat3x3->m[4], mat3x3->m[5],
                            mat3x3->m[6], mat3x3->m[7], mat3x3->m[8]);

  return string;
}

/* 4x4 matrix public methods */

GType
pgm_mat4x4_get_type (void)
{
  static GType our_type = 0;

  if (G_UNLIKELY (!our_type))
    our_type = g_boxed_type_register_static ("PgmMat4x4",
                                             (GBoxedCopyFunc) pgm_mat4x4_copy,
                                             (GBoxedFreeFunc) pgm_mat4x4_free);

  return our_type;
}

/**
 * pgm_mat4x4_new:
 *
 * Creates a new #PgmMat4x4 with components initilized at 0.0f.
 *
 * MT safe.
 *
 * Returns: a new #PgmMat4x4 instance.
 */
PgmMat4x4*
pgm_mat4x4_new (void)
{
  PgmMat4x4 *mat4x4;

  mat4x4 = g_slice_new0 (PgmMat4x4);

  GST_DEBUG ("created new mat4x4(%p:"
             "[%.1f, %.1f, %.1f, %.1f], [%.1f, %.1f, %.1f, %.1f], "
             "[%.1f, %.1f, %.1f, %.1f], [%.1f, %.1f, %.1f, %.1f])", mat4x4,
             mat4x4->m[0], mat4x4->m[1], mat4x4->m[2], mat4x4->m[3],
             mat4x4->m[4], mat4x4->m[5], mat4x4->m[6], mat4x4->m[7],
             mat4x4->m[8], mat4x4->m[9], mat4x4->m[10], mat4x4->m[11],
             mat4x4->m[12], mat4x4->m[13], mat4x4->m[14], mat4x4->m[15]);

  return mat4x4;
}

/**
 * pgm_mat4x4_new_from_scalars:
 * @v0: the 1st component of the 1st vector.
 * @v1: the 2nd component of the 1st vector.
 * @v2: the 3rd component of the 1st vector.
 * @v3: the 4th component of the 1st vector.
 * @v4: the 1st component of the 2nd vector.
 * @v5: the 2nd component of the 2nd vector.
 * @v6: the 3rd component of the 2nd vector.
 * @v7: the 4th component of the 2nd vector.
 * @v8: the 1st component of the 3rd vector.
 * @v9: the 2nd component of the 3rd vector.
 * @v10: the 3rd component of the 3rd vector.
 * @v11: the 4th component of the 3rd vector.
 * @v12: the 1st component of the 4th vector.
 * @v13: the 2nd component of the 4th vector.
 * @v14: the 3rd component of the 4th vector.
 * @v15: the 4th component of the 4th vector.
 *
 * Creates a new #PgmMat4x4 with components initialized with the given scalars.
 *
 * MT safe.
 *
 * Returns: a new #PgmMat4x4 instance.
 */
PgmMat4x4*
pgm_mat4x4_new_from_scalars (gfloat v0,
                             gfloat v1,
                             gfloat v2,
                             gfloat v3,
                             gfloat v4,
                             gfloat v5,
                             gfloat v6,
                             gfloat v7,
                             gfloat v8,
                             gfloat v9,
                             gfloat v10,
                             gfloat v11,
                             gfloat v12,
                             gfloat v13,
                             gfloat v14,
                             gfloat v15)
{
  PgmMat4x4 *mat4x4;

  mat4x4 = g_slice_new (PgmMat4x4);

  mat4x4->m[0]=v0;   mat4x4->m[1]=v1;   mat4x4->m[2]=v2;   mat4x4->m[3]=v3;
  mat4x4->m[4]=v4;   mat4x4->m[5]=v5;   mat4x4->m[6]=v6;   mat4x4->m[7]=v7;
  mat4x4->m[8]=v8;   mat4x4->m[9]=v9;   mat4x4->m[10]=v10; mat4x4->m[11]=v11;
  mat4x4->m[12]=v12; mat4x4->m[13]=v13; mat4x4->m[14]=v14; mat4x4->m[15]=v15;

  GST_DEBUG ("created new mat4x4(%p:"
             "[%f, %f, %f, %f], [%f, %f, %f, %f], "
             "[%f, %f, %f, %f], [%f, %f, %f, %f])", mat4x4,
             mat4x4->m[0], mat4x4->m[1], mat4x4->m[2], mat4x4->m[3],
             mat4x4->m[4], mat4x4->m[5], mat4x4->m[6], mat4x4->m[7],
             mat4x4->m[8], mat4x4->m[9], mat4x4->m[10], mat4x4->m[11],
             mat4x4->m[12], mat4x4->m[13], mat4x4->m[14], mat4x4->m[15]);

  return mat4x4;
}

/**
 * pgm_mat4x4_new_from_vec4:
 * @v0: the 1st vector.
 * @v1: the 2nd vector.
 * @v2: the 3rd vector.
 * @v3: the 4th vector.
 *
 * Creates a new #PgmMat4x4 with components initialized with the given vectors.
 *
 * MT safe.
 *
 * Returns: a new #PgmMat4x4 instance.
 */
PgmMat4x4*
pgm_mat4x4_new_from_vec4 (const PgmVec4 *v0,
                          const PgmVec4 *v1,
                          const PgmVec4 *v2,
                          const PgmVec4 *v3)
{
  PgmMat4x4 *mat4x4;

  g_return_val_if_fail (v0 != NULL, NULL);
  g_return_val_if_fail (v1 != NULL, NULL);
  g_return_val_if_fail (v2 != NULL, NULL);
  g_return_val_if_fail (v3 != NULL, NULL);

  mat4x4 = g_slice_new (PgmMat4x4);

  mat4x4->m[0]=v0->v[0];  mat4x4->m[1]=v0->v[1];
  mat4x4->m[2]=v0->v[2];  mat4x4->m[3]=v0->v[3];
  mat4x4->m[4]=v1->v[0];  mat4x4->m[5]=v1->v[1];
  mat4x4->m[6]=v1->v[2];  mat4x4->m[7]=v1->v[3];
  mat4x4->m[8]=v2->v[0];  mat4x4->m[9]=v2->v[1];
  mat4x4->m[10]=v2->v[2]; mat4x4->m[11]=v2->v[3];
  mat4x4->m[12]=v3->v[0]; mat4x4->m[13]=v3->v[1];
  mat4x4->m[14]=v3->v[2]; mat4x4->m[15]=v3->v[3];

  GST_DEBUG ("created new mat4x4(%p:"
             "[%f, %f, %f, %f], [%f, %f, %f, %f], "
             "[%f, %f, %f, %f], [%f, %f, %f, %f])", mat4x4,
             mat4x4->m[0], mat4x4->m[1], mat4x4->m[2], mat4x4->m[3],
             mat4x4->m[4], mat4x4->m[5], mat4x4->m[6], mat4x4->m[7],
             mat4x4->m[8], mat4x4->m[9], mat4x4->m[10], mat4x4->m[11],
             mat4x4->m[12], mat4x4->m[13], mat4x4->m[14], mat4x4->m[15]);

  return mat4x4;
}

/**
 * pgm_mat4x4_new_identity:
 *
 * Creates a new #PgmMat4x4 identity matrix.
 *
 * MT safe.
 *
 * Returns: a new #PgmMat4x4 instance.
 */
PgmMat4x4*
pgm_mat4x4_new_identity (void)
{
  PgmMat4x4 *mat4x4;

  mat4x4 = g_slice_new0 (PgmMat4x4);

  mat4x4->m[0] = 1.0f;
  mat4x4->m[5] = 1.0f;
  mat4x4->m[10] = 1.0f;
  mat4x4->m[15] = 1.0f;

  GST_DEBUG ("created new mat4x4(%p:"
             "[%.1f, %.1f, %.1f, %.1f], [%.1f, %.1f, %.1f, %.1f], "
             "[%.1f, %.1f, %.1f, %.1f], [%.1f, %.1f, %.1f, %.1f])", mat4x4,
             mat4x4->m[0], mat4x4->m[1], mat4x4->m[2], mat4x4->m[3],
             mat4x4->m[4], mat4x4->m[5], mat4x4->m[6], mat4x4->m[7],
             mat4x4->m[8], mat4x4->m[9], mat4x4->m[10], mat4x4->m[11],
             mat4x4->m[12], mat4x4->m[13], mat4x4->m[14], mat4x4->m[15]);

  return mat4x4;
}

/**
 * pgm_mat4x4_new_predefined:
 * @predefined: The predefined 4x4 matrix.
 *
 * Creates a new #PgmMat4x4 matrix corresponding to @predefined.
 *
 * MT safe.
 *
 * Returns: a new #PgmMat4x4 instance.
 */
PgmMat4x4*
pgm_mat4x4_new_predefined (PgmMat4x4Predefined predefined)
{
  PgmMat4x4 *mat4x4 = NULL;

  switch (predefined)
    {
    case PGM_MAT4X4_FLIP_HORIZONTAL:
      mat4x4 = pgm_mat4x4_new_from_scalars (MAT4X4_FLIP_HORIZ_SCALARS);
      break;

    case PGM_MAT4X4_FLIP_VERTICAL:
      mat4x4 = pgm_mat4x4_new_from_scalars (MAT4X4_FLIP_VERT_SCALARS);
      break;

    case PGM_MAT4X4_ROTATE_CLOCKWISE:
      mat4x4 = pgm_mat4x4_new_from_scalars (MAT4X4_ROTATE_CW_SCALARS);
      break;

    case PGM_MAT4X4_ROTATE_COUNTER_CLOCKWISE:
      mat4x4 = pgm_mat4x4_new_from_scalars (MAT4X4_ROTATE_CCW_SCALARS);
      break;

    default:
      break;
    }

  return mat4x4;
}

/**
 * pgm_mat4x4_new_translate_from_vec3:
 * @t: A #PgmVec3 object representing the translation.
 *
 * Creates a new #PgmMat4x4 translation matrix.
 *
 * MT safe.
 *
 * Returns: a new #PgmMat4x4 instance.
 */
PgmMat4x4*
pgm_mat4x4_new_translate_from_vec3 (const PgmVec3 *t)
{
  PgmMat4x4 *mat4x4;

  g_return_val_if_fail (t != NULL, NULL);

  mat4x4 = g_slice_new0 (PgmMat4x4);

  mat4x4->m[0] = 1.0f;
  mat4x4->m[3] = t->v[0];
  mat4x4->m[5] = 1.0f;
  mat4x4->m[7] = t->v[1];
  mat4x4->m[10] = 1.0f;
  mat4x4->m[11] = t->v[2];
  mat4x4->m[15] = 1.0f;

  GST_DEBUG ("created new mat4x4(%p:"
             "[%f, %f, %f, %f], [%f, %f, %f, %f], "
             "[%f, %f, %f, %f], [%f, %f, %f, %f])", mat4x4,
             mat4x4->m[0], mat4x4->m[1], mat4x4->m[2], mat4x4->m[3],
             mat4x4->m[4], mat4x4->m[5], mat4x4->m[6], mat4x4->m[7],
             mat4x4->m[8], mat4x4->m[9], mat4x4->m[10], mat4x4->m[11],
             mat4x4->m[12], mat4x4->m[13], mat4x4->m[14], mat4x4->m[15]);

  return mat4x4;
}

/**
 * pgm_mat4x4_new_translate_from_scalars:
 * @tx: The translation on the x cardinal axis.
 * @ty: The translation on the y cardinal axis.
 * @tz: The translation on the z cardinal axis.
 *
 * Creates a new #PgmMat4x4 translation matrix.
 *
 * MT safe.
 *
 * Returns: a new #PgmMat4x4 instance.
 */
PgmMat4x4*
pgm_mat4x4_new_translate_from_scalars (gfloat tx,
                                       gfloat ty,
                                       gfloat tz)
{
  PgmMat4x4 *mat4x4;

  mat4x4 = g_slice_new0 (PgmMat4x4);

  mat4x4->m[0] = 1.0f;
  mat4x4->m[3] = tx;
  mat4x4->m[5] = 1.0f;
  mat4x4->m[7] = ty;
  mat4x4->m[10] = 1.0f;
  mat4x4->m[11] = tz;
  mat4x4->m[15] = 1.0f;

  GST_DEBUG ("created new mat4x4(%p:"
             "[%f, %f, %f, %f], [%f, %f, %f, %f], "
             "[%f, %f, %f, %f], [%f, %f, %f, %f])", mat4x4,
             mat4x4->m[0], mat4x4->m[1], mat4x4->m[2], mat4x4->m[3],
             mat4x4->m[4], mat4x4->m[5], mat4x4->m[6], mat4x4->m[7],
             mat4x4->m[8], mat4x4->m[9], mat4x4->m[10], mat4x4->m[11],
             mat4x4->m[12], mat4x4->m[13], mat4x4->m[14], mat4x4->m[15]);

  return mat4x4;
}

/**
 * pgm_mat4x4_new_scale_from_vec3:
 * @s: A #PgmVec3 object representing the scaling.
 *
 * Creates a new #PgmMat4x4 scaling matrix.
 *
 * MT safe.
 *
 * Returns: a new #PgmMat4x4 instance.
 */
PgmMat4x4*
pgm_mat4x4_new_scale_from_vec3 (const PgmVec3 *s)
{
  PgmMat4x4 *mat4x4;

  g_return_val_if_fail (s != NULL, NULL);

  mat4x4 = g_slice_new0 (PgmMat4x4);

  mat4x4->m[0] = s->v[0];
  mat4x4->m[5] = s->v[1];
  mat4x4->m[10] = s->v[2];
  mat4x4->m[15] = 1.0f;

  GST_DEBUG ("created new mat4x4(%p:"
             "[%f, %f, %f, %f], [%f, %f, %f, %f], "
             "[%f, %f, %f, %f], [%f, %f, %f, %f])", mat4x4,
             mat4x4->m[0], mat4x4->m[1], mat4x4->m[2], mat4x4->m[3],
             mat4x4->m[4], mat4x4->m[5], mat4x4->m[6], mat4x4->m[7],
             mat4x4->m[8], mat4x4->m[9], mat4x4->m[10], mat4x4->m[11],
             mat4x4->m[12], mat4x4->m[13], mat4x4->m[14], mat4x4->m[15]);

  return mat4x4;
}

/**
 * pgm_mat4x4_new_scale_from_scalars:
 * @sx: The scale on the x cardinal axis.
 * @sy: The scale on the y cardinal axis.
 * @sz: The scale on the z cardinal axis.
 *
 * Creates a new #PgmMat4x4 scaling matrix.
 *
 * MT safe.
 *
 * Returns: a new #PgmMat4x4 instance.
 */
PgmMat4x4*
pgm_mat4x4_new_scale_from_scalars (gfloat sx,
                                   gfloat sy,
                                   gfloat sz)
{
  PgmMat4x4 *mat4x4;

  mat4x4 = g_slice_new0 (PgmMat4x4);

  mat4x4->m[0] = sx;
  mat4x4->m[5] = sy;
  mat4x4->m[10] = sz;
  mat4x4->m[15] = 1.0f;

  GST_DEBUG ("created new mat4x4(%p:"
             "[%f, %f, %f, %f], [%f, %f, %f, %f], "
             "[%f, %f, %f, %f], [%f, %f, %f, %f])", mat4x4,
             mat4x4->m[0], mat4x4->m[1], mat4x4->m[2], mat4x4->m[3],
             mat4x4->m[4], mat4x4->m[5], mat4x4->m[6], mat4x4->m[7],
             mat4x4->m[8], mat4x4->m[9], mat4x4->m[10], mat4x4->m[11],
             mat4x4->m[12], mat4x4->m[13], mat4x4->m[14], mat4x4->m[15]);

  return mat4x4;
}

/**
 * pgm_mat4x4_new_rotate_x:
 * @angle: The angle of rotation, in radians.
 *
 * Creates a new #PgmMat4x4 matrix for a rotation about the x cardinal axis.
 *
 * MT safe.
 *
 * Returns: a new #PgmMat4x4 instance.
 */
PgmMat4x4*
pgm_mat4x4_new_rotate_x (gfloat angle)
{
  PgmMat4x4 *mat4x4;
  gfloat s, c;

  mat4x4 = g_slice_new0 (PgmMat4x4);

  PGM_SINCOS (angle, &s, &c);

  mat4x4->m[0] = 1.0f;
  mat4x4->m[5] = c;
  mat4x4->m[6] = -s;
  mat4x4->m[9] = s;
  mat4x4->m[10] = c;
  mat4x4->m[15] = 1.0f;

  GST_DEBUG ("created new mat4x4(%p:"
             "[%f, %f, %f, %f], [%f, %f, %f, %f], "
             "[%f, %f, %f, %f], [%f, %f, %f, %f])", mat4x4,
             mat4x4->m[0], mat4x4->m[1], mat4x4->m[2], mat4x4->m[3],
             mat4x4->m[4], mat4x4->m[5], mat4x4->m[6], mat4x4->m[7],
             mat4x4->m[8], mat4x4->m[9], mat4x4->m[10], mat4x4->m[11],
             mat4x4->m[12], mat4x4->m[13], mat4x4->m[14], mat4x4->m[15]);

  return mat4x4;
}

/**
 * pgm_mat4x4_new_rotate_y:
 * @angle: The angle of rotation, in radians.
 *
 * Creates a new #PgmMat4x4 matrix for a rotation about the y cardinal axis.
 *
 * MT safe.
 *
 * Returns: a new #PgmMat4x4 instance.
 */
PgmMat4x4*
pgm_mat4x4_new_rotate_y (gfloat angle)
{
  PgmMat4x4 *mat4x4;
  gfloat s, c;

  mat4x4 = g_slice_new0 (PgmMat4x4);

  PGM_SINCOS (angle, &s, &c);

  mat4x4->m[0] = c;
  mat4x4->m[2] = s;
  mat4x4->m[5] = 1.0f;
  mat4x4->m[8] = -s;
  mat4x4->m[10] = c;
  mat4x4->m[15] = 1.0f;

  GST_DEBUG ("created new mat4x4(%p:"
             "[%f, %f, %f, %f], [%f, %f, %f, %f], "
             "[%f, %f, %f, %f], [%f, %f, %f, %f])", mat4x4,
             mat4x4->m[0], mat4x4->m[1], mat4x4->m[2], mat4x4->m[3],
             mat4x4->m[4], mat4x4->m[5], mat4x4->m[6], mat4x4->m[7],
             mat4x4->m[8], mat4x4->m[9], mat4x4->m[10], mat4x4->m[11],
             mat4x4->m[12], mat4x4->m[13], mat4x4->m[14], mat4x4->m[15]);

  return mat4x4;
}

/**
 * pgm_mat4x4_new_rotate_z:
 * @angle: The angle of rotation, in radians.
 *
 * Creates a new #PgmMat4x4 matrix for a rotation about the z cardinal axis.
 *
 * MT safe.
 *
 * Returns: a new #PgmMat4x4 instance.
 */
PgmMat4x4*
pgm_mat4x4_new_rotate_z (gfloat angle)
{
  PgmMat4x4 *mat4x4;
  gfloat s, c;

  mat4x4 = g_slice_new0 (PgmMat4x4);

  PGM_SINCOS (angle, &s, &c);

  mat4x4->m[0] = c;
  mat4x4->m[1] = -s;
  mat4x4->m[4] = s;
  mat4x4->m[5] = c;
  mat4x4->m[10] = 1.0f;
  mat4x4->m[15] = 1.0f;

  GST_DEBUG ("created new mat4x4(%p:"
             "[%f, %f, %f, %f], [%f, %f, %f, %f], "
             "[%f, %f, %f, %f], [%f, %f, %f, %f])", mat4x4,
             mat4x4->m[0], mat4x4->m[1], mat4x4->m[2], mat4x4->m[3],
             mat4x4->m[4], mat4x4->m[5], mat4x4->m[6], mat4x4->m[7],
             mat4x4->m[8], mat4x4->m[9], mat4x4->m[10], mat4x4->m[11],
             mat4x4->m[12], mat4x4->m[13], mat4x4->m[14], mat4x4->m[15]);

  return mat4x4;
}

/**
 * pgm_mat4x4_new_rotate_axis_from_vec3:
 * @angle: The angle of rotation, in radians.
 * @axis: A #PgmVec3 object representing the axis of rotation.
 *
 * Creates a new #PgmMat4x4 matrix for a rotation about @axis.
 *
 * MT safe.
 *
 * Returns: a new #PgmMat4x4 instance.
 */
PgmMat4x4*
pgm_mat4x4_new_rotate_axis_from_vec3 (gfloat angle,
                                      const PgmVec3 *axis)
{
  PgmMat4x4 *mat4x4;
  gfloat s, c, a, ax, ay, az;

  g_return_val_if_fail (axis != NULL, NULL);

  mat4x4 = g_slice_new0 (PgmMat4x4);

  PGM_SINCOS (angle, &s, &c);

  a = 1.0f - c;
  ax = a * axis->v[0];
  ay = a * axis->v[1];
  az = a * axis->v[2];

  mat4x4->m[0] = ax * axis->v[0] + c;
  mat4x4->m[1] = ax * axis->v[1] - axis->v[2] * s;
  mat4x4->m[2] = ax * axis->v[2] + axis->v[1] * s;
  mat4x4->m[4] = ay * axis->v[0] + axis->v[2] * s;
  mat4x4->m[5] = ay * axis->v[1] + c;
  mat4x4->m[6] = ay * axis->v[2] - axis->v[0] * s;
  mat4x4->m[8] = az * axis->v[0] - axis->v[1] * s;
  mat4x4->m[9] = az * axis->v[1] + axis->v[0] * s;
  mat4x4->m[10] = az * axis->v[2] + c;
  mat4x4->m[15] = 1.0f;

  GST_DEBUG ("created new mat4x4(%p:"
             "[%f, %f, %f, %f], [%f, %f, %f, %f], "
             "[%f, %f, %f, %f], [%f, %f, %f, %f])", mat4x4,
             mat4x4->m[0], mat4x4->m[1], mat4x4->m[2], mat4x4->m[3],
             mat4x4->m[4], mat4x4->m[5], mat4x4->m[6], mat4x4->m[7],
             mat4x4->m[8], mat4x4->m[9], mat4x4->m[10], mat4x4->m[11],
             mat4x4->m[12], mat4x4->m[13], mat4x4->m[14], mat4x4->m[15]);

  return mat4x4;
}

/**
 * pgm_mat4x4_new_rotate_axis_from_scalars:
 * @angle: The angle of rotation, in radians.
 * @axis_x: The x component of the axis of rotation.
 * @axis_y: The y component of the axis of rotation.
 * @axis_z: The z component of the axis of rotation.
 *
 * Creates a new #PgmMat4x4 matrix for a rotation about the given axis.
 *
 * MT safe.
 *
 * Returns: a new #PgmMat4x4 instance.
 */
PgmMat4x4*
pgm_mat4x4_new_rotate_axis_from_scalars (gfloat angle,
                                         gfloat axis_x,
                                         gfloat axis_y,
                                         gfloat axis_z)
{
  PgmMat4x4 *mat4x4;
  gfloat s, c, a, ax, ay, az;

  mat4x4 = g_slice_new0 (PgmMat4x4);

  PGM_SINCOS (angle, &s, &c);

  a = 1.0f - c;
  ax = a * axis_x;
  ay = a * axis_y;
  az = a * axis_z;

  mat4x4->m[0] = ax * axis_x + c;
  mat4x4->m[1] = ax * axis_y - axis_z * s;
  mat4x4->m[2] = ax * axis_z + axis_y * s;
  mat4x4->m[4] = ay * axis_x + axis_z * s;
  mat4x4->m[5] = ay * axis_y + c;
  mat4x4->m[6] = ay * axis_z - axis_x * s;
  mat4x4->m[8] = az * axis_x - axis_y * s;
  mat4x4->m[9] = az * axis_y + axis_x * s;
  mat4x4->m[10] = az * axis_z + c;
  mat4x4->m[15] = 1.0f;

  GST_DEBUG ("created new mat4x4(%p:"
             "[%f, %f, %f, %f], [%f, %f, %f, %f], "
             "[%f, %f, %f, %f], [%f, %f, %f, %f])", mat4x4,
             mat4x4->m[0], mat4x4->m[1], mat4x4->m[2], mat4x4->m[3],
             mat4x4->m[4], mat4x4->m[5], mat4x4->m[6], mat4x4->m[7],
             mat4x4->m[8], mat4x4->m[9], mat4x4->m[10], mat4x4->m[11],
             mat4x4->m[12], mat4x4->m[13], mat4x4->m[14], mat4x4->m[15]);

  return mat4x4;
}

/**
 * pgm_mat4x4_copy:
 * @mat4x4: a #PgmMat4x4 object.
 *
 * Copies @mat4x4.
 *
 * MT safe.
 *
 * Returns: a newly allocated #PgmMat4x4.
 */
PgmMat4x4*
pgm_mat4x4_copy (PgmMat4x4 *mat4x4)
{
  PgmMat4x4 *new_mat4x4;

  g_return_val_if_fail (mat4x4 != NULL, NULL);

  new_mat4x4 = pgm_mat4x4_new ();
  memcpy (new_mat4x4, mat4x4, sizeof (PgmMat4x4));

  return new_mat4x4;
}

/**
 * pgm_mat4x4_free:
 * @mat4x4: A #PgmMat4x4 object.
 *
 * Frees all resources used by @mat4x4.
 *
 * MT safe.
 */
void
pgm_mat4x4_free (PgmMat4x4 *mat4x4)
{
  g_return_if_fail (mat4x4 != NULL);

  GST_DEBUG ("freeing mat4x4(%p:"
             "[%f, %f, %f, %f], [%f, %f, %f, %f], "
             "[%f, %f, %f, %f], [%f, %f, %f, %f])", mat4x4,
             mat4x4->m[0], mat4x4->m[1], mat4x4->m[2], mat4x4->m[3],
             mat4x4->m[4], mat4x4->m[5], mat4x4->m[6], mat4x4->m[7],
             mat4x4->m[8], mat4x4->m[9], mat4x4->m[10], mat4x4->m[11],
             mat4x4->m[12], mat4x4->m[13], mat4x4->m[14], mat4x4->m[15]);

  g_slice_free (PgmMat4x4, mat4x4);
}

/**
 * pgm_mat4x4_set_from_scalars:
 * @mat4x4: A #PgmMat4x4 object.
 * @v0: the 1st component of the 1st vector.
 * @v1: the 2nd component of the 1st vector.
 * @v2: the 3rd component of the 1st vector.
 * @v3: the 4th component of the 1st vector.
 * @v4: the 1st component of the 2nd vector.
 * @v5: the 2nd component of the 2nd vector.
 * @v6: the 3rd component of the 2nd vector.
 * @v7: the 4th component of the 2nd vector.
 * @v8: the 1st component of the 3rd vector.
 * @v9: the 2nd component of the 3rd vector.
 * @v10: the 3rd component of the 3rd vector.
 * @v11: the 4th component of the 3rd vector.
 * @v12: the 1st component of the 4th vector.
 * @v13: the 2nd component of the 4th vector.
 * @v14: the 3rd component of the 4th vector.
 * @v15: the 4th component of the 4th vector.
 *
 * Set the components of @mat4x4 with the given scalars.
 *
 * MT safe.
 */
void
pgm_mat4x4_set_from_scalars (PgmMat4x4 *mat4x4,
                             gfloat v0,
                             gfloat v1,
                             gfloat v2,
                             gfloat v3,
                             gfloat v4,
                             gfloat v5,
                             gfloat v6,
                             gfloat v7,
                             gfloat v8,
                             gfloat v9,
                             gfloat v10,
                             gfloat v11,
                             gfloat v12,
                             gfloat v13,
                             gfloat v14,
                             gfloat v15)

{
  g_return_if_fail (mat4x4 != NULL);

  mat4x4->m[0]=v0;   mat4x4->m[1]=v1;   mat4x4->m[2]=v2;   mat4x4->m[3]=v3;
  mat4x4->m[4]=v4;   mat4x4->m[5]=v5;   mat4x4->m[6]=v6;   mat4x4->m[7]=v7;
  mat4x4->m[8]=v8;   mat4x4->m[9]=v9;   mat4x4->m[10]=v10; mat4x4->m[11]=v11;
  mat4x4->m[12]=v12; mat4x4->m[13]=v13; mat4x4->m[14]=v14; mat4x4->m[15]=v15;
}

/**
 * pgm_mat4x4_set_from_vec4:
 * @mat4x4: A #PgmMat4x4 object.
 * @v0: the 1st vector.
 * @v1: the 2nd vector.
 * @v2: the 3rd vector.
 * @v3: the 4th vector.
 *
 * Set the components of @mat4x4 with the given vectors.
 *
 * MT safe.
 */
void
pgm_mat4x4_set_from_vec4 (PgmMat4x4 *mat4x4,
                          const PgmVec4 *v0,
                          const PgmVec4 *v1,
                          const PgmVec4 *v2,
                          const PgmVec4 *v3)
{
  g_return_if_fail (mat4x4 != NULL);

  g_return_if_fail (v0 != NULL);
  g_return_if_fail (v1 != NULL);
  g_return_if_fail (v2 != NULL);
  g_return_if_fail (v3 != NULL);

  mat4x4->m[0]=v0->v[0];  mat4x4->m[1]=v0->v[1];
  mat4x4->m[2]=v0->v[2];  mat4x4->m[3]=v0->v[3];
  mat4x4->m[4]=v1->v[0];  mat4x4->m[5]=v1->v[1];
  mat4x4->m[6]=v1->v[2];  mat4x4->m[7]=v1->v[3];
  mat4x4->m[8]=v2->v[0];  mat4x4->m[9]=v2->v[1];
  mat4x4->m[10]=v2->v[2]; mat4x4->m[11]=v2->v[3];
  mat4x4->m[12]=v3->v[0]; mat4x4->m[13]=v3->v[1];
  mat4x4->m[14]=v3->v[2]; mat4x4->m[15]=v3->v[3];
}

/**
 * pgm_mat4x4_set_from_mat4x4:
 * @mat4x4: A #PgmMat4x4 object.
 * @m: A #PgmMat4x4 object.
 *
 * Sets the components of @mat4x4 with the given matrix.
 *
 * MT safe.
 */
void
pgm_mat4x4_set_from_mat4x4 (PgmMat4x4 *mat4x4,
                            const PgmMat4x4 *m)
{
  g_return_if_fail (mat4x4 != NULL);
  g_return_if_fail (m != NULL);

  memcpy (mat4x4, m, sizeof (PgmMat4x4));
}

/**
 * pgm_mat4x4_is_identity:
 * @mat4x4: A #PgmMat4x4 object.
 *
 * Retrieves whether or not @mat4x4 is the identity matrix.
 *
 * MT safe.
 *
 * Returns: #TRUE if @mat4x4 is the identity matrix, #FALSE otherwise.
 */
gboolean
pgm_mat4x4_is_identity (PgmMat4x4 *mat4x4)
{
  g_return_val_if_fail (mat4x4 != NULL, FALSE);

  return (mat4x4->m[0]==1.0f && mat4x4->m[1]==0.0f &&
          mat4x4->m[2]==0.0f && mat4x4->m[3]==0.0f &&
          mat4x4->m[4]==0.0f && mat4x4->m[5]==1.0f &&
          mat4x4->m[6]==0.0f && mat4x4->m[7]==0.0f &&
          mat4x4->m[8]==0.0f && mat4x4->m[9]==0.0f &&
          mat4x4->m[10]==1.0f && mat4x4->m[11]==0.0f &&
          mat4x4->m[12]==0.0f && mat4x4->m[13]==0.0f &&
          mat4x4->m[14]==0.0f && mat4x4->m[15]==1.0f);
}

/**
 * pgm_mat4x4_inverse:
 * @mat4x4: A #PgmMat4x4 object.
 *
 * Retrieves the inverted @mat4x4 matrix.
 *
 * MT safe.
 *
 * Returns: the inverted @mat4x4 in a newly allocated #PgmMat4x4, or NULL if
 * @mat4x4 is non-invertible (singular).
 */
PgmMat4x4*
pgm_mat4x4_inverse (PgmMat4x4 *mat4x4)
{
  gfloat m0, m1, m2, m3, s;
  gfloat *r0, *r1, *r2, *r3;
  gfloat wtmp[4][8];
  gfloat *tmp;

  /* Calculates the inverse matrix by performing the gaussian matrix
   * reduction with partial pivoting followed by back/substitution with
   * the loops manually unrolled. */

  g_return_val_if_fail (mat4x4 != NULL, NULL);

  r0 = wtmp[0], r1 = wtmp[1], r2 = wtmp[2], r3 = wtmp[3];

  r0[0] = mat4x4->m[0], r0[1] = mat4x4->m[1],
  r0[2] = mat4x4->m[2], r0[3] = mat4x4->m[3],
  r0[4] = 1.0, r0[5] = r0[6] = r0[7] = 0.0,

  r1[0] = mat4x4->m[4], r1[1] = mat4x4->m[5],
  r1[2] = mat4x4->m[6], r1[3] = mat4x4->m[7],
  r1[5] = 1.0, r1[4] = r1[6] = r1[7] = 0.0,

  r2[0] = mat4x4->m[8], r2[1] = mat4x4->m[9],
  r2[2] = mat4x4->m[10], r2[3] = mat4x4->m[11],
  r2[6] = 1.0, r2[4] = r2[5] = r2[7] = 0.0,

  r3[0] = mat4x4->m[12], r3[1] = mat4x4->m[13],
  r3[2] = mat4x4->m[14], r3[3] = mat4x4->m[15],
  r3[7] = 1.0, r3[4] = r3[5] = r3[6] = 0.0;

  /* choose pivot, or die */
  if (PGM_FABSF (r3[0]) > PGM_FABSF (r2[0])) { tmp = r3; r3 = r2; r2 = tmp; }
  if (PGM_FABSF (r2[0]) > PGM_FABSF (r1[0])) { tmp = r2; r2 = r1; r1 = tmp; }
  if (PGM_FABSF (r1[0]) > PGM_FABSF (r0[0])) { tmp = r1; r1 = r0; r0 = tmp; }
  if (0.0 == r0[0]) return NULL;

  /* eliminate first variable */
  m1 = r1[0]/r0[0]; m2 = r2[0]/r0[0]; m3 = r3[0]/r0[0];
  s = r0[1]; r1[1] -= m1 * s; r2[1] -= m2 * s; r3[1] -= m3 * s;
  s = r0[2]; r1[2] -= m1 * s; r2[2] -= m2 * s; r3[2] -= m3 * s;
  s = r0[3]; r1[3] -= m1 * s; r2[3] -= m2 * s; r3[3] -= m3 * s;
  s = r0[4]; if (s != 0.0) { r1[4] -= m1 * s; r2[4] -= m2 * s; r3[4] -= m3 * s; }
  s = r0[5]; if (s != 0.0) { r1[5] -= m1 * s; r2[5] -= m2 * s; r3[5] -= m3 * s; }
  s = r0[6]; if (s != 0.0) { r1[6] -= m1 * s; r2[6] -= m2 * s; r3[6] -= m3 * s; }
  s = r0[7]; if (s != 0.0) { r1[7] -= m1 * s; r2[7] -= m2 * s; r3[7] -= m3 * s; }

  /* choose pivot, or die */
  if (PGM_FABSF (r3[1]) > PGM_FABSF(r2[1])) { tmp = r3; r3 = r2; r2 = tmp; }
  if (PGM_FABSF (r2[1]) > PGM_FABSF(r1[1])) { tmp = r2; r2 = r1; r1 = tmp; }
  if (0.0 == r1[1]) return NULL;

  /* eliminate second variable */
  m2 = r2[1]/r1[1]; m3 = r3[1]/r1[1];
  r2[2] -= m2 * r1[2]; r3[2] -= m3 * r1[2];
  r2[3] -= m2 * r1[3]; r3[3] -= m3 * r1[3];
  s = r1[4]; if (0.0 != s) { r2[4] -= m2 * s; r3[4] -= m3 * s; }
  s = r1[5]; if (0.0 != s) { r2[5] -= m2 * s; r3[5] -= m3 * s; }
  s = r1[6]; if (0.0 != s) { r2[6] -= m2 * s; r3[6] -= m3 * s; }
  s = r1[7]; if (0.0 != s) { r2[7] -= m2 * s; r3[7] -= m3 * s; }

  /* choose pivot, or die */
  if (PGM_FABSF (r3[2]) > PGM_FABSF (r2[2])) { tmp = r3; r3 = r2; r2 = tmp; }
  if (0.0 == r2[2]) return NULL;

  /* eliminate third variable */
  m3 = r3[2]/r2[2];
  r3[3] -= m3 * r2[3], r3[4] -= m3 * r2[4],
  r3[5] -= m3 * r2[5], r3[6] -= m3 * r2[6],
  r3[7] -= m3 * r2[7];

  /* last check */
  if (0.0 == r3[3]) return NULL;

  /* back substitute row 3 */
  s = 1.0f/r3[3];
  r3[4] *= s; r3[5] *= s; r3[6] *= s; r3[7] *= s;

  /* back substitute row 2 */
  m2 = r2[3];
  s = 1.0F/r2[2];
  r2[4] = s * (r2[4] - r3[4] * m2), r2[5] = s * (r2[5] - r3[5] * m2),
  r2[6] = s * (r2[6] - r3[6] * m2), r2[7] = s * (r2[7] - r3[7] * m2);
  m1 = r1[3];
  r1[4] -= r3[4] * m1, r1[5] -= r3[5] * m1,
  r1[6] -= r3[6] * m1, r1[7] -= r3[7] * m1;
  m0 = r0[3];
  r0[4] -= r3[4] * m0, r0[5] -= r3[5] * m0,
  r0[6] -= r3[6] * m0, r0[7] -= r3[7] * m0;

  /* back substitute row 1 */
  m1 = r1[2];
  s = 1.0F/r1[1];
  r1[4] = s * (r1[4] - r2[4] * m1), r1[5] = s * (r1[5] - r2[5] * m1),
  r1[6] = s * (r1[6] - r2[6] * m1), r1[7] = s * (r1[7] - r2[7] * m1);
  m0 = r0[2];
  r0[4] -= r2[4] * m0, r0[5] -= r2[5] * m0,
  r0[6] -= r2[6] * m0, r0[7] -= r2[7] * m0;

  /* back substitute row 0 */
  m0 = r0[1];
  s = 1.0F/r0[0];
  r0[4] = s * (r0[4] - r1[4] * m0), r0[5] = s * (r0[5] - r1[5] * m0),
  r0[6] = s * (r0[6] - r1[6] * m0), r0[7] = s * (r0[7] - r1[7] * m0);

  return pgm_mat4x4_new_from_scalars (r0[4], r0[5], r0[6], r0[7],
                                      r1[4], r1[5], r1[6], r1[7],
                                      r2[4], r2[5], r2[6], r2[7],
                                      r3[4], r3[5], r3[6], r3[7]);
}

/**
 * pgm_mat4x4_transpose:
 * @mat4x4: A #PgmMat4x4 object.
 *
 * Retrieves the transposed @mat4x4 matrix.
 *
 * MT safe.
 *
 * Returns: the transposed @mat4x4 in a newly allocated #PgmMat4x4.
 */
PgmMat4x4*
pgm_mat4x4_transpose (PgmMat4x4 *mat4x4)
{
  g_return_val_if_fail (mat4x4 != NULL, NULL);

  return pgm_mat4x4_new_from_scalars
    (mat4x4->m[0], mat4x4->m[4], mat4x4->m[8], mat4x4->m[12],
     mat4x4->m[1], mat4x4->m[5], mat4x4->m[9], mat4x4->m[13],
     mat4x4->m[2], mat4x4->m[6], mat4x4->m[10], mat4x4->m[14],
     mat4x4->m[3], mat4x4->m[7], mat4x4->m[11], mat4x4->m[15]);
}

/**
 * pgm_mat4x4_translate_from_vec3:
 * @mat4x4: A #PgmMat4x4 object.
 * @t: A #PgmVec3 object representing the translation to apply.
 *
 * Post multiplies @mat4x4 by a translation matrix with the product replacing
 * @mat4x4.
 *
 * MT safe.
 */
void
pgm_mat4x4_translate_from_vec3 (PgmMat4x4 *mat4x4,
                                const PgmVec3 *t)
{
  g_return_if_fail (mat4x4 != NULL);
  g_return_if_fail (t != NULL);

  mat4x4->m[3] = mat4x4->m[0]*t->v[0] + mat4x4->m[1]*t->v[1]
    + mat4x4->m[2]*t->v[2] + mat4x4->m[3];
  mat4x4->m[7] = mat4x4->m[4]*t->v[0] + mat4x4->m[5]*t->v[1]
    + mat4x4->m[6]*t->v[2] + mat4x4->m[7];
  mat4x4->m[11] = mat4x4->m[8]*t->v[0] + mat4x4->m[9]*t->v[1]
    + mat4x4->m[10]*t->v[2] + mat4x4->m[11];
  mat4x4->m[15] = mat4x4->m[12]*t->v[0] + mat4x4->m[13]*t->v[1]
    + mat4x4->m[14]*t->v[2] + mat4x4->m[15];
}

/**
 * pgm_mat4x4_translate_from_scalars:
 * @mat4x4: A #PgmMat4x4 object.
 * @tx: The translation on the x cardinal axis.
 * @ty: The translation on the y cardinal axis.
 * @tz: The translation on the z cardinal axis.
 *
 * Post multiplies @mat4x4 by a translation matrix with the product replacing
 * @mat4x4.
 *
 * MT safe.
 */
void
pgm_mat4x4_translate_from_scalars (PgmMat4x4 *mat4x4,
                                   gfloat tx,
                                   gfloat ty,
                                   gfloat tz)
{
  g_return_if_fail (mat4x4 != NULL);

  mat4x4->m[3] = mat4x4->m[0]*tx + mat4x4->m[1]*ty
    + mat4x4->m[2]*tz + mat4x4->m[3];
  mat4x4->m[7] = mat4x4->m[4]*tx + mat4x4->m[5]*ty
    + mat4x4->m[6]*tz + mat4x4->m[7];
  mat4x4->m[11] = mat4x4->m[8]*tx + mat4x4->m[9]*ty
    + mat4x4->m[10]*tz + mat4x4->m[11];
  mat4x4->m[15] = mat4x4->m[12]*tx + mat4x4->m[13]*ty
    + mat4x4->m[14]*tz + mat4x4->m[15];
}

/**
 * pgm_mat4x4_scale_from_vec3:
 * @mat4x4: A #PgmMat4x4 object.
 * @s: A #PgmVec3 object representing the scale to apply.
 *
 * Post multiplies @mat4x4 by a scaling matrix with the product replacing
 * @mat4x4.
 *
 * MT safe.
 */
void
pgm_mat4x4_scale_from_vec3 (PgmMat4x4 *mat4x4,
                            const PgmVec3 *s)
{
  g_return_if_fail (mat4x4 != NULL);
  g_return_if_fail (s != NULL);

  mat4x4->m[0] *= s->v[0];  mat4x4->m[1] *= s->v[1];  mat4x4->m[2] *= s->v[2];
  mat4x4->m[4] *= s->v[0];  mat4x4->m[5] *= s->v[1];  mat4x4->m[6] *= s->v[2];
  mat4x4->m[8] *= s->v[0];  mat4x4->m[9] *= s->v[1];  mat4x4->m[10] *= s->v[2];
  mat4x4->m[12] *= s->v[0]; mat4x4->m[13] *= s->v[1]; mat4x4->m[14] *= s->v[2];
}

/**
 * pgm_mat4x4_scale_from_scalars:
 * @mat4x4: A #PgmMat4x4 object.
 * @sx: The scale to apply on the x axis.
 * @sy: The scale to apply on the y axis.
 * @sz: The scale to apply on the z axis.
 *
 * Post multiplies @mat4x4 by a scaling matrix with the product replacing
 * @mat4x4.
 *
 * MT safe.
 */
void
pgm_mat4x4_scale_from_scalars (PgmMat4x4 *mat4x4,
                               gfloat sx,
                               gfloat sy,
                               gfloat sz)
{
  g_return_if_fail (mat4x4 != NULL);

  mat4x4->m[0] *= sx;  mat4x4->m[1] *= sy;  mat4x4->m[2] *= sz;
  mat4x4->m[4] *= sx;  mat4x4->m[5] *= sy;  mat4x4->m[6] *= sz;
  mat4x4->m[8] *= sx;  mat4x4->m[9] *= sy;  mat4x4->m[10] *= sz;
  mat4x4->m[12] *= sx; mat4x4->m[13] *= sy; mat4x4->m[14] *= sz;
}

/**
 * pgm_mat4x4_rotate_x:
 * @mat4x4: A #PgmMat4x4 object.
 * @angle: The angle of rotation, in radians.
 *
 * Post multiplies @mat4x4 by a rotation matrix with the product replacing
 * @mat4x4.
 *
 * MT safe.
 */
void
pgm_mat4x4_rotate_x (PgmMat4x4 *mat4x4,
                     gfloat angle)
{
  PgmMat4x4 rot, mult;
  gfloat s, c;

  g_return_if_fail (mat4x4 != NULL);

  PGM_SINCOS (angle, &s, &c);

  rot.m[0]=1.0f;  rot.m[1]=0.0f;  rot.m[2]=0.0f;  rot.m[3]=0.0f;
  rot.m[4]=0.0f;  rot.m[5]=c;     rot.m[6]=-s;    rot.m[7]=0.0f;
  rot.m[8]=0.0f;  rot.m[9]=s;     rot.m[10]=c;    rot.m[11]=0.0f;
  rot.m[12]=0.0f; rot.m[13]=0.0f; rot.m[14]=0.0f; rot.m[15]=1.0f;

  mat4x4_multiply_mat4x4_mat4x4 (&mult, mat4x4, &rot);
  pgm_mat4x4_set_from_mat4x4 (mat4x4, &mult);
}

/**
 * pgm_mat4x4_rotate_y:
 * @mat4x4: A #PgmMat4x4 object.
 * @angle: The angle of rotation, in radians.
 *
 * Post multiplies @mat4x4 by a rotation matrix with the product replacing
 * @mat4x4.
 *
 * MT safe.
 */
void
pgm_mat4x4_rotate_y (PgmMat4x4 *mat4x4,
                     gfloat angle)
{
  PgmMat4x4 rot, mult;
  gfloat s, c;

  g_return_if_fail (mat4x4 != NULL);

  PGM_SINCOS (angle, &s, &c);

  rot.m[0]=c;     rot.m[1]=0.0f;  rot.m[2]=s;     rot.m[3]=0.0f;
  rot.m[4]=0.0f;  rot.m[5]=1.0f;  rot.m[6]=0.0f;  rot.m[7]=0.0f;
  rot.m[8]=-s;    rot.m[9]=0.0f;  rot.m[10]=c;    rot.m[11]=0.0f;
  rot.m[12]=0.0f; rot.m[13]=0.0f; rot.m[14]=0.0f; rot.m[15]=1.0f;

  mat4x4_multiply_mat4x4_mat4x4 (&mult, mat4x4, &rot);
  pgm_mat4x4_set_from_mat4x4 (mat4x4, &mult);
}

/**
 * pgm_mat4x4_rotate_z:
 * @mat4x4: A #PgmMat4x4 object.
 * @angle: The angle of rotation, in radians.
 *
 * Post multiplies @mat4x4 by a rotation matrix with the product replacing
 * @mat4x4.
 *
 * MT safe.
 */
void
pgm_mat4x4_rotate_z (PgmMat4x4 *mat4x4,
                     gfloat angle)
{
  PgmMat4x4 rot, mult;
  gfloat s, c;

  g_return_if_fail (mat4x4 != NULL);

  PGM_SINCOS (angle, &s, &c);

  rot.m[0]=c;     rot.m[1]=-s;    rot.m[2]=0.0f;  rot.m[3]=0.0f;
  rot.m[4]=s;     rot.m[5]=c;     rot.m[6]=0.0f;  rot.m[7]=0.0f;
  rot.m[8]=0.0f;  rot.m[9]=0.0f;  rot.m[10]=1.0f; rot.m[11]=0.0f;
  rot.m[12]=0.0f; rot.m[13]=0.0f; rot.m[14]=0.0f; rot.m[15]=1.0f;

  mat4x4_multiply_mat4x4_mat4x4 (&mult, mat4x4, &rot);
  pgm_mat4x4_set_from_mat4x4 (mat4x4, &mult);
}

/**
 * pgm_mat4x4_rotate_axis_from_vec3:
 * @mat4x4: A #PgmMat4x4 object.
 * @angle: The angle of rotation, in radians.
 * @axis: A #PgmVec3 object representing the axis of rotation.
 *
 * Post multiplies @mat4x4 by a rotation matrix with the product replacing
 * @mat4x4.
 *
 * MT safe.
 */
void
pgm_mat4x4_rotate_axis_from_vec3 (PgmMat4x4 *mat4x4,
                                  gfloat angle,
                                  const PgmVec3 *axis)
{
  PgmMat4x4 rot, mult;
  gfloat s, c, a, ax, ay, az;

  g_return_if_fail (mat4x4 != NULL);
  g_return_if_fail (axis != NULL);

  PGM_SINCOS (angle, &s, &c);

  a = 1.0f - c;
  ax = a * axis->v[0];
  ay = a * axis->v[1];
  az = a * axis->v[2];

  rot.m[0] = ax * axis->v[0] + c;
  rot.m[1] = ax * axis->v[1] - axis->v[2] * s;
  rot.m[2] = ax * axis->v[2] + axis->v[1] * s;
  rot.m[3] = 0.0f;
  rot.m[4] = ay * axis->v[0] + axis->v[2] * s;
  rot.m[5] = ay * axis->v[1] + c;
  rot.m[6] = ay * axis->v[2] - axis->v[0] * s;
  rot.m[7] = 0.0f;
  rot.m[8] = az * axis->v[0] - axis->v[1] * s;
  rot.m[9] = az * axis->v[1] + axis->v[0] * s;
  rot.m[10] = az * axis->v[2] + c;
  rot.m[11] = 0.0f;
  rot.m[12] = 0.0f;
  rot.m[13] = 0.0f;
  rot.m[14] = 0.0f;
  rot.m[15] = 1.0f;

  mat4x4_multiply_mat4x4_mat4x4 (&mult, mat4x4, &rot);
  pgm_mat4x4_set_from_mat4x4 (mat4x4, &mult);
}

/**
 * pgm_mat4x4_rotate_axis_from_scalars:
 * @mat4x4: A #PgmMat4x4 object.
 * @angle: The angle of rotation, in radians.
 * @axis_x: The x component of the axis of rotation.
 * @axis_y: The y component of the axis of rotation.
 * @axis_z: The z component of the axis of rotation.
 *
 * Post multiplies @mat4x4 by a rotation matrix with the product replacing
 * @mat4x4.
 *
 * MT safe.
 */
void
pgm_mat4x4_rotate_axis_from_scalars (PgmMat4x4 *mat4x4,
                                     gfloat angle,
                                     gfloat axis_x,
                                     gfloat axis_y,
                                     gfloat axis_z)
{
  PgmMat4x4 rot, mult;
  gfloat s, c, a, ax, ay, az;

  g_return_if_fail (mat4x4 != NULL);

  PGM_SINCOS (angle, &s, &c);

  a = 1.0f - c;
  ax = a * axis_x;
  ay = a * axis_y;
  az = a * axis_z;

  rot.m[0] = ax * axis_x + c;
  rot.m[1] = ax * axis_y - axis_z * s;
  rot.m[2] = ax * axis_z + axis_y * s;
  rot.m[3] = 0.0f;
  rot.m[4] = ay * axis_x + axis_z * s;
  rot.m[5] = ay * axis_y + c;
  rot.m[6] = ay * axis_z - axis_x * s;
  rot.m[7] = 0.0f;
  rot.m[8] = az * axis_x - axis_y * s;
  rot.m[9] = az * axis_y + axis_x * s;
  rot.m[10] = az * axis_z + c;
  rot.m[11] = 0.0f;
  rot.m[12] = 0.0f;
  rot.m[13] = 0.0f;
  rot.m[14] = 0.0f;
  rot.m[15] = 1.0f;

  mat4x4_multiply_mat4x4_mat4x4 (&mult, mat4x4, &rot);
  pgm_mat4x4_set_from_mat4x4 (mat4x4, &mult);
}

/**
 * pgm_mat4x4_add_scalar:
 * @mat4x4: A #PgmMat4x4 object.
 * @s: A scalar.
 *
 * Computes the sum @mat4x4 + @s.
 *
 * MT safe.
 *
 * Returns: the sum @mat4x4 + @s in a newly allocated #PgmMat4x4.
 */
PgmMat4x4*
pgm_mat4x4_add_scalar (PgmMat4x4 *mat4x4,
                       gfloat s)
{
  PgmMat4x4 *sum;

  g_return_val_if_fail (mat4x4 != NULL, NULL);

  sum = pgm_mat4x4_new ();

  sum->m[0] = mat4x4->m[0] + s;   sum->m[1] = mat4x4->m[1] + s;
  sum->m[2] = mat4x4->m[2] + s;   sum->m[3] = mat4x4->m[3] + s;
  sum->m[4] = mat4x4->m[4] + s;   sum->m[5] = mat4x4->m[5] + s;
  sum->m[6] = mat4x4->m[6] + s;   sum->m[7] = mat4x4->m[7] + s;
  sum->m[8] = mat4x4->m[8] + s;   sum->m[9] = mat4x4->m[9] + s;
  sum->m[10] = mat4x4->m[10] + s; sum->m[11] = mat4x4->m[11] + s;
  sum->m[12] = mat4x4->m[12] + s; sum->m[13] = mat4x4->m[13] + s;
  sum->m[14] = mat4x4->m[14] + s; sum->m[15] = mat4x4->m[15] + s;

  return sum;
}

/**
 * pgm_mat4x4_add_mat4x4:
 * @mat4x4: A #PgmMat4x4 object.
 * @m: A #PgmMat4x4 object.
 *
 * Computes the sum @mat4x4 + @m.
 *
 * MT safe.
 *
 * Returns: the sum @mat4x4 + @m in a newly allocated #PgmMat4x4.
 */
PgmMat4x4*
pgm_mat4x4_add_mat4x4 (PgmMat4x4 *mat4x4,
                       const PgmMat4x4 *m)
{
  PgmMat4x4 *sum;

  g_return_val_if_fail (mat4x4 != NULL, NULL);
  g_return_val_if_fail (m != NULL, NULL);

  sum = pgm_mat4x4_new ();

  sum->m[0] = mat4x4->m[0] + m->m[0];    sum->m[1] = mat4x4->m[1] + m->m[1];
  sum->m[2] = mat4x4->m[2] + m->m[2];    sum->m[3] = mat4x4->m[3] + m->m[3];
  sum->m[4] = mat4x4->m[4] + m->m[4];    sum->m[5] = mat4x4->m[5] + m->m[5];
  sum->m[6] = mat4x4->m[6] + m->m[6];    sum->m[7] = mat4x4->m[7] + m->m[7];
  sum->m[8] = mat4x4->m[8] + m->m[8];    sum->m[9] = mat4x4->m[9] + m->m[9];
  sum->m[10] = mat4x4->m[10] + m->m[10]; sum->m[11] = mat4x4->m[11] + m->m[11];
  sum->m[12] = mat4x4->m[12] + m->m[12]; sum->m[13] = mat4x4->m[13] + m->m[13];
  sum->m[14] = mat4x4->m[14] + m->m[14]; sum->m[15] = mat4x4->m[15] + m->m[15];

  return sum;
}

/**
 * pgm_mat4x4_substract_scalar:
 * @mat4x4: A #PgmMat4x4 object.
 * @s: A scalar.
 *
 * Computes the substraction @mat4x4 - @s.
 *
 * MT safe.
 *
 * Returns: the substraction @mat4x4 - @s in a newly allocated #PgmMat4x4.
 */
PgmMat4x4*
pgm_mat4x4_substract_scalar (PgmMat4x4 *mat4x4,
                             gfloat s)
{
  PgmMat4x4 *sub;

  g_return_val_if_fail (mat4x4 != NULL, NULL);

  sub = pgm_mat4x4_new ();

  sub->m[0] = mat4x4->m[0] - s;   sub->m[1] = mat4x4->m[1] - s;
  sub->m[2] = mat4x4->m[2] - s;   sub->m[3] = mat4x4->m[3] - s;
  sub->m[4] = mat4x4->m[4] - s;   sub->m[5] = mat4x4->m[5] - s;
  sub->m[6] = mat4x4->m[6] - s;   sub->m[7] = mat4x4->m[7] - s;
  sub->m[8] = mat4x4->m[8] - s;   sub->m[9] = mat4x4->m[9] - s;
  sub->m[10] = mat4x4->m[10] - s; sub->m[11] = mat4x4->m[11] - s;
  sub->m[12] = mat4x4->m[12] - s; sub->m[13] = mat4x4->m[13] - s;
  sub->m[14] = mat4x4->m[14] - s; sub->m[15] = mat4x4->m[15] - s;

  return sub;
}

/**
 * pgm_mat4x4_substract_mat4x4:
 * @mat4x4: A #PgmMat4x4 object.
 * @m: A #PgmMat4x4 object.
 *
 * Computes the substraction @mat4x4 - @m.
 *
 * MT safe.
 *
 * Returns: the substraction @mat4x4 - @m in a newly allocated #PgmMat4x4.
 */
PgmMat4x4*
pgm_mat4x4_substract_mat4x4 (PgmMat4x4 *mat4x4,
                             const PgmMat4x4 *m)
{
  PgmMat4x4 *sub;

  g_return_val_if_fail (mat4x4 != NULL, NULL);
  g_return_val_if_fail (m != NULL, NULL);

  sub = pgm_mat4x4_new ();

  sub->m[0] = mat4x4->m[0] - m->m[0];    sub->m[1] = mat4x4->m[1] - m->m[1];
  sub->m[2] = mat4x4->m[2] - m->m[2];    sub->m[3] = mat4x4->m[3] - m->m[3];
  sub->m[4] = mat4x4->m[4] - m->m[4];    sub->m[5] = mat4x4->m[5] - m->m[5];
  sub->m[6] = mat4x4->m[6] - m->m[6];    sub->m[7] = mat4x4->m[7] - m->m[7];
  sub->m[8] = mat4x4->m[8] - m->m[8];    sub->m[9] = mat4x4->m[9] - m->m[9];
  sub->m[10] = mat4x4->m[10] - m->m[10]; sub->m[11] = mat4x4->m[11] - m->m[11];
  sub->m[12] = mat4x4->m[12] - m->m[12]; sub->m[13] = mat4x4->m[13] - m->m[13];
  sub->m[14] = mat4x4->m[14] - m->m[14]; sub->m[15] = mat4x4->m[15] - m->m[15];

  return sub;
}

/**
 * pgm_mat4x4_multiply_scalar:
 * @mat4x4: A #PgmMat4x4 object.
 * @s: A scalar.
 *
 * Computes the multiplication @mat4x4 . @s.
 *
 * MT safe.
 *
 * Returns: the multiplication @mat4x4 . @s in a newly allocated #PgmMat4x4.
 */
PgmMat4x4*
pgm_mat4x4_multiply_scalar (PgmMat4x4 *mat4x4,
                            gfloat s)
{
  PgmMat4x4 *mult;

  g_return_val_if_fail (mat4x4 != NULL, NULL);

  mult = pgm_mat4x4_new ();

  mult->m[0] = s * mat4x4->m[0];  mult->m[1] = s * mat4x4->m[1];
  mult->m[2] = s * mat4x4->m[2];  mult->m[3] = s * mat4x4->m[3];
  mult->m[4] = s * mat4x4->m[4];  mult->m[5] = s * mat4x4->m[5];
  mult->m[6] = s * mat4x4->m[6];  mult->m[7] = s * mat4x4->m[7];
  mult->m[8] = s * mat4x4->m[8];  mult->m[9] = s * mat4x4->m[9];
  mult->m[10] = s * mat4x4->m[10]; mult->m[11] = s * mat4x4->m[11];
  mult->m[12] = s * mat4x4->m[12]; mult->m[13] = s * mat4x4->m[13];
  mult->m[14] = s * mat4x4->m[14]; mult->m[15] = s * mat4x4->m[15];

  return mult;
}

/**
 * pgm_mat4x4_multiply_vec4:
 * @mat4x4: A #PgmMat4x4 object.
 * @v: A #PgmVec4 object.
 *
 * Computes the multiplication @mat4x4 . @v.
 *
 * MT safe.
 *
 * Returns: the multiplication @mat4x4 . @v in a newly allocated #PgmVec4.
 */
PgmVec4*
pgm_mat4x4_multiply_vec4 (PgmMat4x4 *mat4x4,
                          const PgmVec4 *v)
{
  PgmMat4x4 *m;
  PgmVec4 *mult;

  g_return_val_if_fail (mat4x4 != NULL, NULL);
  g_return_val_if_fail (v != NULL, NULL);

  mult = pgm_vec4_new ();
  m = mat4x4;

  mult->v[0] = m->m[0]*v->v[0] + m->m[1]*v->v[1]
    + m->m[2]*v->v[2] + m->m[3]*v->v[3];
  mult->v[1] = m->m[4]*v->v[0] + m->m[5]*v->v[1]
    + m->m[6]*v->v[2] + m->m[7]*v->v[3];
  mult->v[2] = m->m[8]*v->v[0] + m->m[9]*v->v[1]
    + m->m[10]*v->v[2] + m->m[11]*v->v[3];
  mult->v[3] = m->m[12]*v->v[0] + m->m[13]*v->v[1]
    + m->m[14]*v->v[2] + m->m[15]*v->v[3];

  return mult;
}

/**
 * pgm_mat4x4_multiply_mat4x4:
 * @mat4x4: A #PgmMat4x4 object.
 * @m: A #PgmMat4x4 object.
 *
 * Computes the multiplication @mat4x4 . @m. Note that this a
 * post-multiplication of @mat4x4 by @m.
 *
 * MT safe.
 *
 * Returns: the multiplication @mat4x4 . @m in a newly allocated #PgmMat4x4.
 */
PgmMat4x4*
pgm_mat4x4_multiply_mat4x4 (PgmMat4x4 *mat4x4,
                            const PgmMat4x4 *m)
{
  PgmMat4x4 *mult;

  g_return_val_if_fail (mat4x4 != NULL, NULL);
  g_return_val_if_fail (m != NULL, NULL);

  mult = pgm_mat4x4_new ();
  mat4x4_multiply_mat4x4_mat4x4 (mult, mat4x4, m);

  return mult;
}

/**
 * pgm_mat4x4_to_string:
 * @mat4x4: A #PgmMat4x4 object.
 *
 * Converts @mat4x4 to a string representation.
 *
 * MT safe.
 *
 * Returns: a newly allocated string representing @mat4x4.
 */
gchar*
pgm_mat4x4_to_string (PgmMat4x4 *mat4x4)
{
  gchar *string;

  g_return_val_if_fail (mat4x4 != NULL, NULL);

  string = g_strdup_printf ("[%f, %f, %f, %f]\n"
                            "|%f, %f, %f, %f|\n"
                            "|%f, %f, %f, %f|\n"
                            "[%f, %f, %f, %f]",
                            mat4x4->m[0], mat4x4->m[1], mat4x4->m[2],
                            mat4x4->m[3], mat4x4->m[4], mat4x4->m[5],
                            mat4x4->m[6], mat4x4->m[7], mat4x4->m[8],
                            mat4x4->m[9], mat4x4->m[10], mat4x4->m[11],
                            mat4x4->m[12], mat4x4->m[13], mat4x4->m[14],
                            mat4x4->m[15]);

  return string;
}

/* Utility functions */

/**
 * pgm_intersection_line_plane:
 * @l1: A #PgmVec3 representing a 1st point of the line.
 * @l2: A #PgmVec3 representing a 2nd point of the line.
 * @p: A #PgmVec3 representing point of the plane.
 * @pu: A #PgmVec3 representing a 1st vector colinear to the plane.
 * @pv: A #PgmVec3 representing a 2nd vector colinear to the plane.
 *
 * Retrieves the intersection point of the line defined by @l1 and @l2 with
 * the plane defined by @p, @pu and @pv.
 *
 * @pu and @pv must not be colinear. If the line belongs to the plane then
 * it's considered as if there is no intersection.
 *
 * MT safe.
 *
 * Returns: the intersection point in a newly allocated #PgmVec3 if any, #NULL
 * otherwise.
 */
PgmVec3*
pgm_intersection_line_plane (const PgmVec3 *l1,
                             const PgmVec3 *l2,
                             const PgmVec3 *p,
                             const PgmVec3 *pu,
                             const PgmVec3 *pv)
{
  gfloat normal[3], ma[3], mn[3];
  gfloat tmp;

  g_return_val_if_fail (l1 != NULL, NULL);
  g_return_val_if_fail (l2 != NULL, NULL);
  g_return_val_if_fail (p != NULL, NULL);
  g_return_val_if_fail (pu != NULL, NULL);
  g_return_val_if_fail (pv != NULL, NULL);

  /* The vec3 methods are not used to avoid too many memory allocations. This
   * function being used a lot for the mouse picking, speed is the priority. */

  normal[0] = pu->v[1] * pv->v[2] - pu->v[2] * pv->v[1];
  normal[1] = pu->v[2] * pv->v[0] - pu->v[0] * pv->v[2];
  normal[2] = pu->v[0] * pv->v[1] - pu->v[1] * pv->v[0];

  tmp = (gfloat) sqrt (normal[0]*normal[0] +
                       normal[1]*normal[1] +
                       normal[2]*normal[2]);
  if (tmp == 0.0f)
    return NULL;

  tmp = 1.0f / tmp;
  normal[0] *= tmp;
  normal[1] *= tmp;
  normal[2] *= tmp;

  mn[0] = l2->v[0] - l1->v[0];
  mn[1] = l2->v[1] - l1->v[1];
  mn[2] = l2->v[2] - l1->v[2];

  tmp = normal[0]*mn[0] + normal[1]*mn[1] + normal[2]*mn[2];
  if (tmp == 0.0f)
    return NULL;

  ma[0] = p->v[0] - l1->v[0];
  ma[1] = p->v[1] - l1->v[1];
  ma[2] = p->v[2] - l1->v[2];

  tmp = (normal[0]*ma[0] + normal[1]*ma[1] + normal[2]*ma[2]) / tmp;

  return pgm_vec3_new_from_scalars (l1->v[0] + mn[0] * tmp,
                                    l1->v[1] + mn[1] * tmp,
                                    l1->v[2] + mn[2] * tmp);
}

/**
 * pgm_point_belongs_rectangle:
 * @p: A #PgmVec3 representing the point.
 * @r: A #PgmVec3 representing the top-left corner position of the rectangle.
 * @ru: A #PgmVec3 representing the top edge of the rectangle.
 * @rv: a 3 components vector defining the left edge of the rectangle.
 *
 * Retrieves if the point @p belongs to the rectangle defined by the point
 * @r and the vectors @ru and @rv.
 *
 * MT safe.
 *
 * Returns: #TRUE if @p belongs to the rectangle, #FALSE otherwise.
 */
gboolean
pgm_point_belongs_rectangle (const PgmVec3 *p,
                             const PgmVec3 *r,
                             const PgmVec3 *ru,
                             const PgmVec3 *rv)
{
  PgmVec3 am;
  gfloat a, b;

  g_return_val_if_fail (p != NULL, FALSE);
  g_return_val_if_fail (r != NULL, FALSE);
  g_return_val_if_fail (ru != NULL, FALSE);
  g_return_val_if_fail (rv != NULL, FALSE);

  pgm_vec3_set_from_scalars (&am,
                             p->v[0] - r->v[0],
                             p->v[1] - r->v[1],
                             p->v[2] - r->v[2]);

  a = ru->v[0]*am.v[0] + ru->v[1]*am.v[1] + ru->v[2]*am.v[2];
  if (a < 0)
    return FALSE;

  b = rv->v[0]*am.v[0] + rv->v[1]*am.v[1] + rv->v[2]*am.v[2];
  if (b < 0)
    return FALSE;

  if (a > ru->v[0]*ru->v[0] + ru->v[1]*ru->v[1] + ru->v[2]*ru->v[2])
    return FALSE;

  if (b > rv->v[0]*rv->v[0] + rv->v[1]*rv->v[1] + rv->v[2]*rv->v[2])
    return FALSE;

  return TRUE;
}
