/*	String_Vector_Comparator

PIRL CVS ID: String_Vector_Comparator.java,v 1.9 2012/04/16 06:04:11 castalia Exp

Copyright (C) 2003-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package PIRL.Conductor;

import java.util.Comparator;
import java.util.List;

/**	A <i>String_Vector_Comparator</i> implements the Comparator
	interface to sort Lists of Objects.
<p>
	It is particularly useful, for example, in sorting a set of results
	from a Database query. Use an instance of this class as the
	Comparator in the Collections.sort(List, Comparator) static method,
	which actually does the sorting. The Objects to be compared are
	assumed to be Lists. The Objects at the specified index of each
	List are converted to Strings. If these Strings can be parsed as
	Doubles then these are compared numerically. Otherwise the Strings
	are are compared lexicographically.
<p>
	@author Christian Schaller, Bradford Castalia - UA/PIRL
*/
public class String_Vector_Comparator
	implements Comparator
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Conductor.String_Vector_Comparator (1.9 2012/04/16 06:04:11)";


// Vector index to be compared.
private int
	Index;

private static final String
	NL = System.getProperty ("line.separator");

/*==============================================================================
	Constructors
*/
/**	Construct a String_Vector_Comparator using a sepecified index for the
	Lists to be {@link #compare(Object, Object) compared}.
<p>
	@param	index	The List index to be compared.
*/
public String_Vector_Comparator
	(
	int		index
	)
{Index = index;}

/**	Construct a String_Vector_Comparator that will {@link
	#compare(Object, Object) compare} index zero of the Lists.
*/
public String_Vector_Comparator ()
{this (0);}

/*==============================================================================
	Accessors
*/
/**	Set the List index to be {@link #compare(Object, Object) compared}.
<p>
	@param	index	The List index to be compared.
	@return	This String_Vector_Comparator.
	@throws	IllegalArgumentException	If the index is negative.
*/
public String_Vector_Comparator Index
	(
	int		index
	)
{
if (index < 0)
	throw new IllegalArgumentException (ID + NL
		+ "A negative (" + index + ") index is invalid.");
Index = index;
return this;
}

/**	Get the List index to be {@link #compare(Object, Object) compared}.
<p>
	@return	The List index to be compared.
	@see	#Index(int)
*/
public int Index ()
{return Index;}

/*==============================================================================
	Methods
*/
/**	Compare two Lists.
<p>
	The objects at each List's {@link #Index() Index} entry are first
	checked for being null. If both are null, 0 is returned; if only
	the first is null, -1 is returned; if only the second is null, 1
	is returned.
<p>
	Non-null list entries are converted to a Strings. If these Strings
	can be converted to Doubles, then they are compared numerically using
	Double.compareTo. Otherwise they are compared lexicographically using
	String compareTo.
<p>
	@param	list_0	The first List for comparison.
	@param	list_1	The second List for comparison.
	@return	An integer less than, equal to, or greater than zero as the
		first List {@link #Index() Index} entry is less than, equal to,
		or greater than the second List {@link #Index() Index} entry.
	@throws	IllegalArgumentException	If the arguments are not Lists.
	@throws	ArrayIndexOutOfBoundsException	If the arguments do not
		have at least {@link #Index() Index} + 1 entries.
*/
public int compare
	(
	Object	list_0,
	Object	list_1
	)
{
if (! (list_0 instanceof List) ||
	! (list_1 instanceof List))
	throw new IllegalArgumentException (ID + NL
		+ "The objects to compare must be Lists.");
if (((List)list_0).size () <= Index ||
	((List)list_1).size () <= Index)
	throw new ArrayIndexOutOfBoundsException (ID + NL
		+ "Both List objects to compare must contain at least "
			+ (Index + 1) + " entr"
			+ (((Index + 1) == 1) ? "y" : "ies") + '.');
Object
	object_0 = ((List)list_0).get (Index),
	object_1 = ((List)list_1).get (Index);
if (object_0 == null ||
	object_1 == null)
	{
	if (object_0 == null &&
		object_1 == null)
		return 0;
	if (object_0 == null)
		return -1;
	if (object_1 == null)
		return  1;
	}
String
	string_0 = object_0.toString (),
	string_1 = object_1.toString ();
try
	{
	//	Try numerical comparison.
	Double
		double_0 = new Double (string_0),
		double_1 = new Double (string_1);
	return double_0.compareTo (double_1);
	}
catch (NumberFormatException exception)
	{
	//	Compare lexicographically.
	return string_0.compareTo (string_1);
	}
}

}

