/*	Fields_Map

PIRL CVS ID: Fields_Map.java,v 1.13 2012/04/16 06:08:57 castalia Exp

Copyright (C) 2005-2007  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package	PIRL.Database;

import	java.util.List;
import	java.util.Vector;
import	java.util.Iterator;
import	java.util.Arrays;


/**	A <I>Fields_Map</I> maps a list of field names in user specified
	order to the actual order of field names as returned from a database
	table query.
<p>
	The Fields_Map relates two possible orderings of a list of field
	names: The <i>actual</i> order is the order of entries as they occur
	before being mapped. This is expected to be the order in which table
	record field entries are returned from a database query. The
	<i>user</i> order is the the order in which the user of table records
	expects the entries to be in after being mapped. The user order might
	be thought of as the logical order known to the user of the map as
	typically set by by pre-determined constant index values associated
	with known record field names.
<p>
	The map is constructed by providing a list of field names in user
	order and a list of field names in actual order. Every user field
	name that is found in the actual field names list results in a
	mapping of that name from its index in the user order to its index in
	the actual order. User names that are not found in the list of actual
	field names are mapped to the no-index value. Field names are
	{@link Database#Matches(String, String, boolean) matched} using
	either case sensitive or case insensitive comparison.
<p>
	The map may be used to access a field entry from a record in
	actual order by specifying a field name or a user index.
<p>
	@author		Bradford Castalia, Drew Castalia - idaeim studio
	@version	1.13
*/
public class Fields_Map
{
public static final String
	ID = "PIRL.Database.Fields_Map (1.13 2012/04/16 06:08:57)";

/**	The default for case sensitive field name matches.
<p>
	The initial value is false;
*/
public static boolean
	Case_Sensitive_Default		= false;

/**	Whether to use case sensitive field name matches.
<p>
	<b.N.B.</b>: The case sensistive matching condition is set when the
	Field_Map is constructed and used by the constructor to assemble the
	map of user field name indexes to actual field name indexes. This
	condition is also used for {@link #index(String) user field name to
	actual field index} and {@link #entry(List, String) user field name
	to table record entry} mapping. Changing the condition after the
	Field_Map has been constructed may produce inconsistent mapping
	results.
*/
public boolean
	Case_Sensitive;

//	Original field names as provided to the constructor.
private String
	User_Fields[]				= {},
	Actual_Fields[]				= {};

//	Array of actual field index values in user index order.
private int
	User_to_Actual_Index_Map[]	= {};

//	User_to_Actual_Index_Map sorted with only non-negative unique values.
private int
	Actual_Ordered_Index_Map[]	= null;

private static final String
	NL = Database.NL;

	
// Debug control.
private static final int
	DEBUG_OFF					= 0,
	DEBUG_SETUP					= 1 << 0,
	DEBUG_CONSTRUCTOR			= 1 << 1,
	DEBUG_ACCESSORS				= 1 << 2,
	DEBUG_UTILITY				= 1 << 3,
	DEBUG_ALL					= -1,

	DEBUG						= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Constructs a Fields_Map that maps a List of field names in user
	index order to their index in a List of field names in actual
	(table record) order.
<p>
	User field names that are not found in the List of actual names map
	to a -1 index to indicate missing fields.
<p>
	User field names are compared to actual field names using the {@link
	#Case_Sensitive_Default default case sensitivity}.
<p>
	<b>N.B.</b>: The first actual field name that matches each user
	field name provides the mapping. If the actual field names list
	contains duplicate entries, only the first entry will be mapped.
	Likewise, if case insensitive matching is being done only the first
	actual name that matches a user name regardless of case is mapped.
<p>
	<b>N.B.</b>: A field names list may contain null entries. A null name
	always matches a null name. A non-null name never matches a null
	name.
<p>
	@param	user_fields	A List of user field name Strings to be mapped.
	@param	actual_fields	A List of field name Strings in actual
		(record) order; as returned in the first record of a {@link
		Database#Select(String, String) Database.Select} operation.
	@param	case_sensitive	If true, user field names are compared to
		actual field names with a case sensitive match; otherwise a
		case insensitive match is used.
	@throws	IllegalArgumentException	If either argument is null.
*/
public Fields_Map
	(
	List<String>	user_fields,
	List<String>	actual_fields,
	boolean	case_sensitive
	)
{
/*	Programmer's note:

	The effective constructor uses List arguments rather than array arguments
	because the Arrays.asList method used by constructors on arrays does not
	copy the orginal array, whereas the Collection.toArray method copies the
	elements of the Collection into the array. Since the working constructor
	is always going to make a local copy the field lists, using a List backed
	by a user specified array avoids unnecessary additional copying.
*/
if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		(">>> Fields_Map:" + NL
		+"       user_fields - " + user_fields + NL
		+"     actual_fields - " + actual_fields + NL
		+"    case_sensitive - " + case_sensitive);
Case_Sensitive = case_sensitive;

//	Copy the fields lists.
if (user_fields != null)
	user_fields.toArray (User_Fields = new String[user_fields.size ()]);
if (actual_fields != null)
	actual_fields.toArray (Actual_Fields = new String[actual_fields.size ()]);
if (user_fields == null ||
	actual_fields == null)
	throw new IllegalArgumentException (ID + NL
		+ "Can't construct a Fields_Map with null arguments.");

//	Assemble the user-to-actual indexes map.
int
	index = User_Fields.length;
User_to_Actual_Index_Map = new int[index];
while (index-- > 0)
	{
	User_to_Actual_Index_Map[index] =
		Database.Index
			(Actual_Fields, User_Fields[index], Case_Sensitive);
	if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
		System.out.println
			("    " + index + ':' + User_Fields[index]
				+ " -> " + User_to_Actual_Index_Map[index]
				+ ((User_to_Actual_Index_Map[index] < 0) ?
					"" :
					(":" + Actual_Fields[User_to_Actual_Index_Map[index]])));
	}
if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		("<<< Fields_Map");
}

/**	Constructs a Fields_Map that maps a List of field names in user
	index order to their index in a List of field names in actual
	(table record) order.
<p>
	User field names that are not found in the List of actual names map
	to a -1 index to indicate missing fields.
<p>
	User field names are compared to actual field names using the {@link
	#Case_Sensitive_Default default case sensitivity}.
<p>
	<b>N.B.</b>: The first actual field name that matches each user
	field name provides the mapping. If the actual field names list
	contains duplicate entries, only the first entry will be mapped.
	Likewise, if case insensitive matching is being done only the first
	actual name that matches a user name regardless of case is mapped.
<p>
	<b>N.B.</b>: A field names list may contain null entries. A null name
	always matches a null name. A non-null name never matches a null
	name.
<p>
	@param	user_fields	A List of user field name Strings to be mapped.
	@param	actual_fields	A List of field name Strings in actual
		(record) order; as returned in the first record of a {@link
		Database#Select(String, String) Database.Select} operation.
	@throws	IllegalArgumentException	If either argument is null.
*/
public Fields_Map
	(
	List<String>	user_fields,
	List<String>	actual_fields
	)
{this (user_fields, actual_fields, Case_Sensitive_Default);}

/**	Constructs a Fields_Map that maps an array of field names in user
	index order to their index in an array of field names in actual
	(table record) order.
<p>
	User field names that are not found in the array of actual names map
	to a -1 index to indicate missing fields.
<p>
	<b>N.B.</b>: The first actual field name that matches each user
	field name provides the mapping. If the actual field names list
	contains duplicate entries, only the first entry will be mapped.
	Likewise, if case insensitive matching is being done only the first
	actual name that matches a user name regardless of case is mapped.
<p>
	<b>N.B.</b>: A field names list may contain null entries. A null name
	always matches a null name. A non-null name never matches a null
	name.
<p>
	@param	user_fields	An array of user field name Strings to be mapped.
	@param	actual_fields	An array of field name Strings in actual
		(tablae record) order; as returned in the first record of a {@link
		Database#Select(String, String) Database.Select} operation.
	@param	case_sensitive	If true, user field names are compared to
		actual field names with a case sensitive match; otherwise a
		case insensitive match is used.
	@throws	IllegalArgumentException	If either argument is null.
*/
public Fields_Map
	(
	String[]	user_fields,
	String[]	actual_fields,
	boolean		case_sensitive
	)
{
this
	((user_fields == null) ? (List<String>)null : Arrays.asList (user_fields),
	(actual_fields == null) ? (List<String>)null : Arrays.asList (actual_fields),
	case_sensitive);
}

/**	Constructs a Fields_Map that maps an array of field names in user
	index order to their index in an array of field names in actual
	(table record) order.
<p>
	User field names that are not found in the array of actual names map
	to a -1 index to indicate missing fields.
<p>
	User field names are compared to actual field names using the {@link
	#Case_Sensitive_Default default case sensitivity}.
<p>
	<b>N.B.</b>: The first actual field name that matches each user
	field name provides the mapping. If the actual field names list
	contains duplicate entries, only the first entry will be mapped.
	Likewise, if case insensitive matching is being done only the first
	actual name that matches a user name regardless of case is mapped.
<p>
	<b>N.B.</b>: A field names list may contain null entries. A null name
	always matches a null name. A non-null name never matches a null
	name.
<p>
	@param	user_fields	An array of user field name Strings to be mapped.
	@param	actual_fields	An array of field name Strings in actual
		(table record) order; as returned in the first record of a {@link
		Database#Select(String, String) Database.Select} operation.
	@throws	IllegalArgumentException	If either argument is null.
*/
public Fields_Map
	(
	String[]	user_fields,
	String[]	actual_fields
	)
{
this
	((user_fields == null) ? (List<String>)null : Arrays.asList (user_fields),
	(actual_fields == null) ? (List<String>)null : Arrays.asList (actual_fields),
	Case_Sensitive_Default);
}

/**	Constructs a Fields_Map that maps an array of field names in user
	index order to their index in a List of field names in actual
	(table record) order.
<p>
	User field names that are not found in the List of actual names map
	to a -1 index to indicate missing fields.
<p>
	User field names are compared to actual field names using the {@link
	#Case_Sensitive_Default default case sensitivity}.
<p>
	<b>N.B.</b>: The first actual field name that matches each user
	field name provides the mapping. If the actual field names list
	contains duplicate entries, only the first entry will be mapped.
	Likewise, if case insensitive matching is being done only the first
	actual name that matches a user name regardless of case is mapped.
<p>
	<b>N.B.</b>: A field names list may contain null entries. A null name
	always matches a null name. A non-null name never matches a null
	name.
<p>
	@param	user_fields	An array of user field name Strings to be mapped.
	@param	actual_fields	A List of field name Strings in actual
		(record) order; as returned in the first record of a {@link
		Database#Select(String, String) Database.Select} operation.
	@throws	IllegalArgumentException	If either argument is null.
*/
public Fields_Map
	(
	String[]		user_fields,
	List<String>	actual_fields
	)
{
this
	((user_fields == null) ? (List<String>)null : Arrays.asList (user_fields),
	actual_fields,
	Case_Sensitive_Default);
}

/**	Constructs a Fields_Map that maps a List of field names in user
	index order to their index in an array of field names in actual
	(table record) order.
<p>
	User field names that are not found in the array of actual names map
	to a -1 index to indicate missing fields.
<p>
	User field names are compared to actual field names using the {@link
	#Case_Sensitive_Default default case sensitivity}.
<p>
	<b>N.B.</b>: The first actual field name that matches each user
	field name provides the mapping. If the actual field names list
	contains duplicate entries, only the first entry will be mapped.
	Likewise, if case insensitive matching is being done only the first
	actual name that matches a user name regardless of case is mapped.
<p>
	<b>N.B.</b>: A field names list may contain null entries. A null name
	always matches a null name. A non-null name never matches a null
	name.
<p>
	@param	user_fields	A List of user field name Strings to be mapped.
	@param	actual_fields	An array of field name Strings in actual
		(record) order; as returned in the first record of a {@link
		Database#Select(String, String) Database.Select} operation.
	@throws	IllegalArgumentException	If either argument is null.
*/
public Fields_Map
	(
	List<String>	user_fields,
	String[]		actual_fields
	)
{
this
	(user_fields,
	(actual_fields == null) ? (List<String>)null : Arrays.asList (actual_fields),
	Case_Sensitive_Default);
}

/*==============================================================================
	Accessors
*/
/**	Maps a user field number to an actual record field number.
<p>
	@param	user_field_number	The user field number to map.
	@return	The actual record field number. This will be -1 if the user
		field_number is not within the range of the number of field
		names provided when the Fields_Map was constructed, or it maps
		to a field that was not present when the map was constructed.
*/
public int index
	(
	int		user_field_number
	)
{
if (user_field_number >= 0 &&
	user_field_number < User_to_Actual_Index_Map.length)
	return User_to_Actual_Index_Map[user_field_number];
return -1;
}

/**	Maps a user field name to an actual field number.
<p>
	The specified field name is compared to the list of user field
	names with the {@link #Case_Sensitive case sensitivity} of the
	Field_Map object.
<p>
	@param	user_field_name	The user field name to map.
	@return	The actual field number. This will be -1 if the field_name
		was not matched in the list of user field names provided when the
		Fields_Map was constructed, or it was not mapped to an actual
		field name.
*/
public int index
	(
	String	user_field_name
	)
{
return
	index (Database.Index (User_Fields, user_field_name, Case_Sensitive));
}

/**	Gets a field entry from a record corresponding to a user field number.
<p>
	The field number is mapped to is actual {@link #index(int) index}. If
	this results in a valid record index (non-negative less than the size
	of the record) the corresponding entry from the record is returned as
	a String.
<p>
	@param	record	The record List containing field value Strings.
		If null, null is returned.
	@param	field_number	The user field number before mapping
		to its actual field index.
	@return	The record List String entry for the mapped user field_number.
		This will be null if the user field number does not map to a
		valid record entry.
*/
public String entry
	(
	List<String>	record,
	int				field_number
	)
{
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.print
		(">>> Fields_Map.entry:" + NL
		+"    record - " + record + NL
		+"    field_number - " + field_number);
String
	field_value = null;
if (record != null)
	{
	field_number = index (field_number);
	if (field_number >= 0 && 
		field_number < record.size ())
		field_value = record.get (field_number);
	}
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.print
		("<<< Fields_Map.entry: " + field_value);
return field_value;
}

/**	Gets a field entry from a record corresponding to a user field name.
<p>
	The field name is mapped to its actual {@link #index(String) index}.
	If this results in a valid record index (non-negative less than the
	size of the record) the corresponding entry from the record is
	returned as a String.
<p>
	@param	record	The record List containing field value Strings.
		If null, null is returned.
	@param	field_name	The user field name of the field for which to get
		a record entry.
	@return	The record List String entry for the mapped field_name.
		This will be null if the field name does not map to a valid
		record entry.
*/
public String entry
	(
	List<String>	record,
	String			field_name
	)
{
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.print
		(">>> Fields_Map.entry:" + NL
		+"    record - " + record + NL
		+"    field_name - " + field_name);
String
	field_value = null;
if (record != null)
	{
	int
		field_number = index (field_name);
	if (field_number >= 0 &&
		field_number < record.size ())
		field_value = (String)record.get (field_number);
	}
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.print
		("<<< Fields_Map.entry: " + field_value);
return field_value;
}

/**	Gets a Vector of field names in user index order.
<p>
	Only field names that were mapped when the Fields_Map was constructed
	will be included in the Vector that is returned.
<p>
	@return	A Vector of field names in user index order.
*/
public Vector<String> User_Fields ()
{
Vector<String>
	user_fields = new Vector<String> ();
for (int
		index = 0;
		index < User_to_Actual_Index_Map.length;
		index++)
	if (User_to_Actual_Index_Map[index] >= 0)
		user_fields.add (User_Fields[index]);
return user_fields;
}

/**	Gets a Vector of all the user field names.
<p>
	All the original user field names, as specified in the constructor,
	are included in the Vector that is returned.
<p>
	@return	A Vector that is a copy of the original user field names
		specified in the constructor. This will be empty if the
		constructor was given a null user fields list.
*/
public Vector<String> All_User_Fields ()
{return new Vector<String> (Arrays.asList (User_Fields));}

/**	Gets a Vector of field names in actual index order.
<p>
	Only actual field names that were mapped from user field names when
	the Fields_Map was constructed will be included in the Vector that is
	returned. Thus the returned Vector may be smaller than the number of
	entries in the actual fields name list provided to the constructor.
<p>
	@return	A Vector of field names in actual index order.
*/
public Vector<String> Actual_Fields ()
{
int
	index;
if (Actual_Ordered_Index_Map == null)
	{
	//	Copy the index map.
	index = User_to_Actual_Index_Map.length;
	int
		array[] = new int[index];
	while (index-- > 0)
		array[index] = User_to_Actual_Index_Map[index];

	//	Sort the copied index map into ascending order.
	Arrays.sort (array);

	//	Count the number of non-negative, unique values.
	int
		count = 0,
		previous = -1;	//	Initial previous value is non-mapped.
	for (index = 0;
		 index < array.length;
		 index++)
		{
		if (array[index] != previous)
			{
			//	Count the non-negative, unique value.
			++count;
			previous = array[index];
			}
		else
			//	Duplicate or non-mapped to be excluded.
			array[index] = -1;
		}
	if (count == array.length)
		//	No excluded values.
		Actual_Ordered_Index_Map = array;
	else
		{
		//	Remove excluded values.
		Actual_Ordered_Index_Map = new int[count];
		for (count = 0,
			 index = 0;
			 index < array.length;
			 index++)
			if (array[index] >= 0)
				Actual_Ordered_Index_Map[count++] = array[index];
		}
	}

Vector<String>
	actual_fields = new Vector<String> (Actual_Ordered_Index_Map.length);
for (index = 0;
	 index < Actual_Ordered_Index_Map.length;
	 index++)
	actual_fields.add (Actual_Fields[Actual_Ordered_Index_Map[index]]);
return actual_fields;
}

/**	Gets a Vector of actual field names in unmapped order.
<p>
	All the original actual field names, as specified in the constructor,
	are included in the Vector that is returned.
<p>
	@return	A Vector that is a copy of the original actual field names
		specified in the constructor. This will be an empty if the
		constructor had a null user or actual fields list.
*/
public Vector<String> All_Actual_Fields ()
{return new Vector<String> (Arrays.asList (Actual_Fields));}

/**	Gets a Vector of actual field names in unmapped order.
<p>
	This is a deprecated name for the {@link #All_Actual_Fields()} method.
<p>
	@return	A Vector that is a copy of the original actual field names
		specified in the constructor. This will be an empty if the
		constructor had a null useer or actual fields list.
	@deprecated
*/
public Vector<String> Table_Fields ()
{return All_Actual_Fields ();}


/**	Gets a Vector of unmapped actual field names.
<p>
	@return	A Vector containing the actual field names are not mapped
		to user field names.
*/
public Vector<String> Unmapped_Actual_Fields ()
{
if (Actual_Ordered_Index_Map == null)
	Actual_Fields ();

Vector<String>
	fields = All_Actual_Fields ();
int
	index = Actual_Ordered_Index_Map.length;
while (--index >= 0)
	fields.remove (Actual_Ordered_Index_Map[index]);
return fields;
}

/**	Remaps a record from actual table record order to user fields order.
<p>
	A new record is constructed from {@link #entry(List, int) mapped
	entries} of the specified record for all user fields. Any user field
	that is not mapped to an actual field results in a null record entry.
	Entries of the source record that are not mapped will not be included
	in the returned record; the returned record will be the same size as
	the user field names list that was used to construct the Fields_Map.
<p>
	@param	record	A Vector of field values in actual order. If null,
		null is returned.
	@return	A Vector of mapped record field values in user order.
	@throws	IllegalArgumentException	If the source record has fewer
		entries than the number of user fields.
*/
public Vector<String> Actual_to_User
	(
	Vector<String>	record
	)
{
Vector<String>
	mapped = null;
if (record != null)
	{
	if (record.size () < User_to_Actual_Index_Map.length)
		throw new IllegalArgumentException (ID + NL
			+ "The record to map from actual to user order only has "
				+ record.size () + " of the required "
				+ User_to_Actual_Index_Map.length + " fields.");
	mapped = new Vector<String> (User_to_Actual_Index_Map.length);
	for (int
			index = 0;
			index < User_to_Actual_Index_Map.length;
			index++)
		mapped.add (entry (record, index));
	}
return mapped;
}

/**	Gets the description of this Fields_Map.
<p>
	@return	A String describing this Fields_Map.
*/
public String toString ()
{
String
	description = ID + NL
		+ "    User <- Actual";
int
	actual_index,
	user_index = -1;
while (++user_index < User_Fields.length)
	{
	actual_index = index (User_Fields[user_index]);
	description += NL
		+ String.format ("%8d <- %2d: %s",
			user_index, actual_index, User_Fields[user_index]);
	}

Vector<String>
	unmapped_fields = Unmapped_Actual_Fields ();
if (unmapped_fields.size () > 0)
	{
	description += NL
		+ "    Unmapped -";
	String
		name;
	int
		index = -1,
		size = unmapped_fields.size ();
	while (++index < size)
		{
		name = unmapped_fields.get (index);
		actual_index = -1;
		while (++actual_index < Actual_Fields.length)
			{
			if ((name == null &&
				Actual_Fields[actual_index] == null) ||
				(name != null &&
				name.equals (Actual_Fields[actual_index])))
				{
				description += NL
					+ String.format ("%14d: %s",
						actual_index, Actual_Fields[actual_index]);
				break;
				}
			}
		}
	}
return description;
}

/*==============================================================================
	Utility
*/
/**	Confirms that a list of field names are all mapped to valid record
	locations.
<p>
	Each name in the list is treated as a user field name that is
	required to be mapped to an actual field name. Thus the failure
	of any name in the list to be mapped will throw an exception.
	Nevertheless, all unmapped field names in the list are identified.
<p>
	@param	field_names	An array of user name Strings to be checked. null
		entries are ignored.
	@throws	Database_Exception	if any names fail to map to valid
		record locations. The exception message will list all missing
		field names.
*/
public void Confirm_Fields
	(
	String[]	field_names
	)
	throws Database_Exception
{Confirm_Fields
	((field_names == null) ? (List<String>)null : Arrays.asList (field_names));}

/**	Confirms that a list of user field names are all mapped to valid
	actual fields.
<p>
	Each name in the list is treated as a user field name that is
	required to be mapped to an actual field name. Thus the failure
	of any name in the list to be mapped will throw an exception.
	Nevertheless, all unmapped field names in the list are identified.
<p>
	@param	field_names	The list of names to be checked. If null, nothing
		is done. Any null entries in the list are ignored.
	@throws	Database_Exception	if any of the field names fail to map to
		valid actual locations. The exception message will list all
		unmapped field names.
*/
public void Confirm_Fields
	(
	List<String>	field_names
	)
	throws Database_Exception
{
if ((DEBUG & DEBUG_UTILITY) != 0)
	System.out.println
		(">>> Fields_Map.Confirm_Fields");
if (field_names == null)
	return;
String
	missing_fields = "";
Iterator
	names = field_names.iterator ();
while (names.hasNext ())
	{
	String
		name = (String)names.next ();
	if (name == null)
		continue;
	if ((DEBUG & DEBUG_UTILITY) != 0)
		System.out.println
			("    " + name + " -> " + index (name));
	if (index (name) < 0)
		missing_fields +=
			((missing_fields.length () == 0) ?
				"Missing required fields: " : ", ")
			+ name;
	}
if (missing_fields.length () != 0)
	throw new Database_Exception (ID + NL
		+ missing_fields);
if ((DEBUG & DEBUG_UTILITY) != 0)
	System.out.println
		("<<< Fields_Map.Confirm_Fields");
}

}	//	End of Fields_Map class.
