/*
 * auth.c  Authentication for socket port
 *
 * 1st Version:  @(#)auth.c  1.00  19-Jan-2001 EAS
 *
 */

#include <stdlib.h>
#include <termios.h>
#include <time.h>
#include <signal.h>
#include <ctype.h>
#include <errno.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <grp.h>
#include <netdb.h>
#include <netinet/in.h>
#include <pwd.h>
#include <string.h>
#include <sys/resource.h>
#include <sys/stat.h>
#include <unistd.h>
#include <utmp.h>
#include <crypt.h>

#ifndef NO_CHAP
#include "chap.h"
#ifdef CHAPMS
#include "chap_ms.h"
#endif
#endif

#include "server.h"

#ifdef PORTSLAVE_TACACS
#include "tacc/libtac.h"
#endif

#ifdef HAS_SHADOW
#include <shadow.h>
#endif

#define CRYPT_EP_SIZE 35 /* Maximum encrypted password size (34 bytes for MD5)*/

static char *pw_encrypt(const char *clear, const char *salt)
{
  static char cipher[CRYPT_EP_SIZE];
  char *cp;

  cp = (char *) crypt(clear, salt);
  /* if crypt (a nonstandard crypt) returns a string too large,
     truncate it so we don't overrun buffers and hope there is
     enough security in what's left */
  strncpy(cipher, cp, CRYPT_EP_SIZE);
  cipher[CRYPT_EP_SIZE - 1] = 0;
  return cipher;
}

/* login_local can't be static: called by other modules */
int login_local(struct auth *pai)
{
  struct passwd *pwd;
  char *password;
  int  rtn;

#ifdef HAS_SHADOW
  struct spwd *spwd = NULL;
#endif

  pwd = getpwnam(pai->login);
  if(!pwd)
  {
    /* No username info */
    if(pai->message[0])
      free(pai->message[0]);
    pai->message[0] = xstrdup("Invalid Login.\n");  
    pai->msn = 1;
    return(1);
  }

#ifdef HAS_SHADOW
  if( !strcmp(pwd->pw_passwd, "x") || !strcmp(pwd->pw_passwd, "*") )
  {
    spwd = getspnam(pai->login);
    if(!spwd)
    {
      /* No username info */
      if(pai->message[0])
        free(pai->message[0]);
      pai->message[0] = xstrdup("Invalid Login.\n");  
      pai->msn = 1;
      return(1);
    }
    password = spwd->sp_pwdp;
  }
  else
#endif    /* HAS_SHADOW */
  {
    password = pwd->pw_passwd;
  }

  if(*password == '\0' && pai->passwd == '\0')
  {
    rtn = 0;
  }
  else
  {
    char salt[12];
    if(!strncmp(password, "$1$", 3))
    {
      memcpy(salt, password, sizeof(salt) - 1);
      salt[sizeof(salt) - 1] = '\0';
    }
    else
    {
      memcpy(salt, password, 2);
      salt[2] = '\0';
    }
    if((rtn = strcmp(pw_encrypt(pai->passwd, salt), password)))
    {
      /* No username info */
      if(pai->message[0])
        free(pai->message[0]);
      pai->message[0] = xstrdup("Invalid Login.\n");  
      pai->msn = 1;
    }
  }

  endpwent();  /* stop access to password file */
  endgrent();  /* stop access to group file */
#ifdef HAS_SHADOW
  endspent();  /* stop access to shadow passwd file */
#endif
  if(!rtn && pai->proto == P_AUTOPPP)
  { 
    pai->proto = P_PPP;
    if(pai->address == 0 && lineconf.rem_host)
      pai->address = lineconf.rem_host;
  }
  if(rtn == 0)
    pai->authenticated = 1;
  return(rtn);
}

static int tac_client(struct auth *pai)
{
  if(pai->message[0])
    free(pai->message[0]);
  pai->message[0] = xstrdup("Authentication protocol is not available yet\n");
  pai->msn = 1;
  return(1);
}

#ifdef UNUSED
/* This isn't ready yet */
static int check_users_access(char *users, char *the_user)
{
  char *p, ch[32];
  int  deny_access;

  if(users == NULL || *users == 0)
    return 0;

  deny_access = strchr(users, '!') ? 1 : 0;

  for(p=ch; users; users++)
  {
    switch(*users)
    {
    case 0:
      users = (char *)-1;
      /* fall through */
    case '!':
    case ',':
    case ' ':
    case '\t':
      if(p != ch)
      {
        *p = 0;
        if (!strcmp(ch, the_user))
          return(deny_access);
        p = ch;
      }
    break;
    default:
      *p++ = *users;
    break;
    }
  }

  /* user not found only accept if users should be denied */
  return(!deny_access);
}
#endif

int do_local_or_server_authentication(struct auth *pai, bool ppp)
{
  unsigned int x;

  pai->msn = 0;
  pai->do_acct = false;
  pai->authenticated = false;

  switch (lineconf.authtype)
  {
  case AUTH_NONE:
    return(0);

  case AUTH_RADIUS_LOCAL:
    if(!rad_client(pai, ppp))
    {
      pai->do_acct = 1;
      goto check_access;
    }
    if(!login_local(pai))
      goto check_access;
  break;

  case AUTH_LOCAL_RADIUS:
    if(!login_local(pai))
      goto check_access;
    if(!rad_client(pai, ppp))
    {
      pai->do_acct = 1;
      goto check_access;
    }
  break;

  case AUTH_RADIUS:
    if(!rad_client(pai, ppp))
    {
      pai->do_acct = 1;
      goto check_access;
    }
  break;

  case AUTH_TACACS:
    if(tac_client(pai))
      goto check_access;
  break;

  case AUTH_LOCAL_TACACS:
    if(!login_local(pai))
      goto check_access;
    if(tac_client(pai))
      goto check_access;
  break;

  case AUTH_TACACS_LOCAL:
    if(tac_client(pai))
      goto check_access;
    if(!login_local(pai))
      goto check_access;
  break;

  case AUTH_LOCAL:
    if(!login_local(pai))
      goto check_access;
  break;

  case AUTH_REMOTE:
    goto check_access;
  }

  if(pai->msn)
  {
    for(x = 0; x < pai->msn; x++)
      write(STDOUT_FILENO, pai->message[x], strlen(pai->message[x]));
    write(STDOUT_FILENO, "\r\n", 2);
  }
  else
  {
    write(STDOUT_FILENO, "Authentication failure\r\n", 24);
  }
  return(1);

check_access:
/*
  if (check_users_access(lineconf.users, pai->login)) {
    write(STDOUT_FILENO, "Port access permission denied\r\n", 32);
    return(1);
  }
*/

  return(0);
}

#ifndef NO_CHAP
int do_chap_authentication(struct auth *pai, u_char *remmd, chap_state *cstate)
{
  pai->msn = 0;
  pai->do_acct = false;
  pai->authenticated = false;

  switch (lineconf.authtype)
  {
  case AUTH_NONE:
    return CHAP_FAILURE;

  case AUTH_RADIUS_LOCAL:
  case AUTH_LOCAL_RADIUS:
  case AUTH_RADIUS:
    if(rad_client_chap(pai, remmd, cstate) == CHAP_SUCCESS)
    {
      pai->do_acct = 1;
      pai->done_chap_once++;
      if(pai->done_chap_once > 1)
        goto check_access;
      return CHAP_SUCCESS;
    }
  break;

  case AUTH_REMOTE:
    goto check_access;
  }
  return CHAP_FAILURE;

check_access:
/*
  if (check_users_access(lineconf.users, pai->login)) {
    write(STDOUT_FILENO, "Port access permission denied\r\n", 32);
    return CHAP_FAILURE;
  }
*/

  return CHAP_SUCCESS;
}
#endif
