/***************************************************************************
 *   Copyright (C) 2007-2008 by Anistratov Oleg                            *
 *   ower86@gmail.com                                                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation;                         *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 ***************************************************************************/

#ifndef INTERFACES_H
#define INTERFACES_H

#include <QObject>

#include <QHostAddress>
#include <QByteArray>
#include <QIcon>
#include <QMenu>
#include <QToolBar>

/**
 * @class QChatBasicPlugin
 * @author Anistratov Oleg <ower@users.sourceforge.net>
 * @brief Basic abstract interface for qchat plugins
 *
 * \note If plugin needs to send data it must use for this purpose signal with following signature:\n
 * \code
 *  void sendData(const QString & plugin_id,
 *                const QMap < QString, QByteArray > & parametrs,
 *                quint64 destination_user_id,
 *                AbstractChatCore::DataType data_type,
 *                const QString & channel_id,
 *                uint channel_type,
 *                const QString & message);
 * \endcode
 *
 * \b plugin_id is used to determine to which plugin data is arrived \n
 * As \b data_type in most cases you need to use a AbstractChatCore::PLUGIN_DATA
 *
 *
*/
class QChatBasicPlugin
{
  private:
    QString m_path;
  public:
    virtual ~QChatBasicPlugin(){};

    virtual void load() = 0;
    virtual void unload() = 0;

    /**
     * @brief Path to plugin
     * \return full path to plugin
     */
    const QString& path() const {return m_path;}

    void setPath(const QString& p){m_path = p;}

    /**
     * @brief Plugin's icon
     * \return pointer to plugin's icon or NULL pointer if there is no icon
     */
    virtual QIcon* icon() const = 0;

    /**
     * @brief Plugin's settings page
     * \return pointer to widget with plugin's settings or NULL pointer if plugin have no settings
     */
    virtual QWidget* settingsPage() const = 0;

    /**
     * @brief Name of a plugin
     * \return name of a plugin
     */
    virtual QString name() const = 0;

    /**
     * @brief Information about plugin
     * \return 'about information' which will be placed in 'about plugin' section
     */
    virtual QString about() const = 0;

    /**
     * @brief Must process data which arrived from network
     * @param parametrs - QMap with key - QString parametrName and value - QByteArray parametrValue
     */
    virtual void processData(QMap<QString, QByteArray> parametrs) = 0;
};

/**
  @class QChatWidgetPlugin
  @author Anistratov Oleg <ower@users.sourceforge.net>
  @brief Abstract interface for qchat plugins with GUI
*/
class QChatWidgetPlugin : public QChatBasicPlugin
{
  public:
    virtual ~QChatWidgetPlugin(){};

    /**
     * @brief Plugin's main widget
     * \return Plugin's main widget
     */
    virtual QWidget* widget() = 0;

//     virtual void updateMenus(QMenu*) const = 0;

    /**
     * @brief Setting up toolbar object with plugin's actions
     *
     * This function is calls when plugin's widget is activating, it must setup toolbar object with plugin's actions
     */
    virtual void setupToolBar(QToolBar*) const = 0;
};

Q_DECLARE_INTERFACE(QChatBasicPlugin, "QChat.QChatBasicPlugin/0.1");
Q_DECLARE_INTERFACE(QChatWidgetPlugin, "QChat.QChatWidgetPlugin/0.1");

#endif
