% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tidy.R
\name{tidy_source}
\alias{tidy_source}
\title{Reformat R code}
\usage{
tidy_source(
  source = "clipboard",
  comment = getOption("formatR.comment", TRUE),
  blank = getOption("formatR.blank", TRUE),
  arrow = getOption("formatR.arrow", FALSE),
  pipe = getOption("formatR.pipe", FALSE),
  brace.newline = getOption("formatR.brace.newline", FALSE),
  indent = getOption("formatR.indent", 4),
  wrap = getOption("formatR.wrap", TRUE),
  width.cutoff = getOption("formatR.width", getOption("width")),
  args.newline = getOption("formatR.args.newline", FALSE),
  output = TRUE,
  text = NULL,
  ...
)
}
\arguments{
\item{source}{A character string: file path to the source code (defaults to
the clipboard).}

\item{comment}{Whether to keep comments.}

\item{blank}{Whether to keep blank lines.}

\item{arrow}{Whether to substitute the assignment operator \code{=} with
\code{<-}.}

\item{pipe}{Whether to substitute the \pkg{magrittr} pipe \code{\%>\%} with
R's native pipe operator \code{|>}.}

\item{brace.newline}{Whether to put the left brace \code{\{} to a new line.}

\item{indent}{Number of spaces to indent the code.}

\item{wrap}{Whether to wrap comments to the linewidth determined by
\code{width.cutoff} (roxygen comments will never be wrapped).}

\item{width.cutoff}{An integer in \code{[20, 500]}: if a line's character
length is at or over this number, the function will try to break it into a
new line. In other words, this is the \emph{lower bound} of the line width.
See \sQuote{Details} if an upper bound is desired instead.}

\item{args.newline}{Whether to start the arguments of a function call on a
new line instead of after the function name and \code{(} when the arguments
cannot fit one line.}

\item{output}{Whether to output to the console or a file using
\code{\link{cat}()}.}

\item{text}{An alternative way to specify the input: if \code{NULL}, the
function will use the \code{source} argument; if a character vector
containing the source code, the function will use this and ignore the
\code{source} argument.}

\item{...}{Other arguments passed to \code{\link{cat}()}, e.g. \code{file}
(this can be useful for batch-processing R scripts, e.g.
\code{tidy_source(source = 'input.R', file = 'output.R')}).}
}
\value{
A list with components \item{text.tidy}{the reformatted code as a
  character vector} \item{text.mask}{the code containing comments, which are
  masked in assignments or with the weird operator}.
}
\description{
Read R code from a file or the clipboard and reformat it. This function is
based on \code{\link{parse}()} and \code{\link{deparse}()}, but it does
several other things, such as preserving blank lines and comments,
substituting the assignment operator \code{=} with \code{<-}, and
re-indenting code with a specified number of spaces.
}
\details{
A value of the argument \code{width.cutoff} wrapped in \code{\link{I}()}
(e.g., \code{I(60)}) will be treated as the \emph{upper bound} of the line
width. The corresponding argument to \code{deparse()} is a lower bound, so
the function will perform a binary search for a width value that can make
\code{deparse()} return code with line width smaller than or equal to the
\code{width.cutoff} value. If the search fails, a warning will signal,
suppressible by global option \code{options(formatR.width.warning = FALSE)}.
}
\note{
Be sure to read the reference to know other limitations.
}
\examples{
library(formatR)

## a messy R script
messy = system.file("format", "messy.R", package = "formatR")
tidy_source(messy)

## use the 'text' argument
src = readLines(messy)

## source code
cat(src, sep = "\n")

## the formatted version
tidy_source(text = src)

## preserve blank lines
tidy_source(text = src, blank = TRUE)

## indent with 2 spaces
tidy_source(text = src, indent = 2)

## discard comments!
tidy_source(text = src, comment = FALSE)

## wanna see the gory truth??
tidy_source(text = src, output = FALSE)$text.mask


## tidy up the source code of image demo
x = file.path(system.file(package = "graphics"), "demo", "image.R")

# to console
tidy_source(x)

# to a file
f = tempfile()
tidy_source(x, blank = TRUE, file = f)

## check the original code here and see the difference
file.show(x)
file.show(f)

## use global options
options(comment = TRUE, blank = FALSE)
tidy_source(x)

## if you've copied R code into the clipboard
if (interactive()) {
    tidy_source("clipboard")
    ## write into clipboard again
    tidy_source("clipboard", file = "clipboard")
}

## the if-else structure
tidy_source(text = c("{if(TRUE)1 else 2; if(FALSE){1+1", "## comments", "} else 2}"))
}
\references{
\url{https://yihui.org/formatR/} (an introduction to this
  package, with examples and further notes)
}
\seealso{
\code{\link{parse}()}, \code{\link{deparse}()}
}
\author{
Yihui Xie <\url{https://yihui.org}> with substantial contribution
  from Yixuan Qiu <\url{https://yixuan.blog}>
}
