;;  Filename : test-char-cmp.scm
;;  About    : unit test for R5RS char comparison predicates
;;
;;  Copyright (C) 2006 YAMAMOTO Kengo <yamaken AT bp.iij4u.or.jp>
;;  Copyright (c) 2007-2008 SigScheme Project <uim-en AT googlegroups.com>
;;
;;  All rights reserved.
;;
;;  Redistribution and use in source and binary forms, with or without
;;  modification, are permitted provided that the following conditions
;;  are met:
;;
;;  1. Redistributions of source code must retain the above copyright
;;     notice, this list of conditions and the following disclaimer.
;;  2. Redistributions in binary form must reproduce the above copyright
;;     notice, this list of conditions and the following disclaimer in the
;;     documentation and/or other materials provided with the distribution.
;;  3. Neither the name of authors nor the names of its contributors
;;     may be used to endorse or promote products derived from this software
;;     without specific prior written permission.
;;
;;  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
;;  IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
;;  THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
;;  PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
;;  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
;;  EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
;;  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
;;  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
;;  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
;;  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
;;  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

(require-extension (unittest))

(if (not (symbol-bound? 'char=?))
    (test-skip "R5RS characters is not enabled"))

(define tn test-name)

;; SigScheme's case-insensitive comparison conforms to the foldcase'ed
;; comparison described in R6RS and SRFI-13, although R5RS does not specify
;; comparison between alphabetic and non-alphabetic char.
;;
;; This specification is needed to produce natural result on sort functions
;; with these case-insensitive predicates as comparator.
;;
;;   (a-sort '(#\a #\c #\B #\D #\1 #\[ #\$ #\_) char-ci<?)
;;     => (#\$ #\1 #\a #\B #\c #\D #\[ #\_)  ;; the "natural result"
;;
;;     => (#\$ #\1 #\B #\D #\[ #\_ #\a #\c)  ;; "unnatural result"
;;
;; See also:
;;
;;   - Description around 'char-foldcase' in R6RS (R5.92) Standard Libraries
;;     http://www.r6rs.org/document/lib-html/r6rs-lib-Z-H-3.html#node_sec_1.1
;;   - "Case mapping and case-folding" and "Comparison" section of SRFI-13

;; char=?
(tn "char=? #\\x00")
(assert-true   (tn) (char=? #\x00 #\x00))
(assert-false  (tn) (char=? #\x00 #\newline))
(assert-false  (tn) (char=? #\x00 #\space))
(assert-false  (tn) (char=? #\x00 #\!))
(assert-false  (tn) (char=? #\x00 #\0))
(assert-false  (tn) (char=? #\x00 #\9))
(assert-false  (tn) (char=? #\x00 #\A))
(assert-false  (tn) (char=? #\x00 #\B))
(assert-false  (tn) (char=? #\x00 #\Z))
(assert-false  (tn) (char=? #\x00 #\_))
(assert-false  (tn) (char=? #\x00 #\a))
(assert-false  (tn) (char=? #\x00 #\b))
(assert-false  (tn) (char=? #\x00 #\z))
(assert-false  (tn) (char=? #\x00 #\~))
(assert-false  (tn) (char=? #\x00 #\x7f))
(tn "char=? #\\newline")
(assert-false  (tn) (char=? #\newline #\x00))
(assert-true   (tn) (char=? #\newline #\newline))
(assert-false  (tn) (char=? #\newline #\space))
(assert-false  (tn) (char=? #\newline #\!))
(assert-false  (tn) (char=? #\newline #\0))
(assert-false  (tn) (char=? #\newline #\9))
(assert-false  (tn) (char=? #\newline #\A))
(assert-false  (tn) (char=? #\newline #\B))
(assert-false  (tn) (char=? #\newline #\Z))
(assert-false  (tn) (char=? #\newline #\_))
(assert-false  (tn) (char=? #\newline #\a))
(assert-false  (tn) (char=? #\newline #\b))
(assert-false  (tn) (char=? #\newline #\z))
(assert-false  (tn) (char=? #\newline #\~))
(assert-false  (tn) (char=? #\newline #\x7f))
(tn "char=? #\\space")
(assert-false  (tn) (char=? #\space #\x00))
(assert-false  (tn) (char=? #\space #\newline))
(assert-true   (tn) (char=? #\space #\space))
(assert-false  (tn) (char=? #\space #\!))
(assert-false  (tn) (char=? #\space #\0))
(assert-false  (tn) (char=? #\space #\9))
(assert-false  (tn) (char=? #\space #\A))
(assert-false  (tn) (char=? #\space #\B))
(assert-false  (tn) (char=? #\space #\Z))
(assert-false  (tn) (char=? #\space #\_))
(assert-false  (tn) (char=? #\space #\a))
(assert-false  (tn) (char=? #\space #\b))
(assert-false  (tn) (char=? #\space #\z))
(assert-false  (tn) (char=? #\space #\~))
(assert-false  (tn) (char=? #\space #\x7f))
(tn "char=? #\\!")
(assert-false  (tn) (char=? #\! #\x00))
(assert-false  (tn) (char=? #\! #\newline))
(assert-false  (tn) (char=? #\! #\space))
(assert-true   (tn) (char=? #\! #\!))
(assert-false  (tn) (char=? #\! #\0))
(assert-false  (tn) (char=? #\! #\9))
(assert-false  (tn) (char=? #\! #\A))
(assert-false  (tn) (char=? #\! #\B))
(assert-false  (tn) (char=? #\! #\Z))
(assert-false  (tn) (char=? #\! #\_))
(assert-false  (tn) (char=? #\! #\a))
(assert-false  (tn) (char=? #\! #\b))
(assert-false  (tn) (char=? #\! #\z))
(assert-false  (tn) (char=? #\! #\~))
(assert-false  (tn) (char=? #\! #\x7f))
(tn "char=? #\\0")
(assert-false  (tn) (char=? #\0 #\x00))
(assert-false  (tn) (char=? #\0 #\newline))
(assert-false  (tn) (char=? #\0 #\space))
(assert-false  (tn) (char=? #\0 #\!))
(assert-true   (tn) (char=? #\0 #\0))
(assert-false  (tn) (char=? #\0 #\9))
(assert-false  (tn) (char=? #\0 #\A))
(assert-false  (tn) (char=? #\0 #\B))
(assert-false  (tn) (char=? #\0 #\Z))
(assert-false  (tn) (char=? #\0 #\_))
(assert-false  (tn) (char=? #\0 #\a))
(assert-false  (tn) (char=? #\0 #\b))
(assert-false  (tn) (char=? #\0 #\z))
(assert-false  (tn) (char=? #\0 #\~))
(assert-false  (tn) (char=? #\0 #\x7f))
(tn "char=? #\\9")
(assert-false  (tn) (char=? #\9 #\x00))
(assert-false  (tn) (char=? #\9 #\newline))
(assert-false  (tn) (char=? #\9 #\space))
(assert-false  (tn) (char=? #\9 #\!))
(assert-false  (tn) (char=? #\9 #\0))
(assert-true   (tn) (char=? #\9 #\9))
(assert-false  (tn) (char=? #\9 #\A))
(assert-false  (tn) (char=? #\9 #\B))
(assert-false  (tn) (char=? #\9 #\Z))
(assert-false  (tn) (char=? #\9 #\_))
(assert-false  (tn) (char=? #\9 #\a))
(assert-false  (tn) (char=? #\9 #\b))
(assert-false  (tn) (char=? #\9 #\z))
(assert-false  (tn) (char=? #\9 #\~))
(assert-false  (tn) (char=? #\9 #\x7f))
(tn "char=? #\\A")
(assert-false  (tn) (char=? #\A #\x00))
(assert-false  (tn) (char=? #\A #\newline))
(assert-false  (tn) (char=? #\A #\space))
(assert-false  (tn) (char=? #\A #\!))
(assert-false  (tn) (char=? #\A #\0))
(assert-false  (tn) (char=? #\A #\9))
(assert-true   (tn) (char=? #\A #\A))
(assert-false  (tn) (char=? #\A #\B))
(assert-false  (tn) (char=? #\A #\Z))
(assert-false  (tn) (char=? #\A #\_))
(assert-false  (tn) (char=? #\A #\a))
(assert-false  (tn) (char=? #\A #\b))
(assert-false  (tn) (char=? #\A #\z))
(assert-false  (tn) (char=? #\A #\~))
(assert-false  (tn) (char=? #\A #\x7f))
(tn "char=? #\\B")
(assert-false  (tn) (char=? #\B #\x00))
(assert-false  (tn) (char=? #\B #\newline))
(assert-false  (tn) (char=? #\B #\space))
(assert-false  (tn) (char=? #\B #\!))
(assert-false  (tn) (char=? #\B #\0))
(assert-false  (tn) (char=? #\B #\9))
(assert-false  (tn) (char=? #\B #\A))
(assert-true   (tn) (char=? #\B #\B))
(assert-false  (tn) (char=? #\B #\Z))
(assert-false  (tn) (char=? #\B #\_))
(assert-false  (tn) (char=? #\B #\a))
(assert-false  (tn) (char=? #\B #\b))
(assert-false  (tn) (char=? #\B #\z))
(assert-false  (tn) (char=? #\B #\~))
(assert-false  (tn) (char=? #\B #\x7f))
(tn "char=? #\\Z")
(assert-false  (tn) (char=? #\Z #\x00))
(assert-false  (tn) (char=? #\Z #\newline))
(assert-false  (tn) (char=? #\Z #\space))
(assert-false  (tn) (char=? #\Z #\!))
(assert-false  (tn) (char=? #\Z #\0))
(assert-false  (tn) (char=? #\Z #\9))
(assert-false  (tn) (char=? #\Z #\A))
(assert-false  (tn) (char=? #\Z #\B))
(assert-true   (tn) (char=? #\Z #\Z))
(assert-false  (tn) (char=? #\Z #\_))
(assert-false  (tn) (char=? #\Z #\a))
(assert-false  (tn) (char=? #\Z #\b))
(assert-false  (tn) (char=? #\Z #\z))
(assert-false  (tn) (char=? #\Z #\~))
(assert-false  (tn) (char=? #\Z #\x7f))
(tn "char=? #\\_")
(assert-false  (tn) (char=? #\_ #\x00))
(assert-false  (tn) (char=? #\_ #\newline))
(assert-false  (tn) (char=? #\_ #\space))
(assert-false  (tn) (char=? #\_ #\!))
(assert-false  (tn) (char=? #\_ #\0))
(assert-false  (tn) (char=? #\_ #\9))
(assert-false  (tn) (char=? #\_ #\A))
(assert-false  (tn) (char=? #\_ #\B))
(assert-false  (tn) (char=? #\_ #\Z))
(assert-true   (tn) (char=? #\_ #\_))
(assert-false  (tn) (char=? #\_ #\a))
(assert-false  (tn) (char=? #\_ #\b))
(assert-false  (tn) (char=? #\_ #\z))
(assert-false  (tn) (char=? #\_ #\~))
(assert-false  (tn) (char=? #\_ #\x7f))
(tn "char=? #\\a")
(assert-false  (tn) (char=? #\a #\x00))
(assert-false  (tn) (char=? #\a #\newline))
(assert-false  (tn) (char=? #\a #\space))
(assert-false  (tn) (char=? #\a #\!))
(assert-false  (tn) (char=? #\a #\0))
(assert-false  (tn) (char=? #\a #\9))
(assert-false  (tn) (char=? #\a #\A))
(assert-false  (tn) (char=? #\a #\B))
(assert-false  (tn) (char=? #\a #\Z))
(assert-false  (tn) (char=? #\a #\_))
(assert-true   (tn) (char=? #\a #\a))
(assert-false  (tn) (char=? #\a #\b))
(assert-false  (tn) (char=? #\a #\z))
(assert-false  (tn) (char=? #\a #\~))
(assert-false  (tn) (char=? #\a #\x7f))
(tn "char=? #\\b")
(assert-false  (tn) (char=? #\b #\x00))
(assert-false  (tn) (char=? #\b #\newline))
(assert-false  (tn) (char=? #\b #\space))
(assert-false  (tn) (char=? #\b #\!))
(assert-false  (tn) (char=? #\b #\0))
(assert-false  (tn) (char=? #\b #\9))
(assert-false  (tn) (char=? #\b #\A))
(assert-false  (tn) (char=? #\b #\B))
(assert-false  (tn) (char=? #\b #\Z))
(assert-false  (tn) (char=? #\b #\_))
(assert-false  (tn) (char=? #\b #\a))
(assert-true   (tn) (char=? #\b #\b))
(assert-false  (tn) (char=? #\b #\z))
(assert-false  (tn) (char=? #\b #\~))
(assert-false  (tn) (char=? #\b #\x7f))
(tn "char=? #\\z")
(assert-false  (tn) (char=? #\z #\x00))
(assert-false  (tn) (char=? #\z #\newline))
(assert-false  (tn) (char=? #\z #\space))
(assert-false  (tn) (char=? #\z #\!))
(assert-false  (tn) (char=? #\z #\0))
(assert-false  (tn) (char=? #\z #\9))
(assert-false  (tn) (char=? #\z #\A))
(assert-false  (tn) (char=? #\z #\B))
(assert-false  (tn) (char=? #\z #\Z))
(assert-false  (tn) (char=? #\z #\_))
(assert-false  (tn) (char=? #\z #\a))
(assert-false  (tn) (char=? #\z #\b))
(assert-true   (tn) (char=? #\z #\z))
(assert-false  (tn) (char=? #\z #\~))
(assert-false  (tn) (char=? #\z #\x7f))
(tn "char=? #\\~")
(assert-false  (tn) (char=? #\~ #\x00))
(assert-false  (tn) (char=? #\~ #\newline))
(assert-false  (tn) (char=? #\~ #\space))
(assert-false  (tn) (char=? #\~ #\!))
(assert-false  (tn) (char=? #\~ #\0))
(assert-false  (tn) (char=? #\~ #\9))
(assert-false  (tn) (char=? #\~ #\A))
(assert-false  (tn) (char=? #\~ #\B))
(assert-false  (tn) (char=? #\~ #\Z))
(assert-false  (tn) (char=? #\~ #\_))
(assert-false  (tn) (char=? #\~ #\a))
(assert-false  (tn) (char=? #\~ #\b))
(assert-false  (tn) (char=? #\~ #\z))
(assert-true   (tn) (char=? #\~ #\~))
(assert-false  (tn) (char=? #\~ #\x7f))
(tn "char=? #\\x7f")
(assert-false  (tn) (char=? #\x7f #\x00))
(assert-false  (tn) (char=? #\x7f #\newline))
(assert-false  (tn) (char=? #\x7f #\space))
(assert-false  (tn) (char=? #\x7f #\!))
(assert-false  (tn) (char=? #\x7f #\0))
(assert-false  (tn) (char=? #\x7f #\9))
(assert-false  (tn) (char=? #\x7f #\A))
(assert-false  (tn) (char=? #\x7f #\B))
(assert-false  (tn) (char=? #\x7f #\Z))
(assert-false  (tn) (char=? #\x7f #\_))
(assert-false  (tn) (char=? #\x7f #\a))
(assert-false  (tn) (char=? #\x7f #\b))
(assert-false  (tn) (char=? #\x7f #\z))
(assert-false  (tn) (char=? #\x7f #\~))
(assert-true   (tn) (char=? #\x7f #\x7f))

;; char<?
(tn "char<? #\\x00")
(assert-false  (tn) (char<? #\x00 #\x00))
(assert-true   (tn) (char<? #\x00 #\newline))
(assert-true   (tn) (char<? #\x00 #\space))
(assert-true   (tn) (char<? #\x00 #\!))
(assert-true   (tn) (char<? #\x00 #\0))
(assert-true   (tn) (char<? #\x00 #\9))
(assert-true   (tn) (char<? #\x00 #\A))
(assert-true   (tn) (char<? #\x00 #\B))
(assert-true   (tn) (char<? #\x00 #\Z))
(assert-true   (tn) (char<? #\x00 #\_))
(assert-true   (tn) (char<? #\x00 #\a))
(assert-true   (tn) (char<? #\x00 #\b))
(assert-true   (tn) (char<? #\x00 #\z))
(assert-true   (tn) (char<? #\x00 #\~))
(assert-true   (tn) (char<? #\x00 #\x7f))
(tn "char<? #\\newline")
(assert-false  (tn) (char<? #\newline #\x00))
(assert-false  (tn) (char<? #\newline #\newline))
(assert-true   (tn) (char<? #\newline #\space))
(assert-true   (tn) (char<? #\newline #\!))
(assert-true   (tn) (char<? #\newline #\0))
(assert-true   (tn) (char<? #\newline #\9))
(assert-true   (tn) (char<? #\newline #\A))
(assert-true   (tn) (char<? #\newline #\B))
(assert-true   (tn) (char<? #\newline #\Z))
(assert-true   (tn) (char<? #\newline #\_))
(assert-true   (tn) (char<? #\newline #\a))
(assert-true   (tn) (char<? #\newline #\b))
(assert-true   (tn) (char<? #\newline #\z))
(assert-true   (tn) (char<? #\newline #\~))
(assert-true   (tn) (char<? #\newline #\x7f))
(tn "char<? #\\space")
(assert-false  (tn) (char<? #\space #\x00))
(assert-false  (tn) (char<? #\space #\newline))
(assert-false  (tn) (char<? #\space #\space))
(assert-true   (tn) (char<? #\space #\!))
(assert-true   (tn) (char<? #\space #\0))
(assert-true   (tn) (char<? #\space #\9))
(assert-true   (tn) (char<? #\space #\A))
(assert-true   (tn) (char<? #\space #\B))
(assert-true   (tn) (char<? #\space #\Z))
(assert-true   (tn) (char<? #\space #\_))
(assert-true   (tn) (char<? #\space #\a))
(assert-true   (tn) (char<? #\space #\b))
(assert-true   (tn) (char<? #\space #\z))
(assert-true   (tn) (char<? #\space #\~))
(assert-true   (tn) (char<? #\space #\x7f))
(tn "char<? #\\!")
(assert-false  (tn) (char<? #\! #\x00))
(assert-false  (tn) (char<? #\! #\newline))
(assert-false  (tn) (char<? #\! #\space))
(assert-false  (tn) (char<? #\! #\!))
(assert-true   (tn) (char<? #\! #\0))
(assert-true   (tn) (char<? #\! #\9))
(assert-true   (tn) (char<? #\! #\A))
(assert-true   (tn) (char<? #\! #\B))
(assert-true   (tn) (char<? #\! #\Z))
(assert-true   (tn) (char<? #\! #\_))
(assert-true   (tn) (char<? #\! #\a))
(assert-true   (tn) (char<? #\! #\b))
(assert-true   (tn) (char<? #\! #\z))
(assert-true   (tn) (char<? #\! #\~))
(assert-true   (tn) (char<? #\! #\x7f))
(tn "char<? #\\0")
(assert-false  (tn) (char<? #\0 #\x00))
(assert-false  (tn) (char<? #\0 #\newline))
(assert-false  (tn) (char<? #\0 #\space))
(assert-false  (tn) (char<? #\0 #\!))
(assert-false  (tn) (char<? #\0 #\0))
(assert-true   (tn) (char<? #\0 #\9))
(assert-true   (tn) (char<? #\0 #\A))
(assert-true   (tn) (char<? #\0 #\B))
(assert-true   (tn) (char<? #\0 #\Z))
(assert-true   (tn) (char<? #\0 #\_))
(assert-true   (tn) (char<? #\0 #\a))
(assert-true   (tn) (char<? #\0 #\b))
(assert-true   (tn) (char<? #\0 #\z))
(assert-true   (tn) (char<? #\0 #\~))
(assert-true   (tn) (char<? #\0 #\x7f))
(tn "char<? #\\9")
(assert-false  (tn) (char<? #\9 #\x00))
(assert-false  (tn) (char<? #\9 #\newline))
(assert-false  (tn) (char<? #\9 #\space))
(assert-false  (tn) (char<? #\9 #\!))
(assert-false  (tn) (char<? #\9 #\0))
(assert-false  (tn) (char<? #\9 #\9))
(assert-true   (tn) (char<? #\9 #\A))
(assert-true   (tn) (char<? #\9 #\B))
(assert-true   (tn) (char<? #\9 #\Z))
(assert-true   (tn) (char<? #\9 #\_))
(assert-true   (tn) (char<? #\9 #\a))
(assert-true   (tn) (char<? #\9 #\b))
(assert-true   (tn) (char<? #\9 #\z))
(assert-true   (tn) (char<? #\9 #\~))
(assert-true   (tn) (char<? #\9 #\x7f))
(tn "char<? #\\A")
(assert-false  (tn) (char<? #\A #\x00))
(assert-false  (tn) (char<? #\A #\newline))
(assert-false  (tn) (char<? #\A #\space))
(assert-false  (tn) (char<? #\A #\!))
(assert-false  (tn) (char<? #\A #\0))
(assert-false  (tn) (char<? #\A #\9))
(assert-false  (tn) (char<? #\A #\A))
(assert-true   (tn) (char<? #\A #\B))
(assert-true   (tn) (char<? #\A #\Z))
(assert-true   (tn) (char<? #\A #\_))
(assert-true   (tn) (char<? #\A #\a))
(assert-true   (tn) (char<? #\A #\b))
(assert-true   (tn) (char<? #\A #\z))
(assert-true   (tn) (char<? #\A #\~))
(assert-true   (tn) (char<? #\A #\x7f))
(tn "char<? #\\B")
(assert-false  (tn) (char<? #\B #\x00))
(assert-false  (tn) (char<? #\B #\newline))
(assert-false  (tn) (char<? #\B #\space))
(assert-false  (tn) (char<? #\B #\!))
(assert-false  (tn) (char<? #\B #\0))
(assert-false  (tn) (char<? #\B #\9))
(assert-false  (tn) (char<? #\B #\A))
(assert-false  (tn) (char<? #\B #\B))
(assert-true   (tn) (char<? #\B #\Z))
(assert-true   (tn) (char<? #\B #\_))
(assert-true   (tn) (char<? #\B #\a))
(assert-true   (tn) (char<? #\B #\b))
(assert-true   (tn) (char<? #\B #\z))
(assert-true   (tn) (char<? #\B #\~))
(assert-true   (tn) (char<? #\B #\x7f))
(tn "char<? #\\Z")
(assert-false  (tn) (char<? #\Z #\x00))
(assert-false  (tn) (char<? #\Z #\newline))
(assert-false  (tn) (char<? #\Z #\space))
(assert-false  (tn) (char<? #\Z #\!))
(assert-false  (tn) (char<? #\Z #\0))
(assert-false  (tn) (char<? #\Z #\9))
(assert-false  (tn) (char<? #\Z #\A))
(assert-false  (tn) (char<? #\Z #\B))
(assert-false  (tn) (char<? #\Z #\Z))
(assert-true   (tn) (char<? #\Z #\_))
(assert-true   (tn) (char<? #\Z #\a))
(assert-true   (tn) (char<? #\Z #\b))
(assert-true   (tn) (char<? #\Z #\z))
(assert-true   (tn) (char<? #\Z #\~))
(assert-true   (tn) (char<? #\Z #\x7f))
(tn "char<? #\\_")
(assert-false  (tn) (char<? #\_ #\x00))
(assert-false  (tn) (char<? #\_ #\newline))
(assert-false  (tn) (char<? #\_ #\space))
(assert-false  (tn) (char<? #\_ #\!))
(assert-false  (tn) (char<? #\_ #\0))
(assert-false  (tn) (char<? #\_ #\9))
(assert-false  (tn) (char<? #\_ #\A))
(assert-false  (tn) (char<? #\_ #\B))
(assert-false  (tn) (char<? #\_ #\Z))
(assert-false  (tn) (char<? #\_ #\_))
(assert-true   (tn) (char<? #\_ #\a))
(assert-true   (tn) (char<? #\_ #\b))
(assert-true   (tn) (char<? #\_ #\z))
(assert-true   (tn) (char<? #\_ #\~))
(assert-true   (tn) (char<? #\_ #\x7f))
(tn "char<? #\\a")
(assert-false  (tn) (char<? #\a #\x00))
(assert-false  (tn) (char<? #\a #\newline))
(assert-false  (tn) (char<? #\a #\space))
(assert-false  (tn) (char<? #\a #\!))
(assert-false  (tn) (char<? #\a #\0))
(assert-false  (tn) (char<? #\a #\9))
(assert-false  (tn) (char<? #\a #\A))
(assert-false  (tn) (char<? #\a #\B))
(assert-false  (tn) (char<? #\a #\Z))
(assert-false  (tn) (char<? #\a #\_))
(assert-false  (tn) (char<? #\a #\a))
(assert-true   (tn) (char<? #\a #\b))
(assert-true   (tn) (char<? #\a #\z))
(assert-true   (tn) (char<? #\a #\~))
(assert-true   (tn) (char<? #\a #\x7f))
(tn "char<? #\\b")
(assert-false  (tn) (char<? #\b #\x00))
(assert-false  (tn) (char<? #\b #\newline))
(assert-false  (tn) (char<? #\b #\space))
(assert-false  (tn) (char<? #\b #\!))
(assert-false  (tn) (char<? #\b #\0))
(assert-false  (tn) (char<? #\b #\9))
(assert-false  (tn) (char<? #\b #\A))
(assert-false  (tn) (char<? #\b #\B))
(assert-false  (tn) (char<? #\b #\Z))
(assert-false  (tn) (char<? #\b #\_))
(assert-false  (tn) (char<? #\b #\a))
(assert-false  (tn) (char<? #\b #\b))
(assert-true   (tn) (char<? #\b #\z))
(assert-true   (tn) (char<? #\b #\~))
(assert-true   (tn) (char<? #\b #\x7f))
(tn "char<? #\\z")
(assert-false  (tn) (char<? #\z #\x00))
(assert-false  (tn) (char<? #\z #\newline))
(assert-false  (tn) (char<? #\z #\space))
(assert-false  (tn) (char<? #\z #\!))
(assert-false  (tn) (char<? #\z #\0))
(assert-false  (tn) (char<? #\z #\9))
(assert-false  (tn) (char<? #\z #\A))
(assert-false  (tn) (char<? #\z #\B))
(assert-false  (tn) (char<? #\z #\Z))
(assert-false  (tn) (char<? #\z #\_))
(assert-false  (tn) (char<? #\z #\a))
(assert-false  (tn) (char<? #\z #\b))
(assert-false  (tn) (char<? #\z #\z))
(assert-true   (tn) (char<? #\z #\~))
(assert-true   (tn) (char<? #\z #\x7f))
(tn "char<? #\\~")
(assert-false  (tn) (char<? #\~ #\x00))
(assert-false  (tn) (char<? #\~ #\newline))
(assert-false  (tn) (char<? #\~ #\space))
(assert-false  (tn) (char<? #\~ #\!))
(assert-false  (tn) (char<? #\~ #\0))
(assert-false  (tn) (char<? #\~ #\9))
(assert-false  (tn) (char<? #\~ #\A))
(assert-false  (tn) (char<? #\~ #\B))
(assert-false  (tn) (char<? #\~ #\Z))
(assert-false  (tn) (char<? #\~ #\_))
(assert-false  (tn) (char<? #\~ #\a))
(assert-false  (tn) (char<? #\~ #\b))
(assert-false  (tn) (char<? #\~ #\z))
(assert-false  (tn) (char<? #\~ #\~))
(assert-true   (tn) (char<? #\~ #\x7f))
(tn "char<? #\\x7f")
(assert-false  (tn) (char<? #\x7f #\x00))
(assert-false  (tn) (char<? #\x7f #\newline))
(assert-false  (tn) (char<? #\x7f #\space))
(assert-false  (tn) (char<? #\x7f #\!))
(assert-false  (tn) (char<? #\x7f #\0))
(assert-false  (tn) (char<? #\x7f #\9))
(assert-false  (tn) (char<? #\x7f #\A))
(assert-false  (tn) (char<? #\x7f #\B))
(assert-false  (tn) (char<? #\x7f #\Z))
(assert-false  (tn) (char<? #\x7f #\_))
(assert-false  (tn) (char<? #\x7f #\a))
(assert-false  (tn) (char<? #\x7f #\b))
(assert-false  (tn) (char<? #\x7f #\z))
(assert-false  (tn) (char<? #\x7f #\~))
(assert-false  (tn) (char<? #\x7f #\x7f))

;; char>?
(tn "char>? #\\x00")
(assert-false  (tn) (char>? #\x00 #\x00))
(assert-false  (tn) (char>? #\x00 #\newline))
(assert-false  (tn) (char>? #\x00 #\space))
(assert-false  (tn) (char>? #\x00 #\!))
(assert-false  (tn) (char>? #\x00 #\0))
(assert-false  (tn) (char>? #\x00 #\9))
(assert-false  (tn) (char>? #\x00 #\A))
(assert-false  (tn) (char>? #\x00 #\B))
(assert-false  (tn) (char>? #\x00 #\Z))
(assert-false  (tn) (char>? #\x00 #\_))
(assert-false  (tn) (char>? #\x00 #\a))
(assert-false  (tn) (char>? #\x00 #\b))
(assert-false  (tn) (char>? #\x00 #\z))
(assert-false  (tn) (char>? #\x00 #\~))
(assert-false  (tn) (char>? #\x00 #\x7f))
(tn "char>? #\\newline")
(assert-true   (tn) (char>? #\newline #\x00))
(assert-false  (tn) (char>? #\newline #\newline))
(assert-false  (tn) (char>? #\newline #\space))
(assert-false  (tn) (char>? #\newline #\!))
(assert-false  (tn) (char>? #\newline #\0))
(assert-false  (tn) (char>? #\newline #\9))
(assert-false  (tn) (char>? #\newline #\A))
(assert-false  (tn) (char>? #\newline #\B))
(assert-false  (tn) (char>? #\newline #\Z))
(assert-false  (tn) (char>? #\newline #\_))
(assert-false  (tn) (char>? #\newline #\a))
(assert-false  (tn) (char>? #\newline #\b))
(assert-false  (tn) (char>? #\newline #\z))
(assert-false  (tn) (char>? #\newline #\~))
(assert-false  (tn) (char>? #\newline #\x7f))
(tn "char>? #\\space")
(assert-true   (tn) (char>? #\space #\x00))
(assert-true   (tn) (char>? #\space #\newline))
(assert-false  (tn) (char>? #\space #\space))
(assert-false  (tn) (char>? #\space #\!))
(assert-false  (tn) (char>? #\space #\0))
(assert-false  (tn) (char>? #\space #\9))
(assert-false  (tn) (char>? #\space #\A))
(assert-false  (tn) (char>? #\space #\B))
(assert-false  (tn) (char>? #\space #\Z))
(assert-false  (tn) (char>? #\space #\_))
(assert-false  (tn) (char>? #\space #\a))
(assert-false  (tn) (char>? #\space #\b))
(assert-false  (tn) (char>? #\space #\z))
(assert-false  (tn) (char>? #\space #\~))
(assert-false  (tn) (char>? #\space #\x7f))
(tn "char>? #\\!")
(assert-true   (tn) (char>? #\! #\x00))
(assert-true   (tn) (char>? #\! #\newline))
(assert-true   (tn) (char>? #\! #\space))
(assert-false  (tn) (char>? #\! #\!))
(assert-false  (tn) (char>? #\! #\0))
(assert-false  (tn) (char>? #\! #\9))
(assert-false  (tn) (char>? #\! #\A))
(assert-false  (tn) (char>? #\! #\B))
(assert-false  (tn) (char>? #\! #\Z))
(assert-false  (tn) (char>? #\! #\_))
(assert-false  (tn) (char>? #\! #\a))
(assert-false  (tn) (char>? #\! #\b))
(assert-false  (tn) (char>? #\! #\z))
(assert-false  (tn) (char>? #\! #\~))
(assert-false  (tn) (char>? #\! #\x7f))
(tn "char>? #\\0")
(assert-true   (tn) (char>? #\0 #\x00))
(assert-true   (tn) (char>? #\0 #\newline))
(assert-true   (tn) (char>? #\0 #\space))
(assert-true   (tn) (char>? #\0 #\!))
(assert-false  (tn) (char>? #\0 #\0))
(assert-false  (tn) (char>? #\0 #\9))
(assert-false  (tn) (char>? #\0 #\A))
(assert-false  (tn) (char>? #\0 #\B))
(assert-false  (tn) (char>? #\0 #\Z))
(assert-false  (tn) (char>? #\0 #\_))
(assert-false  (tn) (char>? #\0 #\a))
(assert-false  (tn) (char>? #\0 #\b))
(assert-false  (tn) (char>? #\0 #\z))
(assert-false  (tn) (char>? #\0 #\~))
(assert-false  (tn) (char>? #\0 #\x7f))
(tn "char>? #\\9")
(assert-true   (tn) (char>? #\9 #\x00))
(assert-true   (tn) (char>? #\9 #\newline))
(assert-true   (tn) (char>? #\9 #\space))
(assert-true   (tn) (char>? #\9 #\!))
(assert-true   (tn) (char>? #\9 #\0))
(assert-false  (tn) (char>? #\9 #\9))
(assert-false  (tn) (char>? #\9 #\A))
(assert-false  (tn) (char>? #\9 #\B))
(assert-false  (tn) (char>? #\9 #\Z))
(assert-false  (tn) (char>? #\9 #\_))
(assert-false  (tn) (char>? #\9 #\a))
(assert-false  (tn) (char>? #\9 #\b))
(assert-false  (tn) (char>? #\9 #\z))
(assert-false  (tn) (char>? #\9 #\~))
(assert-false  (tn) (char>? #\9 #\x7f))
(tn "char>? #\\A")
(assert-true   (tn) (char>? #\A #\x00))
(assert-true   (tn) (char>? #\A #\newline))
(assert-true   (tn) (char>? #\A #\space))
(assert-true   (tn) (char>? #\A #\!))
(assert-true   (tn) (char>? #\A #\0))
(assert-true   (tn) (char>? #\A #\9))
(assert-false  (tn) (char>? #\A #\A))
(assert-false  (tn) (char>? #\A #\B))
(assert-false  (tn) (char>? #\A #\Z))
(assert-false  (tn) (char>? #\A #\_))
(assert-false  (tn) (char>? #\A #\a))
(assert-false  (tn) (char>? #\A #\b))
(assert-false  (tn) (char>? #\A #\z))
(assert-false  (tn) (char>? #\A #\~))
(assert-false  (tn) (char>? #\A #\x7f))
(tn "char>? #\\B")
(assert-true   (tn) (char>? #\B #\x00))
(assert-true   (tn) (char>? #\B #\newline))
(assert-true   (tn) (char>? #\B #\space))
(assert-true   (tn) (char>? #\B #\!))
(assert-true   (tn) (char>? #\B #\0))
(assert-true   (tn) (char>? #\B #\9))
(assert-true   (tn) (char>? #\B #\A))
(assert-false  (tn) (char>? #\B #\B))
(assert-false  (tn) (char>? #\B #\Z))
(assert-false  (tn) (char>? #\B #\_))
(assert-false  (tn) (char>? #\B #\a))
(assert-false  (tn) (char>? #\B #\b))
(assert-false  (tn) (char>? #\B #\z))
(assert-false  (tn) (char>? #\B #\~))
(assert-false  (tn) (char>? #\B #\x7f))
(tn "char>? #\\Z")
(assert-true   (tn) (char>? #\Z #\x00))
(assert-true   (tn) (char>? #\Z #\newline))
(assert-true   (tn) (char>? #\Z #\space))
(assert-true   (tn) (char>? #\Z #\!))
(assert-true   (tn) (char>? #\Z #\0))
(assert-true   (tn) (char>? #\Z #\9))
(assert-true   (tn) (char>? #\Z #\A))
(assert-true   (tn) (char>? #\Z #\B))
(assert-false  (tn) (char>? #\Z #\Z))
(assert-false  (tn) (char>? #\Z #\_))
(assert-false  (tn) (char>? #\Z #\a))
(assert-false  (tn) (char>? #\Z #\b))
(assert-false  (tn) (char>? #\Z #\z))
(assert-false  (tn) (char>? #\Z #\~))
(assert-false  (tn) (char>? #\Z #\x7f))
(tn "char>? #\\_")
(assert-true   (tn) (char>? #\_ #\x00))
(assert-true   (tn) (char>? #\_ #\newline))
(assert-true   (tn) (char>? #\_ #\space))
(assert-true   (tn) (char>? #\_ #\!))
(assert-true   (tn) (char>? #\_ #\0))
(assert-true   (tn) (char>? #\_ #\9))
(assert-true   (tn) (char>? #\_ #\A))
(assert-true   (tn) (char>? #\_ #\B))
(assert-true   (tn) (char>? #\_ #\Z))
(assert-false  (tn) (char>? #\_ #\_))
(assert-false  (tn) (char>? #\_ #\a))
(assert-false  (tn) (char>? #\_ #\b))
(assert-false  (tn) (char>? #\_ #\z))
(assert-false  (tn) (char>? #\_ #\~))
(assert-false  (tn) (char>? #\_ #\x7f))
(tn "char>? #\\a")
(assert-true   (tn) (char>? #\a #\x00))
(assert-true   (tn) (char>? #\a #\newline))
(assert-true   (tn) (char>? #\a #\space))
(assert-true   (tn) (char>? #\a #\!))
(assert-true   (tn) (char>? #\a #\0))
(assert-true   (tn) (char>? #\a #\9))
(assert-true   (tn) (char>? #\a #\A))
(assert-true   (tn) (char>? #\a #\B))
(assert-true   (tn) (char>? #\a #\Z))
(assert-true   (tn) (char>? #\a #\_))
(assert-false  (tn) (char>? #\a #\a))
(assert-false  (tn) (char>? #\a #\b))
(assert-false  (tn) (char>? #\a #\z))
(assert-false  (tn) (char>? #\a #\~))
(assert-false  (tn) (char>? #\a #\x7f))
(tn "char>? #\\b")
(assert-true   (tn) (char>? #\b #\x00))
(assert-true   (tn) (char>? #\b #\newline))
(assert-true   (tn) (char>? #\b #\space))
(assert-true   (tn) (char>? #\b #\!))
(assert-true   (tn) (char>? #\b #\0))
(assert-true   (tn) (char>? #\b #\9))
(assert-true   (tn) (char>? #\b #\A))
(assert-true   (tn) (char>? #\b #\B))
(assert-true   (tn) (char>? #\b #\Z))
(assert-true   (tn) (char>? #\b #\_))
(assert-true   (tn) (char>? #\b #\a))
(assert-false  (tn) (char>? #\b #\b))
(assert-false  (tn) (char>? #\b #\z))
(assert-false  (tn) (char>? #\b #\~))
(assert-false  (tn) (char>? #\b #\x7f))
(tn "char>? #\\z")
(assert-true   (tn) (char>? #\z #\x00))
(assert-true   (tn) (char>? #\z #\newline))
(assert-true   (tn) (char>? #\z #\space))
(assert-true   (tn) (char>? #\z #\!))
(assert-true   (tn) (char>? #\z #\0))
(assert-true   (tn) (char>? #\z #\9))
(assert-true   (tn) (char>? #\z #\A))
(assert-true   (tn) (char>? #\z #\B))
(assert-true   (tn) (char>? #\z #\Z))
(assert-true   (tn) (char>? #\z #\_))
(assert-true   (tn) (char>? #\z #\a))
(assert-true   (tn) (char>? #\z #\b))
(assert-false  (tn) (char>? #\z #\z))
(assert-false  (tn) (char>? #\z #\~))
(assert-false  (tn) (char>? #\z #\x7f))
(tn "char>? #\\~")
(assert-true   (tn) (char>? #\~ #\x00))
(assert-true   (tn) (char>? #\~ #\newline))
(assert-true   (tn) (char>? #\~ #\space))
(assert-true   (tn) (char>? #\~ #\!))
(assert-true   (tn) (char>? #\~ #\0))
(assert-true   (tn) (char>? #\~ #\9))
(assert-true   (tn) (char>? #\~ #\A))
(assert-true   (tn) (char>? #\~ #\B))
(assert-true   (tn) (char>? #\~ #\Z))
(assert-true   (tn) (char>? #\~ #\_))
(assert-true   (tn) (char>? #\~ #\a))
(assert-true   (tn) (char>? #\~ #\b))
(assert-true   (tn) (char>? #\~ #\z))
(assert-false  (tn) (char>? #\~ #\~))
(assert-false  (tn) (char>? #\~ #\x7f))
(tn "char>? #\\x7f")
(assert-true   (tn) (char>? #\x7f #\x00))
(assert-true   (tn) (char>? #\x7f #\newline))
(assert-true   (tn) (char>? #\x7f #\space))
(assert-true   (tn) (char>? #\x7f #\!))
(assert-true   (tn) (char>? #\x7f #\0))
(assert-true   (tn) (char>? #\x7f #\9))
(assert-true   (tn) (char>? #\x7f #\A))
(assert-true   (tn) (char>? #\x7f #\B))
(assert-true   (tn) (char>? #\x7f #\Z))
(assert-true   (tn) (char>? #\x7f #\_))
(assert-true   (tn) (char>? #\x7f #\a))
(assert-true   (tn) (char>? #\x7f #\b))
(assert-true   (tn) (char>? #\x7f #\z))
(assert-true   (tn) (char>? #\x7f #\~))
(assert-false  (tn) (char>? #\x7f #\x7f))

;; char<=?
(tn "char<=? #\\x00")
(assert-true   (tn) (char<=? #\x00 #\x00))
(assert-true   (tn) (char<=? #\x00 #\newline))
(assert-true   (tn) (char<=? #\x00 #\space))
(assert-true   (tn) (char<=? #\x00 #\!))
(assert-true   (tn) (char<=? #\x00 #\0))
(assert-true   (tn) (char<=? #\x00 #\9))
(assert-true   (tn) (char<=? #\x00 #\A))
(assert-true   (tn) (char<=? #\x00 #\B))
(assert-true   (tn) (char<=? #\x00 #\Z))
(assert-true   (tn) (char<=? #\x00 #\_))
(assert-true   (tn) (char<=? #\x00 #\a))
(assert-true   (tn) (char<=? #\x00 #\b))
(assert-true   (tn) (char<=? #\x00 #\z))
(assert-true   (tn) (char<=? #\x00 #\~))
(assert-true   (tn) (char<=? #\x00 #\x7f))
(tn "char<=? #\\newline")
(assert-false  (tn) (char<=? #\newline #\x00))
(assert-true   (tn) (char<=? #\newline #\newline))
(assert-true   (tn) (char<=? #\newline #\space))
(assert-true   (tn) (char<=? #\newline #\!))
(assert-true   (tn) (char<=? #\newline #\0))
(assert-true   (tn) (char<=? #\newline #\9))
(assert-true   (tn) (char<=? #\newline #\A))
(assert-true   (tn) (char<=? #\newline #\B))
(assert-true   (tn) (char<=? #\newline #\Z))
(assert-true   (tn) (char<=? #\newline #\_))
(assert-true   (tn) (char<=? #\newline #\a))
(assert-true   (tn) (char<=? #\newline #\b))
(assert-true   (tn) (char<=? #\newline #\z))
(assert-true   (tn) (char<=? #\newline #\~))
(assert-true   (tn) (char<=? #\newline #\x7f))
(tn "char<=? #\\space")
(assert-false  (tn) (char<=? #\space #\x00))
(assert-false  (tn) (char<=? #\space #\newline))
(assert-true   (tn) (char<=? #\space #\space))
(assert-true   (tn) (char<=? #\space #\!))
(assert-true   (tn) (char<=? #\space #\0))
(assert-true   (tn) (char<=? #\space #\9))
(assert-true   (tn) (char<=? #\space #\A))
(assert-true   (tn) (char<=? #\space #\B))
(assert-true   (tn) (char<=? #\space #\Z))
(assert-true   (tn) (char<=? #\space #\_))
(assert-true   (tn) (char<=? #\space #\a))
(assert-true   (tn) (char<=? #\space #\b))
(assert-true   (tn) (char<=? #\space #\z))
(assert-true   (tn) (char<=? #\space #\~))
(assert-true   (tn) (char<=? #\space #\x7f))
(tn "char<=? #\\!")
(assert-false  (tn) (char<=? #\! #\x00))
(assert-false  (tn) (char<=? #\! #\newline))
(assert-false  (tn) (char<=? #\! #\space))
(assert-true   (tn) (char<=? #\! #\!))
(assert-true   (tn) (char<=? #\! #\0))
(assert-true   (tn) (char<=? #\! #\9))
(assert-true   (tn) (char<=? #\! #\A))
(assert-true   (tn) (char<=? #\! #\B))
(assert-true   (tn) (char<=? #\! #\Z))
(assert-true   (tn) (char<=? #\! #\_))
(assert-true   (tn) (char<=? #\! #\a))
(assert-true   (tn) (char<=? #\! #\b))
(assert-true   (tn) (char<=? #\! #\z))
(assert-true   (tn) (char<=? #\! #\~))
(assert-true   (tn) (char<=? #\! #\x7f))
(tn "char<=? #\\0")
(assert-false  (tn) (char<=? #\0 #\x00))
(assert-false  (tn) (char<=? #\0 #\newline))
(assert-false  (tn) (char<=? #\0 #\space))
(assert-false  (tn) (char<=? #\0 #\!))
(assert-true   (tn) (char<=? #\0 #\0))
(assert-true   (tn) (char<=? #\0 #\9))
(assert-true   (tn) (char<=? #\0 #\A))
(assert-true   (tn) (char<=? #\0 #\B))
(assert-true   (tn) (char<=? #\0 #\Z))
(assert-true   (tn) (char<=? #\0 #\_))
(assert-true   (tn) (char<=? #\0 #\a))
(assert-true   (tn) (char<=? #\0 #\b))
(assert-true   (tn) (char<=? #\0 #\z))
(assert-true   (tn) (char<=? #\0 #\~))
(assert-true   (tn) (char<=? #\0 #\x7f))
(tn "char<=? #\\9")
(assert-false  (tn) (char<=? #\9 #\x00))
(assert-false  (tn) (char<=? #\9 #\newline))
(assert-false  (tn) (char<=? #\9 #\space))
(assert-false  (tn) (char<=? #\9 #\!))
(assert-false  (tn) (char<=? #\9 #\0))
(assert-true   (tn) (char<=? #\9 #\9))
(assert-true   (tn) (char<=? #\9 #\A))
(assert-true   (tn) (char<=? #\9 #\B))
(assert-true   (tn) (char<=? #\9 #\Z))
(assert-true   (tn) (char<=? #\9 #\_))
(assert-true   (tn) (char<=? #\9 #\a))
(assert-true   (tn) (char<=? #\9 #\b))
(assert-true   (tn) (char<=? #\9 #\z))
(assert-true   (tn) (char<=? #\9 #\~))
(assert-true   (tn) (char<=? #\9 #\x7f))
(tn "char<=? #\\A")
(assert-false  (tn) (char<=? #\A #\x00))
(assert-false  (tn) (char<=? #\A #\newline))
(assert-false  (tn) (char<=? #\A #\space))
(assert-false  (tn) (char<=? #\A #\!))
(assert-false  (tn) (char<=? #\A #\0))
(assert-false  (tn) (char<=? #\A #\9))
(assert-true   (tn) (char<=? #\A #\A))
(assert-true   (tn) (char<=? #\A #\B))
(assert-true   (tn) (char<=? #\A #\Z))
(assert-true   (tn) (char<=? #\A #\_))
(assert-true   (tn) (char<=? #\A #\a))
(assert-true   (tn) (char<=? #\A #\b))
(assert-true   (tn) (char<=? #\A #\z))
(assert-true   (tn) (char<=? #\A #\~))
(assert-true   (tn) (char<=? #\A #\x7f))
(tn "char<=? #\\B")
(assert-false  (tn) (char<=? #\B #\x00))
(assert-false  (tn) (char<=? #\B #\newline))
(assert-false  (tn) (char<=? #\B #\space))
(assert-false  (tn) (char<=? #\B #\!))
(assert-false  (tn) (char<=? #\B #\0))
(assert-false  (tn) (char<=? #\B #\9))
(assert-false  (tn) (char<=? #\B #\A))
(assert-true   (tn) (char<=? #\B #\B))
(assert-true   (tn) (char<=? #\B #\Z))
(assert-true   (tn) (char<=? #\B #\_))
(assert-true   (tn) (char<=? #\B #\a))
(assert-true   (tn) (char<=? #\B #\b))
(assert-true   (tn) (char<=? #\B #\z))
(assert-true   (tn) (char<=? #\B #\~))
(assert-true   (tn) (char<=? #\B #\x7f))
(tn "char<=? #\\Z")
(assert-false  (tn) (char<=? #\Z #\x00))
(assert-false  (tn) (char<=? #\Z #\newline))
(assert-false  (tn) (char<=? #\Z #\space))
(assert-false  (tn) (char<=? #\Z #\!))
(assert-false  (tn) (char<=? #\Z #\0))
(assert-false  (tn) (char<=? #\Z #\9))
(assert-false  (tn) (char<=? #\Z #\A))
(assert-false  (tn) (char<=? #\Z #\B))
(assert-true   (tn) (char<=? #\Z #\Z))
(assert-true   (tn) (char<=? #\Z #\_))
(assert-true   (tn) (char<=? #\Z #\a))
(assert-true   (tn) (char<=? #\Z #\b))
(assert-true   (tn) (char<=? #\Z #\z))
(assert-true   (tn) (char<=? #\Z #\~))
(assert-true   (tn) (char<=? #\Z #\x7f))
(tn "char<=? #\\_")
(assert-false  (tn) (char<=? #\_ #\x00))
(assert-false  (tn) (char<=? #\_ #\newline))
(assert-false  (tn) (char<=? #\_ #\space))
(assert-false  (tn) (char<=? #\_ #\!))
(assert-false  (tn) (char<=? #\_ #\0))
(assert-false  (tn) (char<=? #\_ #\9))
(assert-false  (tn) (char<=? #\_ #\A))
(assert-false  (tn) (char<=? #\_ #\B))
(assert-false  (tn) (char<=? #\_ #\Z))
(assert-true   (tn) (char<=? #\_ #\_))
(assert-true   (tn) (char<=? #\_ #\a))
(assert-true   (tn) (char<=? #\_ #\b))
(assert-true   (tn) (char<=? #\_ #\z))
(assert-true   (tn) (char<=? #\_ #\~))
(assert-true   (tn) (char<=? #\_ #\x7f))
(tn "char<=? #\\a")
(assert-false  (tn) (char<=? #\a #\x00))
(assert-false  (tn) (char<=? #\a #\newline))
(assert-false  (tn) (char<=? #\a #\space))
(assert-false  (tn) (char<=? #\a #\!))
(assert-false  (tn) (char<=? #\a #\0))
(assert-false  (tn) (char<=? #\a #\9))
(assert-false  (tn) (char<=? #\a #\A))
(assert-false  (tn) (char<=? #\a #\B))
(assert-false  (tn) (char<=? #\a #\Z))
(assert-false  (tn) (char<=? #\a #\_))
(assert-true   (tn) (char<=? #\a #\a))
(assert-true   (tn) (char<=? #\a #\b))
(assert-true   (tn) (char<=? #\a #\z))
(assert-true   (tn) (char<=? #\a #\~))
(assert-true   (tn) (char<=? #\a #\x7f))
(tn "char<=? #\\b")
(assert-false  (tn) (char<=? #\b #\x00))
(assert-false  (tn) (char<=? #\b #\newline))
(assert-false  (tn) (char<=? #\b #\space))
(assert-false  (tn) (char<=? #\b #\!))
(assert-false  (tn) (char<=? #\b #\0))
(assert-false  (tn) (char<=? #\b #\9))
(assert-false  (tn) (char<=? #\b #\A))
(assert-false  (tn) (char<=? #\b #\B))
(assert-false  (tn) (char<=? #\b #\Z))
(assert-false  (tn) (char<=? #\b #\_))
(assert-false  (tn) (char<=? #\b #\a))
(assert-true   (tn) (char<=? #\b #\b))
(assert-true   (tn) (char<=? #\b #\z))
(assert-true   (tn) (char<=? #\b #\~))
(assert-true   (tn) (char<=? #\b #\x7f))
(tn "char<=? #\\z")
(assert-false  (tn) (char<=? #\z #\x00))
(assert-false  (tn) (char<=? #\z #\newline))
(assert-false  (tn) (char<=? #\z #\space))
(assert-false  (tn) (char<=? #\z #\!))
(assert-false  (tn) (char<=? #\z #\0))
(assert-false  (tn) (char<=? #\z #\9))
(assert-false  (tn) (char<=? #\z #\A))
(assert-false  (tn) (char<=? #\z #\B))
(assert-false  (tn) (char<=? #\z #\Z))
(assert-false  (tn) (char<=? #\z #\_))
(assert-false  (tn) (char<=? #\z #\a))
(assert-false  (tn) (char<=? #\z #\b))
(assert-true   (tn) (char<=? #\z #\z))
(assert-true   (tn) (char<=? #\z #\~))
(assert-true   (tn) (char<=? #\z #\x7f))
(tn "char<=? #\\~")
(assert-false  (tn) (char<=? #\~ #\x00))
(assert-false  (tn) (char<=? #\~ #\newline))
(assert-false  (tn) (char<=? #\~ #\space))
(assert-false  (tn) (char<=? #\~ #\!))
(assert-false  (tn) (char<=? #\~ #\0))
(assert-false  (tn) (char<=? #\~ #\9))
(assert-false  (tn) (char<=? #\~ #\A))
(assert-false  (tn) (char<=? #\~ #\B))
(assert-false  (tn) (char<=? #\~ #\Z))
(assert-false  (tn) (char<=? #\~ #\_))
(assert-false  (tn) (char<=? #\~ #\a))
(assert-false  (tn) (char<=? #\~ #\b))
(assert-false  (tn) (char<=? #\~ #\z))
(assert-true   (tn) (char<=? #\~ #\~))
(assert-true   (tn) (char<=? #\~ #\x7f))
(tn "char<=? #\\x7f")
(assert-false  (tn) (char<=? #\x7f #\x00))
(assert-false  (tn) (char<=? #\x7f #\newline))
(assert-false  (tn) (char<=? #\x7f #\space))
(assert-false  (tn) (char<=? #\x7f #\!))
(assert-false  (tn) (char<=? #\x7f #\0))
(assert-false  (tn) (char<=? #\x7f #\9))
(assert-false  (tn) (char<=? #\x7f #\A))
(assert-false  (tn) (char<=? #\x7f #\B))
(assert-false  (tn) (char<=? #\x7f #\Z))
(assert-false  (tn) (char<=? #\x7f #\_))
(assert-false  (tn) (char<=? #\x7f #\a))
(assert-false  (tn) (char<=? #\x7f #\b))
(assert-false  (tn) (char<=? #\x7f #\z))
(assert-false  (tn) (char<=? #\x7f #\~))
(assert-true   (tn) (char<=? #\x7f #\x7f))

;; char>=?
(tn "char>=? #\\x00")
(assert-true   (tn) (char>=? #\x00 #\x00))
(assert-false  (tn) (char>=? #\x00 #\newline))
(assert-false  (tn) (char>=? #\x00 #\space))
(assert-false  (tn) (char>=? #\x00 #\!))
(assert-false  (tn) (char>=? #\x00 #\0))
(assert-false  (tn) (char>=? #\x00 #\9))
(assert-false  (tn) (char>=? #\x00 #\A))
(assert-false  (tn) (char>=? #\x00 #\B))
(assert-false  (tn) (char>=? #\x00 #\Z))
(assert-false  (tn) (char>=? #\x00 #\_))
(assert-false  (tn) (char>=? #\x00 #\a))
(assert-false  (tn) (char>=? #\x00 #\b))
(assert-false  (tn) (char>=? #\x00 #\z))
(assert-false  (tn) (char>=? #\x00 #\~))
(assert-false  (tn) (char>=? #\x00 #\x7f))
(tn "char>=? #\\newline")
(assert-true   (tn) (char>=? #\newline #\x00))
(assert-true   (tn) (char>=? #\newline #\newline))
(assert-false  (tn) (char>=? #\newline #\space))
(assert-false  (tn) (char>=? #\newline #\!))
(assert-false  (tn) (char>=? #\newline #\0))
(assert-false  (tn) (char>=? #\newline #\9))
(assert-false  (tn) (char>=? #\newline #\A))
(assert-false  (tn) (char>=? #\newline #\B))
(assert-false  (tn) (char>=? #\newline #\Z))
(assert-false  (tn) (char>=? #\newline #\_))
(assert-false  (tn) (char>=? #\newline #\a))
(assert-false  (tn) (char>=? #\newline #\b))
(assert-false  (tn) (char>=? #\newline #\z))
(assert-false  (tn) (char>=? #\newline #\~))
(assert-false  (tn) (char>=? #\newline #\x7f))
(tn "char>=? #\\space")
(assert-true   (tn) (char>=? #\space #\x00))
(assert-true   (tn) (char>=? #\space #\newline))
(assert-true   (tn) (char>=? #\space #\space))
(assert-false  (tn) (char>=? #\space #\!))
(assert-false  (tn) (char>=? #\space #\0))
(assert-false  (tn) (char>=? #\space #\9))
(assert-false  (tn) (char>=? #\space #\A))
(assert-false  (tn) (char>=? #\space #\B))
(assert-false  (tn) (char>=? #\space #\Z))
(assert-false  (tn) (char>=? #\space #\_))
(assert-false  (tn) (char>=? #\space #\a))
(assert-false  (tn) (char>=? #\space #\b))
(assert-false  (tn) (char>=? #\space #\z))
(assert-false  (tn) (char>=? #\space #\~))
(assert-false  (tn) (char>=? #\space #\x7f))
(tn "char>=? #\\!")
(assert-true   (tn) (char>=? #\! #\x00))
(assert-true   (tn) (char>=? #\! #\newline))
(assert-true   (tn) (char>=? #\! #\space))
(assert-true   (tn) (char>=? #\! #\!))
(assert-false  (tn) (char>=? #\! #\0))
(assert-false  (tn) (char>=? #\! #\9))
(assert-false  (tn) (char>=? #\! #\A))
(assert-false  (tn) (char>=? #\! #\B))
(assert-false  (tn) (char>=? #\! #\Z))
(assert-false  (tn) (char>=? #\! #\_))
(assert-false  (tn) (char>=? #\! #\a))
(assert-false  (tn) (char>=? #\! #\b))
(assert-false  (tn) (char>=? #\! #\z))
(assert-false  (tn) (char>=? #\! #\~))
(assert-false  (tn) (char>=? #\! #\x7f))
(tn "char>=? #\\0")
(assert-true   (tn) (char>=? #\0 #\x00))
(assert-true   (tn) (char>=? #\0 #\newline))
(assert-true   (tn) (char>=? #\0 #\space))
(assert-true   (tn) (char>=? #\0 #\!))
(assert-true   (tn) (char>=? #\0 #\0))
(assert-false  (tn) (char>=? #\0 #\9))
(assert-false  (tn) (char>=? #\0 #\A))
(assert-false  (tn) (char>=? #\0 #\B))
(assert-false  (tn) (char>=? #\0 #\Z))
(assert-false  (tn) (char>=? #\0 #\_))
(assert-false  (tn) (char>=? #\0 #\a))
(assert-false  (tn) (char>=? #\0 #\b))
(assert-false  (tn) (char>=? #\0 #\z))
(assert-false  (tn) (char>=? #\0 #\~))
(assert-false  (tn) (char>=? #\0 #\x7f))
(tn "char>=? #\\9")
(assert-true   (tn) (char>=? #\9 #\x00))
(assert-true   (tn) (char>=? #\9 #\newline))
(assert-true   (tn) (char>=? #\9 #\space))
(assert-true   (tn) (char>=? #\9 #\!))
(assert-true   (tn) (char>=? #\9 #\0))
(assert-true   (tn) (char>=? #\9 #\9))
(assert-false  (tn) (char>=? #\9 #\A))
(assert-false  (tn) (char>=? #\9 #\B))
(assert-false  (tn) (char>=? #\9 #\Z))
(assert-false  (tn) (char>=? #\9 #\_))
(assert-false  (tn) (char>=? #\9 #\a))
(assert-false  (tn) (char>=? #\9 #\b))
(assert-false  (tn) (char>=? #\9 #\z))
(assert-false  (tn) (char>=? #\9 #\~))
(assert-false  (tn) (char>=? #\9 #\x7f))
(tn "char>=? #\\A")
(assert-true   (tn) (char>=? #\A #\x00))
(assert-true   (tn) (char>=? #\A #\newline))
(assert-true   (tn) (char>=? #\A #\space))
(assert-true   (tn) (char>=? #\A #\!))
(assert-true   (tn) (char>=? #\A #\0))
(assert-true   (tn) (char>=? #\A #\9))
(assert-true   (tn) (char>=? #\A #\A))
(assert-false  (tn) (char>=? #\A #\B))
(assert-false  (tn) (char>=? #\A #\Z))
(assert-false  (tn) (char>=? #\A #\_))
(assert-false  (tn) (char>=? #\A #\a))
(assert-false  (tn) (char>=? #\A #\b))
(assert-false  (tn) (char>=? #\A #\z))
(assert-false  (tn) (char>=? #\A #\~))
(assert-false  (tn) (char>=? #\A #\x7f))
(tn "char>=? #\\B")
(assert-true   (tn) (char>=? #\B #\x00))
(assert-true   (tn) (char>=? #\B #\newline))
(assert-true   (tn) (char>=? #\B #\space))
(assert-true   (tn) (char>=? #\B #\!))
(assert-true   (tn) (char>=? #\B #\0))
(assert-true   (tn) (char>=? #\B #\9))
(assert-true   (tn) (char>=? #\B #\A))
(assert-true   (tn) (char>=? #\B #\B))
(assert-false  (tn) (char>=? #\B #\Z))
(assert-false  (tn) (char>=? #\B #\_))
(assert-false  (tn) (char>=? #\B #\a))
(assert-false  (tn) (char>=? #\B #\b))
(assert-false  (tn) (char>=? #\B #\z))
(assert-false  (tn) (char>=? #\B #\~))
(assert-false  (tn) (char>=? #\B #\x7f))
(tn "char>=? #\\Z")
(assert-true   (tn) (char>=? #\Z #\x00))
(assert-true   (tn) (char>=? #\Z #\newline))
(assert-true   (tn) (char>=? #\Z #\space))
(assert-true   (tn) (char>=? #\Z #\!))
(assert-true   (tn) (char>=? #\Z #\0))
(assert-true   (tn) (char>=? #\Z #\9))
(assert-true   (tn) (char>=? #\Z #\A))
(assert-true   (tn) (char>=? #\Z #\B))
(assert-true   (tn) (char>=? #\Z #\Z))
(assert-false  (tn) (char>=? #\Z #\_))
(assert-false  (tn) (char>=? #\Z #\a))
(assert-false  (tn) (char>=? #\Z #\b))
(assert-false  (tn) (char>=? #\Z #\z))
(assert-false  (tn) (char>=? #\Z #\~))
(assert-false  (tn) (char>=? #\Z #\x7f))
(tn "char>=? #\\_")
(assert-true   (tn) (char>=? #\_ #\x00))
(assert-true   (tn) (char>=? #\_ #\newline))
(assert-true   (tn) (char>=? #\_ #\space))
(assert-true   (tn) (char>=? #\_ #\!))
(assert-true   (tn) (char>=? #\_ #\0))
(assert-true   (tn) (char>=? #\_ #\9))
(assert-true   (tn) (char>=? #\_ #\A))
(assert-true   (tn) (char>=? #\_ #\B))
(assert-true   (tn) (char>=? #\_ #\Z))
(assert-true   (tn) (char>=? #\_ #\_))
(assert-false  (tn) (char>=? #\_ #\a))
(assert-false  (tn) (char>=? #\_ #\b))
(assert-false  (tn) (char>=? #\_ #\z))
(assert-false  (tn) (char>=? #\_ #\~))
(assert-false  (tn) (char>=? #\_ #\x7f))
(tn "char>=? #\\a")
(assert-true   (tn) (char>=? #\a #\x00))
(assert-true   (tn) (char>=? #\a #\newline))
(assert-true   (tn) (char>=? #\a #\space))
(assert-true   (tn) (char>=? #\a #\!))
(assert-true   (tn) (char>=? #\a #\0))
(assert-true   (tn) (char>=? #\a #\9))
(assert-true   (tn) (char>=? #\a #\A))
(assert-true   (tn) (char>=? #\a #\B))
(assert-true   (tn) (char>=? #\a #\Z))
(assert-true   (tn) (char>=? #\a #\_))
(assert-true   (tn) (char>=? #\a #\a))
(assert-false  (tn) (char>=? #\a #\b))
(assert-false  (tn) (char>=? #\a #\z))
(assert-false  (tn) (char>=? #\a #\~))
(assert-false  (tn) (char>=? #\a #\x7f))
(tn "char>=? #\\b")
(assert-true   (tn) (char>=? #\b #\x00))
(assert-true   (tn) (char>=? #\b #\newline))
(assert-true   (tn) (char>=? #\b #\space))
(assert-true   (tn) (char>=? #\b #\!))
(assert-true   (tn) (char>=? #\b #\0))
(assert-true   (tn) (char>=? #\b #\9))
(assert-true   (tn) (char>=? #\b #\A))
(assert-true   (tn) (char>=? #\b #\B))
(assert-true   (tn) (char>=? #\b #\Z))
(assert-true   (tn) (char>=? #\b #\_))
(assert-true   (tn) (char>=? #\b #\a))
(assert-true   (tn) (char>=? #\b #\b))
(assert-false  (tn) (char>=? #\b #\z))
(assert-false  (tn) (char>=? #\b #\~))
(assert-false  (tn) (char>=? #\b #\x7f))
(tn "char>=? #\\z")
(assert-true   (tn) (char>=? #\z #\x00))
(assert-true   (tn) (char>=? #\z #\newline))
(assert-true   (tn) (char>=? #\z #\space))
(assert-true   (tn) (char>=? #\z #\!))
(assert-true   (tn) (char>=? #\z #\0))
(assert-true   (tn) (char>=? #\z #\9))
(assert-true   (tn) (char>=? #\z #\A))
(assert-true   (tn) (char>=? #\z #\B))
(assert-true   (tn) (char>=? #\z #\Z))
(assert-true   (tn) (char>=? #\z #\_))
(assert-true   (tn) (char>=? #\z #\a))
(assert-true   (tn) (char>=? #\z #\b))
(assert-true   (tn) (char>=? #\z #\z))
(assert-false  (tn) (char>=? #\z #\~))
(assert-false  (tn) (char>=? #\z #\x7f))
(tn "char>=? #\\~")
(assert-true   (tn) (char>=? #\~ #\x00))
(assert-true   (tn) (char>=? #\~ #\newline))
(assert-true   (tn) (char>=? #\~ #\space))
(assert-true   (tn) (char>=? #\~ #\!))
(assert-true   (tn) (char>=? #\~ #\0))
(assert-true   (tn) (char>=? #\~ #\9))
(assert-true   (tn) (char>=? #\~ #\A))
(assert-true   (tn) (char>=? #\~ #\B))
(assert-true   (tn) (char>=? #\~ #\Z))
(assert-true   (tn) (char>=? #\~ #\_))
(assert-true   (tn) (char>=? #\~ #\a))
(assert-true   (tn) (char>=? #\~ #\b))
(assert-true   (tn) (char>=? #\~ #\z))
(assert-true   (tn) (char>=? #\~ #\~))
(assert-false  (tn) (char>=? #\~ #\x7f))
(tn "char>=? #\\x7f")
(assert-true   (tn) (char>=? #\x7f #\x00))
(assert-true   (tn) (char>=? #\x7f #\newline))
(assert-true   (tn) (char>=? #\x7f #\space))
(assert-true   (tn) (char>=? #\x7f #\!))
(assert-true   (tn) (char>=? #\x7f #\0))
(assert-true   (tn) (char>=? #\x7f #\9))
(assert-true   (tn) (char>=? #\x7f #\A))
(assert-true   (tn) (char>=? #\x7f #\B))
(assert-true   (tn) (char>=? #\x7f #\Z))
(assert-true   (tn) (char>=? #\x7f #\_))
(assert-true   (tn) (char>=? #\x7f #\a))
(assert-true   (tn) (char>=? #\x7f #\b))
(assert-true   (tn) (char>=? #\x7f #\z))
(assert-true   (tn) (char>=? #\x7f #\~))
(assert-true   (tn) (char>=? #\x7f #\x7f))

;; char-ci=?
(tn "char-ci=? #\\x00")
(assert-true   (tn) (char-ci=? #\x00 #\x00))
(assert-false  (tn) (char-ci=? #\x00 #\newline))
(assert-false  (tn) (char-ci=? #\x00 #\space))
(assert-false  (tn) (char-ci=? #\x00 #\!))
(assert-false  (tn) (char-ci=? #\x00 #\0))
(assert-false  (tn) (char-ci=? #\x00 #\9))
(assert-false  (tn) (char-ci=? #\x00 #\A))
(assert-false  (tn) (char-ci=? #\x00 #\B))
(assert-false  (tn) (char-ci=? #\x00 #\Z))
(assert-false  (tn) (char-ci=? #\x00 #\_))
(assert-false  (tn) (char-ci=? #\x00 #\a))
(assert-false  (tn) (char-ci=? #\x00 #\b))
(assert-false  (tn) (char-ci=? #\x00 #\z))
(assert-false  (tn) (char-ci=? #\x00 #\~))
(assert-false  (tn) (char-ci=? #\x00 #\x7f))
(tn "char-ci=? #\\newline")
(assert-false  (tn) (char-ci=? #\newline #\x00))
(assert-true   (tn) (char-ci=? #\newline #\newline))
(assert-false  (tn) (char-ci=? #\newline #\space))
(assert-false  (tn) (char-ci=? #\newline #\!))
(assert-false  (tn) (char-ci=? #\newline #\0))
(assert-false  (tn) (char-ci=? #\newline #\9))
(assert-false  (tn) (char-ci=? #\newline #\A))
(assert-false  (tn) (char-ci=? #\newline #\B))
(assert-false  (tn) (char-ci=? #\newline #\Z))
(assert-false  (tn) (char-ci=? #\newline #\_))
(assert-false  (tn) (char-ci=? #\newline #\a))
(assert-false  (tn) (char-ci=? #\newline #\b))
(assert-false  (tn) (char-ci=? #\newline #\z))
(assert-false  (tn) (char-ci=? #\newline #\~))
(assert-false  (tn) (char-ci=? #\newline #\x7f))
(tn "char-ci=? #\\space")
(assert-false  (tn) (char-ci=? #\space #\x00))
(assert-false  (tn) (char-ci=? #\space #\newline))
(assert-true   (tn) (char-ci=? #\space #\space))
(assert-false  (tn) (char-ci=? #\space #\!))
(assert-false  (tn) (char-ci=? #\space #\0))
(assert-false  (tn) (char-ci=? #\space #\9))
(assert-false  (tn) (char-ci=? #\space #\A))
(assert-false  (tn) (char-ci=? #\space #\B))
(assert-false  (tn) (char-ci=? #\space #\Z))
(assert-false  (tn) (char-ci=? #\space #\_))
(assert-false  (tn) (char-ci=? #\space #\a))
(assert-false  (tn) (char-ci=? #\space #\b))
(assert-false  (tn) (char-ci=? #\space #\z))
(assert-false  (tn) (char-ci=? #\space #\~))
(assert-false  (tn) (char-ci=? #\space #\x7f))
(tn "char-ci=? #\\!")
(assert-false  (tn) (char-ci=? #\! #\x00))
(assert-false  (tn) (char-ci=? #\! #\newline))
(assert-false  (tn) (char-ci=? #\! #\space))
(assert-true   (tn) (char-ci=? #\! #\!))
(assert-false  (tn) (char-ci=? #\! #\0))
(assert-false  (tn) (char-ci=? #\! #\9))
(assert-false  (tn) (char-ci=? #\! #\A))
(assert-false  (tn) (char-ci=? #\! #\B))
(assert-false  (tn) (char-ci=? #\! #\Z))
(assert-false  (tn) (char-ci=? #\! #\_))
(assert-false  (tn) (char-ci=? #\! #\a))
(assert-false  (tn) (char-ci=? #\! #\b))
(assert-false  (tn) (char-ci=? #\! #\z))
(assert-false  (tn) (char-ci=? #\! #\~))
(assert-false  (tn) (char-ci=? #\! #\x7f))
(tn "char-ci=? #\\0")
(assert-false  (tn) (char-ci=? #\0 #\x00))
(assert-false  (tn) (char-ci=? #\0 #\newline))
(assert-false  (tn) (char-ci=? #\0 #\space))
(assert-false  (tn) (char-ci=? #\0 #\!))
(assert-true   (tn) (char-ci=? #\0 #\0))
(assert-false  (tn) (char-ci=? #\0 #\9))
(assert-false  (tn) (char-ci=? #\0 #\A))
(assert-false  (tn) (char-ci=? #\0 #\B))
(assert-false  (tn) (char-ci=? #\0 #\Z))
(assert-false  (tn) (char-ci=? #\0 #\_))
(assert-false  (tn) (char-ci=? #\0 #\a))
(assert-false  (tn) (char-ci=? #\0 #\b))
(assert-false  (tn) (char-ci=? #\0 #\z))
(assert-false  (tn) (char-ci=? #\0 #\~))
(assert-false  (tn) (char-ci=? #\0 #\x7f))
(tn "char-ci=? #\\9")
(assert-false  (tn) (char-ci=? #\9 #\x00))
(assert-false  (tn) (char-ci=? #\9 #\newline))
(assert-false  (tn) (char-ci=? #\9 #\space))
(assert-false  (tn) (char-ci=? #\9 #\!))
(assert-false  (tn) (char-ci=? #\9 #\0))
(assert-true   (tn) (char-ci=? #\9 #\9))
(assert-false  (tn) (char-ci=? #\9 #\A))
(assert-false  (tn) (char-ci=? #\9 #\B))
(assert-false  (tn) (char-ci=? #\9 #\Z))
(assert-false  (tn) (char-ci=? #\9 #\_))
(assert-false  (tn) (char-ci=? #\9 #\a))
(assert-false  (tn) (char-ci=? #\9 #\b))
(assert-false  (tn) (char-ci=? #\9 #\z))
(assert-false  (tn) (char-ci=? #\9 #\~))
(assert-false  (tn) (char-ci=? #\9 #\x7f))
(tn "char-ci=? #\\A")
(assert-false  (tn) (char-ci=? #\A #\x00))
(assert-false  (tn) (char-ci=? #\A #\newline))
(assert-false  (tn) (char-ci=? #\A #\space))
(assert-false  (tn) (char-ci=? #\A #\!))
(assert-false  (tn) (char-ci=? #\A #\0))
(assert-false  (tn) (char-ci=? #\A #\9))
(assert-true   (tn) (char-ci=? #\A #\A))
(assert-false  (tn) (char-ci=? #\A #\B))
(assert-false  (tn) (char-ci=? #\A #\Z))
(assert-false  (tn) (char-ci=? #\A #\_))
(assert-true   (tn) (char-ci=? #\A #\a))
(assert-false  (tn) (char-ci=? #\A #\b))
(assert-false  (tn) (char-ci=? #\A #\z))
(assert-false  (tn) (char-ci=? #\A #\~))
(assert-false  (tn) (char-ci=? #\A #\x7f))
(tn "char-ci=? #\\B")
(assert-false  (tn) (char-ci=? #\B #\x00))
(assert-false  (tn) (char-ci=? #\B #\newline))
(assert-false  (tn) (char-ci=? #\B #\space))
(assert-false  (tn) (char-ci=? #\B #\!))
(assert-false  (tn) (char-ci=? #\B #\0))
(assert-false  (tn) (char-ci=? #\B #\9))
(assert-false  (tn) (char-ci=? #\B #\A))
(assert-true   (tn) (char-ci=? #\B #\B))
(assert-false  (tn) (char-ci=? #\B #\Z))
(assert-false  (tn) (char-ci=? #\B #\_))
(assert-false  (tn) (char-ci=? #\B #\a))
(assert-true   (tn) (char-ci=? #\B #\b))
(assert-false  (tn) (char-ci=? #\B #\z))
(assert-false  (tn) (char-ci=? #\B #\~))
(assert-false  (tn) (char-ci=? #\B #\x7f))
(tn "char-ci=? #\\Z")
(assert-false  (tn) (char-ci=? #\Z #\x00))
(assert-false  (tn) (char-ci=? #\Z #\newline))
(assert-false  (tn) (char-ci=? #\Z #\space))
(assert-false  (tn) (char-ci=? #\Z #\!))
(assert-false  (tn) (char-ci=? #\Z #\0))
(assert-false  (tn) (char-ci=? #\Z #\9))
(assert-false  (tn) (char-ci=? #\Z #\A))
(assert-false  (tn) (char-ci=? #\Z #\B))
(assert-true   (tn) (char-ci=? #\Z #\Z))
(assert-false  (tn) (char-ci=? #\Z #\_))
(assert-false  (tn) (char-ci=? #\Z #\a))
(assert-false  (tn) (char-ci=? #\Z #\b))
(assert-true   (tn) (char-ci=? #\Z #\z))
(assert-false  (tn) (char-ci=? #\Z #\~))
(assert-false  (tn) (char-ci=? #\Z #\x7f))
(tn "char-ci=? #\\_")
(assert-false  (tn) (char-ci=? #\_ #\x00))
(assert-false  (tn) (char-ci=? #\_ #\newline))
(assert-false  (tn) (char-ci=? #\_ #\space))
(assert-false  (tn) (char-ci=? #\_ #\!))
(assert-false  (tn) (char-ci=? #\_ #\0))
(assert-false  (tn) (char-ci=? #\_ #\9))
(assert-false  (tn) (char-ci=? #\_ #\A))
(assert-false  (tn) (char-ci=? #\_ #\B))
(assert-false  (tn) (char-ci=? #\_ #\Z))
(assert-true   (tn) (char-ci=? #\_ #\_))
(assert-false  (tn) (char-ci=? #\_ #\a))
(assert-false  (tn) (char-ci=? #\_ #\b))
(assert-false  (tn) (char-ci=? #\_ #\z))
(assert-false  (tn) (char-ci=? #\_ #\~))
(assert-false  (tn) (char-ci=? #\_ #\x7f))
(tn "char-ci=? #\\a")
(assert-false  (tn) (char-ci=? #\a #\x00))
(assert-false  (tn) (char-ci=? #\a #\newline))
(assert-false  (tn) (char-ci=? #\a #\space))
(assert-false  (tn) (char-ci=? #\a #\!))
(assert-false  (tn) (char-ci=? #\a #\0))
(assert-false  (tn) (char-ci=? #\a #\9))
(assert-true   (tn) (char-ci=? #\a #\A))
(assert-false  (tn) (char-ci=? #\a #\B))
(assert-false  (tn) (char-ci=? #\a #\Z))
(assert-false  (tn) (char-ci=? #\a #\_))
(assert-true   (tn) (char-ci=? #\a #\a))
(assert-false  (tn) (char-ci=? #\a #\b))
(assert-false  (tn) (char-ci=? #\a #\z))
(assert-false  (tn) (char-ci=? #\a #\~))
(assert-false  (tn) (char-ci=? #\a #\x7f))
(tn "char-ci=? #\\b")
(assert-false  (tn) (char-ci=? #\b #\x00))
(assert-false  (tn) (char-ci=? #\b #\newline))
(assert-false  (tn) (char-ci=? #\b #\space))
(assert-false  (tn) (char-ci=? #\b #\!))
(assert-false  (tn) (char-ci=? #\b #\0))
(assert-false  (tn) (char-ci=? #\b #\9))
(assert-false  (tn) (char-ci=? #\b #\A))
(assert-true   (tn) (char-ci=? #\b #\B))
(assert-false  (tn) (char-ci=? #\b #\Z))
(assert-false  (tn) (char-ci=? #\b #\_))
(assert-false  (tn) (char-ci=? #\b #\a))
(assert-true   (tn) (char-ci=? #\b #\b))
(assert-false  (tn) (char-ci=? #\b #\z))
(assert-false  (tn) (char-ci=? #\b #\~))
(assert-false  (tn) (char-ci=? #\b #\x7f))
(tn "char-ci=? #\\z")
(assert-false  (tn) (char-ci=? #\z #\x00))
(assert-false  (tn) (char-ci=? #\z #\newline))
(assert-false  (tn) (char-ci=? #\z #\space))
(assert-false  (tn) (char-ci=? #\z #\!))
(assert-false  (tn) (char-ci=? #\z #\0))
(assert-false  (tn) (char-ci=? #\z #\9))
(assert-false  (tn) (char-ci=? #\z #\A))
(assert-false  (tn) (char-ci=? #\z #\B))
(assert-true   (tn) (char-ci=? #\z #\Z))
(assert-false  (tn) (char-ci=? #\z #\_))
(assert-false  (tn) (char-ci=? #\z #\a))
(assert-false  (tn) (char-ci=? #\z #\b))
(assert-true   (tn) (char-ci=? #\z #\z))
(assert-false  (tn) (char-ci=? #\z #\~))
(assert-false  (tn) (char-ci=? #\z #\x7f))
(tn "char-ci=? #\\~")
(assert-false  (tn) (char-ci=? #\~ #\x00))
(assert-false  (tn) (char-ci=? #\~ #\newline))
(assert-false  (tn) (char-ci=? #\~ #\space))
(assert-false  (tn) (char-ci=? #\~ #\!))
(assert-false  (tn) (char-ci=? #\~ #\0))
(assert-false  (tn) (char-ci=? #\~ #\9))
(assert-false  (tn) (char-ci=? #\~ #\A))
(assert-false  (tn) (char-ci=? #\~ #\B))
(assert-false  (tn) (char-ci=? #\~ #\Z))
(assert-false  (tn) (char-ci=? #\~ #\_))
(assert-false  (tn) (char-ci=? #\~ #\a))
(assert-false  (tn) (char-ci=? #\~ #\b))
(assert-false  (tn) (char-ci=? #\~ #\z))
(assert-true   (tn) (char-ci=? #\~ #\~))
(assert-false  (tn) (char-ci=? #\~ #\x7f))
(tn "char-ci=? #\\x7f")
(assert-false  (tn) (char-ci=? #\x7f #\x00))
(assert-false  (tn) (char-ci=? #\x7f #\newline))
(assert-false  (tn) (char-ci=? #\x7f #\space))
(assert-false  (tn) (char-ci=? #\x7f #\!))
(assert-false  (tn) (char-ci=? #\x7f #\0))
(assert-false  (tn) (char-ci=? #\x7f #\9))
(assert-false  (tn) (char-ci=? #\x7f #\A))
(assert-false  (tn) (char-ci=? #\x7f #\B))
(assert-false  (tn) (char-ci=? #\x7f #\Z))
(assert-false  (tn) (char-ci=? #\x7f #\_))
(assert-false  (tn) (char-ci=? #\x7f #\a))
(assert-false  (tn) (char-ci=? #\x7f #\b))
(assert-false  (tn) (char-ci=? #\x7f #\z))
(assert-false  (tn) (char-ci=? #\x7f #\~))
(assert-true   (tn) (char-ci=? #\x7f #\x7f))

;; char-ci<?
(tn "char-ci<? #\\x00")
(assert-false  (tn) (char-ci<? #\x00 #\x00))
(assert-true   (tn) (char-ci<? #\x00 #\newline))
(assert-true   (tn) (char-ci<? #\x00 #\space))
(assert-true   (tn) (char-ci<? #\x00 #\!))
(assert-true   (tn) (char-ci<? #\x00 #\0))
(assert-true   (tn) (char-ci<? #\x00 #\9))
(assert-true   (tn) (char-ci<? #\x00 #\A))
(assert-true   (tn) (char-ci<? #\x00 #\B))
(assert-true   (tn) (char-ci<? #\x00 #\Z))
(assert-true   (tn) (char-ci<? #\x00 #\_))
(assert-true   (tn) (char-ci<? #\x00 #\a))
(assert-true   (tn) (char-ci<? #\x00 #\b))
(assert-true   (tn) (char-ci<? #\x00 #\z))
(assert-true   (tn) (char-ci<? #\x00 #\~))
(assert-true   (tn) (char-ci<? #\x00 #\x7f))
(tn "char-ci<? #\\newline")
(assert-false  (tn) (char-ci<? #\newline #\x00))
(assert-false  (tn) (char-ci<? #\newline #\newline))
(assert-true   (tn) (char-ci<? #\newline #\space))
(assert-true   (tn) (char-ci<? #\newline #\!))
(assert-true   (tn) (char-ci<? #\newline #\0))
(assert-true   (tn) (char-ci<? #\newline #\9))
(assert-true   (tn) (char-ci<? #\newline #\A))
(assert-true   (tn) (char-ci<? #\newline #\B))
(assert-true   (tn) (char-ci<? #\newline #\Z))
(assert-true   (tn) (char-ci<? #\newline #\_))
(assert-true   (tn) (char-ci<? #\newline #\a))
(assert-true   (tn) (char-ci<? #\newline #\b))
(assert-true   (tn) (char-ci<? #\newline #\z))
(assert-true   (tn) (char-ci<? #\newline #\~))
(assert-true   (tn) (char-ci<? #\newline #\x7f))
(tn "char-ci<? #\\space")
(assert-false  (tn) (char-ci<? #\space #\x00))
(assert-false  (tn) (char-ci<? #\space #\newline))
(assert-false  (tn) (char-ci<? #\space #\space))
(assert-true   (tn) (char-ci<? #\space #\!))
(assert-true   (tn) (char-ci<? #\space #\0))
(assert-true   (tn) (char-ci<? #\space #\9))
(assert-true   (tn) (char-ci<? #\space #\A))
(assert-true   (tn) (char-ci<? #\space #\B))
(assert-true   (tn) (char-ci<? #\space #\Z))
(assert-true   (tn) (char-ci<? #\space #\_))
(assert-true   (tn) (char-ci<? #\space #\a))
(assert-true   (tn) (char-ci<? #\space #\b))
(assert-true   (tn) (char-ci<? #\space #\z))
(assert-true   (tn) (char-ci<? #\space #\~))
(assert-true   (tn) (char-ci<? #\space #\x7f))
(tn "char-ci<? #\\!")
(assert-false  (tn) (char-ci<? #\! #\x00))
(assert-false  (tn) (char-ci<? #\! #\newline))
(assert-false  (tn) (char-ci<? #\! #\space))
(assert-false  (tn) (char-ci<? #\! #\!))
(assert-true   (tn) (char-ci<? #\! #\0))
(assert-true   (tn) (char-ci<? #\! #\9))
(assert-true   (tn) (char-ci<? #\! #\A))
(assert-true   (tn) (char-ci<? #\! #\B))
(assert-true   (tn) (char-ci<? #\! #\Z))
(assert-true   (tn) (char-ci<? #\! #\_))
(assert-true   (tn) (char-ci<? #\! #\a))
(assert-true   (tn) (char-ci<? #\! #\b))
(assert-true   (tn) (char-ci<? #\! #\z))
(assert-true   (tn) (char-ci<? #\! #\~))
(assert-true   (tn) (char-ci<? #\! #\x7f))
(tn "char-ci<? #\\0")
(assert-false  (tn) (char-ci<? #\0 #\x00))
(assert-false  (tn) (char-ci<? #\0 #\newline))
(assert-false  (tn) (char-ci<? #\0 #\space))
(assert-false  (tn) (char-ci<? #\0 #\!))
(assert-false  (tn) (char-ci<? #\0 #\0))
(assert-true   (tn) (char-ci<? #\0 #\9))
(assert-true   (tn) (char-ci<? #\0 #\A))
(assert-true   (tn) (char-ci<? #\0 #\B))
(assert-true   (tn) (char-ci<? #\0 #\Z))
(assert-true   (tn) (char-ci<? #\0 #\_))
(assert-true   (tn) (char-ci<? #\0 #\a))
(assert-true   (tn) (char-ci<? #\0 #\b))
(assert-true   (tn) (char-ci<? #\0 #\z))
(assert-true   (tn) (char-ci<? #\0 #\~))
(assert-true   (tn) (char-ci<? #\0 #\x7f))
(tn "char-ci<? #\\9")
(assert-false  (tn) (char-ci<? #\9 #\x00))
(assert-false  (tn) (char-ci<? #\9 #\newline))
(assert-false  (tn) (char-ci<? #\9 #\space))
(assert-false  (tn) (char-ci<? #\9 #\!))
(assert-false  (tn) (char-ci<? #\9 #\0))
(assert-false  (tn) (char-ci<? #\9 #\9))
(assert-true   (tn) (char-ci<? #\9 #\A))
(assert-true   (tn) (char-ci<? #\9 #\B))
(assert-true   (tn) (char-ci<? #\9 #\Z))
(assert-true   (tn) (char-ci<? #\9 #\_))
(assert-true   (tn) (char-ci<? #\9 #\a))
(assert-true   (tn) (char-ci<? #\9 #\b))
(assert-true   (tn) (char-ci<? #\9 #\z))
(assert-true   (tn) (char-ci<? #\9 #\~))
(assert-true   (tn) (char-ci<? #\9 #\x7f))
(tn "char-ci<? #\\A")
(assert-false  (tn) (char-ci<? #\A #\x00))
(assert-false  (tn) (char-ci<? #\A #\newline))
(assert-false  (tn) (char-ci<? #\A #\space))
(assert-false  (tn) (char-ci<? #\A #\!))
(assert-false  (tn) (char-ci<? #\A #\0))
(assert-false  (tn) (char-ci<? #\A #\9))
(assert-false  (tn) (char-ci<? #\A #\A))
(assert-true   (tn) (char-ci<? #\A #\B))
(assert-true   (tn) (char-ci<? #\A #\Z))
(assert-false  (tn) (char-ci<? #\A #\_))
(assert-false  (tn) (char-ci<? #\A #\a))
(assert-true   (tn) (char-ci<? #\A #\b))
(assert-true   (tn) (char-ci<? #\A #\z))
(assert-true   (tn) (char-ci<? #\A #\~))
(assert-true   (tn) (char-ci<? #\A #\x7f))
(tn "char-ci<? #\\B")
(assert-false  (tn) (char-ci<? #\B #\x00))
(assert-false  (tn) (char-ci<? #\B #\newline))
(assert-false  (tn) (char-ci<? #\B #\space))
(assert-false  (tn) (char-ci<? #\B #\!))
(assert-false  (tn) (char-ci<? #\B #\0))
(assert-false  (tn) (char-ci<? #\B #\9))
(assert-false  (tn) (char-ci<? #\B #\A))
(assert-false  (tn) (char-ci<? #\B #\B))
(assert-true   (tn) (char-ci<? #\B #\Z))
(assert-false  (tn) (char-ci<? #\B #\_))
(assert-false  (tn) (char-ci<? #\B #\a))
(assert-false  (tn) (char-ci<? #\B #\b))
(assert-true   (tn) (char-ci<? #\B #\z))
(assert-true   (tn) (char-ci<? #\B #\~))
(assert-true   (tn) (char-ci<? #\B #\x7f))
(tn "char-ci<? #\\Z")
(assert-false  (tn) (char-ci<? #\Z #\x00))
(assert-false  (tn) (char-ci<? #\Z #\newline))
(assert-false  (tn) (char-ci<? #\Z #\space))
(assert-false  (tn) (char-ci<? #\Z #\!))
(assert-false  (tn) (char-ci<? #\Z #\0))
(assert-false  (tn) (char-ci<? #\Z #\9))
(assert-false  (tn) (char-ci<? #\Z #\A))
(assert-false  (tn) (char-ci<? #\Z #\B))
(assert-false  (tn) (char-ci<? #\Z #\Z))
(assert-false  (tn) (char-ci<? #\Z #\_))
(assert-false  (tn) (char-ci<? #\Z #\a))
(assert-false  (tn) (char-ci<? #\Z #\b))
(assert-false  (tn) (char-ci<? #\Z #\z))
(assert-true   (tn) (char-ci<? #\Z #\~))
(assert-true   (tn) (char-ci<? #\Z #\x7f))
(tn "char-ci<? #\\_")
(assert-false  (tn) (char-ci<? #\_ #\x00))
(assert-false  (tn) (char-ci<? #\_ #\newline))
(assert-false  (tn) (char-ci<? #\_ #\space))
(assert-false  (tn) (char-ci<? #\_ #\!))
(assert-false  (tn) (char-ci<? #\_ #\0))
(assert-false  (tn) (char-ci<? #\_ #\9))
(assert-true   (tn) (char-ci<? #\_ #\A))
(assert-true   (tn) (char-ci<? #\_ #\B))
(assert-true   (tn) (char-ci<? #\_ #\Z))
(assert-false  (tn) (char-ci<? #\_ #\_))
(assert-true   (tn) (char-ci<? #\_ #\a))
(assert-true   (tn) (char-ci<? #\_ #\b))
(assert-true   (tn) (char-ci<? #\_ #\z))
(assert-true   (tn) (char-ci<? #\_ #\~))
(assert-true   (tn) (char-ci<? #\_ #\x7f))
(tn "char-ci<? #\\a")
(assert-false  (tn) (char-ci<? #\a #\x00))
(assert-false  (tn) (char-ci<? #\a #\newline))
(assert-false  (tn) (char-ci<? #\a #\space))
(assert-false  (tn) (char-ci<? #\a #\!))
(assert-false  (tn) (char-ci<? #\a #\0))
(assert-false  (tn) (char-ci<? #\a #\9))
(assert-false  (tn) (char-ci<? #\a #\A))
(assert-true   (tn) (char-ci<? #\a #\B))
(assert-true   (tn) (char-ci<? #\a #\Z))
(assert-false  (tn) (char-ci<? #\a #\_))
(assert-false  (tn) (char-ci<? #\a #\a))
(assert-true   (tn) (char-ci<? #\a #\b))
(assert-true   (tn) (char-ci<? #\a #\z))
(assert-true   (tn) (char-ci<? #\a #\~))
(assert-true   (tn) (char-ci<? #\a #\x7f))
(tn "char-ci<? #\\b")
(assert-false  (tn) (char-ci<? #\b #\x00))
(assert-false  (tn) (char-ci<? #\b #\newline))
(assert-false  (tn) (char-ci<? #\b #\space))
(assert-false  (tn) (char-ci<? #\b #\!))
(assert-false  (tn) (char-ci<? #\b #\0))
(assert-false  (tn) (char-ci<? #\b #\9))
(assert-false  (tn) (char-ci<? #\b #\A))
(assert-false  (tn) (char-ci<? #\b #\B))
(assert-true   (tn) (char-ci<? #\b #\Z))
(assert-false  (tn) (char-ci<? #\b #\_))
(assert-false  (tn) (char-ci<? #\b #\a))
(assert-false  (tn) (char-ci<? #\b #\b))
(assert-true   (tn) (char-ci<? #\b #\z))
(assert-true   (tn) (char-ci<? #\b #\~))
(assert-true   (tn) (char-ci<? #\b #\x7f))
(tn "char-ci<? #\\z")
(assert-false  (tn) (char-ci<? #\z #\x00))
(assert-false  (tn) (char-ci<? #\z #\newline))
(assert-false  (tn) (char-ci<? #\z #\space))
(assert-false  (tn) (char-ci<? #\z #\!))
(assert-false  (tn) (char-ci<? #\z #\0))
(assert-false  (tn) (char-ci<? #\z #\9))
(assert-false  (tn) (char-ci<? #\z #\A))
(assert-false  (tn) (char-ci<? #\z #\B))
(assert-false  (tn) (char-ci<? #\z #\Z))
(assert-false  (tn) (char-ci<? #\z #\_))
(assert-false  (tn) (char-ci<? #\z #\a))
(assert-false  (tn) (char-ci<? #\z #\b))
(assert-false  (tn) (char-ci<? #\z #\z))
(assert-true   (tn) (char-ci<? #\z #\~))
(assert-true   (tn) (char-ci<? #\z #\x7f))
(tn "char-ci<? #\\~")
(assert-false  (tn) (char-ci<? #\~ #\x00))
(assert-false  (tn) (char-ci<? #\~ #\newline))
(assert-false  (tn) (char-ci<? #\~ #\space))
(assert-false  (tn) (char-ci<? #\~ #\!))
(assert-false  (tn) (char-ci<? #\~ #\0))
(assert-false  (tn) (char-ci<? #\~ #\9))
(assert-false  (tn) (char-ci<? #\~ #\A))
(assert-false  (tn) (char-ci<? #\~ #\B))
(assert-false  (tn) (char-ci<? #\~ #\Z))
(assert-false  (tn) (char-ci<? #\~ #\_))
(assert-false  (tn) (char-ci<? #\~ #\a))
(assert-false  (tn) (char-ci<? #\~ #\b))
(assert-false  (tn) (char-ci<? #\~ #\z))
(assert-false  (tn) (char-ci<? #\~ #\~))
(assert-true   (tn) (char-ci<? #\~ #\x7f))
(tn "char-ci<? #\\x7f")
(assert-false  (tn) (char-ci<? #\x7f #\x00))
(assert-false  (tn) (char-ci<? #\x7f #\newline))
(assert-false  (tn) (char-ci<? #\x7f #\space))
(assert-false  (tn) (char-ci<? #\x7f #\!))
(assert-false  (tn) (char-ci<? #\x7f #\0))
(assert-false  (tn) (char-ci<? #\x7f #\9))
(assert-false  (tn) (char-ci<? #\x7f #\A))
(assert-false  (tn) (char-ci<? #\x7f #\B))
(assert-false  (tn) (char-ci<? #\x7f #\Z))
(assert-false  (tn) (char-ci<? #\x7f #\_))
(assert-false  (tn) (char-ci<? #\x7f #\a))
(assert-false  (tn) (char-ci<? #\x7f #\b))
(assert-false  (tn) (char-ci<? #\x7f #\z))
(assert-false  (tn) (char-ci<? #\x7f #\~))
(assert-false  (tn) (char-ci<? #\x7f #\x7f))

;; char-ci>?
(tn "char-ci>? #\\x00")
(assert-false  (tn) (char-ci>? #\x00 #\x00))
(assert-false  (tn) (char-ci>? #\x00 #\newline))
(assert-false  (tn) (char-ci>? #\x00 #\space))
(assert-false  (tn) (char-ci>? #\x00 #\!))
(assert-false  (tn) (char-ci>? #\x00 #\0))
(assert-false  (tn) (char-ci>? #\x00 #\9))
(assert-false  (tn) (char-ci>? #\x00 #\A))
(assert-false  (tn) (char-ci>? #\x00 #\B))
(assert-false  (tn) (char-ci>? #\x00 #\Z))
(assert-false  (tn) (char-ci>? #\x00 #\_))
(assert-false  (tn) (char-ci>? #\x00 #\a))
(assert-false  (tn) (char-ci>? #\x00 #\b))
(assert-false  (tn) (char-ci>? #\x00 #\z))
(assert-false  (tn) (char-ci>? #\x00 #\~))
(assert-false  (tn) (char-ci>? #\x00 #\x7f))
(tn "char-ci>? #\\newline")
(assert-true   (tn) (char-ci>? #\newline #\x00))
(assert-false  (tn) (char-ci>? #\newline #\newline))
(assert-false  (tn) (char-ci>? #\newline #\space))
(assert-false  (tn) (char-ci>? #\newline #\!))
(assert-false  (tn) (char-ci>? #\newline #\0))
(assert-false  (tn) (char-ci>? #\newline #\9))
(assert-false  (tn) (char-ci>? #\newline #\A))
(assert-false  (tn) (char-ci>? #\newline #\B))
(assert-false  (tn) (char-ci>? #\newline #\Z))
(assert-false  (tn) (char-ci>? #\newline #\_))
(assert-false  (tn) (char-ci>? #\newline #\a))
(assert-false  (tn) (char-ci>? #\newline #\b))
(assert-false  (tn) (char-ci>? #\newline #\z))
(assert-false  (tn) (char-ci>? #\newline #\~))
(assert-false  (tn) (char-ci>? #\newline #\x7f))
(tn "char-ci>? #\\space")
(assert-true   (tn) (char-ci>? #\space #\x00))
(assert-true   (tn) (char-ci>? #\space #\newline))
(assert-false  (tn) (char-ci>? #\space #\space))
(assert-false  (tn) (char-ci>? #\space #\!))
(assert-false  (tn) (char-ci>? #\space #\0))
(assert-false  (tn) (char-ci>? #\space #\9))
(assert-false  (tn) (char-ci>? #\space #\A))
(assert-false  (tn) (char-ci>? #\space #\B))
(assert-false  (tn) (char-ci>? #\space #\Z))
(assert-false  (tn) (char-ci>? #\space #\_))
(assert-false  (tn) (char-ci>? #\space #\a))
(assert-false  (tn) (char-ci>? #\space #\b))
(assert-false  (tn) (char-ci>? #\space #\z))
(assert-false  (tn) (char-ci>? #\space #\~))
(assert-false  (tn) (char-ci>? #\space #\x7f))
(tn "char-ci>? #\\!")
(assert-true   (tn) (char-ci>? #\! #\x00))
(assert-true   (tn) (char-ci>? #\! #\newline))
(assert-true   (tn) (char-ci>? #\! #\space))
(assert-false  (tn) (char-ci>? #\! #\!))
(assert-false  (tn) (char-ci>? #\! #\0))
(assert-false  (tn) (char-ci>? #\! #\9))
(assert-false  (tn) (char-ci>? #\! #\A))
(assert-false  (tn) (char-ci>? #\! #\B))
(assert-false  (tn) (char-ci>? #\! #\Z))
(assert-false  (tn) (char-ci>? #\! #\_))
(assert-false  (tn) (char-ci>? #\! #\a))
(assert-false  (tn) (char-ci>? #\! #\b))
(assert-false  (tn) (char-ci>? #\! #\z))
(assert-false  (tn) (char-ci>? #\! #\~))
(assert-false  (tn) (char-ci>? #\! #\x7f))
(tn "char-ci>? #\\0")
(assert-true   (tn) (char-ci>? #\0 #\x00))
(assert-true   (tn) (char-ci>? #\0 #\newline))
(assert-true   (tn) (char-ci>? #\0 #\space))
(assert-true   (tn) (char-ci>? #\0 #\!))
(assert-false  (tn) (char-ci>? #\0 #\0))
(assert-false  (tn) (char-ci>? #\0 #\9))
(assert-false  (tn) (char-ci>? #\0 #\A))
(assert-false  (tn) (char-ci>? #\0 #\B))
(assert-false  (tn) (char-ci>? #\0 #\Z))
(assert-false  (tn) (char-ci>? #\0 #\_))
(assert-false  (tn) (char-ci>? #\0 #\a))
(assert-false  (tn) (char-ci>? #\0 #\b))
(assert-false  (tn) (char-ci>? #\0 #\z))
(assert-false  (tn) (char-ci>? #\0 #\~))
(assert-false  (tn) (char-ci>? #\0 #\x7f))
(tn "char-ci>? #\\9")
(assert-true   (tn) (char-ci>? #\9 #\x00))
(assert-true   (tn) (char-ci>? #\9 #\newline))
(assert-true   (tn) (char-ci>? #\9 #\space))
(assert-true   (tn) (char-ci>? #\9 #\!))
(assert-true   (tn) (char-ci>? #\9 #\0))
(assert-false  (tn) (char-ci>? #\9 #\9))
(assert-false  (tn) (char-ci>? #\9 #\A))
(assert-false  (tn) (char-ci>? #\9 #\B))
(assert-false  (tn) (char-ci>? #\9 #\Z))
(assert-false  (tn) (char-ci>? #\9 #\_))
(assert-false  (tn) (char-ci>? #\9 #\a))
(assert-false  (tn) (char-ci>? #\9 #\b))
(assert-false  (tn) (char-ci>? #\9 #\z))
(assert-false  (tn) (char-ci>? #\9 #\~))
(assert-false  (tn) (char-ci>? #\9 #\x7f))
(tn "char-ci>? #\\A")
(assert-true   (tn) (char-ci>? #\A #\x00))
(assert-true   (tn) (char-ci>? #\A #\newline))
(assert-true   (tn) (char-ci>? #\A #\space))
(assert-true   (tn) (char-ci>? #\A #\!))
(assert-true   (tn) (char-ci>? #\A #\0))
(assert-true   (tn) (char-ci>? #\A #\9))
(assert-false  (tn) (char-ci>? #\A #\A))
(assert-false  (tn) (char-ci>? #\A #\B))
(assert-false  (tn) (char-ci>? #\A #\Z))
(assert-true   (tn) (char-ci>? #\A #\_))
(assert-false  (tn) (char-ci>? #\A #\a))
(assert-false  (tn) (char-ci>? #\A #\b))
(assert-false  (tn) (char-ci>? #\A #\z))
(assert-false  (tn) (char-ci>? #\A #\~))
(assert-false  (tn) (char-ci>? #\A #\x7f))
(tn "char-ci>? #\\B")
(assert-true   (tn) (char-ci>? #\B #\x00))
(assert-true   (tn) (char-ci>? #\B #\newline))
(assert-true   (tn) (char-ci>? #\B #\space))
(assert-true   (tn) (char-ci>? #\B #\!))
(assert-true   (tn) (char-ci>? #\B #\0))
(assert-true   (tn) (char-ci>? #\B #\9))
(assert-true   (tn) (char-ci>? #\B #\A))
(assert-false  (tn) (char-ci>? #\B #\B))
(assert-false  (tn) (char-ci>? #\B #\Z))
(assert-true   (tn) (char-ci>? #\B #\_))
(assert-true   (tn) (char-ci>? #\B #\a))
(assert-false  (tn) (char-ci>? #\B #\b))
(assert-false  (tn) (char-ci>? #\B #\z))
(assert-false  (tn) (char-ci>? #\B #\~))
(assert-false  (tn) (char-ci>? #\B #\x7f))
(tn "char-ci>? #\\Z")
(assert-true   (tn) (char-ci>? #\Z #\x00))
(assert-true   (tn) (char-ci>? #\Z #\newline))
(assert-true   (tn) (char-ci>? #\Z #\space))
(assert-true   (tn) (char-ci>? #\Z #\!))
(assert-true   (tn) (char-ci>? #\Z #\0))
(assert-true   (tn) (char-ci>? #\Z #\9))
(assert-true   (tn) (char-ci>? #\Z #\A))
(assert-true   (tn) (char-ci>? #\Z #\B))
(assert-false  (tn) (char-ci>? #\Z #\Z))
(assert-true   (tn) (char-ci>? #\Z #\_))
(assert-true   (tn) (char-ci>? #\Z #\a))
(assert-true   (tn) (char-ci>? #\Z #\b))
(assert-false  (tn) (char-ci>? #\Z #\z))
(assert-false  (tn) (char-ci>? #\Z #\~))
(assert-false  (tn) (char-ci>? #\Z #\x7f))
(tn "char-ci>? #\\_")
(assert-true   (tn) (char-ci>? #\_ #\x00))
(assert-true   (tn) (char-ci>? #\_ #\newline))
(assert-true   (tn) (char-ci>? #\_ #\space))
(assert-true   (tn) (char-ci>? #\_ #\!))
(assert-true   (tn) (char-ci>? #\_ #\0))
(assert-true   (tn) (char-ci>? #\_ #\9))
(assert-false  (tn) (char-ci>? #\_ #\A))
(assert-false  (tn) (char-ci>? #\_ #\B))
(assert-false  (tn) (char-ci>? #\_ #\Z))
(assert-false  (tn) (char-ci>? #\_ #\_))
(assert-false  (tn) (char-ci>? #\_ #\a))
(assert-false  (tn) (char-ci>? #\_ #\b))
(assert-false  (tn) (char-ci>? #\_ #\z))
(assert-false  (tn) (char-ci>? #\_ #\~))
(assert-false  (tn) (char-ci>? #\_ #\x7f))
(tn "char-ci>? #\\a")
(assert-true   (tn) (char-ci>? #\a #\x00))
(assert-true   (tn) (char-ci>? #\a #\newline))
(assert-true   (tn) (char-ci>? #\a #\space))
(assert-true   (tn) (char-ci>? #\a #\!))
(assert-true   (tn) (char-ci>? #\a #\0))
(assert-true   (tn) (char-ci>? #\a #\9))
(assert-false  (tn) (char-ci>? #\a #\A))
(assert-false  (tn) (char-ci>? #\a #\B))
(assert-false  (tn) (char-ci>? #\a #\Z))
(assert-true   (tn) (char-ci>? #\a #\_))
(assert-false  (tn) (char-ci>? #\a #\a))
(assert-false  (tn) (char-ci>? #\a #\b))
(assert-false  (tn) (char-ci>? #\a #\z))
(assert-false  (tn) (char-ci>? #\a #\~))
(assert-false  (tn) (char-ci>? #\a #\x7f))
(tn "char-ci>? #\\b")
(assert-true   (tn) (char-ci>? #\b #\x00))
(assert-true   (tn) (char-ci>? #\b #\newline))
(assert-true   (tn) (char-ci>? #\b #\space))
(assert-true   (tn) (char-ci>? #\b #\!))
(assert-true   (tn) (char-ci>? #\b #\0))
(assert-true   (tn) (char-ci>? #\b #\9))
(assert-true   (tn) (char-ci>? #\b #\A))
(assert-false  (tn) (char-ci>? #\b #\B))
(assert-false  (tn) (char-ci>? #\b #\Z))
(assert-true   (tn) (char-ci>? #\b #\_))
(assert-true   (tn) (char-ci>? #\b #\a))
(assert-false  (tn) (char-ci>? #\b #\b))
(assert-false  (tn) (char-ci>? #\b #\z))
(assert-false  (tn) (char-ci>? #\b #\~))
(assert-false  (tn) (char-ci>? #\b #\x7f))
(tn "char-ci>? #\\z")
(assert-true   (tn) (char-ci>? #\z #\x00))
(assert-true   (tn) (char-ci>? #\z #\newline))
(assert-true   (tn) (char-ci>? #\z #\space))
(assert-true   (tn) (char-ci>? #\z #\!))
(assert-true   (tn) (char-ci>? #\z #\0))
(assert-true   (tn) (char-ci>? #\z #\9))
(assert-true   (tn) (char-ci>? #\z #\A))
(assert-true   (tn) (char-ci>? #\z #\B))
(assert-false  (tn) (char-ci>? #\z #\Z))
(assert-true   (tn) (char-ci>? #\z #\_))
(assert-true   (tn) (char-ci>? #\z #\a))
(assert-true   (tn) (char-ci>? #\z #\b))
(assert-false  (tn) (char-ci>? #\z #\z))
(assert-false  (tn) (char-ci>? #\z #\~))
(assert-false  (tn) (char-ci>? #\z #\x7f))
(tn "char-ci>? #\\~")
(assert-true   (tn) (char-ci>? #\~ #\x00))
(assert-true   (tn) (char-ci>? #\~ #\newline))
(assert-true   (tn) (char-ci>? #\~ #\space))
(assert-true   (tn) (char-ci>? #\~ #\!))
(assert-true   (tn) (char-ci>? #\~ #\0))
(assert-true   (tn) (char-ci>? #\~ #\9))
(assert-true   (tn) (char-ci>? #\~ #\A))
(assert-true   (tn) (char-ci>? #\~ #\B))
(assert-true   (tn) (char-ci>? #\~ #\Z))
(assert-true   (tn) (char-ci>? #\~ #\_))
(assert-true   (tn) (char-ci>? #\~ #\a))
(assert-true   (tn) (char-ci>? #\~ #\b))
(assert-true   (tn) (char-ci>? #\~ #\z))
(assert-false  (tn) (char-ci>? #\~ #\~))
(assert-false  (tn) (char-ci>? #\~ #\x7f))
(tn "char-ci>? #\\x7f")
(assert-true   (tn) (char-ci>? #\x7f #\x00))
(assert-true   (tn) (char-ci>? #\x7f #\newline))
(assert-true   (tn) (char-ci>? #\x7f #\space))
(assert-true   (tn) (char-ci>? #\x7f #\!))
(assert-true   (tn) (char-ci>? #\x7f #\0))
(assert-true   (tn) (char-ci>? #\x7f #\9))
(assert-true   (tn) (char-ci>? #\x7f #\A))
(assert-true   (tn) (char-ci>? #\x7f #\B))
(assert-true   (tn) (char-ci>? #\x7f #\Z))
(assert-true   (tn) (char-ci>? #\x7f #\_))
(assert-true   (tn) (char-ci>? #\x7f #\a))
(assert-true   (tn) (char-ci>? #\x7f #\b))
(assert-true   (tn) (char-ci>? #\x7f #\z))
(assert-true   (tn) (char-ci>? #\x7f #\~))
(assert-false  (tn) (char-ci>? #\x7f #\x7f))

;; char-ci<=?
(tn "char-ci<=? #\\x00")
(assert-true   (tn) (char-ci<=? #\x00 #\x00))
(assert-true   (tn) (char-ci<=? #\x00 #\newline))
(assert-true   (tn) (char-ci<=? #\x00 #\space))
(assert-true   (tn) (char-ci<=? #\x00 #\!))
(assert-true   (tn) (char-ci<=? #\x00 #\0))
(assert-true   (tn) (char-ci<=? #\x00 #\9))
(assert-true   (tn) (char-ci<=? #\x00 #\A))
(assert-true   (tn) (char-ci<=? #\x00 #\B))
(assert-true   (tn) (char-ci<=? #\x00 #\Z))
(assert-true   (tn) (char-ci<=? #\x00 #\_))
(assert-true   (tn) (char-ci<=? #\x00 #\a))
(assert-true   (tn) (char-ci<=? #\x00 #\b))
(assert-true   (tn) (char-ci<=? #\x00 #\z))
(assert-true   (tn) (char-ci<=? #\x00 #\~))
(assert-true   (tn) (char-ci<=? #\x00 #\x7f))
(tn "char-ci<=? #\\newline")
(assert-false  (tn) (char-ci<=? #\newline #\x00))
(assert-true   (tn) (char-ci<=? #\newline #\newline))
(assert-true   (tn) (char-ci<=? #\newline #\space))
(assert-true   (tn) (char-ci<=? #\newline #\!))
(assert-true   (tn) (char-ci<=? #\newline #\0))
(assert-true   (tn) (char-ci<=? #\newline #\9))
(assert-true   (tn) (char-ci<=? #\newline #\A))
(assert-true   (tn) (char-ci<=? #\newline #\B))
(assert-true   (tn) (char-ci<=? #\newline #\Z))
(assert-true   (tn) (char-ci<=? #\newline #\_))
(assert-true   (tn) (char-ci<=? #\newline #\a))
(assert-true   (tn) (char-ci<=? #\newline #\b))
(assert-true   (tn) (char-ci<=? #\newline #\z))
(assert-true   (tn) (char-ci<=? #\newline #\~))
(assert-true   (tn) (char-ci<=? #\newline #\x7f))
(tn "char-ci<=? #\\space")
(assert-false  (tn) (char-ci<=? #\space #\x00))
(assert-false  (tn) (char-ci<=? #\space #\newline))
(assert-true   (tn) (char-ci<=? #\space #\space))
(assert-true   (tn) (char-ci<=? #\space #\!))
(assert-true   (tn) (char-ci<=? #\space #\0))
(assert-true   (tn) (char-ci<=? #\space #\9))
(assert-true   (tn) (char-ci<=? #\space #\A))
(assert-true   (tn) (char-ci<=? #\space #\B))
(assert-true   (tn) (char-ci<=? #\space #\Z))
(assert-true   (tn) (char-ci<=? #\space #\_))
(assert-true   (tn) (char-ci<=? #\space #\a))
(assert-true   (tn) (char-ci<=? #\space #\b))
(assert-true   (tn) (char-ci<=? #\space #\z))
(assert-true   (tn) (char-ci<=? #\space #\~))
(assert-true   (tn) (char-ci<=? #\space #\x7f))
(tn "char-ci<=? #\\!")
(assert-false  (tn) (char-ci<=? #\! #\x00))
(assert-false  (tn) (char-ci<=? #\! #\newline))
(assert-false  (tn) (char-ci<=? #\! #\space))
(assert-true   (tn) (char-ci<=? #\! #\!))
(assert-true   (tn) (char-ci<=? #\! #\0))
(assert-true   (tn) (char-ci<=? #\! #\9))
(assert-true   (tn) (char-ci<=? #\! #\A))
(assert-true   (tn) (char-ci<=? #\! #\B))
(assert-true   (tn) (char-ci<=? #\! #\Z))
(assert-true   (tn) (char-ci<=? #\! #\_))
(assert-true   (tn) (char-ci<=? #\! #\a))
(assert-true   (tn) (char-ci<=? #\! #\b))
(assert-true   (tn) (char-ci<=? #\! #\z))
(assert-true   (tn) (char-ci<=? #\! #\~))
(assert-true   (tn) (char-ci<=? #\! #\x7f))
(tn "char-ci<=? #\\0")
(assert-false  (tn) (char-ci<=? #\0 #\x00))
(assert-false  (tn) (char-ci<=? #\0 #\newline))
(assert-false  (tn) (char-ci<=? #\0 #\space))
(assert-false  (tn) (char-ci<=? #\0 #\!))
(assert-true   (tn) (char-ci<=? #\0 #\0))
(assert-true   (tn) (char-ci<=? #\0 #\9))
(assert-true   (tn) (char-ci<=? #\0 #\A))
(assert-true   (tn) (char-ci<=? #\0 #\B))
(assert-true   (tn) (char-ci<=? #\0 #\Z))
(assert-true   (tn) (char-ci<=? #\0 #\_))
(assert-true   (tn) (char-ci<=? #\0 #\a))
(assert-true   (tn) (char-ci<=? #\0 #\b))
(assert-true   (tn) (char-ci<=? #\0 #\z))
(assert-true   (tn) (char-ci<=? #\0 #\~))
(assert-true   (tn) (char-ci<=? #\0 #\x7f))
(tn "char-ci<=? #\\9")
(assert-false  (tn) (char-ci<=? #\9 #\x00))
(assert-false  (tn) (char-ci<=? #\9 #\newline))
(assert-false  (tn) (char-ci<=? #\9 #\space))
(assert-false  (tn) (char-ci<=? #\9 #\!))
(assert-false  (tn) (char-ci<=? #\9 #\0))
(assert-true   (tn) (char-ci<=? #\9 #\9))
(assert-true   (tn) (char-ci<=? #\9 #\A))
(assert-true   (tn) (char-ci<=? #\9 #\B))
(assert-true   (tn) (char-ci<=? #\9 #\Z))
(assert-true   (tn) (char-ci<=? #\9 #\_))
(assert-true   (tn) (char-ci<=? #\9 #\a))
(assert-true   (tn) (char-ci<=? #\9 #\b))
(assert-true   (tn) (char-ci<=? #\9 #\z))
(assert-true   (tn) (char-ci<=? #\9 #\~))
(assert-true   (tn) (char-ci<=? #\9 #\x7f))
(tn "char-ci<=? #\\A")
(assert-false  (tn) (char-ci<=? #\A #\x00))
(assert-false  (tn) (char-ci<=? #\A #\newline))
(assert-false  (tn) (char-ci<=? #\A #\space))
(assert-false  (tn) (char-ci<=? #\A #\!))
(assert-false  (tn) (char-ci<=? #\A #\0))
(assert-false  (tn) (char-ci<=? #\A #\9))
(assert-true   (tn) (char-ci<=? #\A #\A))
(assert-true   (tn) (char-ci<=? #\A #\B))
(assert-true   (tn) (char-ci<=? #\A #\Z))
(assert-false  (tn) (char-ci<=? #\A #\_))
(assert-true   (tn) (char-ci<=? #\A #\a))
(assert-true   (tn) (char-ci<=? #\A #\b))
(assert-true   (tn) (char-ci<=? #\A #\z))
(assert-true   (tn) (char-ci<=? #\A #\~))
(assert-true   (tn) (char-ci<=? #\A #\x7f))
(tn "char-ci<=? #\\B")
(assert-false  (tn) (char-ci<=? #\B #\x00))
(assert-false  (tn) (char-ci<=? #\B #\newline))
(assert-false  (tn) (char-ci<=? #\B #\space))
(assert-false  (tn) (char-ci<=? #\B #\!))
(assert-false  (tn) (char-ci<=? #\B #\0))
(assert-false  (tn) (char-ci<=? #\B #\9))
(assert-false  (tn) (char-ci<=? #\B #\A))
(assert-true   (tn) (char-ci<=? #\B #\B))
(assert-true   (tn) (char-ci<=? #\B #\Z))
(assert-false  (tn) (char-ci<=? #\B #\_))
(assert-false  (tn) (char-ci<=? #\B #\a))
(assert-true   (tn) (char-ci<=? #\B #\b))
(assert-true   (tn) (char-ci<=? #\B #\z))
(assert-true   (tn) (char-ci<=? #\B #\~))
(assert-true   (tn) (char-ci<=? #\B #\x7f))
(tn "char-ci<=? #\\Z")
(assert-false  (tn) (char-ci<=? #\Z #\x00))
(assert-false  (tn) (char-ci<=? #\Z #\newline))
(assert-false  (tn) (char-ci<=? #\Z #\space))
(assert-false  (tn) (char-ci<=? #\Z #\!))
(assert-false  (tn) (char-ci<=? #\Z #\0))
(assert-false  (tn) (char-ci<=? #\Z #\9))
(assert-false  (tn) (char-ci<=? #\Z #\A))
(assert-false  (tn) (char-ci<=? #\Z #\B))
(assert-true   (tn) (char-ci<=? #\Z #\Z))
(assert-false  (tn) (char-ci<=? #\Z #\_))
(assert-false  (tn) (char-ci<=? #\Z #\a))
(assert-false  (tn) (char-ci<=? #\Z #\b))
(assert-true   (tn) (char-ci<=? #\Z #\z))
(assert-true   (tn) (char-ci<=? #\Z #\~))
(assert-true   (tn) (char-ci<=? #\Z #\x7f))
(tn "char-ci<=? #\\_")
(assert-false  (tn) (char-ci<=? #\_ #\x00))
(assert-false  (tn) (char-ci<=? #\_ #\newline))
(assert-false  (tn) (char-ci<=? #\_ #\space))
(assert-false  (tn) (char-ci<=? #\_ #\!))
(assert-false  (tn) (char-ci<=? #\_ #\0))
(assert-false  (tn) (char-ci<=? #\_ #\9))
(assert-true   (tn) (char-ci<=? #\_ #\A))
(assert-true   (tn) (char-ci<=? #\_ #\B))
(assert-true   (tn) (char-ci<=? #\_ #\Z))
(assert-true   (tn) (char-ci<=? #\_ #\_))
(assert-true   (tn) (char-ci<=? #\_ #\a))
(assert-true   (tn) (char-ci<=? #\_ #\b))
(assert-true   (tn) (char-ci<=? #\_ #\z))
(assert-true   (tn) (char-ci<=? #\_ #\~))
(assert-true   (tn) (char-ci<=? #\_ #\x7f))
(tn "char-ci<=? #\\a")
(assert-false  (tn) (char-ci<=? #\a #\x00))
(assert-false  (tn) (char-ci<=? #\a #\newline))
(assert-false  (tn) (char-ci<=? #\a #\space))
(assert-false  (tn) (char-ci<=? #\a #\!))
(assert-false  (tn) (char-ci<=? #\a #\0))
(assert-false  (tn) (char-ci<=? #\a #\9))
(assert-true   (tn) (char-ci<=? #\a #\A))
(assert-true   (tn) (char-ci<=? #\a #\B))
(assert-true   (tn) (char-ci<=? #\a #\Z))
(assert-false  (tn) (char-ci<=? #\a #\_))
(assert-true   (tn) (char-ci<=? #\a #\a))
(assert-true   (tn) (char-ci<=? #\a #\b))
(assert-true   (tn) (char-ci<=? #\a #\z))
(assert-true   (tn) (char-ci<=? #\a #\~))
(assert-true   (tn) (char-ci<=? #\a #\x7f))
(tn "char-ci<=? #\\b")
(assert-false  (tn) (char-ci<=? #\b #\x00))
(assert-false  (tn) (char-ci<=? #\b #\newline))
(assert-false  (tn) (char-ci<=? #\b #\space))
(assert-false  (tn) (char-ci<=? #\b #\!))
(assert-false  (tn) (char-ci<=? #\b #\0))
(assert-false  (tn) (char-ci<=? #\b #\9))
(assert-false  (tn) (char-ci<=? #\b #\A))
(assert-true   (tn) (char-ci<=? #\b #\B))
(assert-true   (tn) (char-ci<=? #\b #\Z))
(assert-false  (tn) (char-ci<=? #\b #\_))
(assert-false  (tn) (char-ci<=? #\b #\a))
(assert-true   (tn) (char-ci<=? #\b #\b))
(assert-true   (tn) (char-ci<=? #\b #\z))
(assert-true   (tn) (char-ci<=? #\b #\~))
(assert-true   (tn) (char-ci<=? #\b #\x7f))
(tn "char-ci<=? #\\z")
(assert-false  (tn) (char-ci<=? #\z #\x00))
(assert-false  (tn) (char-ci<=? #\z #\newline))
(assert-false  (tn) (char-ci<=? #\z #\space))
(assert-false  (tn) (char-ci<=? #\z #\!))
(assert-false  (tn) (char-ci<=? #\z #\0))
(assert-false  (tn) (char-ci<=? #\z #\9))
(assert-false  (tn) (char-ci<=? #\z #\A))
(assert-false  (tn) (char-ci<=? #\z #\B))
(assert-true   (tn) (char-ci<=? #\z #\Z))
(assert-false  (tn) (char-ci<=? #\z #\_))
(assert-false  (tn) (char-ci<=? #\z #\a))
(assert-false  (tn) (char-ci<=? #\z #\b))
(assert-true   (tn) (char-ci<=? #\z #\z))
(assert-true   (tn) (char-ci<=? #\z #\~))
(assert-true   (tn) (char-ci<=? #\z #\x7f))
(tn "char-ci<=? #\\~")
(assert-false  (tn) (char-ci<=? #\~ #\x00))
(assert-false  (tn) (char-ci<=? #\~ #\newline))
(assert-false  (tn) (char-ci<=? #\~ #\space))
(assert-false  (tn) (char-ci<=? #\~ #\!))
(assert-false  (tn) (char-ci<=? #\~ #\0))
(assert-false  (tn) (char-ci<=? #\~ #\9))
(assert-false  (tn) (char-ci<=? #\~ #\A))
(assert-false  (tn) (char-ci<=? #\~ #\B))
(assert-false  (tn) (char-ci<=? #\~ #\Z))
(assert-false  (tn) (char-ci<=? #\~ #\_))
(assert-false  (tn) (char-ci<=? #\~ #\a))
(assert-false  (tn) (char-ci<=? #\~ #\b))
(assert-false  (tn) (char-ci<=? #\~ #\z))
(assert-true   (tn) (char-ci<=? #\~ #\~))
(assert-true   (tn) (char-ci<=? #\~ #\x7f))
(tn "char-ci<=? #\\x7f")
(assert-false  (tn) (char-ci<=? #\x7f #\x00))
(assert-false  (tn) (char-ci<=? #\x7f #\newline))
(assert-false  (tn) (char-ci<=? #\x7f #\space))
(assert-false  (tn) (char-ci<=? #\x7f #\!))
(assert-false  (tn) (char-ci<=? #\x7f #\0))
(assert-false  (tn) (char-ci<=? #\x7f #\9))
(assert-false  (tn) (char-ci<=? #\x7f #\A))
(assert-false  (tn) (char-ci<=? #\x7f #\B))
(assert-false  (tn) (char-ci<=? #\x7f #\Z))
(assert-false  (tn) (char-ci<=? #\x7f #\_))
(assert-false  (tn) (char-ci<=? #\x7f #\a))
(assert-false  (tn) (char-ci<=? #\x7f #\b))
(assert-false  (tn) (char-ci<=? #\x7f #\z))
(assert-false  (tn) (char-ci<=? #\x7f #\~))
(assert-true   (tn) (char-ci<=? #\x7f #\x7f))

;; char-ci>=?
(tn "char-ci>=? #\\x00")
(assert-true   (tn) (char-ci>=? #\x00 #\x00))
(assert-false  (tn) (char-ci>=? #\x00 #\newline))
(assert-false  (tn) (char-ci>=? #\x00 #\space))
(assert-false  (tn) (char-ci>=? #\x00 #\!))
(assert-false  (tn) (char-ci>=? #\x00 #\0))
(assert-false  (tn) (char-ci>=? #\x00 #\9))
(assert-false  (tn) (char-ci>=? #\x00 #\A))
(assert-false  (tn) (char-ci>=? #\x00 #\B))
(assert-false  (tn) (char-ci>=? #\x00 #\Z))
(assert-false  (tn) (char-ci>=? #\x00 #\_))
(assert-false  (tn) (char-ci>=? #\x00 #\a))
(assert-false  (tn) (char-ci>=? #\x00 #\b))
(assert-false  (tn) (char-ci>=? #\x00 #\z))
(assert-false  (tn) (char-ci>=? #\x00 #\~))
(assert-false  (tn) (char-ci>=? #\x00 #\x7f))
(tn "char-ci>=? #\\newline")
(assert-true   (tn) (char-ci>=? #\newline #\x00))
(assert-true   (tn) (char-ci>=? #\newline #\newline))
(assert-false  (tn) (char-ci>=? #\newline #\space))
(assert-false  (tn) (char-ci>=? #\newline #\!))
(assert-false  (tn) (char-ci>=? #\newline #\0))
(assert-false  (tn) (char-ci>=? #\newline #\9))
(assert-false  (tn) (char-ci>=? #\newline #\A))
(assert-false  (tn) (char-ci>=? #\newline #\B))
(assert-false  (tn) (char-ci>=? #\newline #\Z))
(assert-false  (tn) (char-ci>=? #\newline #\_))
(assert-false  (tn) (char-ci>=? #\newline #\a))
(assert-false  (tn) (char-ci>=? #\newline #\b))
(assert-false  (tn) (char-ci>=? #\newline #\z))
(assert-false  (tn) (char-ci>=? #\newline #\~))
(assert-false  (tn) (char-ci>=? #\newline #\x7f))
(tn "char-ci>=? #\\space")
(assert-true   (tn) (char-ci>=? #\space #\x00))
(assert-true   (tn) (char-ci>=? #\space #\newline))
(assert-true   (tn) (char-ci>=? #\space #\space))
(assert-false  (tn) (char-ci>=? #\space #\!))
(assert-false  (tn) (char-ci>=? #\space #\0))
(assert-false  (tn) (char-ci>=? #\space #\9))
(assert-false  (tn) (char-ci>=? #\space #\A))
(assert-false  (tn) (char-ci>=? #\space #\B))
(assert-false  (tn) (char-ci>=? #\space #\Z))
(assert-false  (tn) (char-ci>=? #\space #\_))
(assert-false  (tn) (char-ci>=? #\space #\a))
(assert-false  (tn) (char-ci>=? #\space #\b))
(assert-false  (tn) (char-ci>=? #\space #\z))
(assert-false  (tn) (char-ci>=? #\space #\~))
(assert-false  (tn) (char-ci>=? #\space #\x7f))
(tn "char-ci>=? #\\!")
(assert-true   (tn) (char-ci>=? #\! #\x00))
(assert-true   (tn) (char-ci>=? #\! #\newline))
(assert-true   (tn) (char-ci>=? #\! #\space))
(assert-true   (tn) (char-ci>=? #\! #\!))
(assert-false  (tn) (char-ci>=? #\! #\0))
(assert-false  (tn) (char-ci>=? #\! #\9))
(assert-false  (tn) (char-ci>=? #\! #\A))
(assert-false  (tn) (char-ci>=? #\! #\B))
(assert-false  (tn) (char-ci>=? #\! #\Z))
(assert-false  (tn) (char-ci>=? #\! #\_))
(assert-false  (tn) (char-ci>=? #\! #\a))
(assert-false  (tn) (char-ci>=? #\! #\b))
(assert-false  (tn) (char-ci>=? #\! #\z))
(assert-false  (tn) (char-ci>=? #\! #\~))
(assert-false  (tn) (char-ci>=? #\! #\x7f))
(tn "char-ci>=? #\\0")
(assert-true   (tn) (char-ci>=? #\0 #\x00))
(assert-true   (tn) (char-ci>=? #\0 #\newline))
(assert-true   (tn) (char-ci>=? #\0 #\space))
(assert-true   (tn) (char-ci>=? #\0 #\!))
(assert-true   (tn) (char-ci>=? #\0 #\0))
(assert-false  (tn) (char-ci>=? #\0 #\9))
(assert-false  (tn) (char-ci>=? #\0 #\A))
(assert-false  (tn) (char-ci>=? #\0 #\B))
(assert-false  (tn) (char-ci>=? #\0 #\Z))
(assert-false  (tn) (char-ci>=? #\0 #\_))
(assert-false  (tn) (char-ci>=? #\0 #\a))
(assert-false  (tn) (char-ci>=? #\0 #\b))
(assert-false  (tn) (char-ci>=? #\0 #\z))
(assert-false  (tn) (char-ci>=? #\0 #\~))
(assert-false  (tn) (char-ci>=? #\0 #\x7f))
(tn "char-ci>=? #\\9")
(assert-true   (tn) (char-ci>=? #\9 #\x00))
(assert-true   (tn) (char-ci>=? #\9 #\newline))
(assert-true   (tn) (char-ci>=? #\9 #\space))
(assert-true   (tn) (char-ci>=? #\9 #\!))
(assert-true   (tn) (char-ci>=? #\9 #\0))
(assert-true   (tn) (char-ci>=? #\9 #\9))
(assert-false  (tn) (char-ci>=? #\9 #\A))
(assert-false  (tn) (char-ci>=? #\9 #\B))
(assert-false  (tn) (char-ci>=? #\9 #\Z))
(assert-false  (tn) (char-ci>=? #\9 #\_))
(assert-false  (tn) (char-ci>=? #\9 #\a))
(assert-false  (tn) (char-ci>=? #\9 #\b))
(assert-false  (tn) (char-ci>=? #\9 #\z))
(assert-false  (tn) (char-ci>=? #\9 #\~))
(assert-false  (tn) (char-ci>=? #\9 #\x7f))
(tn "char-ci>=? #\\A")
(assert-true   (tn) (char-ci>=? #\A #\x00))
(assert-true   (tn) (char-ci>=? #\A #\newline))
(assert-true   (tn) (char-ci>=? #\A #\space))
(assert-true   (tn) (char-ci>=? #\A #\!))
(assert-true   (tn) (char-ci>=? #\A #\0))
(assert-true   (tn) (char-ci>=? #\A #\9))
(assert-true   (tn) (char-ci>=? #\A #\A))
(assert-false  (tn) (char-ci>=? #\A #\B))
(assert-false  (tn) (char-ci>=? #\A #\Z))
(assert-true   (tn) (char-ci>=? #\A #\_))
(assert-true   (tn) (char-ci>=? #\A #\a))
(assert-false  (tn) (char-ci>=? #\A #\b))
(assert-false  (tn) (char-ci>=? #\A #\z))
(assert-false  (tn) (char-ci>=? #\A #\~))
(assert-false  (tn) (char-ci>=? #\A #\x7f))
(tn "char-ci>=? #\\B")
(assert-true   (tn) (char-ci>=? #\B #\x00))
(assert-true   (tn) (char-ci>=? #\B #\newline))
(assert-true   (tn) (char-ci>=? #\B #\space))
(assert-true   (tn) (char-ci>=? #\B #\!))
(assert-true   (tn) (char-ci>=? #\B #\0))
(assert-true   (tn) (char-ci>=? #\B #\9))
(assert-true   (tn) (char-ci>=? #\B #\A))
(assert-true   (tn) (char-ci>=? #\B #\B))
(assert-false  (tn) (char-ci>=? #\B #\Z))
(assert-true   (tn) (char-ci>=? #\B #\_))
(assert-true   (tn) (char-ci>=? #\B #\a))
(assert-true   (tn) (char-ci>=? #\B #\b))
(assert-false  (tn) (char-ci>=? #\B #\z))
(assert-false  (tn) (char-ci>=? #\B #\~))
(assert-false  (tn) (char-ci>=? #\B #\x7f))
(tn "char-ci>=? #\\Z")
(assert-true   (tn) (char-ci>=? #\Z #\x00))
(assert-true   (tn) (char-ci>=? #\Z #\newline))
(assert-true   (tn) (char-ci>=? #\Z #\space))
(assert-true   (tn) (char-ci>=? #\Z #\!))
(assert-true   (tn) (char-ci>=? #\Z #\0))
(assert-true   (tn) (char-ci>=? #\Z #\9))
(assert-true   (tn) (char-ci>=? #\Z #\A))
(assert-true   (tn) (char-ci>=? #\Z #\B))
(assert-true   (tn) (char-ci>=? #\Z #\Z))
(assert-true   (tn) (char-ci>=? #\Z #\_))
(assert-true   (tn) (char-ci>=? #\Z #\a))
(assert-true   (tn) (char-ci>=? #\Z #\b))
(assert-true   (tn) (char-ci>=? #\Z #\z))
(assert-false  (tn) (char-ci>=? #\Z #\~))
(assert-false  (tn) (char-ci>=? #\Z #\x7f))
(tn "char-ci>=? #\\_")
(assert-true   (tn) (char-ci>=? #\_ #\x00))
(assert-true   (tn) (char-ci>=? #\_ #\newline))
(assert-true   (tn) (char-ci>=? #\_ #\space))
(assert-true   (tn) (char-ci>=? #\_ #\!))
(assert-true   (tn) (char-ci>=? #\_ #\0))
(assert-true   (tn) (char-ci>=? #\_ #\9))
(assert-false  (tn) (char-ci>=? #\_ #\A))
(assert-false  (tn) (char-ci>=? #\_ #\B))
(assert-false  (tn) (char-ci>=? #\_ #\Z))
(assert-true   (tn) (char-ci>=? #\_ #\_))
(assert-false  (tn) (char-ci>=? #\_ #\a))
(assert-false  (tn) (char-ci>=? #\_ #\b))
(assert-false  (tn) (char-ci>=? #\_ #\z))
(assert-false  (tn) (char-ci>=? #\_ #\~))
(assert-false  (tn) (char-ci>=? #\_ #\x7f))
(tn "char-ci>=? #\\a")
(assert-true   (tn) (char-ci>=? #\a #\x00))
(assert-true   (tn) (char-ci>=? #\a #\newline))
(assert-true   (tn) (char-ci>=? #\a #\space))
(assert-true   (tn) (char-ci>=? #\a #\!))
(assert-true   (tn) (char-ci>=? #\a #\0))
(assert-true   (tn) (char-ci>=? #\a #\9))
(assert-true   (tn) (char-ci>=? #\a #\A))
(assert-false  (tn) (char-ci>=? #\a #\B))
(assert-false  (tn) (char-ci>=? #\a #\Z))
(assert-true   (tn) (char-ci>=? #\a #\_))
(assert-true   (tn) (char-ci>=? #\a #\a))
(assert-false  (tn) (char-ci>=? #\a #\b))
(assert-false  (tn) (char-ci>=? #\a #\z))
(assert-false  (tn) (char-ci>=? #\a #\~))
(assert-false  (tn) (char-ci>=? #\a #\x7f))
(tn "char-ci>=? #\\b")
(assert-true   (tn) (char-ci>=? #\b #\x00))
(assert-true   (tn) (char-ci>=? #\b #\newline))
(assert-true   (tn) (char-ci>=? #\b #\space))
(assert-true   (tn) (char-ci>=? #\b #\!))
(assert-true   (tn) (char-ci>=? #\b #\0))
(assert-true   (tn) (char-ci>=? #\b #\9))
(assert-true   (tn) (char-ci>=? #\b #\A))
(assert-true   (tn) (char-ci>=? #\b #\B))
(assert-false  (tn) (char-ci>=? #\b #\Z))
(assert-true   (tn) (char-ci>=? #\b #\_))
(assert-true   (tn) (char-ci>=? #\b #\a))
(assert-true   (tn) (char-ci>=? #\b #\b))
(assert-false  (tn) (char-ci>=? #\b #\z))
(assert-false  (tn) (char-ci>=? #\b #\~))
(assert-false  (tn) (char-ci>=? #\b #\x7f))
(tn "char-ci>=? #\\z")
(assert-true   (tn) (char-ci>=? #\z #\x00))
(assert-true   (tn) (char-ci>=? #\z #\newline))
(assert-true   (tn) (char-ci>=? #\z #\space))
(assert-true   (tn) (char-ci>=? #\z #\!))
(assert-true   (tn) (char-ci>=? #\z #\0))
(assert-true   (tn) (char-ci>=? #\z #\9))
(assert-true   (tn) (char-ci>=? #\z #\A))
(assert-true   (tn) (char-ci>=? #\z #\B))
(assert-true   (tn) (char-ci>=? #\z #\Z))
(assert-true   (tn) (char-ci>=? #\z #\_))
(assert-true   (tn) (char-ci>=? #\z #\a))
(assert-true   (tn) (char-ci>=? #\z #\b))
(assert-true   (tn) (char-ci>=? #\z #\z))
(assert-false  (tn) (char-ci>=? #\z #\~))
(assert-false  (tn) (char-ci>=? #\z #\x7f))
(tn "char-ci>=? #\\~")
(assert-true   (tn) (char-ci>=? #\~ #\x00))
(assert-true   (tn) (char-ci>=? #\~ #\newline))
(assert-true   (tn) (char-ci>=? #\~ #\space))
(assert-true   (tn) (char-ci>=? #\~ #\!))
(assert-true   (tn) (char-ci>=? #\~ #\0))
(assert-true   (tn) (char-ci>=? #\~ #\9))
(assert-true   (tn) (char-ci>=? #\~ #\A))
(assert-true   (tn) (char-ci>=? #\~ #\B))
(assert-true   (tn) (char-ci>=? #\~ #\Z))
(assert-true   (tn) (char-ci>=? #\~ #\_))
(assert-true   (tn) (char-ci>=? #\~ #\a))
(assert-true   (tn) (char-ci>=? #\~ #\b))
(assert-true   (tn) (char-ci>=? #\~ #\z))
(assert-true   (tn) (char-ci>=? #\~ #\~))
(assert-false  (tn) (char-ci>=? #\~ #\x7f))
(tn "char-ci>=? #\\x7f")
(assert-true   (tn) (char-ci>=? #\x7f #\x00))
(assert-true   (tn) (char-ci>=? #\x7f #\newline))
(assert-true   (tn) (char-ci>=? #\x7f #\space))
(assert-true   (tn) (char-ci>=? #\x7f #\!))
(assert-true   (tn) (char-ci>=? #\x7f #\0))
(assert-true   (tn) (char-ci>=? #\x7f #\9))
(assert-true   (tn) (char-ci>=? #\x7f #\A))
(assert-true   (tn) (char-ci>=? #\x7f #\B))
(assert-true   (tn) (char-ci>=? #\x7f #\Z))
(assert-true   (tn) (char-ci>=? #\x7f #\_))
(assert-true   (tn) (char-ci>=? #\x7f #\a))
(assert-true   (tn) (char-ci>=? #\x7f #\b))
(assert-true   (tn) (char-ci>=? #\x7f #\z))
(assert-true   (tn) (char-ci>=? #\x7f #\~))
(assert-true   (tn) (char-ci>=? #\x7f #\x7f))

(total-report)
