/*
 * Copyright (C) 2023, KylinSoft Co., Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
**/
#include "dategroupwidget.h"
#include <QDebug>

DategroupWidget::DategroupWidget(QWidget *parent)
    : QWidget(parent)
{
    dateEdit          = new DateEdit(this); // 日期编辑
    calendarWidget    = new CalendarWidget; // 日历
    hourComboBox      = new QComboBox(this); // 时
    minComboBox       = new QComboBox(this); // 分
    secComboBox       = new QComboBox(this); // 秒
    hourMinColonLabel = new QLabel(":");
    MinSecColonLabel  = new QLabel(":");

    QHBoxLayout *layout = new QHBoxLayout(this);
    QSpacerItem *setDateSpacer = new QSpacerItem(16, 20, QSizePolicy::Policy::Fixed, QSizePolicy::Policy::Fixed);
    layout->setMargin(0);
    layout->setSpacing(0);
    layout->addWidget(dateEdit);
    layout->addSpacerItem(setDateSpacer);
    layout->addWidget(hourComboBox);
    layout->addWidget(hourMinColonLabel);
    layout->addWidget(minComboBox);
    layout->addWidget(MinSecColonLabel);
    layout->addWidget(secComboBox);
    layout->addStretch();
    hourComboBox->setFixedWidth(100);
    minComboBox->setFixedWidth(64);
    secComboBox->setFixedWidth(64);
    hourMinColonLabel->setFixedWidth(16);
    MinSecColonLabel->setFixedWidth(16);
    hourMinColonLabel->setAlignment(Qt::AlignCenter);
    MinSecColonLabel->setAlignment(Qt::AlignCenter);
    dateEdit->setCalendarPopup(true);
    dateEdit->setCalendarWidget(calendarWidget);
#ifdef Nile
    resetDateEdit(kdk_system_get_shortformat());
#endif

    areaInterface = new QDBusInterface("org.ukui.ukcc.session",
                                       "/Area",
                                       "org.ukui.ukcc.session.Area",
                                       QDBusConnection::sessionBus(),
                                       this);
    if (!areaInterface->isValid()) {
        qCritical() << "org.ukui.ukcc.session.Area DBus error:" << areaInterface->lastError();
    }
    QDBusConnection::sessionBus().connect("org.ukui.ukcc.session",
                                          "/Area",
                                          "org.ukui.ukcc.session.Area",
                                          "changed",
                                          this,
                                          SLOT(dataChangedSlot(QString)));
#ifdef Nile
    QDBusConnection::sessionBus().connect("com.kylin.kysdk.DateServer",
                                          "/com/kylin/kysdk/Date",
                                          "com.kylin.kysdk.DateInterface",
                                          "ShortDateSignal",
                                          this,
                                          SLOT(resetDateEdit(QString)));
#endif
    initHour();
    initMinAndSec();
    initConnect();
    updateTime();
    timerID = startTimer(1000);
}

DategroupWidget::~DategroupWidget()
{
    killTimer(timerID);
}

void DategroupWidget::dataChangedSlot(QString key)
{
    if (key == "timeFormat") {
        initHour();
    }
}

void DategroupWidget::resetDateEdit(QString date) {
    QString dateformat;
    int locale = 0;
    int i = 0;


    QLocale l1ocale = QLocale::system();
    if ("zh_CN" == l1ocale.name()){
        locale = 1;
    } else if ("bo_CN" == l1ocale.name()){
        locale = 2;
    }

    dateformat = date;
    while(i < dateformat.length()) {
        if (dateformat.at(i) != 'M' &&
                dateformat.at(i) != 'd' &&
                dateformat.at(i) != 'y')
            break;
        else
            ++i;
    }

    QString split = dateformat.at(i);
    if (locale == 0) {
        if (dateformat.at(dateformat.length() - 3) == split)
            dateEdit->setDisplayFormat("M" + split + "d" + split + "yy");
        else
            dateEdit->setDisplayFormat("MM" + split + "dd" + split + "yyyy");
    } else {
        if (dateformat.at(2) == split)
            dateEdit->setDisplayFormat("yy" + split + "M" + split + "d");
        else
            dateEdit->setDisplayFormat("yyyy" + split + "MM" + split + "dd");
    }
}

void DategroupWidget::initHour()
{
    hourComboBox->clear();
    QString AMname = QLocale::system().amText();
    QString PMname = QLocale::system().pmText();
    if (areaInterface->property("timeFormat").toString() != "24") {
        if (AMname == QString("上午") || AMname == QString("སྔ་དྲོ་")) {
            hourComboBox->addItem(AMname + QString::number(12));
            for (int i = 1 ; i <= 11; i++) {
                hourComboBox->addItem(AMname + QString::number(i));
            }
            hourComboBox->addItem(PMname + QString::number(12));
            for (int i = 1 ; i <= 11; i++) {
                hourComboBox->addItem(PMname + QString::number(i));
            }
        } else {
            hourComboBox->addItem(QString::number(12) + " " + AMname);
            for (int i = 1 ; i <= 11; i++) {
                hourComboBox->addItem(QString::number(i) + " " +AMname);
            }
            hourComboBox->addItem(QString::number(12) + " " +PMname);
            for (int i = 1 ; i <= 11; i++) {
                hourComboBox->addItem(QString::number(i) + "  " + PMname);
            }
        }
    } else {
        for (int h = 0; h < 24; h++){
            hourComboBox->addItem(QString::number(h));
        }
    }
}

void DategroupWidget::initMinAndSec()
{
    for (int m = 0; m < 60; m++) {
        minComboBox->addItem(QString::number(m));
    }

    for (int s = 0; s < 60; s++) {
        secComboBox->addItem(QString::number(s));
    }
}

void DategroupWidget::initConnect()
{
    connect(dateEdit, &DateEdit::changeDate, this, [=]() {
        QDate d(dateEdit->date());
        QTime t(hourComboBox->currentIndex(), minComboBox->currentIndex(), secComboBox->currentIndex());
        Q_EMIT dateChanged(d, t);
    });

    connect(hourComboBox, static_cast<void (QComboBox::*)(int)>(&QComboBox::activated), this, [=]() {
        QDate d(dateEdit->date());
        QTime t(hourComboBox->currentIndex(), minComboBox->currentIndex(), secComboBox->currentIndex());
        Q_EMIT dateChanged(d, t);
    });

    connect(minComboBox, static_cast<void (QComboBox::*)(int)>(&QComboBox::activated), this, [=]() {
        QDate d(dateEdit->date());
        QTime t(hourComboBox->currentIndex(), minComboBox->currentIndex(), secComboBox->currentIndex());
        Q_EMIT dateChanged(d, t);
    });

    connect(secComboBox, static_cast<void (QComboBox::*)(int)>(&QComboBox::activated), this, [=]() {
        QDate d(dateEdit->date());
        QTime t(hourComboBox->currentIndex(), minComboBox->currentIndex(), secComboBox->currentIndex());
        Q_EMIT dateChanged(d, t);
    });
}

void DategroupWidget::timerEvent(QTimerEvent *e)
{
    if (e->timerId() == timerID) {
        updateTime();
    }
}

void DategroupWidget::updateTime()
{
    QDateTime Ctime = QDateTime::currentDateTime();
    dateEdit->blockSignals(true);
    hourComboBox->blockSignals(true);
    minComboBox->blockSignals(true);
    secComboBox->blockSignals(true);
    if (!dateEdit->hasFocus()) {
        dateEdit->setDate(Ctime.date());
    }
    hourComboBox->setCurrentIndex(Ctime.time().hour());
    minComboBox->setCurrentIndex(Ctime.time().minute());
    secComboBox->setCurrentIndex(Ctime.time().second());

    dateEdit->blockSignals(false);
    hourComboBox->blockSignals(false);
    minComboBox->blockSignals(false);
    secComboBox->blockSignals(false);
}
